% Generated by roxygen2 (4.0.1): do not edit by hand
\name{cluster.boot}
\alias{cluster.boot}
\title{Bootstrapped multi-way standard error clustering}
\usage{
cluster.boot(model, cluster, parallel = FALSE, use_white = NULL,
  force_posdef = FALSE, R = 300, debug = FALSE)
}
\arguments{
\item{model}{The estimated model, usually an \code{lm} or \code{glm} class object}

\item{cluster}{A vector, \code{matrix}, or \code{data.frame} of cluster variables,
where each column is a separate variable.  If the vector \code{1:nrow(data)}
is used, the function effectively produces a regular
heteroskedasticity-robust matrix.}

\item{parallel}{Scalar or list.  If a list, use the list as a list
of connected processing cores/clusters.  A scalar indicates no
parallelization.  See the parallel package.}

\item{use_white}{\code{Logical} or \code{NULL}.  See description below.}

\item{force_posdef}{Logical.  Force the eigenvalues of the variance-covariance
matrix to be positive.}

\item{R}{\code{Integer}. The number of bootstrap replicates; passed directly to \code{boot}.}

\item{debug}{\code{Logical}.  Print internal values useful for debugging to
the console.}
}
\value{
a \eqn{k} x \eqn{k} variance-covariance matrix of type \code{matrix}
}
\description{
Return a bootstrapped multi-way cluster-robust variance-covariance matrix
}
\details{
This function implements cluster bootstrapping (also known as the block bootstrap)
for variance-covariance matrices, following Cameron, Gelbach, & Miller (CGM) (2008).
Usage is generally similar to the \code{cluster.vcov} function in this package, but this
function does not support degrees of freedome corrections or leverage adjustments.

In the terminology that CGM (2008) use, this function implements
\emph{pairs cluster bootstrap-se}.

Multi-way clustering is handled as described by Petersen (2009) and generalized
according to Cameron, Gelbach, & Miller (2011).  This means that cluster.boot
estimates a set of variance-covariance matrices \emph{for the variables} separately
and then sums them (subtracting some matrices and adding others).
The method described by CGM (2011) estimates a set of variance-covariance matrices
\emph{for the residuals} (sometimes referred to as the meat of the sandwich estimator)
and sums them appropriately.  Whether you sum the meat matrices and then compute
the model's variance-covariance matrix or you compute a series of model matrices
and sum those is mathematically irrelevant, but may lead to (very) minor numerical
differences.

Unlike the \code{cluster.vcov} function, this function does not depend upon the \code{estfun}
function from the \bold{sandwich} package, although it does make use of the \code{vcovHC}
function for computing White HC0 variance-covariance matrices.

Parallelization (if used) is handled by the \bold{boot} package.
}
\examples{
\dontrun{
library(lmtest)
data(petersen)
m1 <- lm(y ~ x, data = petersen)

# Cluster by firm
boot_firm <- cluster.boot(m1, petersen$firmid)
coeftest(m1, boot_firm)

# Cluster by year
boot_year <- cluster.boot(m1, petersen$year)
coeftest(m1, boot_year)

# Double cluster by firm and year
boot_both <- cluster.boot(m1, cbind(petersen$firmid, petersen$year))
coeftest(m1, boot_both)

# Go multicore using the parallel package
require(parallel)
cl <- makeCluster(4)
boot_both <- cluster.boot(m1, cbind(petersen$firmid, petersen$year), parallel = cl)
stopCluster(cl)
coeftest(m1, boot_both)
}
}
\author{
Nathaniel Graham \email{npgraham1@gmail.com}
}
\references{
Cameron, A. C., Gelbach, J. B., & Miller, D. L. (2008). Bootstrap-based improvements
for inference with clustered errors. The Review of Economics and Statistics, 90(3), 414-427.

Cameron, A. C., Gelbach, J. B., & Miller, D. L. (2011). Robust inference with multiway
clustering. Journal of Business & Economic Statistics, 29(2).

Petersen, M. A. (2009). Estimating standard errors in finance panel data sets: Comparing
approaches. Review of financial studies, 22(1), 435-480.
}
\keyword{block}
\keyword{boot}
\keyword{bootstrap}
\keyword{clustering}
\keyword{errors}
\keyword{multi-way}
\keyword{robust}
\keyword{standard}

