% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rewire.R
\name{rewire_graph}
\alias{rewire_graph}
\title{Graph rewiring algorithms}
\usage{
rewire_graph(graph, p, algorithm = "endpoints", both.ends = FALSE,
  self = FALSE, multiple = FALSE,
  undirected = getOption("diffnet.undirected"), copy.first = TRUE)
}
\arguments{
\item{graph}{Any class of accepted graph format (see \code{\link{netdiffuseR-graphs}})}

\item{p}{Either a [0,1] vector with rewiring probabilities (\code{algorithm="endpoints"}),
or an integer vector with number of iterations (\code{algorithm="swap"}).}

\item{algorithm}{Character scalar. Either \code{"swap"} or \code{"endpoints"}.}

\item{both.ends}{Logical scalar. When \code{TRUE} rewires both ends.}

\item{self}{Logical scalar. When \code{TRUE}, allows loops (self edges).}

\item{multiple}{Logical scalar. When \code{TRUE} allows multiple edges.}

\item{undirected}{Logical scalar. \code{TRUE} when the graph is undirected.}

\item{copy.first}{Logical scalar. When \code{TRUE} and \code{graph} is dynamic uses
the first slice as a baseline for the rest of slices (see details).}
}
\description{
Changes the structure of a graph by altering ties.
}
\details{
Both algorithms are implemented sequentially, this is, edge-wise checking
self edges and multiple edges over the changing graph; in other words, at step
\eqn{m} (in which either a new endpoint or edge is chosen, depending on the algorithm),
the algorithms verify whether the proposed change creates either multiple edges
or self edges using the resulting graph at step \eqn{m-1}.

The main difference between the two algorithms is that the \code{"swap"} algorithm
preserves the degree sequence of the graph and \code{"endpoints"} does not.
The \code{"swap"} algorithm is specially useful to asses the non-randomness of
a graph's structural properties, furthermore it is this algorithm the one used
in the \code{\link{struct_test}} routine implemented in \pkg{netdiffuseR}.

Rewiring assumes a weighted network, hence \eqn{G(i,j) = k = G(i',j')},
where \eqn{i',j'} are the new end points of the edge and \eqn{k} may not be equal
to one.

In the case of dynamic graphs, when \code{copy.first=TRUE}, after rewiring the
first slice--\eqn{t=1}--the rest of slices are generated by rewiring the rewired
version of the first slice. Formally:

\deqn{%
G(t)' = \left\{\begin{array}{ll}
R(G(t)) & \mbox{if }t=1 \\
R(G(1)') & \mbox{otherwise}
\end{array}
\right.
}{%
G(t)' = R(G(t))  if t=1,
        R(G(1)') otherwise
}

Where \eqn{G(t)} is the t-th slice, \eqn{G(t)'} is the t-th rewired slice, and
\eqn{R} is the rewiring function. Otherwise, \code{copy.first=FALSE} (default),
The rewiring function is simply \eqn{G(t)' = R(G(t))}.

The following sections describe the way both algorithms were implemented.
}
\section{\emph{Swap} algorithm}{

The \code{"swap"} algorithm chooses randomly two edges \eqn{(a,b)} and
\eqn{(c,d)} and swaps the 'right' endpoint of boths such that we get
\eqn{(a,d)} and \eqn{(c,b)} (considering self and multiple edges).

Following Milo et al. (2004) testing procedure, the algorithm shows to be
well behaved in terms of been unbiased, so after each iteration each possible
structure of the graph has the same probability of been generated. The algorithm
has been implemented as follows:

Let \eqn{E} be the set of edges of the graph \eqn{G}. For \eqn{i=1} to \eqn{p}, do:
\enumerate{
 \item Choose \eqn{e0=(a, b)} from \eqn{E}. If \code{!self & a == b} then go to the last step.
 \item Choose \eqn{e1=(c, d)} from \eqn{E}. If \code{!self & c == d } then go to the last step.
 \item Define \eqn{e0'=(a, d)} and \eqn{e1' = (c, b)}. If \code{!multiple & [G[e0']!= 0 | G[e1'] != 0]} then go to the last step.
 \item Define \eqn{v0 = G[e0]} and \eqn{v1 = G[e1]}, set \eqn{G[e0]=0} and \eqn{G[e1]=0}
 (and the same to the diagonally opposed coordinates in the case of undirected graphs)
 \item Set \eqn{G[e0'] = v0} and \eqn{G[e1'] = v1} (and so with the diagonally opposed coordinates
 in the case of undirected graphs).
 \item Next i.
}

Milo et al. (2004) suggests that in order for the rewired graph to be independent
from the original one researchers usually iterate around \code{nlinks(graph)*100}
times, so \code{p=nlinks(graph)*100} is recommended.
}

\section{\emph{Endpoints} algorithm}{


This reconnect either one or both of the endpoints of the edge randomly. As a big
difference with the swap algorithm is that this does not preserves the degree
sequence of the graph (at most the outgoing degree sequence). The algorithm is
implemented as follows:

Let \eqn{G} be the baseline graph and \eqn{G'} be a copy of it. Then, For \eqn{l=1} to \eqn{|E|} do:

\enumerate{
 \item Pick the \eqn{l}-th edge from \eqn{E}, define it as \eqn{e = (i,j)}.
 \item Draw \eqn{r} from \eqn{U(0,1)}, if \eqn{r > p} go to the last step.
 \item If \code{!undirected & i < j} go to the last step.
 \item Randomly select a vertex \eqn{j'} (and \eqn{i'} if \code{both_ends==TRUE}).
       And define \eqn{e'=(i, j')} (or \eqn{e'=(i', j')} if \code{both_ends==TRUE}).
 \item If \code{!self &} \code{i==j}' (or if \code{both_ends==TRUE & i'==j'}) go to the last step.
 \item If \code{!multiple & G'[e']!= 0} then go to the last step.
 \item Define \eqn{v = G[e]}, set \eqn{G'[e] = 0} and \eqn{G'[e'] = v} (and the
       same to the diagonally opposed coordinates in the case of undirected graphs).
 \item Next \eqn{l}.
}

The endpoints algorithm is used by default in \code{\link{rdiffnet}} and used
to be the default in \code{\link{struct_test}} (now \code{swap} is the default).
}
\examples{
# Checking the consistency of the "swap" ------------------------------------

# A graph with known structure (see Milo 2004)
n <- 5
x <- matrix(0, ncol=n, nrow=n)
x <- as(x, "dgCMatrix")
x[1,c(-1,-n)] <- 1
x[c(-1,-n),n] <- 1

x

# Simulations (increase the number for more precission)
set.seed(8612)
nsim <- 1e4
w <- sapply(seq_len(nsim), function(y) {
 # Creating the new graph
 g <- rewire_graph(x,p=nlinks(x)*100, algorithm = "swap")

 # Categorizing (tag of the generated structure)
 paste0(as.vector(g), collapse="")
})

# Counting
coded <- as.integer(as.factor(w))

plot(table(coded)/nsim*100, type="p", ylab="Frequency \%", xlab="Class of graph", pch=3,
 main="Distribution of classes generated by rewiring")

# Marking the original structure
baseline <- paste0(as.vector(x), collapse="")
points(x=7,y=table(as.factor(w))[baseline]/nsim*100, pch=3, col="red")

}
\author{
George G. Vega Yon
}
\references{
Watts, D. J., & Strogatz, S. H. (1998). Collectivedynamics of "small-world" networks.
Nature, 393(6684), 440–442. \url{http://doi.org/10.1038/30918}

Milo, R., Kashtan, N., Itzkovitz, S., Newman, M. E. J., & Alon, U.
(2004). On the uniform generation of random graphs with prescribed degree sequences.
Arxiv Preprint condmat0312028, cond-mat/0, 1–4. Retrieved from
\url{http://arxiv.org/abs/cond-mat/0312028}
}
\seealso{
Other simulation functions: \code{\link{rdiffnet}},
  \code{\link{rgraph_ba}}, \code{\link{rgraph_er}},
  \code{\link{rgraph_ws}}, \code{\link{ring_lattice}}
}

