
<!-- README.md is generated from README.Rmd. Please edit that file -->

# nlmixr2extra

<!-- badges: start -->

[![R-CMD-check](https://github.com/nlmixr2/nlmixr2extra/workflows/R-CMD-check/badge.svg)](https://github.com/nlmixr2/nlmixr2extra/actions)
[![Codecov test
coverage](https://codecov.io/gh/nlmixr2/nlmixr2extra/branch/main/graph/badge.svg)](https://app.codecov.io/gh/nlmixr2/nlmixr2extra?branch=main)
[![CRAN
version](http://www.r-pkg.org/badges/version/nlmixr2extra)](https://cran.r-project.org/package=nlmixr2extra)
[![CRAN total
downloads](https://cranlogs.r-pkg.org/badges/grand-total/nlmixr2extra)](https://cran.r-project.org/package=nlmixr2extra)
[![CRAN total
downloads](https://cranlogs.r-pkg.org/badges/nlmixr2extra)](https://cran.r-project.org/package=nlmixr2extra)
[![CodeFactor](https://www.codefactor.io/repository/github/nlmixr2/nlmixr2extra/badge)](https://www.codefactor.io/repository/github/nlmixr2/nlmixr2extra)
![r-universe](https://nlmixr2.r-universe.dev/badges/nlmixr2extra)
<!-- badges: end -->

The goal of nlmixr2extra is to provide the tools to help with common
pharmacometric tasks with nlmixr2 models like bootstrapping, covariate
selection etc.

## Installation

You can install the development version of nlmixr2extra from
[GitHub](https://github.com/) with:

``` r
# install.packages("remotes")
remotes::install_github("nlmixr2/nlmixr2data")
remotes::install_github("nlmixr2/lotri")
remotes::install_github("nlmixr2/rxode2")
remotes::install_github("nlmixr2/nlmixr2est")
remotes::install_github("nlmixr2/nlmixr2extra")
```

## Example of a `bootstrapFit()`

This is a basic example of bootstrapping provided by this package

``` r
library(nlmixr2est)
#> Loading required package: nlmixr2data
library(nlmixr2extra)
## basic example code
## The basic model consiss of an ini block that has initial estimates
one.compartment <- function() {
  ini({
    tka <- 0.45 # Log Ka
    tcl <- 1 # Log Cl
    tv <- 3.45    # Log V
    eta.ka ~ 0.6
    eta.cl ~ 0.3
    eta.v ~ 0.1
    add.sd <- 0.7
  })
  # and a model block with the error sppecification and model specification
  model({
    ka <- exp(tka + eta.ka)
    cl <- exp(tcl + eta.cl)
    v <- exp(tv + eta.v)
    d/dt(depot) = -ka * depot
    d/dt(center) = ka * depot - cl / v * center
    cp = center / v
    cp ~ add(add.sd)
  })
}

## The fit is performed by the function nlmixr/nlmix2 specifying the model, data and estimate
fit <- nlmixr2(one.compartment, theo_sd,  est="saem", saemControl(print=0))
#> ℹ parameter labels from comments will be replaced by 'label()'
#> → loading into symengine environment...
#> → pruning branches (`if`/`else`) of saem model...
#> ✔ done
#> → finding duplicate expressions in saem model...
#> [====|====|====|>---------------------------------]  33%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  
#> → optimizing duplicate expressions in saem model...
#> [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  
#> ✔ done
#> rxode2 2.0.9 using 4 threads (see ?getRxThreads)
#> Calculating covariance matrix
#> >-------------------------------------------------]  01%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  
#> → loading into symengine environment...
#> → pruning branches (`if`/`else`) of saem model...
#> ✔ done
#> → finding duplicate expressions in saem predOnly model 0...
#> [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  
#> → finding duplicate expressions in saem predOnly model 1...
#> [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  
#> → optimizing duplicate expressions in saem predOnly model 1...
#> [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  
#> → finding duplicate expressions in saem predOnly model 2...
#> [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  
#> ✔ done
#> → Calculating residuals/tables
#> ✔ done
#> → compress origData in nlmixr2 object, save 5952
#> → compress phiM in nlmixr2 object, save 62360
#> → compress parHist in nlmixr2 object, save 9560
#> → compress saem0 in nlmixr2 object, save 28720

fit2 <- suppressMessages(bootstrapFit(fit))
#> [====|====|====|>---------------------------------]  33%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.288803    0.978875    3.446285    0.392401    0.088720    0.016800    1.797015    
#> 002: 0.364956    1.006541    3.454819    0.372781    0.084284    0.015960    1.185648    
#> 003: 0.510180    0.904762    3.474734    0.354142    0.080070    0.015162    0.918608    
#> 004: 0.469132    0.928467    3.474083    0.336435    0.076067    0.014404    0.843936    
#> 005: 0.549086    0.944019    3.471888    0.344033    0.072263    0.013684    0.812700    
#> 006: 0.507476    0.961289    3.458957    0.341217    0.078991    0.013000    0.743293    
#> 007: 0.469826    0.948615    3.456028    0.324156    0.075041    0.012350    0.767495    
#> 008: 0.461403    0.965787    3.442739    0.396810    0.073617    0.011732    0.761684    
#> 009: 0.440973    0.936870    3.437581    0.422620    0.074051    0.011146    0.771547    
#> 010: 0.419993    0.955247    3.441024    0.414608    0.070348    0.010588    0.780521    
#> 011: 0.506499    0.966527    3.441234    0.458716    0.074833    0.010059    0.765749    
#> 012: 0.493076    0.965366    3.430157    0.435780    0.071091    0.011213    0.738729    
#> 013: 0.415532    0.966124    3.430387    0.413991    0.068155    0.013143    0.743716    
#> 014: 0.474001    0.946162    3.446693    0.426304    0.086004    0.013508    0.750853    
#> 015: 0.506237    0.954114    3.462542    0.404989    0.081704    0.012832    0.739427    
#> 016: 0.507395    0.958252    3.464558    0.447393    0.099510    0.012191    0.745778    
#> 017: 0.502623    0.959698    3.468936    0.443470    0.098485    0.011581    0.740544    
#> 018: 0.499805    0.947233    3.451106    0.421297    0.108327    0.011002    0.743342    
#> 019: 0.463627    0.950666    3.448514    0.431765    0.102910    0.010452    0.734969    
#> 020: 0.511487    0.964835    3.455564    0.442147    0.097765    0.010004    0.740815    
#> 021: 0.498569    0.943445    3.464077    0.420039    0.092877    0.009504    0.740021    
#> 022: 0.497442    0.940285    3.469202    0.463414    0.088233    0.009029    0.749821    
#> 023: 0.573617    0.947463    3.481081    0.440243    0.083821    0.008577    0.766046    
#> 024: 0.566936    0.961609    3.475120    0.435935    0.079630    0.008148    0.772363    
#> 025: 0.533325    0.936007    3.458758    0.460408    0.076570    0.008190    0.771306    
#> 026: 0.504430    0.961734    3.458389    0.437387    0.084167    0.011287    0.755922    
#> 027: 0.455874    0.987119    3.451109    0.415518    0.087975    0.012124    0.790582    
#> 028: 0.497673    0.953105    3.457788    0.394742    0.090866    0.011518    0.750350    
#> 029: 0.524332    0.947197    3.450710    0.440898    0.087640    0.010942    0.765437    
#> 030: 0.543604    0.958975    3.446713    0.418853    0.088824    0.010395    0.761807    
#> 031: 0.518314    0.936689    3.459394    0.515493    0.084383    0.009875    0.729652    
#> 032: 0.522224    0.944341    3.458723    0.583728    0.089635    0.009381    0.724806    
#> 033: 0.500102    0.955974    3.455613    0.554541    0.101250    0.008912    0.731551    
#> 034: 0.532469    0.957643    3.456762    0.526814    0.096188    0.008872    0.722828    
#> 035: 0.464806    0.938234    3.443717    0.500473    0.091378    0.008428    0.730470    
#> 036: 0.477341    0.951170    3.446830    0.475450    0.099147    0.008007    0.731938    
#> 037: 0.518269    0.958704    3.453297    0.451677    0.094190    0.007606    0.750267    
#> 038: 0.509890    0.960283    3.440294    0.429093    0.089480    0.007598    0.735975    
#> 039: 0.486821    0.955821    3.456073    0.407639    0.087368    0.007218    0.743754    
#> 040: 0.491917    0.963409    3.461726    0.387257    0.084039    0.006857    0.751194    
#> 041: 0.483130    0.946241    3.459153    0.388285    0.088669    0.006514    0.724504    
#> 042: 0.485180    0.946132    3.446349    0.368871    0.088233    0.006339    0.740779    
#> 043: 0.422046    0.973716    3.434888    0.350427    0.083821    0.006117    0.742111    
#> 044: 0.428742    0.977601    3.436592    0.335622    0.094692    0.006374    0.749974    
#> 045: 0.426329    0.979645    3.429591    0.335578    0.089957    0.006055    0.757419    
#> 046: 0.432972    0.969997    3.437091    0.318800    0.085459    0.005752    0.756750    
#> 047: 0.448389    0.964076    3.440040    0.346197    0.081186    0.005465    0.749620    
#> 048: 0.451875    0.945788    3.437565    0.396699    0.079399    0.005191    0.751811    
#> 049: 0.439602    0.942155    3.442474    0.429846    0.084800    0.005683    0.743807    
#> 050: 0.517837    0.918489    3.448426    0.448612    0.094424    0.006945    0.765733    
#> 051: 0.478900    0.924461    3.445303    0.426181    0.089703    0.006867    0.766745    
#> 052: 0.449570    0.949465    3.446793    0.404872    0.092798    0.006743    0.747830    
#> 053: 0.471306    0.952123    3.444047    0.392100    0.095943    0.006406    0.735880    
#> 054: 0.441670    0.971257    3.443328    0.376509    0.106932    0.006086    0.741067    
#> 055: 0.455486    0.959621    3.437541    0.377678    0.104389    0.005781    0.743856    
#> 056: 0.469347    0.941851    3.440332    0.417126    0.136588    0.005492    0.764651    
#> 057: 0.475796    0.931409    3.450799    0.430272    0.129759    0.006295    0.774194    
#> 058: 0.471262    0.932520    3.444068    0.408758    0.123271    0.007261    0.778025    
#> 059: 0.441325    0.915503    3.455493    0.388321    0.117108    0.009561    0.790378    
#> 060: 0.474100    0.909014    3.466437    0.368905    0.123584    0.009083    0.788505    
#> 061: 0.535053    0.937604    3.464648    0.404767    0.131965    0.008628    0.764447    
#> 062: 0.525797    0.962440    3.466482    0.418208    0.125366    0.008197    0.770208    
#> 063: 0.511458    0.922591    3.461686    0.412103    0.119098    0.009369    0.756928    
#> 064: 0.469058    0.931515    3.462516    0.391498    0.113143    0.008900    0.762781    
#> 065: 0.509476    0.941934    3.450363    0.404834    0.107486    0.008455    0.770649    
#> 066: 0.520119    0.955202    3.457171    0.434849    0.102112    0.008032    0.737724    
#> 067: 0.486520    0.944835    3.441301    0.413107    0.098765    0.007631    0.737381    
#> 068: 0.470243    0.965602    3.448452    0.441279    0.128808    0.007249    0.740466    
#> 069: 0.472063    0.946924    3.443311    0.453382    0.122368    0.006887    0.744896    
#> 070: 0.463350    0.961480    3.455510    0.430713    0.117555    0.006936    0.746043    
#> 071: 0.457413    0.964532    3.436077    0.442671    0.113715    0.006753    0.752737    
#> 072: 0.436475    0.928230    3.444981    0.422472    0.108029    0.006416    0.759115    
#> 073: 0.476644    0.954000    3.459707    0.414139    0.117840    0.006387    0.751274    
#> 074: 0.447828    0.919595    3.457887    0.405758    0.113772    0.006476    0.729825    
#> 075: 0.498616    0.921909    3.479366    0.385470    0.108083    0.006153    0.742484    
#> 076: 0.512583    0.940805    3.482913    0.379778    0.107899    0.006068    0.759221    
#> 077: 0.544691    0.915642    3.474145    0.441685    0.106063    0.005773    0.763239    
#> 078: 0.540503    0.941830    3.474269    0.489171    0.104573    0.006367    0.736902    
#> 079: 0.522371    0.950949    3.474903    0.464712    0.112726    0.007651    0.755575    
#> 080: 0.508734    0.951866    3.468378    0.441477    0.118943    0.010981    0.780721    
#> 081: 0.504811    0.937237    3.468959    0.419403    0.112996    0.012391    0.772237    
#> 082: 0.526606    0.922567    3.481072    0.432144    0.107346    0.011772    0.761130    
#> 083: 0.557222    0.969039    3.481450    0.433290    0.101979    0.011183    0.761912    
#> 084: 0.531451    0.962029    3.470704    0.436897    0.103667    0.010624    0.758900    
#> 085: 0.507147    0.961483    3.473442    0.427155    0.112207    0.010093    0.764063    
#> 086: 0.530548    0.940320    3.468151    0.479643    0.106596    0.009588    0.752891    
#> 087: 0.493182    0.941121    3.451665    0.455826    0.101267    0.009109    0.750047    
#> 088: 0.419334    0.954925    3.447406    0.433035    0.098326    0.008653    0.772469    
#> 089: 0.430799    0.978210    3.429080    0.447842    0.107078    0.008221    0.758035    
#> 090: 0.459594    0.945198    3.433787    0.425450    0.109974    0.007810    0.762033    
#> 091: 0.464047    0.942131    3.430523    0.404509    0.104475    0.007419    0.768378    
#> 092: 0.423904    0.969480    3.429625    0.393773    0.099251    0.007048    0.748858    
#> 093: 0.401111    0.976491    3.425267    0.412789    0.094289    0.006741    0.748389    
#> 094: 0.442180    0.979174    3.423586    0.451546    0.100485    0.008059    0.743660    
#> 095: 0.441106    0.985149    3.430352    0.449848    0.104361    0.007656    0.741488    
#> 096: 0.413562    0.969325    3.428369    0.444031    0.101412    0.007273    0.734259    
#> 097: 0.459272    0.985031    3.430003    0.421829    0.096342    0.006910    0.740791    
#> 098: 0.453960    0.998615    3.429766    0.454905    0.104526    0.006564    0.741354    
#> 099: 0.419222    0.954711    3.422077    0.432160    0.120947    0.006236    0.730669    
#> 100: 0.436425    0.989973    3.423301    0.410552    0.114899    0.005924    0.752592    
#> 101: 0.363424    0.966448    3.406477    0.423428    0.109154    0.005628    0.757607    
#> 102: 0.380330    0.974623    3.399920    0.443425    0.103697    0.005346    0.756598    
#> 103: 0.422332    0.981215    3.412543    0.430551    0.102869    0.005079    0.743735    
#> 104: 0.420135    0.977924    3.424860    0.437135    0.108996    0.004825    0.744525    
#> 105: 0.464498    0.963721    3.429126    0.467772    0.117760    0.004713    0.751498    
#> 106: 0.441836    0.971341    3.427214    0.444384    0.111872    0.004478    0.760660    
#> 107: 0.469697    0.966332    3.431592    0.422165    0.106349    0.004660    0.759925    
#> 108: 0.453203    0.963519    3.424006    0.401056    0.111735    0.004427    0.762516    
#> 109: 0.449129    0.965072    3.433759    0.447038    0.106149    0.004206    0.768293    
#> 110: 0.473089    0.947578    3.424740    0.424686    0.100841    0.003995    0.764174    
#> 111: 0.485182    0.963313    3.433335    0.447556    0.100716    0.003796    0.756380    
#> 112: 0.428718    0.956322    3.424501    0.448877    0.103546    0.003606    0.764068    
#> 113: 0.446540    0.978737    3.433397    0.489231    0.104708    0.003425    0.731813    
#> 114: 0.438656    0.973719    3.434709    0.464770    0.099473    0.003254    0.746007    
#> 115: 0.470419    0.938681    3.436043    0.441531    0.098791    0.003091    0.766273    
#> 116: 0.476104    0.955957    3.441140    0.419455    0.106805    0.002937    0.750867    
#> 117: 0.453670    0.959092    3.440723    0.420248    0.101465    0.002959    0.764741    
#> 118: 0.490860    0.938870    3.443741    0.454150    0.096392    0.002811    0.760336    
#> 119: 0.451697    0.956656    3.440340    0.431443    0.098932    0.003810    0.765562    
#> 120: 0.394145    0.958557    3.438028    0.409871    0.094237    0.004594    0.761570    
#> 121: 0.427551    0.986137    3.431223    0.389377    0.096651    0.004487    0.753006    
#> 122: 0.439188    0.969990    3.432917    0.447196    0.118226    0.004663    0.737358    
#> 123: 0.519760    0.989493    3.440383    0.474418    0.112315    0.005278    0.756673    
#> 124: 0.484024    1.004367    3.435405    0.450697    0.106699    0.005832    0.750003    
#> 125: 0.485707    0.987394    3.431322    0.428162    0.101364    0.006043    0.732638    
#> 126: 0.430954    0.994236    3.421378    0.406754    0.096296    0.005838    0.756869    
#> 127: 0.423875    0.994805    3.419960    0.394904    0.091481    0.005546    0.756201    
#> 128: 0.434773    0.962520    3.436852    0.408054    0.098022    0.005885    0.738128    
#> 129: 0.469388    0.966084    3.447186    0.446969    0.093121    0.005591    0.744223    
#> 130: 0.528366    0.962156    3.452884    0.459002    0.099687    0.006295    0.741606    
#> 131: 0.481980    0.948117    3.441670    0.481178    0.115363    0.006436    0.739382    
#> 132: 0.467723    0.970790    3.436009    0.457119    0.109595    0.006114    0.755441    
#> 133: 0.435932    0.944731    3.436085    0.434263    0.121056    0.006126    0.739174    
#> 134: 0.461495    0.929387    3.446464    0.430911    0.134542    0.006191    0.744995    
#> 135: 0.472340    0.957669    3.442927    0.434591    0.127815    0.005881    0.736639    
#> 136: 0.474256    0.934193    3.444043    0.441157    0.123855    0.005587    0.736324    
#> 137: 0.505845    0.922343    3.450535    0.421987    0.117663    0.005308    0.761050    
#> 138: 0.451694    0.955319    3.448829    0.400887    0.111779    0.005043    0.752507    
#> 139: 0.439039    0.961695    3.450229    0.380843    0.106190    0.004790    0.763887    
#> 140: 0.464308    0.977783    3.450012    0.363441    0.116952    0.004986    0.751776    
#> 141: 0.458861    0.939787    3.458431    0.345269    0.112632    0.004736    0.741117    
#> 142: 0.487723    0.931551    3.469847    0.346406    0.119205    0.005518    0.744071    
#> 143: 0.458718    0.946390    3.468955    0.348256    0.113245    0.005430    0.759131    
#> 144: 0.529996    0.961842    3.474024    0.363795    0.107582    0.005159    0.737832    
#> 145: 0.483242    0.970463    3.457576    0.365647    0.102203    0.004901    0.716821    
#> 146: 0.494759    0.976448    3.460801    0.347364    0.097093    0.004656    0.745750    
#> 147: 0.450969    0.942690    3.456482    0.413325    0.096508    0.004640    0.723905    
#> 148: 0.478025    0.947088    3.451994    0.413991    0.091683    0.004846    0.719231    
#> 149: 0.447375    0.957809    3.449055    0.395559    0.096620    0.004720    0.761867    
#> 150: 0.467183    0.954304    3.441026    0.375781    0.092439    0.004581    0.763224    
#> 151: 0.439408    0.979423    3.437082    0.381180    0.090979    0.004352    0.742385    
#> 152: 0.428882    0.968376    3.439029    0.421073    0.092207    0.003179    0.754409    
#> 153: 0.443055    0.981745    3.442171    0.406970    0.092866    0.002931    0.760870    
#> 154: 0.488322    0.995699    3.442702    0.452628    0.089253    0.002953    0.772505    
#> 155: 0.473431    0.962597    3.438086    0.403344    0.108593    0.003132    0.760016    
#> 156: 0.495660    0.947440    3.456187    0.416763    0.122177    0.002565    0.755968    
#> 157: 0.484785    0.931260    3.453258    0.475733    0.129758    0.003171    0.760634    
#> 158: 0.516732    0.949053    3.454186    0.423354    0.123405    0.002719    0.739213    
#> 159: 0.467722    0.951746    3.454156    0.430671    0.126974    0.002088    0.740260    
#> 160: 0.459446    0.927070    3.455133    0.398576    0.127631    0.002634    0.739849    
#> 161: 0.467911    0.947480    3.451781    0.420824    0.126574    0.002681    0.743885    
#> 162: 0.459405    0.968112    3.439447    0.426089    0.110677    0.002047    0.752310    
#> 163: 0.463123    0.962020    3.434702    0.458844    0.146922    0.002028    0.750370    
#> 164: 0.459105    0.966549    3.431019    0.449204    0.121394    0.002178    0.727098    
#> 165: 0.473393    0.958834    3.440640    0.437648    0.110515    0.002823    0.729620    
#> 166: 0.489012    0.993194    3.438564    0.411402    0.125159    0.002733    0.747717    
#> 167: 0.472154    0.980911    3.441748    0.380931    0.110477    0.002843    0.746862    
#> 168: 0.498113    0.970005    3.446624    0.359183    0.102551    0.003176    0.742132    
#> 169: 0.463266    0.983002    3.444956    0.363983    0.109333    0.002582    0.765161    
#> 170: 0.484727    0.956382    3.445273    0.434579    0.115131    0.003259    0.746367    
#> 171: 0.456108    0.949285    3.448712    0.373559    0.121753    0.003342    0.745744    
#> 172: 0.477177    0.991482    3.443321    0.437363    0.102956    0.003036    0.734714    
#> 173: 0.464353    0.976521    3.437631    0.438937    0.100818    0.003391    0.740831    
#> 174: 0.446915    0.976857    3.440733    0.338572    0.085301    0.004233    0.721075    
#> 175: 0.420167    0.989785    3.427637    0.341420    0.098488    0.003773    0.736391    
#> 176: 0.449420    0.993313    3.434901    0.423242    0.092468    0.003505    0.732045    
#> 177: 0.466618    0.961370    3.440405    0.386342    0.094418    0.003115    0.748267    
#> 178: 0.493899    0.947550    3.443843    0.425145    0.097029    0.003229    0.739673    
#> 179: 0.506294    0.968380    3.439980    0.398604    0.093805    0.003805    0.751596    
#> 180: 0.442553    0.981432    3.437645    0.493055    0.085104    0.004086    0.736739    
#> 181: 0.429998    0.991809    3.424742    0.423989    0.081626    0.004012    0.742858    
#> 182: 0.405708    0.982561    3.424090    0.370298    0.088101    0.003663    0.763971    
#> 183: 0.414927    0.995409    3.424486    0.340061    0.079909    0.003197    0.786911    
#> 184: 0.404763    0.985948    3.421457    0.397777    0.085608    0.003252    0.772746    
#> 185: 0.412959    0.988062    3.418124    0.378339    0.112376    0.002404    0.756268    
#> 186: 0.413852    0.988544    3.426744    0.339053    0.095202    0.002582    0.738944    
#> 187: 0.421847    0.966032    3.435079    0.352461    0.108327    0.002809    0.743877    
#> 188: 0.385113    0.976193    3.425364    0.370981    0.099044    0.002015    0.762471    
#> 189: 0.438061    0.997796    3.427983    0.395440    0.082898    0.002261    0.763536    
#> 190: 0.432364    0.983883    3.433558    0.386783    0.091704    0.002096    0.748394    
#> 191: 0.427803    0.976015    3.427885    0.425679    0.101425    0.002267    0.751794    
#> 192: 0.448061    0.974555    3.430797    0.460236    0.097076    0.002130    0.749212    
#> 193: 0.455619    0.988489    3.433868    0.397224    0.101441    0.001862    0.738211    
#> 194: 0.434446    0.970611    3.428784    0.404154    0.121711    0.002044    0.740942    
#> 195: 0.414566    0.956487    3.429804    0.458587    0.106098    0.002548    0.742081    
#> 196: 0.446255    0.959684    3.437557    0.421609    0.112315    0.002176    0.748796    
#> 197: 0.426148    0.969609    3.445027    0.384099    0.109645    0.002441    0.756016    
#> 198: 0.470119    0.948722    3.438140    0.438263    0.106622    0.003690    0.747920    
#> 199: 0.420581    0.953792    3.438336    0.455127    0.115806    0.003771    0.749115    
#> 200: 0.448832    0.953222    3.444956    0.381452    0.123954    0.004150    0.760650    
#> 201: 0.445597    0.952101    3.445784    0.377768    0.113816    0.005814    0.749324    
#> 202: 0.463523    0.954899    3.445861    0.409530    0.112791    0.006010    0.747730    
#> 203: 0.474453    0.958576    3.444508    0.404531    0.109405    0.006253    0.746563    
#> 204: 0.474862    0.954314    3.446486    0.410124    0.103148    0.006477    0.748830    
#> 205: 0.473134    0.959256    3.447723    0.408211    0.099814    0.006493    0.753053    
#> 206: 0.472591    0.961986    3.448561    0.415798    0.098660    0.006449    0.753038    
#> 207: 0.472908    0.960697    3.448238    0.407244    0.099952    0.006365    0.751949    
#> 208: 0.473164    0.960600    3.448398    0.405027    0.100754    0.006388    0.750095    
#> 209: 0.476017    0.959642    3.448107    0.408530    0.100135    0.006491    0.749882    
#> 210: 0.478340    0.957422    3.448597    0.403661    0.098980    0.006553    0.748658    
#> 211: 0.482571    0.956641    3.449629    0.396951    0.097621    0.006486    0.747652    
#> 212: 0.484260    0.954916    3.450463    0.392645    0.097143    0.006324    0.746281    
#> 213: 0.487760    0.952665    3.450796    0.395544    0.098035    0.006301    0.746148    
#> 214: 0.488931    0.953299    3.451170    0.394172    0.098391    0.006278    0.745632    
#> 215: 0.489238    0.954140    3.451329    0.390499    0.098164    0.006254    0.746249    
#> 216: 0.490826    0.952677    3.451736    0.392020    0.098896    0.006278    0.746925    
#> 217: 0.490611    0.952583    3.451879    0.390196    0.099121    0.006277    0.746491    
#> 218: 0.491391    0.953786    3.451658    0.391448    0.099333    0.006213    0.746290    
#> 219: 0.489916    0.953514    3.450638    0.391567    0.099321    0.006179    0.746435    
#> 220: 0.487565    0.954968    3.450235    0.393611    0.099076    0.006121    0.746174    
#> 221: 0.486153    0.954821    3.450016    0.393592    0.099541    0.006073    0.745538    
#> 222: 0.485833    0.954089    3.449703    0.393543    0.099945    0.006040    0.745790    
#> 223: 0.486649    0.953859    3.449395    0.394456    0.099386    0.006020    0.745801    
#> 224: 0.487098    0.954323    3.449642    0.394924    0.098998    0.006022    0.745431    
#> 225: 0.488271    0.951774    3.450346    0.394413    0.099054    0.006040    0.745596    
#> 226: 0.488052    0.949673    3.450657    0.393993    0.099194    0.005973    0.746219    
#> 227: 0.487874    0.949378    3.450981    0.393600    0.099215    0.005901    0.746362    
#> 228: 0.487046    0.950055    3.450770    0.393068    0.099586    0.005847    0.746074    
#> 229: 0.485658    0.950029    3.450367    0.394159    0.099709    0.005779    0.745550    
#> 230: 0.485451    0.949955    3.449995    0.394831    0.099399    0.005721    0.744981    
#> 231: 0.484383    0.950458    3.449924    0.394904    0.100038    0.005664    0.744683    
#> 232: 0.482828    0.951263    3.449684    0.396646    0.100162    0.005630    0.744234    
#> 233: 0.482093    0.950808    3.449546    0.397059    0.100038    0.005624    0.744734    
#> 234: 0.480907    0.950993    3.449467    0.396264    0.099734    0.005601    0.744730    
#> 235: 0.480246    0.951152    3.449383    0.394824    0.099246    0.005573    0.744963    
#> 236: 0.480390    0.951400    3.449193    0.396312    0.099133    0.005572    0.744945    
#> 237: 0.481428    0.951414    3.449327    0.398389    0.098920    0.005612    0.744757    
#> 238: 0.482075    0.951386    3.449387    0.399710    0.098770    0.005651    0.744821    
#> 239: 0.481109    0.952143    3.449208    0.399990    0.098945    0.005674    0.745364    
#> 240: 0.480586    0.952094    3.449051    0.398592    0.098568    0.005679    0.745240    
#> 241: 0.479765    0.951365    3.448910    0.397312    0.098290    0.005700    0.745095    
#> 242: 0.478733    0.950744    3.448703    0.396497    0.097835    0.005712    0.745727    
#> 243: 0.477111    0.951098    3.448584    0.396578    0.097271    0.005741    0.746172    
#> 244: 0.476169    0.951062    3.448496    0.398062    0.096650    0.005784    0.746383    
#> 245: 0.475610    0.950455    3.448631    0.398453    0.096390    0.005793    0.747003    
#> 246: 0.475641    0.950182    3.448722    0.398402    0.096082    0.005845    0.747046    
#> 247: 0.475596    0.949503    3.448875    0.397902    0.096018    0.005855    0.747086    
#> 248: 0.476000    0.949753    3.448903    0.398498    0.096172    0.005867    0.746797    
#> 249: 0.475607    0.950043    3.448892    0.398605    0.096176    0.005872    0.746420    
#> 250: 0.476127    0.950218    3.448878    0.398567    0.095900    0.005896    0.746441    
#> 251: 0.476103    0.951340    3.448813    0.400201    0.096189    0.005865    0.746591    
#> 252: 0.476567    0.951682    3.449044    0.400622    0.096152    0.005844    0.746631    
#> 253: 0.476752    0.952214    3.449132    0.401597    0.096344    0.005833    0.746512    
#> 254: 0.476986    0.952553    3.449246    0.401130    0.096611    0.005823    0.746524    
#> 255: 0.477328    0.952928    3.449393    0.401939    0.096421    0.005845    0.746414    
#> 256: 0.477759    0.953037    3.449379    0.402339    0.096261    0.005861    0.746146    
#> 257: 0.477144    0.953101    3.449225    0.402578    0.096354    0.005876    0.746116    
#> 258: 0.476306    0.953035    3.449126    0.401943    0.096657    0.005852    0.746227    
#> 259: 0.476213    0.953357    3.448957    0.403086    0.096826    0.005851    0.746259    
#> 260: 0.475936    0.953807    3.448939    0.403172    0.096851    0.005855    0.746224    
#> 261: 0.476170    0.953613    3.448934    0.403544    0.096855    0.005841    0.746212    
#> 262: 0.475967    0.953858    3.448904    0.402583    0.096876    0.005840    0.746223    
#> 263: 0.475385    0.954335    3.448712    0.402244    0.096782    0.005872    0.746297    
#> 264: 0.474819    0.954346    3.448553    0.402232    0.096691    0.005870    0.746593    
#> 265: 0.474649    0.954249    3.448494    0.402857    0.096332    0.005903    0.747090    
#> 266: 0.475226    0.954089    3.448540    0.403845    0.096342    0.005920    0.747212    
#> 267: 0.475337    0.953962    3.448674    0.405248    0.096440    0.005924    0.747271    
#> 268: 0.475563    0.953718    3.448586    0.405701    0.096549    0.005888    0.747200    
#> 269: 0.476555    0.953684    3.448596    0.406889    0.096911    0.005878    0.747134    
#> 270: 0.476975    0.953930    3.448477    0.407952    0.096963    0.005859    0.746846    
#> 271: 0.477264    0.954117    3.448368    0.409322    0.097052    0.005851    0.746706    
#> 272: 0.477564    0.954053    3.448327    0.409296    0.096880    0.005844    0.747198    
#> 273: 0.477809    0.954369    3.448333    0.409097    0.096681    0.005837    0.747560    
#> 274: 0.477356    0.954628    3.448399    0.408275    0.096640    0.005830    0.747708    
#> 275: 0.476947    0.954883    3.448312    0.407797    0.096462    0.005827    0.747724    
#> 276: 0.476741    0.955127    3.448473    0.407740    0.096393    0.005835    0.747818    
#> 277: 0.476761    0.955401    3.448612    0.407983    0.096533    0.005833    0.747775    
#> 278: 0.476653    0.955276    3.448676    0.407711    0.096385    0.005829    0.747978    
#> 279: 0.476110    0.955289    3.448650    0.407023    0.096544    0.005817    0.748069    
#> 280: 0.476091    0.955169    3.448760    0.406225    0.096579    0.005819    0.747987    
#> 281: 0.475623    0.955142    3.448815    0.405973    0.096330    0.005828    0.748141    
#> 282: 0.475295    0.955299    3.448940    0.405550    0.096225    0.005838    0.747995    
#> 283: 0.475357    0.955296    3.449056    0.405990    0.096212    0.005842    0.748095    
#> 284: 0.475053    0.955355    3.449014    0.405925    0.096071    0.005854    0.747748    
#> 285: 0.474639    0.955598    3.448801    0.406177    0.095995    0.005885    0.747703    
#> 286: 0.474034    0.955838    3.448529    0.406790    0.096041    0.005904    0.747588    
#> 287: 0.473161    0.956105    3.448229    0.406606    0.096040    0.005903    0.747601    
#> 288: 0.472779    0.956137    3.447940    0.405638    0.096108    0.005910    0.747925    
#> 289: 0.472881    0.956009    3.447887    0.404644    0.096195    0.005907    0.748077    
#> 290: 0.472888    0.955993    3.447810    0.404316    0.096417    0.005913    0.748028    
#> 291: 0.472870    0.956097    3.447777    0.404629    0.096528    0.005921    0.747851    
#> 292: 0.472846    0.956215    3.447670    0.405002    0.096731    0.005920    0.747942    
#> 293: 0.472747    0.956147    3.447615    0.405469    0.096715    0.005928    0.747982    
#> 294: 0.472379    0.956342    3.447538    0.405468    0.096690    0.005942    0.747932    
#> 295: 0.472209    0.956298    3.447472    0.405782    0.096765    0.005919    0.747974    
#> 296: 0.472147    0.956238    3.447444    0.405786    0.096833    0.005911    0.748228    
#> 297: 0.472183    0.956476    3.447417    0.405720    0.096863    0.005910    0.748366    
#> 298: 0.472555    0.956313    3.447408    0.406383    0.096857    0.005930    0.748328    
#> 299: 0.472844    0.956530    3.447425    0.407127    0.096923    0.005942    0.748033    
#> 300: 0.472739    0.956371    3.447502    0.407063    0.096827    0.005955    0.747788    
#> 301: 0.473072    0.956127    3.447491    0.406952    0.096604    0.005958    0.747869    
#> 302: 0.473381    0.956137    3.447517    0.406979    0.096388    0.005945    0.747861    
#> 303: 0.473658    0.956151    3.447541    0.406852    0.096368    0.005944    0.747800    
#> 304: 0.473739    0.956361    3.447488    0.407193    0.096352    0.005955    0.747950    
#> 305: 0.473469    0.956045    3.447450    0.407246    0.096375    0.005945    0.748025    
#> 306: 0.473250    0.955899    3.447316    0.407675    0.096293    0.005952    0.748113    
#> 307: 0.473320    0.955856    3.447261    0.407810    0.096293    0.005949    0.748297    
#> 308: 0.473292    0.955841    3.447225    0.409204    0.096294    0.005940    0.748475    
#> 309: 0.473633    0.955620    3.447180    0.409758    0.096223    0.005946    0.748419    
#> 310: 0.473759    0.955503    3.447111    0.409823    0.096076    0.005946    0.748311    
#> 311: 0.473591    0.955524    3.447014    0.410162    0.095938    0.005945    0.748135    
#> 312: 0.473370    0.955486    3.446878    0.410356    0.095864    0.005939    0.748267    
#> 313: 0.472808    0.955336    3.446790    0.410247    0.095864    0.005943    0.748459    
#> 314: 0.472470    0.955524    3.446689    0.410372    0.095801    0.005945    0.748429    
#> 315: 0.472197    0.955737    3.446631    0.410564    0.095899    0.005939    0.748362    
#> 316: 0.472311    0.955769    3.446652    0.410804    0.096101    0.005931    0.748542    
#> 317: 0.472642    0.955736    3.446708    0.411669    0.096255    0.005925    0.748454    
#> 318: 0.472747    0.955635    3.446708    0.411715    0.096372    0.005923    0.748340    
#> 319: 0.472856    0.955582    3.446708    0.411767    0.096437    0.005924    0.748282    
#> 320: 0.472692    0.955609    3.446596    0.411337    0.096701    0.005923    0.748320    
#> 321: 0.472335    0.955763    3.446403    0.411050    0.096867    0.005914    0.748308    
#> 322: 0.472131    0.955777    3.446157    0.411378    0.097010    0.005905    0.748363    
#> 323: 0.471379    0.955921    3.446039    0.411343    0.096983    0.005896    0.748541    
#> 324: 0.471340    0.955990    3.446013    0.411461    0.096910    0.005876    0.748436    
#> 325: 0.471517    0.956174    3.445997    0.411708    0.096875    0.005866    0.748240    
#> 326: 0.471638    0.956074    3.445913    0.411830    0.096800    0.005854    0.748214    
#> 327: 0.472067    0.956135    3.445944    0.412561    0.096762    0.005854    0.748281    
#> 328: 0.472355    0.956319    3.445929    0.413500    0.096736    0.005849    0.748264    
#> 329: 0.472587    0.956314    3.446018    0.414114    0.096790    0.005857    0.748279    
#> 330: 0.472538    0.956222    3.446016    0.414917    0.096740    0.005871    0.748305    
#> 331: 0.472437    0.956424    3.445920    0.415132    0.096683    0.005866    0.748225    
#> 332: 0.472540    0.956307    3.445914    0.415135    0.096867    0.005857    0.748249    
#> 333: 0.472155    0.956383    3.445785    0.414715    0.096773    0.005844    0.748163    
#> 334: 0.471782    0.956443    3.445570    0.415111    0.096728    0.005839    0.748092    
#> 335: 0.471371    0.956612    3.445400    0.414599    0.096590    0.005843    0.748069    
#> 336: 0.470933    0.956903    3.445301    0.414110    0.096531    0.005856    0.748006    
#> 337: 0.470534    0.957238    3.445198    0.413751    0.096412    0.005865    0.748041    
#> 338: 0.470465    0.957430    3.445161    0.413934    0.096487    0.005859    0.748098    
#> 339: 0.470221    0.957592    3.445063    0.413970    0.096586    0.005860    0.747959    
#> 340: 0.470090    0.957591    3.444986    0.413943    0.096846    0.005852    0.748018    
#> 341: 0.470148    0.957595    3.445004    0.414100    0.097144    0.005841    0.748071    
#> 342: 0.470003    0.957456    3.444987    0.414080    0.097241    0.005829    0.748089    
#> 343: 0.469533    0.957631    3.445007    0.413803    0.097293    0.005818    0.748218    
#> 344: 0.469524    0.957755    3.445014    0.413587    0.097387    0.005814    0.748380    
#> 345: 0.469572    0.957682    3.445095    0.413608    0.097648    0.005809    0.748390    
#> 346: 0.469599    0.957626    3.445095    0.414005    0.097754    0.005795    0.748303    
#> 347: 0.469728    0.957376    3.445032    0.413671    0.097933    0.005800    0.748331    
#> 348: 0.469395    0.957404    3.444971    0.413671    0.098130    0.005806    0.748384    
#> 349: 0.469307    0.957212    3.444938    0.413809    0.098147    0.005817    0.748368    
#> 350: 0.469098    0.957042    3.444878    0.413575    0.098273    0.005826    0.748308    
#> 351: 0.469158    0.957156    3.444817    0.413379    0.098327    0.005835    0.748345    
#> 352: 0.468806    0.957269    3.444772    0.413463    0.098380    0.005842    0.748464    
#> 353: 0.468687    0.957485    3.444659    0.413712    0.098319    0.005841    0.748457    
#> 354: 0.468700    0.957692    3.444485    0.413772    0.098218    0.005845    0.748385    
#> 355: 0.468885    0.957781    3.444367    0.414159    0.098230    0.005847    0.748348    
#> 356: 0.468965    0.957852    3.444376    0.414023    0.098378    0.005836    0.748470    
#> 357: 0.469033    0.957944    3.444366    0.414051    0.098488    0.005825    0.748441    
#> 358: 0.468900    0.957940    3.444374    0.414846    0.098597    0.005815    0.748295    
#> 359: 0.468475    0.958166    3.444316    0.414782    0.098649    0.005810    0.748247    
#> 360: 0.468357    0.958261    3.444202    0.414848    0.098696    0.005811    0.748105    
#> 361: 0.468055    0.958361    3.444076    0.414872    0.098668    0.005802    0.748042    
#> 362: 0.468020    0.958303    3.443994    0.415304    0.098634    0.005802    0.747907    
#> 363: 0.468018    0.958266    3.443963    0.415488    0.098615    0.005797    0.747793    
#> 364: 0.468099    0.958344    3.443919    0.415742    0.098629    0.005791    0.747756    
#> 365: 0.467883    0.958325    3.443901    0.415943    0.098709    0.005787    0.747755    
#> 366: 0.467967    0.958446    3.443880    0.416027    0.098720    0.005783    0.747695    
#> 367: 0.468249    0.958298    3.443954    0.416334    0.098644    0.005776    0.747701    
#> 368: 0.468400    0.958358    3.443961    0.416046    0.098632    0.005777    0.747628    
#> 369: 0.468406    0.958449    3.443968    0.415415    0.098576    0.005775    0.747634    
#> 370: 0.468461    0.958506    3.443984    0.415046    0.098569    0.005775    0.747653    
#> 371: 0.468369    0.958596    3.443902    0.414713    0.098536    0.005764    0.747672    
#> 372: 0.467956    0.958707    3.443850    0.414551    0.098542    0.005758    0.747732    
#> 373: 0.467434    0.958940    3.443743    0.414550    0.098503    0.005760    0.747759    
#> 374: 0.467612    0.958964    3.443686    0.414714    0.098413    0.005757    0.747746    
#> 375: 0.467469    0.958924    3.443677    0.415088    0.098369    0.005750    0.747763    
#> 376: 0.467426    0.959060    3.443663    0.415139    0.098279    0.005754    0.747680    
#> 377: 0.467293    0.959124    3.443649    0.415233    0.098242    0.005755    0.747577    
#> 378: 0.467232    0.959191    3.443607    0.415272    0.098287    0.005761    0.747545    
#> 379: 0.467047    0.959117    3.443580    0.415379    0.098243    0.005756    0.747531    
#> 380: 0.466971    0.959083    3.443564    0.415362    0.098391    0.005757    0.747460    
#> 381: 0.466783    0.959056    3.443526    0.415358    0.098376    0.005757    0.747444    
#> 382: 0.466698    0.959091    3.443511    0.415581    0.098420    0.005754    0.747306    
#> 383: 0.466936    0.958993    3.443550    0.415785    0.098400    0.005757    0.747283    
#> 384: 0.466845    0.958961    3.443557    0.416117    0.098371    0.005761    0.747186    
#> 385: 0.466965    0.958990    3.443553    0.416318    0.098394    0.005759    0.747117    
#> 386: 0.467076    0.959092    3.443615    0.416040    0.098395    0.005760    0.747230    
#> 387: 0.467125    0.959115    3.443649    0.415536    0.098366    0.005762    0.747260    
#> 388: 0.467248    0.959189    3.443645    0.415123    0.098309    0.005760    0.747339    
#> 389: 0.467242    0.959026    3.443655    0.414962    0.098314    0.005761    0.747447    
#> 390: 0.467052    0.959141    3.443585    0.414706    0.098348    0.005759    0.747433    
#> 391: 0.467115    0.959132    3.443561    0.414577    0.098289    0.005755    0.747378    
#> 392: 0.467249    0.959145    3.443569    0.414797    0.098275    0.005752    0.747377    
#> 393: 0.467312    0.959073    3.443541    0.414853    0.098262    0.005756    0.747378    
#> 394: 0.467361    0.959065    3.443510    0.414902    0.098309    0.005754    0.747312    
#> 395: 0.467672    0.959222    3.443514    0.415208    0.098294    0.005756    0.747323    
#> 396: 0.467763    0.959287    3.443529    0.415146    0.098296    0.005753    0.747308    
#> 397: 0.467707    0.959242    3.443542    0.414793    0.098234    0.005754    0.747387    
#> 398: 0.467816    0.959212    3.443527    0.414836    0.098165    0.005755    0.747494    
#> 399: 0.467877    0.959249    3.443573    0.414649    0.098163    0.005754    0.747571    
#> 400: 0.467994    0.959334    3.443617    0.414586    0.098187    0.005751    0.747620    
#> 401: 0.467904    0.959371    3.443610    0.414427    0.098265    0.005753    0.747606    
#> 402: 0.467953    0.959496    3.443587    0.414245    0.098386    0.005749    0.747622    
#> 403: 0.468174    0.959527    3.443570    0.414055    0.098530    0.005744    0.747603    
#> 404: 0.468003    0.959646    3.443470    0.414121    0.098590    0.005740    0.747579    
#> 405: 0.467758    0.959748    3.443419    0.414449    0.098561    0.005741    0.747580    
#> 406: 0.467665    0.959895    3.443376    0.414415    0.098554    0.005747    0.747622    
#> 407: 0.467703    0.959859    3.443403    0.414601    0.098633    0.005747    0.747633    
#> 408: 0.467952    0.959908    3.443393    0.414495    0.098677    0.005745    0.747620    
#> 409: 0.468038    0.959771    3.443374    0.414450    0.098653    0.005742    0.747520    
#> 410: 0.467958    0.959731    3.443340    0.414276    0.098678    0.005738    0.747507    
#> 411: 0.468108    0.959571    3.443316    0.413950    0.098585    0.005742    0.747598    
#> 412: 0.468170    0.959687    3.443281    0.413661    0.098693    0.005742    0.747595    
#> 413: 0.468125    0.959764    3.443230    0.413403    0.098698    0.005740    0.747756    
#> 414: 0.467952    0.959762    3.443222    0.413426    0.098664    0.005738    0.747717    
#> 415: 0.468030    0.959684    3.443213    0.413429    0.098680    0.005740    0.747760    
#> 416: 0.468138    0.959841    3.443208    0.413168    0.098732    0.005740    0.747792    
#> 417: 0.468035    0.959895    3.443185    0.412870    0.098751    0.005741    0.747837    
#> 418: 0.468012    0.959857    3.443181    0.412587    0.098788    0.005744    0.747845    
#> 419: 0.468153    0.959923    3.443174    0.412558    0.098793    0.005747    0.747912    
#> 420: 0.468251    0.959915    3.443179    0.412910    0.098817    0.005744    0.747848    
#> 421: 0.468232    0.959905    3.443226    0.413023    0.098854    0.005745    0.747851    
#> 422: 0.468423    0.959834    3.443236    0.412918    0.098803    0.005739    0.747939    
#> 423: 0.468559    0.959765    3.443261    0.413272    0.098839    0.005732    0.747919    
#> 424: 0.468405    0.959799    3.443219    0.413647    0.098798    0.005724    0.747896    
#> 425: 0.468302    0.959851    3.443200    0.413695    0.098655    0.005721    0.747993    
#> 426: 0.468278    0.960018    3.443167    0.413846    0.098659    0.005713    0.747999    
#> 427: 0.468067    0.959994    3.443151    0.413729    0.098621    0.005706    0.748020    
#> 428: 0.467919    0.960093    3.443098    0.413862    0.098706    0.005697    0.748011    
#> 429: 0.467915    0.960167    3.443083    0.413774    0.098779    0.005687    0.748029    
#> 430: 0.467786    0.960303    3.443018    0.413508    0.098807    0.005683    0.748065    
#> 431: 0.467817    0.960301    3.442973    0.413437    0.098764    0.005680    0.747979    
#> 432: 0.467767    0.960316    3.442966    0.413594    0.098704    0.005679    0.747851    
#> 433: 0.467804    0.960429    3.442970    0.413417    0.098628    0.005684    0.747853    
#> 434: 0.467660    0.960405    3.442982    0.413169    0.098594    0.005685    0.747852    
#> 435: 0.467742    0.960422    3.442971    0.413253    0.098545    0.005689    0.747833    
#> 436: 0.467759    0.960381    3.442963    0.413189    0.098502    0.005688    0.747904    
#> 437: 0.467910    0.960306    3.442953    0.413480    0.098498    0.005688    0.747912    
#> 438: 0.467826    0.960497    3.442937    0.413535    0.098414    0.005682    0.747845    
#> 439: 0.467802    0.960588    3.442888    0.413649    0.098313    0.005675    0.747852    
#> 440: 0.467537    0.960600    3.442893    0.413756    0.098278    0.005666    0.747897    
#> 441: 0.467482    0.960509    3.442921    0.414178    0.098290    0.005665    0.747815    
#> 442: 0.467395    0.960515    3.442947    0.414463    0.098285    0.005669    0.747751    
#> 443: 0.467402    0.960594    3.442950    0.414364    0.098225    0.005673    0.747767    
#> 444: 0.467412    0.960715    3.442906    0.414441    0.098206    0.005679    0.747712    
#> 445: 0.467335    0.960732    3.442861    0.414546    0.098219    0.005684    0.747697    
#> 446: 0.467232    0.960799    3.442817    0.414384    0.098224    0.005685    0.747647    
#> 447: 0.467361    0.960856    3.442776    0.414609    0.098204    0.005688    0.747562    
#> 448: 0.467381    0.960876    3.442766    0.414559    0.098150    0.005693    0.747493    
#> 449: 0.467514    0.960967    3.442857    0.414379    0.098163    0.005695    0.747569    
#> 450: 0.467516    0.960971    3.442916    0.414002    0.098211    0.005695    0.747609    
#> 451: 0.467513    0.961009    3.442923    0.413860    0.098311    0.005688    0.747684    
#> 452: 0.467515    0.960975    3.442908    0.413505    0.098375    0.005683    0.747726    
#> 453: 0.467415    0.961060    3.442851    0.413518    0.098363    0.005675    0.747690    
#> 454: 0.467455    0.961205    3.442808    0.413833    0.098323    0.005664    0.747735    
#> 455: 0.467591    0.961213    3.442762    0.413992    0.098349    0.005661    0.747730    
#> 456: 0.467526    0.961222    3.442750    0.414071    0.098345    0.005657    0.747705    
#> 457: 0.467508    0.961206    3.442767    0.414116    0.098342    0.005650    0.747662    
#> 458: 0.467524    0.961329    3.442773    0.414193    0.098357    0.005645    0.747621    
#> 459: 0.467510    0.961473    3.442780    0.414047    0.098445    0.005642    0.747617    
#> 460: 0.467408    0.961485    3.442766    0.413953    0.098497    0.005641    0.747633    
#> 461: 0.467450    0.961504    3.442789    0.413823    0.098498    0.005641    0.747669    
#> 462: 0.467403    0.961558    3.442822    0.413817    0.098485    0.005640    0.747720    
#> 463: 0.467492    0.961616    3.442849    0.413862    0.098487    0.005638    0.747753    
#> 464: 0.467628    0.961640    3.442869    0.414232    0.098455    0.005641    0.747714    
#> 465: 0.467820    0.961509    3.442866    0.414640    0.098448    0.005635    0.747740    
#> 466: 0.467894    0.961547    3.442853    0.414529    0.098440    0.005630    0.747724    
#> 467: 0.467821    0.961496    3.442868    0.414630    0.098443    0.005627    0.747657    
#> 468: 0.467897    0.961454    3.442901    0.414677    0.098407    0.005629    0.747687    
#> 469: 0.467966    0.961449    3.442953    0.414345    0.098336    0.005626    0.747742    
#> 470: 0.467886    0.961416    3.442935    0.414094    0.098290    0.005625    0.747743    
#> 471: 0.467801    0.961388    3.442909    0.413757    0.098269    0.005618    0.747746    
#> 472: 0.467671    0.961413    3.442906    0.413755    0.098273    0.005613    0.747744    
#> 473: 0.467594    0.961392    3.442890    0.413785    0.098300    0.005610    0.747682    
#> 474: 0.467464    0.961410    3.442849    0.413729    0.098339    0.005614    0.747706    
#> 475: 0.467429    0.961507    3.442819    0.413890    0.098267    0.005624    0.747716    
#> 476: 0.467318    0.961530    3.442762    0.414066    0.098235    0.005637    0.747760    
#> 477: 0.467250    0.961593    3.442738    0.414170    0.098168    0.005644    0.747668    
#> 478: 0.467085    0.961648    3.442715    0.414296    0.098150    0.005647    0.747601    
#> 479: 0.467050    0.961717    3.442689    0.414453    0.098098    0.005652    0.747597    
#> 480: 0.466978    0.961771    3.442706    0.414665    0.098075    0.005653    0.747701    
#> 481: 0.466942    0.961761    3.442721    0.414791    0.098095    0.005655    0.747739    
#> 482: 0.467047    0.961783    3.442726    0.414999    0.098139    0.005655    0.747822    
#> 483: 0.466909    0.961716    3.442729    0.415096    0.098172    0.005658    0.747870    
#> 484: 0.466844    0.961596    3.442746    0.415135    0.098204    0.005659    0.747877    
#> 485: 0.466924    0.961579    3.442773    0.415181    0.098196    0.005658    0.747985    
#> 486: 0.466950    0.961635    3.442754    0.415315    0.098199    0.005654    0.747973    
#> 487: 0.467184    0.961736    3.442817    0.415460    0.098189    0.005651    0.747948    
#> 488: 0.467215    0.961831    3.442870    0.415281    0.098223    0.005646    0.747977    
#> 489: 0.467294    0.961844    3.442902    0.415264    0.098304    0.005642    0.748027    
#> 490: 0.467419    0.961832    3.442951    0.415148    0.098342    0.005640    0.748087    
#> 491: 0.467493    0.961851    3.443015    0.414984    0.098314    0.005645    0.748079    
#> 492: 0.467474    0.961932    3.443048    0.414778    0.098299    0.005651    0.748107    
#> 493: 0.467419    0.961919    3.443065    0.414681    0.098244    0.005652    0.748128    
#> 494: 0.467545    0.961971    3.443054    0.414504    0.098199    0.005648    0.748098    
#> 495: 0.467560    0.961990    3.443052    0.414218    0.098217    0.005647    0.748057    
#> 496: 0.467539    0.962002    3.443030    0.413959    0.098138    0.005643    0.748109    
#> 497: 0.467412    0.962010    3.443011    0.413833    0.098175    0.005643    0.748117    
#> 498: 0.467310    0.961987    3.442986    0.413627    0.098145    0.005642    0.748102    
#> 499: 0.467312    0.961989    3.442977    0.413368    0.098210    0.005634    0.748086    
#> 500: 0.467319    0.961955    3.442982    0.413112    0.098198    0.005630    0.748038    
#> [====|====|=>-------------------------------------]  25%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.276862    0.963110    3.442772    0.392401    0.082618    0.016800    1.625740    
#> 002: 0.174057    0.975732    3.480638    0.372781    0.078487    0.015960    1.025434    
#> 003: 0.198178    0.939341    3.490411    0.354142    0.082780    0.015162    0.765062    
#> 004: 0.171537    0.972482    3.481355    0.336435    0.078641    0.014404    0.669190    
#> 005: 0.219392    0.988445    3.471006    0.319613    0.074709    0.013684    0.664856    
#> 006: 0.151168    0.991140    3.462856    0.303633    0.070973    0.013000    0.613562    
#> 007: 0.105070    1.002967    3.445105    0.288451    0.067425    0.012557    0.613010    
#> 008: 0.112484    1.021849    3.437652    0.274029    0.064053    0.017968    0.596359    
#> 009: 0.110905    0.995126    3.443777    0.260327    0.060851    0.017777    0.611620    
#> 010: 0.105940    1.003418    3.437615    0.247311    0.057808    0.017403    0.616221    
#> 011: 0.134192    1.033418    3.438937    0.239450    0.062559    0.018668    0.614083    
#> 012: 0.146196    1.025657    3.439932    0.227477    0.059431    0.018895    0.606848    
#> 013: 0.101538    1.020971    3.435755    0.216103    0.056460    0.022010    0.589733    
#> 014: 0.138344    1.008866    3.447941    0.205298    0.053637    0.020910    0.593488    
#> 015: 0.127402    1.021197    3.443022    0.195033    0.059969    0.019864    0.573729    
#> 016: 0.127523    1.035848    3.447540    0.209938    0.056970    0.022412    0.578746    
#> 017: 0.127509    1.036092    3.448395    0.199441    0.056758    0.021291    0.583418    
#> 018: 0.110811    1.023053    3.433620    0.189469    0.063778    0.020227    0.587597    
#> 019: 0.074384    1.023423    3.432221    0.198764    0.060589    0.019731    0.597311    
#> 020: 0.107420    1.025891    3.447188    0.192145    0.057559    0.020824    0.601611    
#> 021: 0.105403    1.023217    3.443901    0.182537    0.054681    0.019783    0.596330    
#> 022: 0.093528    1.031057    3.452809    0.173411    0.051947    0.019828    0.589619    
#> 023: 0.174224    1.019513    3.463348    0.172872    0.049350    0.018837    0.614282    
#> 024: 0.184804    1.032817    3.465642    0.188097    0.046882    0.017895    0.604322    
#> 025: 0.164740    0.999970    3.453476    0.184618    0.044538    0.023714    0.619575    
#> 026: 0.123558    1.018584    3.442703    0.181860    0.044128    0.027765    0.609966    
#> 027: 0.090941    1.039848    3.424300    0.194204    0.041921    0.029617    0.611991    
#> 028: 0.128623    1.031234    3.444826    0.184493    0.039825    0.031518    0.589847    
#> 029: 0.119075    1.038180    3.435605    0.175269    0.045727    0.029943    0.608945    
#> 030: 0.126320    1.048379    3.423082    0.166505    0.053658    0.028445    0.601595    
#> 031: 0.107906    1.027947    3.441379    0.169474    0.051243    0.027023    0.591406    
#> 032: 0.134814    1.012309    3.459496    0.164869    0.048681    0.025672    0.591637    
#> 033: 0.124950    1.030437    3.446104    0.177984    0.046247    0.024388    0.612061    
#> 034: 0.156912    1.026494    3.457252    0.210093    0.049883    0.023169    0.606154    
#> 035: 0.122445    1.008673    3.440112    0.209530    0.047389    0.022011    0.587197    
#> 036: 0.128442    1.021564    3.440071    0.199053    0.048561    0.020910    0.599089    
#> 037: 0.115558    1.026829    3.450383    0.212235    0.046133    0.019865    0.606036    
#> 038: 0.136455    1.029273    3.438616    0.201623    0.043826    0.020307    0.614160    
#> 039: 0.122794    1.024961    3.451532    0.191542    0.043479    0.025086    0.583717    
#> 040: 0.138272    1.022269    3.466645    0.182667    0.041305    0.024466    0.606747    
#> 041: 0.107487    1.032849    3.443938    0.173534    0.045064    0.023998    0.609379    
#> 042: 0.157864    1.022053    3.451493    0.173820    0.047335    0.025634    0.586127    
#> 043: 0.103606    1.028609    3.435704    0.165129    0.044969    0.024352    0.604484    
#> 044: 0.093662    1.020747    3.445047    0.156872    0.042720    0.023135    0.601391    
#> 045: 0.109176    1.035552    3.434089    0.177080    0.045321    0.022863    0.585993    
#> 046: 0.105178    1.034279    3.438693    0.173947    0.045784    0.021720    0.592758    
#> 047: 0.098902    1.045571    3.433867    0.195962    0.043495    0.020634    0.585139    
#> 048: 0.105432    1.035761    3.432011    0.211963    0.041320    0.019602    0.589712    
#> 049: 0.108162    1.031924    3.443287    0.201364    0.046473    0.019893    0.590337    
#> 050: 0.153826    1.012300    3.459733    0.191296    0.051593    0.022288    0.592897    
#> 051: 0.158467    1.014435    3.456154    0.188686    0.049014    0.025432    0.597764    
#> 052: 0.158569    1.022518    3.450869    0.184388    0.046563    0.024160    0.589307    
#> 053: 0.128639    1.004133    3.448742    0.237435    0.044235    0.022952    0.596631    
#> 054: 0.134245    1.019406    3.449831    0.225563    0.060188    0.021804    0.599732    
#> 055: 0.127338    1.026467    3.450115    0.214285    0.057306    0.020714    0.609455    
#> 056: 0.149492    1.001354    3.452448    0.203571    0.062731    0.019679    0.595272    
#> 057: 0.152995    1.004235    3.463343    0.193392    0.059595    0.019251    0.609788    
#> 058: 0.161448    1.016687    3.450399    0.183723    0.056615    0.018289    0.608914    
#> 059: 0.128073    0.991429    3.448618    0.174537    0.053784    0.017374    0.610841    
#> 060: 0.106384    1.022344    3.441153    0.171151    0.051095    0.017130    0.614776    
#> 061: 0.164492    1.022803    3.457175    0.210041    0.059135    0.018063    0.608573    
#> 062: 0.173510    1.042124    3.465497    0.208619    0.077798    0.019993    0.612660    
#> 063: 0.168574    1.009043    3.454135    0.198709    0.073908    0.025499    0.607549    
#> 064: 0.143567    1.021045    3.460460    0.188773    0.070213    0.024224    0.618644    
#> 065: 0.137684    0.999904    3.443480    0.186399    0.066702    0.023013    0.612281    
#> 066: 0.151939    1.015803    3.455334    0.195551    0.063367    0.021941    0.589391    
#> 067: 0.147744    1.004389    3.450619    0.198729    0.060198    0.026801    0.584927    
#> 068: 0.098073    1.026711    3.449803    0.194158    0.057189    0.027099    0.590862    
#> 069: 0.129704    1.033870    3.445234    0.187574    0.054329    0.025744    0.603386    
#> 070: 0.126789    1.022904    3.450890    0.195000    0.054667    0.024457    0.610259    
#> 071: 0.122049    1.024517    3.442149    0.188792    0.059380    0.023234    0.601665    
#> 072: 0.108509    1.003036    3.460289    0.179353    0.057690    0.022072    0.605092    
#> 073: 0.169665    1.011445    3.478357    0.170385    0.054805    0.020969    0.605444    
#> 074: 0.147694    1.005078    3.456352    0.161866    0.052776    0.019920    0.596721    
#> 075: 0.162950    0.985113    3.484240    0.153772    0.053658    0.018924    0.598775    
#> 076: 0.159852    1.013479    3.474825    0.167887    0.053304    0.019675    0.591357    
#> 077: 0.197826    0.997025    3.478403    0.176955    0.050639    0.018691    0.605771    
#> 078: 0.188967    0.987844    3.479528    0.174235    0.048107    0.019322    0.587422    
#> 079: 0.161901    0.997593    3.468231    0.165524    0.045702    0.022659    0.603880    
#> 080: 0.174933    0.987626    3.475068    0.192399    0.050034    0.023814    0.610563    
#> 081: 0.158574    0.995975    3.464982    0.182779    0.047532    0.024306    0.594937    
#> 082: 0.147685    0.994954    3.467895    0.173640    0.045156    0.023090    0.601741    
#> 083: 0.163361    1.028817    3.462020    0.180532    0.046339    0.021936    0.588290    
#> 084: 0.161810    1.026841    3.455847    0.175102    0.044022    0.020839    0.598773    
#> 085: 0.134360    1.037199    3.459311    0.179221    0.047284    0.019797    0.611265    
#> 086: 0.143728    1.031764    3.455556    0.175617    0.056058    0.018807    0.575183    
#> 087: 0.128328    1.033802    3.441014    0.179702    0.063543    0.018102    0.586172    
#> 088: 0.080074    1.034618    3.438786    0.222213    0.060366    0.021446    0.593268    
#> 089: 0.082394    1.036494    3.417989    0.219452    0.059473    0.020373    0.597621    
#> 090: 0.093266    1.032683    3.419898    0.215909    0.056499    0.019563    0.601958    
#> 091: 0.057473    1.026925    3.408999    0.205113    0.053675    0.018585    0.603742    
#> 092: 0.084674    1.038165    3.413953    0.199366    0.055566    0.019400    0.620616    
#> 093: 0.031585    1.044028    3.403819    0.189397    0.063887    0.018430    0.638479    
#> 094: 0.035929    1.044442    3.407989    0.189580    0.061040    0.018301    0.630566    
#> 095: 0.058635    1.075100    3.410316    0.211058    0.061597    0.017386    0.613112    
#> 096: 0.055027    1.054590    3.417940    0.216060    0.058517    0.020389    0.611169    
#> 097: 0.102731    1.047048    3.420594    0.205257    0.055591    0.019370    0.600505    
#> 098: 0.063603    1.060328    3.424066    0.199993    0.052812    0.020664    0.605269    
#> 099: 0.062249    1.039801    3.411419    0.189994    0.050171    0.019631    0.611386    
#> 100: 0.072400    1.046359    3.416128    0.180494    0.047662    0.021651    0.594822    
#> 101: 0.076104    1.034169    3.413058    0.171469    0.060192    0.021023    0.608500    
#> 102: 0.058713    1.046437    3.409202    0.178860    0.060594    0.021927    0.600833    
#> 103: 0.069705    1.047755    3.425562    0.190227    0.057564    0.020975    0.593226    
#> 104: 0.104906    1.039353    3.439109    0.200126    0.054686    0.020246    0.599583    
#> 105: 0.150826    1.038220    3.441209    0.239991    0.064456    0.020615    0.594768    
#> 106: 0.126762    1.031240    3.436717    0.227992    0.065033    0.019584    0.590642    
#> 107: 0.138607    1.034160    3.445399    0.216592    0.061781    0.018605    0.596343    
#> 108: 0.138749    1.025227    3.451122    0.205763    0.058692    0.017912    0.611589    
#> 109: 0.131832    1.022363    3.459938    0.195475    0.055757    0.017017    0.607968    
#> 110: 0.142670    1.024129    3.444250    0.185701    0.063931    0.016166    0.604288    
#> 111: 0.143383    1.018020    3.450783    0.176416    0.061669    0.015358    0.600043    
#> 112: 0.117480    1.014724    3.438670    0.167595    0.058586    0.014590    0.616459    
#> 113: 0.109154    1.034649    3.437561    0.183271    0.060015    0.013860    0.581860    
#> 114: 0.119613    1.031050    3.443476    0.174108    0.063166    0.014547    0.595508    
#> 115: 0.133048    1.005022    3.441529    0.165402    0.060007    0.013819    0.613272    
#> 116: 0.107342    1.029169    3.434435    0.163905    0.058707    0.013128    0.617942    
#> 117: 0.109298    1.045720    3.435498    0.168967    0.071233    0.012472    0.598598    
#> 118: 0.121305    1.023396    3.443205    0.162176    0.067672    0.014240    0.594883    
#> 119: 0.107750    1.045360    3.443002    0.173199    0.064288    0.020661    0.594476    
#> 120: 0.086279    1.028982    3.442981    0.164539    0.061074    0.019628    0.587745    
#> 121: 0.097125    1.065352    3.429990    0.164089    0.058020    0.019081    0.592719    
#> 122: 0.112959    1.047729    3.433252    0.212649    0.063023    0.021034    0.574857    
#> 123: 0.121483    1.059685    3.424661    0.241440    0.059872    0.019982    0.582440    
#> 124: 0.127533    1.067299    3.431188    0.249090    0.056878    0.023839    0.594193    
#> 125: 0.112753    1.049932    3.418171    0.236636    0.054034    0.022647    0.586255    
#> 126: 0.111540    1.028787    3.424599    0.224804    0.051333    0.023764    0.603800    
#> 127: 0.120283    1.026509    3.429438    0.218907    0.048766    0.022576    0.612558    
#> 128: 0.080504    1.037721    3.443182    0.207962    0.051391    0.021447    0.611931    
#> 129: 0.125606    1.046021    3.442776    0.197564    0.061564    0.020375    0.612069    
#> 130: 0.153182    1.053429    3.445169    0.191402    0.058486    0.021184    0.584110    
#> 131: 0.121243    1.027043    3.435998    0.188946    0.055562    0.020125    0.581846    
#> 132: 0.101513    1.032368    3.424566    0.179498    0.061015    0.019118    0.592625    
#> 133: 0.105657    1.031168    3.433439    0.193043    0.057964    0.018974    0.583658    
#> 134: 0.124261    1.021238    3.442744    0.183391    0.055066    0.021365    0.596752    
#> 135: 0.092637    1.041026    3.433615    0.175997    0.052313    0.023162    0.584239    
#> 136: 0.112561    1.020020    3.443755    0.194350    0.049697    0.022003    0.583092    
#> 137: 0.141725    1.002611    3.452136    0.193804    0.047212    0.020903    0.604420    
#> 138: 0.119187    1.019056    3.445614    0.197870    0.044852    0.021538    0.591103    
#> 139: 0.128448    1.027503    3.451080    0.192968    0.042609    0.027428    0.606851    
#> 140: 0.122713    1.029583    3.449463    0.183319    0.047368    0.026056    0.618277    
#> 141: 0.114742    1.010101    3.456180    0.174153    0.046524    0.024754    0.597713    
#> 142: 0.140235    1.014383    3.466497    0.165446    0.054861    0.023516    0.600803    
#> 143: 0.130404    1.008319    3.460307    0.157173    0.052118    0.022340    0.613976    
#> 144: 0.169552    1.023030    3.468901    0.169246    0.049512    0.021223    0.592163    
#> 145: 0.137393    1.032123    3.443288    0.186516    0.047172    0.020162    0.579630    
#> 146: 0.143364    1.019175    3.457945    0.197901    0.044813    0.019478    0.599970    
#> 147: 0.109880    1.020131    3.445688    0.224383    0.046835    0.019723    0.577426    
#> 148: 0.099368    1.020417    3.453487    0.213163    0.050343    0.018889    0.581398    
#> 149: 0.123524    1.029583    3.440442    0.202505    0.070870    0.017944    0.595503    
#> 150: 0.120505    1.027537    3.428942    0.192380    0.070859    0.017047    0.591556    
#> 151: 0.083878    1.059563    3.425986    0.182761    0.067316    0.016195    0.598628    
#> 152: 0.084230    1.050303    3.424222    0.206743    0.054884    0.018592    0.588834    
#> 153: 0.101280    1.051466    3.439955    0.190974    0.052070    0.017224    0.586933    
#> 154: 0.151637    1.044455    3.440024    0.203282    0.045091    0.015668    0.603054    
#> 155: 0.119157    1.025371    3.439492    0.157911    0.057116    0.016627    0.582547    
#> 156: 0.112679    1.034561    3.443272    0.162669    0.063495    0.015887    0.606149    
#> 157: 0.100389    1.026060    3.448592    0.142282    0.057187    0.014611    0.614455    
#> 158: 0.124420    1.039142    3.445171    0.151249    0.060859    0.016611    0.587527    
#> 159: 0.094063    1.033500    3.446070    0.176566    0.053752    0.014880    0.583867    
#> 160: 0.126772    1.003028    3.456835    0.180103    0.064185    0.014891    0.602620    
#> 161: 0.134720    1.018830    3.449758    0.181509    0.056860    0.016142    0.604474    
#> 162: 0.141729    1.036199    3.447031    0.146985    0.059876    0.014010    0.607841    
#> 163: 0.143431    1.030871    3.435869    0.165245    0.052889    0.013528    0.611273    
#> 164: 0.125423    1.034391    3.440240    0.184468    0.049571    0.014914    0.609692    
#> 165: 0.163617    1.013338    3.456348    0.176806    0.047249    0.017420    0.597747    
#> 166: 0.165772    1.029115    3.461729    0.165464    0.044973    0.016999    0.598343    
#> 167: 0.174697    1.037744    3.463935    0.188807    0.045779    0.016853    0.615433    
#> 168: 0.149295    1.038393    3.456320    0.195738    0.049006    0.020246    0.595921    
#> 169: 0.140292    1.053575    3.447932    0.167910    0.056693    0.018819    0.610397    
#> 170: 0.131264    1.029831    3.450881    0.196988    0.048494    0.016484    0.591205    
#> 171: 0.132628    1.019872    3.451892    0.172584    0.053849    0.018869    0.573159    
#> 172: 0.129310    1.030909    3.448486    0.181990    0.044909    0.020536    0.580103    
#> 173: 0.130517    1.039460    3.433515    0.206074    0.050611    0.023425    0.564128    
#> 174: 0.113784    1.027765    3.445772    0.197685    0.047303    0.023800    0.580934    
#> 175: 0.120164    1.028995    3.434487    0.187185    0.043810    0.022713    0.585232    
#> 176: 0.086274    1.038540    3.437341    0.205668    0.051357    0.021743    0.588218    
#> 177: 0.106228    1.026906    3.444738    0.152969    0.046840    0.021654    0.596751    
#> 178: 0.126398    1.011981    3.452559    0.161205    0.046962    0.019625    0.594454    
#> 179: 0.142304    1.016613    3.449374    0.171852    0.055034    0.019133    0.602866    
#> 180: 0.146028    1.018224    3.455990    0.171652    0.049199    0.020404    0.603001    
#> 181: 0.143157    1.005671    3.442007    0.177962    0.052825    0.023105    0.604513    
#> 182: 0.123803    1.014747    3.434604    0.176048    0.046649    0.024015    0.617675    
#> 183: 0.116027    1.048564    3.436334    0.175548    0.048699    0.022906    0.630618    
#> 184: 0.136367    1.022398    3.444361    0.164094    0.043196    0.021846    0.610049    
#> 185: 0.118849    1.015596    3.445051    0.175137    0.039653    0.024736    0.596936    
#> 186: 0.113584    1.019879    3.451785    0.152580    0.045660    0.027708    0.588348    
#> 187: 0.123952    1.006163    3.454518    0.155152    0.042919    0.028224    0.601895    
#> 188: 0.095660    1.036325    3.449795    0.158864    0.044609    0.025130    0.616759    
#> 189: 0.148599    1.043674    3.451182    0.193759    0.041277    0.025470    0.597828    
#> 190: 0.122308    1.026047    3.453477    0.174117    0.034931    0.022755    0.594787    
#> 191: 0.122678    1.028618    3.446268    0.180394    0.039008    0.019131    0.587107    
#> 192: 0.114915    1.029738    3.446538    0.201724    0.040447    0.018402    0.589867    
#> 193: 0.125898    1.036699    3.447922    0.188204    0.055112    0.017749    0.575836    
#> 194: 0.121707    1.036957    3.432158    0.197352    0.055272    0.014479    0.598202    
#> 195: 0.094497    1.025253    3.435811    0.165890    0.047266    0.014833    0.589330    
#> 196: 0.129660    1.036811    3.440919    0.145869    0.051003    0.014437    0.609575    
#> 197: 0.101738    1.043454    3.460050    0.169830    0.047147    0.018052    0.606498    
#> 198: 0.136899    1.023855    3.449010    0.171453    0.042434    0.018712    0.582141    
#> 199: 0.143090    1.013835    3.449867    0.203124    0.056304    0.017265    0.585093    
#> 200: 0.123291    1.027339    3.444656    0.177022    0.066306    0.018699    0.594920    
#> 201: 0.111945    1.028089    3.446718    0.169100    0.057903    0.022876    0.591929    
#> 202: 0.106388    1.031730    3.444162    0.181757    0.051752    0.023264    0.589299    
#> 203: 0.117173    1.029571    3.442624    0.181512    0.049463    0.022940    0.588749    
#> 204: 0.124454    1.021801    3.447464    0.180809    0.048150    0.022871    0.591027    
#> 205: 0.126235    1.024935    3.450247    0.177670    0.047333    0.022553    0.596111    
#> 206: 0.127087    1.028759    3.450610    0.179875    0.047808    0.022540    0.596561    
#> 207: 0.123541    1.028235    3.449804    0.178281    0.047885    0.022487    0.597143    
#> 208: 0.120949    1.028877    3.448931    0.178303    0.047198    0.022521    0.596746    
#> 209: 0.122769    1.028017    3.449189    0.177731    0.047233    0.022739    0.595645    
#> 210: 0.124205    1.027550    3.449601    0.176713    0.047423    0.022648    0.594719    
#> 211: 0.126480    1.027702    3.450594    0.175527    0.047095    0.022750    0.593517    
#> 212: 0.128383    1.026534    3.451279    0.176590    0.046990    0.022728    0.593754    
#> 213: 0.129097    1.024231    3.452010    0.179200    0.046662    0.022857    0.595430    
#> 214: 0.130653    1.023135    3.452573    0.180953    0.046857    0.023057    0.594880    
#> 215: 0.131361    1.022349    3.453149    0.184034    0.046505    0.023033    0.595450    
#> 216: 0.133966    1.021213    3.453838    0.184811    0.046723    0.022746    0.596505    
#> 217: 0.134020    1.021551    3.453383    0.185967    0.047002    0.022329    0.596536    
#> 218: 0.134377    1.021933    3.453015    0.188006    0.046830    0.022067    0.596533    
#> 219: 0.132987    1.021771    3.451886    0.188468    0.046731    0.021862    0.596589    
#> 220: 0.131052    1.022253    3.451443    0.188148    0.046606    0.021815    0.596331    
#> 221: 0.129631    1.022388    3.450799    0.188207    0.046443    0.021693    0.596019    
#> 222: 0.128872    1.022647    3.450040    0.187976    0.046709    0.021449    0.595895    
#> 223: 0.128749    1.022813    3.449839    0.187643    0.046734    0.021389    0.594904    
#> 224: 0.130163    1.023498    3.449952    0.187773    0.046676    0.021489    0.594986    
#> 225: 0.130272    1.022488    3.450170    0.187382    0.046374    0.021469    0.595127    
#> 226: 0.130855    1.022080    3.450292    0.187030    0.046144    0.021397    0.594958    
#> 227: 0.130637    1.022100    3.450047    0.185583    0.046147    0.021320    0.595376    
#> 228: 0.129586    1.023733    3.449361    0.185143    0.046735    0.021322    0.595733    
#> 229: 0.128801    1.024100    3.448954    0.184660    0.046903    0.021218    0.595546    
#> 230: 0.128908    1.024306    3.448509    0.184949    0.047037    0.021117    0.595359    
#> 231: 0.128969    1.024682    3.448717    0.184970    0.047098    0.021210    0.595503    
#> 232: 0.128639    1.024630    3.448797    0.185367    0.047154    0.021225    0.595228    
#> 233: 0.128542    1.024233    3.448870    0.184920    0.047144    0.021128    0.595325    
#> 234: 0.127494    1.023729    3.448778    0.183890    0.047156    0.021200    0.595359    
#> 235: 0.127301    1.023354    3.448769    0.182455    0.046950    0.021087    0.595830    
#> 236: 0.126742    1.023778    3.448464    0.181331    0.047007    0.020960    0.596119    
#> 237: 0.126648    1.023732    3.448399    0.181127    0.047223    0.020864    0.596321    
#> 238: 0.127500    1.024056    3.448272    0.181249    0.047429    0.020782    0.596906    
#> 239: 0.126737    1.024524    3.447738    0.180771    0.047578    0.020748    0.596969    
#> 240: 0.126543    1.024335    3.447744    0.179840    0.047775    0.020789    0.597180    
#> 241: 0.126551    1.023571    3.447812    0.179125    0.047781    0.020786    0.597605    
#> 242: 0.126722    1.023195    3.447733    0.179097    0.047609    0.020773    0.598162    
#> 243: 0.126315    1.023428    3.447711    0.179250    0.047669    0.020740    0.598632    
#> 244: 0.126382    1.023270    3.447770    0.179903    0.047826    0.020669    0.598885    
#> 245: 0.127408    1.022530    3.448115    0.180494    0.048179    0.020536    0.599493    
#> 246: 0.127931    1.022417    3.448252    0.180271    0.048391    0.020429    0.599665    
#> 247: 0.128332    1.022330    3.448350    0.180074    0.048692    0.020300    0.599860    
#> 248: 0.128084    1.022576    3.448203    0.179956    0.048700    0.020249    0.599954    
#> 249: 0.127799    1.022696    3.448360    0.179445    0.048666    0.020257    0.599948    
#> 250: 0.128628    1.022677    3.448543    0.179392    0.048642    0.020319    0.599831    
#> 251: 0.128796    1.022739    3.448773    0.179386    0.048642    0.020359    0.599871    
#> 252: 0.128960    1.022688    3.449098    0.178991    0.048600    0.020396    0.599838    
#> 253: 0.129056    1.022870    3.449348    0.178864    0.048660    0.020399    0.599849    
#> 254: 0.129245    1.023214    3.449285    0.178788    0.048860    0.020397    0.599710    
#> 255: 0.128772    1.023497    3.449156    0.179117    0.048910    0.020451    0.599314    
#> 256: 0.128667    1.023628    3.448991    0.179256    0.048960    0.020431    0.599154    
#> 257: 0.128527    1.023464    3.448950    0.179073    0.048914    0.020354    0.599279    
#> 258: 0.128211    1.023313    3.448828    0.179973    0.048838    0.020304    0.599380    
#> 259: 0.127837    1.023392    3.448562    0.180395    0.048976    0.020329    0.599462    
#> 260: 0.127615    1.023583    3.448593    0.180834    0.048988    0.020341    0.599157    
#> 261: 0.127733    1.023535    3.448789    0.181190    0.049050    0.020353    0.599004    
#> 262: 0.128152    1.023691    3.449028    0.181168    0.049214    0.020339    0.599011    
#> 263: 0.128237    1.023901    3.449021    0.180913    0.049313    0.020319    0.598981    
#> 264: 0.128031    1.023963    3.448887    0.180546    0.049511    0.020303    0.599297    
#> 265: 0.128026    1.023788    3.448937    0.180603    0.049669    0.020303    0.599454    
#> 266: 0.128169    1.023846    3.449025    0.180444    0.049653    0.020283    0.599497    
#> 267: 0.128580    1.023712    3.449350    0.180360    0.049785    0.020280    0.599444    
#> 268: 0.128956    1.023105    3.449443    0.180562    0.049747    0.020275    0.599561    
#> 269: 0.129990    1.022744    3.449715    0.180516    0.049897    0.020280    0.599788    
#> 270: 0.130454    1.022604    3.449838    0.180523    0.049949    0.020228    0.599744    
#> 271: 0.130569    1.022734    3.449753    0.180513    0.049983    0.020245    0.599819    
#> 272: 0.130674    1.022694    3.449674    0.180203    0.049997    0.020231    0.600072    
#> 273: 0.130675    1.023046    3.449719    0.180143    0.050185    0.020227    0.600185    
#> 274: 0.130517    1.023302    3.449792    0.179963    0.050122    0.020212    0.600224    
#> 275: 0.130486    1.023521    3.449688    0.179774    0.050150    0.020198    0.600432    
#> 276: 0.130493    1.023671    3.449691    0.180189    0.050225    0.020232    0.600333    
#> 277: 0.130339    1.023848    3.449654    0.180279    0.050243    0.020226    0.600416    
#> 278: 0.130168    1.024085    3.449613    0.180049    0.050148    0.020216    0.600417    
#> 279: 0.129913    1.024149    3.449603    0.180182    0.050187    0.020218    0.600519    
#> 280: 0.129747    1.024334    3.449485    0.180077    0.050188    0.020177    0.600456    
#> 281: 0.129373    1.024590    3.449433    0.180083    0.050152    0.020179    0.600454    
#> 282: 0.129234    1.024750    3.449491    0.179750    0.050301    0.020110    0.600434    
#> 283: 0.129183    1.024802    3.449501    0.179721    0.050288    0.020052    0.600508    
#> 284: 0.129208    1.024712    3.449488    0.179677    0.050342    0.019986    0.600613    
#> 285: 0.128854    1.024770    3.449309    0.179192    0.050458    0.019902    0.600747    
#> 286: 0.128658    1.024748    3.449206    0.178709    0.050776    0.019818    0.600766    
#> 287: 0.128554    1.024975    3.449090    0.178209    0.051005    0.019725    0.600794    
#> 288: 0.128720    1.025015    3.449110    0.177736    0.051155    0.019676    0.601226    
#> 289: 0.128737    1.024987    3.449088    0.177347    0.051280    0.019657    0.601502    
#> 290: 0.128932    1.024939    3.449123    0.177093    0.051373    0.019610    0.601709    
#> 291: 0.129131    1.025007    3.449250    0.176958    0.051508    0.019573    0.601488    
#> 292: 0.129123    1.025178    3.449151    0.176729    0.051553    0.019534    0.601329    
#> 293: 0.128727    1.025334    3.448906    0.176727    0.051585    0.019525    0.601240    
#> 294: 0.128255    1.025599    3.448657    0.177070    0.051580    0.019518    0.601253    
#> 295: 0.127990    1.025477    3.448573    0.177169    0.051606    0.019495    0.601139    
#> 296: 0.127765    1.025407    3.448472    0.177095    0.051567    0.019490    0.601156    
#> 297: 0.127923    1.025442    3.448538    0.177181    0.051515    0.019490    0.601030    
#> 298: 0.128059    1.025251    3.448581    0.177263    0.051473    0.019474    0.600781    
#> 299: 0.128131    1.025205    3.448593    0.177203    0.051342    0.019503    0.600735    
#> 300: 0.128222    1.025020    3.448639    0.177290    0.051290    0.019509    0.600667    
#> 301: 0.128516    1.024890    3.448573    0.177582    0.051282    0.019528    0.600670    
#> 302: 0.128511    1.025029    3.448542    0.177628    0.051267    0.019537    0.600512    
#> 303: 0.128330    1.025134    3.448446    0.177521    0.051245    0.019596    0.600311    
#> 304: 0.128294    1.025397    3.448319    0.177781    0.051193    0.019646    0.600261    
#> 305: 0.128092    1.025287    3.448203    0.177758    0.051178    0.019658    0.600201    
#> 306: 0.127637    1.025138    3.448020    0.177958    0.051122    0.019662    0.600237    
#> 307: 0.127543    1.025058    3.447912    0.178221    0.051180    0.019679    0.600273    
#> 308: 0.127441    1.025039    3.447871    0.178578    0.051123    0.019665    0.600159    
#> 309: 0.127457    1.024856    3.447813    0.178944    0.051139    0.019673    0.600171    
#> 310: 0.127527    1.024782    3.447681    0.178886    0.051110    0.019653    0.600253    
#> 311: 0.127221    1.024830    3.447502    0.179189    0.051139    0.019635    0.600117    
#> 312: 0.126970    1.024793    3.447348    0.179414    0.051140    0.019648    0.600046    
#> 313: 0.126798    1.024548    3.447195    0.179757    0.051061    0.019643    0.600170    
#> 314: 0.126361    1.024647    3.447026    0.179762    0.051069    0.019637    0.600205    
#> 315: 0.125825    1.024802    3.446883    0.179760    0.051064    0.019615    0.600167    
#> 316: 0.125669    1.024984    3.446815    0.179621    0.051120    0.019602    0.600140    
#> 317: 0.125812    1.024966    3.446820    0.179820    0.051066    0.019604    0.600251    
#> 318: 0.125863    1.025086    3.446813    0.179654    0.051033    0.019600    0.600201    
#> 319: 0.125698    1.025051    3.446819    0.179467    0.051048    0.019603    0.600191    
#> 320: 0.125220    1.025102    3.446686    0.179131    0.051088    0.019599    0.600273    
#> 321: 0.125019    1.025127    3.446561    0.178740    0.051075    0.019577    0.600403    
#> 322: 0.124709    1.025343    3.446400    0.178498    0.051070    0.019547    0.600472    
#> 323: 0.124001    1.025539    3.446256    0.178480    0.051042    0.019532    0.600568    
#> 324: 0.123652    1.025640    3.446160    0.178539    0.051086    0.019508    0.600470    
#> 325: 0.123395    1.025825    3.445992    0.178436    0.051111    0.019517    0.600403    
#> 326: 0.123215    1.025731    3.445883    0.178296    0.051091    0.019529    0.600524    
#> 327: 0.123130    1.025733    3.445878    0.178120    0.051148    0.019542    0.600645    
#> 328: 0.123313    1.025758    3.445952    0.178214    0.051145    0.019529    0.600601    
#> 329: 0.123506    1.025909    3.445939    0.178213    0.051123    0.019497    0.600538    
#> 330: 0.123438    1.025940    3.446042    0.178259    0.051101    0.019516    0.600396    
#> 331: 0.123533    1.026093    3.446082    0.178146    0.051097    0.019531    0.600304    
#> 332: 0.123751    1.026094    3.446201    0.177942    0.051072    0.019534    0.600461    
#> 333: 0.123570    1.026034    3.446138    0.177796    0.051071    0.019516    0.600546    
#> 334: 0.123381    1.026007    3.446009    0.178038    0.051011    0.019527    0.600458    
#> 335: 0.123237    1.026152    3.445896    0.178129    0.050962    0.019548    0.600501    
#> 336: 0.122981    1.026301    3.445816    0.178186    0.050966    0.019572    0.600485    
#> 337: 0.122759    1.026646    3.445692    0.178148    0.051044    0.019572    0.600507    
#> 338: 0.122535    1.026811    3.445590    0.178152    0.051165    0.019534    0.600472    
#> 339: 0.122354    1.027020    3.445447    0.178255    0.051261    0.019497    0.600401    
#> 340: 0.122150    1.027028    3.445329    0.178094    0.051401    0.019442    0.600417    
#> 341: 0.122131    1.027119    3.445282    0.177874    0.051477    0.019394    0.600394    
#> 342: 0.121939    1.027056    3.445266    0.177659    0.051521    0.019343    0.600367    
#> 343: 0.121718    1.027052    3.445250    0.177573    0.051544    0.019304    0.600366    
#> 344: 0.121806    1.027070    3.445308    0.177426    0.051526    0.019277    0.600307    
#> 345: 0.121837    1.027050    3.445345    0.177217    0.051529    0.019244    0.600366    
#> 346: 0.121779    1.026992    3.445330    0.177270    0.051572    0.019208    0.600230    
#> 347: 0.121911    1.026886    3.445307    0.177382    0.051586    0.019207    0.600095    
#> 348: 0.121814    1.026890    3.445290    0.177305    0.051595    0.019183    0.600062    
#> 349: 0.121749    1.026834    3.445266    0.177327    0.051564    0.019166    0.600069    
#> 350: 0.121569    1.026797    3.445205    0.177318    0.051555    0.019149    0.600017    
#> 351: 0.121646    1.026923    3.445141    0.177133    0.051544    0.019128    0.599988    
#> 352: 0.121426    1.027044    3.445090    0.176988    0.051555    0.019122    0.600013    
#> 353: 0.121333    1.027199    3.444987    0.176994    0.051492    0.019132    0.599875    
#> 354: 0.121320    1.027256    3.444876    0.177041    0.051412    0.019140    0.599743    
#> 355: 0.121572    1.027122    3.444900    0.177139    0.051364    0.019139    0.599678    
#> 356: 0.121670    1.027022    3.444906    0.177127    0.051396    0.019130    0.599791    
#> 357: 0.121806    1.027011    3.444920    0.177020    0.051395    0.019121    0.599720    
#> 358: 0.121879    1.026996    3.445026    0.176967    0.051360    0.019140    0.599632    
#> 359: 0.121837    1.027042    3.445059    0.176986    0.051379    0.019144    0.599514    
#> 360: 0.121913    1.027017    3.445048    0.177043    0.051366    0.019155    0.599375    
#> 361: 0.121862    1.026887    3.445062    0.176971    0.051351    0.019176    0.599293    
#> 362: 0.121912    1.026799    3.445077    0.176815    0.051330    0.019215    0.599123    
#> 363: 0.121819    1.026791    3.445025    0.176923    0.051280    0.019229    0.598946    
#> 364: 0.121936    1.026828    3.444994    0.177123    0.051265    0.019263    0.598926    
#> 365: 0.121763    1.026726    3.445005    0.177133    0.051232    0.019306    0.598981    
#> 366: 0.121620    1.026684    3.444946    0.177022    0.051224    0.019323    0.598969    
#> 367: 0.121660    1.026584    3.444960    0.176913    0.051230    0.019326    0.598883    
#> 368: 0.121835    1.026589    3.444948    0.176748    0.051151    0.019345    0.598869    
#> 369: 0.121841    1.026519    3.444913    0.176692    0.051165    0.019333    0.598917    
#> 370: 0.121921    1.026408    3.444970    0.176517    0.051181    0.019350    0.598933    
#> 371: 0.121871    1.026445    3.444882    0.176297    0.051278    0.019323    0.598878    
#> 372: 0.121679    1.026538    3.444797    0.176151    0.051352    0.019295    0.598847    
#> 373: 0.121501    1.026692    3.444699    0.176249    0.051358    0.019275    0.598857    
#> 374: 0.121489    1.026766    3.444637    0.176255    0.051313    0.019271    0.598790    
#> 375: 0.121428    1.026783    3.444629    0.176369    0.051298    0.019275    0.598739    
#> 376: 0.121407    1.026815    3.444637    0.176432    0.051281    0.019278    0.598635    
#> 377: 0.121294    1.026874    3.444591    0.176498    0.051335    0.019275    0.598517    
#> 378: 0.121207    1.026859    3.444550    0.176529    0.051302    0.019281    0.598499    
#> 379: 0.120965    1.026810    3.444502    0.176623    0.051304    0.019263    0.598527    
#> 380: 0.120716    1.026820    3.444414    0.176677    0.051298    0.019264    0.598450    
#> 381: 0.120492    1.026794    3.444342    0.176757    0.051350    0.019260    0.598406    
#> 382: 0.120314    1.026840    3.444257    0.176909    0.051361    0.019238    0.598302    
#> 383: 0.120470    1.026834    3.444266    0.177155    0.051304    0.019240    0.598258    
#> 384: 0.120426    1.026821    3.444258    0.177445    0.051303    0.019264    0.598173    
#> 385: 0.120497    1.026857    3.444287    0.177437    0.051272    0.019303    0.598125    
#> 386: 0.120558    1.026942    3.444373    0.177394    0.051245    0.019329    0.598148    
#> 387: 0.120845    1.026982    3.444502    0.177230    0.051227    0.019343    0.598137    
#> 388: 0.121067    1.026932    3.444557    0.177025    0.051226    0.019352    0.598182    
#> 389: 0.121043    1.026796    3.444574    0.176846    0.051256    0.019357    0.598278    
#> 390: 0.120965    1.026814    3.444557    0.176757    0.051317    0.019316    0.598249    
#> 391: 0.121099    1.026747    3.444592    0.176834    0.051333    0.019290    0.598145    
#> 392: 0.121140    1.026698    3.444646    0.176958    0.051342    0.019270    0.598080    
#> 393: 0.121059    1.026600    3.444577    0.177020    0.051370    0.019270    0.598049    
#> 394: 0.121027    1.026617    3.444502    0.176970    0.051352    0.019267    0.598007    
#> 395: 0.121116    1.026664    3.444450    0.176856    0.051324    0.019287    0.598079    
#> 396: 0.121132    1.026713    3.444420    0.176847    0.051282    0.019310    0.598048    
#> 397: 0.121206    1.026739    3.444446    0.176850    0.051256    0.019347    0.598101    
#> 398: 0.121327    1.026699    3.444480    0.176744    0.051205    0.019367    0.598193    
#> 399: 0.121523    1.026723    3.444553    0.176574    0.051173    0.019381    0.598289    
#> 400: 0.121742    1.026777    3.444608    0.176491    0.051175    0.019390    0.598267    
#> 401: 0.121947    1.026783    3.444658    0.176526    0.051167    0.019409    0.598273    
#> 402: 0.122028    1.026853    3.444703    0.176425    0.051179    0.019416    0.598282    
#> 403: 0.122187    1.026893    3.444702    0.176343    0.051215    0.019417    0.598234    
#> 404: 0.122225    1.026938    3.444639    0.176263    0.051196    0.019416    0.598236    
#> 405: 0.122326    1.026939    3.444677    0.176263    0.051120    0.019420    0.598245    
#> 406: 0.122429    1.026979    3.444700    0.176277    0.051105    0.019442    0.598343    
#> 407: 0.122455    1.027076    3.444678    0.176310    0.051079    0.019456    0.598321    
#> 408: 0.122596    1.027201    3.444682    0.176347    0.051094    0.019461    0.598399    
#> 409: 0.122617    1.027231    3.444676    0.176487    0.051064    0.019462    0.598382    
#> 410: 0.122565    1.027281    3.444574    0.176535    0.051064    0.019447    0.598458    
#> 411: 0.122515    1.027206    3.444518    0.176566    0.051049    0.019443    0.598513    
#> 412: 0.122506    1.027269    3.444496    0.176647    0.051077    0.019438    0.598538    
#> 413: 0.122644    1.027274    3.444535    0.176754    0.051057    0.019436    0.598552    
#> 414: 0.122656    1.027250    3.444553    0.176831    0.051031    0.019441    0.598479    
#> 415: 0.122753    1.027214    3.444581    0.176825    0.050995    0.019438    0.598472    
#> 416: 0.122816    1.027281    3.444591    0.176812    0.050968    0.019433    0.598449    
#> 417: 0.122695    1.027340    3.444577    0.176831    0.050982    0.019410    0.598465    
#> 418: 0.122521    1.027335    3.444569    0.176787    0.050966    0.019393    0.598496    
#> 419: 0.122451    1.027412    3.444533    0.176814    0.050975    0.019370    0.598548    
#> 420: 0.122404    1.027501    3.444472    0.176846    0.051054    0.019336    0.598538    
#> 421: 0.122370    1.027581    3.444471    0.176804    0.051073    0.019310    0.598536    
#> 422: 0.122506    1.027533    3.444505    0.176842    0.051081    0.019281    0.598594    
#> 423: 0.122704    1.027464    3.444543    0.176975    0.051155    0.019253    0.598578    
#> 424: 0.122675    1.027414    3.444508    0.177012    0.051215    0.019234    0.598570    
#> 425: 0.122693    1.027436    3.444489    0.177124    0.051228    0.019232    0.598549    
#> 426: 0.122747    1.027512    3.444506    0.177045    0.051285    0.019210    0.598530    
#> 427: 0.122736    1.027509    3.444521    0.176994    0.051355    0.019188    0.598485    
#> 428: 0.122637    1.027539    3.444499    0.176913    0.051368    0.019185    0.598552    
#> 429: 0.122633    1.027507    3.444513    0.176785    0.051371    0.019196    0.598634    
#> 430: 0.122582    1.027541    3.444506    0.176763    0.051358    0.019208    0.598661    
#> 431: 0.122528    1.027549    3.444453    0.176760    0.051331    0.019201    0.598625    
#> 432: 0.122412    1.027610    3.444427    0.176805    0.051328    0.019202    0.598557    
#> 433: 0.122316    1.027693    3.444436    0.176774    0.051284    0.019211    0.598552    
#> 434: 0.122171    1.027624    3.444437    0.176717    0.051272    0.019208    0.598473    
#> 435: 0.122194    1.027589    3.444440    0.176767    0.051265    0.019206    0.598376    
#> 436: 0.122163    1.027544    3.444475    0.176840    0.051293    0.019202    0.598346    
#> 437: 0.122216    1.027487    3.444468    0.176821    0.051326    0.019176    0.598363    
#> 438: 0.122223    1.027596    3.444406    0.176762    0.051352    0.019151    0.598394    
#> 439: 0.122325    1.027725    3.444416    0.176671    0.051342    0.019135    0.598383    
#> 440: 0.122305    1.027797    3.444424    0.176661    0.051351    0.019140    0.598326    
#> 441: 0.122308    1.027742    3.444476    0.176681    0.051332    0.019136    0.598229    
#> 442: 0.122248    1.027766    3.444498    0.176825    0.051340    0.019131    0.598115    
#> 443: 0.122224    1.027794    3.444460    0.176828    0.051331    0.019115    0.598075    
#> 444: 0.122162    1.027823    3.444355    0.176892    0.051306    0.019105    0.598047    
#> 445: 0.122089    1.027875    3.444293    0.176830    0.051311    0.019090    0.598053    
#> 446: 0.121848    1.027996    3.444202    0.176737    0.051291    0.019080    0.598013    
#> 447: 0.121857    1.028047    3.444181    0.176797    0.051279    0.019085    0.597971    
#> 448: 0.121840    1.028073    3.444195    0.176768    0.051274    0.019105    0.597894    
#> 449: 0.121867    1.028117    3.444265    0.176758    0.051247    0.019122    0.597894    
#> 450: 0.121740    1.028133    3.444251    0.176669    0.051204    0.019130    0.597940    
#> 451: 0.121668    1.028142    3.444219    0.176627    0.051155    0.019130    0.598047    
#> 452: 0.121635    1.028143    3.444169    0.176581    0.051146    0.019131    0.598108    
#> 453: 0.121537    1.028126    3.444161    0.176642    0.051167    0.019134    0.598077    
#> 454: 0.121439    1.028277    3.444144    0.176645    0.051159    0.019133    0.598061    
#> 455: 0.121400    1.028386    3.444107    0.176658    0.051160    0.019137    0.598079    
#> 456: 0.121279    1.028388    3.444059    0.176658    0.051155    0.019144    0.598111    
#> 457: 0.121234    1.028346    3.444086    0.176591    0.051165    0.019141    0.598097    
#> 458: 0.121129    1.028475    3.444056    0.176549    0.051206    0.019124    0.598121    
#> 459: 0.120957    1.028562    3.444008    0.176498    0.051219    0.019093    0.598163    
#> 460: 0.120776    1.028649    3.443900    0.176509    0.051227    0.019083    0.598189    
#> 461: 0.120685    1.028675    3.443888    0.176483    0.051233    0.019061    0.598210    
#> 462: 0.120601    1.028753    3.443874    0.176385    0.051236    0.019037    0.598258    
#> 463: 0.120572    1.028796    3.443902    0.176291    0.051267    0.019007    0.598320    
#> 464: 0.120744    1.028783    3.443938    0.176317    0.051278    0.018996    0.598296    
#> 465: 0.120771    1.028652    3.443979    0.176289    0.051279    0.018984    0.598334    
#> 466: 0.120819    1.028654    3.443990    0.176191    0.051287    0.018972    0.598307    
#> 467: 0.120849    1.028606    3.444051    0.176107    0.051280    0.018969    0.598251    
#> 468: 0.120958    1.028608    3.444118    0.176113    0.051262    0.018973    0.598223    
#> 469: 0.121089    1.028600    3.444159    0.176149    0.051238    0.018974    0.598206    
#> 470: 0.121088    1.028586    3.444144    0.176095    0.051214    0.018969    0.598196    
#> 471: 0.121038    1.028566    3.444121    0.175967    0.051208    0.018969    0.598135    
#> 472: 0.121035    1.028541    3.444099    0.175952    0.051189    0.018974    0.598114    
#> 473: 0.120926    1.028585    3.444071    0.175866    0.051181    0.018968    0.598039    
#> 474: 0.120878    1.028609    3.444069    0.175727    0.051181    0.018967    0.598060    
#> 475: 0.120967    1.028653    3.444095    0.175655    0.051196    0.018967    0.598072    
#> 476: 0.120964    1.028670    3.444096    0.175630    0.051185    0.018968    0.598143    
#> 477: 0.120978    1.028728    3.444124    0.175604    0.051182    0.018985    0.598060    
#> 478: 0.120936    1.028752    3.444119    0.175654    0.051162    0.018992    0.598030    
#> 479: 0.121085    1.028730    3.444170    0.175722    0.051138    0.019000    0.598008    
#> 480: 0.121172    1.028722    3.444185    0.175747    0.051128    0.018994    0.597995    
#> 481: 0.121205    1.028687    3.444208    0.175742    0.051117    0.019002    0.597967    
#> 482: 0.121168    1.028651    3.444179    0.175852    0.051097    0.018996    0.598028    
#> 483: 0.121056    1.028660    3.444146    0.175954    0.051111    0.018992    0.597999    
#> 484: 0.120998    1.028579    3.444168    0.176058    0.051134    0.018986    0.598015    
#> 485: 0.121026    1.028521    3.444183    0.176066    0.051124    0.018979    0.598073    
#> 486: 0.120995    1.028507    3.444181    0.176087    0.051123    0.018960    0.598042    
#> 487: 0.121172    1.028541    3.444236    0.176278    0.051143    0.018959    0.597974    
#> 488: 0.121304    1.028534    3.444281    0.176454    0.051109    0.018953    0.597942    
#> 489: 0.121450    1.028499    3.444313    0.176616    0.051086    0.018953    0.597899    
#> 490: 0.121553    1.028518    3.444380    0.176628    0.051066    0.018968    0.597880    
#> 491: 0.121649    1.028543    3.444435    0.176687    0.051040    0.018975    0.597859    
#> 492: 0.121677    1.028575    3.444424    0.176741    0.050993    0.018985    0.597919    
#> 493: 0.121708    1.028581    3.444466    0.176845    0.050952    0.019000    0.597896    
#> 494: 0.121844    1.028540    3.444506    0.176721    0.050894    0.019015    0.597887    
#> 495: 0.121944    1.028522    3.444551    0.176662    0.050872    0.019026    0.597883    
#> 496: 0.122038    1.028531    3.444575    0.176504    0.050845    0.019030    0.597960    
#> 497: 0.122010    1.028503    3.444599    0.176346    0.050855    0.019026    0.598026    
#> 498: 0.122028    1.028506    3.444589    0.176301    0.050856    0.019023    0.598010    
#> 499: 0.122000    1.028553    3.444553    0.176310    0.050876    0.019019    0.597960    
#> 500: 0.121978    1.028501    3.444529    0.176315    0.050905    0.019010    0.597927    
#> [====|====|=>-------------------------------------]  25%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.264298    0.987488    3.449494    0.392401    0.085846    0.016800    1.614386    
#> 002: 0.225841    1.088359    3.499667    0.372781    0.081554    0.015960    1.067967    
#> 003: 0.377509    1.101516    3.522684    0.354142    0.077476    0.015162    0.751035    
#> 004: 0.365263    1.138871    3.522908    0.348502    0.073602    0.014404    0.669017    
#> 005: 0.425377    1.143001    3.510348    0.371393    0.069922    0.013684    0.619016    
#> 006: 0.402004    1.149988    3.513463    0.389036    0.066426    0.013000    0.591955    
#> 007: 0.403224    1.165763    3.519168    0.372127    0.063105    0.012350    0.591965    
#> 008: 0.372004    1.161113    3.505388    0.412339    0.059949    0.012833    0.577063    
#> 009: 0.346067    1.143020    3.504066    0.391722    0.056952    0.012804    0.588107    
#> 010: 0.418701    1.142449    3.512619    0.504841    0.054104    0.016370    0.582473    
#> 011: 0.434897    1.171425    3.516762    0.556550    0.051399    0.017642    0.562629    
#> 012: 0.444349    1.149548    3.519632    0.528723    0.048829    0.016760    0.571436    
#> 013: 0.422508    1.136719    3.522562    0.507560    0.046388    0.016401    0.579177    
#> 014: 0.469984    1.127652    3.537180    0.482182    0.044068    0.015581    0.580301    
#> 015: 0.466606    1.135647    3.534125    0.458073    0.041865    0.014802    0.567459    
#> 016: 0.458560    1.152671    3.528984    0.460220    0.039772    0.014062    0.582954    
#> 017: 0.392592    1.160623    3.523088    0.437209    0.037783    0.013358    0.571095    
#> 018: 0.441867    1.164540    3.516139    0.447513    0.035894    0.015702    0.576188    
#> 019: 0.412574    1.154753    3.523364    0.425137    0.034099    0.014917    0.562038    
#> 020: 0.452193    1.151146    3.521319    0.438060    0.032394    0.014370    0.564873    
#> 021: 0.448355    1.139191    3.542216    0.416157    0.030775    0.015704    0.558984    
#> 022: 0.455179    1.139725    3.533312    0.422501    0.029236    0.015082    0.562662    
#> 023: 0.531837    1.135578    3.550646    0.534802    0.027774    0.018540    0.576217    
#> 024: 0.516023    1.157191    3.555801    0.522816    0.026385    0.017613    0.560332    
#> 025: 0.495778    1.147717    3.540535    0.527162    0.025066    0.016733    0.566537    
#> 026: 0.471702    1.162266    3.537385    0.500804    0.023813    0.015896    0.577833    
#> 027: 0.455951    1.166954    3.534754    0.475764    0.026457    0.015101    0.567678    
#> 028: 0.464221    1.158816    3.529864    0.544382    0.025134    0.015140    0.549773    
#> 029: 0.471908    1.156183    3.523531    0.600095    0.023878    0.015592    0.563027    
#> 030: 0.502378    1.163734    3.530547    0.653440    0.022684    0.014963    0.572758    
#> 031: 0.479503    1.150881    3.542763    0.620768    0.021550    0.015522    0.563626    
#> 032: 0.453658    1.141973    3.542186    0.589729    0.020472    0.014746    0.562132    
#> 033: 0.437823    1.148159    3.529159    0.560243    0.019449    0.014009    0.563888    
#> 034: 0.480633    1.143184    3.535376    0.532231    0.018476    0.013308    0.575865    
#> 035: 0.457330    1.132083    3.524463    0.505619    0.017552    0.012643    0.568746    
#> 036: 0.443032    1.139445    3.526864    0.480338    0.018934    0.012011    0.581007    
#> 037: 0.457015    1.132971    3.534795    0.456321    0.017987    0.013727    0.589033    
#> 038: 0.441107    1.137937    3.524416    0.436445    0.017476    0.015201    0.585604    
#> 039: 0.452398    1.135266    3.536364    0.454468    0.017909    0.015908    0.567719    
#> 040: 0.454421    1.133552    3.541737    0.458484    0.021428    0.015699    0.563725    
#> 041: 0.444443    1.131675    3.530840    0.483393    0.020357    0.014914    0.566876    
#> 042: 0.498123    1.116855    3.532681    0.560359    0.019339    0.014169    0.565265    
#> 043: 0.460472    1.129623    3.521140    0.579199    0.018372    0.015209    0.555621    
#> 044: 0.431097    1.127737    3.525904    0.550239    0.017453    0.016009    0.559229    
#> 045: 0.410299    1.135868    3.520133    0.522727    0.018372    0.016351    0.556559    
#> 046: 0.426820    1.135611    3.524358    0.512961    0.017453    0.015690    0.565961    
#> 047: 0.438613    1.136635    3.517361    0.487575    0.017587    0.016294    0.567682    
#> 048: 0.419137    1.135254    3.520742    0.463197    0.022160    0.015479    0.577435    
#> 049: 0.410167    1.153695    3.519580    0.440037    0.021052    0.015263    0.578907    
#> 050: 0.467191    1.132171    3.528896    0.532676    0.025686    0.015377    0.566468    
#> 051: 0.450320    1.151338    3.524641    0.506042    0.024401    0.016144    0.564523    
#> 052: 0.460122    1.148668    3.532932    0.480740    0.023181    0.015337    0.564792    
#> 053: 0.450978    1.125830    3.533082    0.520275    0.022022    0.018008    0.561442    
#> 054: 0.419491    1.142835    3.528484    0.494261    0.022727    0.017108    0.578992    
#> 055: 0.422128    1.139162    3.525142    0.469548    0.021590    0.016252    0.581061    
#> 056: 0.450379    1.136535    3.530811    0.490451    0.022476    0.015545    0.594551    
#> 057: 0.415436    1.142683    3.528416    0.465928    0.021353    0.015520    0.598788    
#> 058: 0.458490    1.158208    3.512263    0.453013    0.022772    0.014744    0.583831    
#> 059: 0.377185    1.126646    3.512082    0.471412    0.021886    0.016530    0.593914    
#> 060: 0.380214    1.130266    3.520667    0.447842    0.020792    0.016956    0.613849    
#> 061: 0.455990    1.126212    3.532927    0.472345    0.019868    0.016108    0.592609    
#> 062: 0.420399    1.138291    3.530867    0.448754    0.018875    0.016724    0.619411    
#> 063: 0.461717    1.129034    3.539718    0.486641    0.017931    0.017210    0.610212    
#> 064: 0.497392    1.130530    3.556505    0.462309    0.018577    0.017463    0.602442    
#> 065: 0.508290    1.130336    3.532636    0.439193    0.020130    0.018287    0.589476    
#> 066: 0.497620    1.125870    3.543118    0.489580    0.019124    0.017373    0.577796    
#> 067: 0.479693    1.127616    3.534795    0.535479    0.020190    0.016504    0.573811    
#> 068: 0.463577    1.141756    3.545267    0.508705    0.023070    0.015679    0.567600    
#> 069: 0.476195    1.160294    3.525046    0.620457    0.021916    0.019331    0.569393    
#> 070: 0.422656    1.157157    3.529334    0.589434    0.020821    0.019409    0.571299    
#> 071: 0.436489    1.181802    3.505054    0.559962    0.019780    0.019575    0.571112    
#> 072: 0.413855    1.165867    3.500612    0.584135    0.018791    0.018639    0.568304    
#> 073: 0.442695    1.170137    3.516411    0.576099    0.017851    0.017707    0.574049    
#> 074: 0.398827    1.159422    3.507939    0.547294    0.017000    0.016822    0.572785    
#> 075: 0.398201    1.167126    3.522038    0.520853    0.016150    0.019267    0.568079    
#> 076: 0.451159    1.158683    3.530245    0.518174    0.017238    0.018776    0.588823    
#> 077: 0.467053    1.148376    3.526762    0.522404    0.016376    0.020994    0.583924    
#> 078: 0.474526    1.144623    3.532264    0.568006    0.015557    0.020160    0.572490    
#> 079: 0.479100    1.150086    3.532473    0.559267    0.014779    0.020025    0.589609    
#> 080: 0.509303    1.146231    3.540056    0.566786    0.014603    0.020486    0.608396    
#> 081: 0.524870    1.146683    3.538006    0.562517    0.014194    0.021029    0.591733    
#> 082: 0.465962    1.136272    3.541684    0.534391    0.013485    0.024442    0.596922    
#> 083: 0.457454    1.143275    3.535244    0.508746    0.013843    0.023220    0.590439    
#> 084: 0.461519    1.159594    3.527481    0.630079    0.015950    0.023035    0.575842    
#> 085: 0.475665    1.151229    3.542343    0.598575    0.017517    0.023409    0.585039    
#> 086: 0.475150    1.139611    3.535631    0.568647    0.016641    0.022239    0.574332    
#> 087: 0.445165    1.138750    3.534273    0.540214    0.016675    0.021127    0.575714    
#> 088: 0.406545    1.138227    3.539697    0.513204    0.015841    0.020071    0.582775    
#> 089: 0.415548    1.153275    3.510499    0.487543    0.015049    0.019067    0.589839    
#> 090: 0.406012    1.163097    3.506592    0.487844    0.014297    0.018114    0.578347    
#> 091: 0.410783    1.159327    3.503111    0.537460    0.013582    0.017509    0.574149    
#> 092: 0.413894    1.175314    3.505128    0.510587    0.014586    0.018155    0.572476    
#> 093: 0.379663    1.156926    3.510891    0.485058    0.014105    0.019594    0.579041    
#> 094: 0.399736    1.163271    3.500152    0.460805    0.015369    0.018614    0.578991    
#> 095: 0.409482    1.158156    3.520372    0.437764    0.016211    0.017982    0.567313    
#> 096: 0.388474    1.161390    3.509854    0.522728    0.015401    0.019484    0.569145    
#> 097: 0.437399    1.172485    3.511052    0.506629    0.015462    0.019569    0.575400    
#> 098: 0.438939    1.168127    3.519816    0.481297    0.017619    0.018590    0.566164    
#> 099: 0.444099    1.152528    3.515157    0.457232    0.016738    0.017661    0.569044    
#> 100: 0.428408    1.161196    3.522292    0.434371    0.017435    0.016778    0.565562    
#> 101: 0.417018    1.150973    3.517701    0.436870    0.016563    0.015939    0.566612    
#> 102: 0.420067    1.136044    3.507106    0.451272    0.015735    0.016353    0.568782    
#> 103: 0.430822    1.146523    3.517343    0.485579    0.014949    0.015535    0.571036    
#> 104: 0.450581    1.144799    3.535349    0.498951    0.014201    0.017792    0.582325    
#> 105: 0.471962    1.144327    3.525860    0.538126    0.013491    0.018584    0.576818    
#> 106: 0.446330    1.146023    3.523073    0.534528    0.014479    0.017654    0.581934    
#> 107: 0.455733    1.150546    3.523430    0.507802    0.017292    0.016772    0.569650    
#> 108: 0.410687    1.150967    3.519793    0.482412    0.021000    0.017485    0.574101    
#> 109: 0.451971    1.138431    3.543199    0.458291    0.019950    0.016611    0.572342    
#> 110: 0.435140    1.144087    3.527622    0.436975    0.018952    0.015780    0.581385    
#> 111: 0.466250    1.149605    3.531078    0.462873    0.018202    0.015008    0.562137    
#> 112: 0.422419    1.136038    3.522068    0.445478    0.019525    0.014257    0.583233    
#> 113: 0.438638    1.149417    3.528988    0.489796    0.019035    0.013723    0.564801    
#> 114: 0.473883    1.146662    3.533007    0.524814    0.021335    0.015047    0.573993    
#> 115: 0.474177    1.126653    3.527545    0.536684    0.022459    0.016164    0.578425    
#> 116: 0.480444    1.139509    3.527759    0.549615    0.021336    0.017306    0.583381    
#> 117: 0.428947    1.158107    3.537877    0.522134    0.020603    0.019531    0.591414    
#> 118: 0.438292    1.155721    3.539016    0.496027    0.022517    0.018554    0.584262    
#> 119: 0.473528    1.149559    3.528740    0.471226    0.021391    0.020717    0.584125    
#> 120: 0.431964    1.146079    3.532820    0.471836    0.020321    0.019681    0.588501    
#> 121: 0.466255    1.149991    3.532894    0.479941    0.022906    0.018697    0.583765    
#> 122: 0.457848    1.134096    3.535504    0.514402    0.021760    0.021388    0.573286    
#> 123: 0.486029    1.155585    3.531482    0.501794    0.020672    0.020425    0.568381    
#> 124: 0.433330    1.173513    3.528197    0.496975    0.019977    0.021274    0.575992    
#> 125: 0.430502    1.187806    3.516061    0.502571    0.021751    0.020210    0.567964    
#> 126: 0.397881    1.200609    3.499825    0.477443    0.024381    0.019199    0.574768    
#> 127: 0.419520    1.179882    3.495512    0.454111    0.023162    0.018239    0.579786    
#> 128: 0.401443    1.164215    3.521902    0.465020    0.022004    0.017327    0.575964    
#> 129: 0.432471    1.164554    3.525496    0.459471    0.024196    0.016461    0.583909    
#> 130: 0.457883    1.172413    3.524637    0.491919    0.022986    0.015638    0.578179    
#> 131: 0.436189    1.157674    3.508842    0.467323    0.021837    0.018518    0.554213    
#> 132: 0.378545    1.166918    3.494508    0.443957    0.020745    0.017592    0.576026    
#> 133: 0.373832    1.163928    3.502459    0.482876    0.019708    0.016819    0.579252    
#> 134: 0.396003    1.152400    3.517471    0.521507    0.018722    0.015986    0.576108    
#> 135: 0.428762    1.137812    3.526801    0.495431    0.017786    0.019333    0.572206    
#> 136: 0.446572    1.139794    3.529072    0.505726    0.016987    0.018783    0.563321    
#> 137: 0.483396    1.130474    3.529205    0.586144    0.016138    0.018702    0.577420    
#> 138: 0.468300    1.128532    3.539047    0.556837    0.015331    0.017767    0.586567    
#> 139: 0.445901    1.130808    3.539290    0.528995    0.014565    0.018022    0.585631    
#> 140: 0.462016    1.143816    3.542473    0.502844    0.013836    0.018081    0.585778    
#> 141: 0.450356    1.129344    3.541677    0.477702    0.013145    0.017470    0.599125    
#> 142: 0.462787    1.113625    3.553151    0.453817    0.014229    0.016596    0.598732    
#> 143: 0.434597    1.123578    3.544021    0.431126    0.016837    0.015766    0.582066    
#> 144: 0.492271    1.115176    3.558593    0.460199    0.015995    0.016919    0.577353    
#> 145: 0.454069    1.126784    3.537153    0.454456    0.015195    0.016073    0.569369    
#> 146: 0.454709    1.130749    3.550393    0.500304    0.014436    0.018470    0.579293    
#> 147: 0.413684    1.137567    3.516226    0.533725    0.016135    0.017930    0.574375    
#> 148: 0.403759    1.141708    3.533905    0.535466    0.015328    0.018082    0.569017    
#> 149: 0.417975    1.141539    3.525994    0.508692    0.014562    0.017487    0.584383    
#> 150: 0.454272    1.138975    3.516220    0.498171    0.014896    0.016613    0.590942    
#> 151: 0.410774    1.159895    3.523022    0.473263    0.017991    0.016852    0.569665    
#> 152: 0.412664    1.153581    3.530042    0.464938    0.020028    0.015593    0.571460    
#> 153: 0.425309    1.161695    3.540253    0.429817    0.021704    0.017066    0.577794    
#> 154: 0.461413    1.158869    3.538366    0.447180    0.019855    0.019834    0.586409    
#> 155: 0.482390    1.149915    3.534014    0.496344    0.012126    0.022662    0.578756    
#> 156: 0.482232    1.151433    3.534198    0.491253    0.013160    0.024242    0.575551    
#> 157: 0.475161    1.140670    3.531968    0.511175    0.009696    0.022757    0.566755    
#> 158: 0.461267    1.151619    3.532497    0.474492    0.010133    0.021075    0.558557    
#> 159: 0.435862    1.153798    3.537943    0.527877    0.011317    0.018779    0.568471    
#> 160: 0.458068    1.139640    3.537649    0.459280    0.013925    0.016702    0.571157    
#> 161: 0.482272    1.140876    3.541420    0.509171    0.015006    0.020607    0.567353    
#> 162: 0.490076    1.156339    3.538780    0.480961    0.011610    0.020077    0.582067    
#> 163: 0.478365    1.139875    3.533731    0.489783    0.012042    0.019267    0.574302    
#> 164: 0.459353    1.153413    3.527069    0.469741    0.013780    0.017386    0.579382    
#> 165: 0.463336    1.159583    3.531311    0.496126    0.013873    0.014680    0.593908    
#> 166: 0.465653    1.174426    3.529556    0.445204    0.015416    0.011623    0.593755    
#> 167: 0.424589    1.190363    3.526249    0.460800    0.020375    0.012718    0.583041    
#> 168: 0.446693    1.189057    3.528699    0.530624    0.018504    0.015072    0.571301    
#> 169: 0.458500    1.186041    3.515141    0.524113    0.019641    0.011000    0.587317    
#> 170: 0.477881    1.157715    3.529480    0.575918    0.019612    0.012914    0.569907    
#> 171: 0.459860    1.156524    3.525632    0.490575    0.020280    0.009519    0.568401    
#> 172: 0.412088    1.171498    3.524206    0.412801    0.019013    0.010254    0.576354    
#> 173: 0.399827    1.182364    3.508685    0.401507    0.018651    0.013211    0.587859    
#> 174: 0.407020    1.184912    3.511077    0.442270    0.018178    0.013857    0.579272    
#> 175: 0.418557    1.185204    3.509474    0.471513    0.018846    0.015077    0.580127    
#> 176: 0.449475    1.185718    3.519655    0.506856    0.020383    0.015821    0.565787    
#> 177: 0.437639    1.169126    3.530458    0.471634    0.022312    0.013455    0.568963    
#> 178: 0.437228    1.155474    3.531681    0.460021    0.022323    0.014065    0.579482    
#> 179: 0.448945    1.160710    3.523997    0.457395    0.020239    0.016442    0.573416    
#> 180: 0.456320    1.151539    3.526495    0.521504    0.020515    0.018926    0.576673    
#> 181: 0.436812    1.127781    3.515648    0.472573    0.015198    0.015371    0.580530    
#> 182: 0.402018    1.143803    3.507705    0.460377    0.018886    0.013234    0.580286    
#> 183: 0.455725    1.163818    3.518862    0.457375    0.017512    0.016084    0.580816    
#> 184: 0.446118    1.144647    3.529011    0.430450    0.016654    0.015280    0.571622    
#> 185: 0.429575    1.168352    3.518117    0.480861    0.016576    0.014916    0.578266    
#> 186: 0.409108    1.177342    3.516789    0.459596    0.018590    0.015115    0.583990    
#> 187: 0.452654    1.159579    3.529855    0.543802    0.012757    0.019981    0.571271    
#> 188: 0.411588    1.178546    3.521992    0.492319    0.010582    0.019035    0.591313    
#> 189: 0.465468    1.188398    3.532168    0.595708    0.012885    0.025169    0.585025    
#> 190: 0.488437    1.182285    3.529334    0.699264    0.012588    0.022485    0.564341    
#> 191: 0.425410    1.191007    3.510975    0.566516    0.013320    0.018885    0.558281    
#> 192: 0.421043    1.184424    3.506100    0.590934    0.015219    0.016706    0.572594    
#> 193: 0.448708    1.180674    3.519216    0.539788    0.016376    0.015137    0.568203    
#> 194: 0.405193    1.161912    3.497362    0.488672    0.016613    0.017102    0.585980    
#> 195: 0.348954    1.167950    3.496906    0.495883    0.014675    0.016923    0.581683    
#> 196: 0.384896    1.175123    3.505253    0.469609    0.017618    0.018423    0.569984    
#> 197: 0.375072    1.188483    3.517605    0.387311    0.016207    0.020594    0.591554    
#> 198: 0.426786    1.164777    3.513673    0.410284    0.014357    0.018388    0.573847    
#> 199: 0.425649    1.159751    3.516854    0.489389    0.012970    0.019409    0.567184    
#> 200: 0.400603    1.155010    3.517000    0.526487    0.010875    0.017115    0.582615    
#> 201: 0.398347    1.147202    3.517231    0.480590    0.011834    0.015213    0.582565    
#> 202: 0.404844    1.147661    3.520956    0.475691    0.012935    0.014260    0.582895    
#> 203: 0.425541    1.147071    3.523493    0.468553    0.013910    0.014717    0.581522    
#> 204: 0.437342    1.143892    3.525753    0.472411    0.014675    0.014688    0.583692    
#> 205: 0.441311    1.144359    3.526825    0.472610    0.015916    0.014868    0.582446    
#> 206: 0.442464    1.146843    3.527550    0.478543    0.015509    0.014593    0.583821    
#> 207: 0.444855    1.147748    3.526902    0.479324    0.015413    0.014419    0.583297    
#> 208: 0.445754    1.148427    3.526880    0.486072    0.015317    0.014732    0.581369    
#> 209: 0.445962    1.147716    3.526485    0.489599    0.014917    0.015336    0.580156    
#> 210: 0.447234    1.147523    3.526973    0.494020    0.014871    0.015572    0.578835    
#> 211: 0.447904    1.148354    3.527706    0.489720    0.014880    0.015603    0.578496    
#> 212: 0.449448    1.148863    3.528480    0.486366    0.014792    0.015742    0.578840    
#> 213: 0.452576    1.147036    3.529472    0.491165    0.014890    0.015924    0.579332    
#> 214: 0.455931    1.146063    3.529624    0.495685    0.014986    0.016320    0.578939    
#> 215: 0.456723    1.145267    3.530054    0.502716    0.014947    0.016708    0.578942    
#> 216: 0.460119    1.143014    3.530835    0.503340    0.014771    0.017155    0.579605    
#> 217: 0.456939    1.142051    3.531228    0.498488    0.014507    0.017363    0.580113    
#> 218: 0.456447    1.141600    3.530538    0.496660    0.014538    0.017505    0.580114    
#> 219: 0.454828    1.141337    3.529901    0.495419    0.014418    0.017497    0.579409    
#> 220: 0.451899    1.142023    3.529341    0.498154    0.014474    0.017493    0.579087    
#> 221: 0.451313    1.142244    3.528609    0.500333    0.014488    0.017570    0.578173    
#> 222: 0.450766    1.142441    3.528025    0.503815    0.014351    0.017664    0.578065    
#> 223: 0.452421    1.142033    3.528261    0.507219    0.014270    0.017863    0.577597    
#> 224: 0.454108    1.142005    3.528906    0.506554    0.014310    0.017926    0.577587    
#> 225: 0.453456    1.141508    3.529649    0.504200    0.014457    0.017946    0.577495    
#> 226: 0.452714    1.141059    3.529465    0.501124    0.014485    0.017863    0.577708    
#> 227: 0.453307    1.141009    3.529496    0.497548    0.014578    0.017665    0.577489    
#> 228: 0.452743    1.141687    3.529405    0.494743    0.014636    0.017525    0.577920    
#> 229: 0.451043    1.141843    3.529008    0.493028    0.014587    0.017437    0.577606    
#> 230: 0.451205    1.141834    3.529132    0.489884    0.014568    0.017453    0.577758    
#> 231: 0.450904    1.142347    3.529135    0.489582    0.014589    0.017386    0.577505    
#> 232: 0.450349    1.142386    3.529255    0.489701    0.014573    0.017358    0.577590    
#> 233: 0.449716    1.141641    3.529279    0.489105    0.014558    0.017431    0.578054    
#> 234: 0.450311    1.141600    3.529391    0.490912    0.014588    0.017498    0.578301    
#> 235: 0.451189    1.141613    3.529484    0.492307    0.014606    0.017442    0.578319    
#> 236: 0.451227    1.141841    3.529247    0.491649    0.014611    0.017414    0.578222    
#> 237: 0.451600    1.142012    3.529244    0.490130    0.014623    0.017433    0.578264    
#> 238: 0.452244    1.142273    3.529444    0.489935    0.014646    0.017523    0.578601    
#> 239: 0.452590    1.142739    3.529641    0.488929    0.014592    0.017635    0.578309    
#> 240: 0.452141    1.142880    3.529804    0.487306    0.014571    0.017682    0.577935    
#> 241: 0.452185    1.142716    3.529944    0.486646    0.014576    0.017715    0.577786    
#> 242: 0.452173    1.142491    3.529846    0.486957    0.014622    0.017742    0.577987    
#> 243: 0.451859    1.142786    3.529616    0.486710    0.014686    0.017770    0.578152    
#> 244: 0.451614    1.142574    3.529725    0.486533    0.014676    0.017820    0.578203    
#> 245: 0.451442    1.142316    3.529793    0.485789    0.014687    0.017760    0.578582    
#> 246: 0.451324    1.142161    3.529815    0.485132    0.014636    0.017793    0.578781    
#> 247: 0.451179    1.141958    3.530008    0.483936    0.014627    0.017831    0.578676    
#> 248: 0.451168    1.142030    3.530079    0.483672    0.014698    0.017807    0.578290    
#> 249: 0.451036    1.141954    3.530216    0.484351    0.014669    0.017793    0.578366    
#> 250: 0.451539    1.141972    3.530278    0.483846    0.014627    0.017771    0.578359    
#> 251: 0.451474    1.142229    3.530455    0.482227    0.014582    0.017725    0.578486    
#> 252: 0.451866    1.142528    3.530765    0.482624    0.014576    0.017729    0.578464    
#> 253: 0.452453    1.142629    3.530957    0.483816    0.014563    0.017750    0.578599    
#> 254: 0.453226    1.142959    3.531169    0.484911    0.014513    0.017750    0.578644    
#> 255: 0.453805    1.142957    3.531371    0.486010    0.014548    0.017726    0.578778    
#> 256: 0.454738    1.142855    3.531516    0.486120    0.014583    0.017677    0.578619    
#> 257: 0.455215    1.142709    3.531543    0.485807    0.014572    0.017647    0.578358    
#> 258: 0.455400    1.142296    3.531727    0.485099    0.014581    0.017600    0.578396    
#> 259: 0.455715    1.141945    3.531903    0.485608    0.014585    0.017548    0.578240    
#> 260: 0.455934    1.141817    3.531977    0.485691    0.014610    0.017554    0.578192    
#> 261: 0.456281    1.141653    3.532075    0.486131    0.014620    0.017562    0.578307    
#> 262: 0.456115    1.141639    3.532098    0.484963    0.014619    0.017582    0.578340    
#> 263: 0.455834    1.141856    3.532096    0.485284    0.014620    0.017585    0.578474    
#> 264: 0.455098    1.141932    3.531874    0.485375    0.014674    0.017550    0.578575    
#> 265: 0.454919    1.142060    3.531863    0.485188    0.014731    0.017528    0.578838    
#> 266: 0.455813    1.142315    3.532068    0.484518    0.014679    0.017544    0.579177    
#> 267: 0.455933    1.142334    3.532239    0.485058    0.014647    0.017528    0.579239    
#> 268: 0.455862    1.142327    3.532266    0.485254    0.014667    0.017519    0.579392    
#> 269: 0.456050    1.142178    3.532290    0.485038    0.014707    0.017479    0.579306    
#> 270: 0.455848    1.142175    3.532120    0.484740    0.014695    0.017410    0.579301    
#> 271: 0.455676    1.142125    3.531918    0.484788    0.014691    0.017351    0.579370    
#> 272: 0.455472    1.142097    3.531830    0.484663    0.014698    0.017305    0.579464    
#> 273: 0.454966    1.142286    3.531726    0.484813    0.014744    0.017278    0.579412    
#> 274: 0.454648    1.142475    3.531663    0.485149    0.014790    0.017274    0.579149    
#> 275: 0.454099    1.142612    3.531327    0.485046    0.014841    0.017224    0.578968    
#> 276: 0.453986    1.142685    3.531257    0.485428    0.014856    0.017169    0.578816    
#> 277: 0.453588    1.142780    3.531230    0.485678    0.014836    0.017161    0.578943    
#> 278: 0.453556    1.142853    3.531255    0.486354    0.014830    0.017145    0.579032    
#> 279: 0.453530    1.142869    3.531236    0.486986    0.014789    0.017174    0.579189    
#> 280: 0.453273    1.143096    3.531159    0.487226    0.014781    0.017223    0.579279    
#> 281: 0.453105    1.143424    3.531047    0.486821    0.014774    0.017244    0.579336    
#> 282: 0.453051    1.143690    3.530993    0.486602    0.014774    0.017280    0.579280    
#> 283: 0.453006    1.143864    3.530976    0.487105    0.014790    0.017336    0.579323    
#> 284: 0.452823    1.144015    3.530909    0.487481    0.014820    0.017366    0.579232    
#> 285: 0.452674    1.144065    3.530719    0.487102    0.014856    0.017407    0.579242    
#> 286: 0.452548    1.144077    3.530605    0.486895    0.014894    0.017412    0.579465    
#> 287: 0.452633    1.144119    3.530511    0.486341    0.014914    0.017434    0.579882    
#> 288: 0.453063    1.144156    3.530572    0.486704    0.014885    0.017461    0.579949    
#> 289: 0.453680    1.144114    3.530686    0.486913    0.014879    0.017474    0.580052    
#> 290: 0.453809    1.144034    3.530753    0.486457    0.014919    0.017486    0.579913    
#> 291: 0.453557    1.144260    3.530698    0.486060    0.014940    0.017488    0.579791    
#> 292: 0.453159    1.144402    3.530596    0.485618    0.014938    0.017483    0.579832    
#> 293: 0.452663    1.144677    3.530393    0.485036    0.014934    0.017490    0.579766    
#> 294: 0.452182    1.144937    3.530117    0.484919    0.014939    0.017497    0.579831    
#> 295: 0.451293    1.145048    3.529820    0.484572    0.014965    0.017508    0.579728    
#> 296: 0.450934    1.145121    3.529597    0.484860    0.014974    0.017503    0.579911    
#> 297: 0.450460    1.145313    3.529289    0.485537    0.014998    0.017502    0.579972    
#> 298: 0.449869    1.145424    3.529070    0.485856    0.015006    0.017476    0.579857    
#> 299: 0.449179    1.145626    3.528805    0.485918    0.015017    0.017462    0.579725    
#> 300: 0.448776    1.145768    3.528725    0.486575    0.014996    0.017492    0.579651    
#> 301: 0.448724    1.145707    3.528669    0.486221    0.014972    0.017495    0.579753    
#> 302: 0.448878    1.145759    3.528655    0.485952    0.014954    0.017500    0.579695    
#> 303: 0.448820    1.145894    3.528643    0.485415    0.014948    0.017485    0.579570    
#> 304: 0.448599    1.146102    3.528547    0.484922    0.014954    0.017444    0.579594    
#> 305: 0.448221    1.146024    3.528500    0.484338    0.014948    0.017409    0.579677    
#> 306: 0.447751    1.145988    3.528421    0.484004    0.014943    0.017366    0.579730    
#> 307: 0.447536    1.145857    3.528344    0.483873    0.014937    0.017346    0.579814    
#> 308: 0.446972    1.145770    3.528248    0.483623    0.014938    0.017338    0.579817    
#> 309: 0.446849    1.145489    3.528280    0.483747    0.014945    0.017342    0.579888    
#> 310: 0.446827    1.145504    3.528225    0.483485    0.014965    0.017335    0.579940    
#> 311: 0.446643    1.145562    3.528210    0.483277    0.014969    0.017337    0.580051    
#> 312: 0.446307    1.145545    3.528095    0.482351    0.014975    0.017351    0.580247    
#> 313: 0.446071    1.145367    3.527980    0.481841    0.014978    0.017391    0.580250    
#> 314: 0.445480    1.145390    3.527775    0.481928    0.014970    0.017435    0.580296    
#> 315: 0.445014    1.145384    3.527640    0.481471    0.014953    0.017466    0.580315    
#> 316: 0.445098    1.145334    3.527605    0.481353    0.014954    0.017482    0.580288    
#> 317: 0.444989    1.145266    3.527645    0.481721    0.014984    0.017514    0.580323    
#> 318: 0.444931    1.145350    3.527610    0.481566    0.015020    0.017529    0.580204    
#> 319: 0.444909    1.145370    3.527624    0.481897    0.015033    0.017548    0.580251    
#> 320: 0.444361    1.145449    3.527479    0.481612    0.015034    0.017560    0.580427    
#> 321: 0.444359    1.145478    3.527410    0.481645    0.015022    0.017588    0.580515    
#> 322: 0.444360    1.145490    3.527308    0.481419    0.015004    0.017606    0.580542    
#> 323: 0.443983    1.145495    3.527272    0.480948    0.014972    0.017614    0.580494    
#> 324: 0.444011    1.145583    3.527310    0.481262    0.014960    0.017634    0.580373    
#> 325: 0.443847    1.145750    3.527235    0.480925    0.014944    0.017650    0.580339    
#> 326: 0.443772    1.145651    3.527221    0.480846    0.014911    0.017659    0.580345    
#> 327: 0.443878    1.145660    3.527279    0.480605    0.014872    0.017687    0.580356    
#> 328: 0.443903    1.145752    3.527268    0.480860    0.014857    0.017695    0.580357    
#> 329: 0.443963    1.145792    3.527290    0.480716    0.014873    0.017685    0.580226    
#> 330: 0.443934    1.145862    3.527292    0.480503    0.014896    0.017656    0.580209    
#> 331: 0.444416    1.145876    3.527390    0.481323    0.014909    0.017616    0.580168    
#> 332: 0.444598    1.145769    3.527518    0.481657    0.014934    0.017580    0.580235    
#> 333: 0.444547    1.145743    3.527458    0.481647    0.014964    0.017544    0.580290    
#> 334: 0.444210    1.145716    3.527363    0.481791    0.014961    0.017552    0.580240    
#> 335: 0.444025    1.145727    3.527260    0.481578    0.014958    0.017536    0.580219    
#> 336: 0.443791    1.145846    3.527161    0.481442    0.014944    0.017561    0.580189    
#> 337: 0.443322    1.146085    3.526925    0.481943    0.014952    0.017569    0.580099    
#> 338: 0.442887    1.146244    3.526835    0.481727    0.014966    0.017580    0.580086    
#> 339: 0.442758    1.146349    3.526805    0.481833    0.014972    0.017602    0.580023    
#> 340: 0.442802    1.146305    3.526804    0.481788    0.014979    0.017608    0.579992    
#> 341: 0.442993    1.146276    3.526874    0.481987    0.014976    0.017605    0.579975    
#> 342: 0.442947    1.146124    3.526914    0.482050    0.014960    0.017613    0.579911    
#> 343: 0.442898    1.146087    3.526989    0.481778    0.014916    0.017620    0.579940    
#> 344: 0.443088    1.146035    3.527104    0.481832    0.014895    0.017638    0.579981    
#> 345: 0.443104    1.146035    3.527153    0.481655    0.014898    0.017651    0.579964    
#> 346: 0.443065    1.146159    3.527062    0.481740    0.014875    0.017673    0.579964    
#> 347: 0.443263    1.146058    3.527006    0.481873    0.014871    0.017703    0.580047    
#> 348: 0.442896    1.146142    3.526899    0.481431    0.014852    0.017725    0.580099    
#> 349: 0.442898    1.146228    3.526899    0.481487    0.014837    0.017740    0.580051    
#> 350: 0.442598    1.146279    3.526849    0.481708    0.014835    0.017736    0.579989    
#> 351: 0.442573    1.146473    3.526793    0.481946    0.014855    0.017721    0.579929    
#> 352: 0.442336    1.146656    3.526696    0.481753    0.014874    0.017730    0.580008    
#> 353: 0.442519    1.146837    3.526640    0.481747    0.014904    0.017703    0.580057    
#> 354: 0.442462    1.147004    3.526494    0.481913    0.014926    0.017721    0.580030    
#> 355: 0.442602    1.147030    3.526439    0.482209    0.014945    0.017722    0.579953    
#> 356: 0.442568    1.146994    3.526430    0.482582    0.014964    0.017713    0.580049    
#> 357: 0.442735    1.146924    3.526447    0.483029    0.014951    0.017710    0.580124    
#> 358: 0.442897    1.146931    3.526530    0.483684    0.014910    0.017740    0.580145    
#> 359: 0.442924    1.146999    3.526533    0.484148    0.014888    0.017780    0.580074    
#> 360: 0.442914    1.147041    3.526478    0.484008    0.014861    0.017830    0.579948    
#> 361: 0.442917    1.147137    3.526414    0.484128    0.014845    0.017859    0.579832    
#> 362: 0.442868    1.147189    3.526358    0.484247    0.014835    0.017867    0.579674    
#> 363: 0.442807    1.147160    3.526335    0.484187    0.014818    0.017889    0.579576    
#> 364: 0.442788    1.147124    3.526315    0.483759    0.014784    0.017926    0.579560    
#> 365: 0.442691    1.147097    3.526285    0.483931    0.014794    0.017990    0.579579    
#> 366: 0.442728    1.147072    3.526201    0.483738    0.014779    0.018026    0.579521    
#> 367: 0.442754    1.147058    3.526165    0.483950    0.014767    0.018027    0.579463    
#> 368: 0.442964    1.147104    3.526062    0.485063    0.014762    0.018020    0.579404    
#> 369: 0.443108    1.147091    3.526034    0.485455    0.014749    0.018017    0.579460    
#> 370: 0.442873    1.147156    3.525971    0.485134    0.014739    0.018012    0.579469    
#> 371: 0.442896    1.147177    3.525914    0.485094    0.014718    0.018019    0.579448    
#> 372: 0.442706    1.147202    3.525873    0.485411    0.014713    0.018031    0.579483    
#> 373: 0.442530    1.147351    3.525790    0.485801    0.014735    0.018034    0.579431    
#> 374: 0.442574    1.147444    3.525734    0.486121    0.014754    0.018040    0.579384    
#> 375: 0.442490    1.147518    3.525664    0.486507    0.014741    0.018035    0.579395    
#> 376: 0.442418    1.147550    3.525598    0.486411    0.014731    0.018045    0.579362    
#> 377: 0.442368    1.147623    3.525610    0.486388    0.014746    0.018024    0.579339    
#> 378: 0.442170    1.147638    3.525520    0.486294    0.014754    0.018004    0.579405    
#> 379: 0.442079    1.147524    3.525518    0.486399    0.014763    0.017998    0.579412    
#> 380: 0.441921    1.147472    3.525539    0.486226    0.014800    0.017991    0.579378    
#> 381: 0.441820    1.147401    3.525536    0.486300    0.014824    0.017973    0.579336    
#> 382: 0.441827    1.147357    3.525534    0.486934    0.014841    0.017979    0.579254    
#> 383: 0.441760    1.147332    3.525492    0.487121    0.014851    0.017993    0.579202    
#> 384: 0.441680    1.147368    3.525471    0.486862    0.014855    0.017983    0.579240    
#> 385: 0.441789    1.147435    3.525497    0.486531    0.014861    0.017955    0.579245    
#> 386: 0.442000    1.147550    3.525558    0.486236    0.014882    0.017940    0.579220    
#> 387: 0.442137    1.147532    3.525639    0.485827    0.014889    0.017938    0.579221    
#> 388: 0.442246    1.147507    3.525743    0.485505    0.014909    0.017914    0.579201    
#> 389: 0.442411    1.147344    3.525816    0.485374    0.014905    0.017889    0.579233    
#> 390: 0.442434    1.147296    3.525821    0.485396    0.014897    0.017878    0.579281    
#> 391: 0.442740    1.147241    3.525850    0.485383    0.014887    0.017914    0.579289    
#> 392: 0.442973    1.147189    3.525926    0.485748    0.014882    0.017934    0.579309    
#> 393: 0.443056    1.147120    3.525981    0.485815    0.014878    0.017934    0.579262    
#> 394: 0.443399    1.147051    3.526049    0.486373    0.014868    0.017949    0.579230    
#> 395: 0.443925    1.147028    3.526163    0.486956    0.014860    0.017953    0.579254    
#> 396: 0.444236    1.147059    3.526255    0.487432    0.014862    0.017969    0.579279    
#> 397: 0.444413    1.147026    3.526372    0.487356    0.014869    0.017979    0.579324    
#> 398: 0.444588    1.146920    3.526459    0.487277    0.014879    0.017975    0.579363    
#> 399: 0.444868    1.146847    3.526602    0.487265    0.014890    0.017970    0.579330    
#> 400: 0.445060    1.146868    3.526680    0.487199    0.014894    0.017965    0.579313    
#> 401: 0.445206    1.146930    3.526690    0.487302    0.014899    0.017946    0.579333    
#> 402: 0.445255    1.146975    3.526689    0.487051    0.014907    0.017930    0.579352    
#> 403: 0.445344    1.146994    3.526660    0.486874    0.014916    0.017913    0.579332    
#> 404: 0.445268    1.147019    3.526575    0.486939    0.014942    0.017900    0.579208    
#> 405: 0.445189    1.147083    3.526565    0.487104    0.014966    0.017887    0.579204    
#> 406: 0.445186    1.147078    3.526555    0.487219    0.014985    0.017894    0.579225    
#> 407: 0.445244    1.147117    3.526576    0.487182    0.014987    0.017892    0.579233    
#> 408: 0.445328    1.147184    3.526616    0.486770    0.014984    0.017881    0.579321    
#> 409: 0.445350    1.147192    3.526583    0.486560    0.014979    0.017857    0.579333    
#> 410: 0.445111    1.147229    3.526522    0.486355    0.014989    0.017854    0.579293    
#> 411: 0.444980    1.147154    3.526475    0.486235    0.014995    0.017841    0.579263    
#> 412: 0.444895    1.147199    3.526410    0.486167    0.015003    0.017829    0.579193    
#> 413: 0.444875    1.147234    3.526366    0.486369    0.015000    0.017818    0.579140    
#> 414: 0.444629    1.147209    3.526327    0.486015    0.014996    0.017850    0.579125    
#> 415: 0.444906    1.147177    3.526325    0.486199    0.014989    0.017863    0.579146    
#> 416: 0.445135    1.147154    3.526368    0.486757    0.014985    0.017860    0.579132    
#> 417: 0.445241    1.147114    3.526367    0.487030    0.015000    0.017855    0.579089    
#> 418: 0.445163    1.147129    3.526360    0.487113    0.014998    0.017857    0.579029    
#> 419: 0.445277    1.147172    3.526354    0.487252    0.014979    0.017851    0.578996    
#> 420: 0.445247    1.147221    3.526340    0.487040    0.014944    0.017851    0.578943    
#> 421: 0.445059    1.147286    3.526334    0.486676    0.014933    0.017862    0.578898    
#> 422: 0.445172    1.147275    3.526373    0.486366    0.014925    0.017870    0.578905    
#> 423: 0.445355    1.147196    3.526419    0.486550    0.014932    0.017871    0.578851    
#> 424: 0.445361    1.147146    3.526359    0.486722    0.014936    0.017889    0.578836    
#> 425: 0.445351    1.147164    3.526297    0.487509    0.014942    0.017915    0.578866    
#> 426: 0.445503    1.147230    3.526298    0.487890    0.014929    0.017927    0.578866    
#> 427: 0.445406    1.147303    3.526266    0.487933    0.014919    0.017946    0.578813    
#> 428: 0.445363    1.147347    3.526270    0.487970    0.014916    0.017946    0.578832    
#> 429: 0.445331    1.147323    3.526281    0.487898    0.014910    0.017941    0.578864    
#> 430: 0.445255    1.147299    3.526325    0.487737    0.014915    0.017950    0.578832    
#> 431: 0.445262    1.147318    3.526321    0.487606    0.014902    0.017952    0.578774    
#> 432: 0.445381    1.147387    3.526330    0.487984    0.014897    0.017964    0.578734    
#> 433: 0.445447    1.147516    3.526347    0.488205    0.014905    0.017974    0.578708    
#> 434: 0.445161    1.147566    3.526251    0.488153    0.014907    0.017980    0.578616    
#> 435: 0.445023    1.147634    3.526188    0.488121    0.014910    0.017987    0.578537    
#> 436: 0.445030    1.147680    3.526193    0.488172    0.014893    0.017978    0.578498    
#> 437: 0.444952    1.147747    3.526185    0.487767    0.014881    0.017978    0.578500    
#> 438: 0.445102    1.147877    3.526211    0.487588    0.014881    0.017963    0.578530    
#> 439: 0.445071    1.147939    3.526198    0.487413    0.014866    0.017961    0.578557    
#> 440: 0.445136    1.147972    3.526202    0.487283    0.014851    0.017958    0.578583    
#> 441: 0.445288    1.147952    3.526197    0.487198    0.014849    0.017954    0.578558    
#> 442: 0.445178    1.147961    3.526251    0.487019    0.014838    0.017948    0.578544    
#> 443: 0.445139    1.147967    3.526229    0.486888    0.014832    0.017948    0.578531    
#> 444: 0.445015    1.147988    3.526180    0.486779    0.014827    0.017950    0.578473    
#> 445: 0.444966    1.148024    3.526154    0.486838    0.014857    0.017943    0.578448    
#> 446: 0.444819    1.148089    3.526097    0.486915    0.014861    0.017942    0.578438    
#> 447: 0.444899    1.148108    3.526074    0.487085    0.014862    0.017940    0.578420    
#> 448: 0.444816    1.148140    3.526034    0.487086    0.014857    0.017947    0.578399    
#> 449: 0.444767    1.148213    3.526075    0.487109    0.014855    0.017955    0.578443    
#> 450: 0.444939    1.148207    3.526079    0.487449    0.014850    0.017950    0.578443    
#> 451: 0.445065    1.148245    3.526077    0.487683    0.014848    0.017950    0.578455    
#> 452: 0.444889    1.148341    3.526033    0.487771    0.014842    0.017954    0.578371    
#> 453: 0.444647    1.148400    3.525989    0.487694    0.014851    0.017947    0.578336    
#> 454: 0.444706    1.148506    3.525966    0.488211    0.014850    0.017940    0.578310    
#> 455: 0.444775    1.148540    3.525928    0.488864    0.014854    0.017933    0.578284    
#> 456: 0.444694    1.148554    3.525902    0.488993    0.014850    0.017938    0.578274    
#> 457: 0.444612    1.148569    3.525915    0.488841    0.014836    0.017932    0.578219    
#> 458: 0.444536    1.148646    3.525953    0.488712    0.014824    0.017939    0.578189    
#> 459: 0.444261    1.148730    3.525928    0.488589    0.014832    0.017953    0.578156    
#> 460: 0.444134    1.148788    3.525882    0.488685    0.014821    0.017974    0.578117    
#> 461: 0.443924    1.148850    3.525841    0.488500    0.014817    0.017986    0.578093    
#> 462: 0.443736    1.148916    3.525822    0.488503    0.014806    0.017984    0.578085    
#> 463: 0.443647    1.148953    3.525800    0.488394    0.014790    0.017980    0.578103    
#> 464: 0.443775    1.149009    3.525800    0.488322    0.014782    0.017972    0.578055    
#> 465: 0.443847    1.148970    3.525820    0.488244    0.014769    0.017979    0.577958    
#> 466: 0.444015    1.148985    3.525802    0.488577    0.014749    0.017988    0.577900    
#> 467: 0.443938    1.148966    3.525810    0.488883    0.014736    0.017995    0.577869    
#> 468: 0.444093    1.148969    3.525821    0.489457    0.014730    0.018002    0.577868    
#> 469: 0.444147    1.149026    3.525826    0.489606    0.014720    0.018018    0.577942    
#> 470: 0.444177    1.149021    3.525807    0.489976    0.014699    0.018038    0.577962    
#> 471: 0.444141    1.149015    3.525764    0.490095    0.014689    0.018040    0.577975    
#> 472: 0.444057    1.149020    3.525735    0.490195    0.014665    0.018045    0.577962    
#> 473: 0.443974    1.149079    3.525681    0.490391    0.014652    0.018055    0.577938    
#> 474: 0.443883    1.149080    3.525625    0.490469    0.014653    0.018063    0.577929    
#> 475: 0.443976    1.149128    3.525615    0.490405    0.014645    0.018063    0.577959    
#> 476: 0.443787    1.149116    3.525565    0.490210    0.014639    0.018058    0.578028    
#> 477: 0.443624    1.149126    3.525561    0.490129    0.014626    0.018049    0.578003    
#> 478: 0.443493    1.149111    3.525550    0.490194    0.014631    0.018039    0.578054    
#> 479: 0.443391    1.149148    3.525537    0.490146    0.014627    0.018030    0.578039    
#> 480: 0.443352    1.149169    3.525489    0.490519    0.014619    0.018024    0.578065    
#> 481: 0.443213    1.149199    3.525414    0.490645    0.014612    0.018024    0.578047    
#> 482: 0.443175    1.149233    3.525325    0.491053    0.014608    0.018034    0.578068    
#> 483: 0.443051    1.149262    3.525286    0.491043    0.014588    0.018040    0.578102    
#> 484: 0.443048    1.149210    3.525296    0.491090    0.014574    0.018050    0.578088    
#> 485: 0.443037    1.149221    3.525297    0.491282    0.014585    0.018059    0.578090    
#> 486: 0.442965    1.149288    3.525305    0.491366    0.014588    0.018063    0.578092    
#> 487: 0.442953    1.149377    3.525319    0.491468    0.014590    0.018077    0.578073    
#> 488: 0.442921    1.149349    3.525331    0.491638    0.014581    0.018097    0.578042    
#> 489: 0.442983    1.149308    3.525353    0.491560    0.014580    0.018115    0.578072    
#> 490: 0.443095    1.149340    3.525398    0.491471    0.014569    0.018135    0.578097    
#> 491: 0.443190    1.149406    3.525447    0.491561    0.014565    0.018152    0.578122    
#> 492: 0.443144    1.149462    3.525450    0.491422    0.014567    0.018170    0.578176    
#> 493: 0.443084    1.149482    3.525454    0.491506    0.014574    0.018188    0.578220    
#> 494: 0.443128    1.149552    3.525439    0.491571    0.014566    0.018190    0.578192    
#> 495: 0.443170    1.149586    3.525479    0.491439    0.014564    0.018200    0.578141    
#> 496: 0.443334    1.149644    3.525495    0.491537    0.014563    0.018205    0.578121    
#> 497: 0.443405    1.149654    3.525518    0.491613    0.014571    0.018199    0.578098    
#> 498: 0.443414    1.149671    3.525509    0.491523    0.014566    0.018190    0.578076    
#> 499: 0.443403    1.149693    3.525487    0.491627    0.014580    0.018186    0.578042    
#> 500: 0.443409    1.149689    3.525456    0.491844    0.014588    0.018182    0.578040    
#> [====|====|=>-------------------------------------]  25%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.285703    0.977533    3.437770    0.392401    0.073420    0.016800    1.752371    
#> 002: 0.251485    0.987421    3.473279    0.372781    0.070373    0.017127    1.241543    
#> 003: 0.370588    0.938761    3.468346    0.354142    0.066854    0.016271    0.964191    
#> 004: 0.294877    0.942215    3.476637    0.336435    0.063511    0.018901    0.861213    
#> 005: 0.360402    0.957368    3.474270    0.319613    0.074906    0.018120    0.791706    
#> 006: 0.336868    0.983111    3.469746    0.303633    0.081784    0.017214    0.751446    
#> 007: 0.286458    0.987992    3.457578    0.288451    0.090282    0.016353    0.758156    
#> 008: 0.320017    0.975105    3.456555    0.274029    0.085768    0.016646    0.750979    
#> 009: 0.292689    0.953067    3.457493    0.260327    0.081479    0.015813    0.744001    
#> 010: 0.287474    0.960488    3.454180    0.247311    0.077405    0.015023    0.755524    
#> 011: 0.344369    0.978682    3.462126    0.234945    0.081012    0.014803    0.773311    
#> 012: 0.342161    0.985384    3.454780    0.223198    0.081132    0.014063    0.755161    
#> 013: 0.280916    0.959230    3.456295    0.212038    0.077076    0.014893    0.765322    
#> 014: 0.344496    0.962023    3.470364    0.201436    0.073222    0.018877    0.776317    
#> 015: 0.352556    0.958923    3.478139    0.191364    0.090894    0.017934    0.735124    
#> 016: 0.340930    0.973705    3.487117    0.181796    0.105496    0.017037    0.743923    
#> 017: 0.357880    0.970374    3.488769    0.172706    0.106998    0.016185    0.731702    
#> 018: 0.371839    0.950730    3.486488    0.164071    0.101648    0.015376    0.750333    
#> 019: 0.337605    0.955369    3.480118    0.155867    0.099883    0.014725    0.743344    
#> 020: 0.354416    0.944042    3.485487    0.169644    0.110535    0.015342    0.737034    
#> 021: 0.338584    0.942103    3.487205    0.161162    0.122801    0.015065    0.723702    
#> 022: 0.369768    0.954254    3.500321    0.153104    0.116661    0.014617    0.735557    
#> 023: 0.435612    0.944162    3.510004    0.157243    0.110828    0.013890    0.753740    
#> 024: 0.435498    0.950872    3.514449    0.149381    0.108249    0.013195    0.769585    
#> 025: 0.402279    0.914109    3.507325    0.141912    0.102836    0.012535    0.758803    
#> 026: 0.380140    0.960830    3.493419    0.134816    0.097695    0.011909    0.763206    
#> 027: 0.345943    0.991285    3.476475    0.128984    0.096527    0.011313    0.767778    
#> 028: 0.370110    0.961834    3.483939    0.156493    0.111684    0.010748    0.737333    
#> 029: 0.352456    0.980805    3.472289    0.163557    0.114096    0.014056    0.749675    
#> 030: 0.372927    0.976088    3.470662    0.155380    0.108391    0.013353    0.741287    
#> 031: 0.352995    0.963051    3.485760    0.173748    0.102972    0.012685    0.728110    
#> 032: 0.343946    0.961333    3.480941    0.175902    0.100909    0.012051    0.723113    
#> 033: 0.351368    0.971717    3.473956    0.167107    0.104669    0.011449    0.738807    
#> 034: 0.368619    0.978366    3.480464    0.158752    0.099436    0.010876    0.735510    
#> 035: 0.339426    0.961355    3.471307    0.161852    0.094464    0.010332    0.734978    
#> 036: 0.310139    0.999939    3.460878    0.170345    0.101854    0.009816    0.758495    
#> 037: 0.299317    0.981836    3.461774    0.161827    0.112896    0.009325    0.752080    
#> 038: 0.341219    0.989776    3.452109    0.153736    0.113903    0.009265    0.759112    
#> 039: 0.342135    0.991559    3.460113    0.163314    0.108208    0.008802    0.755064    
#> 040: 0.294817    0.988652    3.466232    0.155148    0.115414    0.008496    0.752039    
#> 041: 0.280569    0.977644    3.463753    0.172567    0.110312    0.008071    0.743822    
#> 042: 0.341966    0.969119    3.466162    0.163939    0.105807    0.007667    0.741326    
#> 043: 0.309070    0.987439    3.457023    0.158189    0.105218    0.007542    0.729701    
#> 044: 0.269880    1.002954    3.454531    0.175294    0.111017    0.007796    0.749993    
#> 045: 0.288720    1.015485    3.442753    0.166530    0.108506    0.007407    0.765625    
#> 046: 0.314918    0.988681    3.450440    0.176689    0.104704    0.007036    0.751677    
#> 047: 0.336900    0.990518    3.444908    0.175214    0.118914    0.008358    0.733608    
#> 048: 0.337373    0.976649    3.448102    0.175344    0.112968    0.007940    0.747971    
#> 049: 0.313796    0.990643    3.450329    0.211287    0.107319    0.007543    0.741079    
#> 050: 0.352376    0.955256    3.456588    0.200723    0.115002    0.009902    0.766319    
#> 051: 0.353456    0.986472    3.453780    0.190686    0.115454    0.010587    0.777560    
#> 052: 0.346719    0.998997    3.461061    0.181152    0.110021    0.010058    0.730820    
#> 053: 0.342845    0.990099    3.462636    0.172094    0.115816    0.009984    0.767889    
#> 054: 0.273990    0.990432    3.458702    0.163490    0.112204    0.012176    0.761971    
#> 055: 0.291931    0.983087    3.456092    0.186500    0.125594    0.011568    0.761182    
#> 056: 0.300706    0.957021    3.466100    0.189873    0.128790    0.010989    0.779394    
#> 057: 0.310279    0.935492    3.484204    0.180379    0.127182    0.010440    0.785129    
#> 058: 0.349043    0.939942    3.470290    0.184289    0.121420    0.009918    0.777280    
#> 059: 0.340560    0.913494    3.477712    0.175075    0.121409    0.009422    0.770547    
#> 060: 0.341793    0.928220    3.481271    0.166321    0.119828    0.011937    0.765794    
#> 061: 0.381562    0.940488    3.488033    0.158005    0.113837    0.014851    0.768680    
#> 062: 0.361317    0.949741    3.491996    0.150105    0.110597    0.017002    0.776195    
#> 063: 0.382989    0.937472    3.486983    0.142599    0.105067    0.016152    0.786655    
#> 064: 0.377524    0.956262    3.488225    0.144910    0.109172    0.015344    0.748138    
#> 065: 0.380455    0.930162    3.483723    0.149359    0.106757    0.014577    0.753875    
#> 066: 0.395271    0.950741    3.487531    0.162921    0.101419    0.014691    0.749158    
#> 067: 0.350659    0.949121    3.481319    0.154775    0.101190    0.013956    0.754511    
#> 068: 0.313681    0.954607    3.480644    0.149228    0.096131    0.013259    0.777211    
#> 069: 0.340530    0.948181    3.472619    0.162400    0.098379    0.012596    0.771011    
#> 070: 0.356118    0.955681    3.487786    0.154280    0.131302    0.011966    0.757143    
#> 071: 0.334019    0.970082    3.468288    0.146566    0.124737    0.011368    0.756105    
#> 072: 0.274179    0.964331    3.457117    0.162534    0.118500    0.010799    0.755091    
#> 073: 0.333344    0.968187    3.475501    0.154407    0.112575    0.010259    0.742712    
#> 074: 0.311631    0.962943    3.474800    0.155113    0.112375    0.009746    0.739083    
#> 075: 0.328714    0.949271    3.485011    0.152901    0.110952    0.009444    0.726512    
#> 076: 0.377646    0.967742    3.501164    0.184279    0.108899    0.008972    0.734746    
#> 077: 0.379376    0.931999    3.487968    0.176905    0.117449    0.008523    0.732765    
#> 078: 0.379521    0.977264    3.490799    0.168060    0.117424    0.010282    0.729878    
#> 079: 0.376173    0.976187    3.498977    0.159657    0.136833    0.014173    0.754758    
#> 080: 0.368643    0.959763    3.487002    0.151674    0.129991    0.013465    0.792402    
#> 081: 0.368601    0.979690    3.475445    0.146817    0.123492    0.012791    0.767999    
#> 082: 0.368112    0.959660    3.483747    0.151442    0.117317    0.012152    0.760819    
#> 083: 0.367664    1.001591    3.486095    0.148085    0.111451    0.011544    0.735796    
#> 084: 0.367708    0.991281    3.477393    0.140681    0.116001    0.010967    0.733979    
#> 085: 0.343801    0.981317    3.486815    0.138090    0.112983    0.010661    0.752907    
#> 086: 0.343763    0.980159    3.474513    0.145479    0.107334    0.011380    0.738933    
#> 087: 0.320235    0.985615    3.459962    0.162680    0.101967    0.010811    0.726852    
#> 088: 0.255417    0.967711    3.457429    0.165402    0.096869    0.013222    0.761546    
#> 089: 0.304233    0.986868    3.450905    0.170037    0.092025    0.014177    0.743665    
#> 090: 0.293873    0.986232    3.449935    0.161535    0.103920    0.013468    0.754274    
#> 091: 0.276872    0.987200    3.436653    0.153459    0.107934    0.012795    0.768540    
#> 092: 0.274356    1.011064    3.433172    0.157593    0.123338    0.012155    0.763542    
#> 093: 0.268397    1.000306    3.434977    0.175280    0.117171    0.011547    0.747810    
#> 094: 0.249326    1.008422    3.436290    0.196195    0.111599    0.010970    0.757699    
#> 095: 0.285923    1.002991    3.452749    0.186385    0.109510    0.010421    0.757246    
#> 096: 0.280465    0.981337    3.454764    0.177066    0.104034    0.011709    0.759351    
#> 097: 0.319649    1.007947    3.453413    0.191704    0.105990    0.011124    0.732524    
#> 098: 0.313330    0.994487    3.461110    0.212866    0.103844    0.010568    0.752942    
#> 099: 0.313077    0.972962    3.449567    0.202222    0.110730    0.010039    0.762283    
#> 100: 0.297735    0.972175    3.458473    0.192111    0.105193    0.010066    0.737448    
#> 101: 0.260247    0.975246    3.443597    0.182506    0.099934    0.010141    0.741399    
#> 102: 0.268104    0.994636    3.437741    0.174067    0.097368    0.011028    0.740318    
#> 103: 0.319974    1.004773    3.457027    0.187613    0.119891    0.010477    0.717166    
#> 104: 0.296148    0.999044    3.461352    0.178232    0.127764    0.009953    0.751616    
#> 105: 0.305714    0.991276    3.452658    0.169321    0.130727    0.009455    0.741524    
#> 106: 0.325520    0.988883    3.447812    0.160854    0.124191    0.008982    0.734888    
#> 107: 0.338664    0.996497    3.454195    0.152812    0.117981    0.008533    0.755455    
#> 108: 0.300787    0.981500    3.444175    0.145171    0.112082    0.008107    0.761314    
#> 109: 0.264940    1.004239    3.453276    0.148021    0.106478    0.007701    0.749998    
#> 110: 0.284513    0.990081    3.436151    0.184528    0.127587    0.007316    0.736610    
#> 111: 0.274267    0.982369    3.444115    0.177654    0.121207    0.006950    0.741532    
#> 112: 0.279929    0.962405    3.448562    0.175163    0.131960    0.006603    0.747082    
#> 113: 0.295681    0.986314    3.448911    0.210497    0.143196    0.006273    0.737536    
#> 114: 0.302566    0.965919    3.455750    0.207775    0.154988    0.005959    0.743695    
#> 115: 0.346799    0.954779    3.458859    0.220794    0.147238    0.005661    0.754241    
#> 116: 0.345254    1.002724    3.463008    0.209754    0.139876    0.006111    0.752778    
#> 117: 0.334522    0.969886    3.470579    0.199266    0.132883    0.007336    0.744835    
#> 118: 0.320645    0.950529    3.471920    0.189303    0.127433    0.006969    0.753151    
#> 119: 0.335622    0.950730    3.470496    0.179838    0.121061    0.007428    0.754963    
#> 120: 0.327816    0.952915    3.471613    0.170846    0.115008    0.008586    0.746797    
#> 121: 0.333923    0.983784    3.469890    0.162304    0.109258    0.008157    0.763977    
#> 122: 0.331690    0.952639    3.472934    0.154188    0.103795    0.007749    0.755904    
#> 123: 0.380659    0.983169    3.471374    0.156447    0.098605    0.008306    0.739674    
#> 124: 0.375405    1.004332    3.468752    0.148624    0.095360    0.009136    0.754422    
#> 125: 0.374698    1.018049    3.456216    0.141193    0.105336    0.010114    0.756536    
#> 126: 0.318342    1.006843    3.444139    0.134133    0.100069    0.010922    0.756256    
#> 127: 0.319000    0.990213    3.441438    0.138825    0.095975    0.010988    0.763563    
#> 128: 0.297730    0.983977    3.464489    0.139583    0.125530    0.011856    0.751669    
#> 129: 0.314641    0.997953    3.460348    0.138795    0.121998    0.011263    0.761260    
#> 130: 0.332500    0.998818    3.458120    0.131855    0.124652    0.011588    0.727488    
#> 131: 0.329414    0.959995    3.454671    0.145181    0.118420    0.013154    0.757457    
#> 132: 0.290496    0.995206    3.436844    0.140475    0.112499    0.012497    0.769078    
#> 133: 0.302047    1.001060    3.444051    0.149520    0.106874    0.011872    0.754289    
#> 134: 0.323337    0.973735    3.461113    0.165828    0.104322    0.011278    0.749786    
#> 135: 0.316132    0.962438    3.462944    0.163850    0.142155    0.010714    0.743812    
#> 136: 0.330600    0.950304    3.467516    0.157149    0.135048    0.010179    0.734946    
#> 137: 0.356689    0.959489    3.469835    0.182660    0.128295    0.009670    0.735110    
#> 138: 0.340239    0.982173    3.467612    0.177827    0.124103    0.009186    0.742716    
#> 139: 0.372199    0.965688    3.483518    0.168936    0.117898    0.009327    0.751913    
#> 140: 0.353124    0.972462    3.483141    0.161248    0.138115    0.008861    0.757886    
#> 141: 0.323467    0.949067    3.477702    0.153186    0.149984    0.008418    0.746798    
#> 142: 0.349510    0.946089    3.482242    0.160933    0.142485    0.007997    0.750941    
#> 143: 0.327715    0.969999    3.480179    0.152887    0.135360    0.008474    0.748992    
#> 144: 0.405543    0.956999    3.502744    0.168361    0.128592    0.008233    0.739223    
#> 145: 0.391672    0.965477    3.484244    0.163763    0.122163    0.008878    0.743576    
#> 146: 0.345666    0.960622    3.475872    0.155575    0.118156    0.008434    0.753888    
#> 147: 0.315491    0.941998    3.469241    0.152170    0.112248    0.008636    0.737614    
#> 148: 0.331789    0.965973    3.478179    0.158771    0.116676    0.009256    0.736278    
#> 149: 0.342226    0.968492    3.476704    0.172590    0.110842    0.008793    0.749708    
#> 150: 0.342058    0.969634    3.462395    0.188981    0.116455    0.008353    0.739681    
#> 151: 0.306768    0.986081    3.461212    0.179532    0.114559    0.007935    0.742857    
#> 152: 0.302447    0.973049    3.468967    0.166293    0.114605    0.007080    0.745171    
#> 153: 0.275406    0.966450    3.467464    0.176449    0.124013    0.008259    0.762823    
#> 154: 0.346722    0.978265    3.473040    0.161452    0.117799    0.008627    0.748848    
#> 155: 0.328024    0.969848    3.471776    0.164444    0.137290    0.008085    0.747675    
#> 156: 0.326070    0.952958    3.487683    0.209524    0.144324    0.007865    0.746242    
#> 157: 0.364962    0.934664    3.488353    0.167914    0.138898    0.006346    0.758724    
#> 158: 0.395727    0.971774    3.488022    0.149503    0.108067    0.007472    0.740917    
#> 159: 0.346737    0.964706    3.489747    0.139731    0.119915    0.007065    0.742616    
#> 160: 0.376433    0.928869    3.495068    0.143277    0.098974    0.007859    0.755509    
#> 161: 0.372647    0.951010    3.490462    0.158690    0.114956    0.007931    0.748858    
#> 162: 0.358083    0.964713    3.471052    0.181844    0.113281    0.008627    0.729852    
#> 163: 0.367556    0.971731    3.467201    0.167801    0.114244    0.009842    0.742506    
#> 164: 0.330431    0.973833    3.468864    0.156509    0.118747    0.011576    0.742360    
#> 165: 0.367987    0.944031    3.481676    0.185125    0.115934    0.011343    0.750037    
#> 166: 0.387067    0.996216    3.483751    0.154473    0.108995    0.006518    0.746917    
#> 167: 0.362016    1.003169    3.478239    0.151900    0.120812    0.007504    0.748394    
#> 168: 0.369896    0.969699    3.479759    0.175211    0.129902    0.007886    0.757543    
#> 169: 0.342548    1.022983    3.473686    0.187688    0.136838    0.008104    0.769390    
#> 170: 0.321686    1.005603    3.456538    0.185152    0.127334    0.008182    0.745360    
#> 171: 0.285976    0.997992    3.455568    0.187990    0.110971    0.008268    0.734950    
#> 172: 0.287932    1.002960    3.447217    0.187796    0.099917    0.006688    0.729937    
#> 173: 0.298994    1.003839    3.441279    0.138025    0.106324    0.008378    0.752469    
#> 174: 0.290779    1.007463    3.443468    0.194894    0.111121    0.008249    0.740346    
#> 175: 0.268619    1.022125    3.431917    0.183476    0.112073    0.009123    0.750116    
#> 176: 0.299030    1.015211    3.451067    0.182576    0.100150    0.010396    0.757393    
#> 177: 0.322984    0.980834    3.467252    0.148484    0.100458    0.013122    0.717782    
#> 178: 0.346859    0.966482    3.467227    0.168704    0.111768    0.009644    0.739589    
#> 179: 0.362990    0.988107    3.461864    0.163609    0.116646    0.009151    0.751681    
#> 180: 0.315622    0.997137    3.458458    0.214558    0.112773    0.008446    0.751074    
#> 181: 0.301839    1.006821    3.437416    0.188869    0.114120    0.007110    0.733943    
#> 182: 0.253527    1.010412    3.427261    0.224600    0.120941    0.006695    0.744244    
#> 183: 0.292003    1.028879    3.426884    0.227689    0.112469    0.007350    0.741534    
#> 184: 0.293260    1.010258    3.429706    0.233463    0.124658    0.006439    0.734472    
#> 185: 0.272338    1.024996    3.430626    0.205958    0.136665    0.005972    0.745856    
#> 186: 0.290624    1.025062    3.436168    0.181778    0.109890    0.005940    0.755710    
#> 187: 0.290390    0.992977    3.443728    0.162831    0.111968    0.006667    0.742590    
#> 188: 0.266345    1.001809    3.438815    0.168924    0.124964    0.007035    0.768909    
#> 189: 0.293317    1.010249    3.442835    0.157062    0.109652    0.007950    0.758581    
#> 190: 0.282991    1.007108    3.447552    0.199057    0.122798    0.007230    0.731900    
#> 191: 0.259096    0.996676    3.441718    0.150463    0.115783    0.006990    0.752628    
#> 192: 0.286461    1.013445    3.452796    0.142725    0.130342    0.006103    0.736423    
#> 193: 0.302777    1.005741    3.451344    0.130180    0.117507    0.006596    0.728569    
#> 194: 0.302853    0.987416    3.444304    0.121330    0.098020    0.008396    0.750682    
#> 195: 0.283720    0.980243    3.450511    0.136981    0.114899    0.008555    0.731727    
#> 196: 0.292907    0.978208    3.461650    0.154251    0.132112    0.006914    0.745082    
#> 197: 0.284139    0.969642    3.473724    0.162594    0.116520    0.006590    0.765632    
#> 198: 0.344514    0.963557    3.467880    0.117598    0.128898    0.006388    0.742862    
#> 199: 0.317642    0.976782    3.466714    0.132825    0.101848    0.008325    0.748552    
#> 200: 0.321937    0.984284    3.465853    0.132931    0.113315    0.007140    0.765072    
#> 201: 0.321506    0.975686    3.465683    0.139913    0.113032    0.007104    0.759407    
#> 202: 0.320035    0.977871    3.465594    0.134386    0.111469    0.007846    0.753291    
#> 203: 0.326821    0.978147    3.463408    0.135284    0.107163    0.008278    0.752747    
#> 204: 0.333176    0.972494    3.464319    0.137969    0.107281    0.008461    0.752934    
#> 205: 0.329595    0.980121    3.464068    0.143863    0.105932    0.008828    0.753706    
#> 206: 0.327189    0.984667    3.462601    0.148055    0.106042    0.008821    0.757053    
#> 207: 0.323138    0.982269    3.461253    0.150097    0.107517    0.008707    0.756921    
#> 208: 0.322344    0.983621    3.460685    0.157038    0.108155    0.008551    0.754704    
#> 209: 0.323460    0.981529    3.460043    0.160139    0.109793    0.008671    0.753571    
#> 210: 0.323812    0.979245    3.459781    0.160513    0.109705    0.008753    0.751847    
#> 211: 0.323342    0.980482    3.459493    0.160257    0.110579    0.008755    0.750521    
#> 212: 0.324947    0.979056    3.460257    0.160559    0.110851    0.008764    0.748913    
#> 213: 0.327625    0.975417    3.461391    0.165956    0.109812    0.008770    0.748612    
#> 214: 0.329246    0.974435    3.461801    0.166029    0.108821    0.008815    0.748931    
#> 215: 0.329114    0.974939    3.461973    0.165881    0.108684    0.008862    0.748336    
#> 216: 0.330131    0.973880    3.461708    0.164903    0.109308    0.008854    0.748739    
#> 217: 0.330633    0.975064    3.461446    0.163430    0.110204    0.008960    0.748237    
#> 218: 0.329605    0.976202    3.460718    0.163828    0.110489    0.008981    0.747169    
#> 219: 0.328230    0.975715    3.459783    0.164944    0.109983    0.008906    0.746970    
#> 220: 0.326652    0.976261    3.459681    0.165690    0.109863    0.008749    0.746918    
#> 221: 0.325581    0.976991    3.459637    0.165569    0.110784    0.008618    0.746987    
#> 222: 0.325087    0.977057    3.459368    0.165936    0.111871    0.008505    0.748393    
#> 223: 0.326393    0.977634    3.459049    0.166700    0.112430    0.008441    0.747924    
#> 224: 0.327920    0.978954    3.459248    0.165979    0.113400    0.008431    0.747849    
#> 225: 0.327440    0.978042    3.459511    0.165614    0.113787    0.008458    0.747485    
#> 226: 0.327118    0.977580    3.459288    0.164731    0.113487    0.008445    0.747431    
#> 227: 0.326057    0.977892    3.459380    0.165114    0.113709    0.008363    0.747595    
#> 228: 0.324996    0.978793    3.459043    0.165086    0.113706    0.008310    0.747970    
#> 229: 0.324119    0.979055    3.458802    0.165622    0.113375    0.008270    0.747417    
#> 230: 0.323461    0.978835    3.458586    0.166161    0.113190    0.008257    0.746950    
#> 231: 0.323658    0.979413    3.458974    0.167219    0.113796    0.008255    0.746843    
#> 232: 0.321917    0.979894    3.459086    0.167280    0.113838    0.008268    0.747264    
#> 233: 0.321670    0.979434    3.459067    0.166181    0.114234    0.008286    0.747175    
#> 234: 0.320947    0.978687    3.459113    0.165548    0.114215    0.008312    0.747361    
#> 235: 0.321371    0.979103    3.458741    0.165528    0.114694    0.008271    0.747712    
#> 236: 0.322154    0.979537    3.458715    0.166337    0.115143    0.008353    0.748293    
#> 237: 0.323353    0.979677    3.458913    0.167929    0.114908    0.008444    0.748650    
#> 238: 0.323840    0.979530    3.458934    0.168386    0.114767    0.008455    0.748936    
#> 239: 0.323484    0.979826    3.458958    0.168544    0.114504    0.008507    0.748798    
#> 240: 0.323647    0.979549    3.459131    0.167681    0.114132    0.008527    0.748728    
#> 241: 0.324385    0.978566    3.459666    0.166510    0.113916    0.008558    0.748746    
#> 242: 0.324321    0.977912    3.459782    0.166493    0.114137    0.008595    0.749104    
#> 243: 0.323968    0.978669    3.459893    0.165756    0.114126    0.008636    0.749544    
#> 244: 0.323851    0.978372    3.460101    0.165591    0.114152    0.008674    0.749719    
#> 245: 0.323423    0.977752    3.460281    0.165858    0.113931    0.008683    0.749912    
#> 246: 0.323546    0.977235    3.460588    0.165557    0.113779    0.008736    0.750198    
#> 247: 0.323009    0.976927    3.460906    0.165882    0.114158    0.008755    0.749970    
#> 248: 0.322664    0.977015    3.460918    0.165847    0.114606    0.008767    0.750052    
#> 249: 0.321896    0.977409    3.460838    0.166353    0.115270    0.008722    0.749848    
#> 250: 0.322446    0.977573    3.460946    0.166708    0.115379    0.008712    0.750018    
#> 251: 0.322716    0.978052    3.461217    0.167556    0.115818    0.008708    0.750221    
#> 252: 0.323121    0.978151    3.461533    0.168089    0.116161    0.008739    0.750431    
#> 253: 0.323481    0.978054    3.461856    0.168392    0.115918    0.008777    0.750114    
#> 254: 0.323085    0.978504    3.462081    0.168004    0.115765    0.008782    0.749763    
#> 255: 0.322902    0.979000    3.462307    0.167848    0.115671    0.008781    0.749560    
#> 256: 0.323023    0.978941    3.462175    0.167564    0.115615    0.008799    0.749490    
#> 257: 0.322446    0.979029    3.462003    0.167500    0.115633    0.008817    0.749587    
#> 258: 0.322221    0.978755    3.461973    0.167342    0.115429    0.008838    0.749652    
#> 259: 0.322243    0.978969    3.461888    0.167303    0.115181    0.008839    0.749686    
#> 260: 0.322245    0.979697    3.461991    0.167165    0.115185    0.008844    0.749642    
#> 261: 0.322305    0.979855    3.462023    0.167281    0.115228    0.008875    0.749428    
#> 262: 0.322662    0.980109    3.462084    0.166990    0.115073    0.008911    0.749590    
#> 263: 0.322453    0.980366    3.462131    0.166723    0.115302    0.008973    0.749634    
#> 264: 0.321861    0.980584    3.461975    0.166326    0.115153    0.008963    0.750253    
#> 265: 0.321593    0.980710    3.461818    0.165755    0.115244    0.008998    0.750405    
#> 266: 0.322033    0.980991    3.461772    0.165504    0.115184    0.009020    0.750498    
#> 267: 0.322237    0.980904    3.461793    0.165298    0.115105    0.009038    0.750553    
#> 268: 0.322312    0.980416    3.461606    0.165234    0.114919    0.009027    0.750753    
#> 269: 0.322684    0.980330    3.461560    0.165285    0.114921    0.009014    0.751018    
#> 270: 0.323168    0.980264    3.461463    0.165306    0.114997    0.008973    0.750700    
#> 271: 0.323201    0.980119    3.461306    0.165587    0.114801    0.008975    0.750875    
#> 272: 0.322886    0.980029    3.461117    0.166238    0.114572    0.008973    0.751040    
#> 273: 0.322865    0.980157    3.461102    0.166220    0.114498    0.008955    0.751344    
#> 274: 0.322558    0.980010    3.461253    0.166384    0.114916    0.008971    0.751372    
#> 275: 0.322342    0.980256    3.461135    0.166661    0.114852    0.008977    0.751313    
#> 276: 0.322113    0.980205    3.461275    0.166572    0.114813    0.008993    0.751154    
#> 277: 0.322025    0.980310    3.461371    0.166644    0.114910    0.009009    0.750868    
#> 278: 0.321995    0.980381    3.461509    0.166416    0.115005    0.009009    0.750786    
#> 279: 0.321844    0.980085    3.461692    0.166110    0.114996    0.008997    0.750916    
#> 280: 0.321834    0.980263    3.461758    0.165754    0.114867    0.009012    0.750659    
#> 281: 0.321740    0.980183    3.461910    0.165554    0.114598    0.009030    0.750517    
#> 282: 0.321699    0.980258    3.461965    0.165392    0.114374    0.009040    0.750339    
#> 283: 0.321881    0.980162    3.462007    0.165401    0.114235    0.009049    0.750562    
#> 284: 0.321678    0.980160    3.462031    0.165057    0.114215    0.009052    0.750446    
#> 285: 0.321996    0.980215    3.462027    0.165253    0.114212    0.009056    0.750577    
#> 286: 0.321999    0.980427    3.461974    0.165504    0.114190    0.009050    0.750757    
#> 287: 0.321990    0.980620    3.461887    0.165695    0.114115    0.009048    0.750983    
#> 288: 0.322413    0.980686    3.461858    0.165950    0.114320    0.009057    0.750941    
#> 289: 0.322787    0.980701    3.461895    0.166333    0.114230    0.009061    0.750879    
#> 290: 0.322982    0.980552    3.461929    0.166458    0.114081    0.009079    0.750917    
#> 291: 0.323358    0.980748    3.462149    0.166266    0.114052    0.009129    0.750695    
#> 292: 0.323393    0.980380    3.462354    0.166165    0.113973    0.009147    0.750772    
#> 293: 0.323288    0.980140    3.462358    0.166207    0.113973    0.009144    0.750703    
#> 294: 0.323126    0.980067    3.462332    0.166229    0.113955    0.009143    0.750539    
#> 295: 0.323089    0.979636    3.462358    0.166369    0.114010    0.009150    0.750552    
#> 296: 0.323285    0.979421    3.462458    0.166200    0.113842    0.009191    0.750687    
#> 297: 0.323707    0.979234    3.462562    0.166333    0.113670    0.009239    0.750728    
#> 298: 0.323808    0.979055    3.462513    0.166150    0.113445    0.009268    0.750686    
#> 299: 0.324005    0.979000    3.462549    0.166254    0.113458    0.009269    0.750665    
#> 300: 0.324258    0.978798    3.462718    0.166130    0.113440    0.009301    0.750491    
#> 301: 0.324481    0.978680    3.462701    0.165775    0.113601    0.009317    0.750571    
#> 302: 0.324344    0.978843    3.462617    0.165676    0.113621    0.009321    0.750423    
#> 303: 0.324290    0.978957    3.462603    0.165650    0.113696    0.009299    0.750200    
#> 304: 0.324363    0.979294    3.462444    0.165434    0.113717    0.009292    0.750163    
#> 305: 0.324039    0.979003    3.462401    0.165368    0.113658    0.009287    0.750221    
#> 306: 0.323599    0.979054    3.462251    0.165516    0.113575    0.009279    0.750235    
#> 307: 0.323762    0.979056    3.462171    0.165601    0.113468    0.009272    0.750197    
#> 308: 0.323564    0.979018    3.462080    0.165773    0.113560    0.009267    0.750102    
#> 309: 0.323361    0.978823    3.461979    0.165815    0.113578    0.009288    0.750269    
#> 310: 0.323306    0.978760    3.461808    0.166087    0.113615    0.009275    0.750128    
#> 311: 0.322926    0.978835    3.461677    0.166121    0.113569    0.009272    0.749963    
#> 312: 0.322685    0.978917    3.461548    0.166123    0.113622    0.009295    0.749919    
#> 313: 0.322360    0.978700    3.461393    0.166373    0.113858    0.009322    0.750054    
#> 314: 0.321844    0.978740    3.461309    0.166006    0.113956    0.009331    0.750138    
#> 315: 0.321272    0.978976    3.461192    0.165907    0.114014    0.009349    0.750187    
#> 316: 0.321261    0.979032    3.461060    0.165741    0.113916    0.009361    0.750224    
#> 317: 0.321234    0.978992    3.461081    0.165689    0.113754    0.009356    0.750162    
#> 318: 0.321094    0.979129    3.460976    0.165830    0.113687    0.009338    0.749945    
#> 319: 0.321090    0.979217    3.460940    0.165894    0.113618    0.009343    0.749850    
#> 320: 0.321011    0.979301    3.460872    0.165981    0.113523    0.009353    0.749867    
#> 321: 0.320870    0.979340    3.460740    0.166313    0.113407    0.009360    0.749969    
#> 322: 0.320804    0.979440    3.460644    0.166138    0.113324    0.009348    0.750050    
#> 323: 0.320290    0.979605    3.460508    0.165986    0.113198    0.009333    0.750097    
#> 324: 0.320174    0.979872    3.460456    0.165604    0.113260    0.009340    0.750005    
#> 325: 0.319806    0.980119    3.460300    0.165440    0.113171    0.009350    0.749866    
#> 326: 0.319725    0.979913    3.460239    0.165244    0.113148    0.009358    0.749844    
#> 327: 0.319920    0.980016    3.460282    0.165049    0.113250    0.009379    0.749867    
#> 328: 0.320047    0.980284    3.460342    0.164801    0.113257    0.009382    0.749836    
#> 329: 0.320349    0.980330    3.460439    0.164786    0.113306    0.009399    0.749794    
#> 330: 0.320515    0.980263    3.460520    0.164847    0.113281    0.009409    0.749790    
#> 331: 0.320748    0.980266    3.460601    0.164844    0.113282    0.009407    0.749682    
#> 332: 0.320832    0.980176    3.460642    0.164646    0.113233    0.009399    0.749660    
#> 333: 0.320896    0.980192    3.460663    0.164732    0.113303    0.009392    0.749591    
#> 334: 0.320813    0.980043    3.460617    0.164794    0.113451    0.009373    0.749385    
#> 335: 0.320617    0.980156    3.460514    0.164637    0.113677    0.009367    0.749355    
#> 336: 0.320236    0.980282    3.460416    0.164575    0.113875    0.009370    0.749539    
#> 337: 0.320275    0.980452    3.460296    0.164670    0.113925    0.009361    0.749590    
#> 338: 0.320377    0.980369    3.460316    0.164657    0.113935    0.009364    0.749559    
#> 339: 0.320291    0.980352    3.460333    0.164621    0.113972    0.009380    0.749580    
#> 340: 0.320285    0.980072    3.460459    0.164581    0.114011    0.009377    0.749481    
#> 341: 0.320589    0.980065    3.460527    0.164581    0.114088    0.009371    0.749551    
#> 342: 0.320431    0.979874    3.460552    0.164465    0.114126    0.009374    0.749437    
#> 343: 0.320402    0.979860    3.460627    0.164623    0.114028    0.009367    0.749422    
#> 344: 0.320542    0.979668    3.460672    0.164755    0.114075    0.009387    0.749411    
#> 345: 0.320703    0.979614    3.460711    0.164861    0.113975    0.009409    0.749317    
#> 346: 0.320745    0.979660    3.460669    0.165014    0.113893    0.009418    0.749115    
#> 347: 0.320716    0.979730    3.460574    0.164836    0.113852    0.009425    0.749185    
#> 348: 0.320575    0.979833    3.460537    0.164830    0.113798    0.009437    0.749208    
#> 349: 0.320592    0.979797    3.460515    0.164917    0.113739    0.009454    0.749159    
#> 350: 0.320454    0.979667    3.460462    0.164760    0.113625    0.009448    0.749207    
#> 351: 0.320433    0.979912    3.460402    0.164775    0.113489    0.009438    0.749331    
#> 352: 0.320032    0.980141    3.460366    0.164931    0.113415    0.009439    0.749376    
#> 353: 0.320006    0.980335    3.460308    0.165130    0.113375    0.009431    0.749261    
#> 354: 0.319910    0.980467    3.460179    0.165279    0.113365    0.009442    0.749128    
#> 355: 0.320120    0.980464    3.460106    0.165417    0.113446    0.009449    0.749140    
#> 356: 0.320066    0.980427    3.460159    0.165519    0.113437    0.009457    0.749292    
#> 357: 0.320144    0.980555    3.460173    0.165512    0.113489    0.009456    0.749255    
#> 358: 0.320211    0.980617    3.460239    0.165481    0.113666    0.009446    0.749141    
#> 359: 0.320322    0.980691    3.460275    0.165486    0.113641    0.009457    0.749107    
#> 360: 0.320298    0.980629    3.460235    0.165451    0.113554    0.009478    0.749051    
#> 361: 0.320112    0.980684    3.460168    0.165572    0.113519    0.009477    0.749062    
#> 362: 0.320151    0.980572    3.460137    0.165610    0.113506    0.009470    0.748956    
#> 363: 0.320199    0.980570    3.460106    0.165506    0.113568    0.009471    0.748849    
#> 364: 0.320225    0.980592    3.460064    0.165477    0.113664    0.009473    0.748800    
#> 365: 0.320280    0.980546    3.460098    0.165473    0.113700    0.009476    0.748851    
#> 366: 0.320364    0.980597    3.460054    0.165674    0.113606    0.009468    0.748920    
#> 367: 0.320375    0.980624    3.460078    0.165708    0.113553    0.009466    0.748961    
#> 368: 0.320464    0.980750    3.460033    0.165878    0.113485    0.009472    0.748838    
#> 369: 0.320447    0.980733    3.459973    0.165883    0.113530    0.009471    0.748890    
#> 370: 0.320221    0.980774    3.459910    0.165779    0.113595    0.009478    0.748950    
#> 371: 0.320176    0.980895    3.459801    0.165579    0.113615    0.009472    0.749055    
#> 372: 0.320094    0.981054    3.459740    0.165383    0.113676    0.009462    0.749173    
#> 373: 0.319885    0.981293    3.459688    0.165305    0.113715    0.009454    0.749242    
#> 374: 0.319888    0.981416    3.459658    0.165333    0.113739    0.009441    0.749190    
#> 375: 0.319736    0.981483    3.459629    0.165406    0.113762    0.009425    0.749255    
#> 376: 0.319548    0.981521    3.459632    0.165249    0.113765    0.009420    0.749339    
#> 377: 0.319432    0.981509    3.459631    0.165197    0.113701    0.009408    0.749233    
#> 378: 0.319347    0.981519    3.459595    0.165078    0.113713    0.009401    0.749235    
#> 379: 0.319246    0.981403    3.459584    0.164879    0.113707    0.009395    0.749292    
#> 380: 0.319093    0.981408    3.459556    0.164835    0.113675    0.009394    0.749209    
#> 381: 0.319045    0.981383    3.459592    0.164876    0.113617    0.009396    0.749105    
#> 382: 0.319147    0.981314    3.459614    0.164904    0.113547    0.009405    0.748950    
#> 383: 0.319295    0.981402    3.459590    0.164851    0.113617    0.009408    0.748912    
#> 384: 0.319157    0.981386    3.459606    0.164963    0.113539    0.009412    0.748954    
#> 385: 0.319211    0.981391    3.459631    0.164862    0.113468    0.009410    0.749024    
#> 386: 0.319244    0.981569    3.459656    0.164739    0.113432    0.009407    0.749115    
#> 387: 0.319448    0.981669    3.459749    0.164788    0.113355    0.009420    0.749082    
#> 388: 0.319642    0.981665    3.459829    0.164765    0.113327    0.009428    0.749015    
#> 389: 0.319720    0.981539    3.459871    0.164753    0.113344    0.009434    0.748988    
#> 390: 0.319600    0.981644    3.459850    0.164872    0.113347    0.009450    0.748932    
#> 391: 0.319600    0.981620    3.459854    0.164911    0.113289    0.009466    0.748926    
#> 392: 0.319483    0.981593    3.459858    0.164890    0.113245    0.009477    0.748977    
#> 393: 0.319470    0.981518    3.459816    0.164795    0.113259    0.009489    0.748912    
#> 394: 0.319344    0.981489    3.459771    0.164844    0.113272    0.009488    0.748922    
#> 395: 0.319492    0.981554    3.459749    0.164895    0.113244    0.009482    0.748951    
#> 396: 0.319597    0.981659    3.459759    0.164825    0.113248    0.009475    0.748920    
#> 397: 0.319567    0.981704    3.459761    0.164808    0.113181    0.009478    0.748938    
#> 398: 0.319549    0.981665    3.459751    0.164946    0.113219    0.009474    0.748970    
#> 399: 0.319584    0.981708    3.459792    0.164896    0.113158    0.009468    0.748919    
#> 400: 0.319688    0.981731    3.459804    0.164843    0.113159    0.009471    0.748897    
#> 401: 0.319712    0.981758    3.459826    0.164711    0.113198    0.009469    0.748843    
#> 402: 0.319745    0.981953    3.459825    0.164565    0.113093    0.009459    0.748897    
#> 403: 0.319897    0.981857    3.459818    0.164477    0.113142    0.009442    0.749005    
#> 404: 0.320022    0.981833    3.459795    0.164441    0.113098    0.009427    0.748953    
#> 405: 0.320099    0.981910    3.459861    0.164534    0.113139    0.009421    0.748879    
#> 406: 0.320186    0.981901    3.459912    0.164481    0.113088    0.009420    0.748959    
#> 407: 0.320241    0.981993    3.459971    0.164419    0.113055    0.009428    0.748937    
#> 408: 0.320234    0.982041    3.459997    0.164394    0.113025    0.009427    0.748962    
#> 409: 0.320173    0.981952    3.460011    0.164311    0.113059    0.009423    0.748879    
#> 410: 0.320190    0.981955    3.459969    0.164172    0.113052    0.009412    0.748908    
#> 411: 0.320309    0.981774    3.459971    0.163950    0.113131    0.009408    0.749076    
#> 412: 0.320190    0.981778    3.459922    0.163843    0.113178    0.009395    0.749084    
#> 413: 0.320101    0.981771    3.459858    0.163965    0.113183    0.009384    0.749013    
#> 414: 0.319909    0.981814    3.459832    0.163915    0.113110    0.009382    0.748992    
#> 415: 0.319999    0.981763    3.459822    0.163868    0.113040    0.009380    0.748950    
#> 416: 0.320114    0.981832    3.459837    0.163867    0.113085    0.009374    0.748897    
#> 417: 0.320213    0.981761    3.459860    0.163937    0.113077    0.009369    0.748866    
#> 418: 0.320091    0.981800    3.459873    0.164020    0.113109    0.009366    0.748805    
#> 419: 0.320209    0.981892    3.459889    0.163988    0.113051    0.009374    0.748786    
#> 420: 0.320350    0.981951    3.459925    0.164018    0.113106    0.009372    0.748687    
#> 421: 0.320411    0.981957    3.459964    0.164044    0.113142    0.009373    0.748590    
#> 422: 0.320705    0.981802    3.460024    0.163991    0.113185    0.009370    0.748660    
#> 423: 0.320968    0.981698    3.460115    0.164047    0.113122    0.009380    0.748752    
#> 424: 0.321122    0.981663    3.460148    0.164136    0.113070    0.009388    0.748748    
#> 425: 0.321344    0.981773    3.460177    0.164302    0.113080    0.009386    0.748859    
#> 426: 0.321401    0.981876    3.460185    0.164555    0.113075    0.009376    0.748858    
#> 427: 0.321301    0.981850    3.460156    0.164720    0.112999    0.009374    0.748828    
#> 428: 0.321041    0.981928    3.460120    0.164783    0.112930    0.009367    0.748858    
#> 429: 0.320872    0.981979    3.460088    0.164856    0.112957    0.009352    0.748846    
#> 430: 0.320786    0.982095    3.460068    0.165042    0.112938    0.009354    0.748775    
#> 431: 0.320581    0.982076    3.460015    0.165143    0.112881    0.009360    0.748754    
#> 432: 0.320527    0.982186    3.459977    0.165303    0.112867    0.009368    0.748718    
#> 433: 0.320539    0.982210    3.459989    0.165383    0.112836    0.009371    0.748683    
#> 434: 0.320412    0.982230    3.459941    0.165368    0.112807    0.009364    0.748616    
#> 435: 0.320380    0.982275    3.459897    0.165393    0.112788    0.009359    0.748542    
#> 436: 0.320302    0.982171    3.459904    0.165380    0.112869    0.009352    0.748605    
#> 437: 0.320242    0.981999    3.459932    0.165451    0.112800    0.009352    0.748589    
#> 438: 0.320167    0.982102    3.459951    0.165644    0.112782    0.009346    0.748505    
#> 439: 0.320230    0.982175    3.459968    0.165709    0.112802    0.009343    0.748457    
#> 440: 0.320149    0.982219    3.459989    0.165729    0.112778    0.009338    0.748408    
#> 441: 0.320148    0.982216    3.460003    0.165637    0.112792    0.009332    0.748361    
#> 442: 0.320128    0.982295    3.459993    0.165517    0.112742    0.009331    0.748282    
#> 443: 0.319980    0.982371    3.459933    0.165471    0.112807    0.009324    0.748283    
#> 444: 0.319983    0.982444    3.459876    0.165398    0.112951    0.009318    0.748275    
#> 445: 0.319916    0.982541    3.459811    0.165221    0.112976    0.009325    0.748253    
#> 446: 0.319681    0.982657    3.459710    0.165098    0.113040    0.009323    0.748226    
#> 447: 0.319597    0.982707    3.459659    0.165007    0.113106    0.009323    0.748229    
#> 448: 0.319432    0.982769    3.459652    0.164942    0.113118    0.009313    0.748205    
#> 449: 0.319324    0.982833    3.459712    0.164911    0.113175    0.009306    0.748191    
#> 450: 0.319371    0.982852    3.459729    0.164863    0.113150    0.009300    0.748178    
#> 451: 0.319488    0.982825    3.459744    0.164804    0.113182    0.009298    0.748219    
#> 452: 0.319312    0.982924    3.459675    0.164769    0.113195    0.009295    0.748291    
#> 453: 0.319042    0.983002    3.459580    0.164806    0.113198    0.009292    0.748216    
#> 454: 0.318905    0.983174    3.459510    0.164903    0.113205    0.009283    0.748193    
#> 455: 0.318809    0.983274    3.459440    0.164962    0.113228    0.009277    0.748250    
#> 456: 0.318687    0.983308    3.459378    0.165055    0.113193    0.009264    0.748259    
#> 457: 0.318570    0.983277    3.459344    0.165161    0.113129    0.009254    0.748217    
#> 458: 0.318504    0.983364    3.459396    0.165296    0.113163    0.009260    0.748162    
#> 459: 0.318407    0.983442    3.459400    0.165299    0.113223    0.009276    0.748142    
#> 460: 0.318370    0.983324    3.459386    0.165307    0.113184    0.009283    0.748116    
#> 461: 0.318249    0.983304    3.459407    0.165352    0.113211    0.009283    0.748080    
#> 462: 0.318214    0.983322    3.459451    0.165427    0.113219    0.009281    0.748068    
#> 463: 0.318209    0.983368    3.459476    0.165560    0.113197    0.009273    0.747999    
#> 464: 0.318319    0.983320    3.459491    0.165607    0.113224    0.009257    0.747935    
#> 465: 0.318292    0.983145    3.459513    0.165716    0.113258    0.009247    0.747952    
#> 466: 0.318436    0.983173    3.459504    0.165778    0.113288    0.009236    0.747962    
#> 467: 0.318441    0.983129    3.459513    0.165875    0.113263    0.009222    0.747918    
#> 468: 0.318580    0.983119    3.459536    0.165969    0.113249    0.009211    0.747955    
#> 469: 0.318621    0.983220    3.459543    0.165908    0.113241    0.009201    0.748047    
#> 470: 0.318652    0.983211    3.459581    0.165841    0.113286    0.009192    0.748020    
#> 471: 0.318656    0.983126    3.459627    0.165789    0.113305    0.009190    0.748032    
#> 472: 0.318673    0.983044    3.459696    0.165752    0.113320    0.009194    0.747978    
#> 473: 0.318698    0.982955    3.459751    0.165726    0.113313    0.009198    0.748042    
#> 474: 0.318758    0.982784    3.459773    0.165693    0.113267    0.009199    0.748177    
#> 475: 0.318877    0.982747    3.459787    0.165648    0.113239    0.009190    0.748152    
#> 476: 0.319032    0.982686    3.459807    0.165620    0.113276    0.009182    0.748186    
#> 477: 0.318998    0.982634    3.459854    0.165606    0.113352    0.009174    0.748117    
#> 478: 0.318939    0.982613    3.459854    0.165554    0.113382    0.009168    0.748131    
#> 479: 0.318996    0.982623    3.459852    0.165485    0.113504    0.009160    0.748137    
#> 480: 0.319089    0.982526    3.459895    0.165538    0.113508    0.009147    0.748204    
#> 481: 0.319136    0.982569    3.459919    0.165488    0.113537    0.009137    0.748218    
#> 482: 0.319309    0.982588    3.459935    0.165403    0.113582    0.009127    0.748242    
#> 483: 0.319347    0.982588    3.459936    0.165287    0.113592    0.009116    0.748248    
#> 484: 0.319353    0.982515    3.459946    0.165211    0.113597    0.009108    0.748340    
#> 485: 0.319415    0.982518    3.459936    0.165129    0.113594    0.009104    0.748423    
#> 486: 0.319462    0.982542    3.459965    0.165065    0.113582    0.009102    0.748404    
#> 487: 0.319698    0.982520    3.460052    0.165056    0.113564    0.009101    0.748363    
#> 488: 0.319858    0.982565    3.460122    0.164940    0.113623    0.009098    0.748352    
#> 489: 0.319998    0.982521    3.460177    0.164873    0.113640    0.009104    0.748362    
#> 490: 0.320095    0.982529    3.460224    0.164854    0.113671    0.009105    0.748345    
#> 491: 0.320153    0.982589    3.460280    0.164808    0.113666    0.009106    0.748361    
#> 492: 0.320148    0.982655    3.460310    0.164673    0.113666    0.009109    0.748489    
#> 493: 0.320080    0.982662    3.460334    0.164670    0.113723    0.009106    0.748590    
#> 494: 0.320131    0.982690    3.460328    0.164622    0.113644    0.009099    0.748607    
#> 495: 0.320267    0.982617    3.460372    0.164641    0.113614    0.009094    0.748625    
#> 496: 0.320419    0.982623    3.460369    0.164777    0.113614    0.009088    0.748639    
#> 497: 0.320300    0.982635    3.460366    0.164741    0.113539    0.009088    0.748712    
#> 498: 0.320350    0.982595    3.460374    0.164777    0.113580    0.009076    0.748677    
#> 499: 0.320414    0.982547    3.460349    0.164756    0.113645    0.009069    0.748657    
#> 500: 0.320415    0.982526    3.460333    0.164729    0.113669    0.009062    0.748617    
#> [====|====|=>-------------------------------------]  25%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.239263    0.965823    3.447834    0.392401    0.080836    0.016800    1.729700    
#> 002: 0.215852    1.060072    3.479413    0.372781    0.076795    0.015960    1.103679    
#> 003: 0.335298    1.043900    3.515980    0.354142    0.072955    0.015162    0.846487    
#> 004: 0.303474    1.074896    3.519251    0.336435    0.076039    0.014404    0.767727    
#> 005: 0.319838    1.096924    3.499673    0.319613    0.073924    0.013684    0.713149    
#> 006: 0.310673    1.109767    3.494590    0.303633    0.091568    0.013000    0.680211    
#> 007: 0.294424    1.136370    3.490190    0.288451    0.097008    0.012350    0.675690    
#> 008: 0.288254    1.100875    3.484809    0.274029    0.099901    0.011732    0.682310    
#> 009: 0.262387    1.069820    3.494661    0.260327    0.100562    0.011919    0.691440    
#> 010: 0.301960    1.072290    3.495552    0.252366    0.098256    0.013146    0.694571    
#> 011: 0.344277    1.089358    3.511343    0.239748    0.093343    0.012498    0.699547    
#> 012: 0.319822    1.077080    3.501856    0.250175    0.088676    0.016384    0.674423    
#> 013: 0.282440    1.072748    3.494411    0.237666    0.085058    0.020752    0.675545    
#> 014: 0.340685    1.067582    3.509584    0.234457    0.090647    0.019714    0.691400    
#> 015: 0.356440    1.074286    3.522613    0.222734    0.086115    0.019321    0.655536    
#> 016: 0.347586    1.102811    3.528767    0.211597    0.091242    0.018355    0.672269    
#> 017: 0.322350    1.110310    3.516406    0.201018    0.091816    0.018442    0.656699    
#> 018: 0.350716    1.104008    3.506481    0.190967    0.087225    0.017520    0.668624    
#> 019: 0.322464    1.099524    3.509597    0.205977    0.082864    0.017613    0.664345    
#> 020: 0.322767    1.109589    3.504543    0.195678    0.082773    0.021959    0.681318    
#> 021: 0.336035    1.091565    3.515686    0.185894    0.078634    0.023449    0.662446    
#> 022: 0.341900    1.095974    3.528171    0.203993    0.074703    0.022277    0.666139    
#> 023: 0.399977    1.093112    3.535656    0.196708    0.081908    0.021163    0.673300    
#> 024: 0.414118    1.116520    3.525973    0.196946    0.077813    0.020105    0.684015    
#> 025: 0.375598    1.093468    3.525116    0.195131    0.085987    0.019099    0.660920    
#> 026: 0.352150    1.103746    3.516622    0.185374    0.081996    0.020247    0.676108    
#> 027: 0.295059    1.109484    3.511492    0.188427    0.080964    0.019234    0.684567    
#> 028: 0.357186    1.099319    3.514288    0.179005    0.076916    0.018403    0.666620    
#> 029: 0.373882    1.085997    3.516000    0.193420    0.073070    0.019646    0.693253    
#> 030: 0.354120    1.114033    3.514582    0.226171    0.069417    0.023336    0.666871    
#> 031: 0.328561    1.095281    3.520631    0.214863    0.066634    0.022169    0.665910    
#> 032: 0.343896    1.078006    3.531602    0.204120    0.068041    0.021475    0.664093    
#> 033: 0.349871    1.100065    3.523374    0.212174    0.071979    0.021084    0.661873    
#> 034: 0.366170    1.106679    3.524850    0.219980    0.068380    0.022510    0.669000    
#> 035: 0.323034    1.080276    3.517090    0.211253    0.078674    0.021384    0.647961    
#> 036: 0.350052    1.095685    3.520227    0.200691    0.081864    0.020315    0.658419    
#> 037: 0.332352    1.105369    3.511296    0.190656    0.090148    0.019299    0.682359    
#> 038: 0.368963    1.086927    3.514821    0.209181    0.085640    0.018334    0.690595    
#> 039: 0.371657    1.085876    3.532343    0.198722    0.094389    0.017418    0.661848    
#> 040: 0.357231    1.086046    3.526509    0.205452    0.092637    0.016547    0.673733    
#> 041: 0.334907    1.072983    3.523159    0.212771    0.088005    0.015719    0.661568    
#> 042: 0.368327    1.080161    3.525309    0.202132    0.084432    0.016185    0.654629    
#> 043: 0.329319    1.104778    3.511554    0.192026    0.088811    0.015376    0.663488    
#> 044: 0.292772    1.093717    3.508311    0.185088    0.086951    0.014607    0.666215    
#> 045: 0.275149    1.130161    3.490070    0.190799    0.094399    0.013877    0.669115    
#> 046: 0.303733    1.101476    3.508084    0.181259    0.094912    0.013183    0.672426    
#> 047: 0.336850    1.116795    3.497036    0.184454    0.090167    0.012524    0.653256    
#> 048: 0.331488    1.082300    3.505690    0.175232    0.091162    0.016904    0.674596    
#> 049: 0.313543    1.088729    3.515928    0.166470    0.086604    0.016746    0.699366    
#> 050: 0.382627    1.058806    3.521681    0.198043    0.086004    0.016398    0.711849    
#> 051: 0.348572    1.070422    3.526214    0.188141    0.109550    0.018683    0.694148    
#> 052: 0.335191    1.102959    3.522042    0.195121    0.104072    0.017749    0.670736    
#> 053: 0.345279    1.062674    3.525041    0.185365    0.110062    0.016861    0.694411    
#> 054: 0.307717    1.084941    3.521183    0.176096    0.104559    0.016586    0.707342    
#> 055: 0.310796    1.087101    3.510020    0.190705    0.099331    0.018294    0.707576    
#> 056: 0.323676    1.068790    3.511058    0.213357    0.095021    0.017379    0.686064    
#> 057: 0.341078    1.053595    3.526541    0.221000    0.097662    0.016510    0.686221    
#> 058: 0.319237    1.071359    3.510228    0.209950    0.092779    0.017910    0.685706    
#> 059: 0.328907    1.044572    3.518511    0.199452    0.091649    0.020645    0.689903    
#> 060: 0.339366    1.065570    3.521075    0.189479    0.105808    0.020400    0.699398    
#> 061: 0.367886    1.079360    3.527792    0.226294    0.106294    0.019380    0.700271    
#> 062: 0.338310    1.097528    3.536668    0.214979    0.115006    0.018411    0.715734    
#> 063: 0.385426    1.053112    3.534773    0.204230    0.109256    0.017490    0.720410    
#> 064: 0.377292    1.069688    3.538906    0.194019    0.103793    0.021135    0.696577    
#> 065: 0.371973    1.052351    3.526492    0.184318    0.101208    0.020079    0.691353    
#> 066: 0.371873    1.096685    3.525512    0.175102    0.096147    0.019075    0.669359    
#> 067: 0.333636    1.082958    3.509946    0.218057    0.091340    0.020505    0.664919    
#> 068: 0.325129    1.077112    3.525310    0.212841    0.105078    0.021304    0.692271    
#> 069: 0.299133    1.096937    3.508916    0.258150    0.099843    0.020239    0.683107    
#> 070: 0.343176    1.092906    3.525132    0.245243    0.117160    0.019227    0.672996    
#> 071: 0.300204    1.116326    3.491254    0.232981    0.122617    0.018266    0.670171    
#> 072: 0.281621    1.095825    3.490139    0.221332    0.116486    0.017352    0.662500    
#> 073: 0.319141    1.107485    3.505269    0.228353    0.110662    0.016656    0.653843    
#> 074: 0.291152    1.080098    3.504250    0.228772    0.105129    0.015824    0.667594    
#> 075: 0.332717    1.060230    3.530767    0.217333    0.099872    0.015032    0.672354    
#> 076: 0.360593    1.095885    3.539958    0.206466    0.103411    0.014370    0.677047    
#> 077: 0.403296    1.066099    3.537397    0.196143    0.098241    0.014628    0.669519    
#> 078: 0.397098    1.075045    3.537299    0.218068    0.093329    0.016522    0.657847    
#> 079: 0.364777    1.102714    3.518344    0.234620    0.103092    0.015696    0.679836    
#> 080: 0.354588    1.073729    3.525050    0.245119    0.097938    0.016557    0.692015    
#> 081: 0.357067    1.095523    3.516943    0.232863    0.093041    0.015729    0.678409    
#> 082: 0.355226    1.085702    3.527113    0.221220    0.088389    0.016936    0.670652    
#> 083: 0.362211    1.104638    3.528563    0.221160    0.083969    0.017683    0.658703    
#> 084: 0.344544    1.094878    3.527097    0.210102    0.086235    0.016799    0.659537    
#> 085: 0.356895    1.111034    3.529714    0.199597    0.097031    0.016690    0.663040    
#> 086: 0.343415    1.098432    3.512711    0.189617    0.092179    0.015855    0.659315    
#> 087: 0.315948    1.112311    3.500050    0.194093    0.087570    0.016092    0.666701    
#> 088: 0.292751    1.110763    3.503373    0.224312    0.087379    0.016400    0.703183    
#> 089: 0.292094    1.124202    3.488579    0.213097    0.111132    0.015580    0.689361    
#> 090: 0.302266    1.098559    3.495599    0.214465    0.105576    0.014801    0.681658    
#> 091: 0.281835    1.088159    3.481516    0.246589    0.100297    0.015031    0.682524    
#> 092: 0.281522    1.110611    3.477272    0.235985    0.095282    0.015026    0.686754    
#> 093: 0.250420    1.106244    3.486076    0.224186    0.090518    0.017019    0.689959    
#> 094: 0.282722    1.123439    3.482884    0.212976    0.086726    0.020449    0.669733    
#> 095: 0.286382    1.125690    3.498633    0.202328    0.095970    0.020758    0.679037    
#> 096: 0.268553    1.112012    3.492258    0.206241    0.100481    0.019720    0.681720    
#> 097: 0.329728    1.115985    3.500368    0.195929    0.095457    0.018734    0.678160    
#> 098: 0.318719    1.145166    3.516481    0.202385    0.096208    0.017798    0.688628    
#> 099: 0.291306    1.103905    3.491194    0.195592    0.114969    0.016908    0.654112    
#> 100: 0.312756    1.117246    3.496655    0.185812    0.109221    0.016062    0.658729    
#> 101: 0.287749    1.101089    3.486727    0.176522    0.103760    0.015259    0.667175    
#> 102: 0.282540    1.094486    3.483180    0.167696    0.098572    0.014496    0.673340    
#> 103: 0.315014    1.101007    3.497165    0.176015    0.093643    0.013771    0.659164    
#> 104: 0.301747    1.089015    3.512527    0.184183    0.107198    0.013083    0.673622    
#> 105: 0.358705    1.080196    3.523349    0.188790    0.101838    0.012429    0.683951    
#> 106: 0.381375    1.069728    3.527688    0.191515    0.100756    0.011837    0.672371    
#> 107: 0.390449    1.086748    3.523679    0.196655    0.097530    0.014457    0.686289    
#> 108: 0.350477    1.067494    3.524602    0.190696    0.102800    0.013735    0.693935    
#> 109: 0.310302    1.096363    3.527226    0.199926    0.097660    0.013048    0.692421    
#> 110: 0.332498    1.104728    3.496249    0.189930    0.092777    0.012395    0.691010    
#> 111: 0.305298    1.102457    3.497116    0.180433    0.088138    0.011776    0.678763    
#> 112: 0.296900    1.100595    3.489681    0.189395    0.083731    0.013297    0.673752    
#> 113: 0.318946    1.128920    3.499266    0.179925    0.086402    0.013366    0.655282    
#> 114: 0.322702    1.111157    3.505900    0.174724    0.082082    0.014023    0.669241    
#> 115: 0.369633    1.083159    3.514516    0.178614    0.080993    0.013322    0.710213    
#> 116: 0.337560    1.118464    3.516866    0.178968    0.083553    0.017261    0.700874    
#> 117: 0.307898    1.126996    3.513253    0.171697    0.079375    0.017215    0.710018    
#> 118: 0.327664    1.117799    3.514442    0.199415    0.078867    0.016930    0.692407    
#> 119: 0.331204    1.116211    3.503765    0.239935    0.086431    0.016084    0.688990    
#> 120: 0.320738    1.108632    3.508114    0.227938    0.088951    0.015280    0.666566    
#> 121: 0.330270    1.133597    3.501922    0.216541    0.103844    0.014516    0.685314    
#> 122: 0.336055    1.121643    3.507461    0.205714    0.101018    0.013790    0.655516    
#> 123: 0.356843    1.126604    3.504511    0.196113    0.099330    0.013100    0.666774    
#> 124: 0.350567    1.134007    3.508183    0.190017    0.103239    0.012445    0.667520    
#> 125: 0.367316    1.133937    3.508564    0.180516    0.098077    0.013578    0.675754    
#> 126: 0.295152    1.116805    3.495890    0.171490    0.093174    0.013812    0.685870    
#> 127: 0.307042    1.116413    3.488302    0.162916    0.088569    0.016694    0.690360    
#> 128: 0.306645    1.097501    3.508716    0.155239    0.094870    0.015913    0.687912    
#> 129: 0.333331    1.126820    3.508242    0.191517    0.096994    0.021189    0.697533    
#> 130: 0.355631    1.124869    3.507740    0.203682    0.096101    0.020238    0.663815    
#> 131: 0.334301    1.096315    3.508670    0.199447    0.098725    0.019226    0.673841    
#> 132: 0.287124    1.099887    3.487013    0.206313    0.093789    0.018265    0.673168    
#> 133: 0.278261    1.089827    3.492391    0.195998    0.098621    0.017351    0.702789    
#> 134: 0.354010    1.063549    3.521021    0.217981    0.093690    0.016484    0.676853    
#> 135: 0.332703    1.073521    3.520422    0.214763    0.095990    0.015660    0.671776    
#> 136: 0.320761    1.090965    3.515685    0.204025    0.117347    0.014877    0.656636    
#> 137: 0.384359    1.072592    3.523673    0.199104    0.111480    0.015710    0.661644    
#> 138: 0.342579    1.099345    3.508944    0.196950    0.105906    0.016680    0.652405    
#> 139: 0.359524    1.102656    3.528834    0.187102    0.100611    0.015846    0.668465    
#> 140: 0.369857    1.109376    3.525641    0.202448    0.096667    0.020485    0.676837    
#> 141: 0.348947    1.067773    3.525462    0.192936    0.099505    0.019461    0.659703    
#> 142: 0.357299    1.070904    3.536054    0.183289    0.094530    0.018487    0.673253    
#> 143: 0.325977    1.098416    3.523737    0.204033    0.089803    0.026204    0.677228    
#> 144: 0.359646    1.081483    3.530397    0.193832    0.085313    0.024894    0.646577    
#> 145: 0.334576    1.092815    3.508411    0.189123    0.082016    0.023649    0.652144    
#> 146: 0.320723    1.102294    3.513755    0.179689    0.077915    0.022467    0.662166    
#> 147: 0.307848    1.092027    3.510176    0.197947    0.081774    0.021683    0.642788    
#> 148: 0.316039    1.099576    3.518101    0.197966    0.084581    0.023797    0.659109    
#> 149: 0.329208    1.088550    3.514582    0.188068    0.082916    0.022607    0.690742    
#> 150: 0.342771    1.071349    3.511291    0.180477    0.078770    0.021477    0.689809    
#> 151: 0.307686    1.115880    3.497327    0.173511    0.088027    0.020403    0.688273    
#> 152: 0.294699    1.100536    3.507885    0.192346    0.073071    0.016167    0.659189    
#> 153: 0.297219    1.120586    3.515700    0.194718    0.085352    0.016589    0.681493    
#> 154: 0.369841    1.095360    3.524895    0.194612    0.082496    0.014314    0.701684    
#> 155: 0.334243    1.063793    3.519982    0.197610    0.083207    0.013210    0.694147    
#> 156: 0.358616    1.070761    3.520229    0.194920    0.119687    0.011012    0.697026    
#> 157: 0.372719    1.065663    3.535534    0.205385    0.131255    0.008679    0.677880    
#> 158: 0.391962    1.092792    3.527173    0.194802    0.103452    0.007983    0.681356    
#> 159: 0.328023    1.079678    3.526151    0.188211    0.090783    0.011124    0.684150    
#> 160: 0.365087    1.060510    3.539845    0.188426    0.099210    0.013037    0.686372    
#> 161: 0.350418    1.082110    3.529160    0.210706    0.101061    0.012823    0.677967    
#> 162: 0.325209    1.082338    3.507557    0.207823    0.109269    0.011367    0.697284    
#> 163: 0.334331    1.081982    3.508376    0.246028    0.124771    0.011881    0.668627    
#> 164: 0.337277    1.083284    3.516996    0.192287    0.101174    0.011493    0.691090    
#> 165: 0.386536    1.065694    3.532192    0.203210    0.094965    0.016124    0.680854    
#> 166: 0.381154    1.095967    3.539347    0.150424    0.099810    0.016342    0.673166    
#> 167: 0.381818    1.086058    3.535405    0.140016    0.095165    0.016278    0.684328    
#> 168: 0.353428    1.095238    3.526361    0.181278    0.105163    0.012650    0.682719    
#> 169: 0.335055    1.133649    3.510832    0.196154    0.101800    0.009683    0.708478    
#> 170: 0.339149    1.103259    3.509227    0.188176    0.115166    0.008004    0.682968    
#> 171: 0.303138    1.103394    3.504118    0.161983    0.097622    0.008173    0.666744    
#> 172: 0.312126    1.114483    3.502948    0.169281    0.108868    0.009231    0.672669    
#> 173: 0.297305    1.102799    3.496781    0.156317    0.108929    0.009469    0.675525    
#> 174: 0.309920    1.100816    3.504955    0.147818    0.102634    0.010164    0.681606    
#> 175: 0.296006    1.107124    3.492141    0.189163    0.112413    0.009802    0.683385    
#> 176: 0.302238    1.117944    3.501763    0.184931    0.087045    0.008080    0.687109    
#> 177: 0.309876    1.105879    3.510100    0.162669    0.098367    0.009963    0.680750    
#> 178: 0.340738    1.082095    3.513506    0.139852    0.107743    0.010539    0.679890    
#> 179: 0.343149    1.132516    3.505178    0.156187    0.102674    0.010990    0.675138    
#> 180: 0.311219    1.108503    3.510119    0.141000    0.091819    0.011690    0.682861    
#> 181: 0.315003    1.111548    3.497122    0.132970    0.088446    0.013306    0.669884    
#> 182: 0.281705    1.122396    3.490455    0.122931    0.103605    0.015716    0.693775    
#> 183: 0.304877    1.125525    3.493478    0.096833    0.095067    0.016118    0.691410    
#> 184: 0.295456    1.126807    3.500407    0.112322    0.094276    0.016048    0.691812    
#> 185: 0.266724    1.130017    3.489565    0.118592    0.087387    0.013745    0.712856    
#> 186: 0.285197    1.139626    3.493698    0.110697    0.084563    0.018977    0.701814    
#> 187: 0.303683    1.117151    3.508707    0.128580    0.095441    0.014411    0.692610    
#> 188: 0.269277    1.134255    3.491335    0.156893    0.105373    0.010867    0.701493    
#> 189: 0.312191    1.128881    3.498351    0.159611    0.111403    0.009887    0.707302    
#> 190: 0.301849    1.112158    3.504788    0.137157    0.103934    0.011886    0.696534    
#> 191: 0.275244    1.111990    3.494132    0.142374    0.121285    0.011540    0.708913    
#> 192: 0.314971    1.124358    3.508706    0.175630    0.101428    0.011220    0.684295    
#> 193: 0.363709    1.106171    3.512721    0.195794    0.089673    0.012393    0.665840    
#> 194: 0.329157    1.097741    3.503794    0.199686    0.109028    0.012934    0.684814    
#> 195: 0.304017    1.090980    3.503050    0.180124    0.121610    0.014351    0.683341    
#> 196: 0.304208    1.102571    3.510934    0.154118    0.111824    0.012489    0.690135    
#> 197: 0.293069    1.099973    3.532890    0.119192    0.115632    0.012546    0.703249    
#> 198: 0.364455    1.087987    3.521166    0.129714    0.100829    0.014308    0.682861    
#> 199: 0.319787    1.082677    3.515737    0.154210    0.092980    0.015247    0.687043    
#> 200: 0.326970    1.096207    3.514566    0.145705    0.106301    0.013774    0.697055    
#> 201: 0.324267    1.079168    3.522869    0.159800    0.095662    0.016956    0.691954    
#> 202: 0.330726    1.077194    3.519658    0.177181    0.093962    0.016208    0.680289    
#> 203: 0.344676    1.076279    3.516930    0.186935    0.091083    0.016359    0.679350    
#> 204: 0.348164    1.072565    3.518329    0.186772    0.090658    0.016951    0.680184    
#> 205: 0.347347    1.076177    3.519161    0.190619    0.088593    0.017555    0.682497    
#> 206: 0.343972    1.083083    3.518744    0.187275    0.088533    0.017699    0.684588    
#> 207: 0.337983    1.084969    3.516567    0.188248    0.089359    0.017956    0.684308    
#> 208: 0.335685    1.086091    3.515978    0.188914    0.089860    0.017821    0.684146    
#> 209: 0.337844    1.085310    3.515479    0.195575    0.090163    0.017724    0.682029    
#> 210: 0.340205    1.085882    3.515135    0.197184    0.088722    0.017663    0.680588    
#> 211: 0.340718    1.086728    3.515796    0.195512    0.088295    0.017670    0.679178    
#> 212: 0.341800    1.087337    3.515282    0.193075    0.088431    0.017450    0.678977    
#> 213: 0.342390    1.087358    3.515133    0.193316    0.089181    0.017208    0.679661    
#> 214: 0.344059    1.087407    3.515377    0.192498    0.089578    0.016963    0.679849    
#> 215: 0.342816    1.087522    3.515732    0.191484    0.090096    0.016805    0.679701    
#> 216: 0.344357    1.086200    3.515986    0.191936    0.092015    0.016704    0.680457    
#> 217: 0.344441    1.086887    3.516441    0.190863    0.094214    0.016515    0.679104    
#> 218: 0.343015    1.087733    3.516195    0.191704    0.095558    0.016337    0.678452    
#> 219: 0.340735    1.087410    3.515262    0.192380    0.095661    0.016075    0.678367    
#> 220: 0.339207    1.088341    3.514701    0.193892    0.095992    0.015917    0.677610    
#> 221: 0.338222    1.089038    3.514151    0.194145    0.096010    0.015728    0.676377    
#> 222: 0.338282    1.088731    3.513526    0.194731    0.096574    0.015551    0.676524    
#> 223: 0.339800    1.088275    3.513387    0.194931    0.096760    0.015314    0.676391    
#> 224: 0.341534    1.088356    3.513559    0.195726    0.096888    0.015182    0.676146    
#> 225: 0.341613    1.086695    3.513947    0.195693    0.096833    0.015120    0.676254    
#> 226: 0.341572    1.085517    3.513869    0.195171    0.096958    0.015023    0.676380    
#> 227: 0.341558    1.085291    3.513987    0.195564    0.096462    0.015040    0.675879    
#> 228: 0.339837    1.086549    3.513053    0.196227    0.097434    0.014990    0.675926    
#> 229: 0.338271    1.086625    3.512689    0.196233    0.097629    0.014983    0.675452    
#> 230: 0.338085    1.086492    3.512531    0.195888    0.098102    0.015011    0.675192    
#> 231: 0.337512    1.087631    3.512664    0.195523    0.098861    0.015032    0.675344    
#> 232: 0.336676    1.087851    3.512900    0.195724    0.099136    0.015039    0.675221    
#> 233: 0.337185    1.086376    3.513391    0.196539    0.099183    0.015067    0.675481    
#> 234: 0.337973    1.085897    3.513963    0.197325    0.099323    0.015100    0.675181    
#> 235: 0.338218    1.085572    3.514043    0.197724    0.099366    0.015100    0.674724    
#> 236: 0.338447    1.085900    3.513894    0.197863    0.099390    0.015069    0.674576    
#> 237: 0.339310    1.086077    3.514037    0.198601    0.099438    0.015076    0.674848    
#> 238: 0.340048    1.086163    3.514302    0.199543    0.099476    0.015127    0.675120    
#> 239: 0.339816    1.086402    3.514299    0.199010    0.099616    0.015123    0.675143    
#> 240: 0.340335    1.085914    3.514553    0.197961    0.099343    0.015171    0.675004    
#> 241: 0.340566    1.084794    3.514931    0.197509    0.099159    0.015149    0.675243    
#> 242: 0.340556    1.084176    3.514888    0.197601    0.098921    0.015187    0.675343    
#> 243: 0.339626    1.084781    3.514760    0.198104    0.098803    0.015198    0.675718    
#> 244: 0.338835    1.084900    3.514752    0.198786    0.098586    0.015301    0.675570    
#> 245: 0.338778    1.084591    3.514807    0.198280    0.098303    0.015295    0.675642    
#> 246: 0.338263    1.084529    3.514665    0.198270    0.098180    0.015308    0.675257    
#> 247: 0.337597    1.084751    3.514830    0.197955    0.098303    0.015268    0.675316    
#> 248: 0.337038    1.085488    3.514784    0.198153    0.097940    0.015316    0.675185    
#> 249: 0.337195    1.085637    3.514900    0.198465    0.097763    0.015424    0.675046    
#> 250: 0.337729    1.085966    3.515050    0.198812    0.097621    0.015598    0.674848    
#> 251: 0.337920    1.086591    3.515296    0.199196    0.097456    0.015719    0.674658    
#> 252: 0.338833    1.086807    3.515635    0.200439    0.097104    0.015780    0.674588    
#> 253: 0.338769    1.086999    3.515787    0.201077    0.096578    0.015868    0.674544    
#> 254: 0.338785    1.087466    3.515926    0.201805    0.096415    0.015898    0.674312    
#> 255: 0.338797    1.087931    3.516080    0.202610    0.096131    0.015960    0.674351    
#> 256: 0.338942    1.088221    3.515847    0.202674    0.096077    0.016013    0.674245    
#> 257: 0.338639    1.088409    3.515611    0.202875    0.096157    0.016028    0.673933    
#> 258: 0.337794    1.088626    3.515314    0.203447    0.096204    0.015990    0.673912    
#> 259: 0.336715    1.089207    3.515002    0.203949    0.096135    0.015979    0.674020    
#> 260: 0.335798    1.089829    3.514767    0.203541    0.096124    0.015961    0.673955    
#> 261: 0.334985    1.090413    3.514548    0.203555    0.096221    0.015914    0.674202    
#> 262: 0.334782    1.091010    3.514276    0.203105    0.095980    0.015907    0.674752    
#> 263: 0.334517    1.091397    3.514072    0.203160    0.095832    0.015891    0.675186    
#> 264: 0.334212    1.091564    3.513871    0.203556    0.095683    0.015852    0.675369    
#> 265: 0.334199    1.091789    3.513907    0.203671    0.095322    0.015854    0.675403    
#> 266: 0.334887    1.091820    3.513930    0.203663    0.095339    0.015852    0.675427    
#> 267: 0.335299    1.091838    3.514194    0.204520    0.095468    0.015873    0.675230    
#> 268: 0.335489    1.091580    3.514292    0.204508    0.095549    0.015892    0.675341    
#> 269: 0.336134    1.091363    3.514500    0.204202    0.095612    0.015926    0.675403    
#> 270: 0.336280    1.091638    3.514378    0.204190    0.095603    0.015924    0.675276    
#> 271: 0.335965    1.091873    3.514221    0.204194    0.095458    0.015922    0.675319    
#> 272: 0.335895    1.091739    3.514207    0.203844    0.095356    0.015913    0.675425    
#> 273: 0.335685    1.092057    3.514252    0.203881    0.095333    0.015907    0.675282    
#> 274: 0.335216    1.092157    3.514275    0.203625    0.095302    0.015864    0.675301    
#> 275: 0.335213    1.092029    3.514286    0.203339    0.095250    0.015842    0.675175    
#> 276: 0.335171    1.092133    3.514413    0.203330    0.095256    0.015801    0.675106    
#> 277: 0.334986    1.092279    3.514470    0.203192    0.095253    0.015782    0.674924    
#> 278: 0.335112    1.092187    3.514708    0.203265    0.095249    0.015759    0.674871    
#> 279: 0.335060    1.091864    3.514891    0.203572    0.095271    0.015723    0.675096    
#> 280: 0.335044    1.091769    3.514934    0.203233    0.095267    0.015697    0.674895    
#> 281: 0.334786    1.091813    3.514988    0.202980    0.095191    0.015695    0.674848    
#> 282: 0.334838    1.091916    3.514969    0.203126    0.095072    0.015709    0.675041    
#> 283: 0.334905    1.091891    3.515024    0.203538    0.095217    0.015772    0.675131    
#> 284: 0.334661    1.091821    3.514985    0.203639    0.095286    0.015803    0.675075    
#> 285: 0.334662    1.091907    3.514893    0.204091    0.095247    0.015847    0.675038    
#> 286: 0.334484    1.092164    3.514733    0.204317    0.095426    0.015850    0.674960    
#> 287: 0.334235    1.092311    3.514578    0.203901    0.095459    0.015858    0.675157    
#> 288: 0.334403    1.092255    3.514611    0.203670    0.095592    0.015894    0.675240    
#> 289: 0.334670    1.092236    3.514640    0.203033    0.095574    0.015927    0.675285    
#> 290: 0.334873    1.092020    3.514717    0.202819    0.095558    0.015933    0.675467    
#> 291: 0.334989    1.092036    3.514872    0.202723    0.095534    0.015957    0.675283    
#> 292: 0.334694    1.091922    3.514864    0.202572    0.095621    0.015938    0.675329    
#> 293: 0.334682    1.091945    3.514721    0.202587    0.095528    0.015918    0.675220    
#> 294: 0.334570    1.091991    3.514652    0.202822    0.095449    0.015899    0.675127    
#> 295: 0.334607    1.091666    3.514611    0.202904    0.095259    0.015877    0.675061    
#> 296: 0.334353    1.091768    3.514532    0.203145    0.095284    0.015860    0.675103    
#> 297: 0.334263    1.092081    3.514457    0.203233    0.095310    0.015818    0.675107    
#> 298: 0.334301    1.092123    3.514422    0.203480    0.095197    0.015834    0.675050    
#> 299: 0.334608    1.092206    3.514393    0.203690    0.095045    0.015841    0.674929    
#> 300: 0.335053    1.092255    3.514607    0.203756    0.094990    0.015836    0.674767    
#> 301: 0.335406    1.092166    3.514678    0.203781    0.094891    0.015818    0.674819    
#> 302: 0.335460    1.092306    3.514654    0.204055    0.094776    0.015807    0.674737    
#> 303: 0.335258    1.092524    3.514506    0.204189    0.094668    0.015794    0.674596    
#> 304: 0.335172    1.092829    3.514405    0.204313    0.094570    0.015788    0.674490    
#> 305: 0.335077    1.092630    3.514372    0.204428    0.094532    0.015811    0.674392    
#> 306: 0.334530    1.092619    3.514215    0.204112    0.094456    0.015836    0.674393    
#> 307: 0.334338    1.092658    3.514052    0.204091    0.094345    0.015830    0.674587    
#> 308: 0.334080    1.092499    3.513993    0.204268    0.094324    0.015816    0.674809    
#> 309: 0.334149    1.092213    3.513962    0.204506    0.094250    0.015816    0.674938    
#> 310: 0.334362    1.092049    3.513878    0.204861    0.094163    0.015801    0.674932    
#> 311: 0.334364    1.091958    3.513978    0.204794    0.094118    0.015803    0.674875    
#> 312: 0.334416    1.091802    3.513974    0.204582    0.094100    0.015796    0.674769    
#> 313: 0.334330    1.091552    3.513930    0.204342    0.094099    0.015783    0.674756    
#> 314: 0.334067    1.091661    3.513803    0.204314    0.094073    0.015757    0.674584    
#> 315: 0.333611    1.091826    3.513687    0.204382    0.094020    0.015738    0.674552    
#> 316: 0.333660    1.091924    3.513620    0.204214    0.094077    0.015732    0.674654    
#> 317: 0.333679    1.091956    3.513628    0.204170    0.094174    0.015731    0.674722    
#> 318: 0.333590    1.092220    3.513532    0.204097    0.094155    0.015748    0.674705    
#> 319: 0.333525    1.092281    3.513486    0.204004    0.094089    0.015770    0.674771    
#> 320: 0.333182    1.092267    3.513394    0.203776    0.094207    0.015804    0.674744    
#> 321: 0.333052    1.092299    3.513332    0.203636    0.094295    0.015828    0.674796    
#> 322: 0.332976    1.092191    3.513296    0.203757    0.094339    0.015808    0.674870    
#> 323: 0.332648    1.092148    3.513297    0.203975    0.094395    0.015805    0.674900    
#> 324: 0.332530    1.092452    3.513232    0.204369    0.094407    0.015787    0.674797    
#> 325: 0.332365    1.092591    3.513152    0.204377    0.094402    0.015773    0.674704    
#> 326: 0.332311    1.092476    3.513054    0.204189    0.094392    0.015755    0.674716    
#> 327: 0.332306    1.092506    3.513022    0.204114    0.094372    0.015758    0.674584    
#> 328: 0.332197    1.092631    3.512949    0.204215    0.094345    0.015754    0.674383    
#> 329: 0.332094    1.092678    3.512926    0.204181    0.094373    0.015772    0.674212    
#> 330: 0.332024    1.092834    3.512867    0.204163    0.094293    0.015789    0.674112    
#> 331: 0.332380    1.092903    3.512945    0.204073    0.094227    0.015813    0.674060    
#> 332: 0.332618    1.092734    3.513127    0.204134    0.094068    0.015861    0.674025    
#> 333: 0.332732    1.092699    3.513147    0.204218    0.093914    0.015884    0.673978    
#> 334: 0.332676    1.092766    3.513034    0.204184    0.093842    0.015876    0.673895    
#> 335: 0.332560    1.092730    3.512943    0.204165    0.093625    0.015885    0.673873    
#> 336: 0.332279    1.092806    3.512893    0.204143    0.093421    0.015925    0.673867    
#> 337: 0.332281    1.093120    3.512777    0.204122    0.093289    0.015961    0.673894    
#> 338: 0.332367    1.093123    3.512843    0.203974    0.093190    0.015987    0.673836    
#> 339: 0.332341    1.093190    3.512834    0.203869    0.093147    0.016015    0.673825    
#> 340: 0.332410    1.093009    3.512919    0.203790    0.093193    0.016037    0.673835    
#> 341: 0.332674    1.093016    3.512990    0.203731    0.093263    0.016055    0.673815    
#> 342: 0.332632    1.092806    3.513075    0.203466    0.093302    0.016074    0.673783    
#> 343: 0.332624    1.092737    3.513118    0.203316    0.093281    0.016067    0.673743    
#> 344: 0.332832    1.092641    3.513240    0.203090    0.093281    0.016071    0.673799    
#> 345: 0.333027    1.092576    3.513311    0.203019    0.093288    0.016083    0.673734    
#> 346: 0.332987    1.092576    3.513230    0.203013    0.093238    0.016083    0.673682    
#> 347: 0.333003    1.092572    3.513180    0.203262    0.093201    0.016103    0.673808    
#> 348: 0.332820    1.092649    3.513124    0.203057    0.093270    0.016128    0.673737    
#> 349: 0.332669    1.092610    3.513111    0.202946    0.093277    0.016165    0.673717    
#> 350: 0.332281    1.092586    3.513004    0.203150    0.093269    0.016183    0.673695    
#> 351: 0.332078    1.092831    3.512865    0.203357    0.093220    0.016211    0.673667    
#> 352: 0.331592    1.093094    3.512710    0.203673    0.093142    0.016234    0.673791    
#> 353: 0.331269    1.093423    3.512591    0.203984    0.093115    0.016255    0.673708    
#> 354: 0.331228    1.093543    3.512447    0.204063    0.093010    0.016270    0.673602    
#> 355: 0.331338    1.093611    3.512406    0.204234    0.092974    0.016284    0.673607    
#> 356: 0.331319    1.093571    3.512388    0.204135    0.092983    0.016293    0.673666    
#> 357: 0.331279    1.093747    3.512386    0.203929    0.092893    0.016307    0.673646    
#> 358: 0.331030    1.093867    3.512344    0.203927    0.092830    0.016309    0.673535    
#> 359: 0.330945    1.094002    3.512306    0.203925    0.092809    0.016285    0.673474    
#> 360: 0.330832    1.094129    3.512167    0.203928    0.092810    0.016261    0.673383    
#> 361: 0.330642    1.094207    3.512069    0.203834    0.092786    0.016229    0.673383    
#> 362: 0.330637    1.094150    3.512061    0.203819    0.092754    0.016212    0.673212    
#> 363: 0.330687    1.094083    3.512037    0.203822    0.092703    0.016212    0.673072    
#> 364: 0.330812    1.094198    3.512007    0.204071    0.092801    0.016187    0.673072    
#> 365: 0.330727    1.094195    3.511994    0.204102    0.092843    0.016190    0.673032    
#> 366: 0.330473    1.094462    3.511859    0.203907    0.092813    0.016196    0.673148    
#> 367: 0.330400    1.094473    3.511817    0.203834    0.092716    0.016201    0.673136    
#> 368: 0.330677    1.094431    3.511852    0.203822    0.092665    0.016207    0.673091    
#> 369: 0.330704    1.094299    3.511859    0.203781    0.092751    0.016183    0.673149    
#> 370: 0.330766    1.094254    3.511867    0.203532    0.092833    0.016172    0.673206    
#> 371: 0.330643    1.094332    3.511773    0.203352    0.092835    0.016151    0.673250    
#> 372: 0.330491    1.094373    3.511715    0.203375    0.092856    0.016140    0.673214    
#> 373: 0.330206    1.094515    3.511609    0.203361    0.092877    0.016134    0.673173    
#> 374: 0.330112    1.094624    3.511525    0.203318    0.092822    0.016126    0.673113    
#> 375: 0.329921    1.094666    3.511502    0.203364    0.092739    0.016111    0.673191    
#> 376: 0.329835    1.094755    3.511476    0.203281    0.092686    0.016103    0.673166    
#> 377: 0.329518    1.094842    3.511410    0.203207    0.092656    0.016103    0.673081    
#> 378: 0.329235    1.094857    3.511341    0.203241    0.092650    0.016103    0.673137    
#> 379: 0.329153    1.094797    3.511344    0.203276    0.092637    0.016096    0.673120    
#> 380: 0.329113    1.094769    3.511337    0.203413    0.092676    0.016076    0.673145    
#> 381: 0.329035    1.094783    3.511338    0.203506    0.092682    0.016067    0.673177    
#> 382: 0.329054    1.094804    3.511362    0.203495    0.092674    0.016062    0.673117    
#> 383: 0.329174    1.094813    3.511375    0.203334    0.092714    0.016047    0.673066    
#> 384: 0.329064    1.094756    3.511406    0.203199    0.092664    0.016038    0.673088    
#> 385: 0.329068    1.094760    3.511430    0.203268    0.092613    0.016051    0.673043    
#> 386: 0.329172    1.094866    3.511476    0.203221    0.092589    0.016070    0.673027    
#> 387: 0.329418    1.094975    3.511601    0.203030    0.092575    0.016080    0.673014    
#> 388: 0.329476    1.095011    3.511653    0.202906    0.092502    0.016091    0.672964    
#> 389: 0.329566    1.094841    3.511677    0.202966    0.092450    0.016099    0.672976    
#> 390: 0.329548    1.094841    3.511687    0.202810    0.092427    0.016125    0.672984    
#> 391: 0.329696    1.094744    3.511773    0.202825    0.092373    0.016132    0.672952    
#> 392: 0.329862    1.094700    3.511886    0.203038    0.092380    0.016144    0.672899    
#> 393: 0.329951    1.094691    3.511871    0.203059    0.092365    0.016147    0.672865    
#> 394: 0.330026    1.094731    3.511848    0.203080    0.092366    0.016140    0.672833    
#> 395: 0.330241    1.094752    3.511872    0.202944    0.092381    0.016134    0.672859    
#> 396: 0.330327    1.094754    3.511903    0.203050    0.092392    0.016116    0.672740    
#> 397: 0.330523    1.094692    3.511967    0.202938    0.092413    0.016101    0.672696    
#> 398: 0.330662    1.094558    3.512050    0.202797    0.092437    0.016100    0.672742    
#> 399: 0.330814    1.094551    3.512132    0.202688    0.092505    0.016105    0.672652    
#> 400: 0.330883    1.094545    3.512208    0.202484    0.092520    0.016109    0.672586    
#> 401: 0.330874    1.094609    3.512204    0.202418    0.092597    0.016117    0.672594    
#> 402: 0.330865    1.094839    3.512221    0.202254    0.092630    0.016122    0.672596    
#> 403: 0.331122    1.094871    3.512244    0.202231    0.092621    0.016139    0.672564    
#> 404: 0.331209    1.094828    3.512274    0.202023    0.092604    0.016143    0.672535    
#> 405: 0.331321    1.094844    3.512293    0.201925    0.092535    0.016150    0.672583    
#> 406: 0.331420    1.094853    3.512314    0.201850    0.092549    0.016165    0.672612    
#> 407: 0.331574    1.094755    3.512472    0.201842    0.092607    0.016176    0.672560    
#> 408: 0.331689    1.094701    3.512531    0.201709    0.092652    0.016185    0.672620    
#> 409: 0.331684    1.094634    3.512517    0.201572    0.092580    0.016184    0.672543    
#> 410: 0.331492    1.094696    3.512387    0.201450    0.092628    0.016155    0.672646    
#> 411: 0.331509    1.094547    3.512298    0.201497    0.092649    0.016151    0.672722    
#> 412: 0.331549    1.094689    3.512230    0.201379    0.092674    0.016140    0.672743    
#> 413: 0.331507    1.094757    3.512210    0.201172    0.092672    0.016131    0.672806    
#> 414: 0.331454    1.094705    3.512209    0.201112    0.092612    0.016128    0.672805    
#> 415: 0.331492    1.094782    3.512199    0.201175    0.092600    0.016123    0.672720    
#> 416: 0.331442    1.094905    3.512198    0.201183    0.092575    0.016116    0.672686    
#> 417: 0.331377    1.094904    3.512157    0.201225    0.092550    0.016104    0.672650    
#> 418: 0.331214    1.095044    3.512136    0.201164    0.092557    0.016102    0.672580    
#> 419: 0.331081    1.095188    3.512095    0.201174    0.092540    0.016103    0.672552    
#> 420: 0.330955    1.095255    3.512028    0.201031    0.092510    0.016101    0.672577    
#> 421: 0.330730    1.095319    3.511975    0.200876    0.092449    0.016100    0.672619    
#> 422: 0.330887    1.095279    3.511956    0.200900    0.092506    0.016092    0.672655    
#> 423: 0.330927    1.095159    3.511934    0.200822    0.092648    0.016085    0.672685    
#> 424: 0.330924    1.095154    3.511911    0.200734    0.092741    0.016076    0.672649    
#> 425: 0.331128    1.095264    3.511926    0.200725    0.092793    0.016066    0.672721    
#> 426: 0.331277    1.095369    3.511892    0.200778    0.092789    0.016064    0.672657    
#> 427: 0.331231    1.095384    3.511856    0.200853    0.092712    0.016064    0.672572    
#> 428: 0.331143    1.095462    3.511861    0.201012    0.092655    0.016066    0.672590    
#> 429: 0.331088    1.095523    3.511843    0.201036    0.092651    0.016062    0.672581    
#> 430: 0.331042    1.095614    3.511820    0.200882    0.092657    0.016066    0.672627    
#> 431: 0.330940    1.095610    3.511790    0.200790    0.092632    0.016061    0.672569    
#> 432: 0.330905    1.095685    3.511808    0.200777    0.092636    0.016065    0.672528    
#> 433: 0.330946    1.095780    3.511850    0.200704    0.092541    0.016083    0.672512    
#> 434: 0.330882    1.095808    3.511830    0.200648    0.092528    0.016090    0.672474    
#> 435: 0.330870    1.095826    3.511822    0.200690    0.092572    0.016070    0.672448    
#> 436: 0.330844    1.095816    3.511842    0.200700    0.092666    0.016047    0.672557    
#> 437: 0.330851    1.095818    3.511835    0.200642    0.092678    0.016036    0.672478    
#> 438: 0.330890    1.095943    3.511802    0.200627    0.092648    0.016026    0.672446    
#> 439: 0.330821    1.096032    3.511768    0.200666    0.092629    0.016012    0.672393    
#> 440: 0.330729    1.096086    3.511791    0.200717    0.092652    0.015992    0.672345    
#> 441: 0.330681    1.096109    3.511813    0.200853    0.092688    0.015986    0.672300    
#> 442: 0.330621    1.096195    3.511822    0.200803    0.092681    0.015971    0.672318    
#> 443: 0.330431    1.096226    3.511753    0.200817    0.092604    0.015960    0.672390    
#> 444: 0.330435    1.096307    3.511711    0.200827    0.092559    0.015969    0.672387    
#> 445: 0.330361    1.096424    3.511664    0.200952    0.092508    0.015992    0.672383    
#> 446: 0.330113    1.096535    3.511560    0.201099    0.092497    0.015996    0.672379    
#> 447: 0.330108    1.096585    3.511504    0.201116    0.092519    0.016004    0.672394    
#> 448: 0.330053    1.096611    3.511515    0.201067    0.092521    0.016007    0.672317    
#> 449: 0.330242    1.096653    3.511633    0.200968    0.092559    0.016009    0.672305    
#> 450: 0.330238    1.096621    3.511682    0.200887    0.092547    0.016021    0.672292    
#> 451: 0.330453    1.096565    3.511702    0.200814    0.092569    0.016025    0.672330    
#> 452: 0.330592    1.096564    3.511730    0.200657    0.092536    0.016010    0.672372    
#> 453: 0.330501    1.096540    3.511717    0.200540    0.092533    0.016000    0.672332    
#> 454: 0.330489    1.096677    3.511719    0.200473    0.092506    0.015993    0.672272    
#> 455: 0.330541    1.096751    3.511726    0.200464    0.092481    0.015985    0.672219    
#> 456: 0.330575    1.096711    3.511731    0.200414    0.092464    0.015981    0.672175    
#> 457: 0.330578    1.096681    3.511784    0.200361    0.092496    0.015973    0.672115    
#> 458: 0.330565    1.096825    3.511838    0.200308    0.092561    0.015954    0.672193    
#> 459: 0.330434    1.096967    3.511819    0.200251    0.092585    0.015941    0.672304    
#> 460: 0.330230    1.096978    3.511748    0.200213    0.092670    0.015929    0.672391    
#> 461: 0.329994    1.097044    3.511705    0.200227    0.092773    0.015930    0.672451    
#> 462: 0.329717    1.097086    3.511673    0.200189    0.092770    0.015943    0.672472    
#> 463: 0.329715    1.097127    3.511706    0.200061    0.092798    0.015945    0.672473    
#> 464: 0.329840    1.097104    3.511739    0.199968    0.092758    0.015941    0.672403    
#> 465: 0.329880    1.096962    3.511743    0.199914    0.092716    0.015940    0.672441    
#> 466: 0.329870    1.097038    3.511686    0.199821    0.092704    0.015937    0.672450    
#> 467: 0.329779    1.097017    3.511669    0.199919    0.092710    0.015937    0.672370    
#> 468: 0.329951    1.097044    3.511718    0.199942    0.092719    0.015926    0.672411    
#> 469: 0.330081    1.097184    3.511743    0.199943    0.092678    0.015923    0.672517    
#> 470: 0.330051    1.097231    3.511729    0.199980    0.092621    0.015923    0.672546    
#> 471: 0.329987    1.097202    3.511709    0.199921    0.092570    0.015917    0.672529    
#> 472: 0.330030    1.097157    3.511720    0.199786    0.092534    0.015908    0.672547    
#> 473: 0.329980    1.097131    3.511738    0.199783    0.092557    0.015901    0.672501    
#> 474: 0.330039    1.097024    3.511792    0.199805    0.092539    0.015899    0.672502    
#> 475: 0.330181    1.097034    3.511842    0.199775    0.092529    0.015901    0.672475    
#> 476: 0.330298    1.096947    3.511871    0.199755    0.092549    0.015893    0.672504    
#> 477: 0.330369    1.096905    3.511952    0.199692    0.092531    0.015893    0.672473    
#> 478: 0.330302    1.096880    3.512001    0.199600    0.092504    0.015894    0.672466    
#> 479: 0.330387    1.096891    3.512033    0.199570    0.092554    0.015893    0.672447    
#> 480: 0.330484    1.096865    3.512057    0.199648    0.092619    0.015895    0.672417    
#> 481: 0.330495    1.096794    3.512062    0.199649    0.092689    0.015882    0.672402    
#> 482: 0.330673    1.096754    3.512062    0.199777    0.092740    0.015860    0.672459    
#> 483: 0.330744    1.096720    3.512081    0.199872    0.092794    0.015847    0.672402    
#> 484: 0.330811    1.096618    3.512091    0.200112    0.092801    0.015836    0.672456    
#> 485: 0.330948    1.096582    3.512135    0.200207    0.092755    0.015841    0.672487    
#> 486: 0.331023    1.096621    3.512153    0.200348    0.092742    0.015837    0.672440    
#> 487: 0.331259    1.096588    3.512234    0.200615    0.092749    0.015829    0.672385    
#> 488: 0.331291    1.096559    3.512265    0.200684    0.092772    0.015812    0.672360    
#> 489: 0.331308    1.096521    3.512259    0.200789    0.092775    0.015801    0.672345    
#> 490: 0.331351    1.096523    3.512324    0.200776    0.092764    0.015789    0.672344    
#> 491: 0.331345    1.096562    3.512388    0.200691    0.092705    0.015798    0.672302    
#> 492: 0.331284    1.096581    3.512408    0.200602    0.092697    0.015806    0.672332    
#> 493: 0.331108    1.096608    3.512384    0.200415    0.092708    0.015803    0.672399    
#> 494: 0.331117    1.096711    3.512342    0.200258    0.092758    0.015792    0.672389    
#> 495: 0.331091    1.096756    3.512293    0.200242    0.092809    0.015792    0.672356    
#> 496: 0.331129    1.096841    3.512239    0.200153    0.092763    0.015796    0.672438    
#> 497: 0.331009    1.096866    3.512208    0.200125    0.092771    0.015806    0.672445    
#> 498: 0.331003    1.096880    3.512181    0.199974    0.092766    0.015806    0.672429    
#> 499: 0.330933    1.096915    3.512147    0.199931    0.092766    0.015809    0.672393    
#> 500: 0.330957    1.096838    3.512146    0.199836    0.092756    0.015818    0.672423    
#> [====|====|=>-------------------------------------]  25%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.295977    0.979353    3.441212    0.392401    0.078575    0.016800    1.617246    
#> 002: 0.327385    0.988221    3.474111    0.372781    0.074646    0.016383    1.042192    
#> 003: 0.387091    0.950326    3.470951    0.354142    0.070914    0.015677    0.881254    
#> 004: 0.317648    0.951122    3.463282    0.336435    0.071158    0.014893    0.791885    
#> 005: 0.350152    0.961683    3.450722    0.319613    0.079435    0.014148    0.732200    
#> 006: 0.328955    0.974117    3.452293    0.303633    0.104941    0.013441    0.692369    
#> 007: 0.302851    0.967263    3.445029    0.288451    0.108275    0.012769    0.705526    
#> 008: 0.288828    0.950357    3.438183    0.290781    0.124960    0.012130    0.702092    
#> 009: 0.280618    0.942429    3.431140    0.296191    0.118712    0.011524    0.706846    
#> 010: 0.321270    0.938709    3.433151    0.411892    0.112776    0.012434    0.711109    
#> 011: 0.369639    0.944385    3.448656    0.391297    0.107137    0.015028    0.712035    
#> 012: 0.360723    0.929043    3.443881    0.420393    0.115628    0.014276    0.698145    
#> 013: 0.325883    0.936056    3.436852    0.399374    0.109847    0.013563    0.704819    
#> 014: 0.338648    0.943606    3.437594    0.379405    0.104355    0.012884    0.703096    
#> 015: 0.356091    0.938888    3.456344    0.360435    0.099137    0.013322    0.689159    
#> 016: 0.369630    0.950703    3.464313    0.342413    0.094180    0.012664    0.700122    
#> 017: 0.345842    0.970377    3.460949    0.340569    0.090352    0.012258    0.686857    
#> 018: 0.365483    0.962383    3.447275    0.327689    0.086528    0.013655    0.695927    
#> 019: 0.358712    0.959287    3.452258    0.354251    0.082201    0.013520    0.676120    
#> 020: 0.369489    0.963832    3.461948    0.336539    0.087776    0.013128    0.677847    
#> 021: 0.363448    0.949513    3.467114    0.319712    0.083388    0.013660    0.679638    
#> 022: 0.363353    0.938163    3.471913    0.303726    0.079218    0.013333    0.686375    
#> 023: 0.400413    0.946364    3.473060    0.356184    0.104617    0.013728    0.692310    
#> 024: 0.439763    0.959571    3.473113    0.385967    0.099386    0.014430    0.671212    
#> 025: 0.397921    0.941273    3.467160    0.428505    0.094417    0.013708    0.682710    
#> 026: 0.377630    0.969728    3.449329    0.407080    0.097988    0.013287    0.684568    
#> 027: 0.323948    0.968071    3.448219    0.386726    0.096825    0.013668    0.704306    
#> 028: 0.356280    0.956296    3.457558    0.382990    0.091984    0.014189    0.673860    
#> 029: 0.380839    0.958714    3.458842    0.363841    0.087924    0.014577    0.698251    
#> 030: 0.392475    0.973393    3.451558    0.368892    0.083528    0.014605    0.699997    
#> 031: 0.369201    0.936382    3.460699    0.384131    0.080854    0.015137    0.689543    
#> 032: 0.352673    0.931526    3.463520    0.364924    0.077488    0.017880    0.684953    
#> 033: 0.327835    0.957092    3.458122    0.346678    0.091027    0.018011    0.687681    
#> 034: 0.364824    0.970342    3.455531    0.334978    0.086475    0.017238    0.698189    
#> 035: 0.346356    0.944179    3.451443    0.318229    0.096605    0.016376    0.678416    
#> 036: 0.333128    0.960692    3.448950    0.321486    0.091774    0.016174    0.685580    
#> 037: 0.308269    0.958549    3.438142    0.305411    0.087186    0.015365    0.705302    
#> 038: 0.338992    0.956705    3.435412    0.373686    0.093249    0.014597    0.685272    
#> 039: 0.346925    0.947267    3.444686    0.355001    0.099267    0.013867    0.669350    
#> 040: 0.328807    0.956351    3.451615    0.388961    0.099903    0.013174    0.675160    
#> 041: 0.351449    0.961507    3.450087    0.480025    0.101004    0.012515    0.670669    
#> 042: 0.371807    0.954724    3.448435    0.456024    0.102564    0.011889    0.676196    
#> 043: 0.344029    0.960224    3.441357    0.433223    0.103208    0.011295    0.668626    
#> 044: 0.358038    0.949671    3.442731    0.411562    0.106442    0.010730    0.686695    
#> 045: 0.351142    0.976578    3.436968    0.401778    0.114149    0.010194    0.676938    
#> 046: 0.305561    0.972278    3.428846    0.381689    0.108441    0.009684    0.679874    
#> 047: 0.316294    0.983713    3.422523    0.371463    0.113720    0.009200    0.670393    
#> 048: 0.311633    0.964730    3.424163    0.352889    0.115709    0.008740    0.670924    
#> 049: 0.306136    0.975706    3.428562    0.335245    0.109924    0.008303    0.696923    
#> 050: 0.357095    0.925735    3.442017    0.373060    0.104428    0.007888    0.723830    
#> 051: 0.305694    0.948135    3.439632    0.354407    0.109539    0.007691    0.712561    
#> 052: 0.325732    0.955651    3.445581    0.336687    0.106406    0.007306    0.703076    
#> 053: 0.302036    0.952080    3.439783    0.319852    0.126672    0.007153    0.699116    
#> 054: 0.288834    0.946879    3.445487    0.303860    0.120338    0.009534    0.718505    
#> 055: 0.294447    0.961979    3.444667    0.288667    0.114321    0.013846    0.704457    
#> 056: 0.335631    0.943849    3.457733    0.274233    0.108605    0.013153    0.706157    
#> 057: 0.394370    0.925702    3.476464    0.260522    0.104389    0.012496    0.711404    
#> 058: 0.351069    0.933524    3.466878    0.265783    0.099170    0.011871    0.704792    
#> 059: 0.341690    0.891091    3.471246    0.282837    0.098403    0.013090    0.716338    
#> 060: 0.344376    0.904371    3.477666    0.268695    0.095175    0.012436    0.748326    
#> 061: 0.376205    0.921318    3.464302    0.276841    0.099727    0.011814    0.761527    
#> 062: 0.345873    0.947444    3.464983    0.339734    0.100094    0.011338    0.750716    
#> 063: 0.386698    0.902336    3.468645    0.322747    0.095089    0.010771    0.738065    
#> 064: 0.372753    0.943882    3.466910    0.361156    0.090335    0.010355    0.710849    
#> 065: 0.386371    0.923939    3.458305    0.343098    0.094418    0.010880    0.727106    
#> 066: 0.367694    0.947263    3.457093    0.333246    0.089698    0.010336    0.704051    
#> 067: 0.377019    0.929967    3.449515    0.338727    0.089844    0.012937    0.691195    
#> 068: 0.335118    0.946169    3.450582    0.347158    0.118299    0.012290    0.709762    
#> 069: 0.339223    0.938015    3.443112    0.329800    0.117399    0.011892    0.696636    
#> 070: 0.324235    0.958615    3.455959    0.335822    0.111529    0.014779    0.674315    
#> 071: 0.323069    0.965430    3.445600    0.319031    0.105953    0.014040    0.686126    
#> 072: 0.341644    0.922831    3.455999    0.303079    0.100655    0.013338    0.682476    
#> 073: 0.357624    0.944685    3.469627    0.398517    0.110939    0.013188    0.666408    
#> 074: 0.359734    0.909226    3.466858    0.378591    0.105393    0.012703    0.676601    
#> 075: 0.361972    0.909528    3.476322    0.375160    0.105497    0.012068    0.685619    
#> 076: 0.364906    0.919004    3.485201    0.356402    0.100222    0.011465    0.694291    
#> 077: 0.431460    0.912104    3.470533    0.357005    0.097610    0.010950    0.680144    
#> 078: 0.393033    0.942792    3.468381    0.374745    0.098366    0.010403    0.696800    
#> 079: 0.371733    0.968897    3.466229    0.363131    0.111513    0.009883    0.712986    
#> 080: 0.396055    0.952442    3.466704    0.344975    0.105937    0.011018    0.700223    
#> 081: 0.394064    0.965258    3.457646    0.327726    0.100640    0.011201    0.686562    
#> 082: 0.384349    0.934042    3.478774    0.316067    0.095608    0.010641    0.709570    
#> 083: 0.372315    0.927097    3.464878    0.333555    0.090828    0.010109    0.719281    
#> 084: 0.370594    0.940420    3.460012    0.316877    0.102502    0.009604    0.699579    
#> 085: 0.357130    0.949355    3.460485    0.301034    0.114116    0.009124    0.691645    
#> 086: 0.338169    0.968706    3.454196    0.296490    0.108411    0.008667    0.710998    
#> 087: 0.321125    0.978981    3.438983    0.317276    0.102990    0.010337    0.688275    
#> 088: 0.274255    0.966181    3.436801    0.368075    0.097841    0.009820    0.711377    
#> 089: 0.303796    0.966311    3.432405    0.349672    0.112858    0.009329    0.707131    
#> 090: 0.289919    0.962883    3.427729    0.334893    0.107215    0.008862    0.706726    
#> 091: 0.288395    0.981054    3.418055    0.318149    0.101855    0.008419    0.702736    
#> 092: 0.267753    0.998634    3.413616    0.349591    0.107949    0.009225    0.703659    
#> 093: 0.264035    0.982028    3.417217    0.332111    0.102551    0.011037    0.685287    
#> 094: 0.306813    0.973191    3.417730    0.368569    0.097424    0.010779    0.670338    
#> 095: 0.302162    1.005221    3.431164    0.350140    0.092552    0.012809    0.665235    
#> 096: 0.247269    0.992625    3.427265    0.332633    0.105365    0.012168    0.666589    
#> 097: 0.322577    0.992011    3.425931    0.316001    0.107784    0.011560    0.675562    
#> 098: 0.328545    0.987349    3.440955    0.326210    0.102708    0.011691    0.684700    
#> 099: 0.302922    0.953133    3.425219    0.335989    0.106592    0.011106    0.677794    
#> 100: 0.291014    0.960099    3.431956    0.319190    0.101262    0.010551    0.682282    
#> 101: 0.263873    0.958839    3.409871    0.328364    0.096199    0.010023    0.681239    
#> 102: 0.283522    0.973937    3.406273    0.311946    0.091389    0.009522    0.704683    
#> 103: 0.274211    0.970224    3.419156    0.300389    0.100591    0.009046    0.704949    
#> 104: 0.301739    0.967277    3.432990    0.319531    0.108496    0.008879    0.692155    
#> 105: 0.276220    0.968119    3.430290    0.303554    0.103071    0.008435    0.700833    
#> 106: 0.319034    0.952573    3.435341    0.325163    0.097918    0.008955    0.690067    
#> 107: 0.367995    0.953315    3.439893    0.422139    0.123070    0.008507    0.700859    
#> 108: 0.361992    0.943592    3.439530    0.401032    0.119147    0.008082    0.703305    
#> 109: 0.341970    0.949293    3.457492    0.380980    0.147602    0.007678    0.711430    
#> 110: 0.355522    0.957774    3.437860    0.373691    0.140221    0.007294    0.689555    
#> 111: 0.327470    0.957267    3.438450    0.355006    0.133210    0.006929    0.684329    
#> 112: 0.300579    0.954631    3.428816    0.337256    0.126550    0.006583    0.701028    
#> 113: 0.307311    0.990180    3.429988    0.328951    0.120222    0.006254    0.685438    
#> 114: 0.318375    0.969039    3.434830    0.387675    0.114211    0.005941    0.687979    
#> 115: 0.337898    0.948024    3.437186    0.421758    0.113050    0.005644    0.699623    
#> 116: 0.336661    0.975204    3.441813    0.400670    0.109391    0.005437    0.702441    
#> 117: 0.299546    0.985796    3.438511    0.380636    0.103921    0.005174    0.696397    
#> 118: 0.321977    0.962156    3.439616    0.361604    0.111949    0.004915    0.685261    
#> 119: 0.313434    0.978154    3.438626    0.343524    0.118364    0.004830    0.702097    
#> 120: 0.274576    0.970366    3.433384    0.326877    0.114701    0.004589    0.712963    
#> 121: 0.303506    0.996520    3.430171    0.310533    0.113171    0.004930    0.720790    
#> 122: 0.315013    0.949625    3.438663    0.295006    0.107512    0.004683    0.738491    
#> 123: 0.340544    0.991633    3.436194    0.303737    0.102137    0.004449    0.725676    
#> 124: 0.321338    1.000386    3.423891    0.328265    0.101520    0.004227    0.713506    
#> 125: 0.291832    0.990859    3.417601    0.333125    0.112089    0.004015    0.714789    
#> 126: 0.275394    1.000206    3.411098    0.316469    0.116951    0.003814    0.706616    
#> 127: 0.282422    0.971230    3.412888    0.317808    0.127926    0.003764    0.710859    
#> 128: 0.258332    0.991325    3.422064    0.301918    0.121530    0.003575    0.723010    
#> 129: 0.329472    0.984894    3.431313    0.317364    0.115453    0.004382    0.715098    
#> 130: 0.339308    0.985288    3.438713    0.316460    0.121753    0.004830    0.688339    
#> 131: 0.323163    0.963356    3.428512    0.338567    0.119904    0.004588    0.686941    
#> 132: 0.309257    0.959025    3.419000    0.396475    0.124664    0.004915    0.701377    
#> 133: 0.290978    0.953673    3.422917    0.376651    0.136928    0.005378    0.690583    
#> 134: 0.301356    0.941523    3.425850    0.385017    0.130082    0.007642    0.688939    
#> 135: 0.320852    0.947664    3.431722    0.402280    0.123578    0.007259    0.686839    
#> 136: 0.318022    0.941198    3.433590    0.382166    0.124010    0.006896    0.673840    
#> 137: 0.396094    0.968449    3.439848    0.428373    0.117809    0.006552    0.689593    
#> 138: 0.330459    0.955536    3.434269    0.413795    0.115809    0.006224    0.695829    
#> 139: 0.308134    0.969908    3.432239    0.393105    0.110018    0.005913    0.689060    
#> 140: 0.318877    0.971337    3.442912    0.373450    0.109444    0.007455    0.683131    
#> 141: 0.306644    0.941084    3.450137    0.354777    0.105111    0.008135    0.695214    
#> 142: 0.380928    0.947486    3.459292    0.380654    0.106118    0.007729    0.701537    
#> 143: 0.367016    0.963012    3.464927    0.395879    0.116022    0.008489    0.691909    
#> 144: 0.368064    0.962661    3.459886    0.376085    0.120757    0.008064    0.691668    
#> 145: 0.335828    0.960680    3.452588    0.357281    0.114719    0.007661    0.679971    
#> 146: 0.320798    0.964524    3.462607    0.385667    0.109635    0.007278    0.692646    
#> 147: 0.324606    0.946556    3.458098    0.366384    0.104153    0.009206    0.668839    
#> 148: 0.324589    0.964701    3.464443    0.348065    0.101259    0.010024    0.673335    
#> 149: 0.343290    0.948005    3.459835    0.330662    0.096196    0.011645    0.679387    
#> 150: 0.349677    0.927387    3.453099    0.314128    0.091386    0.011063    0.692348    
#> 151: 0.312752    0.970304    3.449924    0.298641    0.095214    0.010510    0.684982    
#> 152: 0.323951    0.964568    3.449685    0.332403    0.102573    0.006270    0.686221    
#> 153: 0.307251    0.961378    3.450561    0.306530    0.104923    0.007331    0.688321    
#> 154: 0.373684    0.975379    3.458834    0.283196    0.120629    0.008426    0.701157    
#> 155: 0.317829    0.949998    3.442733    0.268756    0.112938    0.006998    0.688415    
#> 156: 0.337827    0.927531    3.454748    0.287107    0.100383    0.006751    0.698207    
#> 157: 0.339674    0.932768    3.464557    0.337315    0.120319    0.008107    0.697424    
#> 158: 0.388322    0.966421    3.461884    0.325517    0.110484    0.007268    0.696749    
#> 159: 0.340777    0.982323    3.452727    0.313978    0.092131    0.007339    0.696225    
#> 160: 0.342608    0.944948    3.456891    0.299880    0.088457    0.006797    0.695958    
#> 161: 0.367373    0.966481    3.450451    0.333206    0.084602    0.007369    0.689902    
#> 162: 0.348406    0.965267    3.447886    0.330862    0.102980    0.005689    0.720320    
#> 163: 0.356398    0.964333    3.435335    0.362740    0.125633    0.005961    0.690285    
#> 164: 0.334395    0.967445    3.436222    0.337679    0.097346    0.004211    0.697983    
#> 165: 0.328315    0.940323    3.441270    0.367145    0.110587    0.003920    0.693935    
#> 166: 0.326105    0.978826    3.442509    0.322525    0.129342    0.003703    0.688868    
#> 167: 0.318901    0.965856    3.447487    0.329045    0.127316    0.003511    0.694733    
#> 168: 0.354465    0.974235    3.446781    0.388799    0.130074    0.003618    0.692698    
#> 169: 0.335342    0.981292    3.438019    0.343648    0.114649    0.004275    0.705305    
#> 170: 0.344459    0.975671    3.438042    0.390360    0.117344    0.004658    0.701267    
#> 171: 0.288456    0.971792    3.437602    0.312902    0.111041    0.004398    0.686685    
#> 172: 0.280096    0.972343    3.436578    0.272059    0.138975    0.004238    0.696273    
#> 173: 0.307472    0.950568    3.438832    0.345027    0.117321    0.003687    0.702754    
#> 174: 0.324425    0.954575    3.443615    0.365866    0.130200    0.003395    0.692382    
#> 175: 0.329285    0.945626    3.438744    0.367922    0.134975    0.002852    0.696926    
#> 176: 0.344587    0.954934    3.441322    0.379901    0.125512    0.002720    0.701399    
#> 177: 0.337252    0.941472    3.445978    0.369857    0.136192    0.002047    0.716179    
#> 178: 0.348032    0.933510    3.446190    0.316250    0.141182    0.002040    0.704457    
#> 179: 0.368905    0.969642    3.446339    0.301198    0.109782    0.002872    0.708812    
#> 180: 0.328430    0.982372    3.446424    0.286479    0.124224    0.003275    0.707773    
#> 181: 0.307016    0.963574    3.426702    0.290912    0.095577    0.002889    0.700853    
#> 182: 0.277870    0.979520    3.421017    0.329459    0.126333    0.002932    0.694308    
#> 183: 0.293368    1.003598    3.418691    0.299435    0.127866    0.002881    0.694938    
#> 184: 0.277006    0.988361    3.418228    0.286771    0.112820    0.003052    0.696521    
#> 185: 0.277912    0.992783    3.408261    0.314174    0.119188    0.002907    0.720732    
#> 186: 0.246278    0.991196    3.414621    0.290707    0.125427    0.003116    0.704314    
#> 187: 0.295242    0.986245    3.419513    0.345858    0.118541    0.002537    0.702185    
#> 188: 0.267706    1.000426    3.412282    0.308642    0.126679    0.002496    0.723560    
#> 189: 0.310763    1.001117    3.419490    0.311388    0.113715    0.002802    0.721574    
#> 190: 0.304700    0.982403    3.428400    0.316794    0.104526    0.002465    0.689867    
#> 191: 0.289507    0.986442    3.422638    0.277095    0.117619    0.002321    0.688730    
#> 192: 0.272168    0.998696    3.422906    0.383171    0.107007    0.003241    0.694102    
#> 193: 0.333030    0.987813    3.426134    0.392984    0.085546    0.002824    0.687581    
#> 194: 0.302285    0.978434    3.422081    0.381665    0.110355    0.003583    0.692803    
#> 195: 0.245531    0.983699    3.414091    0.347208    0.123576    0.003469    0.701206    
#> 196: 0.267846    0.974530    3.421803    0.319030    0.132763    0.002846    0.722154    
#> 197: 0.265496    0.970133    3.430536    0.347110    0.124833    0.002611    0.729582    
#> 198: 0.331854    0.943879    3.424252    0.347280    0.123771    0.002601    0.707889    
#> 199: 0.281374    0.967475    3.425891    0.362270    0.127177    0.002604    0.705822    
#> 200: 0.275383    0.974487    3.428171    0.412116    0.110598    0.002547    0.711074    
#> 201: 0.297875    0.967801    3.430297    0.442994    0.106739    0.002966    0.706849    
#> 202: 0.309667    0.970041    3.430621    0.469308    0.112744    0.002831    0.707378    
#> 203: 0.323865    0.968356    3.430703    0.487443    0.114513    0.002729    0.707786    
#> 204: 0.334959    0.961650    3.432319    0.478821    0.118906    0.002741    0.706985    
#> 205: 0.339274    0.965684    3.433074    0.474597    0.120857    0.002709    0.707520    
#> 206: 0.337149    0.970171    3.433487    0.463181    0.125169    0.002751    0.707546    
#> 207: 0.338321    0.970981    3.433208    0.464297    0.127588    0.002677    0.706728    
#> 208: 0.337945    0.969915    3.432717    0.461096    0.128445    0.002595    0.705311    
#> 209: 0.344893    0.967116    3.433044    0.461951    0.127988    0.002607    0.706202    
#> 210: 0.344633    0.963347    3.433351    0.458321    0.127957    0.002571    0.705712    
#> 211: 0.345341    0.963722    3.433862    0.451998    0.128166    0.002496    0.704588    
#> 212: 0.345807    0.962792    3.434306    0.440340    0.128798    0.002451    0.705107    
#> 213: 0.347083    0.960962    3.434440    0.433695    0.130690    0.002431    0.706682    
#> 214: 0.346502    0.961282    3.434033    0.429249    0.130985    0.002392    0.705773    
#> 215: 0.344672    0.962719    3.433561    0.424535    0.131115    0.002356    0.705724    
#> 216: 0.344389    0.962953    3.433469    0.421022    0.131580    0.002334    0.707234    
#> 217: 0.340367    0.963889    3.433135    0.418751    0.132014    0.002347    0.707739    
#> 218: 0.339492    0.965229    3.432700    0.415278    0.132754    0.002310    0.707608    
#> 219: 0.336703    0.966426    3.431660    0.413066    0.132134    0.002322    0.707147    
#> 220: 0.333847    0.967773    3.431211    0.414619    0.131363    0.002319    0.706896    
#> 221: 0.332281    0.967928    3.430622    0.414318    0.130997    0.002311    0.706754    
#> 222: 0.330975    0.967661    3.430226    0.411831    0.130513    0.002338    0.706950    
#> 223: 0.329996    0.967911    3.430068    0.409318    0.131002    0.002348    0.707133    
#> 224: 0.329720    0.967095    3.430174    0.409589    0.131399    0.002352    0.706354    
#> 225: 0.328560    0.965191    3.430512    0.408599    0.131836    0.002374    0.706305    
#> 226: 0.328351    0.964427    3.430464    0.406770    0.132384    0.002394    0.706691    
#> 227: 0.327180    0.964175    3.430595    0.404201    0.132384    0.002410    0.707603    
#> 228: 0.325673    0.964484    3.430267    0.402541    0.132927    0.002410    0.707711    
#> 229: 0.324445    0.964204    3.429877    0.400740    0.133695    0.002400    0.707693    
#> 230: 0.324077    0.964504    3.429468    0.400057    0.133559    0.002388    0.707443    
#> 231: 0.322869    0.965034    3.429305    0.399663    0.133643    0.002389    0.707231    
#> 232: 0.321619    0.965754    3.429366    0.396924    0.133452    0.002387    0.707262    
#> 233: 0.320327    0.965526    3.429336    0.397377    0.133082    0.002399    0.707698    
#> 234: 0.318725    0.966057    3.429254    0.396718    0.132625    0.002401    0.707458    
#> 235: 0.318194    0.966373    3.429028    0.394135    0.132814    0.002411    0.707510    
#> 236: 0.317210    0.966348    3.428856    0.393415    0.132569    0.002424    0.707283    
#> 237: 0.316865    0.966679    3.428698    0.391967    0.131747    0.002445    0.707611    
#> 238: 0.316579    0.966889    3.428422    0.392426    0.131682    0.002484    0.707579    
#> 239: 0.315980    0.967610    3.428205    0.392231    0.131492    0.002491    0.707534    
#> 240: 0.315746    0.967562    3.428123    0.391871    0.131153    0.002482    0.707136    
#> 241: 0.315833    0.967023    3.428070    0.392199    0.130562    0.002477    0.707098    
#> 242: 0.314841    0.966519    3.427924    0.392325    0.130632    0.002489    0.707147    
#> 243: 0.314270    0.967054    3.427735    0.392230    0.130866    0.002508    0.707250    
#> 244: 0.313973    0.967159    3.427671    0.392292    0.130682    0.002541    0.706867    
#> 245: 0.314211    0.966887    3.427641    0.391377    0.130236    0.002557    0.706963    
#> 246: 0.314820    0.966884    3.427703    0.390154    0.129757    0.002591    0.706868    
#> 247: 0.314853    0.966686    3.427953    0.387605    0.129155    0.002612    0.707106    
#> 248: 0.314449    0.966801    3.428049    0.386182    0.128506    0.002621    0.707129    
#> 249: 0.314034    0.966747    3.428020    0.384726    0.128190    0.002628    0.706922    
#> 250: 0.314964    0.967259    3.428065    0.384537    0.127939    0.002649    0.706804    
#> 251: 0.315410    0.967945    3.428082    0.383962    0.127360    0.002649    0.706959    
#> 252: 0.315477    0.968559    3.428121    0.384626    0.127103    0.002644    0.706859    
#> 253: 0.315273    0.968814    3.428119    0.385432    0.126425    0.002658    0.706719    
#> 254: 0.315456    0.968975    3.428191    0.386778    0.126216    0.002667    0.706647    
#> 255: 0.316179    0.969561    3.428325    0.387818    0.125932    0.002691    0.706844    
#> 256: 0.316097    0.969672    3.428404    0.386520    0.125913    0.002709    0.706889    
#> 257: 0.315868    0.969334    3.428349    0.386042    0.126005    0.002723    0.706836    
#> 258: 0.315660    0.969206    3.428266    0.384863    0.126351    0.002721    0.707017    
#> 259: 0.315291    0.969047    3.428225    0.384004    0.126293    0.002715    0.706876    
#> 260: 0.314958    0.969354    3.428293    0.384101    0.126252    0.002707    0.706686    
#> 261: 0.315394    0.969184    3.428377    0.384871    0.125958    0.002703    0.706743    
#> 262: 0.315500    0.969254    3.428481    0.384174    0.125721    0.002700    0.706645    
#> 263: 0.315192    0.969262    3.428555    0.383452    0.125507    0.002707    0.706666    
#> 264: 0.314812    0.969173    3.428578    0.383464    0.125518    0.002708    0.706930    
#> 265: 0.314189    0.969317    3.428589    0.384289    0.125284    0.002713    0.706901    
#> 266: 0.314650    0.969420    3.428535    0.385276    0.125272    0.002716    0.706946    
#> 267: 0.315215    0.969577    3.428585    0.386728    0.125230    0.002713    0.706792    
#> 268: 0.315453    0.969315    3.428558    0.386314    0.125280    0.002709    0.706713    
#> 269: 0.315940    0.969206    3.428552    0.385893    0.125042    0.002711    0.706568    
#> 270: 0.316056    0.969658    3.428421    0.386538    0.124837    0.002710    0.706208    
#> 271: 0.315686    0.970116    3.428185    0.386903    0.124507    0.002716    0.706204    
#> 272: 0.315414    0.970178    3.427986    0.385890    0.124475    0.002713    0.706385    
#> 273: 0.314779    0.970526    3.427805    0.385353    0.124371    0.002717    0.706580    
#> 274: 0.314080    0.971152    3.427647    0.385178    0.124208    0.002709    0.706763    
#> 275: 0.313849    0.971318    3.427458    0.383867    0.124117    0.002709    0.706947    
#> 276: 0.313619    0.971522    3.427542    0.383836    0.123978    0.002732    0.706939    
#> 277: 0.313686    0.971788    3.427710    0.383846    0.123944    0.002759    0.706686    
#> 278: 0.313814    0.971869    3.427810    0.384103    0.123840    0.002774    0.706539    
#> 279: 0.313704    0.971785    3.427856    0.383768    0.123750    0.002792    0.706622    
#> 280: 0.313384    0.971788    3.427938    0.382998    0.123631    0.002793    0.706573    
#> 281: 0.312759    0.971669    3.427990    0.381631    0.123322    0.002786    0.706742    
#> 282: 0.312562    0.971857    3.427988    0.381028    0.123092    0.002798    0.706784    
#> 283: 0.312404    0.972067    3.428032    0.380609    0.123220    0.002812    0.707065    
#> 284: 0.312178    0.972105    3.428024    0.379773    0.123095    0.002821    0.706983    
#> 285: 0.312461    0.972313    3.427971    0.379362    0.122934    0.002830    0.706837    
#> 286: 0.312650    0.972391    3.427901    0.379192    0.122747    0.002838    0.706723    
#> 287: 0.312212    0.972673    3.427798    0.378626    0.122583    0.002841    0.706385    
#> 288: 0.312331    0.972667    3.427719    0.378148    0.122329    0.002840    0.706252    
#> 289: 0.312543    0.972878    3.427668    0.377985    0.122350    0.002837    0.706112    
#> 290: 0.312399    0.972959    3.427642    0.377767    0.122515    0.002835    0.706152    
#> 291: 0.312501    0.973038    3.427686    0.376780    0.122621    0.002833    0.706207    
#> 292: 0.312502    0.972825    3.427727    0.376153    0.122607    0.002829    0.706278    
#> 293: 0.312344    0.972759    3.427713    0.375543    0.122559    0.002825    0.706152    
#> 294: 0.312161    0.972763    3.427707    0.374984    0.122411    0.002821    0.706166    
#> 295: 0.312256    0.972361    3.427703    0.375157    0.122312    0.002816    0.706052    
#> 296: 0.312436    0.972317    3.427779    0.375898    0.122217    0.002822    0.705981    
#> 297: 0.312597    0.972436    3.427836    0.375874    0.122307    0.002829    0.705878    
#> 298: 0.312776    0.972481    3.427943    0.376287    0.122353    0.002836    0.705766    
#> 299: 0.313061    0.972421    3.428027    0.376434    0.122349    0.002834    0.705652    
#> 300: 0.313112    0.972228    3.428097    0.376089    0.122190    0.002825    0.705569    
#> 301: 0.313351    0.971864    3.428131    0.375782    0.122013    0.002818    0.705708    
#> 302: 0.313500    0.971866    3.428176    0.375298    0.121858    0.002816    0.705725    
#> 303: 0.313390    0.971890    3.428168    0.374851    0.121603    0.002816    0.705612    
#> 304: 0.313489    0.972202    3.428118    0.375021    0.121454    0.002821    0.705546    
#> 305: 0.313349    0.971997    3.428097    0.374907    0.121314    0.002832    0.705552    
#> 306: 0.313207    0.971959    3.428057    0.374609    0.121180    0.002838    0.705568    
#> 307: 0.313096    0.971799    3.428015    0.374050    0.121074    0.002835    0.705530    
#> 308: 0.312744    0.971697    3.427942    0.374227    0.121149    0.002833    0.705569    
#> 309: 0.312611    0.971454    3.427892    0.373957    0.121183    0.002826    0.705640    
#> 310: 0.312750    0.971429    3.427807    0.373622    0.121049    0.002824    0.705707    
#> 311: 0.312746    0.971408    3.427778    0.373441    0.120862    0.002832    0.705699    
#> 312: 0.312235    0.971290    3.427716    0.372804    0.120859    0.002831    0.705772    
#> 313: 0.311927    0.971065    3.427624    0.372147    0.120967    0.002828    0.705940    
#> 314: 0.311662    0.971241    3.427525    0.372067    0.121015    0.002825    0.705918    
#> 315: 0.311563    0.971498    3.427501    0.371493    0.121181    0.002832    0.705903    
#> 316: 0.311360    0.971642    3.427473    0.370959    0.121148    0.002837    0.706031    
#> 317: 0.311206    0.971677    3.427492    0.370345    0.121171    0.002836    0.706273    
#> 318: 0.311226    0.971715    3.427447    0.370051    0.121316    0.002835    0.706330    
#> 319: 0.311117    0.971772    3.427443    0.370152    0.121484    0.002834    0.706311    
#> 320: 0.310866    0.971794    3.427395    0.370372    0.121572    0.002834    0.706367    
#> 321: 0.310658    0.972003    3.427307    0.369949    0.121653    0.002837    0.706448    
#> 322: 0.310519    0.971962    3.427195    0.369492    0.121536    0.002838    0.706381    
#> 323: 0.310045    0.971823    3.427147    0.369577    0.121377    0.002842    0.706287    
#> 324: 0.309988    0.971910    3.427185    0.369554    0.121328    0.002847    0.706184    
#> 325: 0.310102    0.971865    3.427150    0.369154    0.121310    0.002846    0.706092    
#> 326: 0.310144    0.971633    3.427094    0.368644    0.121403    0.002842    0.705999    
#> 327: 0.310285    0.971740    3.427101    0.368197    0.121287    0.002839    0.705996    
#> 328: 0.310338    0.971883    3.427070    0.368033    0.121233    0.002835    0.705907    
#> 329: 0.310218    0.971906    3.427053    0.367589    0.121202    0.002833    0.705985    
#> 330: 0.309999    0.971961    3.427039    0.367038    0.121087    0.002832    0.706036    
#> 331: 0.310094    0.972134    3.427047    0.366731    0.120997    0.002831    0.706046    
#> 332: 0.310005    0.972173    3.427048    0.366453    0.120988    0.002833    0.705998    
#> 333: 0.309782    0.972309    3.426988    0.366072    0.121003    0.002839    0.705938    
#> 334: 0.309686    0.972239    3.426868    0.366068    0.120882    0.002843    0.705933    
#> 335: 0.309251    0.972429    3.426724    0.366031    0.120815    0.002848    0.705883    
#> 336: 0.308687    0.972651    3.426613    0.366225    0.120659    0.002856    0.705919    
#> 337: 0.308422    0.973007    3.426470    0.366604    0.120522    0.002859    0.706010    
#> 338: 0.308069    0.973086    3.426398    0.366581    0.120406    0.002853    0.706077    
#> 339: 0.307808    0.973253    3.426341    0.366412    0.120367    0.002850    0.706087    
#> 340: 0.307816    0.973109    3.426345    0.366196    0.120310    0.002855    0.706099    
#> 341: 0.307924    0.973094    3.426357    0.365706    0.120249    0.002858    0.706193    
#> 342: 0.307640    0.972971    3.426316    0.365487    0.120229    0.002856    0.706085    
#> 343: 0.307262    0.972977    3.426306    0.365141    0.120201    0.002852    0.706052    
#> 344: 0.307248    0.973013    3.426351    0.364594    0.120283    0.002852    0.706073    
#> 345: 0.307261    0.972888    3.426408    0.363910    0.120344    0.002853    0.706135    
#> 346: 0.307156    0.972870    3.426416    0.363078    0.120303    0.002846    0.706096    
#> 347: 0.307334    0.972797    3.426362    0.363570    0.120260    0.002842    0.706024    
#> 348: 0.307085    0.972827    3.426272    0.363725    0.120288    0.002846    0.706056    
#> 349: 0.306767    0.972787    3.426186    0.363723    0.120384    0.002850    0.706122    
#> 350: 0.306473    0.972740    3.426130    0.363777    0.120441    0.002853    0.706066    
#> 351: 0.306463    0.973042    3.426065    0.363966    0.120535    0.002858    0.706036    
#> 352: 0.306231    0.973231    3.426037    0.364161    0.120588    0.002860    0.706066    
#> 353: 0.306262    0.973430    3.426005    0.363881    0.120543    0.002857    0.705988    
#> 354: 0.306334    0.973464    3.425932    0.363677    0.120585    0.002854    0.705990    
#> 355: 0.306584    0.973264    3.425880    0.363742    0.120623    0.002858    0.706036    
#> 356: 0.306529    0.973243    3.425877    0.363972    0.120585    0.002854    0.706150    
#> 357: 0.306788    0.973338    3.425861    0.364052    0.120564    0.002849    0.706203    
#> 358: 0.306766    0.973295    3.425883    0.364058    0.120474    0.002848    0.706025    
#> 359: 0.306743    0.973328    3.425880    0.364274    0.120456    0.002848    0.705863    
#> 360: 0.306778    0.973244    3.425834    0.364376    0.120409    0.002846    0.705736    
#> 361: 0.306599    0.973119    3.425798    0.364414    0.120301    0.002844    0.705681    
#> 362: 0.306613    0.973061    3.425756    0.364258    0.120303    0.002842    0.705486    
#> 363: 0.306577    0.973037    3.425720    0.364182    0.120305    0.002841    0.705327    
#> 364: 0.306705    0.973009    3.425691    0.364173    0.120281    0.002843    0.705291    
#> 365: 0.306752    0.972933    3.425712    0.363980    0.120336    0.002845    0.705329    
#> 366: 0.306858    0.972989    3.425703    0.363589    0.120308    0.002848    0.705381    
#> 367: 0.306993    0.972945    3.425714    0.363628    0.120306    0.002844    0.705308    
#> 368: 0.307014    0.972945    3.425688    0.363495    0.120328    0.002840    0.705264    
#> 369: 0.306886    0.972892    3.425714    0.363129    0.120404    0.002841    0.705256    
#> 370: 0.306814    0.973048    3.425718    0.362703    0.120486    0.002839    0.705305    
#> 371: 0.306718    0.973087    3.425659    0.362546    0.120478    0.002835    0.705354    
#> 372: 0.306312    0.973179    3.425573    0.362773    0.120452    0.002833    0.705373    
#> 373: 0.306072    0.973157    3.425484    0.362997    0.120526    0.002831    0.705456    
#> 374: 0.306076    0.973169    3.425395    0.362985    0.120593    0.002828    0.705461    
#> 375: 0.305943    0.973231    3.425328    0.363145    0.120716    0.002825    0.705475    
#> 376: 0.305901    0.973307    3.425271    0.363391    0.120657    0.002821    0.705449    
#> 377: 0.305911    0.973349    3.425239    0.363562    0.120574    0.002818    0.705471    
#> 378: 0.305830    0.973335    3.425195    0.363546    0.120489    0.002818    0.705493    
#> 379: 0.305510    0.973259    3.425170    0.363247    0.120411    0.002819    0.705413    
#> 380: 0.305397    0.973164    3.425161    0.363294    0.120427    0.002815    0.705405    
#> 381: 0.305126    0.973076    3.425128    0.363042    0.120412    0.002811    0.705528    
#> 382: 0.305078    0.973120    3.425107    0.362766    0.120378    0.002807    0.705432    
#> 383: 0.305007    0.973109    3.425097    0.362456    0.120390    0.002801    0.705457    
#> 384: 0.304887    0.973090    3.425102    0.362456    0.120415    0.002796    0.705463    
#> 385: 0.305070    0.972986    3.425151    0.362786    0.120377    0.002801    0.705460    
#> 386: 0.305277    0.973084    3.425205    0.363106    0.120372    0.002804    0.705497    
#> 387: 0.305391    0.973272    3.425226    0.362750    0.120298    0.002806    0.705593    
#> 388: 0.305390    0.973220    3.425248    0.362416    0.120347    0.002806    0.705688    
#> 389: 0.305366    0.973120    3.425221    0.362114    0.120268    0.002801    0.705803    
#> 390: 0.305303    0.973103    3.425190    0.362046    0.120251    0.002796    0.705840    
#> 391: 0.305485    0.973092    3.425170    0.362061    0.120295    0.002791    0.705853    
#> 392: 0.305582    0.973119    3.425192    0.362400    0.120305    0.002786    0.705767    
#> 393: 0.305496    0.973093    3.425158    0.362566    0.120325    0.002784    0.705665    
#> 394: 0.305516    0.973144    3.425121    0.362658    0.120371    0.002782    0.705604    
#> 395: 0.305608    0.973281    3.425103    0.362545    0.120404    0.002779    0.705634    
#> 396: 0.305758    0.973383    3.425100    0.362597    0.120425    0.002777    0.705584    
#> 397: 0.305756    0.973322    3.425104    0.362789    0.120447    0.002774    0.705596    
#> 398: 0.305827    0.973249    3.425121    0.362837    0.120521    0.002777    0.705625    
#> 399: 0.305807    0.973205    3.425177    0.362932    0.120506    0.002780    0.705583    
#> 400: 0.305898    0.973119    3.425209    0.362829    0.120574    0.002780    0.705604    
#> 401: 0.306030    0.973117    3.425231    0.362659    0.120532    0.002784    0.705577    
#> 402: 0.306034    0.973312    3.425245    0.362473    0.120514    0.002790    0.705565    
#> 403: 0.306184    0.973339    3.425259    0.362378    0.120396    0.002799    0.705479    
#> 404: 0.306064    0.973450    3.425224    0.361990    0.120407    0.002803    0.705422    
#> 405: 0.305997    0.973520    3.425220    0.361780    0.120397    0.002806    0.705495    
#> 406: 0.306016    0.973550    3.425202    0.362134    0.120442    0.002810    0.705429    
#> 407: 0.305941    0.973588    3.425216    0.362317    0.120418    0.002810    0.705446    
#> 408: 0.306043    0.973648    3.425209    0.362287    0.120436    0.002811    0.705446    
#> 409: 0.305885    0.973579    3.425195    0.362437    0.120399    0.002809    0.705384    
#> 410: 0.305767    0.973567    3.425135    0.362266    0.120446    0.002806    0.705358    
#> 411: 0.305703    0.973438    3.425103    0.362370    0.120464    0.002805    0.705335    
#> 412: 0.305556    0.973604    3.425051    0.362533    0.120454    0.002807    0.705385    
#> 413: 0.305484    0.973685    3.425003    0.362630    0.120452    0.002804    0.705460    
#> 414: 0.305410    0.973654    3.424976    0.362569    0.120443    0.002802    0.705433    
#> 415: 0.305375    0.973688    3.424920    0.362546    0.120521    0.002797    0.705527    
#> 416: 0.305228    0.973806    3.424864    0.362618    0.120571    0.002794    0.705499    
#> 417: 0.305016    0.973871    3.424777    0.362878    0.120524    0.002792    0.705501    
#> 418: 0.304765    0.973993    3.424699    0.362887    0.120610    0.002789    0.705499    
#> 419: 0.304681    0.974029    3.424642    0.363156    0.120605    0.002790    0.705460    
#> 420: 0.304670    0.974021    3.424596    0.363053    0.120554    0.002790    0.705480    
#> 421: 0.304592    0.974031    3.424592    0.363175    0.120531    0.002791    0.705414    
#> 422: 0.304591    0.973998    3.424593    0.363227    0.120582    0.002794    0.705339    
#> 423: 0.304774    0.973975    3.424611    0.363031    0.120561    0.002797    0.705332    
#> 424: 0.304814    0.973857    3.424614    0.362842    0.120548    0.002793    0.705344    
#> 425: 0.304754    0.973868    3.424620    0.362863    0.120563    0.002790    0.705303    
#> 426: 0.304817    0.973983    3.424634    0.362843    0.120506    0.002788    0.705263    
#> 427: 0.304725    0.973932    3.424676    0.362763    0.120488    0.002787    0.705203    
#> 428: 0.304699    0.973978    3.424671    0.362668    0.120445    0.002786    0.705214    
#> 429: 0.304585    0.974055    3.424668    0.362393    0.120463    0.002786    0.705171    
#> 430: 0.304503    0.974125    3.424662    0.362376    0.120467    0.002786    0.705216    
#> 431: 0.304304    0.974139    3.424595    0.362222    0.120486    0.002787    0.705151    
#> 432: 0.304064    0.974236    3.424555    0.361876    0.120461    0.002789    0.705160    
#> 433: 0.304005    0.974387    3.424535    0.361565    0.120465    0.002789    0.705185    
#> 434: 0.303768    0.974387    3.424518    0.361518    0.120494    0.002791    0.705142    
#> 435: 0.303763    0.974346    3.424508    0.361296    0.120571    0.002788    0.705138    
#> 436: 0.303717    0.974268    3.424523    0.361047    0.120558    0.002785    0.705148    
#> 437: 0.303717    0.974178    3.424520    0.360793    0.120479    0.002786    0.705108    
#> 438: 0.303835    0.974235    3.424540    0.360673    0.120384    0.002785    0.705087    
#> 439: 0.303938    0.974241    3.424560    0.360611    0.120348    0.002779    0.705060    
#> 440: 0.303934    0.974234    3.424615    0.360521    0.120357    0.002778    0.705129    
#> 441: 0.304015    0.974204    3.424652    0.360631    0.120347    0.002777    0.705123    
#> 442: 0.303793    0.974256    3.424666    0.360513    0.120315    0.002774    0.705059    
#> 443: 0.303744    0.974316    3.424652    0.360409    0.120248    0.002777    0.704987    
#> 444: 0.303587    0.974383    3.424626    0.360447    0.120154    0.002783    0.704933    
#> 445: 0.303495    0.974429    3.424586    0.360352    0.120044    0.002788    0.704957    
#> 446: 0.303277    0.974495    3.424543    0.360230    0.119966    0.002794    0.704930    
#> 447: 0.303310    0.974516    3.424540    0.360119    0.119965    0.002801    0.704859    
#> 448: 0.303174    0.974578    3.424543    0.360102    0.119934    0.002809    0.704762    
#> 449: 0.303136    0.974686    3.424604    0.360071    0.120000    0.002814    0.704698    
#> 450: 0.303240    0.974612    3.424649    0.360139    0.120018    0.002821    0.704669    
#> 451: 0.303396    0.974529    3.424672    0.360355    0.120053    0.002827    0.704679    
#> 452: 0.303423    0.974544    3.424656    0.360621    0.120052    0.002827    0.704654    
#> 453: 0.303500    0.974564    3.424641    0.360884    0.120013    0.002823    0.704653    
#> 454: 0.303575    0.974685    3.424619    0.360960    0.120000    0.002822    0.704661    
#> 455: 0.303607    0.974714    3.424586    0.360863    0.119933    0.002820    0.704638    
#> 456: 0.303581    0.974679    3.424576    0.360728    0.119993    0.002822    0.704662    
#> 457: 0.303474    0.974619    3.424584    0.360862    0.120084    0.002821    0.704625    
#> 458: 0.303433    0.974690    3.424612    0.360670    0.120168    0.002818    0.704579    
#> 459: 0.303259    0.974759    3.424617    0.360496    0.120209    0.002814    0.704638    
#> 460: 0.303066    0.974752    3.424601    0.360433    0.120326    0.002810    0.704706    
#> 461: 0.302964    0.974770    3.424600    0.360366    0.120338    0.002810    0.704760    
#> 462: 0.302830    0.974804    3.424613    0.360264    0.120291    0.002811    0.704691    
#> 463: 0.302771    0.974878    3.424605    0.360050    0.120324    0.002813    0.704640    
#> 464: 0.302732    0.974940    3.424590    0.360059    0.120298    0.002816    0.704541    
#> 465: 0.302766    0.974845    3.424582    0.360123    0.120284    0.002815    0.704537    
#> 466: 0.302837    0.974883    3.424554    0.360109    0.120233    0.002814    0.704550    
#> 467: 0.302802    0.974855    3.424528    0.360315    0.120243    0.002816    0.704416    
#> 468: 0.302937    0.974852    3.424513    0.360338    0.120245    0.002816    0.704506    
#> 469: 0.302945    0.974938    3.424503    0.360255    0.120257    0.002819    0.704557    
#> 470: 0.302900    0.974958    3.424487    0.360435    0.120328    0.002821    0.704497    
#> 471: 0.302833    0.974914    3.424480    0.360559    0.120318    0.002821    0.704501    
#> 472: 0.302934    0.974870    3.424509    0.360803    0.120356    0.002821    0.704480    
#> 473: 0.302933    0.974815    3.424524    0.360896    0.120397    0.002823    0.704414    
#> 474: 0.302920    0.974847    3.424536    0.360929    0.120443    0.002827    0.704433    
#> 475: 0.303040    0.974867    3.424539    0.360937    0.120415    0.002831    0.704378    
#> 476: 0.303024    0.974850    3.424524    0.360750    0.120397    0.002833    0.704404    
#> 477: 0.303006    0.974836    3.424534    0.360626    0.120382    0.002836    0.704377    
#> 478: 0.302886    0.974812    3.424546    0.360575    0.120414    0.002838    0.704382    
#> 479: 0.302882    0.974882    3.424559    0.360757    0.120436    0.002838    0.704404    
#> 480: 0.302890    0.974829    3.424577    0.360911    0.120490    0.002841    0.704388    
#> 481: 0.302877    0.974785    3.424556    0.361214    0.120547    0.002843    0.704397    
#> 482: 0.302920    0.974795    3.424529    0.361580    0.120605    0.002842    0.704380    
#> 483: 0.302823    0.974792    3.424506    0.361616    0.120601    0.002842    0.704370    
#> 484: 0.302855    0.974694    3.424495    0.362113    0.120586    0.002842    0.704396    
#> 485: 0.302801    0.974678    3.424489    0.362225    0.120561    0.002841    0.704439    
#> 486: 0.302858    0.974680    3.424487    0.362444    0.120467    0.002839    0.704449    
#> 487: 0.303018    0.974737    3.424505    0.362748    0.120403    0.002837    0.704410    
#> 488: 0.303053    0.974734    3.424529    0.362868    0.120414    0.002836    0.704395    
#> 489: 0.303201    0.974745    3.424538    0.363086    0.120416    0.002838    0.704394    
#> 490: 0.303351    0.974732    3.424556    0.363204    0.120412    0.002838    0.704438    
#> 491: 0.303483    0.974745    3.424585    0.363346    0.120392    0.002839    0.704375    
#> 492: 0.303507    0.974784    3.424586    0.363462    0.120437    0.002839    0.704412    
#> 493: 0.303461    0.974744    3.424600    0.363276    0.120524    0.002841    0.704404    
#> 494: 0.303584    0.974774    3.424598    0.363142    0.120587    0.002840    0.704368    
#> 495: 0.303692    0.974789    3.424620    0.363031    0.120595    0.002842    0.704355    
#> 496: 0.303854    0.974822    3.424633    0.362969    0.120657    0.002842    0.704369    
#> 497: 0.303898    0.974802    3.424647    0.362945    0.120641    0.002841    0.704392    
#> 498: 0.303963    0.974742    3.424660    0.362948    0.120582    0.002840    0.704334    
#> 499: 0.304025    0.974669    3.424652    0.362957    0.120525    0.002839    0.704318    
#> 500: 0.304048    0.974574    3.424659    0.362716    0.120521    0.002837    0.704275    
#> [====|====|=>-------------------------------------]  25%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.293896    0.965977    3.438175    0.392401    0.070359    0.016800    1.704848    
#> 002: 0.166280    0.934117    3.456379    0.372781    0.066841    0.015960    1.160224    
#> 003: 0.207576    0.829577    3.457981    0.354142    0.063499    0.015162    0.910142    
#> 004: 0.156608    0.868808    3.434042    0.336435    0.060324    0.014404    0.784342    
#> 005: 0.166324    0.869347    3.408560    0.319613    0.057308    0.013684    0.729848    
#> 006: 0.104472    0.887285    3.393649    0.303633    0.054443    0.013000    0.681599    
#> 007: 0.055487    0.916480    3.375085    0.330285    0.066492    0.012727    0.677588    
#> 008: 0.022057    0.929588    3.365587    0.321933    0.063167    0.013351    0.663471    
#> 009: -0.004834   0.926174    3.344418    0.342673    0.060009    0.012683    0.652992    
#> 010: 0.010544    0.921504    3.351760    0.332108    0.057008    0.012049    0.684735    
#> 011: 0.054097    0.938364    3.357923    0.365064    0.064901    0.011657    0.684693    
#> 012: 0.059756    0.950559    3.358984    0.378750    0.061656    0.011376    0.661777    
#> 013: 0.029937    0.947256    3.350364    0.376494    0.060569    0.012361    0.656930    
#> 014: 0.051460    0.921312    3.354535    0.441280    0.057541    0.011743    0.669003    
#> 015: 0.045068    0.927870    3.368705    0.419216    0.062667    0.011156    0.662197    
#> 016: 0.069514    0.931448    3.369612    0.398255    0.060382    0.012968    0.659871    
#> 017: 0.053491    0.936291    3.368108    0.378342    0.059563    0.012417    0.651721    
#> 018: 0.068028    0.940082    3.362446    0.367117    0.056585    0.011796    0.654536    
#> 019: 0.047373    0.939477    3.359924    0.348761    0.060347    0.011206    0.641671    
#> 020: 0.084427    0.944204    3.366793    0.388417    0.057330    0.012217    0.658873    
#> 021: 0.078918    0.923574    3.378900    0.394584    0.054464    0.012146    0.648691    
#> 022: 0.077865    0.922540    3.384650    0.393255    0.051740    0.013736    0.643146    
#> 023: 0.143273    0.912707    3.396430    0.431502    0.049153    0.016844    0.663729    
#> 024: 0.118841    0.937167    3.382323    0.412351    0.055109    0.016002    0.658026    
#> 025: 0.101861    0.921482    3.378581    0.465083    0.054816    0.015202    0.655445    
#> 026: 0.068713    0.949655    3.356781    0.441829    0.053644    0.014442    0.666152    
#> 027: 0.038508    0.963129    3.347736    0.419738    0.054136    0.013720    0.662944    
#> 028: 0.054279    0.952466    3.349374    0.398751    0.057703    0.013034    0.649086    
#> 029: 0.056928    0.944884    3.345223    0.388101    0.054818    0.012382    0.662266    
#> 030: 0.049928    0.946891    3.355373    0.384815    0.053375    0.011763    0.658401    
#> 031: 0.032912    0.937832    3.361610    0.380410    0.056695    0.011175    0.657106    
#> 032: 0.053429    0.927353    3.362097    0.400962    0.053860    0.010616    0.657652    
#> 033: 0.041570    0.939273    3.359185    0.402190    0.051167    0.010085    0.658702    
#> 034: 0.073269    0.938152    3.362226    0.411047    0.048609    0.012218    0.649001    
#> 035: 0.052625    0.941108    3.348487    0.417317    0.056893    0.011607    0.642085    
#> 036: 0.040506    0.954246    3.339559    0.448138    0.054049    0.012320    0.648839    
#> 037: 0.032467    0.952950    3.343608    0.425731    0.051346    0.011704    0.671571    
#> 038: 0.025971    0.941860    3.340286    0.404444    0.049251    0.011119    0.675564    
#> 039: 0.043556    0.946208    3.347816    0.430756    0.046788    0.013857    0.664799    
#> 040: 0.055446    0.949708    3.354924    0.437140    0.044449    0.013505    0.670683    
#> 041: 0.001843    0.936625    3.354404    0.415283    0.051032    0.012830    0.647186    
#> 042: 0.047196    0.943824    3.350579    0.394519    0.049563    0.012189    0.642087    
#> 043: 0.028050    0.964710    3.333470    0.405633    0.051217    0.011579    0.667060    
#> 044: 0.004836    0.969710    3.326239    0.406190    0.051260    0.011000    0.663314    
#> 045: -0.015634   0.973011    3.320423    0.391374    0.054830    0.010450    0.652996    
#> 046: 0.019145    0.955755    3.336686    0.371805    0.052089    0.009928    0.657838    
#> 047: 0.012096    0.957761    3.333666    0.398224    0.049484    0.009800    0.647957    
#> 048: 0.015498    0.962459    3.334945    0.378313    0.047158    0.010792    0.660978    
#> 049: 0.024793    0.963368    3.332072    0.441850    0.052818    0.011364    0.662912    
#> 050: 0.042444    0.933897    3.347593    0.426491    0.055960    0.013079    0.674088    
#> 051: 0.019463    0.951143    3.340479    0.405166    0.053162    0.014739    0.672876    
#> 052: 0.051555    0.961737    3.349212    0.404372    0.050504    0.014002    0.667660    
#> 053: 0.038152    0.961172    3.349086    0.411006    0.048174    0.013478    0.663380    
#> 054: 0.021059    0.949228    3.354144    0.406733    0.056174    0.012804    0.671371    
#> 055: 0.044028    0.953055    3.352105    0.462176    0.061535    0.013250    0.653234    
#> 056: 0.077741    0.927475    3.352968    0.533163    0.058458    0.012815    0.667457    
#> 057: 0.069177    0.928537    3.352127    0.506504    0.064344    0.013195    0.662671    
#> 058: 0.098850    0.933747    3.354150    0.481179    0.061126    0.012535    0.668053    
#> 059: 0.098210    0.897867    3.362910    0.497825    0.058070    0.015459    0.686949    
#> 060: 0.136380    0.897598    3.384477    0.508169    0.055167    0.016525    0.693398    
#> 061: 0.138941    0.896915    3.396689    0.482760    0.052408    0.020364    0.690962    
#> 062: 0.123545    0.923687    3.384542    0.548666    0.053522    0.019346    0.696381    
#> 063: 0.141550    0.911531    3.378769    0.640486    0.050846    0.018379    0.694727    
#> 064: 0.088249    0.915320    3.383388    0.608462    0.048303    0.017460    0.692094    
#> 065: 0.066489    0.914257    3.355796    0.578039    0.054584    0.016587    0.683288    
#> 066: 0.083114    0.937808    3.361907    0.553199    0.051855    0.017044    0.658297    
#> 067: 0.054865    0.934732    3.354382    0.546410    0.049262    0.020014    0.650080    
#> 068: 0.010473    0.961734    3.346416    0.519090    0.047455    0.019013    0.656124    
#> 069: 0.029509    0.955600    3.328629    0.545155    0.045082    0.018062    0.663761    
#> 070: 0.007505    0.951248    3.336029    0.533076    0.046430    0.017159    0.653465    
#> 071: -0.002474   0.976550    3.312732    0.506423    0.046906    0.016301    0.671646    
#> 072: -0.032322   0.948411    3.320381    0.481101    0.044561    0.016732    0.656944    
#> 073: 0.022538    0.959074    3.350368    0.457046    0.042333    0.018140    0.666873    
#> 074: 0.014342    0.933305    3.341472    0.434194    0.041266    0.017233    0.662353    
#> 075: 0.033856    0.931738    3.358060    0.481055    0.041867    0.016372    0.665899    
#> 076: 0.076054    0.932399    3.365991    0.507091    0.039773    0.015627    0.676031    
#> 077: 0.107653    0.922770    3.362149    0.503427    0.040136    0.018580    0.672903    
#> 078: 0.087279    0.924478    3.377476    0.569137    0.040581    0.017851    0.649236    
#> 079: 0.095196    0.943748    3.370236    0.540680    0.038552    0.019186    0.667459    
#> 080: 0.098688    0.930296    3.361378    0.513646    0.036713    0.018227    0.670665    
#> 081: 0.068538    0.939439    3.359656    0.487964    0.038475    0.017316    0.671113    
#> 082: 0.051324    0.937024    3.360629    0.463565    0.044436    0.016450    0.689928    
#> 083: 0.062106    0.957818    3.360182    0.440387    0.044042    0.015627    0.671551    
#> 084: 0.081147    0.951982    3.350319    0.435306    0.049624    0.014846    0.656162    
#> 085: 0.068843    0.952068    3.368442    0.473737    0.053309    0.015631    0.656987    
#> 086: 0.065101    0.948265    3.356947    0.450050    0.050644    0.014849    0.645288    
#> 087: 0.040862    0.937027    3.352545    0.427548    0.048112    0.016620    0.642263    
#> 088: 0.009178    0.938271    3.354816    0.406170    0.046754    0.015789    0.653230    
#> 089: 0.021606    0.956300    3.341866    0.446004    0.044416    0.016864    0.649841    
#> 090: 0.001358    0.954695    3.330913    0.441049    0.044859    0.018262    0.668353    
#> 091: -0.020974   0.950857    3.324053    0.418996    0.042616    0.017440    0.684600    
#> 092: 0.008273    0.970757    3.326124    0.478987    0.040485    0.018947    0.662052    
#> 093: -0.020175   0.969650    3.320530    0.455037    0.040203    0.018000    0.663407    
#> 094: -0.004218   0.985053    3.319320    0.445519    0.044254    0.018290    0.672652    
#> 095: -0.010739   0.984806    3.326676    0.434611    0.043439    0.017376    0.665730    
#> 096: -0.032342   0.968517    3.318669    0.412880    0.041267    0.016507    0.667888    
#> 097: -0.013990   0.984338    3.323365    0.410046    0.039203    0.015682    0.645907    
#> 098: 0.012360    0.978068    3.339850    0.389544    0.040987    0.014898    0.662869    
#> 099: -0.014272   0.952314    3.327259    0.449709    0.044163    0.014153    0.659922    
#> 100: 0.012741    0.959492    3.329969    0.431402    0.055216    0.013445    0.650283    
#> 101: -0.012171   0.962715    3.315945    0.409832    0.052455    0.012773    0.650062    
#> 102: -0.013970   0.967072    3.314135    0.389341    0.049832    0.012134    0.656533    
#> 103: 0.003026    0.974164    3.324770    0.438044    0.053284    0.011527    0.645008    
#> 104: -0.018897   0.962986    3.342952    0.416142    0.050620    0.010951    0.677974    
#> 105: 0.001335    0.960044    3.336266    0.414837    0.059305    0.010403    0.680381    
#> 106: 0.048754    0.963187    3.338483    0.472777    0.056340    0.010747    0.671805    
#> 107: 0.020765    0.976058    3.337065    0.474982    0.058848    0.012104    0.663482    
#> 108: 0.014119    0.956090    3.333887    0.492830    0.058667    0.011499    0.657112    
#> 109: 0.008187    0.958273    3.345482    0.468188    0.055733    0.012588    0.650980    
#> 110: 0.061229    0.937064    3.333921    0.563674    0.054703    0.011959    0.642855    
#> 111: 0.047137    0.953792    3.336750    0.535491    0.054868    0.011361    0.636766    
#> 112: 0.033696    0.936179    3.336593    0.508716    0.058724    0.010793    0.653609    
#> 113: 0.010357    0.966733    3.338882    0.485886    0.055788    0.010253    0.654380    
#> 114: 0.017911    0.979598    3.335621    0.480841    0.061956    0.009740    0.655495    
#> 115: 0.005544    0.954584    3.321954    0.489447    0.067937    0.009253    0.664273    
#> 116: 0.023867    0.962687    3.333787    0.464975    0.064541    0.008791    0.666376    
#> 117: 0.032410    0.969196    3.342321    0.441726    0.061314    0.008661    0.672323    
#> 118: 0.041861    0.949451    3.345301    0.419640    0.058739    0.008580    0.662856    
#> 119: 0.044567    0.967091    3.347722    0.421842    0.055802    0.009688    0.679689    
#> 120: -0.025193   0.954203    3.333439    0.400750    0.053012    0.010541    0.669557    
#> 121: 0.022730    0.983717    3.337016    0.382763    0.052220    0.011377    0.668699    
#> 122: 0.029352    0.943741    3.348322    0.363624    0.049609    0.012692    0.664460    
#> 123: 0.064918    0.974938    3.346777    0.357984    0.053780    0.013109    0.647319    
#> 124: 0.061440    0.984242    3.339201    0.360828    0.055061    0.013978    0.668295    
#> 125: 0.054115    0.962171    3.338561    0.419977    0.052308    0.014491    0.661402    
#> 126: 0.017809    0.967684    3.330849    0.398978    0.049692    0.014157    0.674914    
#> 127: 0.034836    0.965708    3.330093    0.429339    0.051682    0.016610    0.669952    
#> 128: 0.041326    0.961419    3.349245    0.448575    0.049097    0.017906    0.673238    
#> 129: 0.052854    0.958336    3.353530    0.487178    0.046643    0.018584    0.675170    
#> 130: 0.078154    0.956325    3.355448    0.462819    0.044310    0.023310    0.669471    
#> 131: 0.078626    0.927176    3.353226    0.456567    0.042095    0.023847    0.656890    
#> 132: 0.009537    0.950660    3.333669    0.433739    0.046596    0.022655    0.644059    
#> 133: 0.054222    0.942333    3.336770    0.605534    0.044266    0.021522    0.651095    
#> 134: 0.043488    0.932943    3.347155    0.579550    0.042053    0.021671    0.657010    
#> 135: 0.056223    0.937628    3.346441    0.550572    0.039950    0.020587    0.653745    
#> 136: 0.063739    0.925281    3.351693    0.536521    0.037953    0.019558    0.648118    
#> 137: 0.091305    0.911561    3.365934    0.509695    0.036055    0.022419    0.670214    
#> 138: 0.072158    0.927832    3.364752    0.484210    0.037300    0.021298    0.655435    
#> 139: 0.039128    0.938944    3.369077    0.460000    0.036974    0.020233    0.671700    
#> 140: 0.096798    0.949335    3.375114    0.463180    0.035125    0.019221    0.679624    
#> 141: 0.059778    0.916588    3.373067    0.456795    0.033618    0.018260    0.686632    
#> 142: 0.107845    0.903443    3.382150    0.524744    0.033413    0.017347    0.677171    
#> 143: 0.090615    0.910887    3.376005    0.516544    0.037549    0.018708    0.673542    
#> 144: 0.123760    0.915874    3.390308    0.490717    0.036856    0.017772    0.654328    
#> 145: 0.087414    0.937071    3.362383    0.466181    0.039908    0.016890    0.664235    
#> 146: 0.051371    0.943072    3.373081    0.442872    0.039103    0.018134    0.660062    
#> 147: 0.028336    0.926059    3.351986    0.420728    0.039009    0.017227    0.650270    
#> 148: 0.015086    0.941976    3.352356    0.468915    0.048389    0.016366    0.664481    
#> 149: 0.031870    0.929478    3.358965    0.445469    0.045970    0.015679    0.659679    
#> 150: 0.046284    0.945958    3.346410    0.423196    0.052439    0.014895    0.674357    
#> 151: -0.002267   0.967496    3.336881    0.402036    0.049817    0.014150    0.666263    
#> 152: 0.005107    0.964023    3.344443    0.454180    0.042943    0.015953    0.655247    
#> 153: 0.008439    0.961349    3.349995    0.422041    0.060372    0.014285    0.654968    
#> 154: 0.045290    0.958932    3.351319    0.379749    0.050845    0.014058    0.662633    
#> 155: 0.017159    0.953991    3.346990    0.330043    0.051024    0.012161    0.661475    
#> 156: 0.034753    0.942590    3.355808    0.322652    0.045342    0.013194    0.676997    
#> 157: 0.017948    0.943452    3.350604    0.423552    0.061894    0.009634    0.675112    
#> 158: 0.062081    0.948400    3.351694    0.437951    0.061995    0.010151    0.654325    
#> 159: 0.009924    0.938670    3.349606    0.439307    0.050699    0.012417    0.661117    
#> 160: 0.049874    0.929128    3.358436    0.458539    0.055572    0.012169    0.665606    
#> 161: 0.051073    0.941074    3.355451    0.396739    0.049137    0.012609    0.658143    
#> 162: 0.045759    0.947344    3.347513    0.399747    0.049656    0.013895    0.674288    
#> 163: 0.046191    0.952473    3.341230    0.441912    0.042032    0.010459    0.668139    
#> 164: 0.055494    0.960071    3.342262    0.516660    0.048763    0.009106    0.653664    
#> 165: 0.038341    0.952519    3.351579    0.423604    0.056805    0.010322    0.648759    
#> 166: 0.058478    0.963536    3.345754    0.389262    0.063495    0.009555    0.668638    
#> 167: 0.039390    0.990075    3.340433    0.477216    0.053540    0.008740    0.668630    
#> 168: 0.017365    0.988272    3.339759    0.432669    0.055511    0.009098    0.658822    
#> 169: 0.034452    0.990994    3.326991    0.405477    0.059641    0.011237    0.664447    
#> 170: 0.039751    0.967968    3.336015    0.456628    0.054041    0.010782    0.663543    
#> 171: 0.040284    0.965489    3.340305    0.418183    0.058842    0.009691    0.653758    
#> 172: 0.018233    0.977310    3.339522    0.403721    0.051247    0.009754    0.652806    
#> 173: 0.014179    0.972080    3.328190    0.369374    0.056417    0.010047    0.661128    
#> 174: 0.032968    0.953085    3.344785    0.376482    0.060295    0.010325    0.648920    
#> 175: 0.018584    0.954405    3.335194    0.383720    0.066325    0.009749    0.661969    
#> 176: 0.007392    0.956336    3.349265    0.273585    0.059322    0.008296    0.666973    
#> 177: 0.015122    0.961202    3.351518    0.325155    0.058242    0.007344    0.667063    
#> 178: 0.052317    0.949997    3.353211    0.358628    0.055053    0.009175    0.650231    
#> 179: 0.058283    0.975053    3.348830    0.312085    0.055667    0.008546    0.672759    
#> 180: 0.036114    0.960961    3.351455    0.350091    0.055346    0.008388    0.658215    
#> 181: 0.040715    0.946979    3.342043    0.350956    0.055580    0.009561    0.644970    
#> 182: 0.031226    0.946976    3.331679    0.388581    0.058465    0.008841    0.672605    
#> 183: 0.043378    0.963838    3.339679    0.371008    0.065392    0.009499    0.672881    
#> 184: 0.034110    0.941591    3.337244    0.420007    0.054627    0.009296    0.678385    
#> 185: 0.037500    0.957420    3.341155    0.424793    0.048823    0.009376    0.670525    
#> 186: 0.053453    0.975835    3.342514    0.432166    0.061423    0.010235    0.670461    
#> 187: 0.033336    0.948879    3.348969    0.446598    0.064178    0.008628    0.647032    
#> 188: 0.013365    0.960299    3.341521    0.400510    0.062137    0.009603    0.645291    
#> 189: -0.001409   0.968770    3.346204    0.377514    0.063784    0.007576    0.660353    
#> 190: 0.024357    0.967291    3.350200    0.394183    0.052859    0.006964    0.656409    
#> 191: 0.008149    0.968834    3.342688    0.353068    0.054499    0.007441    0.644862    
#> 192: 0.010683    0.969214    3.342915    0.409485    0.054046    0.008997    0.643075    
#> 193: 0.038386    0.974245    3.347799    0.401887    0.036025    0.009454    0.666180    
#> 194: 0.021945    0.952222    3.341052    0.401090    0.058662    0.009407    0.669733    
#> 195: 0.002344    0.943149    3.331229    0.429197    0.054748    0.007155    0.680407    
#> 196: 0.031379    0.948146    3.347481    0.416314    0.065102    0.007095    0.682319    
#> 197: 0.022631    0.943233    3.359694    0.346715    0.053378    0.009209    0.698988    
#> 198: 0.053907    0.945633    3.355945    0.357790    0.051362    0.008343    0.680318    
#> 199: 0.052011    0.936408    3.362412    0.344750    0.054525    0.007356    0.671553    
#> 200: 0.061235    0.935106    3.360288    0.436458    0.073095    0.007355    0.688976    
#> 201: 0.058221    0.921576    3.363150    0.418329    0.064022    0.008447    0.690039    
#> 202: 0.050254    0.924938    3.358219    0.399096    0.064154    0.008696    0.692500    
#> 203: 0.051603    0.924262    3.355675    0.399625    0.062492    0.008564    0.692835    
#> 204: 0.048381    0.923231    3.354528    0.406020    0.062301    0.008673    0.694832    
#> 205: 0.045674    0.932217    3.354495    0.405273    0.061562    0.009057    0.695081    
#> 206: 0.041887    0.936214    3.354366    0.396973    0.061322    0.009176    0.694510    
#> 207: 0.038487    0.937028    3.352984    0.405819    0.060907    0.009297    0.692223    
#> 208: 0.035518    0.938095    3.351934    0.399268    0.058963    0.009294    0.689962    
#> 209: 0.034125    0.937551    3.351302    0.395731    0.058241    0.009279    0.688865    
#> 210: 0.033721    0.936843    3.351287    0.393527    0.057131    0.009144    0.687489    
#> 211: 0.034241    0.938007    3.351871    0.389570    0.056574    0.009090    0.685811    
#> 212: 0.036863    0.938955    3.352438    0.390002    0.056310    0.009219    0.683034    
#> 213: 0.038681    0.939303    3.352460    0.389911    0.055681    0.009563    0.684482    
#> 214: 0.040308    0.939451    3.352678    0.387992    0.054734    0.009775    0.683218    
#> 215: 0.040681    0.940256    3.353031    0.387174    0.054021    0.009848    0.682533    
#> 216: 0.044115    0.939296    3.353499    0.385337    0.054369    0.009837    0.682013    
#> 217: 0.044080    0.940084    3.353805    0.382482    0.054278    0.009841    0.681149    
#> 218: 0.045830    0.940602    3.353886    0.385881    0.054202    0.009937    0.680708    
#> 219: 0.045072    0.940090    3.353295    0.387367    0.053951    0.009994    0.680866    
#> 220: 0.042706    0.940482    3.353002    0.389106    0.053679    0.010016    0.680501    
#> 221: 0.042251    0.941529    3.352689    0.389833    0.053226    0.009934    0.679560    
#> 222: 0.042122    0.941531    3.352565    0.389749    0.053090    0.009876    0.678741    
#> 223: 0.043084    0.941504    3.352721    0.387999    0.052883    0.009915    0.677877    
#> 224: 0.043762    0.942650    3.353098    0.386755    0.053333    0.009903    0.677122    
#> 225: 0.043793    0.941558    3.353669    0.386921    0.053090    0.009881    0.676717    
#> 226: 0.043866    0.940155    3.353949    0.387013    0.053106    0.009835    0.676536    
#> 227: 0.044408    0.939748    3.354356    0.388570    0.053577    0.009774    0.676871    
#> 228: 0.044455    0.940040    3.354221    0.387893    0.054216    0.009775    0.676704    
#> 229: 0.042942    0.939959    3.353632    0.387470    0.054698    0.009673    0.676446    
#> 230: 0.042809    0.939783    3.353256    0.386648    0.054819    0.009549    0.676497    
#> 231: 0.042528    0.940766    3.352975    0.388254    0.055118    0.009450    0.676710    
#> 232: 0.040798    0.940803    3.352884    0.387837    0.055128    0.009428    0.676880    
#> 233: 0.040040    0.940104    3.352935    0.387345    0.055266    0.009435    0.677071    
#> 234: 0.039937    0.939863    3.353101    0.387726    0.055360    0.009404    0.676748    
#> 235: 0.039954    0.939495    3.353098    0.387411    0.055476    0.009321    0.676718    
#> 236: 0.040060    0.939781    3.352990    0.388099    0.055581    0.009282    0.676227    
#> 237: 0.040627    0.939829    3.352975    0.387288    0.055989    0.009238    0.676037    
#> 238: 0.041098    0.939610    3.353153    0.384946    0.056362    0.009221    0.676226    
#> 239: 0.041359    0.939862    3.353355    0.383843    0.056715    0.009256    0.675883    
#> 240: 0.041849    0.939814    3.353543    0.384627    0.056897    0.009286    0.675482    
#> 241: 0.041985    0.939449    3.353791    0.384130    0.056756    0.009287    0.675381    
#> 242: 0.041854    0.939231    3.353846    0.383386    0.056997    0.009335    0.675601    
#> 243: 0.042008    0.939820    3.353903    0.383192    0.057251    0.009376    0.675446    
#> 244: 0.041746    0.940461    3.353891    0.383508    0.057325    0.009416    0.675032    
#> 245: 0.041872    0.940412    3.353858    0.384117    0.057451    0.009391    0.675172    
#> 246: 0.042588    0.940388    3.354074    0.384603    0.057395    0.009409    0.675238    
#> 247: 0.043334    0.939848    3.354406    0.384996    0.057510    0.009441    0.674918    
#> 248: 0.043270    0.940077    3.354487    0.384244    0.057349    0.009444    0.674504    
#> 249: 0.043376    0.940430    3.354623    0.385924    0.057402    0.009425    0.674093    
#> 250: 0.043751    0.940490    3.354781    0.385962    0.057536    0.009422    0.673771    
#> 251: 0.044042    0.940888    3.355038    0.386465    0.057681    0.009428    0.673630    
#> 252: 0.043835    0.940872    3.355200    0.386032    0.057868    0.009442    0.673599    
#> 253: 0.044370    0.940691    3.355455    0.386702    0.057922    0.009480    0.673362    
#> 254: 0.044526    0.941139    3.355599    0.386682    0.057983    0.009537    0.673103    
#> 255: 0.044437    0.941612    3.355525    0.387092    0.058040    0.009596    0.672891    
#> 256: 0.044428    0.941724    3.355241    0.388166    0.058175    0.009600    0.672728    
#> 257: 0.044885    0.941674    3.355135    0.390006    0.058222    0.009571    0.672934    
#> 258: 0.044756    0.941494    3.355020    0.389868    0.058281    0.009520    0.673236    
#> 259: 0.045294    0.941494    3.354856    0.391443    0.058326    0.009494    0.673601    
#> 260: 0.045434    0.941806    3.354744    0.391936    0.058323    0.009467    0.673716    
#> 261: 0.044938    0.941788    3.354716    0.390894    0.058675    0.009462    0.674132    
#> 262: 0.044736    0.942050    3.354657    0.389874    0.058887    0.009433    0.674417    
#> 263: 0.044293    0.942373    3.354387    0.390142    0.058907    0.009455    0.674505    
#> 264: 0.043736    0.942391    3.354248    0.391035    0.058923    0.009448    0.674630    
#> 265: 0.043263    0.942585    3.354090    0.390792    0.059095    0.009434    0.674822    
#> 266: 0.042908    0.942916    3.353885    0.390728    0.059160    0.009418    0.674990    
#> 267: 0.041927    0.943193    3.353708    0.390772    0.059148    0.009436    0.675039    
#> 268: 0.041582    0.943023    3.353465    0.391485    0.059144    0.009434    0.674872    
#> 269: 0.041878    0.942805    3.353394    0.392834    0.059059    0.009440    0.674680    
#> 270: 0.042106    0.942909    3.353280    0.393430    0.058946    0.009430    0.674420    
#> 271: 0.041646    0.943193    3.353002    0.394313    0.058855    0.009440    0.674362    
#> 272: 0.041592    0.943116    3.352828    0.394701    0.058717    0.009419    0.674458    
#> 273: 0.040961    0.943469    3.352676    0.394213    0.058736    0.009402    0.674552    
#> 274: 0.040708    0.944004    3.352657    0.393616    0.058574    0.009383    0.674555    
#> 275: 0.040419    0.944464    3.352447    0.393197    0.058529    0.009373    0.674390    
#> 276: 0.040171    0.944847    3.352436    0.393244    0.058429    0.009385    0.674315    
#> 277: 0.040026    0.944996    3.352559    0.394000    0.058375    0.009401    0.674164    
#> 278: 0.039823    0.944870    3.352601    0.394148    0.058365    0.009395    0.674039    
#> 279: 0.039634    0.944642    3.352565    0.393720    0.058444    0.009396    0.673951    
#> 280: 0.039385    0.944698    3.352576    0.392914    0.058371    0.009401    0.674006    
#> 281: 0.039147    0.944797    3.352590    0.391962    0.058298    0.009410    0.674079    
#> 282: 0.038788    0.944945    3.352550    0.391887    0.058284    0.009404    0.674024    
#> 283: 0.038621    0.945147    3.352484    0.392175    0.058194    0.009410    0.674022    
#> 284: 0.038534    0.945119    3.352404    0.392958    0.058183    0.009417    0.673917    
#> 285: 0.038227    0.945280    3.352222    0.393575    0.058172    0.009391    0.673767    
#> 286: 0.037943    0.945591    3.352034    0.393919    0.058170    0.009359    0.673730    
#> 287: 0.037483    0.945889    3.351760    0.395202    0.058165    0.009346    0.673531    
#> 288: 0.037464    0.946045    3.351542    0.395198    0.058277    0.009324    0.673701    
#> 289: 0.037548    0.946093    3.351409    0.395434    0.058259    0.009314    0.673799    
#> 290: 0.037551    0.945979    3.351317    0.394867    0.058220    0.009311    0.674094    
#> 291: 0.037587    0.946079    3.351312    0.394320    0.058275    0.009290    0.674022    
#> 292: 0.037449    0.946230    3.351177    0.394808    0.058262    0.009285    0.673924    
#> 293: 0.037400    0.946432    3.350997    0.395548    0.058296    0.009292    0.673891    
#> 294: 0.037148    0.946616    3.350824    0.395891    0.058204    0.009294    0.673949    
#> 295: 0.037103    0.946407    3.350673    0.396616    0.058192    0.009292    0.673790    
#> 296: 0.037198    0.946243    3.350635    0.396875    0.058123    0.009324    0.673801    
#> 297: 0.037110    0.946309    3.350551    0.397281    0.057974    0.009364    0.673840    
#> 298: 0.037174    0.946275    3.350442    0.398517    0.057897    0.009387    0.673787    
#> 299: 0.036884    0.946296    3.350332    0.398992    0.057822    0.009394    0.673673    
#> 300: 0.037026    0.946141    3.350368    0.399249    0.057911    0.009371    0.673636    
#> 301: 0.037018    0.945958    3.350331    0.398752    0.057904    0.009365    0.673663    
#> 302: 0.037081    0.946108    3.350263    0.398902    0.057958    0.009367    0.673407    
#> 303: 0.036759    0.946507    3.350083    0.398651    0.057972    0.009392    0.673173    
#> 304: 0.036675    0.946822    3.349843    0.398832    0.057881    0.009420    0.672897    
#> 305: 0.036445    0.946856    3.349704    0.399594    0.057775    0.009471    0.672624    
#> 306: 0.036215    0.946886    3.349546    0.399768    0.057820    0.009501    0.672602    
#> 307: 0.035989    0.946928    3.349403    0.399839    0.057874    0.009510    0.672634    
#> 308: 0.035756    0.946784    3.349317    0.400121    0.057947    0.009546    0.672692    
#> 309: 0.035598    0.946647    3.349260    0.400166    0.058047    0.009568    0.672989    
#> 310: 0.035603    0.946723    3.349139    0.400370    0.058095    0.009559    0.673008    
#> 311: 0.035498    0.946770    3.349099    0.400894    0.058174    0.009554    0.672938    
#> 312: 0.035108    0.946693    3.348965    0.400586    0.058194    0.009545    0.673184    
#> 313: 0.034702    0.946588    3.348858    0.399939    0.058200    0.009537    0.673445    
#> 314: 0.034394    0.946860    3.348643    0.399806    0.058247    0.009529    0.673413    
#> 315: 0.034103    0.947070    3.348515    0.399799    0.058256    0.009525    0.673400    
#> 316: 0.034011    0.947214    3.348363    0.400223    0.058281    0.009511    0.673379    
#> 317: 0.034102    0.947179    3.348352    0.400368    0.058279    0.009500    0.673420    
#> 318: 0.034366    0.947144    3.348357    0.400205    0.058319    0.009491    0.673365    
#> 319: 0.034353    0.947050    3.348356    0.400419    0.058365    0.009467    0.673476    
#> 320: 0.034095    0.947097    3.348264    0.400425    0.058391    0.009465    0.673478    
#> 321: 0.033633    0.947271    3.348035    0.399972    0.058375    0.009442    0.673653    
#> 322: 0.033503    0.947356    3.347904    0.399472    0.058415    0.009440    0.673793    
#> 323: 0.033320    0.947411    3.347825    0.399142    0.058487    0.009428    0.673815    
#> 324: 0.033066    0.947533    3.347825    0.398504    0.058554    0.009405    0.673765    
#> 325: 0.032998    0.947710    3.347774    0.398268    0.058520    0.009400    0.673724    
#> 326: 0.032891    0.947551    3.347684    0.398489    0.058517    0.009388    0.673679    
#> 327: 0.032915    0.947603    3.347696    0.398630    0.058580    0.009381    0.673642    
#> 328: 0.032980    0.947612    3.347677    0.398955    0.058557    0.009381    0.673551    
#> 329: 0.033091    0.947644    3.347694    0.399419    0.058634    0.009381    0.673481    
#> 330: 0.032802    0.947638    3.347672    0.399098    0.058701    0.009365    0.673439    
#> 331: 0.032683    0.947811    3.347584    0.399148    0.058727    0.009355    0.673395    
#> 332: 0.032636    0.947754    3.347562    0.399319    0.058754    0.009337    0.673476    
#> 333: 0.032534    0.947792    3.347485    0.399446    0.058752    0.009328    0.673407    
#> 334: 0.032324    0.947796    3.347348    0.399181    0.058720    0.009306    0.673299    
#> 335: 0.031907    0.947913    3.347167    0.398774    0.058738    0.009294    0.673280    
#> 336: 0.031383    0.948188    3.346988    0.398312    0.058762    0.009281    0.673301    
#> 337: 0.030930    0.948568    3.346794    0.398698    0.058778    0.009272    0.673315    
#> 338: 0.030669    0.948741    3.346710    0.398596    0.058774    0.009253    0.673426    
#> 339: 0.030446    0.948950    3.346638    0.398425    0.058742    0.009267    0.673363    
#> 340: 0.030511    0.948692    3.346705    0.398336    0.058801    0.009277    0.673318    
#> 341: 0.030780    0.948573    3.346807    0.398561    0.058773    0.009281    0.673228    
#> 342: 0.030754    0.948453    3.346860    0.398386    0.058854    0.009265    0.673158    
#> 343: 0.030696    0.948531    3.346901    0.398544    0.058919    0.009272    0.673070    
#> 344: 0.030838    0.948505    3.346906    0.398719    0.058881    0.009282    0.673141    
#> 345: 0.031039    0.948485    3.346958    0.398697    0.058804    0.009287    0.673138    
#> 346: 0.031176    0.948541    3.346962    0.398668    0.058682    0.009291    0.673076    
#> 347: 0.031311    0.948469    3.346925    0.398192    0.058525    0.009325    0.673118    
#> 348: 0.031344    0.948602    3.346918    0.398356    0.058456    0.009363    0.673074    
#> 349: 0.031331    0.948588    3.346890    0.398244    0.058329    0.009388    0.672920    
#> 350: 0.031083    0.948524    3.346827    0.398108    0.058184    0.009422    0.672964    
#> 351: 0.030887    0.948603    3.346727    0.398111    0.058149    0.009441    0.672925    
#> 352: 0.030692    0.948721    3.346715    0.398231    0.058140    0.009473    0.672823    
#> 353: 0.030655    0.948868    3.346689    0.398182    0.058065    0.009488    0.672769    
#> 354: 0.030639    0.948908    3.346646    0.398058    0.057982    0.009507    0.672609    
#> 355: 0.030967    0.948830    3.346701    0.397991    0.057875    0.009531    0.672561    
#> 356: 0.031078    0.948744    3.346744    0.397856    0.057839    0.009541    0.672572    
#> 357: 0.031461    0.948785    3.346820    0.398545    0.057799    0.009558    0.672447    
#> 358: 0.031450    0.948787    3.346887    0.398348    0.057717    0.009575    0.672279    
#> 359: 0.031473    0.948842    3.346943    0.397851    0.057735    0.009587    0.672173    
#> 360: 0.031494    0.948839    3.346897    0.397461    0.057679    0.009584    0.672108    
#> 361: 0.031448    0.948846    3.346847    0.397569    0.057578    0.009586    0.672012    
#> 362: 0.031475    0.948866    3.346806    0.397672    0.057512    0.009593    0.671798    
#> 363: 0.031459    0.948890    3.346760    0.397448    0.057466    0.009596    0.671631    
#> 364: 0.031399    0.948954    3.346731    0.397009    0.057423    0.009596    0.671606    
#> 365: 0.031319    0.948942    3.346734    0.396569    0.057416    0.009604    0.671615    
#> 366: 0.031255    0.949000    3.346684    0.395975    0.057410    0.009606    0.671704    
#> 367: 0.031197    0.949059    3.346687    0.395615    0.057449    0.009585    0.671687    
#> 368: 0.031364    0.949116    3.346696    0.395426    0.057347    0.009576    0.671737    
#> 369: 0.031278    0.949178    3.346685    0.395159    0.057351    0.009563    0.671897    
#> 370: 0.031190    0.949291    3.346664    0.395051    0.057390    0.009555    0.671855    
#> 371: 0.031058    0.949345    3.346586    0.394891    0.057398    0.009549    0.671818    
#> 372: 0.030827    0.949410    3.346521    0.394536    0.057391    0.009546    0.671833    
#> 373: 0.030546    0.949652    3.346395    0.394677    0.057414    0.009543    0.671835    
#> 374: 0.030344    0.949667    3.346289    0.394685    0.057386    0.009549    0.671711    
#> 375: 0.030097    0.949704    3.346203    0.394978    0.057375    0.009553    0.671615    
#> 376: 0.029958    0.949882    3.346123    0.394959    0.057404    0.009551    0.671607    
#> 377: 0.029962    0.949980    3.346086    0.395355    0.057426    0.009553    0.671478    
#> 378: 0.029876    0.949972    3.346012    0.395853    0.057444    0.009553    0.671428    
#> 379: 0.029817    0.949881    3.345969    0.395982    0.057435    0.009563    0.671298    
#> 380: 0.029617    0.949858    3.345915    0.395775    0.057453    0.009561    0.671301    
#> 381: 0.029375    0.949926    3.345807    0.396010    0.057495    0.009545    0.671290    
#> 382: 0.029176    0.950021    3.345785    0.395764    0.057446    0.009543    0.671303    
#> 383: 0.029278    0.950036    3.345790    0.395701    0.057442    0.009545    0.671268    
#> 384: 0.029279    0.950005    3.345866    0.395365    0.057418    0.009555    0.671224    
#> 385: 0.029399    0.949968    3.345927    0.395399    0.057357    0.009583    0.671126    
#> 386: 0.029551    0.950103    3.345985    0.395443    0.057307    0.009600    0.671165    
#> 387: 0.029693    0.950157    3.346069    0.395714    0.057225    0.009632    0.671141    
#> 388: 0.029777    0.950152    3.346142    0.395541    0.057165    0.009652    0.671073    
#> 389: 0.029815    0.950018    3.346147    0.395231    0.057092    0.009668    0.671143    
#> 390: 0.029780    0.949973    3.346166    0.395213    0.057041    0.009682    0.671081    
#> 391: 0.030083    0.949916    3.346201    0.395244    0.057001    0.009678    0.671042    
#> 392: 0.029918    0.949928    3.346237    0.394850    0.057005    0.009675    0.671112    
#> 393: 0.029882    0.949860    3.346246    0.394687    0.056966    0.009681    0.671088    
#> 394: 0.029868    0.949913    3.346221    0.394881    0.056983    0.009684    0.670979    
#> 395: 0.029945    0.950030    3.346183    0.394933    0.057000    0.009683    0.670991    
#> 396: 0.029854    0.950136    3.346154    0.394813    0.057008    0.009676    0.670968    
#> 397: 0.029770    0.950139    3.346154    0.394883    0.056970    0.009678    0.670967    
#> 398: 0.029932    0.950116    3.346154    0.395099    0.056939    0.009672    0.670990    
#> 399: 0.030013    0.950158    3.346192    0.395402    0.056917    0.009668    0.670972    
#> 400: 0.030202    0.950220    3.346214    0.395658    0.056936    0.009664    0.670944    
#> 401: 0.030347    0.950233    3.346201    0.395912    0.056982    0.009658    0.670991    
#> 402: 0.030270    0.950264    3.346176    0.395746    0.056952    0.009668    0.671008    
#> 403: 0.030359    0.950281    3.346169    0.395998    0.056938    0.009672    0.670912    
#> 404: 0.030342    0.950269    3.346078    0.396430    0.056888    0.009677    0.670886    
#> 405: 0.030152    0.950233    3.346030    0.396761    0.056835    0.009679    0.670886    
#> 406: 0.029960    0.950297    3.345972    0.397145    0.056749    0.009688    0.670826    
#> 407: 0.029973    0.950362    3.345971    0.397297    0.056660    0.009687    0.670797    
#> 408: 0.029988    0.950416    3.345913    0.397274    0.056614    0.009686    0.670802    
#> 409: 0.029979    0.950370    3.345868    0.397439    0.056588    0.009692    0.670706    
#> 410: 0.029844    0.950362    3.345793    0.397590    0.056553    0.009695    0.670708    
#> 411: 0.029725    0.950166    3.345774    0.397364    0.056564    0.009692    0.670728    
#> 412: 0.029805    0.950129    3.345767    0.397427    0.056563    0.009692    0.670675    
#> 413: 0.029741    0.950115    3.345694    0.397350    0.056573    0.009700    0.670626    
#> 414: 0.029670    0.950131    3.345651    0.397507    0.056527    0.009700    0.670580    
#> 415: 0.029705    0.950222    3.345649    0.397422    0.056492    0.009716    0.670481    
#> 416: 0.029699    0.950285    3.345668    0.397514    0.056468    0.009730    0.670415    
#> 417: 0.029781    0.950212    3.345664    0.397142    0.056450    0.009730    0.670417    
#> 418: 0.029738    0.950245    3.345691    0.397170    0.056479    0.009716    0.670394    
#> 419: 0.029721    0.950325    3.345711    0.396860    0.056494    0.009707    0.670352    
#> 420: 0.029717    0.950333    3.345720    0.396654    0.056515    0.009691    0.670288    
#> 421: 0.029688    0.950323    3.345756    0.396309    0.056519    0.009676    0.670234    
#> 422: 0.029760    0.950258    3.345755    0.396186    0.056509    0.009669    0.670213    
#> 423: 0.029823    0.950171    3.345741    0.396054    0.056511    0.009661    0.670261    
#> 424: 0.029747    0.950097    3.345734    0.395903    0.056519    0.009663    0.670237    
#> 425: 0.029707    0.950098    3.345740    0.395781    0.056495    0.009673    0.670263    
#> 426: 0.029832    0.950145    3.345748    0.395813    0.056483    0.009672    0.670226    
#> 427: 0.029880    0.950085    3.345822    0.395656    0.056467    0.009696    0.670168    
#> 428: 0.029911    0.950052    3.345855    0.395674    0.056430    0.009703    0.670074    
#> 429: 0.030048    0.950011    3.345891    0.395520    0.056423    0.009712    0.670049    
#> 430: 0.030163    0.949987    3.345918    0.395552    0.056368    0.009731    0.670064    
#> 431: 0.030242    0.949941    3.345903    0.395541    0.056315    0.009731    0.669979    
#> 432: 0.030174    0.950043    3.345887    0.395541    0.056297    0.009744    0.669897    
#> 433: 0.030127    0.950144    3.345890    0.395302    0.056257    0.009751    0.669879    
#> 434: 0.030030    0.950097    3.345862    0.395385    0.056202    0.009754    0.669786    
#> 435: 0.030023    0.950127    3.345814    0.395158    0.056218    0.009752    0.669708    
#> 436: 0.030055    0.950066    3.345827    0.394848    0.056213    0.009743    0.669690    
#> 437: 0.030239    0.950004    3.345843    0.395011    0.056174    0.009741    0.669699    
#> 438: 0.030283    0.950125    3.345832    0.395359    0.056205    0.009728    0.669594    
#> 439: 0.030308    0.950224    3.345833    0.395300    0.056205    0.009721    0.669507    
#> 440: 0.030229    0.950311    3.345842    0.395199    0.056204    0.009708    0.669444    
#> 441: 0.030269    0.950265    3.345899    0.395160    0.056210    0.009711    0.669394    
#> 442: 0.030246    0.950274    3.345932    0.395205    0.056210    0.009708    0.669380    
#> 443: 0.030271    0.950318    3.345898    0.395182    0.056236    0.009703    0.669380    
#> 444: 0.030148    0.950377    3.345841    0.395049    0.056287    0.009702    0.669361    
#> 445: 0.029902    0.950494    3.345746    0.395103    0.056333    0.009702    0.669387    
#> 446: 0.029663    0.950569    3.345679    0.395149    0.056339    0.009691    0.669453    
#> 447: 0.029639    0.950624    3.345648    0.395312    0.056341    0.009691    0.669442    
#> 448: 0.029542    0.950628    3.345620    0.395448    0.056338    0.009696    0.669353    
#> 449: 0.029561    0.950692    3.345646    0.395492    0.056378    0.009695    0.669343    
#> 450: 0.029532    0.950712    3.345639    0.395611    0.056395    0.009701    0.669324    
#> 451: 0.029510    0.950815    3.345574    0.395903    0.056449    0.009703    0.669279    
#> 452: 0.029448    0.950884    3.345524    0.396209    0.056467    0.009702    0.669274    
#> 453: 0.029315    0.950935    3.345451    0.395950    0.056489    0.009699    0.669233    
#> 454: 0.029276    0.951055    3.345406    0.396022    0.056499    0.009698    0.669144    
#> 455: 0.029247    0.951107    3.345381    0.395934    0.056467    0.009702    0.669117    
#> 456: 0.029254    0.951136    3.345359    0.396040    0.056432    0.009710    0.669118    
#> 457: 0.029201    0.951158    3.345339    0.396266    0.056436    0.009715    0.669038    
#> 458: 0.029275    0.951255    3.345405    0.396356    0.056468    0.009717    0.669020    
#> 459: 0.029161    0.951337    3.345410    0.396211    0.056479    0.009717    0.669083    
#> 460: 0.029085    0.951367    3.345398    0.396051    0.056471    0.009728    0.669049    
#> 461: 0.029007    0.951388    3.345408    0.395818    0.056494    0.009741    0.669078    
#> 462: 0.028967    0.951409    3.345422    0.395756    0.056469    0.009753    0.669047    
#> 463: 0.029029    0.951464    3.345441    0.395872    0.056455    0.009754    0.669022    
#> 464: 0.029130    0.951473    3.345424    0.396033    0.056439    0.009752    0.668962    
#> 465: 0.029212    0.951347    3.345434    0.396295    0.056429    0.009746    0.668980    
#> 466: 0.029335    0.951366    3.345433    0.396626    0.056398    0.009738    0.668990    
#> 467: 0.029358    0.951271    3.345479    0.396997    0.056402    0.009733    0.668917    
#> 468: 0.029538    0.951255    3.345537    0.397393    0.056371    0.009735    0.668902    
#> 469: 0.029672    0.951351    3.345598    0.397552    0.056334    0.009743    0.668986    
#> 470: 0.029503    0.951385    3.345586    0.397444    0.056334    0.009757    0.669028    
#> 471: 0.029400    0.951411    3.345558    0.397408    0.056281    0.009758    0.669039    
#> 472: 0.029309    0.951450    3.345565    0.397469    0.056268    0.009757    0.669025    
#> 473: 0.029287    0.951461    3.345561    0.397313    0.056220    0.009759    0.669031    
#> 474: 0.029345    0.951472    3.345553    0.397238    0.056208    0.009760    0.669069    
#> 475: 0.029357    0.951570    3.345538    0.397147    0.056220    0.009760    0.669037    
#> 476: 0.029221    0.951624    3.345505    0.397107    0.056251    0.009754    0.669051    
#> 477: 0.029116    0.951651    3.345487    0.397319    0.056257    0.009748    0.668969    
#> 478: 0.029114    0.951694    3.345489    0.397290    0.056236    0.009746    0.668976    
#> 479: 0.029076    0.951785    3.345489    0.397213    0.056242    0.009741    0.668958    
#> 480: 0.029038    0.951811    3.345468    0.397054    0.056257    0.009744    0.668941    
#> 481: 0.028924    0.951834    3.345419    0.396887    0.056255    0.009751    0.668930    
#> 482: 0.029021    0.951850    3.345385    0.397011    0.056228    0.009758    0.669024    
#> 483: 0.028951    0.951807    3.345364    0.396831    0.056241    0.009757    0.669078    
#> 484: 0.028899    0.951688    3.345369    0.396606    0.056208    0.009754    0.669148    
#> 485: 0.028852    0.951648    3.345377    0.396543    0.056171    0.009763    0.669214    
#> 486: 0.028932    0.951614    3.345429    0.396444    0.056181    0.009760    0.669222    
#> 487: 0.029039    0.951606    3.345507    0.396228    0.056220    0.009756    0.669208    
#> 488: 0.029165    0.951590    3.345572    0.396096    0.056225    0.009752    0.669226    
#> 489: 0.029339    0.951506    3.345613    0.395993    0.056258    0.009752    0.669297    
#> 490: 0.029412    0.951493    3.345649    0.395992    0.056310    0.009749    0.669326    
#> 491: 0.029504    0.951547    3.345699    0.396102    0.056362    0.009745    0.669371    
#> 492: 0.029454    0.951555    3.345719    0.396015    0.056384    0.009733    0.669549    
#> 493: 0.029414    0.951516    3.345715    0.395953    0.056445    0.009744    0.669569    
#> 494: 0.029450    0.951563    3.345679    0.395830    0.056493    0.009748    0.669540    
#> 495: 0.029461    0.951611    3.345674    0.395639    0.056517    0.009744    0.669530    
#> 496: 0.029622    0.951646    3.345658    0.395647    0.056528    0.009735    0.669557    
#> 497: 0.029548    0.951620    3.345636    0.395728    0.056545    0.009730    0.669544    
#> 498: 0.029517    0.951632    3.345604    0.395993    0.056548    0.009723    0.669471    
#> 499: 0.029481    0.951713    3.345554    0.396142    0.056568    0.009718    0.669428    
#> 500: 0.029440    0.951659    3.345537    0.396129    0.056565    0.009719    0.669372    
#> [====|====|=>-------------------------------------]  25%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.325347    0.986689    3.439233    0.392401    0.073871    0.016800    1.811729    
#> 002: 0.257754    1.043928    3.466642    0.372781    0.070177    0.015960    1.149826    
#> 003: 0.391262    1.028652    3.466536    0.354142    0.066668    0.015162    0.883152    
#> 004: 0.338914    1.062755    3.473730    0.336435    0.063335    0.014404    0.807295    
#> 005: 0.383072    1.084681    3.452094    0.319613    0.060168    0.013684    0.745935    
#> 006: 0.368807    1.099472    3.449461    0.418967    0.057160    0.013000    0.691863    
#> 007: 0.307700    1.105879    3.443772    0.398019    0.054302    0.013716    0.686892    
#> 008: 0.321449    1.104058    3.444611    0.395141    0.056545    0.013884    0.680521    
#> 009: 0.313693    1.091005    3.445197    0.480469    0.053718    0.019155    0.680037    
#> 010: 0.306636    1.087391    3.448121    0.456446    0.051032    0.019871    0.673072    
#> 011: 0.427350    1.082266    3.464887    0.508151    0.051517    0.020206    0.683172    
#> 012: 0.437214    1.078015    3.475718    0.482744    0.048941    0.019196    0.660338    
#> 013: 0.389903    1.075882    3.468710    0.489445    0.046494    0.022881    0.659077    
#> 014: 0.446684    1.077219    3.482984    0.514021    0.044170    0.025284    0.657801    
#> 015: 0.466802    1.070239    3.495902    0.531160    0.041961    0.024020    0.660695    
#> 016: 0.455814    1.078655    3.488092    0.504602    0.039863    0.022819    0.656100    
#> 017: 0.430271    1.076615    3.484724    0.516652    0.037870    0.023285    0.635273    
#> 018: 0.449473    1.078985    3.481066    0.507197    0.035976    0.022121    0.646438    
#> 019: 0.399194    1.082696    3.477917    0.521189    0.034178    0.022635    0.646429    
#> 020: 0.388407    1.089127    3.477698    0.495130    0.036149    0.021503    0.646807    
#> 021: 0.403269    1.088879    3.481466    0.508934    0.034342    0.020428    0.645554    
#> 022: 0.387374    1.084028    3.489242    0.556219    0.033787    0.019491    0.649375    
#> 023: 0.501391    1.095087    3.503457    0.528408    0.035286    0.018862    0.676128    
#> 024: 0.474431    1.106269    3.496345    0.513078    0.033522    0.021303    0.657492    
#> 025: 0.418539    1.102299    3.479346    0.528064    0.031846    0.023824    0.646593    
#> 026: 0.377925    1.125524    3.454908    0.519052    0.032099    0.022633    0.647095    
#> 027: 0.366950    1.137459    3.445754    0.510262    0.035530    0.021502    0.648906    
#> 028: 0.403601    1.112277    3.454011    0.523579    0.033754    0.020426    0.629321    
#> 029: 0.386192    1.113791    3.447629    0.497400    0.032597    0.025356    0.645391    
#> 030: 0.363366    1.135787    3.443797    0.472530    0.030967    0.026300    0.632335    
#> 031: 0.336208    1.119843    3.453644    0.448903    0.029796    0.024985    0.635263    
#> 032: 0.358318    1.109803    3.464846    0.484197    0.035078    0.024752    0.634931    
#> 033: 0.379136    1.124602    3.449862    0.573035    0.033324    0.023515    0.655112    
#> 034: 0.415719    1.108826    3.453800    0.544383    0.032361    0.022339    0.658833    
#> 035: 0.385242    1.096123    3.455982    0.517164    0.030743    0.021222    0.643830    
#> 036: 0.362393    1.093832    3.448205    0.491306    0.031806    0.020609    0.639743    
#> 037: 0.370690    1.108481    3.461332    0.489392    0.036958    0.025328    0.638720    
#> 038: 0.427219    1.117238    3.455772    0.567717    0.035110    0.024062    0.655680    
#> 039: 0.389599    1.111091    3.470261    0.539331    0.033355    0.026268    0.627593    
#> 040: 0.405065    1.104685    3.480084    0.512364    0.031687    0.024955    0.645342    
#> 041: 0.400542    1.096707    3.469474    0.486746    0.030103    0.023707    0.642736    
#> 042: 0.427352    1.106612    3.464563    0.570063    0.028598    0.026140    0.642602    
#> 043: 0.369179    1.117656    3.454827    0.541560    0.027168    0.026900    0.643376    
#> 044: 0.371052    1.119194    3.460537    0.546420    0.025809    0.029549    0.637432    
#> 045: 0.356248    1.121219    3.453442    0.519099    0.024519    0.031503    0.631305    
#> 046: 0.393333    1.110163    3.457056    0.507698    0.028537    0.029928    0.641111    
#> 047: 0.398874    1.097042    3.473255    0.491143    0.027792    0.028997    0.634597    
#> 048: 0.406055    1.087687    3.467050    0.593454    0.027659    0.031443    0.639096    
#> 049: 0.422483    1.096684    3.467011    0.563782    0.030699    0.029871    0.664428    
#> 050: 0.434478    1.078097    3.481823    0.535593    0.029164    0.029037    0.662857    
#> 051: 0.395914    1.090076    3.469002    0.559719    0.029708    0.027585    0.650523    
#> 052: 0.418191    1.105532    3.479205    0.561591    0.029323    0.027374    0.646126    
#> 053: 0.414090    1.100586    3.470033    0.545093    0.027857    0.031104    0.657526    
#> 054: 0.385579    1.114771    3.463628    0.583637    0.026464    0.031510    0.647360    
#> 055: 0.409014    1.120571    3.468896    0.554455    0.029762    0.029934    0.663450    
#> 056: 0.392127    1.112075    3.466944    0.526732    0.031826    0.028438    0.651160    
#> 057: 0.413125    1.117000    3.465540    0.519909    0.032916    0.027016    0.654666    
#> 058: 0.397751    1.099022    3.465670    0.493914    0.031271    0.025761    0.651036    
#> 059: 0.375298    1.065136    3.468955    0.527519    0.035318    0.024473    0.656787    
#> 060: 0.393506    1.076081    3.471014    0.611045    0.033552    0.032058    0.673593    
#> 061: 0.445185    1.082400    3.472255    0.706407    0.031874    0.031183    0.657622    
#> 062: 0.406724    1.089112    3.476078    0.671087    0.032051    0.033099    0.663199    
#> 063: 0.449083    1.079498    3.474198    0.637533    0.031283    0.031444    0.677995    
#> 064: 0.422681    1.082879    3.475286    0.605656    0.030217    0.030820    0.673352    
#> 065: 0.429338    1.070789    3.468468    0.575373    0.028706    0.031909    0.657415    
#> 066: 0.412870    1.082635    3.475312    0.546605    0.027386    0.030313    0.638212    
#> 067: 0.404161    1.080810    3.481965    0.519274    0.026016    0.028798    0.636934    
#> 068: 0.365809    1.102382    3.468095    0.493311    0.026926    0.027358    0.652658    
#> 069: 0.377161    1.100361    3.473825    0.468645    0.032939    0.025990    0.653915    
#> 070: 0.397393    1.091013    3.478865    0.445213    0.031292    0.024690    0.647470    
#> 071: 0.386591    1.114662    3.455810    0.422952    0.029727    0.024325    0.669568    
#> 072: 0.379325    1.080961    3.471772    0.416735    0.034783    0.030127    0.664020    
#> 073: 0.378016    1.087229    3.488146    0.423099    0.042243    0.028621    0.673861    
#> 074: 0.382952    1.071634    3.479058    0.426527    0.040131    0.027190    0.665749    
#> 075: 0.422272    1.085161    3.504789    0.466402    0.038124    0.026219    0.653162    
#> 076: 0.457121    1.082396    3.512726    0.443082    0.036218    0.024908    0.675451    
#> 077: 0.465224    1.073958    3.495762    0.485147    0.036940    0.023663    0.677086    
#> 078: 0.382939    1.095236    3.482211    0.507198    0.035298    0.022480    0.650623    
#> 079: 0.415913    1.099954    3.474279    0.524407    0.037627    0.021356    0.656243    
#> 080: 0.429784    1.096208    3.472724    0.520322    0.038344    0.020288    0.657363    
#> 081: 0.397655    1.105522    3.469290    0.526742    0.036427    0.022470    0.641252    
#> 082: 0.394806    1.087765    3.476304    0.500405    0.034606    0.024348    0.648701    
#> 083: 0.389416    1.101597    3.474934    0.524498    0.032876    0.025960    0.642240    
#> 084: 0.390362    1.102922    3.470545    0.498273    0.035062    0.025763    0.641843    
#> 085: 0.400232    1.106851    3.483992    0.485646    0.033309    0.024475    0.649141    
#> 086: 0.427505    1.102697    3.479215    0.476884    0.031643    0.027815    0.634232    
#> 087: 0.415202    1.099429    3.471545    0.453040    0.030061    0.026424    0.658367    
#> 088: 0.338767    1.091138    3.470458    0.430388    0.031155    0.025103    0.651035    
#> 089: 0.374412    1.110435    3.465572    0.408869    0.029597    0.025520    0.647558    
#> 090: 0.384057    1.107120    3.466290    0.443532    0.039218    0.024244    0.651067    
#> 091: 0.386281    1.103038    3.458020    0.483923    0.041355    0.023032    0.648945    
#> 092: 0.389715    1.109493    3.455129    0.471542    0.040039    0.021880    0.643511    
#> 093: 0.349501    1.098062    3.449426    0.531164    0.038049    0.022062    0.640550    
#> 094: 0.379119    1.120459    3.453701    0.504606    0.036147    0.020959    0.651354    
#> 095: 0.407404    1.111924    3.471851    0.606355    0.034339    0.020079    0.638701    
#> 096: 0.360332    1.106181    3.458002    0.589755    0.032622    0.022775    0.650663    
#> 097: 0.394578    1.120500    3.461120    0.560267    0.031742    0.024843    0.639686    
#> 098: 0.404852    1.115526    3.463326    0.581611    0.030155    0.023601    0.643764    
#> 099: 0.400811    1.104193    3.453612    0.625695    0.028647    0.025596    0.634704    
#> 100: 0.390741    1.113046    3.449540    0.649184    0.027215    0.024536    0.637362    
#> 101: 0.341883    1.093157    3.428000    0.616725    0.025854    0.023644    0.637357    
#> 102: 0.347916    1.091575    3.441821    0.585889    0.024561    0.026100    0.647355    
#> 103: 0.386899    1.092097    3.456159    0.556595    0.026099    0.025895    0.642594    
#> 104: 0.389800    1.104158    3.473563    0.552261    0.031392    0.024601    0.653591    
#> 105: 0.400979    1.114290    3.463194    0.530208    0.031159    0.023371    0.650200    
#> 106: 0.397880    1.100812    3.458315    0.580370    0.029733    0.026798    0.644928    
#> 107: 0.434574    1.108534    3.471193    0.551351    0.029242    0.025667    0.656251    
#> 108: 0.428922    1.103392    3.470266    0.523784    0.027780    0.028749    0.664857    
#> 109: 0.415034    1.083557    3.488003    0.497595    0.036034    0.030068    0.647265    
#> 110: 0.450197    1.097152    3.473677    0.472715    0.034232    0.029801    0.650643    
#> 111: 0.429497    1.108827    3.477580    0.471014    0.033457    0.028311    0.645983    
#> 112: 0.428735    1.092911    3.474321    0.511363    0.036780    0.026896    0.649893    
#> 113: 0.403173    1.106041    3.469635    0.499158    0.042214    0.025551    0.628307    
#> 114: 0.383197    1.118601    3.460784    0.474200    0.044259    0.024273    0.642762    
#> 115: 0.379093    1.094691    3.450227    0.482303    0.042046    0.027753    0.665743    
#> 116: 0.408698    1.114188    3.452641    0.533252    0.039944    0.028225    0.652601    
#> 117: 0.382421    1.119440    3.462807    0.506589    0.037946    0.026814    0.654897    
#> 118: 0.402560    1.108116    3.470354    0.481260    0.036843    0.025473    0.660696    
#> 119: 0.409874    1.103869    3.468439    0.478649    0.039002    0.027969    0.664685    
#> 120: 0.370962    1.102600    3.463677    0.488828    0.037052    0.027313    0.650838    
#> 121: 0.371479    1.113628    3.465452    0.481425    0.035200    0.025948    0.649194    
#> 122: 0.398546    1.091048    3.464315    0.530489    0.033440    0.024650    0.639072    
#> 123: 0.402440    1.123875    3.460706    0.557161    0.031768    0.024676    0.634940    
#> 124: 0.416563    1.140991    3.456321    0.535313    0.030179    0.025452    0.652603    
#> 125: 0.396981    1.132043    3.453295    0.526951    0.034420    0.024909    0.631595    
#> 126: 0.392223    1.128260    3.453052    0.529710    0.039670    0.026696    0.635126    
#> 127: 0.407669    1.110746    3.459202    0.526186    0.039486    0.026971    0.639881    
#> 128: 0.405327    1.113518    3.464801    0.585090    0.044431    0.028982    0.662848    
#> 129: 0.428073    1.099612    3.479563    0.555835    0.047426    0.027533    0.657494    
#> 130: 0.411849    1.114178    3.472625    0.528044    0.045054    0.026157    0.643966    
#> 131: 0.412168    1.085281    3.460933    0.501641    0.042802    0.024849    0.661672    
#> 132: 0.383028    1.102753    3.446737    0.505117    0.040662    0.023606    0.654814    
#> 133: 0.359181    1.098501    3.452644    0.479862    0.038739    0.022426    0.658976    
#> 134: 0.379627    1.085144    3.479864    0.455869    0.036947    0.022819    0.666093    
#> 135: 0.381920    1.088356    3.475944    0.433075    0.035100    0.021778    0.660485    
#> 136: 0.411803    1.077611    3.478046    0.411421    0.033345    0.023901    0.659193    
#> 137: 0.424298    1.079041    3.488844    0.411590    0.031677    0.023281    0.675804    
#> 138: 0.402750    1.092849    3.475623    0.399769    0.030319    0.022117    0.664561    
#> 139: 0.383372    1.086571    3.486830    0.437051    0.033673    0.022433    0.682620    
#> 140: 0.426921    1.102370    3.483651    0.507013    0.039840    0.021312    0.685682    
#> 141: 0.395960    1.090293    3.479340    0.488166    0.038741    0.021615    0.648743    
#> 142: 0.417333    1.074042    3.483499    0.508339    0.036804    0.023333    0.655831    
#> 143: 0.417393    1.089099    3.487266    0.490683    0.034964    0.024495    0.647198    
#> 144: 0.444079    1.091473    3.483507    0.556815    0.033216    0.023270    0.649714    
#> 145: 0.418585    1.095333    3.473678    0.528975    0.031884    0.025613    0.636919    
#> 146: 0.397777    1.095971    3.480883    0.502526    0.030290    0.024914    0.650088    
#> 147: 0.354029    1.095186    3.462368    0.477400    0.028775    0.023668    0.644699    
#> 148: 0.375104    1.106738    3.473639    0.478404    0.031346    0.022485    0.645441    
#> 149: 0.382343    1.099427    3.459586    0.459294    0.031651    0.024866    0.653327    
#> 150: 0.448722    1.094495    3.457366    0.542249    0.034836    0.024391    0.669780    
#> 151: 0.388579    1.124747    3.458165    0.542637    0.033105    0.024656    0.655182    
#> 152: 0.381941    1.111790    3.466334    0.488822    0.030980    0.026678    0.640220    
#> 153: 0.369530    1.119545    3.474453    0.481612    0.023730    0.032805    0.654474    
#> 154: 0.400692    1.133580    3.469835    0.452115    0.028429    0.031917    0.645871    
#> 155: 0.396768    1.103991    3.471818    0.482934    0.022678    0.036162    0.643054    
#> 156: 0.409805    1.118122    3.465679    0.484598    0.027037    0.042509    0.647134    
#> 157: 0.397953    1.109039    3.469018    0.494655    0.018431    0.039795    0.644802    
#> 158: 0.433511    1.111876    3.474366    0.498262    0.014229    0.032828    0.650594    
#> 159: 0.379754    1.114629    3.478782    0.472832    0.015772    0.035668    0.654386    
#> 160: 0.447552    1.090456    3.488186    0.591101    0.021795    0.033755    0.657367    
#> 161: 0.386185    1.090793    3.482035    0.450727    0.022135    0.036190    0.659636    
#> 162: 0.412907    1.110935    3.471909    0.462887    0.019866    0.039326    0.652270    
#> 163: 0.416997    1.090451    3.469493    0.536128    0.016813    0.035884    0.652832    
#> 164: 0.421053    1.095659    3.469085    0.528996    0.017458    0.036091    0.649478    
#> 165: 0.384332    1.092887    3.468724    0.494053    0.015993    0.031571    0.650484    
#> 166: 0.418336    1.122226    3.471318    0.449736    0.015248    0.032854    0.674056    
#> 167: 0.409756    1.134588    3.480807    0.484064    0.017004    0.033775    0.663876    
#> 168: 0.395421    1.132458    3.473443    0.483333    0.017673    0.032863    0.651209    
#> 169: 0.396355    1.134477    3.468257    0.459730    0.020027    0.029010    0.661106    
#> 170: 0.386751    1.113418    3.469677    0.408869    0.023447    0.030854    0.650584    
#> 171: 0.354617    1.131025    3.449290    0.428970    0.017569    0.030850    0.655119    
#> 172: 0.341697    1.144967    3.448466    0.429257    0.017030    0.032032    0.653460    
#> 173: 0.371195    1.139966    3.450020    0.488898    0.020673    0.032328    0.644083    
#> 174: 0.390295    1.141703    3.462621    0.498472    0.024541    0.034469    0.655618    
#> 175: 0.387649    1.144027    3.450157    0.525229    0.027039    0.036073    0.643723    
#> 176: 0.346983    1.139648    3.455218    0.512565    0.023702    0.031942    0.632855    
#> 177: 0.402611    1.124269    3.461196    0.594600    0.025997    0.037069    0.624990    
#> 178: 0.417336    1.111330    3.478647    0.569687    0.035560    0.032841    0.633342    
#> 179: 0.493005    1.110658    3.477046    0.588662    0.032333    0.025918    0.644518    
#> 180: 0.426966    1.112842    3.474375    0.488436    0.032465    0.023949    0.650570    
#> 181: 0.423980    1.095114    3.460280    0.499709    0.031167    0.026195    0.665258    
#> 182: 0.370665    1.102622    3.450571    0.431425    0.036708    0.021815    0.663407    
#> 183: 0.354015    1.113396    3.452089    0.382149    0.030422    0.021628    0.660176    
#> 184: 0.359109    1.105806    3.452374    0.458620    0.029540    0.021888    0.657246    
#> 185: 0.363357    1.126841    3.455091    0.419192    0.032365    0.024002    0.655868    
#> 186: 0.343176    1.151928    3.451134    0.441092    0.039490    0.030568    0.655717    
#> 187: 0.382704    1.119687    3.457850    0.454081    0.031832    0.029353    0.672626    
#> 188: 0.339349    1.131275    3.452438    0.521255    0.033897    0.024135    0.676843    
#> 189: 0.385390    1.143810    3.464815    0.536668    0.031273    0.024585    0.663629    
#> 190: 0.365244    1.118902    3.458028    0.458910    0.030118    0.026253    0.651427    
#> 191: 0.344599    1.122746    3.453421    0.504046    0.028086    0.027865    0.641918    
#> 192: 0.350769    1.127999    3.439199    0.487486    0.026078    0.027480    0.645698    
#> 193: 0.417032    1.127412    3.467612    0.490876    0.026395    0.029396    0.648938    
#> 194: 0.413911    1.116799    3.462086    0.467565    0.020085    0.023883    0.653683    
#> 195: 0.400099    1.104000    3.467419    0.499364    0.019642    0.026341    0.656573    
#> 196: 0.415031    1.105358    3.472961    0.559408    0.025637    0.033009    0.643746    
#> 197: 0.367502    1.099099    3.482477    0.495335    0.020465    0.035962    0.658452    
#> 198: 0.475882    1.087094    3.482280    0.553802    0.020471    0.031316    0.644457    
#> 199: 0.411676    1.074172    3.472328    0.464882    0.024501    0.029157    0.644646    
#> 200: 0.412362    1.083695    3.486431    0.443715    0.020679    0.030985    0.663613    
#> 201: 0.397111    1.077269    3.490270    0.409776    0.020725    0.031022    0.673386    
#> 202: 0.390567    1.077296    3.488988    0.412907    0.020761    0.031871    0.673644    
#> 203: 0.405450    1.075158    3.485674    0.434450    0.021205    0.033186    0.672232    
#> 204: 0.412096    1.069499    3.487990    0.453036    0.021479    0.032743    0.667187    
#> 205: 0.413746    1.071656    3.487571    0.453842    0.021542    0.033902    0.666893    
#> 206: 0.410669    1.077225    3.484796    0.468217    0.022083    0.033639    0.666024    
#> 207: 0.405857    1.079980    3.480915    0.458794    0.021772    0.033448    0.664460    
#> 208: 0.402307    1.082758    3.479407    0.455456    0.021948    0.033769    0.662844    
#> 209: 0.402689    1.083349    3.478602    0.458865    0.022072    0.033675    0.660063    
#> 210: 0.404994    1.083254    3.479417    0.462555    0.021920    0.033237    0.658461    
#> 211: 0.406521    1.084344    3.479606    0.466008    0.022249    0.032537    0.657188    
#> 212: 0.406947    1.085176    3.479587    0.467113    0.022300    0.032102    0.656067    
#> 213: 0.408652    1.083053    3.479950    0.466858    0.022726    0.031390    0.656182    
#> 214: 0.411696    1.082059    3.480706    0.469432    0.023142    0.031067    0.656659    
#> 215: 0.411909    1.082394    3.480931    0.473044    0.023360    0.031237    0.656144    
#> 216: 0.416621    1.080911    3.481841    0.483955    0.023421    0.031376    0.656972    
#> 217: 0.420017    1.080836    3.482456    0.493115    0.023159    0.031471    0.656317    
#> 218: 0.421566    1.080980    3.482347    0.502243    0.022821    0.031623    0.655742    
#> 219: 0.422619    1.080804    3.481274    0.508202    0.022606    0.031895    0.655261    
#> 220: 0.420506    1.081500    3.480810    0.507231    0.022361    0.032095    0.655244    
#> 221: 0.418369    1.082205    3.480465    0.503959    0.022143    0.032241    0.655257    
#> 222: 0.417215    1.083130    3.479579    0.505820    0.022143    0.032169    0.654904    
#> 223: 0.417733    1.084043    3.479225    0.506996    0.022201    0.032056    0.654539    
#> 224: 0.419440    1.084967    3.479230    0.508959    0.022313    0.032152    0.654084    
#> 225: 0.419879    1.084555    3.479698    0.512833    0.022591    0.032350    0.653987    
#> 226: 0.418622    1.084273    3.479243    0.516283    0.022728    0.032516    0.653471    
#> 227: 0.417281    1.084589    3.478644    0.516605    0.022816    0.032637    0.652942    
#> 228: 0.415545    1.085681    3.477990    0.515367    0.022747    0.032714    0.653141    
#> 229: 0.415429    1.086156    3.477618    0.516302    0.022630    0.032865    0.652628    
#> 230: 0.415178    1.086446    3.477113    0.517001    0.022724    0.032982    0.652489    
#> 231: 0.414098    1.086880    3.477021    0.514614    0.022820    0.033089    0.651857    
#> 232: 0.413883    1.086850    3.477006    0.515208    0.022957    0.033006    0.651101    
#> 233: 0.413273    1.086243    3.477127    0.518328    0.022911    0.032884    0.650959    
#> 234: 0.412622    1.085879    3.477216    0.516637    0.022724    0.032881    0.651576    
#> 235: 0.413582    1.085566    3.477312    0.518216    0.022780    0.032942    0.651138    
#> 236: 0.413194    1.085720    3.477055    0.516748    0.022893    0.032943    0.651093    
#> 237: 0.413627    1.085680    3.476979    0.516954    0.022919    0.033031    0.651133    
#> 238: 0.414108    1.085992    3.477004    0.516916    0.022899    0.032977    0.651497    
#> 239: 0.414395    1.086618    3.476694    0.517191    0.022954    0.032830    0.651623    
#> 240: 0.413975    1.086435    3.476916    0.515327    0.023015    0.032679    0.651653    
#> 241: 0.414808    1.085953    3.476961    0.515858    0.023232    0.032530    0.651798    
#> 242: 0.414650    1.085348    3.477003    0.514503    0.023483    0.032324    0.652172    
#> 243: 0.415018    1.084907    3.477343    0.512726    0.023774    0.032164    0.652453    
#> 244: 0.415383    1.083997    3.477978    0.508989    0.023856    0.032034    0.653011    
#> 245: 0.416197    1.083230    3.478647    0.507244    0.023927    0.031996    0.653657    
#> 246: 0.417315    1.082739    3.479162    0.506134    0.023930    0.032055    0.654185    
#> 247: 0.418083    1.082530    3.479341    0.506534    0.024006    0.032002    0.654033    
#> 248: 0.418503    1.082560    3.479501    0.506644    0.023986    0.031884    0.653768    
#> 249: 0.418320    1.082267    3.479525    0.506598    0.023988    0.031775    0.653567    
#> 250: 0.418950    1.082513    3.479634    0.507863    0.024010    0.031717    0.653593    
#> 251: 0.419468    1.082806    3.480024    0.509082    0.024094    0.031672    0.653516    
#> 252: 0.420183    1.083027    3.480404    0.510905    0.024133    0.031677    0.653302    
#> 253: 0.420694    1.083038    3.480801    0.510696    0.024096    0.031616    0.653474    
#> 254: 0.421261    1.083477    3.481061    0.509601    0.024071    0.031498    0.653918    
#> 255: 0.421037    1.083671    3.481115    0.508743    0.024008    0.031439    0.653884    
#> 256: 0.421679    1.083566    3.481055    0.510866    0.023925    0.031488    0.653796    
#> 257: 0.421163    1.083316    3.480864    0.511521    0.023781    0.031417    0.653729    
#> 258: 0.420790    1.082992    3.480685    0.512817    0.023754    0.031433    0.653844    
#> 259: 0.420317    1.083018    3.480522    0.513104    0.023776    0.031453    0.654045    
#> 260: 0.420206    1.083165    3.480559    0.512834    0.023798    0.031520    0.654125    
#> 261: 0.420366    1.083054    3.480599    0.512544    0.023824    0.031663    0.654383    
#> 262: 0.420288    1.083228    3.480339    0.513744    0.023853    0.031697    0.654397    
#> 263: 0.420079    1.083351    3.480201    0.514773    0.023811    0.031679    0.654329    
#> 264: 0.419808    1.083519    3.479977    0.516024    0.023798    0.031672    0.654503    
#> 265: 0.419522    1.083631    3.479883    0.516275    0.023847    0.031638    0.654819    
#> 266: 0.419337    1.084034    3.479705    0.517691    0.023838    0.031675    0.654630    
#> 267: 0.419380    1.084242    3.479755    0.519399    0.023788    0.031617    0.654741    
#> 268: 0.419213    1.084356    3.479606    0.519842    0.023760    0.031592    0.654836    
#> 269: 0.419228    1.084377    3.479463    0.519530    0.023796    0.031563    0.654654    
#> 270: 0.419638    1.084628    3.479384    0.519559    0.023842    0.031499    0.654280    
#> 271: 0.420075    1.084827    3.479271    0.519523    0.023773    0.031423    0.654101    
#> 272: 0.420195    1.084837    3.479103    0.518858    0.023733    0.031459    0.654057    
#> 273: 0.419811    1.085140    3.479031    0.518076    0.023739    0.031463    0.653974    
#> 274: 0.419833    1.085234    3.478984    0.517968    0.023758    0.031455    0.653705    
#> 275: 0.419531    1.085406    3.478706    0.517769    0.023787    0.031412    0.653495    
#> 276: 0.419205    1.085567    3.478564    0.517768    0.023759    0.031350    0.653595    
#> 277: 0.418972    1.085779    3.478685    0.516689    0.023724    0.031314    0.653908    
#> 278: 0.418667    1.085945    3.478674    0.516387    0.023747    0.031248    0.653828    
#> 279: 0.419044    1.085902    3.478807    0.516431    0.023758    0.031270    0.653941    
#> 280: 0.418806    1.086050    3.478856    0.516280    0.023839    0.031294    0.653846    
#> 281: 0.418185    1.086241    3.478804    0.516679    0.023871    0.031286    0.653732    
#> 282: 0.418253    1.086430    3.478756    0.518310    0.023923    0.031242    0.653663    
#> 283: 0.418199    1.086562    3.478758    0.519721    0.023963    0.031203    0.653985    
#> 284: 0.417873    1.086595    3.478664    0.520170    0.024048    0.031131    0.653959    
#> 285: 0.417674    1.086616    3.478484    0.520850    0.024170    0.031124    0.654068    
#> 286: 0.417101    1.086834    3.478222    0.522070    0.024324    0.031110    0.654246    
#> 287: 0.416610    1.087216    3.477845    0.523041    0.024415    0.031113    0.654341    
#> 288: 0.416512    1.087303    3.477705    0.522344    0.024446    0.031086    0.654764    
#> 289: 0.416037    1.087519    3.477416    0.521955    0.024467    0.031091    0.654736    
#> 290: 0.416179    1.087747    3.477249    0.522974    0.024480    0.031117    0.654936    
#> 291: 0.416078    1.088017    3.477111    0.523496    0.024484    0.031166    0.655060    
#> 292: 0.415389    1.088309    3.476792    0.522813    0.024488    0.031190    0.655119    
#> 293: 0.415132    1.088531    3.476595    0.521930    0.024457    0.031218    0.655267    
#> 294: 0.414336    1.088824    3.476260    0.522539    0.024458    0.031249    0.655359    
#> 295: 0.413843    1.088705    3.476046    0.522775    0.024444    0.031354    0.655425    
#> 296: 0.413380    1.088784    3.475866    0.522331    0.024436    0.031464    0.655610    
#> 297: 0.413137    1.088989    3.475790    0.522851    0.024486    0.031524    0.655718    
#> 298: 0.413148    1.089000    3.475684    0.522720    0.024486    0.031475    0.655713    
#> 299: 0.413198    1.089096    3.475544    0.522264    0.024495    0.031397    0.655623    
#> 300: 0.413506    1.089141    3.475586    0.522858    0.024468    0.031389    0.655520    
#> 301: 0.413920    1.088974    3.475655    0.522410    0.024461    0.031368    0.655662    
#> 302: 0.414129    1.088926    3.475732    0.521702    0.024434    0.031383    0.655494    
#> 303: 0.414222    1.089037    3.475765    0.521736    0.024439    0.031380    0.655342    
#> 304: 0.414399    1.089244    3.475718    0.522213    0.024419    0.031381    0.655371    
#> 305: 0.414436    1.089205    3.475641    0.522086    0.024438    0.031358    0.655305    
#> 306: 0.414290    1.089161    3.475554    0.521474    0.024459    0.031287    0.655276    
#> 307: 0.413558    1.089070    3.475259    0.521200    0.024515    0.031220    0.655323    
#> 308: 0.412845    1.089102    3.475064    0.521376    0.024581    0.031169    0.655286    
#> 309: 0.412359    1.089046    3.474996    0.521461    0.024579    0.031106    0.655516    
#> 310: 0.412047    1.089250    3.474732    0.521781    0.024651    0.031116    0.655407    
#> 311: 0.412049    1.089342    3.474626    0.522287    0.024661    0.031102    0.655271    
#> 312: 0.411592    1.089321    3.474472    0.523132    0.024707    0.031100    0.655256    
#> 313: 0.411189    1.089206    3.474269    0.523598    0.024750    0.031074    0.655341    
#> 314: 0.410601    1.089309    3.474146    0.523474    0.024729    0.031067    0.655444    
#> 315: 0.410203    1.089491    3.474029    0.523192    0.024710    0.031009    0.655575    
#> 316: 0.409665    1.089808    3.473819    0.522733    0.024711    0.031000    0.655642    
#> 317: 0.409377    1.090007    3.473789    0.521905    0.024752    0.031026    0.655863    
#> 318: 0.409190    1.090208    3.473670    0.521602    0.024726    0.031008    0.655848    
#> 319: 0.408955    1.090349    3.473597    0.521383    0.024730    0.031017    0.655827    
#> 320: 0.408615    1.090500    3.473468    0.521733    0.024692    0.031000    0.655927    
#> 321: 0.408421    1.090711    3.473258    0.522632    0.024685    0.031011    0.655932    
#> 322: 0.408135    1.090883    3.473083    0.522656    0.024667    0.031014    0.655895    
#> 323: 0.407714    1.090945    3.472925    0.522363    0.024686    0.031067    0.655849    
#> 324: 0.407274    1.091153    3.472721    0.522534    0.024745    0.031065    0.655753    
#> 325: 0.406804    1.091508    3.472493    0.522457    0.024768    0.031076    0.655681    
#> 326: 0.406393    1.091638    3.472322    0.522058    0.024763    0.031083    0.655648    
#> 327: 0.406284    1.091777    3.472298    0.522239    0.024796    0.031094    0.655631    
#> 328: 0.406275    1.091825    3.472303    0.521987    0.024783    0.031077    0.655682    
#> 329: 0.406411    1.091870    3.472311    0.521991    0.024768    0.031064    0.655764    
#> 330: 0.405937    1.091880    3.472264    0.521664    0.024734    0.031018    0.655998    
#> 331: 0.405774    1.092071    3.472229    0.521471    0.024789    0.030985    0.655988    
#> 332: 0.405671    1.092127    3.472297    0.521045    0.024812    0.030947    0.656119    
#> 333: 0.405425    1.092138    3.472202    0.520756    0.024875    0.030908    0.656174    
#> 334: 0.405323    1.092049    3.472131    0.520297    0.024927    0.030892    0.656040    
#> 335: 0.405186    1.092116    3.471986    0.520359    0.024970    0.030888    0.655913    
#> 336: 0.404772    1.092321    3.471880    0.520101    0.025023    0.030883    0.655793    
#> 337: 0.404514    1.092702    3.471649    0.520386    0.025089    0.030879    0.655730    
#> 338: 0.404377    1.092949    3.471499    0.520718    0.025146    0.030887    0.655649    
#> 339: 0.403940    1.093125    3.471291    0.520508    0.025166    0.030923    0.655650    
#> 340: 0.403747    1.093170    3.471152    0.520805    0.025169    0.030954    0.655579    
#> 341: 0.403860    1.093152    3.471128    0.521332    0.025187    0.030999    0.655501    
#> 342: 0.403901    1.092975    3.471155    0.521491    0.025151    0.031055    0.655514    
#> 343: 0.403684    1.092920    3.471185    0.521344    0.025136    0.031076    0.655546    
#> 344: 0.403877    1.092830    3.471252    0.521748    0.025096    0.031114    0.655685    
#> 345: 0.403945    1.092855    3.471318    0.521616    0.025038    0.031183    0.655722    
#> 346: 0.403638    1.093023    3.471190    0.521639    0.025010    0.031185    0.655761    
#> 347: 0.403482    1.092989    3.471126    0.521632    0.025039    0.031162    0.655800    
#> 348: 0.403440    1.093089    3.471010    0.521700    0.025029    0.031179    0.655913    
#> 349: 0.403066    1.093124    3.470913    0.522099    0.025015    0.031208    0.655875    
#> 350: 0.402628    1.093144    3.470762    0.522174    0.025012    0.031223    0.655790    
#> 351: 0.402475    1.093361    3.470626    0.522411    0.025000    0.031219    0.655755    
#> 352: 0.402231    1.093507    3.470565    0.522382    0.024964    0.031238    0.655976    
#> 353: 0.402449    1.093598    3.470520    0.522588    0.024950    0.031228    0.656007    
#> 354: 0.402444    1.093712    3.470486    0.522583    0.024972    0.031192    0.655964    
#> 355: 0.402736    1.093734    3.470447    0.522953    0.024973    0.031173    0.655873    
#> 356: 0.402699    1.093715    3.470495    0.522753    0.024967    0.031157    0.655929    
#> 357: 0.402736    1.093809    3.470543    0.522868    0.024949    0.031102    0.655947    
#> 358: 0.402596    1.093908    3.470579    0.522836    0.024956    0.031062    0.655917    
#> 359: 0.402296    1.094090    3.470461    0.522465    0.024913    0.031021    0.655995    
#> 360: 0.402112    1.094233    3.470317    0.522628    0.024883    0.030992    0.655936    
#> 361: 0.401974    1.094404    3.470232    0.522314    0.024880    0.030973    0.655815    
#> 362: 0.401974    1.094406    3.470183    0.522212    0.024896    0.030923    0.655639    
#> 363: 0.401983    1.094394    3.470211    0.522258    0.024909    0.030866    0.655536    
#> 364: 0.402299    1.094434    3.470214    0.522677    0.024912    0.030839    0.655542    
#> 365: 0.402058    1.094412    3.470140    0.522404    0.024932    0.030835    0.655624    
#> 366: 0.402134    1.094420    3.470116    0.522052    0.024933    0.030809    0.655737    
#> 367: 0.402239    1.094481    3.470070    0.522145    0.024930    0.030773    0.655750    
#> 368: 0.402465    1.094587    3.470028    0.522602    0.024922    0.030749    0.655664    
#> 369: 0.402605    1.094625    3.469952    0.523331    0.024922    0.030737    0.655603    
#> 370: 0.402570    1.094745    3.469863    0.523513    0.024926    0.030760    0.655587    
#> 371: 0.402360    1.094854    3.469786    0.523324    0.024917    0.030775    0.655501    
#> 372: 0.402066    1.094933    3.469670    0.523539    0.024888    0.030798    0.655457    
#> 373: 0.401628    1.095096    3.469478    0.523686    0.024857    0.030798    0.655403    
#> 374: 0.401731    1.095111    3.469440    0.523959    0.024814    0.030807    0.655281    
#> 375: 0.401696    1.095103    3.469436    0.524173    0.024812    0.030800    0.655216    
#> 376: 0.401783    1.095150    3.469411    0.524500    0.024799    0.030804    0.655050    
#> 377: 0.401726    1.095239    3.469369    0.524613    0.024792    0.030785    0.654986    
#> 378: 0.401663    1.095246    3.469331    0.524449    0.024770    0.030768    0.655040    
#> 379: 0.401580    1.095125    3.469286    0.524555    0.024776    0.030756    0.655065    
#> 380: 0.401508    1.095176    3.469195    0.524634    0.024792    0.030734    0.655114    
#> 381: 0.401549    1.095214    3.469144    0.525325    0.024770    0.030729    0.655080    
#> 382: 0.401312    1.095266    3.469036    0.525652    0.024787    0.030718    0.654967    
#> 383: 0.401114    1.095291    3.468928    0.525701    0.024780    0.030701    0.654877    
#> 384: 0.400838    1.095431    3.468835    0.525576    0.024760    0.030694    0.654825    
#> 385: 0.400759    1.095485    3.468800    0.525582    0.024784    0.030675    0.654777    
#> 386: 0.400858    1.095639    3.468766    0.525815    0.024842    0.030652    0.654742    
#> 387: 0.400930    1.095726    3.468842    0.525587    0.024901    0.030639    0.654725    
#> 388: 0.400987    1.095755    3.468897    0.525262    0.024900    0.030622    0.654699    
#> 389: 0.400959    1.095665    3.468895    0.525379    0.024897    0.030603    0.654737    
#> 390: 0.400946    1.095716    3.468896    0.525334    0.024871    0.030598    0.654694    
#> 391: 0.401079    1.095699    3.468956    0.525218    0.024856    0.030589    0.654647    
#> 392: 0.401134    1.095681    3.469033    0.525256    0.024838    0.030576    0.654701    
#> 393: 0.401153    1.095608    3.469036    0.525506    0.024861    0.030565    0.654753    
#> 394: 0.401256    1.095641    3.468981    0.525423    0.024856    0.030567    0.654837    
#> 395: 0.401206    1.095797    3.468953    0.525184    0.024850    0.030593    0.654898    
#> 396: 0.401183    1.095862    3.468951    0.524966    0.024869    0.030588    0.654908    
#> 397: 0.401207    1.095789    3.468967    0.524965    0.024868    0.030595    0.655025    
#> 398: 0.401302    1.095705    3.469028    0.524655    0.024889    0.030609    0.655027    
#> 399: 0.401411    1.095757    3.469038    0.524681    0.024918    0.030612    0.655059    
#> 400: 0.401465    1.095877    3.469016    0.524900    0.024928    0.030600    0.655027    
#> 401: 0.401406    1.096049    3.468957    0.524980    0.024951    0.030605    0.655039    
#> 402: 0.401502    1.096069    3.468983    0.524979    0.024941    0.030599    0.655089    
#> 403: 0.401940    1.096092    3.469030    0.525021    0.024934    0.030562    0.655129    
#> 404: 0.402077    1.096160    3.469033    0.524975    0.024937    0.030536    0.655052    
#> 405: 0.402098    1.096195    3.469042    0.525319    0.024954    0.030498    0.654968    
#> 406: 0.402257    1.096220    3.469052    0.525248    0.024960    0.030482    0.654968    
#> 407: 0.402317    1.096273    3.469094    0.525283    0.025004    0.030477    0.654937    
#> 408: 0.402604    1.096401    3.469121    0.525395    0.025078    0.030452    0.654881    
#> 409: 0.402535    1.096447    3.469055    0.525313    0.025097    0.030413    0.654879    
#> 410: 0.402299    1.096507    3.468950    0.525350    0.025127    0.030406    0.654842    
#> 411: 0.402288    1.096399    3.468930    0.525507    0.025165    0.030387    0.654884    
#> 412: 0.402212    1.096414    3.468936    0.525232    0.025195    0.030354    0.654876    
#> 413: 0.402153    1.096428    3.468967    0.524621    0.025191    0.030339    0.654880    
#> 414: 0.401964    1.096394    3.468986    0.524567    0.025197    0.030338    0.654904    
#> 415: 0.402186    1.096388    3.469019    0.524644    0.025186    0.030325    0.654891    
#> 416: 0.402345    1.096430    3.469098    0.524299    0.025190    0.030298    0.654911    
#> 417: 0.402624    1.096399    3.469177    0.524691    0.025203    0.030291    0.654883    
#> 418: 0.402698    1.096376    3.469247    0.524697    0.025221    0.030291    0.654833    
#> 419: 0.402840    1.096418    3.469288    0.524789    0.025234    0.030311    0.654749    
#> 420: 0.403063    1.096399    3.469368    0.524754    0.025233    0.030292    0.654638    
#> 421: 0.403141    1.096379    3.469440    0.524660    0.025221    0.030271    0.654581    
#> 422: 0.403399    1.096326    3.469518    0.524559    0.025224    0.030253    0.654517    
#> 423: 0.403569    1.096237    3.469602    0.524474    0.025215    0.030260    0.654425    
#> 424: 0.403621    1.096177    3.469570    0.524666    0.025210    0.030241    0.654370    
#> 425: 0.403649    1.096187    3.469524    0.525024    0.025217    0.030244    0.654375    
#> 426: 0.403803    1.096234    3.469492    0.525223    0.025226    0.030243    0.654346    
#> 427: 0.403706    1.096180    3.469496    0.525274    0.025228    0.030240    0.654246    
#> 428: 0.403571    1.096199    3.469509    0.525097    0.025222    0.030256    0.654135    
#> 429: 0.403537    1.096237    3.469527    0.525037    0.025211    0.030272    0.654084    
#> 430: 0.403524    1.096295    3.469472    0.524939    0.025208    0.030272    0.654104    
#> 431: 0.403404    1.096314    3.469399    0.524908    0.025209    0.030283    0.654048    
#> 432: 0.403201    1.096400    3.469316    0.524977    0.025199    0.030302    0.654059    
#> 433: 0.403189    1.096515    3.469297    0.525089    0.025190    0.030291    0.654084    
#> 434: 0.403083    1.096516    3.469254    0.524875    0.025185    0.030292    0.654045    
#> 435: 0.403108    1.096559    3.469234    0.524633    0.025173    0.030274    0.654023    
#> 436: 0.403160    1.096541    3.469281    0.524577    0.025184    0.030261    0.654004    
#> 437: 0.403116    1.096525    3.469289    0.524408    0.025195    0.030238    0.654019    
#> 438: 0.402954    1.096675    3.469283    0.524220    0.025214    0.030217    0.654031    
#> 439: 0.402893    1.096775    3.469257    0.524174    0.025226    0.030200    0.653928    
#> 440: 0.402848    1.096822    3.469239    0.523724    0.025228    0.030192    0.653937    
#> 441: 0.402929    1.096804    3.469240    0.523367    0.025222    0.030171    0.653878    
#> 442: 0.402875    1.096778    3.469276    0.523426    0.025212    0.030176    0.653778    
#> 443: 0.402892    1.096815    3.469262    0.523298    0.025228    0.030158    0.653683    
#> 444: 0.402798    1.096871    3.469183    0.523087    0.025243    0.030137    0.653666    
#> 445: 0.402751    1.096940    3.469168    0.522951    0.025261    0.030148    0.653669    
#> 446: 0.402617    1.096985    3.469155    0.523040    0.025277    0.030151    0.653642    
#> 447: 0.402771    1.096995    3.469177    0.523294    0.025296    0.030152    0.653653    
#> 448: 0.402818    1.097029    3.469201    0.523162    0.025289    0.030131    0.653668    
#> 449: 0.402837    1.097106    3.469286    0.522860    0.025307    0.030123    0.653677    
#> 450: 0.402859    1.097107    3.469337    0.522601    0.025317    0.030107    0.653650    
#> 451: 0.402847    1.097180    3.469346    0.522301    0.025324    0.030134    0.653622    
#> 452: 0.402786    1.097229    3.469337    0.522075    0.025332    0.030129    0.653567    
#> 453: 0.402745    1.097276    3.469305    0.522037    0.025327    0.030129    0.653523    
#> 454: 0.402577    1.097318    3.469259    0.521928    0.025341    0.030123    0.653469    
#> 455: 0.402562    1.097274    3.469220    0.522107    0.025317    0.030122    0.653421    
#> 456: 0.402517    1.097244    3.469205    0.522118    0.025320    0.030101    0.653400    
#> 457: 0.402553    1.097247    3.469234    0.522093    0.025312    0.030078    0.653373    
#> 458: 0.402579    1.097323    3.469243    0.522238    0.025306    0.030064    0.653377    
#> 459: 0.402328    1.097445    3.469191    0.522353    0.025301    0.030076    0.653374    
#> 460: 0.402100    1.097475    3.469099    0.522137    0.025291    0.030084    0.653361    
#> 461: 0.402046    1.097503    3.469093    0.521854    0.025282    0.030079    0.653348    
#> 462: 0.401907    1.097536    3.469095    0.521438    0.025283    0.030075    0.653322    
#> 463: 0.401813    1.097584    3.469082    0.521122    0.025269    0.030091    0.653274    
#> 464: 0.401737    1.097630    3.469031    0.520962    0.025265    0.030089    0.653236    
#> 465: 0.401694    1.097562    3.469026    0.520838    0.025262    0.030083    0.653181    
#> 466: 0.401704    1.097565    3.468995    0.520605    0.025247    0.030085    0.653122    
#> 467: 0.401512    1.097594    3.469004    0.520298    0.025263    0.030063    0.653084    
#> 468: 0.401585    1.097603    3.469048    0.520378    0.025276    0.030050    0.653035    
#> 469: 0.401762    1.097679    3.469100    0.520520    0.025304    0.030029    0.653019    
#> 470: 0.401876    1.097707    3.469129    0.520755    0.025335    0.030011    0.653046    
#> 471: 0.401983    1.097686    3.469160    0.520846    0.025348    0.029994    0.653090    
#> 472: 0.402204    1.097619    3.469222    0.520941    0.025370    0.029960    0.653142    
#> 473: 0.402239    1.097636    3.469217    0.520570    0.025381    0.029948    0.653122    
#> 474: 0.402264    1.097634    3.469227    0.520504    0.025369    0.029953    0.653157    
#> 475: 0.402353    1.097663    3.469244    0.520622    0.025361    0.029958    0.653168    
#> 476: 0.402517    1.097624    3.469276    0.521232    0.025348    0.029955    0.653209    
#> 477: 0.402577    1.097603    3.469325    0.521245    0.025337    0.029948    0.653169    
#> 478: 0.402504    1.097565    3.469366    0.521117    0.025354    0.029947    0.653160    
#> 479: 0.402544    1.097589    3.469400    0.520661    0.025388    0.029943    0.653153    
#> 480: 0.402514    1.097625    3.469432    0.520303    0.025420    0.029932    0.653177    
#> 481: 0.402425    1.097670    3.469430    0.519928    0.025419    0.029934    0.653160    
#> 482: 0.402491    1.097670    3.469419    0.519957    0.025428    0.029933    0.653163    
#> 483: 0.402521    1.097665    3.469449    0.520030    0.025434    0.029929    0.653156    
#> 484: 0.402492    1.097575    3.469471    0.519832    0.025430    0.029930    0.653176    
#> 485: 0.402525    1.097609    3.469502    0.519666    0.025427    0.029940    0.653206    
#> 486: 0.402601    1.097696    3.469578    0.519343    0.025435    0.029933    0.653234    
#> 487: 0.402889    1.097783    3.469690    0.519370    0.025442    0.029926    0.653233    
#> 488: 0.402997    1.097730    3.469761    0.519404    0.025455    0.029930    0.653208    
#> 489: 0.403208    1.097683    3.469797    0.519600    0.025455    0.029951    0.653228    
#> 490: 0.403300    1.097752    3.469810    0.520163    0.025471    0.029965    0.653165    
#> 491: 0.403311    1.097834    3.469809    0.520177    0.025490    0.029970    0.653108    
#> 492: 0.403271    1.097926    3.469753    0.520211    0.025507    0.029986    0.653097    
#> 493: 0.403132    1.097974    3.469716    0.520195    0.025510    0.029994    0.653137    
#> 494: 0.403168    1.098029    3.469673    0.520016    0.025512    0.029999    0.653152    
#> 495: 0.403198    1.098092    3.469688    0.519937    0.025502    0.029989    0.653139    
#> 496: 0.403300    1.098157    3.469671    0.520043    0.025523    0.029996    0.653089    
#> 497: 0.403300    1.098143    3.469668    0.520198    0.025524    0.029999    0.653128    
#> 498: 0.403318    1.098141    3.469659    0.520249    0.025523    0.029998    0.653128    
#> 499: 0.403291    1.098217    3.469606    0.520187    0.025541    0.029988    0.653087    
#> 500: 0.403278    1.098233    3.469553    0.520373    0.025529    0.029988    0.653029    
#> [====|====|=>-------------------------------------]  25%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.282846    0.992756    3.449632    0.392401    0.085936    0.016800    1.685169    
#> 002: 0.349988    1.004434    3.486268    0.372781    0.081639    0.016517    1.109585    
#> 003: 0.424115    1.002946    3.491761    0.354142    0.092498    0.015691    0.915694    
#> 004: 0.351560    1.013391    3.482236    0.336435    0.096500    0.015506    0.817481    
#> 005: 0.378164    1.027073    3.469598    0.319613    0.109289    0.014731    0.751494    
#> 006: 0.390916    1.045008    3.460584    0.303666    0.103824    0.014573    0.690806    
#> 007: 0.372798    1.047506    3.456006    0.317673    0.102727    0.016757    0.680046    
#> 008: 0.347569    1.043131    3.450385    0.332746    0.097591    0.015919    0.679885    
#> 009: 0.326456    1.037047    3.442301    0.316108    0.092711    0.017628    0.657956    
#> 010: 0.337312    1.036470    3.447408    0.356554    0.088076    0.020929    0.671407    
#> 011: 0.398063    1.040418    3.457403    0.355088    0.083672    0.021625    0.671579    
#> 012: 0.384385    1.048782    3.458040    0.337334    0.102688    0.020742    0.643757    
#> 013: 0.345320    1.052194    3.445360    0.320467    0.097554    0.021553    0.662339    
#> 014: 0.372420    1.057246    3.462253    0.331911    0.103692    0.027011    0.664203    
#> 015: 0.396492    1.064483    3.461591    0.315315    0.098507    0.025661    0.651483    
#> 016: 0.394783    1.064939    3.469645    0.340300    0.093582    0.024378    0.649193    
#> 017: 0.382005    1.068753    3.463914    0.325954    0.088903    0.025854    0.641230    
#> 018: 0.403945    1.064331    3.454572    0.377825    0.084458    0.028640    0.628837    
#> 019: 0.385387    1.051815    3.456938    0.358934    0.080235    0.027208    0.639557    
#> 020: 0.393355    1.050717    3.457364    0.375530    0.076223    0.028122    0.640398    
#> 021: 0.387840    1.052836    3.456088    0.384469    0.072412    0.026716    0.631413    
#> 022: 0.403367    1.039526    3.476328    0.406510    0.075687    0.029534    0.626247    
#> 023: 0.500403    1.042362    3.480831    0.432402    0.082155    0.028667    0.646093    
#> 024: 0.465926    1.049711    3.487175    0.410782    0.083373    0.027234    0.634104    
#> 025: 0.423804    1.038303    3.482112    0.390243    0.079204    0.025872    0.637724    
#> 026: 0.393618    1.050493    3.462251    0.370731    0.088163    0.024578    0.642465    
#> 027: 0.365502    1.062576    3.454729    0.352194    0.088729    0.023349    0.666573    
#> 028: 0.435533    1.037446    3.472362    0.367778    0.084292    0.025812    0.654520    
#> 029: 0.429659    1.052244    3.456140    0.439424    0.098323    0.028696    0.664838    
#> 030: 0.431135    1.044186    3.459212    0.417453    0.093406    0.027536    0.658948    
#> 031: 0.369387    1.033450    3.464586    0.396580    0.088736    0.026159    0.666683    
#> 032: 0.389605    1.028484    3.472275    0.376751    0.102132    0.024851    0.660247    
#> 033: 0.411796    1.041637    3.474927    0.357913    0.102602    0.023608    0.669822    
#> 034: 0.413061    1.045063    3.479808    0.369036    0.097472    0.022428    0.655423    
#> 035: 0.412871    1.034178    3.468409    0.389375    0.092598    0.021529    0.636728    
#> 036: 0.409863    1.043238    3.470504    0.369906    0.087968    0.024283    0.651825    
#> 037: 0.412224    1.050379    3.466774    0.401820    0.083681    0.025593    0.672394    
#> 038: 0.471475    1.035881    3.477499    0.424830    0.079497    0.026997    0.653802    
#> 039: 0.454718    1.026990    3.477576    0.411081    0.084653    0.025647    0.642276    
#> 040: 0.448622    1.034695    3.483215    0.418016    0.086509    0.024365    0.644741    
#> 041: 0.412630    1.032802    3.475693    0.397115    0.082183    0.026386    0.648792    
#> 042: 0.430783    1.033460    3.464683    0.377259    0.078074    0.030300    0.647390    
#> 043: 0.393636    1.068210    3.451819    0.358396    0.087148    0.031968    0.653383    
#> 044: 0.349554    1.076068    3.450172    0.340477    0.088785    0.030370    0.654820    
#> 045: 0.369872    1.078771    3.448230    0.367591    0.084346    0.028851    0.646061    
#> 046: 0.373941    1.055497    3.453048    0.391602    0.080128    0.028020    0.639505    
#> 047: 0.346746    1.072233    3.447666    0.372022    0.086604    0.026619    0.641030    
#> 048: 0.381966    1.049264    3.461919    0.353421    0.088419    0.027180    0.658240    
#> 049: 0.384216    1.053121    3.463081    0.335750    0.091868    0.025821    0.662939    
#> 050: 0.426093    1.022310    3.478478    0.366873    0.093152    0.027369    0.676072    
#> 051: 0.440538    1.027129    3.463642    0.396261    0.088494    0.026000    0.685664    
#> 052: 0.445495    1.049950    3.483327    0.411166    0.090362    0.024700    0.660292    
#> 053: 0.456031    1.034188    3.468072    0.428162    0.095335    0.024876    0.650309    
#> 054: 0.384639    1.048682    3.478200    0.406754    0.090569    0.033973    0.678346    
#> 055: 0.413099    1.045379    3.464534    0.413772    0.093208    0.032275    0.672057    
#> 056: 0.397169    1.046640    3.463260    0.393084    0.103285    0.030661    0.682948    
#> 057: 0.396103    1.051253    3.472727    0.373429    0.098121    0.029128    0.675881    
#> 058: 0.381591    1.059724    3.454827    0.384731    0.104983    0.027672    0.670452    
#> 059: 0.393070    1.033151    3.466651    0.390429    0.099734    0.026288    0.672730    
#> 060: 0.418911    1.015642    3.475266    0.431120    0.094747    0.029395    0.656742    
#> 061: 0.462165    1.045607    3.473832    0.440471    0.090010    0.034348    0.666829    
#> 062: 0.406349    1.054018    3.476590    0.418447    0.085509    0.038687    0.689631    
#> 063: 0.479167    1.022003    3.484376    0.397525    0.081234    0.036752    0.659684    
#> 064: 0.450894    1.036842    3.490601    0.377649    0.086463    0.034915    0.661076    
#> 065: 0.442592    1.016662    3.478423    0.358766    0.082140    0.037123    0.661706    
#> 066: 0.462783    1.045476    3.483100    0.340828    0.078033    0.036237    0.655130    
#> 067: 0.410541    1.033404    3.469696    0.365625    0.074131    0.034425    0.647702    
#> 068: 0.398429    1.046382    3.471604    0.375517    0.080984    0.032704    0.661781    
#> 069: 0.398135    1.031730    3.458921    0.418142    0.078168    0.031434    0.653249    
#> 070: 0.374676    1.041735    3.457432    0.411007    0.075500    0.032280    0.646326    
#> 071: 0.326762    1.064562    3.424276    0.390456    0.071725    0.030666    0.675192    
#> 072: 0.361047    1.022792    3.449972    0.370934    0.069264    0.037668    0.657748    
#> 073: 0.379412    1.061188    3.466577    0.385480    0.069209    0.035785    0.657768    
#> 074: 0.362906    1.057088    3.465686    0.368048    0.073737    0.033996    0.647901    
#> 075: 0.424961    1.035425    3.479061    0.376775    0.081528    0.032296    0.637456    
#> 076: 0.430146    1.042788    3.484868    0.420303    0.077452    0.035874    0.650271    
#> 077: 0.470228    1.029064    3.484373    0.399288    0.075787    0.034080    0.655071    
#> 078: 0.437468    1.034108    3.487429    0.379323    0.076645    0.032376    0.651118    
#> 079: 0.404368    1.043728    3.472589    0.384017    0.072813    0.032065    0.647697    
#> 080: 0.441720    1.052424    3.476491    0.455222    0.071735    0.032801    0.635698    
#> 081: 0.427875    1.047725    3.477481    0.432461    0.071929    0.031161    0.638332    
#> 082: 0.420255    1.036167    3.479321    0.410838    0.074626    0.029603    0.660848    
#> 083: 0.416362    1.051886    3.482781    0.390296    0.074722    0.028122    0.668449    
#> 084: 0.413957    1.063398    3.470133    0.370781    0.095295    0.026716    0.645431    
#> 085: 0.397184    1.056154    3.474451    0.382130    0.090530    0.025381    0.650746    
#> 086: 0.387697    1.046219    3.466053    0.363023    0.086004    0.027924    0.654407    
#> 087: 0.402138    1.038057    3.460488    0.344872    0.081704    0.026527    0.653567    
#> 088: 0.355901    1.034120    3.467966    0.384629    0.077619    0.026905    0.669141    
#> 089: 0.402250    1.055101    3.460285    0.425335    0.073738    0.031731    0.641690    
#> 090: 0.407157    1.049797    3.452497    0.448578    0.074625    0.031892    0.644369    
#> 091: 0.393704    1.028759    3.455187    0.426149    0.078159    0.030297    0.658304    
#> 092: 0.399966    1.060423    3.451330    0.414488    0.080886    0.028782    0.641747    
#> 093: 0.378201    1.054707    3.460284    0.393764    0.091424    0.027343    0.661599    
#> 094: 0.365579    1.069795    3.445168    0.397002    0.090124    0.025976    0.675688    
#> 095: 0.401182    1.051600    3.469343    0.405186    0.085618    0.026694    0.659511    
#> 096: 0.362267    1.042879    3.469418    0.384927    0.081337    0.027552    0.658486    
#> 097: 0.382123    1.076228    3.457405    0.382403    0.087683    0.028904    0.642313    
#> 098: 0.364608    1.074446    3.464849    0.363283    0.083727    0.027459    0.654017    
#> 099: 0.370344    1.034581    3.452671    0.365407    0.079540    0.027998    0.652571    
#> 100: 0.375578    1.052195    3.460976    0.347136    0.075563    0.026598    0.643017    
#> 101: 0.363643    1.034337    3.440774    0.343996    0.071785    0.025268    0.644139    
#> 102: 0.374980    1.053459    3.444752    0.327360    0.074713    0.024005    0.642053    
#> 103: 0.405603    1.043029    3.463512    0.389109    0.084974    0.025785    0.638176    
#> 104: 0.378992    1.041154    3.468325    0.377417    0.083702    0.028975    0.650962    
#> 105: 0.388414    1.050529    3.467858    0.375008    0.087664    0.027526    0.650646    
#> 106: 0.402645    1.045448    3.473322    0.356258    0.085482    0.026150    0.648250    
#> 107: 0.441760    1.045551    3.470895    0.371604    0.097539    0.025773    0.638527    
#> 108: 0.419225    1.046929    3.471293    0.367535    0.101319    0.025845    0.660018    
#> 109: 0.405696    1.039543    3.476814    0.381609    0.096253    0.028961    0.640638    
#> 110: 0.406956    1.039058    3.463100    0.379422    0.091441    0.027513    0.641360    
#> 111: 0.399967    1.035586    3.461454    0.360451    0.092630    0.026137    0.627206    
#> 112: 0.414136    1.036001    3.458216    0.369382    0.087998    0.024830    0.650070    
#> 113: 0.395115    1.056109    3.469831    0.355009    0.083598    0.028565    0.637269    
#> 114: 0.414229    1.054286    3.466896    0.347201    0.088411    0.031604    0.630774    
#> 115: 0.409820    1.038574    3.469875    0.330095    0.102473    0.034568    0.665145    
#> 116: 0.395533    1.047648    3.456604    0.358473    0.097350    0.032839    0.653203    
#> 117: 0.380718    1.050012    3.473944    0.340550    0.092482    0.031197    0.669718    
#> 118: 0.401301    1.030116    3.469535    0.323522    0.087858    0.029637    0.662081    
#> 119: 0.414334    1.034430    3.478076    0.320357    0.083465    0.031187    0.662130    
#> 120: 0.382231    1.027684    3.469445    0.342108    0.079292    0.032674    0.658969    
#> 121: 0.374773    1.076371    3.462909    0.325002    0.075327    0.031272    0.676585    
#> 122: 0.402555    1.032468    3.478204    0.317496    0.074010    0.029804    0.655681    
#> 123: 0.387272    1.064586    3.453473    0.366246    0.070310    0.029607    0.664935    
#> 124: 0.424160    1.075319    3.460985    0.423194    0.072454    0.032962    0.661358    
#> 125: 0.383790    1.080900    3.443886    0.402034    0.082569    0.031314    0.658325    
#> 126: 0.368786    1.110894    3.421804    0.445358    0.092890    0.029748    0.669823    
#> 127: 0.367382    1.081773    3.430729    0.423090    0.088245    0.028261    0.660593    
#> 128: 0.350150    1.069724    3.454973    0.401935    0.090093    0.026848    0.659190    
#> 129: 0.363094    1.071918    3.441494    0.418013    0.090135    0.025505    0.660190    
#> 130: 0.411617    1.069031    3.456164    0.417662    0.086296    0.029237    0.659121    
#> 131: 0.443701    1.035384    3.457943    0.404796    0.081981    0.029472    0.678771    
#> 132: 0.381355    1.068920    3.445893    0.427097    0.088458    0.027998    0.653659    
#> 133: 0.383420    1.047887    3.448898    0.422566    0.084035    0.030086    0.646459    
#> 134: 0.403009    1.024761    3.469349    0.412421    0.082347    0.036782    0.638856    
#> 135: 0.408923    1.013332    3.471347    0.405835    0.079288    0.036067    0.643085    
#> 136: 0.436396    1.018916    3.476039    0.411435    0.075323    0.040193    0.632384    
#> 137: 0.460713    1.021479    3.489969    0.404925    0.071557    0.038183    0.653364    
#> 138: 0.435389    1.041436    3.475489    0.388744    0.068765    0.036274    0.653675    
#> 139: 0.413290    1.039925    3.485859    0.395323    0.067793    0.035355    0.662166    
#> 140: 0.433465    1.041248    3.477199    0.397125    0.064404    0.039235    0.659535    
#> 141: 0.392247    1.012313    3.474319    0.377269    0.064250    0.037273    0.670287    
#> 142: 0.409848    1.004396    3.488704    0.358405    0.067378    0.035409    0.679122    
#> 143: 0.393321    1.040550    3.484975    0.366105    0.087211    0.033639    0.657743    
#> 144: 0.451548    1.029108    3.489781    0.347800    0.082850    0.031957    0.661348    
#> 145: 0.402481    1.051413    3.463551    0.382161    0.089211    0.030359    0.651245    
#> 146: 0.371796    1.051267    3.472280    0.363053    0.084750    0.028841    0.660006    
#> 147: 0.373636    1.045838    3.460181    0.344901    0.080513    0.027399    0.660377    
#> 148: 0.375131    1.040518    3.479290    0.327656    0.085642    0.026029    0.648831    
#> 149: 0.382702    1.048070    3.471266    0.346704    0.085852    0.028130    0.647998    
#> 150: 0.400661    1.058443    3.446862    0.358048    0.088941    0.026724    0.651884    
#> 151: 0.323042    1.085166    3.442237    0.340146    0.097218    0.025388    0.643204    
#> 152: 0.381834    1.075014    3.449498    0.441393    0.092199    0.026445    0.625755    
#> 153: 0.381971    1.065203    3.460087    0.422105    0.091501    0.030999    0.647478    
#> 154: 0.416305    1.064135    3.469172    0.415078    0.074935    0.037123    0.653082    
#> 155: 0.403127    1.044707    3.467256    0.441789    0.069411    0.033000    0.636787    
#> 156: 0.420227    1.035921    3.462965    0.413245    0.074384    0.030165    0.651585    
#> 157: 0.423529    1.033465    3.473710    0.389338    0.077761    0.033039    0.647378    
#> 158: 0.415911    1.047254    3.458502    0.419288    0.080524    0.030695    0.631754    
#> 159: 0.375909    1.049832    3.466326    0.392905    0.092544    0.028022    0.641667    
#> 160: 0.435021    1.022743    3.476826    0.467155    0.084727    0.030605    0.645108    
#> 161: 0.430084    1.044403    3.465865    0.434292    0.078692    0.034264    0.640275    
#> 162: 0.428305    1.060531    3.453696    0.583495    0.075540    0.034068    0.652567    
#> 163: 0.412052    1.048117    3.457015    0.550205    0.084356    0.033586    0.632220    
#> 164: 0.405330    1.057766    3.458376    0.434099    0.082800    0.032199    0.636129    
#> 165: 0.409389    1.057314    3.466261    0.448111    0.094995    0.026628    0.649112    
#> 166: 0.456421    1.065730    3.463654    0.486432    0.090403    0.027757    0.649228    
#> 167: 0.456124    1.064321    3.480199    0.479334    0.092465    0.031345    0.648601    
#> 168: 0.435927    1.043821    3.485443    0.439885    0.085181    0.028749    0.635027    
#> 169: 0.431534    1.067761    3.468560    0.398981    0.094334    0.026215    0.643765    
#> 170: 0.423737    1.050348    3.479480    0.411673    0.084900    0.027806    0.625663    
#> 171: 0.417218    1.053461    3.464048    0.461667    0.105565    0.024095    0.631520    
#> 172: 0.404006    1.074479    3.458050    0.470989    0.090247    0.027609    0.631512    
#> 173: 0.402322    1.054472    3.445776    0.404400    0.075331    0.025975    0.625620    
#> 174: 0.414415    1.050239    3.468216    0.380687    0.086591    0.026467    0.630336    
#> 175: 0.414959    1.051109    3.462473    0.409083    0.077664    0.030816    0.639072    
#> 176: 0.419487    1.057412    3.475779    0.394852    0.071331    0.033479    0.626189    
#> 177: 0.442697    1.054373    3.476074    0.411796    0.084174    0.035505    0.631703    
#> 178: 0.437113    1.034779    3.471452    0.419352    0.080998    0.028748    0.640742    
#> 179: 0.443964    1.054441    3.475917    0.368391    0.064149    0.028265    0.671861    
#> 180: 0.382723    1.031577    3.469283    0.329143    0.068624    0.027290    0.670645    
#> 181: 0.380037    1.044480    3.445598    0.336541    0.052986    0.022835    0.677526    
#> 182: 0.315586    1.065218    3.435000    0.320354    0.062805    0.022035    0.695636    
#> 183: 0.361546    1.079993    3.443324    0.356034    0.075812    0.019778    0.679922    
#> 184: 0.377082    1.047901    3.450769    0.429027    0.082213    0.022490    0.664771    
#> 185: 0.348804    1.056636    3.445512    0.358939    0.089370    0.019175    0.652799    
#> 186: 0.360309    1.078240    3.449440    0.327111    0.088370    0.021779    0.663173    
#> 187: 0.384037    1.053377    3.459845    0.345706    0.077897    0.025315    0.652246    
#> 188: 0.404464    1.078943    3.455376    0.410599    0.076171    0.028927    0.657565    
#> 189: 0.394097    1.064077    3.462718    0.381902    0.070440    0.028617    0.659098    
#> 190: 0.390193    1.071641    3.466323    0.392218    0.071442    0.025714    0.647981    
#> 191: 0.382534    1.059414    3.458558    0.331045    0.067320    0.026593    0.646368    
#> 192: 0.385164    1.065736    3.473618    0.303543    0.073396    0.029360    0.648189    
#> 193: 0.397439    1.071280    3.460320    0.377268    0.073947    0.031241    0.637076    
#> 194: 0.384309    1.080705    3.448064    0.410570    0.069733    0.031006    0.644436    
#> 195: 0.354407    1.065365    3.450050    0.395515    0.061360    0.031165    0.643623    
#> 196: 0.354620    1.060441    3.455905    0.416675    0.065173    0.034830    0.646729    
#> 197: 0.358118    1.051638    3.462889    0.415747    0.066078    0.037619    0.658506    
#> 198: 0.413369    1.063593    3.456557    0.413465    0.070692    0.036096    0.635617    
#> 199: 0.405720    1.051503    3.455119    0.447290    0.069970    0.034747    0.647582    
#> 200: 0.385421    1.048465    3.465246    0.400814    0.073779    0.025317    0.664664    
#> 201: 0.378803    1.041837    3.462049    0.394971    0.073675    0.026115    0.660966    
#> 202: 0.384732    1.045675    3.464197    0.399771    0.073717    0.027358    0.660389    
#> 203: 0.392140    1.042541    3.462203    0.390016    0.075017    0.027154    0.659356    
#> 204: 0.392302    1.040616    3.464304    0.392160    0.076026    0.026642    0.657298    
#> 205: 0.394656    1.046096    3.465099    0.389613    0.077069    0.026664    0.659993    
#> 206: 0.399138    1.049957    3.467211    0.395293    0.077980    0.026412    0.660543    
#> 207: 0.402693    1.050687    3.467840    0.396502    0.080165    0.026153    0.657692    
#> 208: 0.405617    1.050917    3.467180    0.396200    0.079877    0.026190    0.654893    
#> 209: 0.409969    1.047165    3.467184    0.396806    0.079050    0.026043    0.653887    
#> 210: 0.411478    1.043770    3.468882    0.389754    0.079181    0.026167    0.653514    
#> 211: 0.414641    1.043845    3.470302    0.386880    0.078797    0.026192    0.653271    
#> 212: 0.418538    1.041881    3.471688    0.386469    0.078594    0.026291    0.652974    
#> 213: 0.421719    1.039775    3.473394    0.384473    0.079152    0.026274    0.652376    
#> 214: 0.425791    1.038100    3.475040    0.390025    0.079642    0.026347    0.652179    
#> 215: 0.427263    1.036638    3.476208    0.388894    0.080336    0.026342    0.652359    
#> 216: 0.429675    1.035706    3.476850    0.389107    0.080436    0.026204    0.653477    
#> 217: 0.429517    1.035907    3.477063    0.388986    0.079815    0.026243    0.652407    
#> 218: 0.430614    1.035360    3.476984    0.390641    0.080009    0.026498    0.651922    
#> 219: 0.429770    1.035136    3.476765    0.387112    0.080545    0.026676    0.651994    
#> 220: 0.428307    1.035632    3.476494    0.385615    0.080670    0.026826    0.652821    
#> 221: 0.427214    1.036256    3.476297    0.384314    0.080691    0.026795    0.652863    
#> 222: 0.428103    1.036667    3.475969    0.386281    0.080475    0.026801    0.652916    
#> 223: 0.429477    1.036659    3.475814    0.387023    0.080355    0.026807    0.652535    
#> 224: 0.431941    1.037105    3.476206    0.389308    0.080039    0.026984    0.652181    
#> 225: 0.432232    1.035965    3.476686    0.387316    0.079700    0.027456    0.652011    
#> 226: 0.432498    1.035003    3.476720    0.387242    0.079180    0.027695    0.651481    
#> 227: 0.431670    1.034432    3.476810    0.385047    0.078996    0.027784    0.650838    
#> 228: 0.430061    1.034420    3.476369    0.382505    0.078607    0.027896    0.650555    
#> 229: 0.429787    1.034060    3.476094    0.381416    0.078290    0.027892    0.650242    
#> 230: 0.429753    1.033691    3.476057    0.379981    0.077980    0.027859    0.650524    
#> 231: 0.430407    1.034375    3.476019    0.380126    0.077978    0.027768    0.650395    
#> 232: 0.429250    1.035176    3.475843    0.378879    0.077806    0.027782    0.650317    
#> 233: 0.428066    1.035033    3.475760    0.378761    0.078033    0.027841    0.650015    
#> 234: 0.426364    1.035075    3.475710    0.377659    0.077899    0.027899    0.650416    
#> 235: 0.425574    1.035494    3.475372    0.376862    0.078046    0.027750    0.650583    
#> 236: 0.424668    1.036276    3.475013    0.375483    0.078389    0.027700    0.650816    
#> 237: 0.423813    1.036447    3.474907    0.374889    0.078436    0.027794    0.651359    
#> 238: 0.423343    1.036423    3.474750    0.373650    0.078529    0.027846    0.652233    
#> 239: 0.422508    1.037345    3.474345    0.372611    0.078717    0.027860    0.652220    
#> 240: 0.421688    1.037404    3.474309    0.371554    0.078910    0.027731    0.652023    
#> 241: 0.421710    1.036650    3.474370    0.370764    0.079112    0.027660    0.651974    
#> 242: 0.421692    1.035931    3.474460    0.371231    0.079472    0.027570    0.652284    
#> 243: 0.421612    1.036267    3.474404    0.370850    0.079767    0.027547    0.652553    
#> 244: 0.421448    1.036829    3.474488    0.370639    0.079964    0.027552    0.652757    
#> 245: 0.421184    1.036638    3.474530    0.371110    0.080104    0.027618    0.653123    
#> 246: 0.421805    1.036539    3.474376    0.371441    0.079781    0.027661    0.653201    
#> 247: 0.421368    1.036707    3.474221    0.371477    0.079768    0.027714    0.653303    
#> 248: 0.421335    1.037201    3.474174    0.371704    0.080002    0.027666    0.653121    
#> 249: 0.420747    1.037275    3.474116    0.371149    0.080033    0.027688    0.652855    
#> 250: 0.421276    1.037537    3.474195    0.371906    0.079759    0.027657    0.652595    
#> 251: 0.422196    1.037734    3.474471    0.372739    0.079614    0.027660    0.652575    
#> 252: 0.423234    1.037389    3.475005    0.372737    0.079592    0.027731    0.652656    
#> 253: 0.423598    1.037372    3.475061    0.373024    0.079644    0.027741    0.652751    
#> 254: 0.424289    1.037516    3.475235    0.374297    0.079524    0.027795    0.652973    
#> 255: 0.424922    1.037805    3.475286    0.375949    0.079598    0.027840    0.652676    
#> 256: 0.425371    1.037982    3.475169    0.376658    0.079631    0.027905    0.652641    
#> 257: 0.425137    1.037706    3.475023    0.376892    0.079334    0.027977    0.652631    
#> 258: 0.424499    1.037200    3.474978    0.377181    0.079230    0.027998    0.652535    
#> 259: 0.423981    1.037034    3.474819    0.377008    0.079069    0.028075    0.652447    
#> 260: 0.423577    1.037134    3.474751    0.376851    0.078902    0.028131    0.652217    
#> 261: 0.423510    1.037038    3.474960    0.376968    0.078796    0.028217    0.652045    
#> 262: 0.423456    1.037473    3.474973    0.376407    0.078741    0.028280    0.652016    
#> 263: 0.423541    1.037839    3.474980    0.376034    0.078946    0.028268    0.651908    
#> 264: 0.423730    1.038132    3.475009    0.376557    0.079121    0.028213    0.651997    
#> 265: 0.424003    1.038650    3.474915    0.377213    0.079328    0.028150    0.652255    
#> 266: 0.423917    1.038967    3.474691    0.377858    0.079130    0.028134    0.652139    
#> 267: 0.424234    1.038716    3.474787    0.378529    0.078979    0.028155    0.652287    
#> 268: 0.424422    1.038432    3.474697    0.379785    0.079074    0.028101    0.652213    
#> 269: 0.425257    1.038235    3.474803    0.380770    0.079107    0.028053    0.652330    
#> 270: 0.425481    1.038371    3.474778    0.382135    0.079208    0.027960    0.652069    
#> 271: 0.425354    1.038302    3.474646    0.382223    0.079328    0.027851    0.652227    
#> 272: 0.425179    1.038100    3.474564    0.382875    0.079387    0.027880    0.652479    
#> 273: 0.424572    1.038270    3.474493    0.382781    0.079445    0.027861    0.652670    
#> 274: 0.424350    1.038620    3.474533    0.381502    0.079577    0.027852    0.652739    
#> 275: 0.424250    1.038960    3.474383    0.381212    0.079448    0.027881    0.652734    
#> 276: 0.423753    1.039222    3.474434    0.380173    0.079323    0.027983    0.652710    
#> 277: 0.423078    1.039483    3.474365    0.379554    0.079257    0.028073    0.652663    
#> 278: 0.422514    1.039810    3.474315    0.379248    0.079374    0.028077    0.652783    
#> 279: 0.422303    1.039792    3.474203    0.379772    0.079357    0.028130    0.652926    
#> 280: 0.421886    1.040109    3.474198    0.379683    0.079270    0.028172    0.652865    
#> 281: 0.421374    1.040211    3.474088    0.379853    0.079106    0.028223    0.652815    
#> 282: 0.421069    1.040482    3.474028    0.380777    0.078893    0.028235    0.653023    
#> 283: 0.420863    1.040777    3.473929    0.381021    0.078776    0.028266    0.653194    
#> 284: 0.420321    1.040804    3.473817    0.381137    0.078764    0.028279    0.653152    
#> 285: 0.419952    1.041183    3.473509    0.381131    0.078853    0.028312    0.653296    
#> 286: 0.419467    1.041513    3.473194    0.381191    0.078586    0.028320    0.653472    
#> 287: 0.419183    1.041914    3.473026    0.380858    0.078673    0.028334    0.653456    
#> 288: 0.419120    1.041966    3.473022    0.380743    0.078754    0.028318    0.653754    
#> 289: 0.419225    1.042033    3.472966    0.380107    0.078716    0.028278    0.654068    
#> 290: 0.419820    1.041907    3.473055    0.379984    0.078847    0.028291    0.654159    
#> 291: 0.420040    1.041853    3.473183    0.380092    0.078759    0.028298    0.654212    
#> 292: 0.419837    1.041803    3.473328    0.380245    0.078663    0.028327    0.654354    
#> 293: 0.419854    1.041772    3.473373    0.380128    0.078632    0.028352    0.654374    
#> 294: 0.420004    1.041804    3.473336    0.379925    0.078593    0.028342    0.654206    
#> 295: 0.420428    1.041569    3.473373    0.379998    0.078571    0.028343    0.654107    
#> 296: 0.420485    1.041364    3.473413    0.379557    0.078549    0.028379    0.654207    
#> 297: 0.420423    1.041367    3.473390    0.379527    0.078548    0.028400    0.654172    
#> 298: 0.420143    1.041286    3.473240    0.379299    0.078636    0.028385    0.654030    
#> 299: 0.419946    1.041426    3.473070    0.379654    0.078611    0.028384    0.653854    
#> 300: 0.420100    1.041392    3.473051    0.379934    0.078522    0.028397    0.653725    
#> 301: 0.420075    1.041399    3.472903    0.379787    0.078436    0.028378    0.653794    
#> 302: 0.420197    1.041482    3.472871    0.379546    0.078447    0.028339    0.653590    
#> 303: 0.419722    1.041655    3.472662    0.379379    0.078578    0.028298    0.653417    
#> 304: 0.419377    1.041945    3.472394    0.379742    0.078731    0.028248    0.653434    
#> 305: 0.419066    1.041983    3.472116    0.380428    0.078704    0.028237    0.653492    
#> 306: 0.418361    1.042052    3.471817    0.380734    0.078742    0.028201    0.653373    
#> 307: 0.417670    1.042167    3.471584    0.380560    0.078751    0.028156    0.653434    
#> 308: 0.417192    1.042225    3.471421    0.380877    0.078841    0.028118    0.653492    
#> 309: 0.416808    1.042021    3.471330    0.381003    0.078894    0.028124    0.653626    
#> 310: 0.416794    1.042005    3.471184    0.381037    0.078770    0.028131    0.653694    
#> 311: 0.416867    1.042057    3.471179    0.380954    0.078654    0.028108    0.653690    
#> 312: 0.416903    1.042062    3.471120    0.381695    0.078569    0.028091    0.653879    
#> 313: 0.416857    1.041941    3.471100    0.381722    0.078575    0.028117    0.653933    
#> 314: 0.416491    1.042134    3.470906    0.382121    0.078557    0.028156    0.653862    
#> 315: 0.415993    1.042411    3.470687    0.382305    0.078610    0.028163    0.653744    
#> 316: 0.415629    1.042648    3.470408    0.383902    0.078549    0.028186    0.653591    
#> 317: 0.415619    1.042801    3.470276    0.384805    0.078523    0.028222    0.653490    
#> 318: 0.415399    1.042903    3.470076    0.384820    0.078523    0.028196    0.653288    
#> 319: 0.415271    1.043053    3.469962    0.384693    0.078471    0.028153    0.653277    
#> 320: 0.414688    1.043179    3.469767    0.384542    0.078442    0.028111    0.653280    
#> 321: 0.414240    1.043208    3.469570    0.384285    0.078443    0.028080    0.653301    
#> 322: 0.414097    1.043237    3.469455    0.384662    0.078447    0.028084    0.653210    
#> 323: 0.413780    1.043190    3.469429    0.385094    0.078371    0.028084    0.653184    
#> 324: 0.413804    1.043286    3.469469    0.385576    0.078400    0.028106    0.653040    
#> 325: 0.413798    1.043414    3.469305    0.386285    0.078336    0.028084    0.653010    
#> 326: 0.413567    1.043362    3.469214    0.386384    0.078282    0.028058    0.652873    
#> 327: 0.413571    1.043434    3.469211    0.386873    0.078184    0.028006    0.652848    
#> 328: 0.414054    1.043488    3.469253    0.387124    0.078200    0.027980    0.652787    
#> 329: 0.414354    1.043446    3.469363    0.387198    0.078240    0.027969    0.652853    
#> 330: 0.414127    1.043350    3.469423    0.387383    0.078204    0.027937    0.652820    
#> 331: 0.414163    1.043519    3.469412    0.387345    0.078227    0.027925    0.652807    
#> 332: 0.414000    1.043501    3.469417    0.387470    0.078392    0.027895    0.652754    
#> 333: 0.413742    1.043585    3.469254    0.387555    0.078491    0.027875    0.652706    
#> 334: 0.413461    1.043572    3.469046    0.387614    0.078657    0.027869    0.652748    
#> 335: 0.413125    1.043716    3.468809    0.387974    0.078780    0.027837    0.652771    
#> 336: 0.412822    1.043956    3.468680    0.388375    0.078903    0.027799    0.652797    
#> 337: 0.412638    1.044319    3.468481    0.389026    0.078949    0.027774    0.652839    
#> 338: 0.412447    1.044445    3.468351    0.389663    0.078947    0.027769    0.652860    
#> 339: 0.412252    1.044520    3.468244    0.390210    0.078991    0.027774    0.652812    
#> 340: 0.412320    1.044187    3.468251    0.390754    0.078876    0.027786    0.652824    
#> 341: 0.412382    1.044175    3.468274    0.390935    0.078838    0.027813    0.652681    
#> 342: 0.412419    1.044080    3.468311    0.390982    0.078824    0.027808    0.652640    
#> 343: 0.412338    1.044127    3.468245    0.391290    0.078802    0.027832    0.652622    
#> 344: 0.412454    1.044036    3.468260    0.392208    0.078739    0.027879    0.652631    
#> 345: 0.412516    1.043965    3.468242    0.392403    0.078690    0.027930    0.652678    
#> 346: 0.412574    1.043914    3.468193    0.392710    0.078654    0.027941    0.652597    
#> 347: 0.412585    1.043886    3.468112    0.392880    0.078702    0.027953    0.652697    
#> 348: 0.412112    1.043996    3.467907    0.392806    0.078657    0.027964    0.652724    
#> 349: 0.411876    1.043873    3.467798    0.393156    0.078608    0.027959    0.652744    
#> 350: 0.411633    1.043831    3.467686    0.393131    0.078659    0.027938    0.652756    
#> 351: 0.411458    1.044081    3.467556    0.393143    0.078630    0.027930    0.652865    
#> 352: 0.410975    1.044200    3.467429    0.393266    0.078602    0.028016    0.652940    
#> 353: 0.410905    1.044472    3.467283    0.393832    0.078614    0.028076    0.653016    
#> 354: 0.410608    1.044581    3.467112    0.393912    0.078676    0.028131    0.653097    
#> 355: 0.410732    1.044496    3.467005    0.393849    0.078755    0.028183    0.653110    
#> 356: 0.410483    1.044440    3.466955    0.393528    0.078759    0.028183    0.653268    
#> 357: 0.410559    1.044453    3.467000    0.393641    0.078713    0.028174    0.653225    
#> 358: 0.410539    1.044386    3.467013    0.393407    0.078754    0.028187    0.653197    
#> 359: 0.410469    1.044449    3.466980    0.393278    0.078725    0.028239    0.653077    
#> 360: 0.410229    1.044467    3.466918    0.392841    0.078653    0.028261    0.653014    
#> 361: 0.410045    1.044465    3.466920    0.392692    0.078577    0.028277    0.652964    
#> 362: 0.410118    1.044509    3.466876    0.392740    0.078561    0.028267    0.652871    
#> 363: 0.410028    1.044535    3.466830    0.392997    0.078619    0.028236    0.652716    
#> 364: 0.409993    1.044606    3.466770    0.393157    0.078731    0.028212    0.652702    
#> 365: 0.409794    1.044608    3.466694    0.393286    0.078859    0.028195    0.652677    
#> 366: 0.409449    1.044752    3.466503    0.392953    0.078884    0.028175    0.652737    
#> 367: 0.409241    1.044962    3.466349    0.392516    0.078917    0.028144    0.652764    
#> 368: 0.409256    1.045046    3.466287    0.392282    0.078884    0.028119    0.652719    
#> 369: 0.409204    1.045098    3.466248    0.392242    0.078819    0.028142    0.652661    
#> 370: 0.409031    1.045252    3.466181    0.392053    0.078779    0.028151    0.652601    
#> 371: 0.408801    1.045302    3.466149    0.391545    0.078722    0.028123    0.652528    
#> 372: 0.408651    1.045435    3.466111    0.391228    0.078724    0.028101    0.652484    
#> 373: 0.408289    1.045626    3.465951    0.390950    0.078764    0.028120    0.652459    
#> 374: 0.408310    1.045728    3.465913    0.390721    0.078741    0.028133    0.652453    
#> 375: 0.408360    1.045865    3.465873    0.390688    0.078709    0.028140    0.652445    
#> 376: 0.408363    1.045893    3.465870    0.390376    0.078696    0.028136    0.652422    
#> 377: 0.408235    1.046011    3.465880    0.390047    0.078674    0.028096    0.652438    
#> 378: 0.408089    1.045931    3.465934    0.389975    0.078634    0.028088    0.652455    
#> 379: 0.408012    1.045698    3.465988    0.389825    0.078588    0.028087    0.652481    
#> 380: 0.407763    1.045605    3.466053    0.389714    0.078604    0.028098    0.652522    
#> 381: 0.407343    1.045624    3.465949    0.389595    0.078627    0.028121    0.652543    
#> 382: 0.407097    1.045662    3.465871    0.389492    0.078596    0.028110    0.652478    
#> 383: 0.406957    1.045752    3.465807    0.389729    0.078602    0.028093    0.652320    
#> 384: 0.406743    1.045754    3.465792    0.389492    0.078624    0.028098    0.652246    
#> 385: 0.406814    1.045760    3.465795    0.389685    0.078653    0.028079    0.652133    
#> 386: 0.406907    1.045863    3.465830    0.389629    0.078721    0.028068    0.652131    
#> 387: 0.406842    1.045976    3.465874    0.389609    0.078682    0.028067    0.652128    
#> 388: 0.406640    1.046044    3.465870    0.389516    0.078767    0.028030    0.652128    
#> 389: 0.406455    1.045964    3.465812    0.389245    0.078809    0.028001    0.652174    
#> 390: 0.406222    1.046031    3.465785    0.388911    0.078773    0.027986    0.652133    
#> 391: 0.406375    1.046044    3.465754    0.388785    0.078743    0.027979    0.652052    
#> 392: 0.406601    1.046037    3.465832    0.388869    0.078736    0.027967    0.651999    
#> 393: 0.406623    1.045965    3.465824    0.388810    0.078802    0.027942    0.651962    
#> 394: 0.406721    1.045958    3.465814    0.388756    0.078871    0.027900    0.651954    
#> 395: 0.406736    1.046013    3.465790    0.388748    0.078954    0.027874    0.651971    
#> 396: 0.406858    1.046042    3.465811    0.389195    0.079033    0.027846    0.651872    
#> 397: 0.406837    1.046028    3.465805    0.389245    0.079064    0.027819    0.651830    
#> 398: 0.406775    1.045941    3.465808    0.388944    0.079101    0.027796    0.651848    
#> 399: 0.406909    1.045932    3.465872    0.388827    0.079122    0.027767    0.651800    
#> 400: 0.407125    1.045946    3.465907    0.388841    0.079165    0.027735    0.651754    
#> 401: 0.407232    1.045998    3.465896    0.388573    0.079245    0.027688    0.651837    
#> 402: 0.407223    1.046074    3.465897    0.388283    0.079354    0.027641    0.651917    
#> 403: 0.407364    1.046115    3.465923    0.388261    0.079385    0.027599    0.651914    
#> 404: 0.407396    1.046109    3.465910    0.388135    0.079439    0.027568    0.651936    
#> 405: 0.407414    1.046043    3.465948    0.387783    0.079527    0.027547    0.652025    
#> 406: 0.407373    1.046005    3.465942    0.387437    0.079565    0.027548    0.652136    
#> 407: 0.407329    1.046018    3.465996    0.387277    0.079639    0.027543    0.652208    
#> 408: 0.407331    1.046075    3.465997    0.386895    0.079743    0.027520    0.652300    
#> 409: 0.407157    1.046005    3.465946    0.386504    0.079758    0.027524    0.652327    
#> 410: 0.407015    1.045976    3.465894    0.386335    0.079731    0.027548    0.652307    
#> 411: 0.407067    1.045823    3.465900    0.386459    0.079645    0.027561    0.652387    
#> 412: 0.407020    1.045913    3.465880    0.386023    0.079665    0.027563    0.652377    
#> 413: 0.406899    1.046019    3.465845    0.385815    0.079676    0.027569    0.652477    
#> 414: 0.406750    1.046162    3.465845    0.385919    0.079732    0.027569    0.652488    
#> 415: 0.407066    1.046214    3.465846    0.386216    0.079778    0.027565    0.652571    
#> 416: 0.407188    1.046242    3.465873    0.386149    0.079852    0.027552    0.652653    
#> 417: 0.407340    1.046230    3.465885    0.386461    0.079859    0.027544    0.652599    
#> 418: 0.407167    1.046174    3.465920    0.386262    0.079840    0.027546    0.652513    
#> 419: 0.407166    1.046238    3.465952    0.386068    0.079834    0.027553    0.652418    
#> 420: 0.407208    1.046207    3.465979    0.386092    0.079810    0.027559    0.652279    
#> 421: 0.407206    1.046171    3.466004    0.385970    0.079802    0.027558    0.652229    
#> 422: 0.407269    1.046118    3.465992    0.386135    0.079778    0.027538    0.652225    
#> 423: 0.407345    1.046045    3.466022    0.385916    0.079759    0.027542    0.652227    
#> 424: 0.407313    1.046023    3.466013    0.385655    0.079690    0.027531    0.652252    
#> 425: 0.407317    1.046076    3.466011    0.385501    0.079674    0.027526    0.652290    
#> 426: 0.407396    1.046124    3.466026    0.385553    0.079688    0.027515    0.652220    
#> 427: 0.407384    1.046078    3.466059    0.385823    0.079684    0.027521    0.652175    
#> 428: 0.407215    1.046164    3.466033    0.385537    0.079726    0.027506    0.652159    
#> 429: 0.407155    1.046229    3.466023    0.385365    0.079679    0.027499    0.652124    
#> 430: 0.407085    1.046240    3.465991    0.385374    0.079688    0.027505    0.652124    
#> 431: 0.406993    1.046307    3.465917    0.385426    0.079669    0.027516    0.652075    
#> 432: 0.406690    1.046405    3.465858    0.385438    0.079649    0.027537    0.652066    
#> 433: 0.406483    1.046552    3.465797    0.385265    0.079685    0.027531    0.652068    
#> 434: 0.406388    1.046462    3.465770    0.385100    0.079686    0.027551    0.652011    
#> 435: 0.406405    1.046468    3.465755    0.384942    0.079631    0.027565    0.651987    
#> 436: 0.406493    1.046392    3.465833    0.384973    0.079548    0.027575    0.651986    
#> 437: 0.406614    1.046306    3.465861    0.384990    0.079507    0.027585    0.651902    
#> 438: 0.406522    1.046384    3.465862    0.384932    0.079480    0.027592    0.651821    
#> 439: 0.406379    1.046439    3.465833    0.384796    0.079471    0.027618    0.651719    
#> 440: 0.406349    1.046474    3.465857    0.384613    0.079399    0.027644    0.651786    
#> 441: 0.406533    1.046409    3.465885    0.384858    0.079372    0.027673    0.651681    
#> 442: 0.406577    1.046394    3.465924    0.384982    0.079338    0.027726    0.651573    
#> 443: 0.406470    1.046468    3.465848    0.385134    0.079331    0.027744    0.651532    
#> 444: 0.406204    1.046579    3.465722    0.385147    0.079323    0.027759    0.651475    
#> 445: 0.406000    1.046681    3.465622    0.385499    0.079296    0.027771    0.651477    
#> 446: 0.405834    1.046806    3.465537    0.385724    0.079274    0.027771    0.651464    
#> 447: 0.405988    1.046923    3.465532    0.385875    0.079262    0.027766    0.651432    
#> 448: 0.405927    1.046941    3.465508    0.385675    0.079243    0.027745    0.651432    
#> 449: 0.406049    1.047057    3.465587    0.385486    0.079294    0.027722    0.651478    
#> 450: 0.406006    1.047094    3.465585    0.385345    0.079361    0.027698    0.651550    
#> 451: 0.405948    1.047101    3.465547    0.385306    0.079466    0.027690    0.651603    
#> 452: 0.405850    1.047129    3.465504    0.385111    0.079519    0.027664    0.651624    
#> 453: 0.405675    1.047147    3.465461    0.384771    0.079483    0.027666    0.651617    
#> 454: 0.405429    1.047332    3.465381    0.384642    0.079461    0.027658    0.651595    
#> 455: 0.405404    1.047459    3.465312    0.384863    0.079473    0.027660    0.651553    
#> 456: 0.405224    1.047483    3.465225    0.385276    0.079502    0.027656    0.651588    
#> 457: 0.405216    1.047489    3.465177    0.385741    0.079500    0.027664    0.651577    
#> 458: 0.405105    1.047656    3.465145    0.386008    0.079501    0.027684    0.651580    
#> 459: 0.404859    1.047822    3.465119    0.385975    0.079586    0.027700    0.651632    
#> 460: 0.404689    1.047945    3.465021    0.386050    0.079581    0.027734    0.651595    
#> 461: 0.404551    1.048053    3.464976    0.386018    0.079573    0.027738    0.651525    
#> 462: 0.404309    1.048137    3.464928    0.385955    0.079567    0.027740    0.651477    
#> 463: 0.404342    1.048195    3.464950    0.385917    0.079523    0.027738    0.651464    
#> 464: 0.404417    1.048275    3.464949    0.385709    0.079554    0.027727    0.651389    
#> 465: 0.404439    1.048185    3.464958    0.385559    0.079609    0.027707    0.651330    
#> 466: 0.404332    1.048238    3.464877    0.385371    0.079590    0.027703    0.651302    
#> 467: 0.404036    1.048229    3.464819    0.385413    0.079570    0.027693    0.651235    
#> 468: 0.404093    1.048287    3.464824    0.385570    0.079592    0.027698    0.651125    
#> 469: 0.404128    1.048427    3.464807    0.385789    0.079640    0.027690    0.651085    
#> 470: 0.404030    1.048504    3.464770    0.385823    0.079651    0.027670    0.651084    
#> 471: 0.403912    1.048476    3.464750    0.385837    0.079669    0.027645    0.651060    
#> 472: 0.403921    1.048492    3.464726    0.385913    0.079676    0.027633    0.651105    
#> 473: 0.403786    1.048489    3.464689    0.385817    0.079683    0.027618    0.651094    
#> 474: 0.403626    1.048438    3.464644    0.385578    0.079690    0.027625    0.651148    
#> 475: 0.403655    1.048482    3.464639    0.385326    0.079685    0.027642    0.651108    
#> 476: 0.403659    1.048501    3.464597    0.385231    0.079704    0.027662    0.651123    
#> 477: 0.403594    1.048527    3.464597    0.385213    0.079695    0.027686    0.651063    
#> 478: 0.403535    1.048522    3.464608    0.385247    0.079748    0.027701    0.651098    
#> 479: 0.403692    1.048597    3.464606    0.385646    0.079756    0.027714    0.651106    
#> 480: 0.403718    1.048628    3.464593    0.385830    0.079744    0.027734    0.651095    
#> 481: 0.403737    1.048586    3.464559    0.385951    0.079717    0.027756    0.651040    
#> 482: 0.403795    1.048597    3.464517    0.386077    0.079738    0.027780    0.651041    
#> 483: 0.403654    1.048607    3.464489    0.386089    0.079724    0.027802    0.651010    
#> 484: 0.403598    1.048499    3.464480    0.386213    0.079690    0.027828    0.651008    
#> 485: 0.403475    1.048569    3.464435    0.386429    0.079704    0.027854    0.651061    
#> 486: 0.403386    1.048624    3.464424    0.386602    0.079665    0.027873    0.650993    
#> 487: 0.403541    1.048685    3.464440    0.386848    0.079630    0.027886    0.650971    
#> 488: 0.403484    1.048727    3.464428    0.387085    0.079635    0.027904    0.650890    
#> 489: 0.403668    1.048772    3.464384    0.387440    0.079684    0.027932    0.650877    
#> 490: 0.403704    1.048785    3.464394    0.387523    0.079676    0.027942    0.650846    
#> 491: 0.403677    1.048802    3.464413    0.387327    0.079655    0.027946    0.650789    
#> 492: 0.403661    1.048848    3.464401    0.387136    0.079647    0.027948    0.650861    
#> 493: 0.403603    1.048905    3.464440    0.386817    0.079649    0.027967    0.650943    
#> 494: 0.403662    1.048903    3.464440    0.386591    0.079597    0.027977    0.650945    
#> 495: 0.403735    1.048915    3.464416    0.386870    0.079560    0.027985    0.650933    
#> 496: 0.403791    1.048981    3.464378    0.386877    0.079555    0.027993    0.650948    
#> 497: 0.403948    1.048968    3.464410    0.386950    0.079554    0.027988    0.651020    
#> 498: 0.403981    1.048943    3.464407    0.386928    0.079552    0.028000    0.651025    
#> 499: 0.404032    1.048966    3.464380    0.386988    0.079580    0.028003    0.650954    
#> 500: 0.404091    1.048940    3.464356    0.387205    0.079551    0.028003    0.650911    
#> [====|====|=>-------------------------------------]  25%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.275091    0.969915    3.445814    0.392401    0.075363    0.016800    1.480434    
#> 002: 0.259159    0.984807    3.471664    0.372781    0.071595    0.015960    0.922085    
#> 003: 0.385651    0.943064    3.486683    0.354142    0.070912    0.015162    0.718339    
#> 004: 0.337815    0.954659    3.476391    0.336435    0.074633    0.014404    0.602210    
#> 005: 0.341461    0.961674    3.471336    0.319613    0.073469    0.013684    0.543858    
#> 006: 0.321872    0.968317    3.457272    0.303633    0.071191    0.013000    0.516287    
#> 007: 0.278478    0.972321    3.453355    0.288451    0.067631    0.012350    0.517084    
#> 008: 0.273059    0.972454    3.448448    0.274029    0.067271    0.011830    0.512715    
#> 009: 0.269358    0.966184    3.451756    0.260327    0.063908    0.014542    0.509554    
#> 010: 0.268282    0.981846    3.444442    0.247311    0.060713    0.014630    0.506675    
#> 011: 0.326877    0.980650    3.455037    0.249979    0.060123    0.017123    0.506276    
#> 012: 0.311190    0.974611    3.455448    0.260100    0.057906    0.016267    0.489641    
#> 013: 0.295999    0.974380    3.452335    0.247095    0.057398    0.015454    0.495175    
#> 014: 0.318921    0.980059    3.459633    0.268292    0.054529    0.018206    0.488786    
#> 015: 0.332856    0.981690    3.453369    0.278432    0.051802    0.017296    0.479989    
#> 016: 0.301610    0.998986    3.448184    0.268210    0.049212    0.016750    0.486597    
#> 017: 0.300686    1.000010    3.455122    0.254799    0.046751    0.018638    0.481284    
#> 018: 0.308885    0.997007    3.447534    0.242351    0.044414    0.019572    0.486493    
#> 019: 0.277839    1.000398    3.445679    0.257628    0.042193    0.018594    0.477489    
#> 020: 0.303536    0.997677    3.446855    0.254676    0.050069    0.017664    0.479308    
#> 021: 0.309072    0.989916    3.459577    0.241942    0.049548    0.016781    0.477378    
#> 022: 0.284994    0.979977    3.457414    0.261624    0.047325    0.017343    0.476711    
#> 023: 0.354918    0.984503    3.468320    0.290191    0.044959    0.018005    0.491027    
#> 024: 0.330350    0.992394    3.464545    0.282562    0.042711    0.017105    0.480371    
#> 025: 0.321754    0.981387    3.457304    0.273800    0.040576    0.018103    0.486254    
#> 026: 0.319439    0.997923    3.453563    0.273378    0.038915    0.017198    0.484027    
#> 027: 0.308482    0.996256    3.449329    0.259709    0.044452    0.017884    0.497530    
#> 028: 0.334922    0.980797    3.463900    0.253952    0.042229    0.019247    0.472775    
#> 029: 0.336316    0.982882    3.458347    0.243700    0.041984    0.018284    0.477532    
#> 030: 0.343178    0.987318    3.459132    0.253003    0.041910    0.017370    0.472784    
#> 031: 0.315809    0.973386    3.461257    0.240353    0.049467    0.016502    0.470681    
#> 032: 0.303088    0.974155    3.465484    0.228335    0.046993    0.015677    0.476754    
#> 033: 0.282158    0.978979    3.455199    0.255081    0.046547    0.015856    0.487037    
#> 034: 0.298815    0.991868    3.450123    0.286975    0.052735    0.015063    0.480176    
#> 035: 0.293395    0.977589    3.451277    0.272626    0.050099    0.015748    0.468519    
#> 036: 0.299446    0.984412    3.456489    0.258995    0.047594    0.018386    0.466770    
#> 037: 0.314179    0.984805    3.457104    0.264783    0.053084    0.018880    0.480575    
#> 038: 0.298714    0.986145    3.450922    0.263591    0.053078    0.019340    0.475383    
#> 039: 0.312225    0.988069    3.454015    0.283591    0.050424    0.019941    0.461344    
#> 040: 0.308138    0.981433    3.462406    0.271506    0.047903    0.020552    0.478778    
#> 041: 0.293833    0.969935    3.455023    0.257930    0.045508    0.025934    0.484562    
#> 042: 0.295800    0.977259    3.451402    0.245034    0.043232    0.024637    0.469553    
#> 043: 0.286108    1.001451    3.442982    0.252428    0.041553    0.023406    0.471966    
#> 044: 0.272427    1.002026    3.439693    0.245741    0.044116    0.022235    0.489167    
#> 045: 0.292787    1.010099    3.434128    0.296656    0.041910    0.021944    0.473947    
#> 046: 0.285743    1.003225    3.436350    0.306030    0.048652    0.022635    0.472002    
#> 047: 0.280676    0.997771    3.446006    0.290729    0.046220    0.021504    0.468660    
#> 048: 0.324186    0.991995    3.453462    0.331527    0.043909    0.023998    0.478135    
#> 049: 0.324610    0.991184    3.454917    0.314951    0.050262    0.022799    0.469799    
#> 050: 0.321160    0.969092    3.457050    0.301061    0.047749    0.021659    0.477143    
#> 051: 0.311270    0.973300    3.452240    0.286008    0.054652    0.020576    0.488961    
#> 052: 0.322155    0.965963    3.465430    0.271707    0.051919    0.019547    0.487091    
#> 053: 0.313902    0.964167    3.453033    0.276977    0.049323    0.018570    0.483609    
#> 054: 0.271526    0.980571    3.448312    0.263128    0.046857    0.018866    0.492248    
#> 055: 0.274700    0.983329    3.449518    0.249972    0.044514    0.018516    0.485209    
#> 056: 0.303989    0.979765    3.449443    0.266314    0.042288    0.019608    0.488746    
#> 057: 0.290707    0.977494    3.454714    0.266603    0.040264    0.019283    0.477869    
#> 058: 0.288063    0.986297    3.439058    0.284650    0.050363    0.018319    0.479786    
#> 059: 0.260434    0.986168    3.435477    0.270417    0.047844    0.018142    0.494415    
#> 060: 0.277729    0.974665    3.450337    0.271460    0.045452    0.017422    0.486132    
#> 061: 0.320308    0.982757    3.458269    0.291425    0.043180    0.019157    0.491554    
#> 062: 0.321298    0.980687    3.466095    0.278071    0.048520    0.021874    0.496719    
#> 063: 0.332558    0.962903    3.461886    0.287519    0.058127    0.020780    0.498217    
#> 064: 0.335040    0.961641    3.460377    0.273143    0.055221    0.019741    0.488823    
#> 065: 0.335047    0.981541    3.458806    0.263767    0.052459    0.018754    0.472730    
#> 066: 0.322527    0.988561    3.464164    0.250579    0.053214    0.017816    0.469772    
#> 067: 0.314643    0.971423    3.464049    0.238050    0.050553    0.018331    0.473155    
#> 068: 0.305036    0.970616    3.467357    0.226147    0.048025    0.017975    0.483194    
#> 069: 0.318502    0.969498    3.468929    0.223153    0.045624    0.021507    0.488061    
#> 070: 0.337938    0.976622    3.471519    0.236334    0.043343    0.020432    0.488102    
#> 071: 0.303398    0.977824    3.455056    0.233196    0.041984    0.021323    0.494015    
#> 072: 0.289795    0.972878    3.452652    0.245063    0.041758    0.025338    0.481541    
#> 073: 0.277950    0.979142    3.458995    0.244485    0.043349    0.024072    0.474817    
#> 074: 0.298470    0.975206    3.449107    0.232261    0.041181    0.022868    0.478732    
#> 075: 0.304966    0.983886    3.461221    0.258030    0.049853    0.023059    0.483707    
#> 076: 0.326901    1.002492    3.462906    0.281807    0.047361    0.024287    0.480217    
#> 077: 0.344115    0.980727    3.465105    0.281941    0.044993    0.023137    0.479713    
#> 078: 0.318621    0.978216    3.462366    0.285103    0.042743    0.021980    0.470113    
#> 079: 0.311428    0.980548    3.447979    0.308025    0.040606    0.020881    0.483432    
#> 080: 0.308660    0.988758    3.455470    0.294068    0.039042    0.022681    0.486053    
#> 081: 0.293751    0.987300    3.448569    0.306569    0.037760    0.023528    0.474054    
#> 082: 0.283099    0.985246    3.444740    0.291241    0.036100    0.024584    0.480491    
#> 083: 0.294593    1.002290    3.442582    0.303318    0.037120    0.023355    0.471807    
#> 084: 0.290200    1.002546    3.442707    0.288152    0.035896    0.023387    0.477879    
#> 085: 0.277950    0.989246    3.452997    0.273744    0.034836    0.022218    0.480956    
#> 086: 0.289238    1.004766    3.443602    0.262598    0.042109    0.021670    0.473967    
#> 087: 0.272830    1.004351    3.440605    0.249796    0.045997    0.020587    0.482331    
#> 088: 0.230739    1.008736    3.433493    0.271071    0.045425    0.019829    0.484278    
#> 089: 0.251623    1.015543    3.427050    0.257517    0.043583    0.020205    0.481667    
#> 090: 0.275866    1.004999    3.444875    0.278684    0.045374    0.021273    0.485683    
#> 091: 0.296840    0.990373    3.444630    0.300362    0.043195    0.021242    0.494726    
#> 092: 0.294439    0.989069    3.436253    0.285344    0.041036    0.021155    0.492475    
#> 093: 0.260005    0.990166    3.439390    0.271076    0.041015    0.020735    0.499557    
#> 094: 0.292703    1.000961    3.444306    0.259071    0.048814    0.019698    0.503183    
#> 095: 0.304929    0.998196    3.458213    0.263118    0.046374    0.018713    0.496245    
#> 096: 0.274214    0.984001    3.450471    0.289819    0.044055    0.018712    0.490345    
#> 097: 0.280812    1.010724    3.435254    0.275328    0.046501    0.018379    0.475553    
#> 098: 0.284277    1.014765    3.444282    0.261562    0.045037    0.019559    0.478623    
#> 099: 0.267655    0.987279    3.435019    0.248484    0.042785    0.020660    0.485252    
#> 100: 0.281977    0.999380    3.442106    0.244890    0.040645    0.020962    0.470994    
#> 101: 0.268943    0.990354    3.437840    0.234543    0.039662    0.020393    0.473094    
#> 102: 0.272731    0.993845    3.433665    0.243851    0.038441    0.020986    0.484694    
#> 103: 0.285071    0.989754    3.444892    0.247639    0.036943    0.022100    0.478736    
#> 104: 0.279761    0.990509    3.454715    0.245736    0.045134    0.020995    0.496793    
#> 105: 0.317864    0.980459    3.454699    0.236572    0.042877    0.019946    0.491210    
#> 106: 0.332717    0.976535    3.462567    0.268078    0.040734    0.019720    0.476402    
#> 107: 0.341035    0.977089    3.460116    0.261213    0.038697    0.019463    0.489182    
#> 108: 0.309207    0.996586    3.458496    0.248152    0.043029    0.019298    0.489118    
#> 109: 0.320278    0.984608    3.473691    0.275754    0.046279    0.020797    0.488631    
#> 110: 0.352225    0.961158    3.467968    0.262964    0.043965    0.023316    0.475131    
#> 111: 0.342069    0.967068    3.469147    0.251433    0.042408    0.022150    0.467443    
#> 112: 0.312731    0.967925    3.456491    0.238861    0.044837    0.021042    0.483517    
#> 113: 0.290619    0.983835    3.458469    0.229746    0.042595    0.019990    0.476083    
#> 114: 0.300654    0.984715    3.450871    0.245451    0.047728    0.020119    0.479261    
#> 115: 0.310539    0.970572    3.450758    0.269706    0.045342    0.023280    0.483528    
#> 116: 0.323504    0.979758    3.449982    0.289353    0.043075    0.022116    0.482967    
#> 117: 0.294096    0.976910    3.454974    0.274885    0.041326    0.021011    0.491774    
#> 118: 0.327615    0.964241    3.461005    0.261141    0.039259    0.019960    0.482737    
#> 119: 0.306598    0.978526    3.457472    0.252329    0.037296    0.020839    0.486184    
#> 120: 0.289914    0.969572    3.455947    0.258301    0.039140    0.021182    0.477321    
#> 121: 0.323524    0.995156    3.460420    0.245386    0.040491    0.020682    0.488012    
#> 122: 0.311032    0.981807    3.461139    0.243091    0.043802    0.019648    0.479438    
#> 123: 0.345663    1.000139    3.459246    0.280039    0.046354    0.018666    0.471259    
#> 124: 0.320898    1.012129    3.456668    0.276128    0.044036    0.019483    0.485850    
#> 125: 0.347584    0.994763    3.452390    0.270293    0.053873    0.018509    0.481329    
#> 126: 0.317973    1.000069    3.451023    0.276939    0.051180    0.018269    0.480875    
#> 127: 0.317889    0.989366    3.449208    0.300367    0.049496    0.017356    0.488804    
#> 128: 0.301650    0.998333    3.459234    0.285349    0.047532    0.018485    0.494297    
#> 129: 0.313001    0.987552    3.459689    0.271082    0.045250    0.017561    0.487953    
#> 130: 0.338683    0.992490    3.450107    0.274564    0.051136    0.016683    0.475396    
#> 131: 0.304376    0.984268    3.443463    0.275065    0.049657    0.018285    0.467464    
#> 132: 0.261181    1.002525    3.425238    0.261312    0.051915    0.017561    0.473854    
#> 133: 0.267433    1.011900    3.430058    0.264263    0.049319    0.017965    0.475375    
#> 134: 0.288951    0.986395    3.447818    0.251050    0.046853    0.021822    0.474975    
#> 135: 0.286983    0.978410    3.448819    0.238497    0.044510    0.021777    0.483975    
#> 136: 0.306401    0.959019    3.462610    0.238974    0.042285    0.024458    0.477226    
#> 137: 0.334793    0.958042    3.465211    0.247984    0.047455    0.023235    0.488162    
#> 138: 0.319642    0.971558    3.461899    0.260995    0.050564    0.022073    0.486562    
#> 139: 0.318245    0.980777    3.473409    0.286071    0.048035    0.024615    0.481401    
#> 140: 0.348138    0.965845    3.471121    0.293738    0.046878    0.024036    0.489593    
#> 141: 0.341719    0.957706    3.477005    0.279051    0.044534    0.023638    0.483530    
#> 142: 0.325933    0.951207    3.472900    0.265098    0.042308    0.024186    0.496341    
#> 143: 0.319241    0.971031    3.461546    0.257437    0.042623    0.022976    0.498594    
#> 144: 0.327760    0.980241    3.463132    0.260367    0.040492    0.021827    0.471621    
#> 145: 0.299611    0.982140    3.456212    0.252191    0.038468    0.020736    0.481813    
#> 146: 0.269098    0.994651    3.449923    0.263709    0.043058    0.020594    0.488915    
#> 147: 0.272133    0.990486    3.440106    0.253987    0.044990    0.020023    0.479619    
#> 148: 0.253118    1.006506    3.447122    0.256705    0.042740    0.019639    0.477746    
#> 149: 0.269153    0.994559    3.439466    0.268606    0.045549    0.018657    0.481405    
#> 150: 0.304142    0.989186    3.436603    0.306555    0.044840    0.017942    0.479765    
#> 151: 0.260541    1.001099    3.440336    0.291227    0.042598    0.019338    0.475149    
#> 152: 0.282942    1.012658    3.440550    0.269086    0.037174    0.022029    0.468197    
#> 153: 0.283758    1.009677    3.445116    0.261025    0.037560    0.024927    0.478782    
#> 154: 0.285354    1.000396    3.442830    0.272997    0.036552    0.021891    0.478168    
#> 155: 0.260519    0.987609    3.439504    0.251071    0.036411    0.022930    0.472497    
#> 156: 0.282144    0.985785    3.443793    0.260166    0.035328    0.024296    0.488126    
#> 157: 0.285306    0.977002    3.447654    0.247002    0.033250    0.023050    0.489652    
#> 158: 0.307816    0.997562    3.449146    0.255717    0.033025    0.023882    0.483023    
#> 159: 0.273340    1.000967    3.445992    0.270134    0.039279    0.025169    0.475386    
#> 160: 0.298726    0.980902    3.448823    0.284232    0.041582    0.026079    0.470274    
#> 161: 0.299253    0.980649    3.453477    0.270674    0.029469    0.030804    0.479984    
#> 162: 0.283946    0.988001    3.454510    0.290592    0.029282    0.033955    0.483990    
#> 163: 0.287052    0.986042    3.448442    0.279108    0.029655    0.029972    0.482896    
#> 164: 0.309921    0.995774    3.446977    0.312251    0.025203    0.034942    0.474854    
#> 165: 0.305624    0.984913    3.448521    0.297293    0.027103    0.031502    0.468428    
#> 166: 0.321759    0.996881    3.453914    0.273501    0.026230    0.028645    0.474522    
#> 167: 0.315294    1.002872    3.457410    0.268451    0.033585    0.030520    0.476184    
#> 168: 0.300017    1.003536    3.445712    0.287645    0.031737    0.027839    0.467294    
#> 169: 0.311885    1.025573    3.444292    0.282571    0.034270    0.029123    0.486429    
#> 170: 0.287306    1.015772    3.444832    0.259754    0.034504    0.027336    0.476062    
#> 171: 0.269795    1.009422    3.439691    0.263400    0.030274    0.025103    0.461602    
#> 172: 0.272097    1.009508    3.439269    0.274375    0.038344    0.023077    0.457850    
#> 173: 0.291285    1.014782    3.437289    0.296577    0.034305    0.025200    0.462519    
#> 174: 0.293339    1.010480    3.439352    0.300271    0.032489    0.023044    0.468542    
#> 175: 0.285112    1.000917    3.436429    0.285008    0.033718    0.024694    0.467438    
#> 176: 0.297563    1.010958    3.448727    0.288421    0.035209    0.021382    0.473728    
#> 177: 0.316221    0.993419    3.455951    0.281894    0.032334    0.024909    0.483951    
#> 178: 0.311193    0.991790    3.459772    0.279958    0.045571    0.024344    0.471033    
#> 179: 0.324367    1.012927    3.446652    0.298978    0.050919    0.018694    0.470660    
#> 180: 0.298138    0.993325    3.450090    0.290415    0.044383    0.018197    0.470058    
#> 181: 0.283888    0.988998    3.440586    0.285441    0.044372    0.017555    0.481871    
#> 182: 0.268436    0.996061    3.433200    0.266288    0.040333    0.018299    0.491721    
#> 183: 0.284042    1.013464    3.437952    0.276101    0.050369    0.019179    0.496218    
#> 184: 0.273097    0.997438    3.447496    0.262719    0.042771    0.021505    0.491655    
#> 185: 0.284140    1.008135    3.442654    0.281541    0.041747    0.026598    0.481155    
#> 186: 0.282219    1.010218    3.437154    0.292697    0.049258    0.024226    0.476787    
#> 187: 0.308513    0.998551    3.444085    0.263571    0.038072    0.023759    0.479091    
#> 188: 0.296515    0.997459    3.453344    0.274318    0.039709    0.023644    0.478819    
#> 189: 0.302201    1.003410    3.452372    0.263496    0.044392    0.021197    0.488890    
#> 190: 0.300548    1.006571    3.442294    0.266834    0.045677    0.020391    0.481128    
#> 191: 0.292874    1.009316    3.438722    0.291624    0.042205    0.018477    0.465582    
#> 192: 0.280430    1.013618    3.441323    0.251673    0.045311    0.016726    0.469829    
#> 193: 0.304686    1.009248    3.447771    0.259119    0.045151    0.017735    0.472570    
#> 194: 0.290284    0.989594    3.441217    0.256557    0.042606    0.016515    0.488338    
#> 195: 0.288508    0.999438    3.441993    0.283613    0.037378    0.020080    0.485681    
#> 196: 0.293241    0.989071    3.449179    0.271909    0.042761    0.024089    0.480530    
#> 197: 0.274696    0.997042    3.455890    0.243227    0.041224    0.025207    0.499204    
#> 198: 0.314821    0.981028    3.455216    0.253495    0.038258    0.024789    0.474779    
#> 199: 0.302558    0.980726    3.444142    0.279223    0.039099    0.021546    0.479872    
#> 200: 0.303923    0.981152    3.450186    0.257190    0.035156    0.021507    0.478573    
#> 201: 0.311755    0.982992    3.454746    0.257133    0.036442    0.022820    0.474172    
#> 202: 0.305099    0.986910    3.453099    0.267733    0.035720    0.023247    0.473342    
#> 203: 0.311970    0.985694    3.451686    0.277861    0.035625    0.023069    0.474023    
#> 204: 0.310856    0.983833    3.453559    0.273546    0.036422    0.023134    0.476950    
#> 205: 0.312882    0.985212    3.455538    0.269359    0.037315    0.023178    0.481181    
#> 206: 0.311809    0.989887    3.455688    0.265366    0.038369    0.023155    0.482678    
#> 207: 0.308872    0.990886    3.454101    0.263367    0.038110    0.023392    0.483384    
#> 208: 0.308443    0.990918    3.453240    0.262748    0.037954    0.023481    0.483303    
#> 209: 0.308720    0.989387    3.452623    0.261605    0.038052    0.023312    0.483659    
#> 210: 0.310687    0.988943    3.453132    0.261619    0.038360    0.023050    0.482488    
#> 211: 0.312638    0.988739    3.453850    0.262045    0.039175    0.022858    0.481338    
#> 212: 0.316159    0.986840    3.455227    0.265074    0.038853    0.022547    0.481332    
#> 213: 0.318655    0.984853    3.456276    0.267703    0.038474    0.022458    0.481370    
#> 214: 0.319900    0.983892    3.456706    0.267572    0.038582    0.022126    0.480929    
#> 215: 0.319819    0.983669    3.457546    0.267088    0.038906    0.022010    0.480424    
#> 216: 0.322915    0.982367    3.458567    0.267089    0.039068    0.021959    0.481297    
#> 217: 0.324007    0.982053    3.459392    0.266806    0.039287    0.021702    0.481875    
#> 218: 0.325191    0.981634    3.459743    0.264558    0.039466    0.021649    0.481801    
#> 219: 0.323796    0.981104    3.459346    0.262341    0.039168    0.021640    0.482132    
#> 220: 0.322290    0.980844    3.459118    0.261025    0.038863    0.021911    0.482520    
#> 221: 0.322420    0.981383    3.459064    0.259919    0.038870    0.022076    0.482516    
#> 222: 0.322259    0.981202    3.458897    0.259459    0.039006    0.022369    0.482336    
#> 223: 0.322799    0.981096    3.458759    0.260377    0.038929    0.022611    0.482238    
#> 224: 0.323892    0.981542    3.458665    0.260816    0.038706    0.022768    0.482385    
#> 225: 0.323271    0.980695    3.458608    0.261230    0.038431    0.023037    0.483087    
#> 226: 0.323141    0.979886    3.458669    0.261070    0.038416    0.023268    0.483326    
#> 227: 0.322908    0.979808    3.458674    0.260768    0.038355    0.023471    0.483752    
#> 228: 0.321925    0.980469    3.458344    0.260273    0.038142    0.023603    0.484140    
#> 229: 0.321459    0.980334    3.458133    0.260020    0.037889    0.023717    0.483857    
#> 230: 0.322186    0.980033    3.458272    0.260921    0.037747    0.023748    0.483994    
#> 231: 0.322428    0.980027    3.458438    0.260682    0.037607    0.023739    0.483756    
#> 232: 0.321769    0.980073    3.458481    0.260684    0.037524    0.023681    0.483304    
#> 233: 0.320954    0.979521    3.458480    0.260557    0.037405    0.023580    0.483307    
#> 234: 0.320930    0.979282    3.458572    0.259569    0.037411    0.023553    0.483437    
#> 235: 0.321104    0.979270    3.458733    0.258637    0.037323    0.023505    0.483272    
#> 236: 0.320753    0.979555    3.458646    0.258001    0.037429    0.023370    0.483380    
#> 237: 0.320511    0.979423    3.458698    0.257935    0.037514    0.023241    0.483875    
#> 238: 0.320529    0.979582    3.458819    0.257870    0.037532    0.023145    0.484267    
#> 239: 0.320324    0.979942    3.458687    0.258148    0.037631    0.023087    0.484155    
#> 240: 0.320010    0.979839    3.458690    0.258414    0.037701    0.023048    0.484058    
#> 241: 0.319773    0.979420    3.458681    0.258242    0.037882    0.022986    0.484060    
#> 242: 0.319639    0.979259    3.458490    0.258494    0.038011    0.022849    0.484225    
#> 243: 0.319227    0.979597    3.458350    0.258145    0.038122    0.022800    0.484478    
#> 244: 0.318610    0.979752    3.458503    0.257253    0.038147    0.022694    0.484467    
#> 245: 0.318656    0.979703    3.458825    0.257139    0.038337    0.022698    0.484694    
#> 246: 0.319256    0.979689    3.458836    0.256528    0.038376    0.022615    0.484878    
#> 247: 0.319190    0.979643    3.458896    0.256127    0.038457    0.022560    0.485037    
#> 248: 0.319102    0.979843    3.458737    0.255792    0.038428    0.022457    0.485139    
#> 249: 0.318762    0.979786    3.458816    0.255185    0.038477    0.022398    0.485369    
#> 250: 0.318465    0.979885    3.458848    0.255150    0.038679    0.022301    0.485602    
#> 251: 0.318589    0.979963    3.459021    0.255193    0.038782    0.022223    0.485717    
#> 252: 0.319277    0.980228    3.459242    0.255367    0.038867    0.022146    0.485687    
#> 253: 0.319340    0.980400    3.459252    0.254946    0.039011    0.022057    0.485529    
#> 254: 0.319113    0.980789    3.459107    0.254957    0.039087    0.021949    0.485328    
#> 255: 0.319350    0.981036    3.459220    0.254603    0.039115    0.021852    0.485391    
#> 256: 0.319327    0.980819    3.459230    0.254261    0.039202    0.021731    0.485586    
#> 257: 0.318861    0.980563    3.459232    0.253219    0.039239    0.021622    0.485845    
#> 258: 0.318491    0.980088    3.459278    0.252731    0.039292    0.021596    0.485945    
#> 259: 0.318314    0.979845    3.459312    0.252226    0.039318    0.021605    0.486006    
#> 260: 0.318418    0.979797    3.459418    0.252322    0.039340    0.021626    0.486062    
#> 261: 0.318723    0.979526    3.459681    0.252550    0.039319    0.021622    0.486305    
#> 262: 0.318848    0.979460    3.459734    0.252817    0.039360    0.021599    0.486145    
#> 263: 0.319070    0.979500    3.459773    0.253119    0.039454    0.021587    0.486120    
#> 264: 0.319096    0.979822    3.459703    0.252903    0.039621    0.021508    0.486028    
#> 265: 0.318945    0.980033    3.459675    0.252692    0.039855    0.021448    0.486156    
#> 266: 0.318514    0.980207    3.459548    0.252745    0.039935    0.021378    0.486071    
#> 267: 0.318151    0.980252    3.459629    0.252693    0.039947    0.021322    0.486102    
#> 268: 0.318370    0.980055    3.459626    0.252846    0.039913    0.021245    0.486061    
#> 269: 0.318728    0.979943    3.459643    0.252864    0.039903    0.021190    0.486008    
#> 270: 0.318791    0.980010    3.459580    0.252418    0.039926    0.021120    0.485958    
#> 271: 0.318646    0.979961    3.459543    0.252566    0.040012    0.021050    0.486012    
#> 272: 0.318848    0.979730    3.459527    0.252096    0.040185    0.021019    0.486319    
#> 273: 0.318711    0.979853    3.459502    0.252172    0.040349    0.020994    0.486336    
#> 274: 0.319060    0.980165    3.459553    0.252156    0.040511    0.020958    0.486390    
#> 275: 0.319183    0.980268    3.459541    0.251971    0.040590    0.020925    0.486283    
#> 276: 0.318966    0.980485    3.459493    0.251810    0.040701    0.020905    0.486106    
#> 277: 0.318664    0.980638    3.459507    0.251461    0.040703    0.020911    0.486285    
#> 278: 0.318725    0.980629    3.459620    0.251505    0.040703    0.020869    0.486340    
#> 279: 0.318580    0.980436    3.459642    0.252104    0.040713    0.020849    0.486582    
#> 280: 0.318278    0.980589    3.459543    0.252499    0.040730    0.020855    0.486587    
#> 281: 0.317988    0.980941    3.459447    0.253022    0.040835    0.020842    0.486355    
#> 282: 0.317759    0.981210    3.459435    0.253312    0.040989    0.020810    0.486407    
#> 283: 0.317677    0.981213    3.459407    0.253521    0.041090    0.020745    0.486599    
#> 284: 0.317533    0.981159    3.459361    0.253326    0.041098    0.020673    0.486636    
#> 285: 0.317177    0.981186    3.459202    0.252869    0.041244    0.020588    0.486759    
#> 286: 0.316735    0.981418    3.458949    0.252711    0.041372    0.020541    0.486728    
#> 287: 0.316050    0.981683    3.458617    0.252461    0.041497    0.020522    0.486722    
#> 288: 0.316130    0.981851    3.458511    0.252848    0.041633    0.020511    0.486833    
#> 289: 0.316097    0.981916    3.458458    0.253267    0.041694    0.020530    0.486800    
#> 290: 0.316101    0.982040    3.458380    0.253300    0.041678    0.020505    0.486920    
#> 291: 0.315863    0.982224    3.458372    0.253351    0.041608    0.020480    0.486769    
#> 292: 0.315671    0.982296    3.458306    0.253293    0.041549    0.020456    0.486747    
#> 293: 0.315441    0.982330    3.458181    0.253248    0.041568    0.020455    0.486670    
#> 294: 0.315043    0.982450    3.457959    0.253587    0.041609    0.020450    0.486612    
#> 295: 0.314635    0.982348    3.457805    0.253838    0.041569    0.020497    0.486525    
#> 296: 0.314326    0.982455    3.457661    0.254167    0.041494    0.020573    0.486408    
#> 297: 0.314138    0.982685    3.457569    0.254460    0.041418    0.020630    0.486376    
#> 298: 0.314013    0.982768    3.457398    0.254743    0.041370    0.020649    0.486265    
#> 299: 0.313938    0.982948    3.457233    0.255024    0.041360    0.020641    0.486156    
#> 300: 0.314013    0.982901    3.457210    0.255277    0.041346    0.020653    0.486204    
#> 301: 0.313848    0.982748    3.457072    0.255515    0.041373    0.020670    0.486270    
#> 302: 0.313766    0.982784    3.456950    0.255756    0.041406    0.020685    0.486192    
#> 303: 0.313427    0.982897    3.456834    0.255911    0.041458    0.020693    0.486114    
#> 304: 0.313106    0.983076    3.456714    0.256150    0.041454    0.020717    0.486176    
#> 305: 0.313081    0.983024    3.456624    0.256203    0.041501    0.020736    0.486121    
#> 306: 0.312649    0.983109    3.456428    0.256190    0.041522    0.020718    0.486003    
#> 307: 0.312131    0.983249    3.456244    0.256223    0.041570    0.020680    0.485983    
#> 308: 0.311624    0.983310    3.456106    0.256303    0.041623    0.020683    0.485967    
#> 309: 0.311440    0.983144    3.456061    0.256558    0.041657    0.020689    0.486148    
#> 310: 0.311475    0.983088    3.455996    0.256851    0.041666    0.020714    0.486330    
#> 311: 0.311185    0.983117    3.455902    0.257589    0.041684    0.020715    0.486275    
#> 312: 0.310875    0.983181    3.455750    0.258222    0.041734    0.020698    0.486337    
#> 313: 0.310594    0.983059    3.455592    0.258503    0.041746    0.020687    0.486363    
#> 314: 0.310239    0.983028    3.455484    0.258905    0.041798    0.020694    0.486287    
#> 315: 0.310099    0.983139    3.455353    0.259369    0.041872    0.020704    0.486220    
#> 316: 0.310076    0.983369    3.455241    0.259938    0.041861    0.020726    0.486185    
#> 317: 0.310274    0.983372    3.455285    0.260041    0.041821    0.020724    0.486305    
#> 318: 0.310239    0.983374    3.455242    0.260136    0.041784    0.020719    0.486250    
#> 319: 0.310151    0.983337    3.455230    0.260070    0.041752    0.020709    0.486297    
#> 320: 0.309954    0.983330    3.455177    0.260145    0.041692    0.020682    0.486326    
#> 321: 0.309832    0.983262    3.455143    0.259930    0.041675    0.020651    0.486286    
#> 322: 0.309832    0.983209    3.455117    0.259833    0.041668    0.020640    0.486210    
#> 323: 0.309665    0.983195    3.455106    0.259728    0.041699    0.020621    0.486086    
#> 324: 0.309608    0.983303    3.455083    0.259658    0.041770    0.020600    0.485963    
#> 325: 0.309464    0.983530    3.454944    0.259757    0.041821    0.020579    0.485907    
#> 326: 0.309256    0.983541    3.454823    0.259699    0.041863    0.020567    0.485883    
#> 327: 0.309182    0.983538    3.454788    0.259379    0.041924    0.020540    0.485954    
#> 328: 0.309063    0.983644    3.454712    0.259152    0.041906    0.020521    0.485931    
#> 329: 0.309099    0.983727    3.454782    0.258975    0.041871    0.020485    0.485922    
#> 330: 0.308993    0.983699    3.454817    0.258844    0.041817    0.020442    0.485937    
#> 331: 0.309047    0.983741    3.454936    0.258578    0.041837    0.020426    0.485937    
#> 332: 0.309258    0.983593    3.455094    0.258473    0.041826    0.020412    0.486063    
#> 333: 0.309315    0.983507    3.455098    0.258302    0.041821    0.020405    0.486090    
#> 334: 0.309307    0.983367    3.455068    0.258246    0.041848    0.020383    0.486217    
#> 335: 0.309185    0.983251    3.455060    0.258151    0.041860    0.020350    0.486243    
#> 336: 0.308988    0.983329    3.455037    0.258091    0.041939    0.020324    0.486236    
#> 337: 0.308816    0.983542    3.454937    0.257902    0.042024    0.020294    0.486291    
#> 338: 0.308660    0.983664    3.454861    0.257807    0.042092    0.020258    0.486289    
#> 339: 0.308466    0.983760    3.454789    0.257515    0.042127    0.020236    0.486313    
#> 340: 0.308467    0.983606    3.454785    0.257477    0.042152    0.020223    0.486299    
#> 341: 0.308430    0.983587    3.454803    0.257413    0.042171    0.020208    0.486216    
#> 342: 0.308329    0.983455    3.454821    0.257379    0.042213    0.020202    0.486107    
#> 343: 0.308174    0.983454    3.454822    0.257362    0.042233    0.020196    0.486004    
#> 344: 0.308345    0.983462    3.454864    0.257546    0.042228    0.020185    0.485970    
#> 345: 0.308406    0.983513    3.454869    0.257421    0.042179    0.020174    0.485924    
#> 346: 0.308289    0.983602    3.454792    0.257369    0.042164    0.020135    0.485864    
#> 347: 0.308264    0.983566    3.454764    0.257446    0.042182    0.020117    0.485822    
#> 348: 0.308101    0.983606    3.454707    0.257555    0.042216    0.020093    0.485839    
#> 349: 0.307820    0.983658    3.454613    0.257646    0.042237    0.020076    0.485764    
#> 350: 0.307689    0.983725    3.454551    0.257660    0.042246    0.020063    0.485711    
#> 351: 0.307820    0.983881    3.454513    0.257704    0.042226    0.020047    0.485797    
#> 352: 0.307581    0.984056    3.454482    0.257540    0.042230    0.020051    0.486012    
#> 353: 0.307488    0.984278    3.454416    0.257405    0.042221    0.020046    0.486116    
#> 354: 0.307374    0.984358    3.454338    0.257237    0.042191    0.020037    0.486102    
#> 355: 0.307441    0.984319    3.454279    0.257266    0.042206    0.020038    0.486115    
#> 356: 0.307409    0.984301    3.454250    0.257158    0.042195    0.020046    0.486181    
#> 357: 0.307324    0.984351    3.454260    0.256973    0.042190    0.020054    0.486135    
#> 358: 0.307114    0.984424    3.454218    0.256751    0.042190    0.020036    0.486061    
#> 359: 0.306838    0.984645    3.454076    0.256579    0.042164    0.020017    0.486081    
#> 360: 0.306655    0.984762    3.453946    0.256489    0.042137    0.020005    0.486024    
#> 361: 0.306445    0.984854    3.453869    0.256245    0.042185    0.019984    0.485954    
#> 362: 0.306302    0.984873    3.453825    0.256005    0.042191    0.019972    0.485829    
#> 363: 0.306275    0.984886    3.453814    0.256088    0.042177    0.019950    0.485711    
#> 364: 0.306369    0.984889    3.453793    0.256212    0.042177    0.019945    0.485703    
#> 365: 0.306129    0.984912    3.453726    0.256278    0.042153    0.019940    0.485698    
#> 366: 0.305970    0.985009    3.453632    0.256207    0.042145    0.019939    0.485686    
#> 367: 0.305905    0.985131    3.453574    0.256104    0.042154    0.019943    0.485629    
#> 368: 0.306020    0.985227    3.453525    0.256140    0.042133    0.019953    0.485603    
#> 369: 0.306068    0.985246    3.453518    0.256367    0.042098    0.019965    0.485655    
#> 370: 0.305990    0.985353    3.453494    0.256334    0.042044    0.019990    0.485643    
#> 371: 0.305895    0.985483    3.453435    0.256243    0.042031    0.019996    0.485573    
#> 372: 0.305845    0.985533    3.453411    0.256134    0.042020    0.020002    0.485527    
#> 373: 0.305664    0.985585    3.453378    0.256126    0.041969    0.019998    0.485557    
#> 374: 0.305703    0.985625    3.453312    0.256226    0.041942    0.020005    0.485559    
#> 375: 0.305650    0.985698    3.453290    0.256472    0.041912    0.020034    0.485561    
#> 376: 0.305627    0.985799    3.453239    0.256608    0.041893    0.020057    0.485491    
#> 377: 0.305634    0.985847    3.453245    0.256567    0.041845    0.020059    0.485516    
#> 378: 0.305601    0.985861    3.453235    0.256369    0.041778    0.020068    0.485552    
#> 379: 0.305658    0.985776    3.453289    0.256249    0.041759    0.020071    0.485451    
#> 380: 0.305634    0.985683    3.453284    0.256096    0.041714    0.020071    0.485465    
#> 381: 0.305495    0.985635    3.453262    0.256033    0.041692    0.020076    0.485466    
#> 382: 0.305426    0.985653    3.453239    0.256022    0.041652    0.020074    0.485410    
#> 383: 0.305349    0.985706    3.453180    0.256122    0.041643    0.020079    0.485329    
#> 384: 0.305216    0.985691    3.453203    0.256129    0.041608    0.020085    0.485262    
#> 385: 0.305291    0.985683    3.453217    0.256036    0.041597    0.020098    0.485159    
#> 386: 0.305372    0.985836    3.453195    0.256115    0.041580    0.020115    0.485147    
#> 387: 0.305451    0.985919    3.453248    0.256155    0.041560    0.020133    0.485130    
#> 388: 0.305549    0.985912    3.453312    0.256346    0.041536    0.020155    0.485070    
#> 389: 0.305535    0.985820    3.453317    0.256462    0.041508    0.020180    0.485128    
#> 390: 0.305424    0.985887    3.453260    0.256625    0.041494    0.020176    0.485108    
#> 391: 0.305535    0.985897    3.453229    0.256683    0.041471    0.020172    0.485065    
#> 392: 0.305577    0.985875    3.453245    0.256860    0.041410    0.020169    0.485078    
#> 393: 0.305533    0.985795    3.453212    0.257049    0.041370    0.020182    0.485019    
#> 394: 0.305403    0.985819    3.453187    0.257188    0.041324    0.020200    0.485031    
#> 395: 0.305477    0.985862    3.453168    0.257262    0.041319    0.020236    0.485008    
#> 396: 0.305443    0.985879    3.453140    0.257369    0.041305    0.020248    0.484930    
#> 397: 0.305312    0.985876    3.453093    0.257431    0.041265    0.020254    0.484922    
#> 398: 0.305346    0.985867    3.453089    0.257422    0.041256    0.020263    0.484858    
#> 399: 0.305437    0.985878    3.453149    0.257515    0.041241    0.020270    0.484838    
#> 400: 0.305528    0.985911    3.453163    0.257750    0.041228    0.020266    0.484765    
#> 401: 0.305511    0.985998    3.453127    0.257838    0.041207    0.020254    0.484786    
#> 402: 0.305403    0.986081    3.453094    0.257904    0.041194    0.020248    0.484814    
#> 403: 0.305543    0.986132    3.453113    0.257947    0.041185    0.020244    0.484744    
#> 404: 0.305582    0.986171    3.453067    0.258005    0.041185    0.020235    0.484655    
#> 405: 0.305470    0.986235    3.453043    0.257906    0.041192    0.020232    0.484611    
#> 406: 0.305331    0.986286    3.453001    0.257916    0.041170    0.020228    0.484639    
#> 407: 0.305250    0.986346    3.452963    0.257963    0.041182    0.020224    0.484669    
#> 408: 0.305339    0.986427    3.452934    0.258085    0.041214    0.020214    0.484699    
#> 409: 0.305297    0.986400    3.452929    0.258190    0.041204    0.020215    0.484678    
#> 410: 0.305227    0.986442    3.452892    0.258385    0.041176    0.020234    0.484630    
#> 411: 0.305275    0.986379    3.452916    0.258558    0.041190    0.020242    0.484625    
#> 412: 0.305290    0.986412    3.452910    0.258740    0.041166    0.020255    0.484600    
#> 413: 0.305437    0.986398    3.452941    0.258914    0.041130    0.020274    0.484586    
#> 414: 0.305365    0.986392    3.452953    0.259052    0.041108    0.020284    0.484571    
#> 415: 0.305513    0.986352    3.452959    0.259320    0.041090    0.020302    0.484498    
#> 416: 0.305587    0.986390    3.452993    0.259502    0.041085    0.020311    0.484473    
#> 417: 0.305578    0.986335    3.453008    0.259631    0.041089    0.020327    0.484436    
#> 418: 0.305481    0.986290    3.453048    0.259597    0.041101    0.020326    0.484409    
#> 419: 0.305583    0.986293    3.453076    0.259630    0.041131    0.020333    0.484350    
#> 420: 0.305708    0.986284    3.453131    0.259583    0.041171    0.020332    0.484276    
#> 421: 0.305749    0.986273    3.453209    0.259464    0.041184    0.020341    0.484236    
#> 422: 0.305959    0.986219    3.453281    0.259443    0.041184    0.020350    0.484204    
#> 423: 0.306081    0.986119    3.453326    0.259400    0.041174    0.020359    0.484235    
#> 424: 0.306126    0.986045    3.453324    0.259264    0.041180    0.020344    0.484205    
#> 425: 0.306045    0.986091    3.453291    0.259214    0.041223    0.020337    0.484288    
#> 426: 0.305927    0.986233    3.453215    0.259267    0.041257    0.020336    0.484253    
#> 427: 0.305729    0.986295    3.453177    0.259283    0.041285    0.020325    0.484223    
#> 428: 0.305566    0.986408    3.453140    0.259218    0.041314    0.020314    0.484201    
#> 429: 0.305540    0.986474    3.453155    0.259236    0.041322    0.020316    0.484178    
#> 430: 0.305580    0.986483    3.453147    0.259241    0.041323    0.020318    0.484186    
#> 431: 0.305649    0.986480    3.453153    0.259379    0.041349    0.020309    0.484094    
#> 432: 0.305664    0.986509    3.453163    0.259423    0.041396    0.020298    0.484031    
#> 433: 0.305615    0.986567    3.453149    0.259510    0.041438    0.020281    0.484057    
#> 434: 0.305561    0.986505    3.453153    0.259453    0.041444    0.020279    0.484068    
#> 435: 0.305563    0.986463    3.453148    0.259463    0.041430    0.020278    0.483999    
#> 436: 0.305702    0.986433    3.453232    0.259488    0.041460    0.020287    0.483988    
#> 437: 0.305878    0.986401    3.453276    0.259387    0.041473    0.020277    0.483977    
#> 438: 0.305911    0.986471    3.453280    0.259195    0.041483    0.020285    0.483910    
#> 439: 0.305937    0.986490    3.453311    0.259000    0.041515    0.020280    0.483828    
#> 440: 0.306010    0.986490    3.453381    0.258846    0.041512    0.020284    0.483802    
#> 441: 0.306077    0.986461    3.453413    0.258814    0.041504    0.020294    0.483775    
#> 442: 0.306046    0.986457    3.453409    0.258789    0.041511    0.020298    0.483743    
#> 443: 0.306019    0.986533    3.453362    0.258813    0.041513    0.020309    0.483748    
#> 444: 0.305900    0.986621    3.453279    0.258924    0.041548    0.020307    0.483790    
#> 445: 0.305734    0.986713    3.453211    0.258963    0.041552    0.020304    0.483816    
#> 446: 0.305576    0.986799    3.453144    0.258916    0.041554    0.020309    0.483795    
#> 447: 0.305532    0.986870    3.453098    0.259007    0.041556    0.020323    0.483799    
#> 448: 0.305502    0.986872    3.453060    0.259058    0.041550    0.020341    0.483731    
#> 449: 0.305584    0.986924    3.453112    0.259055    0.041538    0.020364    0.483739    
#> 450: 0.305601    0.986915    3.453113    0.259146    0.041529    0.020390    0.483736    
#> 451: 0.305481    0.986927    3.453076    0.259174    0.041511    0.020415    0.483782    
#> 452: 0.305383    0.986953    3.453048    0.259279    0.041494    0.020431    0.483786    
#> 453: 0.305180    0.987008    3.452988    0.259206    0.041473    0.020434    0.483744    
#> 454: 0.305090    0.987124    3.452952    0.259277    0.041473    0.020442    0.483705    
#> 455: 0.305002    0.987224    3.452894    0.259311    0.041451    0.020440    0.483664    
#> 456: 0.304855    0.987246    3.452857    0.259322    0.041445    0.020445    0.483667    
#> 457: 0.304778    0.987266    3.452865    0.259322    0.041440    0.020456    0.483612    
#> 458: 0.304711    0.987376    3.452840    0.259314    0.041426    0.020462    0.483585    
#> 459: 0.304602    0.987492    3.452816    0.259424    0.041391    0.020482    0.483602    
#> 460: 0.304454    0.987544    3.452740    0.259550    0.041358    0.020504    0.483591    
#> 461: 0.304366    0.987587    3.452697    0.259582    0.041330    0.020519    0.483555    
#> 462: 0.304239    0.987629    3.452678    0.259496    0.041307    0.020538    0.483480    
#> 463: 0.304228    0.987635    3.452708    0.259367    0.041294    0.020554    0.483468    
#> 464: 0.304238    0.987649    3.452698    0.259243    0.041285    0.020564    0.483443    
#> 465: 0.304233    0.987574    3.452715    0.259187    0.041266    0.020567    0.483415    
#> 466: 0.304260    0.987583    3.452718    0.259176    0.041247    0.020580    0.483348    
#> 467: 0.304236    0.987525    3.452752    0.259144    0.041226    0.020595    0.483271    
#> 468: 0.304364    0.987532    3.452804    0.259270    0.041199    0.020624    0.483217    
#> 469: 0.304544    0.987611    3.452828    0.259374    0.041214    0.020637    0.483248    
#> 470: 0.304411    0.987676    3.452780    0.259417    0.041211    0.020644    0.483205    
#> 471: 0.304305    0.987683    3.452757    0.259418    0.041216    0.020636    0.483206    
#> 472: 0.304219    0.987694    3.452721    0.259440    0.041241    0.020626    0.483192    
#> 473: 0.304114    0.987710    3.452707    0.259397    0.041231    0.020630    0.483176    
#> 474: 0.304071    0.987741    3.452675    0.259295    0.041214    0.020637    0.483242    
#> 475: 0.304031    0.987797    3.452657    0.259280    0.041197    0.020637    0.483216    
#> 476: 0.303957    0.987804    3.452634    0.259416    0.041180    0.020641    0.483166    
#> 477: 0.303885    0.987841    3.452636    0.259520    0.041165    0.020642    0.483119    
#> 478: 0.303780    0.987817    3.452642    0.259596    0.041141    0.020644    0.483117    
#> 479: 0.303819    0.987852    3.452625    0.259767    0.041126    0.020648    0.483124    
#> 480: 0.303770    0.987874    3.452618    0.259804    0.041136    0.020637    0.483182    
#> 481: 0.303729    0.987893    3.452589    0.259802    0.041135    0.020634    0.483173    
#> 482: 0.303709    0.987903    3.452540    0.259855    0.041139    0.020642    0.483162    
#> 483: 0.303587    0.987893    3.452513    0.259931    0.041156    0.020635    0.483129    
#> 484: 0.303586    0.987830    3.452517    0.260034    0.041142    0.020644    0.483093    
#> 485: 0.303541    0.987851    3.452515    0.260084    0.041142    0.020655    0.483141    
#> 486: 0.303504    0.987907    3.452522    0.260063    0.041141    0.020663    0.483138    
#> 487: 0.303571    0.987930    3.452555    0.260165    0.041161    0.020662    0.483157    
#> 488: 0.303643    0.987903    3.452589    0.260269    0.041171    0.020661    0.483102    
#> 489: 0.303771    0.987840    3.452604    0.260306    0.041165    0.020662    0.483068    
#> 490: 0.303830    0.987870    3.452631    0.260298    0.041176    0.020664    0.483034    
#> 491: 0.303860    0.987904    3.452662    0.260249    0.041187    0.020663    0.482985    
#> 492: 0.303909    0.987949    3.452661    0.260182    0.041199    0.020659    0.483002    
#> 493: 0.303869    0.987992    3.452659    0.260163    0.041200    0.020674    0.482978    
#> 494: 0.303917    0.988030    3.452653    0.260232    0.041203    0.020678    0.482928    
#> 495: 0.303927    0.988040    3.452620    0.260271    0.041196    0.020670    0.482863    
#> 496: 0.303949    0.988114    3.452605    0.260212    0.041188    0.020674    0.482814    
#> 497: 0.303992    0.988110    3.452618    0.260227    0.041152    0.020675    0.482867    
#> 498: 0.303946    0.988103    3.452595    0.260219    0.041141    0.020679    0.482831    
#> 499: 0.303923    0.988102    3.452566    0.260226    0.041135    0.020677    0.482773    
#> 500: 0.303961    0.988039    3.452572    0.260379    0.041117    0.020678    0.482768    
#> [====|====|=>-------------------------------------]  25%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.308815    0.949402    3.443836    0.392401    0.089004    0.016800    1.708241    
#> 002: 0.332597    0.860670    3.464899    0.372781    0.084553    0.016700    1.066632    
#> 003: 0.347775    0.832892    3.465574    0.354142    0.080326    0.015865    0.895650    
#> 004: 0.291879    0.854988    3.440033    0.336435    0.076310    0.015072    0.825350    
#> 005: 0.310746    0.862287    3.437609    0.319613    0.072494    0.014318    0.792627    
#> 006: 0.294521    0.834829    3.406598    0.303633    0.072438    0.014779    0.720285    
#> 007: 0.234573    0.842986    3.403402    0.288451    0.068816    0.015307    0.715406    
#> 008: 0.224932    0.875330    3.386736    0.339184    0.065376    0.017382    0.683747    
#> 009: 0.185961    0.877687    3.370594    0.361582    0.065028    0.018304    0.657047    
#> 010: 0.168657    0.895193    3.367149    0.345371    0.061777    0.017389    0.672286    
#> 011: 0.227301    0.903980    3.371239    0.384254    0.068282    0.016519    0.669193    
#> 012: 0.207768    0.906947    3.358914    0.371282    0.064868    0.016855    0.646798    
#> 013: 0.158443    0.911776    3.352421    0.352718    0.071546    0.018192    0.667988    
#> 014: 0.227156    0.897515    3.369997    0.340756    0.090867    0.017283    0.673669    
#> 015: 0.240734    0.895015    3.383605    0.347954    0.086324    0.016418    0.652830    
#> 016: 0.235641    0.889584    3.380512    0.343929    0.083529    0.020042    0.654374    
#> 017: 0.227256    0.909466    3.371749    0.371884    0.086466    0.019040    0.642144    
#> 018: 0.197469    0.917725    3.366419    0.365939    0.086692    0.018088    0.643443    
#> 019: 0.149347    0.929266    3.351979    0.347642    0.082357    0.017184    0.639226    
#> 020: 0.193227    0.930404    3.362840    0.330260    0.078239    0.016990    0.649959    
#> 021: 0.208370    0.907869    3.366204    0.334877    0.074327    0.016141    0.639326    
#> 022: 0.222177    0.900692    3.386592    0.339507    0.070611    0.018399    0.641699    
#> 023: 0.308336    0.911635    3.393086    0.375109    0.072862    0.018400    0.667477    
#> 024: 0.277985    0.916997    3.390242    0.410794    0.069219    0.018984    0.654833    
#> 025: 0.277822    0.896436    3.386100    0.390254    0.065758    0.019800    0.643238    
#> 026: 0.207368    0.921223    3.364088    0.370741    0.062470    0.022462    0.656431    
#> 027: 0.237715    0.927684    3.361622    0.387482    0.061986    0.021339    0.664534    
#> 028: 0.236221    0.933028    3.372629    0.368108    0.058952    0.025176    0.648646    
#> 029: 0.236283    0.934752    3.362552    0.474388    0.065816    0.024830    0.637162    
#> 030: 0.199655    0.925177    3.359283    0.450668    0.062526    0.024855    0.640344    
#> 031: 0.177525    0.913962    3.360111    0.489603    0.064897    0.024939    0.652615    
#> 032: 0.157025    0.915270    3.365930    0.465123    0.061652    0.025947    0.655613    
#> 033: 0.160286    0.938215    3.354183    0.441867    0.058569    0.031415    0.628030    
#> 034: 0.177948    0.939061    3.360342    0.419773    0.055641    0.029844    0.643090    
#> 035: 0.196782    0.920103    3.361141    0.398785    0.052901    0.028352    0.631650    
#> 036: 0.194997    0.931484    3.355439    0.429447    0.061967    0.027820    0.624405    
#> 037: 0.198054    0.937916    3.357056    0.409439    0.058868    0.026429    0.650812    
#> 038: 0.205034    0.933148    3.350025    0.455381    0.055925    0.026140    0.645540    
#> 039: 0.175917    0.927560    3.347478    0.478876    0.063323    0.025096    0.644078    
#> 040: 0.192055    0.941553    3.349054    0.454932    0.061116    0.023841    0.656831    
#> 041: 0.190491    0.907508    3.355446    0.438717    0.066147    0.022833    0.649474    
#> 042: 0.205822    0.922137    3.355204    0.416781    0.062914    0.022208    0.639364    
#> 043: 0.181594    0.917712    3.346722    0.395942    0.059769    0.021695    0.636419    
#> 044: 0.166043    0.918402    3.356371    0.376145    0.063766    0.020610    0.637780    
#> 045: 0.176875    0.932636    3.351839    0.369463    0.060577    0.020439    0.645567    
#> 046: 0.189745    0.897530    3.359988    0.404063    0.057548    0.023044    0.648177    
#> 047: 0.170181    0.901423    3.354281    0.383860    0.054671    0.027517    0.643771    
#> 048: 0.184027    0.913325    3.356779    0.378651    0.052387    0.026142    0.645471    
#> 049: 0.170882    0.910401    3.360780    0.381192    0.061937    0.025432    0.646986    
#> 050: 0.214033    0.900246    3.367232    0.362132    0.059226    0.024160    0.667755    
#> 051: 0.199449    0.916848    3.368610    0.354616    0.061546    0.023899    0.650920    
#> 052: 0.216622    0.908294    3.372276    0.433905    0.058468    0.024479    0.633350    
#> 053: 0.214894    0.897413    3.367888    0.437633    0.068725    0.023255    0.634167    
#> 054: 0.203542    0.901849    3.368446    0.415751    0.065948    0.023726    0.663936    
#> 055: 0.224327    0.886283    3.384441    0.394964    0.062651    0.022539    0.667622    
#> 056: 0.238375    0.877577    3.385671    0.387820    0.059522    0.021412    0.660005    
#> 057: 0.208753    0.897246    3.381051    0.390079    0.056546    0.023084    0.670258    
#> 058: 0.214317    0.907915    3.374634    0.370575    0.063174    0.022992    0.666788    
#> 059: 0.220750    0.870268    3.380715    0.406912    0.060015    0.025622    0.669784    
#> 060: 0.196881    0.869414    3.378992    0.386566    0.057015    0.024341    0.658645    
#> 061: 0.231825    0.899671    3.378570    0.395343    0.055799    0.023124    0.674893    
#> 062: 0.221174    0.919215    3.376270    0.375576    0.055130    0.023700    0.688598    
#> 063: 0.275295    0.892824    3.381906    0.410118    0.055231    0.026441    0.676491    
#> 064: 0.244003    0.889287    3.390817    0.389612    0.055917    0.025119    0.684633    
#> 065: 0.250319    0.884592    3.366503    0.370132    0.053121    0.023863    0.664183    
#> 066: 0.220336    0.909608    3.367096    0.387767    0.050465    0.026231    0.640583    
#> 067: 0.210462    0.918028    3.363551    0.411721    0.058497    0.024919    0.625165    
#> 068: 0.181879    0.906640    3.374325    0.391135    0.057987    0.029860    0.646638    
#> 069: 0.193194    0.905257    3.368459    0.371578    0.055087    0.028367    0.655900    
#> 070: 0.192438    0.903302    3.372174    0.377420    0.052333    0.031639    0.647182    
#> 071: 0.174496    0.947724    3.346153    0.458277    0.056102    0.030057    0.641006    
#> 072: 0.179521    0.922884    3.348535    0.531577    0.053297    0.028554    0.631883    
#> 073: 0.213257    0.916973    3.369759    0.531931    0.050632    0.030340    0.640479    
#> 074: 0.202887    0.910569    3.370555    0.505334    0.050360    0.028823    0.639246    
#> 075: 0.212645    0.900838    3.384807    0.480067    0.047842    0.027381    0.644555    
#> 076: 0.253120    0.913953    3.387401    0.456064    0.045450    0.031679    0.660892    
#> 077: 0.286864    0.902283    3.395939    0.446782    0.043177    0.030182    0.652717    
#> 078: 0.217533    0.901477    3.384987    0.424443    0.041019    0.029128    0.640769    
#> 079: 0.208995    0.910144    3.379302    0.403221    0.044859    0.027672    0.667729    
#> 080: 0.250566    0.896315    3.382387    0.434294    0.052457    0.032829    0.641262    
#> 081: 0.236794    0.902024    3.379118    0.412579    0.049834    0.035227    0.634389    
#> 082: 0.223246    0.891202    3.380025    0.391950    0.052789    0.033465    0.639762    
#> 083: 0.221661    0.913841    3.377598    0.372353    0.050149    0.031792    0.654262    
#> 084: 0.212042    0.922146    3.369762    0.357904    0.056519    0.030203    0.639903    
#> 085: 0.228386    0.916200    3.391452    0.371078    0.053693    0.028692    0.639456    
#> 086: 0.233296    0.914019    3.381393    0.362789    0.054049    0.027258    0.640981    
#> 087: 0.202453    0.902532    3.368851    0.353237    0.054346    0.025895    0.632190    
#> 088: 0.152864    0.885982    3.366456    0.335575    0.054098    0.024600    0.673051    
#> 089: 0.202745    0.928418    3.360342    0.404426    0.068731    0.023370    0.640145    
#> 090: 0.191597    0.919341    3.359146    0.384205    0.069049    0.022202    0.646284    
#> 091: 0.164871    0.913708    3.343608    0.364995    0.068996    0.022960    0.647830    
#> 092: 0.155370    0.931873    3.337374    0.349203    0.065547    0.025744    0.660128    
#> 093: 0.149307    0.923988    3.348152    0.373522    0.062269    0.024457    0.680999    
#> 094: 0.169742    0.935573    3.349656    0.397995    0.070179    0.023234    0.669708    
#> 095: 0.177018    0.915590    3.363529    0.395285    0.066670    0.024506    0.657008    
#> 096: 0.137223    0.925123    3.344828    0.375521    0.067802    0.023281    0.636122    
#> 097: 0.192223    0.940290    3.346512    0.450903    0.071130    0.022117    0.626725    
#> 098: 0.210634    0.935214    3.353243    0.479589    0.077072    0.023367    0.646494    
#> 099: 0.177822    0.914634    3.347950    0.455610    0.078404    0.022198    0.642849    
#> 100: 0.203085    0.933044    3.361669    0.432829    0.074541    0.021088    0.631694    
#> 101: 0.155513    0.925302    3.341594    0.411188    0.076879    0.020034    0.643793    
#> 102: 0.190010    0.934695    3.335996    0.390628    0.073035    0.019032    0.661014    
#> 103: 0.201298    0.914442    3.361618    0.472733    0.069383    0.018081    0.647318    
#> 104: 0.199263    0.910494    3.372008    0.495073    0.065914    0.017955    0.653082    
#> 105: 0.196531    0.909498    3.366985    0.470319    0.074458    0.017057    0.659877    
#> 106: 0.192656    0.916309    3.358340    0.446803    0.071734    0.016430    0.643468    
#> 107: 0.184760    0.931222    3.359711    0.424463    0.069941    0.020618    0.635741    
#> 108: 0.171827    0.920649    3.364365    0.403240    0.076809    0.019982    0.644206    
#> 109: 0.202160    0.925001    3.371581    0.383078    0.072969    0.018983    0.653195    
#> 110: 0.214194    0.900452    3.362367    0.421380    0.069320    0.018034    0.650138    
#> 111: 0.202875    0.912171    3.366688    0.401981    0.068304    0.018411    0.639096    
#> 112: 0.161795    0.906745    3.346736    0.381882    0.064889    0.018832    0.655356    
#> 113: 0.170294    0.933665    3.348818    0.424354    0.061644    0.019333    0.645342    
#> 114: 0.189466    0.923227    3.359058    0.403136    0.058562    0.018898    0.657608    
#> 115: 0.204189    0.905856    3.363267    0.402517    0.058567    0.019685    0.654609    
#> 116: 0.199298    0.918712    3.359088    0.426320    0.064550    0.019166    0.637138    
#> 117: 0.182545    0.927592    3.363335    0.425285    0.070309    0.018208    0.654633    
#> 118: 0.206543    0.914768    3.362597    0.470243    0.066793    0.017298    0.656040    
#> 119: 0.184234    0.932013    3.355859    0.446731    0.068185    0.016433    0.665578    
#> 120: 0.157719    0.915689    3.349897    0.424395    0.064776    0.015708    0.664719    
#> 121: 0.186683    0.938521    3.351687    0.473428    0.072864    0.018256    0.662060    
#> 122: 0.192371    0.923309    3.370466    0.449757    0.078708    0.017343    0.668679    
#> 123: 0.191094    0.952353    3.353785    0.427269    0.074773    0.016476    0.676828    
#> 124: 0.170375    0.960807    3.348753    0.405906    0.071034    0.016157    0.661773    
#> 125: 0.212798    0.946177    3.359102    0.385610    0.067483    0.019079    0.670395    
#> 126: 0.195979    0.940073    3.348704    0.383734    0.064963    0.018126    0.671280    
#> 127: 0.209527    0.947716    3.338315    0.431666    0.061715    0.017219    0.673013    
#> 128: 0.182020    0.929038    3.352664    0.410082    0.064139    0.020091    0.670907    
#> 129: 0.176608    0.936659    3.363530    0.389578    0.069937    0.020528    0.678277    
#> 130: 0.213411    0.934975    3.366540    0.370099    0.067561    0.020777    0.657401    
#> 131: 0.162467    0.920562    3.345439    0.376138    0.074027    0.020662    0.653367    
#> 132: 0.132968    0.940064    3.334074    0.357331    0.070326    0.025116    0.648198    
#> 133: 0.171883    0.925497    3.346434    0.439796    0.066809    0.028845    0.624227    
#> 134: 0.203239    0.905002    3.364549    0.431595    0.063469    0.030867    0.636728    
#> 135: 0.224691    0.898679    3.373700    0.410015    0.060295    0.029323    0.643765    
#> 136: 0.250652    0.891640    3.374738    0.474291    0.060763    0.027857    0.631953    
#> 137: 0.273905    0.895475    3.383582    0.484989    0.057725    0.027365    0.643604    
#> 138: 0.250202    0.901072    3.380052    0.460739    0.054839    0.026985    0.638401    
#> 139: 0.216755    0.910549    3.375820    0.437702    0.053271    0.025635    0.645077    
#> 140: 0.205663    0.916226    3.377082    0.415817    0.054839    0.025839    0.666228    
#> 141: 0.213468    0.893874    3.375280    0.395026    0.052097    0.025454    0.645503    
#> 142: 0.221997    0.893063    3.386239    0.375275    0.057909    0.025535    0.649763    
#> 143: 0.207808    0.912390    3.375457    0.356511    0.072856    0.024258    0.663484    
#> 144: 0.255969    0.907065    3.397142    0.376961    0.069213    0.023045    0.636313    
#> 145: 0.229484    0.918197    3.371557    0.358113    0.065753    0.021893    0.650053    
#> 146: 0.182059    0.914150    3.370275    0.344380    0.065140    0.020798    0.649812    
#> 147: 0.168282    0.907884    3.357748    0.335274    0.061883    0.021294    0.649053    
#> 148: 0.190622    0.920681    3.366999    0.352027    0.058789    0.022097    0.650579    
#> 149: 0.195239    0.912172    3.373914    0.452966    0.055850    0.021202    0.654009    
#> 150: 0.219066    0.908129    3.359997    0.430318    0.055602    0.026350    0.657226    
#> 151: 0.183878    0.925486    3.369954    0.408802    0.052822    0.025032    0.653831    
#> 152: 0.200834    0.920190    3.361343    0.404246    0.045856    0.023082    0.655228    
#> 153: 0.198498    0.916895    3.377087    0.369215    0.056494    0.025239    0.660083    
#> 154: 0.236272    0.906459    3.381488    0.418322    0.041238    0.026999    0.654307    
#> 155: 0.214268    0.904600    3.371727    0.397143    0.052692    0.022357    0.649464    
#> 156: 0.217442    0.910401    3.375882    0.404364    0.050758    0.020776    0.655423    
#> 157: 0.263639    0.901074    3.385095    0.459607    0.057393    0.017555    0.654774    
#> 158: 0.242143    0.911678    3.379563    0.387636    0.061439    0.019976    0.650605    
#> 159: 0.232794    0.903835    3.384251    0.406545    0.076468    0.021027    0.664557    
#> 160: 0.255318    0.871870    3.399804    0.371931    0.070178    0.024776    0.657913    
#> 161: 0.250989    0.883197    3.390097    0.388017    0.061610    0.025901    0.651777    
#> 162: 0.248535    0.895040    3.392619    0.429176    0.054336    0.022119    0.651796    
#> 163: 0.278919    0.890992    3.382984    0.409612    0.063580    0.023942    0.639586    
#> 164: 0.266462    0.894504    3.376072    0.462846    0.056882    0.020187    0.638458    
#> 165: 0.254114    0.886219    3.384939    0.446984    0.056118    0.025771    0.643286    
#> 166: 0.277547    0.917039    3.378487    0.627338    0.060727    0.026909    0.646208    
#> 167: 0.252233    0.930173    3.391675    0.471075    0.071512    0.021996    0.651914    
#> 168: 0.272677    0.917399    3.383415    0.482217    0.061215    0.019775    0.634668    
#> 169: 0.281642    0.941792    3.374304    0.521105    0.068996    0.021298    0.657975    
#> 170: 0.280765    0.925995    3.379373    0.586823    0.074193    0.022951    0.652970    
#> 171: 0.263204    0.904595    3.374019    0.561285    0.081505    0.017923    0.627507    
#> 172: 0.205710    0.931967    3.365898    0.409150    0.073372    0.016950    0.629492    
#> 173: 0.207684    0.934545    3.353536    0.438056    0.060504    0.012228    0.644529    
#> 174: 0.224075    0.922264    3.362055    0.440088    0.053259    0.013074    0.662631    
#> 175: 0.215706    0.936261    3.354417    0.403376    0.050530    0.014981    0.652303    
#> 176: 0.223996    0.945142    3.367823    0.394612    0.062204    0.020093    0.634245    
#> 177: 0.240648    0.914336    3.379425    0.373526    0.047565    0.015091    0.660925    
#> 178: 0.231615    0.902548    3.373820    0.401237    0.049617    0.017729    0.655140    
#> 179: 0.252353    0.914469    3.367060    0.463463    0.059417    0.020998    0.644616    
#> 180: 0.239299    0.920549    3.370164    0.502250    0.050920    0.021428    0.643857    
#> 181: 0.233664    0.915623    3.353176    0.450862    0.049199    0.020634    0.648772    
#> 182: 0.186938    0.915071    3.347673    0.450737    0.051849    0.022456    0.662348    
#> 183: 0.195604    0.933513    3.353071    0.460262    0.055133    0.026722    0.640110    
#> 184: 0.203319    0.918802    3.355740    0.480443    0.055301    0.027512    0.630672    
#> 185: 0.152125    0.938671    3.345964    0.424923    0.070961    0.026176    0.631468    
#> 186: 0.159350    0.958811    3.339524    0.402116    0.063800    0.023681    0.635499    
#> 187: 0.188016    0.930473    3.347502    0.493242    0.053081    0.021410    0.633021    
#> 188: 0.168159    0.936066    3.351516    0.451495    0.056884    0.024150    0.641438    
#> 189: 0.166025    0.943049    3.353646    0.428608    0.054951    0.024390    0.648016    
#> 190: 0.199222    0.936102    3.354586    0.473533    0.044725    0.021706    0.642703    
#> 191: 0.183519    0.925965    3.354128    0.422850    0.039656    0.023447    0.637338    
#> 192: 0.162185    0.922722    3.352518    0.400671    0.045587    0.025856    0.633518    
#> 193: 0.186275    0.934878    3.354941    0.400984    0.057243    0.023044    0.644788    
#> 194: 0.191662    0.934201    3.352873    0.442181    0.061642    0.023607    0.647928    
#> 195: 0.160193    0.943361    3.345839    0.434824    0.047049    0.026790    0.636897    
#> 196: 0.138783    0.937098    3.352829    0.398682    0.046874    0.024661    0.651830    
#> 197: 0.138633    0.929951    3.366297    0.382311    0.046207    0.022390    0.651192    
#> 198: 0.202497    0.909843    3.366393    0.342217    0.055815    0.023601    0.640110    
#> 199: 0.197893    0.903242    3.367619    0.413310    0.059911    0.027594    0.632090    
#> 200: 0.196626    0.902703    3.365471    0.443599    0.060390    0.025628    0.636711    
#> 201: 0.198489    0.901522    3.369225    0.432795    0.058286    0.026875    0.633575    
#> 202: 0.201395    0.908811    3.369364    0.427352    0.058415    0.026366    0.633287    
#> 203: 0.204379    0.911347    3.367217    0.406261    0.059371    0.025654    0.635611    
#> 204: 0.199899    0.911588    3.365393    0.404390    0.061012    0.026209    0.636223    
#> 205: 0.202017    0.916852    3.363195    0.407308    0.060280    0.026382    0.640180    
#> 206: 0.198191    0.922619    3.361513    0.415801    0.058725    0.027005    0.642054    
#> 207: 0.193712    0.923988    3.358800    0.409070    0.059836    0.026978    0.641900    
#> 208: 0.194080    0.924743    3.357323    0.410901    0.060035    0.026903    0.640855    
#> 209: 0.192993    0.923353    3.355948    0.412304    0.058276    0.026765    0.643833    
#> 210: 0.193334    0.923488    3.356482    0.411955    0.057658    0.026765    0.644492    
#> 211: 0.195123    0.924702    3.357047    0.404341    0.057345    0.026474    0.644762    
#> 212: 0.198145    0.924091    3.358461    0.400486    0.057148    0.026338    0.644468    
#> 213: 0.200977    0.922602    3.360042    0.400766    0.057198    0.026592    0.644667    
#> 214: 0.204717    0.921893    3.361785    0.402203    0.057040    0.026738    0.644734    
#> 215: 0.207809    0.921102    3.363682    0.402862    0.056755    0.026865    0.645289    
#> 216: 0.210228    0.919530    3.364738    0.403465    0.056700    0.026791    0.646488    
#> 217: 0.209234    0.920483    3.364931    0.401655    0.056291    0.026368    0.646505    
#> 218: 0.210097    0.920177    3.364784    0.404083    0.056765    0.026026    0.646128    
#> 219: 0.210045    0.919779    3.364410    0.407372    0.056412    0.025769    0.645970    
#> 220: 0.209245    0.919634    3.364263    0.408073    0.056300    0.025905    0.646014    
#> 221: 0.209944    0.919402    3.364444    0.410558    0.056157    0.025807    0.645606    
#> 222: 0.210990    0.918647    3.364422    0.414849    0.055473    0.025911    0.645361    
#> 223: 0.210943    0.917704    3.364594    0.413338    0.054787    0.026057    0.645098    
#> 224: 0.211273    0.917737    3.364915    0.411313    0.054417    0.025910    0.645219    
#> 225: 0.210846    0.916739    3.365533    0.411487    0.054107    0.025794    0.645981    
#> 226: 0.211540    0.915830    3.365570    0.411914    0.053929    0.025527    0.646904    
#> 227: 0.210319    0.916244    3.365069    0.411398    0.054296    0.025451    0.647572    
#> 228: 0.209790    0.917087    3.364662    0.410234    0.054462    0.025416    0.647934    
#> 229: 0.208552    0.916649    3.364443    0.409346    0.054757    0.025378    0.647994    
#> 230: 0.208676    0.916064    3.364387    0.408760    0.054500    0.025258    0.648688    
#> 231: 0.208720    0.916055    3.364532    0.408604    0.054417    0.025185    0.648611    
#> 232: 0.208068    0.915678    3.364638    0.407283    0.054248    0.025215    0.648990    
#> 233: 0.206953    0.914915    3.364468    0.406215    0.054274    0.025240    0.649433    
#> 234: 0.206850    0.915018    3.364472    0.406420    0.054521    0.025225    0.649994    
#> 235: 0.206753    0.915092    3.364484    0.405673    0.054452    0.025099    0.650321    
#> 236: 0.205901    0.915349    3.364294    0.403946    0.054469    0.024999    0.650365    
#> 237: 0.206345    0.915125    3.364541    0.403305    0.054558    0.024873    0.651192    
#> 238: 0.206617    0.915114    3.364828    0.402875    0.054691    0.024918    0.651711    
#> 239: 0.206939    0.915020    3.364949    0.402426    0.054737    0.024771    0.651578    
#> 240: 0.206781    0.914912    3.365328    0.400929    0.054796    0.024637    0.651197    
#> 241: 0.207359    0.913848    3.365778    0.399913    0.054864    0.024580    0.651071    
#> 242: 0.207972    0.912784    3.365969    0.398549    0.054699    0.024537    0.651242    
#> 243: 0.207651    0.912972    3.365921    0.396721    0.054688    0.024491    0.651022    
#> 244: 0.207883    0.912606    3.366267    0.397169    0.054550    0.024576    0.650742    
#> 245: 0.208942    0.912026    3.366537    0.397666    0.054615    0.024660    0.651070    
#> 246: 0.209597    0.911560    3.366570    0.399050    0.054573    0.024685    0.651328    
#> 247: 0.209500    0.910947    3.366817    0.398570    0.054589    0.024698    0.651557    
#> 248: 0.209555    0.910969    3.367027    0.398265    0.054680    0.024584    0.651333    
#> 249: 0.209777    0.910735    3.367288    0.397927    0.054803    0.024426    0.651298    
#> 250: 0.210211    0.910824    3.367561    0.396785    0.054848    0.024347    0.651552    
#> 251: 0.211291    0.910922    3.367917    0.397198    0.054674    0.024291    0.651600    
#> 252: 0.212208    0.911231    3.368335    0.396308    0.054724    0.024152    0.652094    
#> 253: 0.212291    0.911647    3.368543    0.395988    0.054761    0.024129    0.652651    
#> 254: 0.212898    0.911870    3.368771    0.395928    0.054754    0.024140    0.652917    
#> 255: 0.212923    0.911787    3.368873    0.395190    0.054678    0.024080    0.653197    
#> 256: 0.212797    0.911700    3.368842    0.393984    0.054658    0.023989    0.653248    
#> 257: 0.212481    0.911570    3.368772    0.394128    0.054918    0.023961    0.653377    
#> 258: 0.211716    0.911305    3.368588    0.392641    0.055054    0.023835    0.653577    
#> 259: 0.211745    0.911051    3.368584    0.392380    0.055123    0.023764    0.653942    
#> 260: 0.211506    0.911095    3.368612    0.393209    0.055279    0.023722    0.653845    
#> 261: 0.211694    0.910848    3.368764    0.393986    0.055333    0.023619    0.653865    
#> 262: 0.211266    0.910854    3.368890    0.393836    0.055189    0.023605    0.653902    
#> 263: 0.210962    0.911130    3.368816    0.393490    0.055246    0.023561    0.654258    
#> 264: 0.210895    0.911281    3.368784    0.392779    0.055289    0.023548    0.654478    
#> 265: 0.210627    0.911375    3.368795    0.391996    0.055257    0.023597    0.654609    
#> 266: 0.210607    0.911656    3.368868    0.391551    0.055158    0.023574    0.654763    
#> 267: 0.210786    0.911601    3.369033    0.391309    0.055132    0.023654    0.654809    
#> 268: 0.211131    0.911363    3.369037    0.392131    0.055173    0.023689    0.654803    
#> 269: 0.211169    0.911537    3.368893    0.392723    0.055248    0.023681    0.654756    
#> 270: 0.210967    0.911523    3.368782    0.391787    0.055198    0.023685    0.654737    
#> 271: 0.210844    0.911623    3.368591    0.392422    0.055044    0.023695    0.654840    
#> 272: 0.210686    0.911642    3.368382    0.393114    0.054887    0.023733    0.654944    
#> 273: 0.210385    0.911794    3.368340    0.393154    0.054830    0.023734    0.654947    
#> 274: 0.209980    0.912082    3.368357    0.392945    0.054750    0.023713    0.654835    
#> 275: 0.209896    0.912244    3.368253    0.392460    0.054658    0.023675    0.654860    
#> 276: 0.209713    0.912411    3.368293    0.392232    0.054456    0.023784    0.654741    
#> 277: 0.209405    0.912544    3.368346    0.392322    0.054327    0.023877    0.654755    
#> 278: 0.209112    0.912582    3.368358    0.391855    0.054204    0.023965    0.654733    
#> 279: 0.208984    0.912325    3.368392    0.391370    0.054112    0.024042    0.654818    
#> 280: 0.208969    0.912453    3.368389    0.390635    0.054059    0.024074    0.654608    
#> 281: 0.208532    0.912623    3.368301    0.390102    0.054065    0.024010    0.654496    
#> 282: 0.208366    0.912762    3.368324    0.389299    0.054152    0.024005    0.654490    
#> 283: 0.208439    0.912861    3.368473    0.388959    0.054312    0.024033    0.654876    
#> 284: 0.208726    0.912672    3.368568    0.388478    0.054311    0.024041    0.654726    
#> 285: 0.208528    0.912700    3.368474    0.388305    0.054358    0.024031    0.654668    
#> 286: 0.208217    0.912714    3.368333    0.387828    0.054448    0.024045    0.654701    
#> 287: 0.207758    0.913008    3.368099    0.387703    0.054354    0.024122    0.654765    
#> 288: 0.207582    0.913092    3.367944    0.387840    0.054460    0.024150    0.654851    
#> 289: 0.207770    0.913027    3.367871    0.387730    0.054527    0.024140    0.654878    
#> 290: 0.207923    0.912876    3.367961    0.387853    0.054775    0.024154    0.654934    
#> 291: 0.207872    0.912921    3.367950    0.388010    0.054836    0.024147    0.654906    
#> 292: 0.207562    0.912945    3.367824    0.387934    0.054868    0.024141    0.654872    
#> 293: 0.206984    0.912960    3.367538    0.387855    0.054825    0.024169    0.654837    
#> 294: 0.206767    0.913078    3.367317    0.387572    0.054832    0.024161    0.654900    
#> 295: 0.206495    0.912964    3.367203    0.387475    0.054876    0.024127    0.654798    
#> 296: 0.206342    0.912902    3.367144    0.387579    0.054785    0.024112    0.654827    
#> 297: 0.206174    0.912964    3.367077    0.387290    0.054735    0.024074    0.654947    
#> 298: 0.205868    0.912942    3.366984    0.387535    0.054840    0.024073    0.654825    
#> 299: 0.205578    0.913064    3.366881    0.387560    0.054782    0.024103    0.654571    
#> 300: 0.205575    0.912964    3.366809    0.387589    0.054719    0.024086    0.654531    
#> 301: 0.205401    0.912877    3.366691    0.387564    0.054610    0.024081    0.654563    
#> 302: 0.205415    0.912838    3.366606    0.387739    0.054582    0.024034    0.654542    
#> 303: 0.205075    0.913039    3.366480    0.387450    0.054553    0.023983    0.654405    
#> 304: 0.204868    0.913431    3.366322    0.387162    0.054603    0.023928    0.654430    
#> 305: 0.204602    0.913482    3.366159    0.387033    0.054635    0.023886    0.654365    
#> 306: 0.204265    0.913406    3.366148    0.386690    0.054563    0.023848    0.654314    
#> 307: 0.204231    0.913423    3.366108    0.386655    0.054509    0.023847    0.654315    
#> 308: 0.203896    0.913402    3.365986    0.386738    0.054508    0.023828    0.654298    
#> 309: 0.203766    0.913242    3.365954    0.386805    0.054468    0.023818    0.654535    
#> 310: 0.203942    0.913171    3.365906    0.387308    0.054414    0.023819    0.654437    
#> 311: 0.203968    0.913138    3.365970    0.387217    0.054408    0.023806    0.654256    
#> 312: 0.203923    0.913130    3.365914    0.387276    0.054361    0.023793    0.654226    
#> 313: 0.203815    0.912920    3.365763    0.387352    0.054343    0.023806    0.654352    
#> 314: 0.203484    0.912996    3.365643    0.387226    0.054317    0.023797    0.654375    
#> 315: 0.203277    0.913237    3.365461    0.387765    0.054389    0.023807    0.654255    
#> 316: 0.203089    0.913297    3.365339    0.387427    0.054506    0.023803    0.654029    
#> 317: 0.203036    0.913356    3.365336    0.387472    0.054608    0.023749    0.654107    
#> 318: 0.202972    0.913537    3.365221    0.387655    0.054703    0.023715    0.654072    
#> 319: 0.202792    0.913524    3.365193    0.387629    0.054702    0.023692    0.654141    
#> 320: 0.202299    0.913466    3.365116    0.387472    0.054766    0.023670    0.654237    
#> 321: 0.201939    0.913594    3.364958    0.387278    0.054842    0.023614    0.654341    
#> 322: 0.201611    0.913775    3.364683    0.387595    0.054971    0.023566    0.654402    
#> 323: 0.200960    0.913981    3.364418    0.387550    0.055017    0.023521    0.654494    
#> 324: 0.200642    0.914109    3.364311    0.387648    0.055076    0.023459    0.654569    
#> 325: 0.200315    0.914302    3.364159    0.387488    0.055058    0.023392    0.654604    
#> 326: 0.200036    0.914361    3.364064    0.387630    0.055126    0.023343    0.654583    
#> 327: 0.199935    0.914357    3.364039    0.387642    0.055091    0.023278    0.654642    
#> 328: 0.199865    0.914461    3.363985    0.387396    0.055132    0.023248    0.654552    
#> 329: 0.200091    0.914538    3.364019    0.387768    0.055222    0.023218    0.654440    
#> 330: 0.199932    0.914577    3.363977    0.387661    0.055294    0.023159    0.654399    
#> 331: 0.200086    0.914637    3.364107    0.387350    0.055282    0.023149    0.654429    
#> 332: 0.200201    0.914503    3.364267    0.387061    0.055230    0.023169    0.654586    
#> 333: 0.200199    0.914442    3.364268    0.386708    0.055179    0.023170    0.654663    
#> 334: 0.200094    0.914377    3.364164    0.386689    0.055167    0.023132    0.654707    
#> 335: 0.199861    0.914415    3.364098    0.386344    0.055139    0.023121    0.654764    
#> 336: 0.199569    0.914465    3.364068    0.385868    0.055190    0.023100    0.654864    
#> 337: 0.199329    0.914698    3.363971    0.385414    0.055211    0.023099    0.654899    
#> 338: 0.199267    0.914746    3.363926    0.385141    0.055237    0.023051    0.654845    
#> 339: 0.199100    0.914948    3.363799    0.385274    0.055239    0.023003    0.654724    
#> 340: 0.198852    0.914864    3.363716    0.385259    0.055216    0.022980    0.654686    
#> 341: 0.198761    0.914814    3.363668    0.385438    0.055245    0.022956    0.654681    
#> 342: 0.198504    0.914664    3.363623    0.385182    0.055170    0.022941    0.654650    
#> 343: 0.198224    0.914635    3.363592    0.385153    0.055183    0.022939    0.654591    
#> 344: 0.198125    0.914650    3.363563    0.385085    0.055209    0.022968    0.654601    
#> 345: 0.198094    0.914657    3.363609    0.385089    0.055278    0.022951    0.654574    
#> 346: 0.198077    0.914671    3.363587    0.384955    0.055324    0.022930    0.654603    
#> 347: 0.198119    0.914572    3.363551    0.385034    0.055359    0.022917    0.654801    
#> 348: 0.198096    0.914642    3.363520    0.385491    0.055384    0.022906    0.654903    
#> 349: 0.198148    0.914602    3.363539    0.385718    0.055463    0.022887    0.654957    
#> 350: 0.198150    0.914472    3.363568    0.385871    0.055625    0.022867    0.655046    
#> 351: 0.198336    0.914507    3.363574    0.385862    0.055758    0.022846    0.655079    
#> 352: 0.198153    0.914552    3.363552    0.385798    0.055892    0.022850    0.655204    
#> 353: 0.198330    0.914688    3.363559    0.385691    0.056041    0.022821    0.655163    
#> 354: 0.198250    0.914746    3.363546    0.385421    0.056163    0.022782    0.655149    
#> 355: 0.198643    0.914676    3.363585    0.385509    0.056236    0.022779    0.655177    
#> 356: 0.198639    0.914602    3.363593    0.385371    0.056361    0.022760    0.655315    
#> 357: 0.198593    0.914645    3.363605    0.385212    0.056384    0.022784    0.655298    
#> 358: 0.198657    0.914663    3.363587    0.385416    0.056396    0.022780    0.655260    
#> 359: 0.198608    0.914718    3.363516    0.385566    0.056472    0.022764    0.655231    
#> 360: 0.198336    0.914724    3.363364    0.385353    0.056539    0.022734    0.655258    
#> 361: 0.198168    0.914796    3.363267    0.385383    0.056626    0.022701    0.655203    
#> 362: 0.198197    0.914802    3.363228    0.385452    0.056661    0.022662    0.655067    
#> 363: 0.198151    0.914826    3.363243    0.385207    0.056674    0.022629    0.654981    
#> 364: 0.198131    0.914910    3.363227    0.385191    0.056731    0.022590    0.654946    
#> 365: 0.198004    0.914931    3.363230    0.385326    0.056916    0.022546    0.654810    
#> 366: 0.197957    0.915000    3.363133    0.385856    0.057042    0.022517    0.654669    
#> 367: 0.198168    0.915047    3.363091    0.386425    0.057110    0.022481    0.654566    
#> 368: 0.198288    0.915104    3.363073    0.386352    0.057146    0.022460    0.654500    
#> 369: 0.198162    0.915074    3.363086    0.386323    0.057138    0.022435    0.654525    
#> 370: 0.198076    0.915082    3.363122    0.386224    0.057169    0.022400    0.654466    
#> 371: 0.197944    0.915103    3.363121    0.386095    0.057146    0.022344    0.654524    
#> 372: 0.197917    0.915097    3.363120    0.386149    0.057197    0.022311    0.654471    
#> 373: 0.197747    0.915204    3.363052    0.386133    0.057249    0.022286    0.654464    
#> 374: 0.197720    0.915286    3.362976    0.386423    0.057243    0.022295    0.654370    
#> 375: 0.197539    0.915350    3.362921    0.386691    0.057260    0.022313    0.654330    
#> 376: 0.197563    0.915430    3.362868    0.386707    0.057337    0.022300    0.654225    
#> 377: 0.197609    0.915484    3.362860    0.386596    0.057423    0.022278    0.654183    
#> 378: 0.197475    0.915534    3.362843    0.386573    0.057436    0.022302    0.654128    
#> 379: 0.197370    0.915440    3.362829    0.386558    0.057388    0.022311    0.654129    
#> 380: 0.197192    0.915385    3.362829    0.386229    0.057389    0.022302    0.654182    
#> 381: 0.197065    0.915401    3.362786    0.385977    0.057364    0.022298    0.654200    
#> 382: 0.197000    0.915384    3.362763    0.386143    0.057331    0.022310    0.654078    
#> 383: 0.196881    0.915351    3.362685    0.386153    0.057347    0.022295    0.653932    
#> 384: 0.196705    0.915364    3.362644    0.386224    0.057377    0.022283    0.653858    
#> 385: 0.196621    0.915425    3.362598    0.385854    0.057432    0.022265    0.653772    
#> 386: 0.196603    0.915509    3.362602    0.385513    0.057463    0.022244    0.653747    
#> 387: 0.196654    0.915612    3.362671    0.385587    0.057422    0.022236    0.653692    
#> 388: 0.196702    0.915560    3.362709    0.385423    0.057454    0.022212    0.653653    
#> 389: 0.196659    0.915392    3.362702    0.385397    0.057454    0.022185    0.653788    
#> 390: 0.196603    0.915467    3.362681    0.385365    0.057461    0.022165    0.653766    
#> 391: 0.196826    0.915426    3.362684    0.385524    0.057436    0.022176    0.653750    
#> 392: 0.196958    0.915423    3.362759    0.385672    0.057452    0.022203    0.653770    
#> 393: 0.197054    0.915386    3.362766    0.385672    0.057434    0.022230    0.653752    
#> 394: 0.197296    0.915337    3.362822    0.385706    0.057437    0.022231    0.653795    
#> 395: 0.197469    0.915409    3.362854    0.385752    0.057426    0.022240    0.653797    
#> 396: 0.197594    0.915422    3.362872    0.385661    0.057390    0.022252    0.653733    
#> 397: 0.197611    0.915344    3.362944    0.385364    0.057388    0.022274    0.653758    
#> 398: 0.197687    0.915270    3.362970    0.385040    0.057329    0.022302    0.653796    
#> 399: 0.197766    0.915247    3.363004    0.384996    0.057283    0.022308    0.653743    
#> 400: 0.197871    0.915231    3.363040    0.384702    0.057274    0.022304    0.653665    
#> 401: 0.197979    0.915236    3.363037    0.384747    0.057296    0.022294    0.653623    
#> 402: 0.198016    0.915288    3.363056    0.384786    0.057374    0.022270    0.653595    
#> 403: 0.198254    0.915275    3.363089    0.384851    0.057433    0.022258    0.653533    
#> 404: 0.198253    0.915285    3.363011    0.385176    0.057475    0.022247    0.653465    
#> 405: 0.198251    0.915338    3.363001    0.385649    0.057530    0.022233    0.653492    
#> 406: 0.198210    0.915397    3.362929    0.385871    0.057635    0.022231    0.653528    
#> 407: 0.198171    0.915481    3.362935    0.385932    0.057723    0.022222    0.653554    
#> 408: 0.198212    0.915587    3.362898    0.385754    0.057743    0.022215    0.653591    
#> 409: 0.198154    0.915513    3.362881    0.385566    0.057726    0.022200    0.653557    
#> 410: 0.198094    0.915505    3.362835    0.385502    0.057747    0.022190    0.653575    
#> 411: 0.197999    0.915350    3.362843    0.385119    0.057729    0.022162    0.653720    
#> 412: 0.198165    0.915357    3.362903    0.385096    0.057724    0.022159    0.653800    
#> 413: 0.198402    0.915389    3.362948    0.385474    0.057737    0.022157    0.653795    
#> 414: 0.198361    0.915457    3.362970    0.385654    0.057787    0.022149    0.653851    
#> 415: 0.198469    0.915491    3.362975    0.386073    0.057866    0.022148    0.653809    
#> 416: 0.198621    0.915510    3.362992    0.386190    0.057906    0.022165    0.653771    
#> 417: 0.198726    0.915469    3.363029    0.386169    0.057909    0.022183    0.653716    
#> 418: 0.198640    0.915392    3.363069    0.386028    0.057917    0.022202    0.653706    
#> 419: 0.198669    0.915430    3.363059    0.386106    0.057943    0.022209    0.653669    
#> 420: 0.198642    0.915479    3.363015    0.386036    0.058011    0.022211    0.653600    
#> 421: 0.198574    0.915513    3.363015    0.385999    0.058113    0.022208    0.653578    
#> 422: 0.198653    0.915452    3.363012    0.385711    0.058227    0.022199    0.653572    
#> 423: 0.198723    0.915367    3.363027    0.385692    0.058284    0.022193    0.653553    
#> 424: 0.198643    0.915377    3.362975    0.385638    0.058338    0.022181    0.653509    
#> 425: 0.198643    0.915438    3.362925    0.385721    0.058373    0.022161    0.653559    
#> 426: 0.198499    0.915548    3.362880    0.385807    0.058414    0.022147    0.653567    
#> 427: 0.198333    0.915589    3.362848    0.385854    0.058392    0.022137    0.653541    
#> 428: 0.198177    0.915666    3.362832    0.385711    0.058405    0.022139    0.653526    
#> 429: 0.198128    0.915716    3.362830    0.385640    0.058422    0.022152    0.653464    
#> 430: 0.198178    0.915749    3.362805    0.386105    0.058435    0.022166    0.653452    
#> 431: 0.198183    0.915772    3.362766    0.386344    0.058418    0.022161    0.653411    
#> 432: 0.198025    0.915847    3.362733    0.386296    0.058423    0.022159    0.653308    
#> 433: 0.198038    0.915908    3.362734    0.386501    0.058455    0.022138    0.653303    
#> 434: 0.197957    0.915809    3.362714    0.386419    0.058453    0.022131    0.653336    
#> 435: 0.198027    0.915794    3.362690    0.386611    0.058421    0.022154    0.653288    
#> 436: 0.198006    0.915809    3.362687    0.386527    0.058364    0.022161    0.653294    
#> 437: 0.197993    0.915815    3.362679    0.386498    0.058328    0.022167    0.653292    
#> 438: 0.197936    0.915917    3.362655    0.386474    0.058265    0.022162    0.653247    
#> 439: 0.197924    0.915970    3.362651    0.386428    0.058223    0.022158    0.653174    
#> 440: 0.198045    0.915998    3.362641    0.386416    0.058225    0.022148    0.653084    
#> 441: 0.198065    0.915973    3.362654    0.386220    0.058290    0.022134    0.653001    
#> 442: 0.197898    0.916036    3.362658    0.386075    0.058348    0.022110    0.652950    
#> 443: 0.197812    0.916100    3.362605    0.386202    0.058322    0.022111    0.652883    
#> 444: 0.197540    0.916230    3.362483    0.386257    0.058311    0.022120    0.652838    
#> 445: 0.197297    0.916392    3.362372    0.386275    0.058280    0.022121    0.652893    
#> 446: 0.197110    0.916528    3.362296    0.386253    0.058244    0.022118    0.652907    
#> 447: 0.197132    0.916586    3.362298    0.386309    0.058206    0.022135    0.652888    
#> 448: 0.197149    0.916599    3.362328    0.386334    0.058174    0.022148    0.652819    
#> 449: 0.197369    0.916570    3.362442    0.386534    0.058167    0.022157    0.652826    
#> 450: 0.197495    0.916477    3.362477    0.386713    0.058147    0.022167    0.652827    
#> 451: 0.197631    0.916416    3.362513    0.386801    0.058149    0.022188    0.652872    
#> 452: 0.197620    0.916407    3.362496    0.386718    0.058144    0.022190    0.652851    
#> 453: 0.197605    0.916414    3.362482    0.386692    0.058147    0.022192    0.652772    
#> 454: 0.197535    0.916577    3.362444    0.386533    0.058175    0.022183    0.652713    
#> 455: 0.197574    0.916670    3.362444    0.386439    0.058232    0.022178    0.652667    
#> 456: 0.197525    0.916660    3.362454    0.386353    0.058274    0.022178    0.652629    
#> 457: 0.197440    0.916686    3.362453    0.386248    0.058320    0.022177    0.652554    
#> 458: 0.197385    0.916801    3.362452    0.386212    0.058340    0.022179    0.652522    
#> 459: 0.197170    0.916965    3.362411    0.386423    0.058357    0.022179    0.652570    
#> 460: 0.196945    0.917084    3.362307    0.386591    0.058391    0.022176    0.652564    
#> 461: 0.196762    0.917178    3.362269    0.386518    0.058447    0.022169    0.652548    
#> 462: 0.196716    0.917170    3.362249    0.386769    0.058443    0.022147    0.652568    
#> 463: 0.196744    0.917180    3.362256    0.387108    0.058445    0.022131    0.652583    
#> 464: 0.196812    0.917195    3.362258    0.387261    0.058435    0.022128    0.652506    
#> 465: 0.196951    0.917050    3.362279    0.387508    0.058430    0.022117    0.652508    
#> 466: 0.197067    0.917051    3.362283    0.387964    0.058430    0.022115    0.652465    
#> 467: 0.197150    0.917025    3.362294    0.388235    0.058422    0.022124    0.652379    
#> 468: 0.197335    0.917025    3.362306    0.388588    0.058386    0.022154    0.652323    
#> 469: 0.197322    0.917129    3.362305    0.388670    0.058366    0.022180    0.652387    
#> 470: 0.197138    0.917188    3.362249    0.388538    0.058317    0.022207    0.652424    
#> 471: 0.196919    0.917188    3.362196    0.388299    0.058309    0.022219    0.652425    
#> 472: 0.196719    0.917235    3.362137    0.388121    0.058292    0.022240    0.652407    
#> 473: 0.196569    0.917264    3.362111    0.388134    0.058270    0.022249    0.652332    
#> 474: 0.196431    0.917273    3.362090    0.388222    0.058292    0.022248    0.652350    
#> 475: 0.196406    0.917401    3.362056    0.388292    0.058304    0.022258    0.652302    
#> 476: 0.196392    0.917394    3.362062    0.388430    0.058319    0.022273    0.652304    
#> 477: 0.196349    0.917423    3.362094    0.388420    0.058297    0.022279    0.652225    
#> 478: 0.196296    0.917409    3.362090    0.388477    0.058283    0.022279    0.652292    
#> 479: 0.196399    0.917396    3.362098    0.388573    0.058278    0.022269    0.652337    
#> 480: 0.196469    0.917377    3.362075    0.388543    0.058252    0.022271    0.652451    
#> 481: 0.196346    0.917437    3.362059    0.388524    0.058270    0.022264    0.652531    
#> 482: 0.196317    0.917496    3.362017    0.388591    0.058288    0.022272    0.652521    
#> 483: 0.196216    0.917495    3.361982    0.388588    0.058295    0.022267    0.652526    
#> 484: 0.196122    0.917380    3.361988    0.388476    0.058312    0.022257    0.652558    
#> 485: 0.196015    0.917378    3.361966    0.388428    0.058367    0.022238    0.652545    
#> 486: 0.196095    0.917404    3.362007    0.388571    0.058387    0.022224    0.652502    
#> 487: 0.196163    0.917423    3.362032    0.388588    0.058393    0.022211    0.652502    
#> 488: 0.196211    0.917394    3.362039    0.388590    0.058427    0.022189    0.652498    
#> 489: 0.196364    0.917381    3.362034    0.388641    0.058500    0.022171    0.652521    
#> 490: 0.196416    0.917424    3.362017    0.388747    0.058516    0.022165    0.652475    
#> 491: 0.196359    0.917496    3.361966    0.388885    0.058521    0.022152    0.652450    
#> 492: 0.196211    0.917568    3.361889    0.388913    0.058506    0.022127    0.652484    
#> 493: 0.196091    0.917630    3.361862    0.389060    0.058510    0.022117    0.652481    
#> 494: 0.196198    0.917682    3.361858    0.389196    0.058527    0.022118    0.652438    
#> 495: 0.196275    0.917719    3.361845    0.389164    0.058535    0.022107    0.652415    
#> 496: 0.196309    0.917771    3.361786    0.389307    0.058509    0.022097    0.652459    
#> 497: 0.196295    0.917761    3.361730    0.389448    0.058543    0.022077    0.652547    
#> 498: 0.196285    0.917760    3.361689    0.389717    0.058544    0.022065    0.652514    
#> 499: 0.196357    0.917786    3.361671    0.389787    0.058549    0.022051    0.652480    
#> 500: 0.196402    0.917747    3.361671    0.389863    0.058528    0.022047    0.652449    
#> [====|====|=>-------------------------------------]  25%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.307688    0.961768    3.436703    0.392401    0.078700    0.016800    1.859964    
#> 002: 0.365439    0.914304    3.438026    0.375040    0.074765    0.015960    1.180578    
#> 003: 0.529011    0.817650    3.436416    0.437323    0.076283    0.015162    0.839675    
#> 004: 0.469170    0.811923    3.424230    0.613146    0.072469    0.014404    0.670135    
#> 005: 0.510348    0.800512    3.408193    0.671672    0.072632    0.013684    0.615784    
#> 006: 0.550992    0.812891    3.400149    0.838114    0.070295    0.013000    0.544539    
#> 007: 0.519369    0.838985    3.385290    0.934592    0.066780    0.012350    0.553346    
#> 008: 0.500456    0.832587    3.384544    0.887862    0.063441    0.014113    0.541030    
#> 009: 0.474731    0.823241    3.372672    0.962897    0.060269    0.013408    0.535034    
#> 010: 0.473263    0.819628    3.373192    1.019820    0.057256    0.012737    0.543114    
#> 011: 0.490827    0.830987    3.378860    0.968829    0.054393    0.012100    0.542792    
#> 012: 0.492356    0.831092    3.371195    0.986704    0.056848    0.012060    0.527708    
#> 013: 0.481087    0.833748    3.374389    0.937369    0.057248    0.011465    0.535607    
#> 014: 0.531997    0.830488    3.380762    0.991948    0.060380    0.011850    0.539552    
#> 015: 0.525919    0.839798    3.389264    0.945203    0.062624    0.011258    0.514674    
#> 016: 0.507289    0.845868    3.387703    0.897943    0.059493    0.011069    0.533351    
#> 017: 0.519809    0.852518    3.387203    0.957291    0.058986    0.011915    0.507680    
#> 018: 0.532084    0.850629    3.377147    0.981462    0.056406    0.011375    0.511151    
#> 019: 0.520487    0.843237    3.380240    1.032457    0.058996    0.011044    0.506745    
#> 020: 0.543270    0.842071    3.381173    1.173932    0.056046    0.013996    0.530365    
#> 021: 0.535562    0.826014    3.381933    1.115235    0.054471    0.013736    0.514771    
#> 022: 0.540138    0.854522    3.391354    1.059474    0.058429    0.013049    0.508866    
#> 023: 0.587191    0.852934    3.390511    1.112370    0.055507    0.012397    0.524117    
#> 024: 0.524196    0.863734    3.385367    1.056752    0.052732    0.012966    0.518327    
#> 025: 0.546534    0.842690    3.378849    1.136671    0.050095    0.014885    0.523602    
#> 026: 0.604042    0.854948    3.379554    1.362577    0.048178    0.016421    0.530420    
#> 027: 0.556387    0.853847    3.378473    1.294448    0.045769    0.016196    0.541674    
#> 028: 0.600472    0.857918    3.382466    1.405731    0.046511    0.016220    0.528194    
#> 029: 0.594565    0.860553    3.371298    1.442368    0.048587    0.016596    0.520520    
#> 030: 0.609603    0.862250    3.364035    1.470393    0.046158    0.015766    0.508776    
#> 031: 0.552063    0.849953    3.372858    1.396873    0.043850    0.014978    0.513710    
#> 032: 0.540256    0.843780    3.377658    1.327030    0.045022    0.014229    0.519816    
#> 033: 0.550521    0.844356    3.375582    1.260678    0.045454    0.013518    0.529058    
#> 034: 0.621063    0.844682    3.379621    1.438711    0.052398    0.012842    0.511416    
#> 035: 0.541892    0.838135    3.370832    1.366775    0.052121    0.013337    0.513408    
#> 036: 0.569310    0.846682    3.377096    1.298437    0.049515    0.014020    0.535720    
#> 037: 0.577890    0.841834    3.378243    1.253735    0.047039    0.013881    0.538597    
#> 038: 0.584069    0.834138    3.380478    1.300925    0.044687    0.015629    0.531350    
#> 039: 0.575399    0.819756    3.389421    1.235879    0.042453    0.014848    0.538254    
#> 040: 0.555988    0.826830    3.392086    1.189129    0.048145    0.015825    0.510417    
#> 041: 0.490584    0.835200    3.384819    1.129673    0.048336    0.016740    0.512592    
#> 042: 0.492014    0.833935    3.378264    1.073189    0.050951    0.015903    0.516709    
#> 043: 0.476847    0.841386    3.368882    1.019530    0.050219    0.015108    0.525689    
#> 044: 0.463306    0.845623    3.367232    0.968553    0.047708    0.014532    0.523768    
#> 045: 0.511158    0.837254    3.373941    1.143438    0.045323    0.013806    0.525157    
#> 046: 0.553507    0.830374    3.380916    1.273970    0.043057    0.015143    0.518508    
#> 047: 0.593861    0.833233    3.385232    1.220603    0.052637    0.014826    0.526229    
#> 048: 0.547118    0.843189    3.373166    1.159572    0.057977    0.014085    0.533417    
#> 049: 0.514813    0.829416    3.381871    1.101594    0.055078    0.013381    0.530236    
#> 050: 0.539505    0.811379    3.379978    1.106404    0.052908    0.013038    0.548899    
#> 051: 0.571144    0.834492    3.374844    1.325474    0.054275    0.017215    0.545327    
#> 052: 0.603307    0.831180    3.386808    1.436325    0.051561    0.018004    0.540889    
#> 053: 0.604980    0.827250    3.380353    1.364509    0.058786    0.017554    0.524313    
#> 054: 0.558453    0.847171    3.383694    1.296283    0.056593    0.017559    0.536868    
#> 055: 0.521365    0.843341    3.382373    1.231469    0.057670    0.016994    0.527328    
#> 056: 0.552415    0.843224    3.377770    1.169896    0.056738    0.016145    0.532177    
#> 057: 0.532965    0.846459    3.383949    1.111401    0.053901    0.016438    0.526938    
#> 058: 0.548019    0.838798    3.372072    1.163441    0.054189    0.015616    0.521916    
#> 059: 0.594673    0.823347    3.381538    1.294921    0.051480    0.015953    0.533143    
#> 060: 0.558388    0.817826    3.383986    1.230175    0.057478    0.015340    0.532712    
#> 061: 0.616233    0.832504    3.385792    1.287622    0.054605    0.016705    0.524635    
#> 062: 0.542562    0.839950    3.388390    1.223241    0.051874    0.016688    0.535736    
#> 063: 0.596958    0.816560    3.385901    1.249337    0.049281    0.015854    0.531658    
#> 064: 0.576447    0.826795    3.391305    1.186870    0.053300    0.015846    0.535550    
#> 065: 0.540328    0.830196    3.383318    1.127527    0.051574    0.015054    0.519394    
#> 066: 0.526003    0.845134    3.377199    1.093994    0.048995    0.014301    0.525532    
#> 067: 0.513187    0.837682    3.377434    1.039294    0.047586    0.013586    0.516407    
#> 068: 0.497915    0.834159    3.379400    0.987329    0.045207    0.014902    0.530032    
#> 069: 0.539194    0.827061    3.380124    1.053776    0.044394    0.014157    0.526273    
#> 070: 0.565353    0.830129    3.381822    1.246211    0.045277    0.015576    0.523494    
#> 071: 0.535323    0.851244    3.371692    1.183901    0.049987    0.014797    0.530801    
#> 072: 0.520187    0.828921    3.380062    1.124706    0.047487    0.014057    0.510279    
#> 073: 0.519605    0.834693    3.389692    1.068470    0.048943    0.015511    0.519284    
#> 074: 0.519871    0.824132    3.384250    1.077254    0.046533    0.016576    0.511574    
#> 075: 0.592887    0.826924    3.389091    1.272470    0.044206    0.016081    0.517538    
#> 076: 0.617771    0.841040    3.398176    1.227179    0.041996    0.015277    0.523073    
#> 077: 0.577859    0.834847    3.393557    1.165820    0.051568    0.015072    0.519999    
#> 078: 0.551715    0.835933    3.395291    1.109821    0.048989    0.015445    0.522672    
#> 079: 0.576943    0.833040    3.388196    1.239405    0.057127    0.014673    0.522534    
#> 080: 0.592678    0.840756    3.388624    1.288776    0.054270    0.013939    0.524726    
#> 081: 0.558500    0.844368    3.387612    1.224337    0.054673    0.013242    0.525165    
#> 082: 0.576500    0.829794    3.389212    1.234817    0.061694    0.012580    0.534631    
#> 083: 0.560549    0.844887    3.389817    1.173076    0.058609    0.011951    0.539545    
#> 084: 0.561408    0.842919    3.389561    1.117564    0.058670    0.013711    0.526829    
#> 085: 0.581383    0.832760    3.391470    1.335632    0.057117    0.015033    0.532611    
#> 086: 0.555735    0.823822    3.386283    1.268851    0.054261    0.015014    0.522474    
#> 087: 0.575630    0.827867    3.382972    1.205408    0.051548    0.016568    0.522310    
#> 088: 0.538694    0.829554    3.390360    1.145138    0.048971    0.015740    0.517805    
#> 089: 0.545018    0.832052    3.385379    1.087881    0.055178    0.014953    0.520956    
#> 090: 0.542477    0.835510    3.380084    1.056163    0.053583    0.014205    0.517133    
#> 091: 0.533124    0.838756    3.376239    1.059391    0.060909    0.015341    0.525403    
#> 092: 0.518819    0.837543    3.366344    1.184704    0.057864    0.017405    0.539245    
#> 093: 0.477063    0.835956    3.369256    1.125469    0.054970    0.016535    0.544694    
#> 094: 0.517137    0.848743    3.375083    1.069196    0.052222    0.020739    0.529320    
#> 095: 0.582628    0.847007    3.376978    1.331834    0.049611    0.021799    0.520872    
#> 096: 0.573507    0.853648    3.367077    1.292765    0.048391    0.020709    0.524456    
#> 097: 0.544731    0.865282    3.364588    1.228127    0.054369    0.019674    0.527681    
#> 098: 0.560019    0.862828    3.370015    1.166720    0.051650    0.019638    0.516297    
#> 099: 0.550517    0.842848    3.368932    1.159812    0.049068    0.018656    0.515155    
#> 100: 0.525996    0.856940    3.373552    1.155843    0.050079    0.017723    0.513014    
#> 101: 0.488873    0.842269    3.358068    1.098051    0.049844    0.016837    0.511975    
#> 102: 0.472160    0.852247    3.356122    1.043148    0.047352    0.015995    0.524456    
#> 103: 0.520185    0.851208    3.365851    1.146542    0.053055    0.015196    0.515364    
#> 104: 0.505387    0.850241    3.376905    1.101013    0.050597    0.014436    0.531093    
#> 105: 0.552065    0.828441    3.375824    1.192816    0.056731    0.013714    0.533059    
#> 106: 0.533956    0.836447    3.378528    1.148573    0.053894    0.013028    0.517126    
#> 107: 0.581448    0.835983    3.380678    1.322590    0.051199    0.012446    0.529074    
#> 108: 0.540276    0.837500    3.382046    1.256460    0.057211    0.013263    0.536216    
#> 109: 0.566187    0.845043    3.392792    1.193637    0.055833    0.014046    0.538786    
#> 110: 0.596333    0.823904    3.377569    1.153576    0.059124    0.013518    0.524979    
#> 111: 0.586327    0.817088    3.384544    1.147212    0.057659    0.012842    0.516227    
#> 112: 0.565314    0.829895    3.377631    1.116543    0.062277    0.013558    0.513999    
#> 113: 0.535476    0.860147    3.376789    1.140778    0.059163    0.013687    0.508577    
#> 114: 0.550534    0.853551    3.370936    1.135398    0.056804    0.013728    0.515804    
#> 115: 0.540331    0.834335    3.372714    1.128756    0.057693    0.015216    0.527668    
#> 116: 0.639143    0.839568    3.371261    1.520651    0.054808    0.015236    0.520203    
#> 117: 0.583891    0.852028    3.378319    1.444618    0.054906    0.014833    0.521522    
#> 118: 0.590392    0.839741    3.377531    1.521718    0.053832    0.015376    0.528138    
#> 119: 0.595085    0.848375    3.371028    1.490900    0.054939    0.014607    0.542266    
#> 120: 0.564659    0.821886    3.379404    1.416355    0.052192    0.013877    0.525233    
#> 121: 0.550179    0.859921    3.372595    1.345538    0.057628    0.013183    0.528344    
#> 122: 0.471669    0.839949    3.375308    1.278261    0.054747    0.012524    0.530644    
#> 123: 0.549677    0.854346    3.374974    1.214348    0.052009    0.011897    0.541224    
#> 124: 0.546885    0.863646    3.376935    1.184773    0.049409    0.011303    0.540520    
#> 125: 0.558662    0.861219    3.369899    1.274697    0.051071    0.012210    0.531330    
#> 126: 0.618598    0.866724    3.360600    1.624337    0.048580    0.011600    0.534532    
#> 127: 0.568919    0.837744    3.361884    1.543121    0.046151    0.011135    0.528165    
#> 128: 0.575737    0.840981    3.373929    1.465965    0.046431    0.011460    0.526234    
#> 129: 0.580470    0.842424    3.381937    1.392666    0.053951    0.013266    0.519717    
#> 130: 0.580295    0.846665    3.372936    1.323151    0.058649    0.012602    0.512091    
#> 131: 0.526438    0.842458    3.360165    1.256993    0.055717    0.011972    0.524740    
#> 132: 0.501244    0.862670    3.363462    1.194144    0.052931    0.012350    0.529120    
#> 133: 0.524573    0.833613    3.372416    1.142231    0.050284    0.011732    0.520799    
#> 134: 0.482306    0.824036    3.380252    1.085119    0.047770    0.012028    0.529843    
#> 135: 0.505385    0.837753    3.382710    1.030863    0.054220    0.011569    0.528349    
#> 136: 0.516199    0.824253    3.381654    0.979320    0.052289    0.011049    0.520452    
#> 137: 0.526226    0.828996    3.386869    0.990230    0.049675    0.014852    0.520204    
#> 138: 0.547198    0.831933    3.379898    1.030234    0.052726    0.014109    0.521087    
#> 139: 0.504278    0.841655    3.384143    1.046785    0.056988    0.013404    0.524051    
#> 140: 0.539712    0.846108    3.386496    1.219479    0.054139    0.012733    0.536342    
#> 141: 0.602951    0.821321    3.382907    1.440374    0.057981    0.012097    0.525435    
#> 142: 0.572872    0.830970    3.392179    1.368356    0.060995    0.011492    0.529394    
#> 143: 0.545427    0.834946    3.381136    1.299938    0.057945    0.012145    0.539032    
#> 144: 0.527388    0.841475    3.386464    1.234941    0.055048    0.014078    0.528029    
#> 145: 0.503153    0.840092    3.369198    1.173194    0.053690    0.013374    0.526378    
#> 146: 0.527264    0.855369    3.370849    1.114534    0.062620    0.012705    0.519121    
#> 147: 0.481101    0.847773    3.370499    1.058808    0.059489    0.013753    0.517034    
#> 148: 0.469216    0.852205    3.381912    1.005867    0.056515    0.013066    0.516655    
#> 149: 0.489283    0.828956    3.387069    0.955574    0.053689    0.013463    0.517853    
#> 150: 0.501956    0.833959    3.374983    0.944268    0.060905    0.014120    0.525010    
#> 151: 0.478496    0.855951    3.374349    0.954980    0.057859    0.013414    0.509986    
#> 152: 0.500336    0.856410    3.371937    1.014309    0.054991    0.011710    0.509320    
#> 153: 0.495812    0.856689    3.376604    1.067009    0.054673    0.010879    0.521775    
#> 154: 0.513142    0.865794    3.378369    1.055509    0.056470    0.011731    0.522264    
#> 155: 0.535212    0.844187    3.374640    1.150954    0.054309    0.010660    0.518109    
#> 156: 0.544246    0.845876    3.385223    1.069107    0.051544    0.013363    0.532734    
#> 157: 0.520664    0.828186    3.385632    1.058282    0.062668    0.015330    0.524923    
#> 158: 0.530759    0.840965    3.378858    1.039019    0.062280    0.012666    0.524137    
#> 159: 0.538649    0.840270    3.383920    1.060782    0.060768    0.015154    0.528411    
#> 160: 0.579200    0.813676    3.389395    1.308755    0.059138    0.017092    0.527545    
#> 161: 0.577733    0.831150    3.384988    1.257118    0.069086    0.015798    0.533406    
#> 162: 0.598766    0.839080    3.373250    1.402001    0.070467    0.013628    0.537606    
#> 163: 0.602375    0.840223    3.373043    1.469362    0.067509    0.014212    0.529024    
#> 164: 0.593418    0.864364    3.369377    1.363603    0.062814    0.012204    0.520991    
#> 165: 0.578195    0.850965    3.374142    1.307823    0.064098    0.012292    0.521336    
#> 166: 0.638874    0.866425    3.373525    1.484470    0.060909    0.012028    0.527527    
#> 167: 0.596045    0.864665    3.372180    1.341942    0.059339    0.010766    0.531233    
#> 168: 0.538022    0.858909    3.375997    1.153309    0.063331    0.010820    0.524258    
#> 169: 0.566716    0.880265    3.374812    1.142972    0.063589    0.011262    0.539710    
#> 170: 0.520787    0.859243    3.378110    1.042700    0.057824    0.012950    0.530038    
#> 171: 0.496420    0.853740    3.369056    1.058787    0.068858    0.010799    0.535649    
#> 172: 0.497445    0.851365    3.367933    1.007421    0.071130    0.009422    0.526641    
#> 173: 0.504006    0.864370    3.361803    1.037669    0.069196    0.011362    0.521715    
#> 174: 0.533232    0.833818    3.376410    1.104375    0.060718    0.010784    0.514303    
#> 175: 0.479255    0.836058    3.362124    1.024355    0.057742    0.011578    0.531478    
#> 176: 0.534096    0.852883    3.379573    1.131397    0.059221    0.011613    0.513598    
#> 177: 0.544374    0.842207    3.374388    1.165519    0.052204    0.010799    0.529095    
#> 178: 0.499227    0.853985    3.369275    1.006261    0.052340    0.009995    0.536787    
#> 179: 0.515645    0.868995    3.368986    0.968979    0.050422    0.011833    0.529584    
#> 180: 0.472034    0.861509    3.374349    0.888158    0.052720    0.011731    0.530311    
#> 181: 0.522887    0.854713    3.360948    1.081780    0.052471    0.012876    0.522149    
#> 182: 0.445317    0.842249    3.360269    0.899937    0.055436    0.012118    0.533026    
#> 183: 0.518968    0.865268    3.354138    1.126628    0.048467    0.014251    0.522570    
#> 184: 0.522960    0.860837    3.367536    1.116283    0.054576    0.014637    0.508507    
#> 185: 0.508932    0.874514    3.363514    1.143469    0.057779    0.015798    0.524013    
#> 186: 0.481188    0.870322    3.361702    1.057831    0.056790    0.015399    0.547158    
#> 187: 0.554517    0.844602    3.378242    1.209533    0.046541    0.013272    0.537071    
#> 188: 0.530533    0.855386    3.376389    1.190530    0.048211    0.013646    0.539422    
#> 189: 0.536463    0.862918    3.378601    1.117939    0.051282    0.011275    0.537443    
#> 190: 0.483957    0.863189    3.370970    1.023183    0.050186    0.011443    0.533387    
#> 191: 0.560417    0.855972    3.374357    1.194725    0.047986    0.012717    0.515548    
#> 192: 0.532960    0.856971    3.373427    1.115412    0.044775    0.012897    0.522526    
#> 193: 0.568442    0.860832    3.375435    1.267410    0.048251    0.013645    0.519664    
#> 194: 0.560775    0.860349    3.368365    1.229383    0.055259    0.015125    0.520676    
#> 195: 0.483197    0.851367    3.367136    1.016945    0.053526    0.014902    0.522282    
#> 196: 0.514699    0.843993    3.370359    1.118577    0.052678    0.013570    0.512438    
#> 197: 0.492289    0.847103    3.379423    1.051559    0.056772    0.012776    0.516882    
#> 198: 0.544142    0.836275    3.375011    1.162676    0.067557    0.011667    0.516267    
#> 199: 0.512869    0.837665    3.371913    1.160775    0.060210    0.011875    0.519754    
#> 200: 0.501521    0.845365    3.368581    1.043770    0.065083    0.010710    0.530114    
#> 201: 0.487286    0.851896    3.371905    0.974320    0.064159    0.010729    0.527463    
#> 202: 0.490790    0.856692    3.371472    0.984599    0.062554    0.011317    0.529126    
#> 203: 0.507117    0.858014    3.370042    1.025399    0.062397    0.010875    0.527065    
#> 204: 0.510681    0.854156    3.369833    1.049719    0.062873    0.011038    0.527718    
#> 205: 0.517332    0.856800    3.369177    1.081855    0.064163    0.011150    0.530060    
#> 206: 0.524300    0.858488    3.369183    1.119316    0.064026    0.011200    0.531102    
#> 207: 0.524418    0.858097    3.368521    1.146913    0.063364    0.011175    0.531478    
#> 208: 0.524823    0.859059    3.367807    1.155540    0.063225    0.010968    0.530711    
#> 209: 0.527755    0.857575    3.367039    1.178847    0.062273    0.010861    0.531269    
#> 210: 0.527034    0.855525    3.367733    1.166250    0.062246    0.010863    0.531785    
#> 211: 0.528436    0.856041    3.368193    1.165469    0.061225    0.010934    0.531102    
#> 212: 0.531600    0.856318    3.368554    1.169728    0.059805    0.011191    0.531408    
#> 213: 0.533003    0.854450    3.369390    1.166656    0.059212    0.011457    0.531741    
#> 214: 0.538777    0.853113    3.369919    1.178904    0.058209    0.011512    0.531968    
#> 215: 0.540911    0.851888    3.370935    1.176562    0.057938    0.011525    0.531398    
#> 216: 0.545804    0.849766    3.371849    1.188602    0.057741    0.011522    0.531406    
#> 217: 0.548851    0.849314    3.372468    1.196265    0.057415    0.011605    0.530382    
#> 218: 0.548794    0.849231    3.372589    1.195872    0.057461    0.011658    0.529864    
#> 219: 0.548046    0.848950    3.371980    1.192105    0.057626    0.011607    0.529544    
#> 220: 0.543418    0.849919    3.371646    1.179615    0.057851    0.011600    0.529383    
#> 221: 0.539378    0.850474    3.371147    1.170766    0.057948    0.011620    0.529560    
#> 222: 0.540418    0.850650    3.370511    1.178982    0.058081    0.011716    0.529152    
#> 223: 0.541821    0.851237    3.370199    1.185739    0.058347    0.011791    0.528823    
#> 224: 0.543737    0.851487    3.370218    1.190631    0.058285    0.011834    0.528581    
#> 225: 0.543910    0.850471    3.370768    1.187499    0.058515    0.011756    0.528709    
#> 226: 0.545711    0.848940    3.370908    1.191153    0.058229    0.011750    0.528593    
#> 227: 0.544619    0.848892    3.370835    1.187537    0.058613    0.011700    0.527947    
#> 228: 0.542491    0.849316    3.370593    1.180050    0.058693    0.011676    0.527615    
#> 229: 0.541171    0.849309    3.370556    1.174590    0.058680    0.011687    0.527094    
#> 230: 0.539626    0.848833    3.370572    1.166908    0.058391    0.011667    0.527151    
#> 231: 0.538286    0.848922    3.370843    1.159666    0.058269    0.011653    0.526671    
#> 232: 0.537774    0.848946    3.371281    1.155630    0.058009    0.011659    0.526063    
#> 233: 0.539515    0.848492    3.371519    1.162728    0.058081    0.011647    0.525801    
#> 234: 0.539906    0.848443    3.371702    1.163439    0.058314    0.011647    0.525472    
#> 235: 0.540563    0.848145    3.371729    1.163376    0.058241    0.011617    0.525330    
#> 236: 0.540307    0.848448    3.371636    1.161739    0.058001    0.011595    0.525227    
#> 237: 0.539831    0.848350    3.371548    1.158655    0.057885    0.011565    0.525423    
#> 238: 0.539558    0.848073    3.371671    1.157062    0.057956    0.011560    0.525883    
#> 239: 0.539456    0.848169    3.371637    1.156060    0.057860    0.011544    0.526144    
#> 240: 0.538558    0.847929    3.371767    1.153548    0.057968    0.011527    0.525876    
#> 241: 0.538615    0.847317    3.371942    1.151785    0.057973    0.011527    0.525490    
#> 242: 0.538762    0.846788    3.372013    1.151128    0.058080    0.011519    0.525693    
#> 243: 0.538622    0.847141    3.372099    1.148810    0.058199    0.011518    0.525871    
#> 244: 0.537863    0.846850    3.372297    1.143711    0.057957    0.011549    0.526172    
#> 245: 0.537894    0.846583    3.372398    1.141510    0.057591    0.011546    0.526825    
#> 246: 0.538592    0.846359    3.372518    1.142358    0.057328    0.011587    0.526854    
#> 247: 0.538741    0.846051    3.372808    1.142321    0.057167    0.011633    0.526732    
#> 248: 0.538563    0.846319    3.372824    1.141225    0.057233    0.011626    0.526450    
#> 249: 0.537769    0.846447    3.372823    1.138500    0.057206    0.011627    0.526563    
#> 250: 0.537829    0.846640    3.372810    1.137890    0.057117    0.011638    0.526664    
#> 251: 0.538615    0.846803    3.372951    1.140760    0.057193    0.011624    0.526594    
#> 252: 0.539011    0.846944    3.373266    1.141302    0.057194    0.011599    0.526722    
#> 253: 0.539321    0.846841    3.373443    1.140979    0.057216    0.011614    0.526808    
#> 254: 0.540107    0.847054    3.373693    1.142394    0.057258    0.011648    0.526780    
#> 255: 0.540453    0.847002    3.373872    1.143157    0.057281    0.011686    0.526828    
#> 256: 0.540003    0.846766    3.373959    1.140203    0.057380    0.011663    0.526892    
#> 257: 0.538948    0.846361    3.374110    1.135429    0.057392    0.011627    0.526997    
#> 258: 0.538902    0.845940    3.374318    1.133932    0.057323    0.011656    0.526883    
#> 259: 0.539183    0.845707    3.374474    1.133230    0.057256    0.011661    0.526925    
#> 260: 0.539191    0.845707    3.374677    1.130184    0.057298    0.011657    0.526912    
#> 261: 0.539167    0.845610    3.374949    1.127339    0.057189    0.011685    0.526757    
#> 262: 0.539755    0.845662    3.375188    1.128621    0.057245    0.011698    0.526678    
#> 263: 0.539629    0.845586    3.375369    1.126757    0.057385    0.011712    0.526767    
#> 264: 0.539734    0.845636    3.375516    1.125763    0.057441    0.011740    0.526886    
#> 265: 0.540774    0.845531    3.375689    1.127954    0.057358    0.011756    0.527149    
#> 266: 0.541780    0.845597    3.375817    1.130726    0.057294    0.011736    0.527364    
#> 267: 0.543145    0.845351    3.375968    1.135213    0.057387    0.011738    0.527429    
#> 268: 0.543978    0.844873    3.376071    1.136387    0.057372    0.011741    0.527417    
#> 269: 0.544840    0.844545    3.376222    1.137723    0.057345    0.011760    0.527540    
#> 270: 0.546155    0.844458    3.376288    1.141661    0.057321    0.011780    0.527445    
#> 271: 0.546434    0.844354    3.376311    1.143791    0.057321    0.011823    0.527646    
#> 272: 0.547129    0.844116    3.376338    1.145654    0.057241    0.011850    0.527779    
#> 273: 0.547697    0.844109    3.376374    1.149761    0.057099    0.011888    0.527909    
#> 274: 0.547824    0.844180    3.376448    1.150321    0.057234    0.011889    0.527961    
#> 275: 0.547142    0.844285    3.376382    1.148032    0.057312    0.011874    0.527836    
#> 276: 0.547438    0.844534    3.376436    1.149917    0.057302    0.011892    0.527853    
#> 277: 0.547170    0.844619    3.376560    1.149421    0.057216    0.011914    0.527809    
#> 278: 0.546530    0.844580    3.376732    1.146815    0.057231    0.011924    0.527810    
#> 279: 0.546409    0.844408    3.376862    1.146107    0.057209    0.011927    0.528057    
#> 280: 0.546175    0.844414    3.376940    1.145156    0.057074    0.011936    0.528015    
#> 281: 0.545994    0.844346    3.377076    1.143951    0.057004    0.011933    0.527862    
#> 282: 0.546465    0.844390    3.377185    1.145418    0.056953    0.011960    0.527751    
#> 283: 0.546460    0.844440    3.377213    1.145234    0.056899    0.011997    0.527718    
#> 284: 0.546675    0.844289    3.377291    1.145757    0.056874    0.012038    0.527554    
#> 285: 0.546596    0.844179    3.377309    1.144880    0.056772    0.012075    0.527556    
#> 286: 0.546367    0.844228    3.377276    1.143173    0.056637    0.012089    0.527411    
#> 287: 0.546178    0.844238    3.377266    1.141807    0.056446    0.012099    0.527516    
#> 288: 0.546313    0.844144    3.377349    1.141035    0.056401    0.012117    0.527694    
#> 289: 0.546498    0.844001    3.377416    1.141054    0.056277    0.012136    0.527657    
#> 290: 0.546926    0.843877    3.377544    1.141977    0.056196    0.012187    0.527665    
#> 291: 0.547844    0.844017    3.377715    1.144799    0.056107    0.012224    0.527616    
#> 292: 0.548076    0.843962    3.377769    1.145364    0.056016    0.012223    0.527636    
#> 293: 0.548279    0.844038    3.377770    1.146508    0.055963    0.012222    0.527640    
#> 294: 0.548492    0.844074    3.377714    1.147135    0.055879    0.012220    0.527598    
#> 295: 0.548658    0.843962    3.377687    1.148653    0.055892    0.012208    0.527620    
#> 296: 0.548106    0.843960    3.377636    1.147192    0.055878    0.012215    0.527716    
#> 297: 0.547472    0.843981    3.377577    1.145568    0.055843    0.012215    0.527710    
#> 298: 0.546999    0.844051    3.377480    1.144506    0.055813    0.012229    0.527629    
#> 299: 0.546604    0.844198    3.377330    1.144799    0.055766    0.012212    0.527631    
#> 300: 0.546182    0.844206    3.377243    1.144415    0.055678    0.012216    0.527750    
#> 301: 0.545841    0.844163    3.377117    1.144360    0.055613    0.012233    0.527860    
#> 302: 0.545508    0.844342    3.376924    1.143407    0.055622    0.012253    0.527921    
#> 303: 0.544945    0.844529    3.376815    1.141616    0.055511    0.012279    0.527747    
#> 304: 0.545132    0.844833    3.376743    1.142262    0.055496    0.012286    0.527807    
#> 305: 0.545156    0.844720    3.376728    1.142233    0.055465    0.012300    0.527907    
#> 306: 0.544677    0.844521    3.376674    1.141631    0.055402    0.012308    0.527871    
#> 307: 0.544123    0.844409    3.376668    1.141258    0.055280    0.012319    0.527886    
#> 308: 0.543849    0.844334    3.376611    1.141625    0.055226    0.012327    0.527836    
#> 309: 0.544000    0.844118    3.376626    1.142445    0.055089    0.012352    0.528038    
#> 310: 0.543802    0.844080    3.376580    1.141903    0.055037    0.012354    0.528065    
#> 311: 0.543799    0.844115    3.376577    1.142036    0.054910    0.012381    0.528126    
#> 312: 0.544534    0.844080    3.376581    1.145359    0.054736    0.012398    0.528193    
#> 313: 0.544855    0.843937    3.376594    1.146333    0.054672    0.012421    0.528208    
#> 314: 0.544638    0.843961    3.376607    1.145284    0.054613    0.012438    0.528233    
#> 315: 0.544741    0.844002    3.376635    1.145509    0.054587    0.012487    0.528152    
#> 316: 0.545437    0.844131    3.376654    1.147866    0.054507    0.012512    0.528072    
#> 317: 0.545432    0.844113    3.376653    1.148236    0.054439    0.012535    0.528093    
#> 318: 0.545254    0.844079    3.376586    1.147930    0.054389    0.012558    0.528058    
#> 319: 0.545233    0.844067    3.376561    1.147537    0.054376    0.012573    0.528081    
#> 320: 0.544908    0.844130    3.376508    1.147154    0.054315    0.012579    0.528072    
#> 321: 0.545045    0.844334    3.376464    1.147789    0.054269    0.012569    0.528122    
#> 322: 0.545038    0.844463    3.376405    1.148535    0.054264    0.012558    0.527990    
#> 323: 0.544742    0.844490    3.376377    1.148088    0.054212    0.012558    0.527966    
#> 324: 0.544504    0.844509    3.376404    1.147586    0.054237    0.012553    0.527837    
#> 325: 0.544638    0.844539    3.376397    1.148304    0.054249    0.012549    0.527674    
#> 326: 0.544739    0.844367    3.376389    1.148775    0.054287    0.012551    0.527608    
#> 327: 0.545025    0.844275    3.376410    1.150011    0.054305    0.012553    0.527545    
#> 328: 0.545235    0.844347    3.376448    1.150630    0.054340    0.012548    0.527484    
#> 329: 0.545359    0.844309    3.376494    1.151184    0.054383    0.012547    0.527476    
#> 330: 0.545446    0.844252    3.376552    1.151497    0.054435    0.012559    0.527394    
#> 331: 0.545614    0.844388    3.376662    1.152026    0.054454    0.012559    0.527286    
#> 332: 0.546382    0.844350    3.376725    1.153957    0.054412    0.012582    0.527143    
#> 333: 0.546433    0.844300    3.376717    1.154372    0.054413    0.012583    0.527063    
#> 334: 0.545987    0.844135    3.376666    1.153819    0.054400    0.012574    0.527116    
#> 335: 0.546004    0.844129    3.376548    1.154966    0.054377    0.012575    0.527171    
#> 336: 0.545658    0.844276    3.376415    1.155254    0.054359    0.012582    0.527189    
#> 337: 0.545449    0.844551    3.376295    1.155502    0.054390    0.012589    0.527188    
#> 338: 0.545290    0.844615    3.376232    1.156005    0.054391    0.012599    0.527142    
#> 339: 0.545013    0.844688    3.376184    1.155994    0.054386    0.012629    0.527068    
#> 340: 0.544706    0.844554    3.376153    1.155943    0.054427    0.012649    0.527042    
#> 341: 0.544543    0.844498    3.376163    1.155252    0.054432    0.012661    0.527025    
#> 342: 0.544056    0.844403    3.376139    1.153620    0.054412    0.012673    0.527033    
#> 343: 0.543778    0.844307    3.376157    1.152425    0.054347    0.012694    0.526949    
#> 344: 0.544350    0.844318    3.376222    1.154831    0.054405    0.012711    0.526813    
#> 345: 0.544505    0.844262    3.376252    1.155298    0.054420    0.012731    0.526781    
#> 346: 0.544645    0.844188    3.376204    1.154983    0.054395    0.012741    0.526827    
#> 347: 0.544506    0.844153    3.376123    1.154717    0.054353    0.012748    0.526852    
#> 348: 0.543971    0.844182    3.376020    1.153954    0.054345    0.012768    0.526802    
#> 349: 0.543730    0.844139    3.375957    1.154445    0.054427    0.012778    0.526724    
#> 350: 0.543222    0.844008    3.375881    1.153399    0.054447    0.012793    0.526729    
#> 351: 0.542797    0.843979    3.375814    1.152563    0.054396    0.012802    0.526752    
#> 352: 0.542266    0.844013    3.375786    1.151325    0.054363    0.012821    0.526839    
#> 353: 0.542446    0.844044    3.375805    1.151443    0.054369    0.012821    0.526827    
#> 354: 0.542336    0.844015    3.375728    1.151478    0.054331    0.012821    0.526821    
#> 355: 0.542598    0.843877    3.375696    1.152356    0.054332    0.012830    0.526804    
#> 356: 0.542790    0.843785    3.375741    1.151903    0.054351    0.012853    0.526848    
#> 357: 0.542992    0.843824    3.375781    1.152105    0.054360    0.012860    0.526754    
#> 358: 0.543043    0.843859    3.375826    1.151636    0.054374    0.012866    0.526659    
#> 359: 0.542929    0.843955    3.375804    1.150708    0.054406    0.012871    0.526566    
#> 360: 0.542790    0.843981    3.375730    1.150153    0.054414    0.012867    0.526512    
#> 361: 0.542603    0.844009    3.375683    1.149323    0.054429    0.012857    0.526402    
#> 362: 0.542538    0.844012    3.375649    1.148929    0.054432    0.012842    0.526267    
#> 363: 0.542711    0.843915    3.375688    1.149063    0.054431    0.012842    0.526151    
#> 364: 0.542628    0.843881    3.375678    1.148170    0.054480    0.012840    0.526127    
#> 365: 0.542447    0.843841    3.375703    1.147258    0.054484    0.012835    0.526091    
#> 366: 0.542258    0.843802    3.375670    1.146575    0.054462    0.012834    0.526085    
#> 367: 0.542144    0.843785    3.375637    1.145940    0.054480    0.012827    0.526049    
#> 368: 0.541957    0.843883    3.375568    1.145114    0.054494    0.012826    0.525998    
#> 369: 0.541831    0.843943    3.375502    1.145240    0.054468    0.012822    0.526016    
#> 370: 0.541687    0.843974    3.375462    1.145065    0.054474    0.012809    0.525995    
#> 371: 0.541666    0.843988    3.375422    1.145455    0.054443    0.012805    0.525934    
#> 372: 0.541543    0.843964    3.375435    1.145061    0.054450    0.012793    0.525875    
#> 373: 0.541079    0.844008    3.375407    1.143720    0.054491    0.012783    0.525810    
#> 374: 0.540919    0.844037    3.375384    1.143129    0.054495    0.012785    0.525681    
#> 375: 0.540826    0.844142    3.375389    1.142216    0.054544    0.012786    0.525623    
#> 376: 0.540671    0.844206    3.375413    1.141169    0.054588    0.012774    0.525622    
#> 377: 0.540494    0.844288    3.375374    1.140189    0.054604    0.012782    0.525564    
#> 378: 0.540270    0.844318    3.375334    1.139929    0.054569    0.012795    0.525515    
#> 379: 0.540076    0.844232    3.375305    1.139699    0.054558    0.012816    0.525493    
#> 380: 0.539830    0.844199    3.375289    1.139287    0.054554    0.012825    0.525447    
#> 381: 0.539552    0.844207    3.375284    1.138471    0.054534    0.012818    0.525475    
#> 382: 0.539557    0.844167    3.375305    1.137859    0.054521    0.012815    0.525349    
#> 383: 0.539676    0.844196    3.375283    1.137837    0.054528    0.012819    0.525273    
#> 384: 0.539354    0.844233    3.375262    1.137301    0.054525    0.012814    0.525282    
#> 385: 0.539282    0.844277    3.375244    1.137203    0.054506    0.012831    0.525229    
#> 386: 0.539384    0.844390    3.375236    1.137512    0.054473    0.012835    0.525258    
#> 387: 0.539330    0.844507    3.375237    1.137428    0.054444    0.012851    0.525259    
#> 388: 0.539409    0.844567    3.375237    1.137703    0.054426    0.012869    0.525279    
#> 389: 0.539184    0.844511    3.375176    1.137720    0.054429    0.012887    0.525406    
#> 390: 0.538894    0.844597    3.375082    1.137019    0.054400    0.012894    0.525411    
#> 391: 0.538651    0.844593    3.375000    1.136634    0.054377    0.012901    0.525374    
#> 392: 0.538784    0.844637    3.374970    1.137065    0.054365    0.012917    0.525279    
#> 393: 0.539023    0.844630    3.374925    1.138462    0.054370    0.012916    0.525226    
#> 394: 0.539294    0.844699    3.374905    1.139853    0.054347    0.012915    0.525240    
#> 395: 0.539470    0.844813    3.374879    1.141453    0.054347    0.012918    0.525297    
#> 396: 0.539558    0.844942    3.374874    1.142439    0.054306    0.012922    0.525324    
#> 397: 0.539362    0.844954    3.374847    1.142719    0.054274    0.012934    0.525354    
#> 398: 0.539395    0.844930    3.374889    1.142989    0.054229    0.012955    0.525419    
#> 399: 0.539707    0.844905    3.374945    1.143794    0.054179    0.012963    0.525388    
#> 400: 0.539667    0.844962    3.374965    1.143448    0.054200    0.012961    0.525367    
#> 401: 0.539511    0.845038    3.374963    1.142686    0.054187    0.012964    0.525414    
#> 402: 0.539332    0.845068    3.374944    1.142109    0.054157    0.012974    0.525402    
#> 403: 0.539400    0.845103    3.374942    1.142312    0.054147    0.012983    0.525387    
#> 404: 0.539505    0.845132    3.374949    1.142180    0.054150    0.012979    0.525349    
#> 405: 0.539418    0.845143    3.374973    1.141856    0.054157    0.012979    0.525330    
#> 406: 0.539087    0.845237    3.374922    1.141054    0.054184    0.012973    0.525259    
#> 407: 0.538913    0.845256    3.374922    1.140693    0.054177    0.012971    0.525180    
#> 408: 0.538889    0.845289    3.374905    1.140071    0.054150    0.012969    0.525161    
#> 409: 0.538640    0.845222    3.374880    1.139178    0.054158    0.012969    0.525136    
#> 410: 0.538473    0.845228    3.374887    1.138443    0.054178    0.012962    0.525139    
#> 411: 0.538623    0.845146    3.374863    1.139093    0.054160    0.012954    0.525186    
#> 412: 0.538401    0.845183    3.374850    1.138498    0.054126    0.012955    0.525244    
#> 413: 0.538358    0.845253    3.374826    1.138222    0.054142    0.012959    0.525248    
#> 414: 0.538498    0.845294    3.374849    1.139334    0.054144    0.012963    0.525253    
#> 415: 0.538480    0.845255    3.374830    1.139370    0.054123    0.012956    0.525248    
#> 416: 0.538344    0.845278    3.374858    1.138590    0.054134    0.012955    0.525269    
#> 417: 0.538580    0.845267    3.374884    1.139408    0.054143    0.012953    0.525193    
#> 418: 0.538546    0.845202    3.374895    1.139492    0.054155    0.012962    0.525163    
#> 419: 0.538862    0.845163    3.374903    1.140701    0.054156    0.012974    0.525105    
#> 420: 0.539093    0.845165    3.374900    1.141515    0.054140    0.012980    0.525053    
#> 421: 0.539014    0.845220    3.374886    1.141666    0.054201    0.012987    0.524999    
#> 422: 0.539022    0.845223    3.374828    1.142106    0.054233    0.012993    0.524953    
#> 423: 0.538801    0.845212    3.374787    1.141522    0.054226    0.012996    0.524905    
#> 424: 0.538492    0.845222    3.374732    1.140747    0.054228    0.013001    0.524838    
#> 425: 0.538331    0.845358    3.374671    1.140531    0.054229    0.013002    0.524889    
#> 426: 0.538159    0.845427    3.374628    1.140771    0.054229    0.012995    0.524866    
#> 427: 0.538089    0.845388    3.374620    1.141177    0.054189    0.013009    0.524832    
#> 428: 0.537849    0.845442    3.374597    1.140973    0.054142    0.013019    0.524835    
#> 429: 0.537656    0.845498    3.374560    1.140754    0.054101    0.013030    0.524830    
#> 430: 0.537460    0.845536    3.374493    1.140750    0.054115    0.013047    0.524822    
#> 431: 0.537385    0.845573    3.374436    1.141258    0.054071    0.013047    0.524753    
#> 432: 0.537423    0.845675    3.374391    1.142188    0.054049    0.013048    0.524724    
#> 433: 0.537407    0.845816    3.374342    1.142598    0.054043    0.013056    0.524736    
#> 434: 0.537196    0.845798    3.374288    1.142321    0.053998    0.013071    0.524745    
#> 435: 0.537258    0.845794    3.374265    1.142633    0.053937    0.013088    0.524754    
#> 436: 0.537415    0.845774    3.374268    1.143366    0.053885    0.013100    0.524829    
#> 437: 0.537522    0.845721    3.374285    1.143584    0.053846    0.013115    0.524936    
#> 438: 0.537428    0.845781    3.374285    1.143161    0.053822    0.013123    0.524943    
#> 439: 0.537572    0.845849    3.374290    1.143410    0.053815    0.013125    0.524874    
#> 440: 0.537592    0.845859    3.374320    1.142957    0.053767    0.013128    0.524817    
#> 441: 0.537892    0.845798    3.374384    1.143266    0.053756    0.013133    0.524782    
#> 442: 0.537929    0.845825    3.374409    1.143306    0.053755    0.013138    0.524762    
#> 443: 0.538058    0.845878    3.374399    1.143672    0.053752    0.013154    0.524761    
#> 444: 0.538030    0.845979    3.374362    1.143832    0.053755    0.013168    0.524735    
#> 445: 0.537970    0.846035    3.374302    1.144139    0.053705    0.013193    0.524718    
#> 446: 0.537809    0.846074    3.374250    1.144154    0.053644    0.013205    0.524675    
#> 447: 0.537725    0.846142    3.374201    1.144242    0.053592    0.013217    0.524691    
#> 448: 0.537544    0.846196    3.374190    1.144073    0.053559    0.013228    0.524648    
#> 449: 0.537565    0.846265    3.374244    1.143985    0.053531    0.013241    0.524634    
#> 450: 0.537476    0.846271    3.374269    1.143339    0.053481    0.013248    0.524656    
#> 451: 0.537534    0.846258    3.374287    1.143587    0.053472    0.013255    0.524629    
#> 452: 0.537574    0.846297    3.374253    1.143814    0.053450    0.013259    0.524583    
#> 453: 0.537651    0.846306    3.374243    1.144192    0.053456    0.013270    0.524522    
#> 454: 0.537530    0.846371    3.374212    1.143859    0.053444    0.013275    0.524488    
#> 455: 0.537528    0.846370    3.374191    1.143843    0.053445    0.013272    0.524430    
#> 456: 0.537305    0.846354    3.374154    1.143647    0.053475    0.013267    0.524405    
#> 457: 0.537228    0.846318    3.374131    1.143931    0.053471    0.013272    0.524352    
#> 458: 0.537208    0.846393    3.374124    1.144389    0.053449    0.013285    0.524262    
#> 459: 0.537129    0.846472    3.374071    1.144696    0.053459    0.013293    0.524179    
#> 460: 0.536863    0.846469    3.373978    1.144480    0.053478    0.013291    0.524179    
#> 461: 0.536555    0.846482    3.373929    1.143774    0.053508    0.013284    0.524199    
#> 462: 0.536446    0.846508    3.373927    1.143546    0.053510    0.013284    0.524221    
#> 463: 0.536458    0.846530    3.373951    1.143328    0.053518    0.013287    0.524212    
#> 464: 0.536442    0.846554    3.373936    1.142824    0.053537    0.013283    0.524222    
#> 465: 0.536353    0.846489    3.373925    1.142167    0.053557    0.013278    0.524217    
#> 466: 0.536198    0.846511    3.373898    1.141217    0.053556    0.013265    0.524226    
#> 467: 0.536085    0.846474    3.373930    1.140748    0.053534    0.013266    0.524184    
#> 468: 0.536241    0.846478    3.373938    1.141063    0.053552    0.013266    0.524142    
#> 469: 0.536553    0.846497    3.373959    1.141664    0.053537    0.013259    0.524114    
#> 470: 0.536587    0.846490    3.373962    1.141973    0.053510    0.013267    0.524072    
#> 471: 0.536590    0.846474    3.373978    1.141796    0.053491    0.013271    0.524112    
#> 472: 0.536529    0.846423    3.374014    1.141507    0.053475    0.013277    0.524168    
#> 473: 0.536412    0.846353    3.374026    1.140953    0.053444    0.013276    0.524144    
#> 474: 0.536311    0.846358    3.374042    1.140500    0.053471    0.013265    0.524184    
#> 475: 0.536316    0.846344    3.374061    1.140304    0.053452    0.013263    0.524201    
#> 476: 0.536414    0.846313    3.374075    1.140486    0.053483    0.013262    0.524206    
#> 477: 0.536449    0.846282    3.374132    1.140395    0.053500    0.013262    0.524174    
#> 478: 0.536307    0.846268    3.374181    1.139752    0.053483    0.013257    0.524262    
#> 479: 0.536369    0.846262    3.374229    1.139904    0.053488    0.013256    0.524303    
#> 480: 0.536204    0.846284    3.374218    1.139398    0.053460    0.013262    0.524461    
#> 481: 0.536108    0.846322    3.374200    1.139387    0.053477    0.013260    0.524549    
#> 482: 0.536095    0.846382    3.374123    1.139814    0.053487    0.013269    0.524625    
#> 483: 0.536108    0.846406    3.374070    1.140766    0.053479    0.013276    0.524740    
#> 484: 0.536163    0.846333    3.374029    1.141713    0.053486    0.013289    0.524829    
#> 485: 0.536231    0.846364    3.373984    1.142914    0.053532    0.013304    0.524870    
#> 486: 0.536153    0.846380    3.373976    1.143122    0.053511    0.013317    0.524917    
#> 487: 0.536237    0.846367    3.373996    1.143262    0.053490    0.013322    0.524915    
#> 488: 0.536375    0.846360    3.374010    1.143579    0.053464    0.013322    0.524926    
#> 489: 0.536462    0.846349    3.373992    1.143786    0.053464    0.013318    0.524906    
#> 490: 0.536620    0.846365    3.374005    1.144288    0.053449    0.013313    0.524884    
#> 491: 0.536558    0.846442    3.374019    1.144104    0.053443    0.013312    0.524838    
#> 492: 0.536601    0.846518    3.374013    1.143984    0.053424    0.013305    0.524893    
#> 493: 0.536587    0.846542    3.374036    1.143749    0.053406    0.013307    0.524883    
#> 494: 0.536550    0.846564    3.374051    1.143651    0.053383    0.013306    0.524865    
#> 495: 0.536730    0.846586    3.374058    1.144361    0.053347    0.013308    0.524823    
#> 496: 0.536886    0.846637    3.374058    1.144593    0.053330    0.013310    0.524847    
#> 497: 0.537139    0.846633    3.374087    1.145418    0.053331    0.013311    0.524851    
#> 498: 0.537280    0.846603    3.374082    1.145871    0.053350    0.013303    0.524838    
#> 499: 0.537326    0.846571    3.374081    1.145852    0.053359    0.013300    0.524820    
#> 500: 0.537568    0.846553    3.374079    1.146410    0.053377    0.013297    0.524806    
#> [====|====|=>-------------------------------------]  25%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.300722    0.966231    3.449219    0.392401    0.077086    0.016800    1.674400    
#> 002: 0.320726    1.011906    3.467853    0.372781    0.073232    0.016620    1.110443    
#> 003: 0.402562    0.916312    3.478587    0.354142    0.069570    0.015789    0.842467    
#> 004: 0.366787    0.908268    3.471623    0.336435    0.066092    0.014999    0.746414    
#> 005: 0.386746    0.938959    3.457407    0.319613    0.062787    0.014249    0.706308    
#> 006: 0.347968    0.928193    3.442754    0.303633    0.059648    0.013537    0.662091    
#> 007: 0.298640    0.938821    3.430205    0.288451    0.061382    0.014457    0.671553    
#> 008: 0.309842    0.938741    3.430263    0.274029    0.076908    0.013734    0.674261    
#> 009: 0.290885    0.924673    3.430795    0.260327    0.074625    0.013047    0.669802    
#> 010: 0.294850    0.930789    3.428338    0.247311    0.073951    0.012395    0.676105    
#> 011: 0.338716    0.927991    3.432525    0.234945    0.095645    0.011775    0.673623    
#> 012: 0.354531    0.940295    3.424391    0.223198    0.090863    0.011186    0.675773    
#> 013: 0.291937    0.938019    3.422559    0.212038    0.092592    0.010988    0.671438    
#> 014: 0.328663    0.932490    3.435067    0.215983    0.087962    0.010712    0.692451    
#> 015: 0.343994    0.933300    3.444470    0.205184    0.083564    0.010177    0.666779    
#> 016: 0.326283    0.944647    3.440578    0.197234    0.079386    0.009668    0.667233    
#> 017: 0.338012    0.952081    3.434802    0.205080    0.075416    0.010087    0.672676    
#> 018: 0.307014    0.944334    3.423832    0.194826    0.079236    0.010987    0.677917    
#> 019: 0.286734    0.955744    3.423533    0.185085    0.085914    0.010437    0.660911    
#> 020: 0.313108    0.956759    3.428171    0.207384    0.081618    0.009915    0.675195    
#> 021: 0.336127    0.937685    3.443975    0.201380    0.077537    0.009420    0.686493    
#> 022: 0.323405    0.928605    3.445651    0.197566    0.073660    0.009594    0.672043    
#> 023: 0.367165    0.928984    3.453261    0.187688    0.071339    0.009125    0.706479    
#> 024: 0.361372    0.951417    3.452806    0.203028    0.075666    0.008669    0.689283    
#> 025: 0.343362    0.934153    3.439467    0.192877    0.084757    0.008236    0.680560    
#> 026: 0.354018    0.956267    3.438543    0.211734    0.083553    0.007968    0.665886    
#> 027: 0.316209    0.980121    3.428147    0.201147    0.079375    0.009560    0.672750    
#> 028: 0.326918    0.948008    3.430020    0.218591    0.075460    0.009082    0.669121    
#> 029: 0.335918    0.958069    3.424813    0.207661    0.096557    0.008628    0.699211    
#> 030: 0.316452    0.964107    3.424352    0.200404    0.091729    0.008196    0.670538    
#> 031: 0.293795    0.945021    3.430518    0.190384    0.087629    0.007787    0.673544    
#> 032: 0.297849    0.933525    3.433680    0.222642    0.083248    0.007830    0.683104    
#> 033: 0.302096    0.945797    3.424987    0.224072    0.079085    0.007536    0.686390    
#> 034: 0.314160    0.960268    3.425197    0.212869    0.077557    0.007209    0.677334    
#> 035: 0.287021    0.943173    3.422968    0.202225    0.093874    0.006849    0.678534    
#> 036: 0.299417    0.959670    3.420044    0.192114    0.089180    0.007109    0.683343    
#> 037: 0.287132    0.976756    3.423828    0.182508    0.084721    0.006927    0.694247    
#> 038: 0.298075    0.961520    3.413731    0.173383    0.096143    0.006581    0.712628    
#> 039: 0.330598    0.933783    3.436115    0.164714    0.099864    0.006872    0.688778    
#> 040: 0.292493    0.948846    3.436985    0.168978    0.094871    0.007652    0.677165    
#> 041: 0.291992    0.932008    3.434366    0.160529    0.098884    0.007541    0.688599    
#> 042: 0.297430    0.940305    3.426436    0.167544    0.093940    0.007164    0.666314    
#> 043: 0.271795    0.947341    3.413550    0.159284    0.089243    0.007402    0.670698    
#> 044: 0.262613    0.960145    3.415137    0.174031    0.084781    0.007032    0.672461    
#> 045: 0.259974    0.980477    3.413838    0.165330    0.080542    0.006680    0.688390    
#> 046: 0.294034    0.968251    3.417410    0.164997    0.076515    0.006346    0.691418    
#> 047: 0.319919    0.948639    3.428774    0.172727    0.077219    0.006409    0.676157    
#> 048: 0.309746    0.947072    3.432789    0.164262    0.076513    0.008820    0.677508    
#> 049: 0.295877    0.954064    3.437198    0.156049    0.081032    0.008837    0.669569    
#> 050: 0.367294    0.920105    3.438292    0.159934    0.086879    0.008395    0.697692    
#> 051: 0.329802    0.919499    3.437993    0.154161    0.082535    0.009925    0.685415    
#> 052: 0.350661    0.941479    3.450735    0.148462    0.078408    0.009429    0.661081    
#> 053: 0.369634    0.928745    3.442876    0.193581    0.074488    0.008958    0.672793    
#> 054: 0.319198    0.922834    3.446380    0.196189    0.079848    0.008510    0.681051    
#> 055: 0.333196    0.924508    3.445262    0.186380    0.086854    0.008084    0.666832    
#> 056: 0.363992    0.927110    3.448564    0.211141    0.099035    0.009084    0.679974    
#> 057: 0.369547    0.920492    3.458967    0.217556    0.102150    0.008630    0.701209    
#> 058: 0.375286    0.917783    3.442822    0.206678    0.097043    0.008198    0.705846    
#> 059: 0.372782    0.892239    3.452158    0.203009    0.108236    0.007788    0.693319    
#> 060: 0.362011    0.878411    3.461159    0.210534    0.102825    0.010355    0.705445    
#> 061: 0.398465    0.895829    3.464539    0.200007    0.097683    0.012261    0.714846    
#> 062: 0.373599    0.917363    3.464491    0.192035    0.107081    0.011648    0.710545    
#> 063: 0.380683    0.883931    3.457771    0.200934    0.101727    0.011074    0.702181    
#> 064: 0.351854    0.913222    3.463012    0.190887    0.096641    0.010520    0.713404    
#> 065: 0.359030    0.888641    3.456283    0.181343    0.091809    0.009994    0.705609    
#> 066: 0.350247    0.925327    3.458736    0.172276    0.099638    0.009494    0.671900    
#> 067: 0.330083    0.903920    3.443520    0.170452    0.094656    0.009051    0.684330    
#> 068: 0.311185    0.937154    3.442147    0.164351    0.099770    0.008598    0.681487    
#> 069: 0.337156    0.930148    3.442392    0.177585    0.094781    0.008168    0.693095    
#> 070: 0.342731    0.930297    3.450256    0.168706    0.090042    0.007760    0.694413    
#> 071: 0.327780    0.959023    3.424988    0.162729    0.085540    0.008128    0.698571    
#> 072: 0.288965    0.936831    3.427803    0.154593    0.089312    0.010161    0.675145    
#> 073: 0.338015    0.930034    3.453873    0.146863    0.084846    0.010244    0.684793    
#> 074: 0.329500    0.917524    3.448911    0.150542    0.080604    0.010113    0.688201    
#> 075: 0.343186    0.908726    3.466087    0.153264    0.097943    0.009607    0.685869    
#> 076: 0.355331    0.918298    3.470037    0.145600    0.093046    0.009127    0.685532    
#> 077: 0.386103    0.911932    3.453252    0.145946    0.088393    0.008670    0.686824    
#> 078: 0.381931    0.925594    3.458207    0.163709    0.083974    0.008237    0.678187    
#> 079: 0.403640    0.919356    3.462706    0.183904    0.080349    0.007911    0.684859    
#> 080: 0.373072    0.918860    3.452859    0.177444    0.094554    0.008644    0.707851    
#> 081: 0.330172    0.922619    3.447548    0.168572    0.089827    0.008880    0.702301    
#> 082: 0.351242    0.902954    3.458152    0.168637    0.085335    0.008436    0.705258    
#> 083: 0.361312    0.929827    3.459085    0.160205    0.082647    0.008014    0.696555    
#> 084: 0.377519    0.910263    3.460499    0.193102    0.081997    0.009266    0.669016    
#> 085: 0.353265    0.927738    3.463436    0.183447    0.079475    0.009761    0.679716    
#> 086: 0.346814    0.938478    3.455149    0.174274    0.084760    0.010621    0.668189    
#> 087: 0.340997    0.925580    3.445154    0.177268    0.080522    0.010090    0.667046    
#> 088: 0.312109    0.935321    3.436850    0.173956    0.076496    0.009585    0.676239    
#> 089: 0.340089    0.937031    3.439254    0.165258    0.086200    0.009656    0.672725    
#> 090: 0.314396    0.937286    3.440777    0.162750    0.081890    0.009173    0.666663    
#> 091: 0.291683    0.913763    3.429215    0.160441    0.077796    0.008715    0.679656    
#> 092: 0.267654    0.946560    3.421015    0.181959    0.076740    0.009256    0.677938    
#> 093: 0.275359    0.947265    3.415484    0.225779    0.083451    0.008793    0.681688    
#> 094: 0.269676    0.967710    3.405517    0.214490    0.097062    0.008353    0.683262    
#> 095: 0.281081    0.977045    3.414439    0.214020    0.092209    0.007936    0.673889    
#> 096: 0.256855    0.968397    3.417584    0.203319    0.087598    0.007539    0.681579    
#> 097: 0.312980    0.973774    3.417176    0.197393    0.083218    0.007842    0.672101    
#> 098: 0.265290    0.988562    3.410171    0.190437    0.080932    0.008320    0.685429    
#> 099: 0.242113    0.961901    3.396593    0.183256    0.081253    0.009758    0.670034    
#> 100: 0.247544    0.981335    3.397618    0.195798    0.084067    0.009270    0.670370    
#> 101: 0.254070    0.946721    3.391846    0.190141    0.081950    0.008807    0.674393    
#> 102: 0.262667    0.969864    3.389453    0.185443    0.079726    0.008366    0.674859    
#> 103: 0.292395    0.955816    3.409276    0.176171    0.075740    0.008476    0.670367    
#> 104: 0.274723    0.955099    3.417973    0.167363    0.073965    0.008052    0.687604    
#> 105: 0.291898    0.971044    3.419737    0.170858    0.081704    0.007650    0.676461    
#> 106: 0.317075    0.951908    3.420409    0.189258    0.094543    0.007267    0.679233    
#> 107: 0.352625    0.961559    3.430900    0.199437    0.096848    0.006974    0.672934    
#> 108: 0.336323    0.935389    3.428828    0.189465    0.092006    0.006625    0.688937    
#> 109: 0.293218    0.961184    3.432602    0.179992    0.099718    0.006294    0.687107    
#> 110: 0.320921    0.932891    3.425293    0.191420    0.094732    0.005979    0.680430    
#> 111: 0.306042    0.933762    3.427415    0.187369    0.095065    0.005680    0.674424    
#> 112: 0.283693    0.923435    3.421426    0.178001    0.097443    0.005396    0.708563    
#> 113: 0.298669    0.961034    3.421243    0.169101    0.092571    0.005126    0.680419    
#> 114: 0.309827    0.953115    3.426872    0.160646    0.088942    0.004870    0.672962    
#> 115: 0.334335    0.930673    3.430259    0.153016    0.096371    0.004626    0.681528    
#> 116: 0.306263    0.940574    3.439771    0.145365    0.091553    0.004878    0.671032    
#> 117: 0.304464    0.944560    3.437426    0.143530    0.086975    0.004634    0.700032    
#> 118: 0.324366    0.926155    3.437787    0.148719    0.082626    0.004936    0.700053    
#> 119: 0.313242    0.932142    3.438492    0.158774    0.093477    0.004931    0.675957    
#> 120: 0.314315    0.922819    3.433326    0.150835    0.091326    0.005733    0.684772    
#> 121: 0.326591    0.961164    3.431496    0.152870    0.086759    0.005980    0.682657    
#> 122: 0.315261    0.946770    3.438123    0.162274    0.085292    0.005790    0.678702    
#> 123: 0.333996    0.954873    3.438325    0.156553    0.087968    0.006533    0.692321    
#> 124: 0.320255    0.972960    3.431091    0.169906    0.083569    0.006206    0.678632    
#> 125: 0.334125    0.971480    3.421512    0.177456    0.092271    0.005896    0.675078    
#> 126: 0.279761    0.982469    3.407941    0.168583    0.087658    0.006349    0.681144    
#> 127: 0.293023    0.949418    3.413249    0.160154    0.086355    0.007156    0.689622    
#> 128: 0.300066    0.936796    3.435343    0.188614    0.084091    0.009232    0.680865    
#> 129: 0.336035    0.955803    3.439649    0.181143    0.087596    0.009906    0.677101    
#> 130: 0.354805    0.948555    3.436598    0.188543    0.083216    0.011779    0.660030    
#> 131: 0.307695    0.942254    3.423362    0.180559    0.098586    0.011190    0.671679    
#> 132: 0.278260    0.956186    3.411181    0.171531    0.101693    0.010631    0.697905    
#> 133: 0.291854    0.935014    3.412689    0.162954    0.106040    0.010650    0.692314    
#> 134: 0.308781    0.944032    3.430802    0.154807    0.100738    0.014299    0.685088    
#> 135: 0.279322    0.958526    3.423923    0.162799    0.095701    0.013584    0.688591    
#> 136: 0.319089    0.929547    3.433134    0.154659    0.090916    0.012904    0.665247    
#> 137: 0.348262    0.934466    3.442944    0.165780    0.086370    0.012259    0.675463    
#> 138: 0.342024    0.944695    3.443432    0.157491    0.082052    0.011646    0.673296    
#> 139: 0.314140    0.924928    3.448283    0.149617    0.087133    0.014142    0.680403    
#> 140: 0.354552    0.933503    3.451166    0.181376    0.082776    0.013435    0.684212    
#> 141: 0.338477    0.921711    3.452960    0.172307    0.083011    0.013029    0.677601    
#> 142: 0.378366    0.908811    3.462434    0.190773    0.080065    0.012378    0.681734    
#> 143: 0.349787    0.926576    3.458411    0.198352    0.093839    0.011759    0.682982    
#> 144: 0.388061    0.921374    3.470290    0.188434    0.089147    0.011171    0.692041    
#> 145: 0.370035    0.931539    3.444863    0.194193    0.086746    0.010613    0.678918    
#> 146: 0.343113    0.931714    3.451629    0.191401    0.082409    0.011755    0.677344    
#> 147: 0.300074    0.924420    3.438341    0.181831    0.078289    0.011168    0.663498    
#> 148: 0.316407    0.942440    3.444516    0.208629    0.074374    0.010609    0.669414    
#> 149: 0.319059    0.924572    3.436894    0.198198    0.085131    0.010079    0.692863    
#> 150: 0.322459    0.955286    3.432731    0.188288    0.080874    0.009575    0.718773    
#> 151: 0.290523    0.972145    3.430735    0.178874    0.078206    0.009120    0.685732    
#> 152: 0.295300    0.961522    3.437663    0.173468    0.077166    0.009477    0.680386    
#> 153: 0.320853    0.939579    3.442561    0.164453    0.090209    0.011333    0.690697    
#> 154: 0.348196    0.924812    3.453352    0.163973    0.084412    0.012407    0.690342    
#> 155: 0.328130    0.914698    3.447001    0.173930    0.074293    0.010038    0.700574    
#> 156: 0.350745    0.927914    3.455842    0.147471    0.081228    0.010244    0.704007    
#> 157: 0.332397    0.911220    3.461303    0.148592    0.082696    0.009924    0.695505    
#> 158: 0.374909    0.923883    3.452351    0.171099    0.086926    0.009561    0.669522    
#> 159: 0.332082    0.938592    3.448499    0.168629    0.085903    0.008528    0.679850    
#> 160: 0.344159    0.917865    3.446820    0.172820    0.111381    0.008192    0.683942    
#> 161: 0.356820    0.927535    3.449511    0.188198    0.073676    0.010413    0.677466    
#> 162: 0.309979    0.947620    3.432030    0.188233    0.075170    0.010366    0.664887    
#> 163: 0.320966    0.923963    3.424278    0.166091    0.071697    0.008764    0.670802    
#> 164: 0.330141    0.936653    3.432902    0.180899    0.062563    0.010507    0.686758    
#> 165: 0.364971    0.933844    3.442471    0.181267    0.080691    0.010003    0.664062    
#> 166: 0.346613    0.948618    3.441243    0.179946    0.082016    0.009571    0.661259    
#> 167: 0.366022    0.970193    3.449105    0.206340    0.076056    0.009107    0.673192    
#> 168: 0.393563    0.943356    3.458770    0.182470    0.077098    0.008925    0.680633    
#> 169: 0.383022    0.973228    3.449918    0.162904    0.077746    0.008256    0.706250    
#> 170: 0.378515    0.925995    3.447130    0.157851    0.071942    0.007924    0.685699    
#> 171: 0.311374    0.941004    3.439042    0.168924    0.075535    0.007893    0.660179    
#> 172: 0.314038    0.967730    3.437661    0.169821    0.071623    0.008607    0.663725    
#> 173: 0.318610    0.961226    3.427018    0.145023    0.069608    0.010255    0.676455    
#> 174: 0.318314    0.949452    3.435775    0.125524    0.074592    0.007939    0.680879    
#> 175: 0.324541    0.960273    3.436915    0.141248    0.085424    0.008474    0.679700    
#> 176: 0.328831    0.952442    3.448946    0.119486    0.088932    0.006677    0.684143    
#> 177: 0.340668    0.934059    3.449022    0.136357    0.095592    0.007791    0.673545    
#> 178: 0.337438    0.902522    3.455205    0.149698    0.089449    0.009997    0.658190    
#> 179: 0.374286    0.929573    3.455849    0.157071    0.081402    0.008204    0.675019    
#> 180: 0.380390    0.924823    3.458417    0.138394    0.076140    0.009394    0.690987    
#> 181: 0.361424    0.913918    3.443958    0.163007    0.066550    0.009616    0.684561    
#> 182: 0.318182    0.929243    3.430719    0.207139    0.084136    0.009403    0.669622    
#> 183: 0.319526    0.953510    3.434861    0.163067    0.072744    0.009266    0.670508    
#> 184: 0.311801    0.915163    3.435026    0.163084    0.062404    0.008426    0.685068    
#> 185: 0.311948    0.936888    3.442151    0.178736    0.061249    0.008660    0.677410    
#> 186: 0.310904    0.965698    3.433283    0.194029    0.074455    0.011439    0.662771    
#> 187: 0.334224    0.942636    3.443836    0.169027    0.064128    0.013843    0.679616    
#> 188: 0.307047    0.952561    3.435809    0.162379    0.082615    0.009715    0.695962    
#> 189: 0.321107    0.962651    3.437315    0.167062    0.089431    0.009826    0.695315    
#> 190: 0.339397    0.944301    3.443014    0.172427    0.087148    0.009018    0.670777    
#> 191: 0.301711    0.936309    3.437870    0.162730    0.086584    0.006907    0.662533    
#> 192: 0.316314    0.948114    3.437559    0.158095    0.075295    0.007484    0.670825    
#> 193: 0.321401    0.960827    3.435100    0.151183    0.077395    0.006754    0.665450    
#> 194: 0.306545    0.945070    3.426498    0.133646    0.099220    0.003558    0.684824    
#> 195: 0.285282    0.942451    3.420037    0.139562    0.075446    0.003650    0.707969    
#> 196: 0.296599    0.943069    3.431227    0.143662    0.093210    0.004211    0.704570    
#> 197: 0.288071    0.938103    3.446617    0.122494    0.085168    0.004897    0.709903    
#> 198: 0.350402    0.932469    3.444662    0.150201    0.067109    0.004926    0.694229    
#> 199: 0.334512    0.908652    3.444065    0.144070    0.101476    0.002787    0.700210    
#> 200: 0.337137    0.913270    3.443507    0.116863    0.105442    0.002796    0.719507    
#> 201: 0.335962    0.912040    3.444838    0.118399    0.097808    0.003249    0.709820    
#> 202: 0.337423    0.921153    3.443124    0.120640    0.096069    0.003282    0.709165    
#> 203: 0.339381    0.928426    3.441935    0.131394    0.095969    0.003293    0.709834    
#> 204: 0.342839    0.926112    3.442397    0.139481    0.096268    0.003257    0.712974    
#> 205: 0.342690    0.928405    3.441759    0.141857    0.097508    0.003318    0.710597    
#> 206: 0.344653    0.932433    3.442791    0.141088    0.098828    0.003247    0.710792    
#> 207: 0.340197    0.931284    3.441837    0.144451    0.098371    0.003191    0.711335    
#> 208: 0.338874    0.931529    3.442154    0.141272    0.097499    0.003187    0.710705    
#> 209: 0.340320    0.928150    3.442413    0.137906    0.097009    0.003283    0.712358    
#> 210: 0.341136    0.925139    3.442924    0.136980    0.097649    0.003304    0.712949    
#> 211: 0.342017    0.925143    3.444033    0.138106    0.097588    0.003267    0.711443    
#> 212: 0.341822    0.923506    3.444611    0.137892    0.097202    0.003228    0.711148    
#> 213: 0.342835    0.923086    3.444646    0.138792    0.097205    0.003277    0.710242    
#> 214: 0.343665    0.922356    3.444628    0.138855    0.098446    0.003363    0.708955    
#> 215: 0.343178    0.922292    3.445012    0.138188    0.098331    0.003496    0.707296    
#> 216: 0.344487    0.922109    3.444974    0.138943    0.098538    0.003560    0.707394    
#> 217: 0.343887    0.923468    3.444681    0.138959    0.097785    0.003564    0.706907    
#> 218: 0.342865    0.924396    3.443968    0.138225    0.097561    0.003612    0.706432    
#> 219: 0.341030    0.924063    3.443063    0.136762    0.096628    0.003691    0.705455    
#> 220: 0.339542    0.923938    3.442819    0.136605    0.095788    0.003705    0.704887    
#> 221: 0.338803    0.924579    3.442728    0.136150    0.094973    0.003719    0.704738    
#> 222: 0.338837    0.924685    3.442612    0.136273    0.094750    0.003792    0.704519    
#> 223: 0.340468    0.924460    3.442726    0.135927    0.095101    0.003836    0.703715    
#> 224: 0.341904    0.925020    3.443352    0.135897    0.095492    0.003841    0.703622    
#> 225: 0.342911    0.923099    3.443867    0.135701    0.095141    0.003876    0.704335    
#> 226: 0.343525    0.921685    3.444055    0.135304    0.095509    0.003909    0.704029    
#> 227: 0.342794    0.921744    3.444307    0.135318    0.095195    0.003974    0.703535    
#> 228: 0.341858    0.922569    3.444244    0.135109    0.095309    0.004059    0.702966    
#> 229: 0.340683    0.921986    3.444208    0.134293    0.095363    0.004118    0.702391    
#> 230: 0.340771    0.921657    3.444114    0.133404    0.095404    0.004151    0.701934    
#> 231: 0.340317    0.922077    3.444288    0.132687    0.095111    0.004210    0.701572    
#> 232: 0.339286    0.921646    3.444657    0.132719    0.094937    0.004260    0.701437    
#> 233: 0.339427    0.921066    3.444791    0.132180    0.094347    0.004333    0.701852    
#> 234: 0.339114    0.921652    3.444880    0.132469    0.094284    0.004407    0.701490    
#> 235: 0.338538    0.921753    3.444640    0.132106    0.094099    0.004399    0.701736    
#> 236: 0.337889    0.922717    3.444256    0.132105    0.094335    0.004429    0.702153    
#> 237: 0.338695    0.922886    3.444249    0.131967    0.094357    0.004477    0.702258    
#> 238: 0.338758    0.922672    3.444346    0.132158    0.094465    0.004485    0.702851    
#> 239: 0.338667    0.923155    3.444325    0.132220    0.094736    0.004530    0.702482    
#> 240: 0.339266    0.922647    3.444361    0.133222    0.094900    0.004563    0.701710    
#> 241: 0.339472    0.921993    3.444453    0.133406    0.094916    0.004545    0.701474    
#> 242: 0.339197    0.922199    3.444288    0.133219    0.094937    0.004538    0.701753    
#> 243: 0.338126    0.922520    3.444277    0.133232    0.094815    0.004555    0.701678    
#> 244: 0.336918    0.922349    3.444272    0.133095    0.095015    0.004551    0.702031    
#> 245: 0.336442    0.922044    3.444300    0.133499    0.095185    0.004565    0.702198    
#> 246: 0.336585    0.922104    3.444547    0.133518    0.094947    0.004599    0.702311    
#> 247: 0.336414    0.921930    3.444914    0.134115    0.094679    0.004605    0.702423    
#> 248: 0.336321    0.922447    3.444877    0.134373    0.094593    0.004601    0.702841    
#> 249: 0.335924    0.923085    3.444653    0.135660    0.094403    0.004587    0.702611    
#> 250: 0.336355    0.923470    3.444709    0.135777    0.094333    0.004604    0.702270    
#> 251: 0.336896    0.924289    3.444822    0.135893    0.094268    0.004609    0.702237    
#> 252: 0.336966    0.924826    3.444939    0.136044    0.094491    0.004603    0.701972    
#> 253: 0.336591    0.925276    3.444912    0.136293    0.094698    0.004630    0.701580    
#> 254: 0.336184    0.925826    3.444890    0.136496    0.094716    0.004641    0.701211    
#> 255: 0.335811    0.926371    3.444883    0.136317    0.094410    0.004644    0.701310    
#> 256: 0.335696    0.926573    3.444786    0.136203    0.094153    0.004639    0.701418    
#> 257: 0.335499    0.926797    3.444586    0.136370    0.093786    0.004626    0.701779    
#> 258: 0.334766    0.926754    3.444485    0.136545    0.093806    0.004630    0.702007    
#> 259: 0.334027    0.926667    3.444425    0.136424    0.093959    0.004638    0.701763    
#> 260: 0.333523    0.926487    3.444606    0.136222    0.093683    0.004647    0.701664    
#> 261: 0.333381    0.926350    3.444781    0.135572    0.093732    0.004658    0.701572    
#> 262: 0.333882    0.926521    3.444856    0.136082    0.093863    0.004657    0.701397    
#> 263: 0.333698    0.926727    3.444883    0.136295    0.093769    0.004664    0.701121    
#> 264: 0.333411    0.927019    3.444708    0.136294    0.093617    0.004656    0.701290    
#> 265: 0.333557    0.927320    3.444614    0.136298    0.093692    0.004641    0.701360    
#> 266: 0.333635    0.928032    3.444426    0.136481    0.093684    0.004637    0.701396    
#> 267: 0.333478    0.928423    3.444255    0.136905    0.093580    0.004638    0.701524    
#> 268: 0.332796    0.928560    3.443930    0.137192    0.093427    0.004637    0.701568    
#> 269: 0.332816    0.928682    3.443731    0.137702    0.093419    0.004654    0.701327    
#> 270: 0.333132    0.928953    3.443606    0.137836    0.093424    0.004649    0.700983    
#> 271: 0.332933    0.929114    3.443405    0.137633    0.093382    0.004644    0.701156    
#> 272: 0.332914    0.928923    3.443254    0.138240    0.093532    0.004644    0.701224    
#> 273: 0.332684    0.929510    3.443102    0.138296    0.093908    0.004642    0.701197    
#> 274: 0.331953    0.929639    3.442922    0.138083    0.093757    0.004641    0.701197    
#> 275: 0.331372    0.929713    3.442726    0.138114    0.093930    0.004633    0.701164    
#> 276: 0.331345    0.929692    3.442849    0.138200    0.094030    0.004645    0.701112    
#> 277: 0.331215    0.929941    3.442924    0.138167    0.093941    0.004651    0.700820    
#> 278: 0.331193    0.929842    3.443029    0.138098    0.093907    0.004654    0.700727    
#> 279: 0.331135    0.929720    3.443055    0.138277    0.093901    0.004672    0.700635    
#> 280: 0.331187    0.930034    3.443083    0.138256    0.093908    0.004669    0.700314    
#> 281: 0.330792    0.930277    3.443093    0.138093    0.093720    0.004673    0.700284    
#> 282: 0.330615    0.930393    3.443154    0.137985    0.093572    0.004677    0.700249    
#> 283: 0.330581    0.930605    3.443090    0.138083    0.093443    0.004672    0.700398    
#> 284: 0.330399    0.930645    3.442994    0.138223    0.093151    0.004671    0.700478    
#> 285: 0.330351    0.930892    3.442874    0.138568    0.093092    0.004672    0.700391    
#> 286: 0.330264    0.931128    3.442751    0.139024    0.093268    0.004671    0.700258    
#> 287: 0.330092    0.931206    3.442647    0.139424    0.093483    0.004676    0.700123    
#> 288: 0.330397    0.931125    3.442573    0.139478    0.093540    0.004664    0.700074    
#> 289: 0.330801    0.931110    3.442580    0.139872    0.093627    0.004667    0.700057    
#> 290: 0.330738    0.931163    3.442482    0.139897    0.093679    0.004671    0.700066    
#> 291: 0.330826    0.931447    3.442518    0.140081    0.093714    0.004674    0.699837    
#> 292: 0.330796    0.931723    3.442442    0.140307    0.093755    0.004666    0.699906    
#> 293: 0.330699    0.931848    3.442345    0.140504    0.093788    0.004659    0.699869    
#> 294: 0.330459    0.932040    3.442253    0.140693    0.093691    0.004663    0.699749    
#> 295: 0.330462    0.931805    3.442229    0.140920    0.093632    0.004679    0.699505    
#> 296: 0.330522    0.931819    3.442253    0.140723    0.093576    0.004697    0.699455    
#> 297: 0.330661    0.932091    3.442229    0.140791    0.093543    0.004724    0.699303    
#> 298: 0.330790    0.932126    3.442163    0.141125    0.093583    0.004725    0.699243    
#> 299: 0.330795    0.932308    3.442051    0.141497    0.093500    0.004721    0.699157    
#> 300: 0.330693    0.932355    3.442049    0.141612    0.093600    0.004705    0.698933    
#> 301: 0.330879    0.932407    3.441984    0.141586    0.093619    0.004697    0.698917    
#> 302: 0.330788    0.932685    3.441816    0.141665    0.093613    0.004690    0.698915    
#> 303: 0.330416    0.933087    3.441716    0.141823    0.093642    0.004695    0.698656    
#> 304: 0.330081    0.933424    3.441556    0.142065    0.093562    0.004707    0.698577    
#> 305: 0.329779    0.933258    3.441443    0.142250    0.093625    0.004724    0.698489    
#> 306: 0.329274    0.933397    3.441242    0.142402    0.093712    0.004733    0.698413    
#> 307: 0.328939    0.933514    3.441078    0.142321    0.093958    0.004717    0.698486    
#> 308: 0.328524    0.933520    3.440998    0.142202    0.094103    0.004715    0.698560    
#> 309: 0.328438    0.933393    3.440905    0.142210    0.094006    0.004718    0.698562    
#> 310: 0.328539    0.933364    3.440788    0.142272    0.093970    0.004720    0.698492    
#> 311: 0.328552    0.933462    3.440690    0.142308    0.094033    0.004728    0.698410    
#> 312: 0.328496    0.933538    3.440581    0.142358    0.094099    0.004730    0.698326    
#> 313: 0.328075    0.933403    3.440491    0.142347    0.094046    0.004734    0.698495    
#> 314: 0.327856    0.933389    3.440393    0.142546    0.094115    0.004736    0.698607    
#> 315: 0.327582    0.933586    3.440263    0.142777    0.094178    0.004735    0.698855    
#> 316: 0.327563    0.933534    3.440227    0.142909    0.094202    0.004728    0.699050    
#> 317: 0.327484    0.933462    3.440270    0.142925    0.094197    0.004722    0.699123    
#> 318: 0.327602    0.933545    3.440198    0.142896    0.094072    0.004722    0.699015    
#> 319: 0.327387    0.933519    3.440155    0.142830    0.094162    0.004719    0.698951    
#> 320: 0.327000    0.933654    3.440013    0.142795    0.094118    0.004721    0.698981    
#> 321: 0.326470    0.933806    3.439783    0.142854    0.094127    0.004714    0.699006    
#> 322: 0.326309    0.934006    3.439638    0.142702    0.094060    0.004713    0.698963    
#> 323: 0.325724    0.934011    3.439501    0.142615    0.093988    0.004711    0.698943    
#> 324: 0.325421    0.934225    3.439410    0.142563    0.093925    0.004706    0.698760    
#> 325: 0.325222    0.934471    3.439271    0.142494    0.093898    0.004701    0.698637    
#> 326: 0.325012    0.934403    3.439140    0.142796    0.093940    0.004694    0.698437    
#> 327: 0.325036    0.934448    3.439078    0.142976    0.093928    0.004699    0.698403    
#> 328: 0.325036    0.934484    3.439001    0.143246    0.093922    0.004701    0.698169    
#> 329: 0.325151    0.934502    3.439001    0.143438    0.093976    0.004701    0.698169    
#> 330: 0.325145    0.934437    3.438994    0.143638    0.093846    0.004706    0.698145    
#> 331: 0.325290    0.934554    3.439019    0.143907    0.093714    0.004706    0.697995    
#> 332: 0.325392    0.934517    3.439017    0.144168    0.093539    0.004707    0.698149    
#> 333: 0.325257    0.934639    3.438892    0.144564    0.093465    0.004728    0.698000    
#> 334: 0.324989    0.934510    3.438741    0.144663    0.093400    0.004748    0.697878    
#> 335: 0.324553    0.934690    3.438553    0.144834    0.093396    0.004751    0.697757    
#> 336: 0.324085    0.934845    3.438446    0.144875    0.093436    0.004753    0.697767    
#> 337: 0.324009    0.935186    3.438352    0.145084    0.093483    0.004755    0.697798    
#> 338: 0.324169    0.935161    3.438378    0.145281    0.093535    0.004760    0.697686    
#> 339: 0.324049    0.935229    3.438372    0.145295    0.093463    0.004765    0.697598    
#> 340: 0.323955    0.935153    3.438367    0.145328    0.093462    0.004771    0.697514    
#> 341: 0.324148    0.935234    3.438441    0.145472    0.093501    0.004780    0.697497    
#> 342: 0.323915    0.935137    3.438456    0.145645    0.093490    0.004785    0.697348    
#> 343: 0.323667    0.935180    3.438528    0.145609    0.093580    0.004779    0.697263    
#> 344: 0.323714    0.935112    3.438574    0.145634    0.093414    0.004779    0.697353    
#> 345: 0.323704    0.935127    3.438613    0.145577    0.093349    0.004782    0.697440    
#> 346: 0.323664    0.935097    3.438572    0.145540    0.093269    0.004785    0.697393    
#> 347: 0.323693    0.935091    3.438474    0.145468    0.093254    0.004808    0.697454    
#> 348: 0.323411    0.935169    3.438376    0.145578    0.093296    0.004824    0.697458    
#> 349: 0.323338    0.935160    3.438280    0.145737    0.093216    0.004832    0.697513    
#> 350: 0.322999    0.935001    3.438199    0.145701    0.093234    0.004839    0.697601    
#> 351: 0.322795    0.935163    3.438108    0.145779    0.093238    0.004839    0.697604    
#> 352: 0.322456    0.935240    3.438072    0.146004    0.093279    0.004848    0.697692    
#> 353: 0.322306    0.935338    3.438015    0.146069    0.093150    0.004856    0.697608    
#> 354: 0.322436    0.935284    3.437971    0.146310    0.093099    0.004866    0.697473    
#> 355: 0.322680    0.935081    3.437975    0.146308    0.093054    0.004875    0.697328    
#> 356: 0.322614    0.935010    3.437984    0.146423    0.093059    0.004873    0.697299    
#> 357: 0.322887    0.935008    3.438013    0.146582    0.093009    0.004883    0.697252    
#> 358: 0.323105    0.935088    3.438095    0.146576    0.092890    0.004888    0.697119    
#> 359: 0.323075    0.935094    3.438134    0.146492    0.092867    0.004890    0.697147    
#> 360: 0.322825    0.935040    3.438083    0.146399    0.092847    0.004887    0.697060    
#> 361: 0.322606    0.934989    3.438002    0.146420    0.092753    0.004896    0.697041    
#> 362: 0.322537    0.934909    3.437956    0.146496    0.092654    0.004910    0.696819    
#> 363: 0.322476    0.934951    3.437885    0.146595    0.092604    0.004909    0.696673    
#> 364: 0.322601    0.934964    3.437880    0.146710    0.092546    0.004911    0.696643    
#> 365: 0.322570    0.934773    3.437928    0.146728    0.092492    0.004917    0.696736    
#> 366: 0.322553    0.934736    3.437868    0.146694    0.092611    0.004924    0.696695    
#> 367: 0.322663    0.934761    3.437861    0.146601    0.092731    0.004930    0.696685    
#> 368: 0.322726    0.934825    3.437810    0.146603    0.092683    0.004934    0.696611    
#> 369: 0.322601    0.934913    3.437739    0.146632    0.092721    0.004943    0.696575    
#> 370: 0.322446    0.934967    3.437665    0.146575    0.092629    0.004962    0.696522    
#> 371: 0.322291    0.935054    3.437551    0.146614    0.092558    0.004972    0.696416    
#> 372: 0.322265    0.935169    3.437503    0.146693    0.092534    0.004978    0.696341    
#> 373: 0.322099    0.935335    3.437454    0.146708    0.092483    0.004978    0.696311    
#> 374: 0.322053    0.935450    3.437373    0.146501    0.092426    0.004977    0.696321    
#> 375: 0.321778    0.935538    3.437276    0.146378    0.092415    0.004979    0.696310    
#> 376: 0.321586    0.935685    3.437188    0.146158    0.092474    0.004990    0.696268    
#> 377: 0.321346    0.935770    3.437160    0.146019    0.092460    0.004986    0.696175    
#> 378: 0.321034    0.935813    3.437088    0.146130    0.092411    0.004985    0.696224    
#> 379: 0.320906    0.935725    3.437094    0.146216    0.092418    0.004991    0.696190    
#> 380: 0.320741    0.935754    3.437055    0.146167    0.092455    0.004992    0.696141    
#> 381: 0.320570    0.935762    3.437006    0.146088    0.092456    0.004991    0.696113    
#> 382: 0.320548    0.935729    3.436975    0.145978    0.092476    0.004989    0.696037    
#> 383: 0.320731    0.935708    3.436995    0.146117    0.092364    0.004990    0.695998    
#> 384: 0.320692    0.935623    3.437036    0.146189    0.092353    0.004998    0.695938    
#> 385: 0.320722    0.935622    3.437067    0.146293    0.092421    0.005012    0.695834    
#> 386: 0.320847    0.935833    3.437080    0.146391    0.092451    0.005020    0.695785    
#> 387: 0.320927    0.936001    3.437083    0.146449    0.092425    0.005034    0.695730    
#> 388: 0.320943    0.936081    3.437069    0.146582    0.092389    0.005043    0.695680    
#> 389: 0.320980    0.935934    3.437076    0.146700    0.092309    0.005056    0.695700    
#> 390: 0.320721    0.936019    3.437030    0.146656    0.092225    0.005064    0.695728    
#> 391: 0.320663    0.935918    3.436983    0.146574    0.092172    0.005068    0.695762    
#> 392: 0.320732    0.935900    3.437020    0.146533    0.092067    0.005071    0.695946    
#> 393: 0.320691    0.935814    3.436999    0.146586    0.092029    0.005075    0.695992    
#> 394: 0.320726    0.935706    3.436979    0.146591    0.091986    0.005088    0.695953    
#> 395: 0.320943    0.935809    3.436963    0.146603    0.091887    0.005100    0.695958    
#> 396: 0.321021    0.935881    3.436937    0.146603    0.091842    0.005107    0.695923    
#> 397: 0.321053    0.935863    3.436917    0.146769    0.091819    0.005115    0.695855    
#> 398: 0.321135    0.935832    3.436940    0.147019    0.091871    0.005116    0.695817    
#> 399: 0.321261    0.935898    3.436980    0.147132    0.091868    0.005116    0.695775    
#> 400: 0.321361    0.936006    3.437001    0.147286    0.091853    0.005115    0.695709    
#> 401: 0.321328    0.936124    3.436985    0.147384    0.091882    0.005119    0.695777    
#> 402: 0.321218    0.936299    3.436904    0.147446    0.091788    0.005125    0.695829    
#> 403: 0.321341    0.936340    3.436856    0.147633    0.091767    0.005124    0.695765    
#> 404: 0.321498    0.936393    3.436825    0.147688    0.091686    0.005120    0.695697    
#> 405: 0.321386    0.936429    3.436831    0.147688    0.091609    0.005121    0.695676    
#> 406: 0.321524    0.936421    3.436845    0.147830    0.091551    0.005129    0.695711    
#> 407: 0.321590    0.936531    3.436869    0.148043    0.091572    0.005133    0.695670    
#> 408: 0.321641    0.936610    3.436837    0.148153    0.091611    0.005130    0.695619    
#> 409: 0.321483    0.936548    3.436793    0.148161    0.091576    0.005130    0.695519    
#> 410: 0.321355    0.936628    3.436706    0.148066    0.091580    0.005134    0.695427    
#> 411: 0.321278    0.936522    3.436658    0.148069    0.091592    0.005134    0.695503    
#> 412: 0.321231    0.936624    3.436608    0.147976    0.091541    0.005135    0.695480    
#> 413: 0.321192    0.936711    3.436547    0.147968    0.091501    0.005133    0.695524    
#> 414: 0.321229    0.936701    3.436577    0.147921    0.091470    0.005128    0.695512    
#> 415: 0.321385    0.936553    3.436623    0.147909    0.091445    0.005133    0.695424    
#> 416: 0.321513    0.936480    3.436659    0.147853    0.091430    0.005138    0.695327    
#> 417: 0.321584    0.936458    3.436654    0.147720    0.091434    0.005140    0.695311    
#> 418: 0.321288    0.936557    3.436614    0.147735    0.091424    0.005137    0.695317    
#> 419: 0.321176    0.936726    3.436544    0.147653    0.091445    0.005138    0.695309    
#> 420: 0.321213    0.936781    3.436537    0.147580    0.091439    0.005141    0.695247    
#> 421: 0.321078    0.936837    3.436561    0.147529    0.091441    0.005140    0.695259    
#> 422: 0.321238    0.936780    3.436581    0.147649    0.091488    0.005139    0.695234    
#> 423: 0.321389    0.936717    3.436585    0.147661    0.091549    0.005138    0.695294    
#> 424: 0.321347    0.936622    3.436568    0.147649    0.091585    0.005134    0.695343    
#> 425: 0.321461    0.936632    3.436564    0.147494    0.091543    0.005127    0.695402    
#> 426: 0.321605    0.936706    3.436571    0.147444    0.091547    0.005121    0.695430    
#> 427: 0.321477    0.936686    3.436569    0.147423    0.091566    0.005122    0.695340    
#> 428: 0.321322    0.936748    3.436538    0.147534    0.091575    0.005126    0.695254    
#> 429: 0.321304    0.936765    3.436515    0.147575    0.091530    0.005133    0.695186    
#> 430: 0.321222    0.936879    3.436467    0.147576    0.091505    0.005143    0.695085    
#> 431: 0.321139    0.936949    3.436408    0.147563    0.091482    0.005149    0.695000    
#> 432: 0.321172    0.937049    3.436423    0.147653    0.091429    0.005158    0.694942    
#> 433: 0.321215    0.937151    3.436467    0.147765    0.091337    0.005166    0.694873    
#> 434: 0.321235    0.937008    3.436548    0.147823    0.091355    0.005182    0.694810    
#> 435: 0.321394    0.936955    3.436593    0.147811    0.091343    0.005186    0.694789    
#> 436: 0.321427    0.936887    3.436636    0.147792    0.091283    0.005187    0.694813    
#> 437: 0.321417    0.936738    3.436638    0.147831    0.091317    0.005187    0.694747    
#> 438: 0.321418    0.936818    3.436656    0.147810    0.091325    0.005188    0.694673    
#> 439: 0.321463    0.936912    3.436647    0.147763    0.091302    0.005186    0.694611    
#> 440: 0.321350    0.936900    3.436667    0.147713    0.091310    0.005187    0.694549    
#> 441: 0.321395    0.936882    3.436712    0.147627    0.091323    0.005188    0.694494    
#> 442: 0.321390    0.936976    3.436718    0.147641    0.091386    0.005186    0.694401    
#> 443: 0.321330    0.937079    3.436707    0.147681    0.091428    0.005184    0.694405    
#> 444: 0.321253    0.937203    3.436658    0.147612    0.091430    0.005179    0.694401    
#> 445: 0.321150    0.937257    3.436594    0.147669    0.091482    0.005173    0.694360    
#> 446: 0.320931    0.937320    3.436519    0.147725    0.091462    0.005165    0.694336    
#> 447: 0.320962    0.937391    3.436486    0.147722    0.091446    0.005167    0.694269    
#> 448: 0.320902    0.937407    3.436483    0.147724    0.091403    0.005171    0.694201    
#> 449: 0.320959    0.937506    3.436540    0.147728    0.091368    0.005176    0.694201    
#> 450: 0.321029    0.937524    3.436550    0.147771    0.091329    0.005185    0.694108    
#> 451: 0.321082    0.937524    3.436519    0.147965    0.091303    0.005194    0.694052    
#> 452: 0.321067    0.937591    3.436478    0.148116    0.091262    0.005196    0.694002    
#> 453: 0.320997    0.937652    3.436438    0.148151    0.091210    0.005192    0.693999    
#> 454: 0.320933    0.937765    3.436411    0.148182    0.091235    0.005190    0.693915    
#> 455: 0.321028    0.937826    3.436377    0.148278    0.091220    0.005187    0.693899    
#> 456: 0.320907    0.937860    3.436341    0.148390    0.091219    0.005188    0.693863    
#> 457: 0.320845    0.937854    3.436334    0.148464    0.091212    0.005191    0.693793    
#> 458: 0.320745    0.937987    3.436318    0.148551    0.091207    0.005189    0.693760    
#> 459: 0.320635    0.938210    3.436281    0.148477    0.091163    0.005189    0.693865    
#> 460: 0.320507    0.938226    3.436244    0.148405    0.091095    0.005193    0.693808    
#> 461: 0.320336    0.938267    3.436232    0.148514    0.091019    0.005201    0.693806    
#> 462: 0.320280    0.938408    3.436235    0.148557    0.090986    0.005206    0.693779    
#> 463: 0.320318    0.938470    3.436223    0.148625    0.090963    0.005208    0.693733    
#> 464: 0.320349    0.938493    3.436213    0.148658    0.090961    0.005208    0.693627    
#> 465: 0.320351    0.938376    3.436210    0.148682    0.090929    0.005204    0.693614    
#> 466: 0.320358    0.938434    3.436168    0.148701    0.090882    0.005203    0.693579    
#> 467: 0.320248    0.938416    3.436142    0.148744    0.090899    0.005202    0.693489    
#> 468: 0.320282    0.938475    3.436115    0.148863    0.090903    0.005208    0.693473    
#> 469: 0.320248    0.938585    3.436097    0.148935    0.090899    0.005214    0.693438    
#> 470: 0.320101    0.938650    3.436060    0.148986    0.090935    0.005221    0.693416    
#> 471: 0.320001    0.938594    3.436049    0.148985    0.090949    0.005221    0.693386    
#> 472: 0.320003    0.938576    3.436087    0.148966    0.090943    0.005225    0.693390    
#> 473: 0.319855    0.938569    3.436099    0.148883    0.090970    0.005228    0.693371    
#> 474: 0.319719    0.938523    3.436074    0.148896    0.090981    0.005240    0.693381    
#> 475: 0.319592    0.938539    3.436053    0.148878    0.090967    0.005248    0.693330    
#> 476: 0.319595    0.938563    3.436003    0.148972    0.090948    0.005255    0.693313    
#> 477: 0.319563    0.938565    3.436002    0.148965    0.090938    0.005260    0.693222    
#> 478: 0.319483    0.938586    3.435981    0.148965    0.090965    0.005262    0.693116    
#> 479: 0.319462    0.938602    3.436005    0.148972    0.090972    0.005264    0.693088    
#> 480: 0.319519    0.938534    3.436043    0.148906    0.090986    0.005265    0.693173    
#> 481: 0.319456    0.938564    3.436026    0.148812    0.090980    0.005264    0.693219    
#> 482: 0.319583    0.938664    3.436004    0.148920    0.090983    0.005264    0.693223    
#> 483: 0.319469    0.938766    3.435943    0.149014    0.090974    0.005265    0.693154    
#> 484: 0.319326    0.938687    3.435901    0.149066    0.090948    0.005268    0.693140    
#> 485: 0.319234    0.938684    3.435851    0.149154    0.090914    0.005275    0.693095    
#> 486: 0.319165    0.938690    3.435821    0.149147    0.090879    0.005278    0.693065    
#> 487: 0.319201    0.938776    3.435822    0.149176    0.090839    0.005281    0.692996    
#> 488: 0.319150    0.938821    3.435833    0.149163    0.090824    0.005284    0.692990    
#> 489: 0.319215    0.938826    3.435840    0.149115    0.090761    0.005288    0.693039    
#> 490: 0.319331    0.938768    3.435893    0.149100    0.090749    0.005292    0.693095    
#> 491: 0.319370    0.938787    3.435939    0.149127    0.090762    0.005292    0.693094    
#> 492: 0.319397    0.938759    3.435950    0.149198    0.090790    0.005288    0.693185    
#> 493: 0.319403    0.938743    3.435965    0.149228    0.090758    0.005293    0.693181    
#> 494: 0.319507    0.938690    3.435985    0.149230    0.090756    0.005291    0.693129    
#> 495: 0.319621    0.938668    3.436026    0.149328    0.090732    0.005290    0.693063    
#> 496: 0.319739    0.938657    3.436038    0.149361    0.090727    0.005290    0.693053    
#> 497: 0.319785    0.938632    3.436054    0.149331    0.090752    0.005292    0.693065    
#> 498: 0.319791    0.938627    3.436025    0.149349    0.090732    0.005292    0.692988    
#> 499: 0.319777    0.938665    3.435989    0.149333    0.090783    0.005293    0.692932    
#> 500: 0.319778    0.938724    3.435969    0.149368    0.090746    0.005294    0.692815    
#> [====|====|=>-------------------------------------]  25%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.346897    0.981633    3.451995    0.392401    0.078006    0.016800    1.704214    
#> 002: 0.480519    1.093896    3.476087    0.372781    0.074106    0.015960    1.099623    
#> 003: 0.748482    1.065340    3.514296    0.374080    0.070400    0.015162    0.743538    
#> 004: 0.749106    1.130604    3.512098    0.404584    0.066880    0.014404    0.650344    
#> 005: 0.817556    1.136390    3.517923    0.540077    0.063536    0.013684    0.580705    
#> 006: 0.848977    1.145012    3.523006    0.565394    0.060360    0.013000    0.519926    
#> 007: 0.838423    1.165947    3.520675    0.573583    0.057342    0.012350    0.519273    
#> 008: 0.812841    1.149767    3.520138    0.581834    0.054474    0.011732    0.525720    
#> 009: 0.783738    1.139139    3.517826    0.552742    0.051751    0.011146    0.527935    
#> 010: 0.813573    1.134500    3.520840    0.654859    0.049163    0.010588    0.530873    
#> 011: 0.884895    1.137444    3.518789    0.697829    0.046705    0.010116    0.521671    
#> 012: 0.946702    1.131088    3.530557    0.753675    0.044370    0.009610    0.514269    
#> 013: 0.863832    1.136530    3.529390    0.715991    0.042151    0.009130    0.522420    
#> 014: 0.935204    1.138702    3.536840    0.750698    0.040044    0.008673    0.515957    
#> 015: 0.957396    1.135945    3.551604    0.780097    0.038042    0.008286    0.496840    
#> 016: 0.950442    1.143158    3.551898    0.741092    0.036139    0.008626    0.507442    
#> 017: 0.932444    1.142232    3.552220    0.704038    0.034333    0.008194    0.502335    
#> 018: 0.931271    1.151509    3.538893    0.668836    0.032616    0.007785    0.507997    
#> 019: 0.885049    1.146465    3.531785    0.660212    0.030985    0.007395    0.487156    
#> 020: 0.909102    1.146799    3.537482    0.627201    0.036253    0.007458    0.491680    
#> 021: 0.900937    1.132484    3.549094    0.624700    0.034441    0.008574    0.494057    
#> 022: 0.903791    1.132519    3.547482    0.656369    0.037800    0.008387    0.496759    
#> 023: 0.964338    1.144695    3.550276    0.680134    0.035910    0.007968    0.521472    
#> 024: 0.979036    1.157034    3.547369    0.783630    0.034115    0.007918    0.496428    
#> 025: 0.987488    1.135233    3.550533    0.837446    0.032409    0.008673    0.503089    
#> 026: 0.976500    1.155026    3.542122    0.819813    0.030789    0.009265    0.528267    
#> 027: 0.936500    1.158384    3.543453    0.778822    0.029249    0.008802    0.514229    
#> 028: 0.906480    1.142928    3.544519    0.739881    0.027787    0.009421    0.498870    
#> 029: 0.924380    1.136571    3.542140    0.702887    0.028602    0.009495    0.504169    
#> 030: 0.944769    1.146388    3.541930    0.762585    0.028839    0.011580    0.505802    
#> 031: 0.876074    1.134469    3.538533    0.724456    0.029862    0.011001    0.501470    
#> 032: 0.804875    1.123402    3.532689    0.688233    0.028369    0.010451    0.505761    
#> 033: 0.878968    1.139295    3.533582    0.676242    0.026950    0.009929    0.498976    
#> 034: 0.931615    1.146135    3.542436    0.720869    0.028606    0.009432    0.493994    
#> 035: 0.881499    1.128919    3.534962    0.684826    0.028406    0.008961    0.488027    
#> 036: 0.905213    1.131580    3.540345    0.684978    0.026986    0.010170    0.499123    
#> 037: 0.891296    1.154772    3.535955    0.664649    0.030380    0.009724    0.507485    
#> 038: 0.891550    1.144098    3.532258    0.696654    0.030803    0.009688    0.514298    
#> 039: 0.929292    1.140129    3.544320    0.688371    0.030103    0.010136    0.504585    
#> 040: 0.921015    1.147802    3.545040    0.697915    0.028598    0.010412    0.513403    
#> 041: 0.943060    1.129048    3.538898    0.820143    0.029610    0.009891    0.500824    
#> 042: 0.930000    1.137420    3.532660    0.830712    0.028130    0.009397    0.503804    
#> 043: 0.864594    1.157194    3.522936    0.789176    0.026723    0.008927    0.505993    
#> 044: 0.889414    1.173524    3.523267    0.749717    0.025875    0.008481    0.504496    
#> 045: 0.898829    1.162876    3.517581    0.760732    0.024581    0.008057    0.504053    
#> 046: 0.924946    1.150270    3.531416    0.738871    0.028887    0.008763    0.489925    
#> 047: 0.937239    1.148440    3.535457    0.763967    0.027443    0.009530    0.499140    
#> 048: 0.914764    1.149360    3.529224    0.791370    0.028590    0.009823    0.492055    
#> 049: 0.907915    1.139353    3.539637    0.751801    0.031170    0.010158    0.499388    
#> 050: 0.977415    1.124591    3.548189    0.917848    0.029611    0.009650    0.527069    
#> 051: 0.919470    1.135270    3.531828    0.871956    0.030928    0.009167    0.515947    
#> 052: 0.934355    1.131821    3.543000    0.828358    0.029381    0.008733    0.513348    
#> 053: 0.944135    1.132870    3.546186    0.786940    0.027912    0.010258    0.501064    
#> 054: 0.945297    1.138464    3.548684    0.752134    0.026517    0.012535    0.508852    
#> 055: 0.925444    1.141316    3.548719    0.714528    0.025191    0.011908    0.517672    
#> 056: 0.924040    1.136412    3.555405    0.678801    0.027602    0.011313    0.518646    
#> 057: 0.926672    1.144089    3.549693    0.653129    0.030245    0.010747    0.524361    
#> 058: 0.914417    1.146574    3.537643    0.637730    0.030478    0.010210    0.523442    
#> 059: 0.886751    1.115728    3.543332    0.679452    0.028954    0.012420    0.518279    
#> 060: 0.920244    1.114735    3.547730    0.722792    0.029156    0.011799    0.525739    
#> 061: 0.963364    1.122747    3.545327    0.874652    0.027699    0.012060    0.521453    
#> 062: 0.938632    1.123630    3.547084    0.830920    0.026314    0.013460    0.546524    
#> 063: 0.956960    1.115998    3.549464    0.789374    0.026908    0.012787    0.536490    
#> 064: 0.930918    1.129259    3.541417    0.749905    0.025563    0.012147    0.521258    
#> 065: 0.976554    1.122481    3.527635    0.857232    0.024284    0.011795    0.522115    
#> 066: 0.950859    1.138659    3.539519    0.834382    0.023255    0.011205    0.494360    
#> 067: 0.949467    1.138598    3.539910    0.795380    0.026557    0.010645    0.505719    
#> 068: 0.931129    1.145983    3.547141    0.816612    0.025229    0.010113    0.511268    
#> 069: 0.920893    1.146058    3.539878    0.775782    0.023967    0.009607    0.515169    
#> 070: 0.895067    1.142272    3.551163    0.736993    0.022769    0.010886    0.511103    
#> 071: 0.874584    1.165775    3.532841    0.700143    0.021631    0.010342    0.513130    
#> 072: 0.862362    1.145869    3.534445    0.665136    0.020549    0.010248    0.507557    
#> 073: 0.940981    1.148531    3.544768    0.767279    0.020897    0.009735    0.500832    
#> 074: 0.888483    1.142621    3.539883    0.728916    0.025060    0.009980    0.500340    
#> 075: 0.934798    1.134146    3.555770    0.743026    0.031448    0.011548    0.505175    
#> 076: 0.905021    1.144576    3.560006    0.705875    0.029967    0.011873    0.530173    
#> 077: 0.978203    1.122928    3.550401    0.670581    0.030974    0.011952    0.524761    
#> 078: 0.937889    1.121711    3.553260    0.671634    0.029425    0.011354    0.526412    
#> 079: 0.971039    1.135201    3.548470    0.814337    0.027954    0.010787    0.523685    
#> 080: 0.971983    1.128628    3.549261    0.773620    0.026556    0.010247    0.521797    
#> 081: 0.929078    1.126799    3.546825    0.798833    0.026442    0.009735    0.512220    
#> 082: 0.938775    1.128725    3.553758    0.758891    0.030099    0.009421    0.520161    
#> 083: 0.936512    1.139290    3.553634    0.750507    0.030384    0.008950    0.518420    
#> 084: 0.910453    1.137549    3.550108    0.712982    0.030455    0.010387    0.509251    
#> 085: 0.904586    1.136858    3.558124    0.677332    0.029292    0.011879    0.497233    
#> 086: 0.923557    1.143855    3.551096    0.680487    0.029133    0.011285    0.494950    
#> 087: 0.893327    1.132576    3.542360    0.646462    0.030756    0.010720    0.493006    
#> 088: 0.917337    1.118633    3.543706    0.809911    0.031007    0.010184    0.499177    
#> 089: 0.904196    1.142237    3.545647    0.769416    0.031922    0.011027    0.495643    
#> 090: 0.912027    1.141778    3.548396    0.730945    0.034696    0.010476    0.503244    
#> 091: 0.913849    1.140372    3.536158    0.694398    0.032961    0.010459    0.509748    
#> 092: 0.876601    1.132821    3.534516    0.724889    0.031313    0.010079    0.512640    
#> 093: 0.844273    1.142689    3.525763    0.693344    0.033620    0.009575    0.520692    
#> 094: 0.893253    1.129802    3.530285    0.658677    0.032037    0.009352    0.516835    
#> 095: 0.913196    1.141249    3.541949    0.724491    0.030435    0.010478    0.499506    
#> 096: 0.876853    1.149980    3.536320    0.688267    0.029636    0.009954    0.499411    
#> 097: 0.905538    1.157621    3.539900    0.653853    0.028752    0.010300    0.490706    
#> 098: 0.926300    1.153810    3.545510    0.642152    0.028993    0.010698    0.488005    
#> 099: 0.918042    1.127589    3.546697    0.647622    0.027543    0.010410    0.494606    
#> 100: 0.923110    1.132761    3.546909    0.723820    0.026478    0.011349    0.491359    
#> 101: 0.867837    1.133191    3.532288    0.687629    0.027299    0.010782    0.493405    
#> 102: 0.866423    1.133210    3.525578    0.674569    0.028255    0.010242    0.491162    
#> 103: 0.865019    1.150140    3.536735    0.640840    0.030718    0.009730    0.483759    
#> 104: 0.891236    1.146069    3.543802    0.661741    0.029582    0.009629    0.501014    
#> 105: 0.940878    1.129873    3.538444    0.781667    0.030504    0.009147    0.519712    
#> 106: 0.963949    1.137739    3.546140    0.818887    0.028979    0.010629    0.494238    
#> 107: 0.967434    1.146837    3.546234    0.812822    0.027530    0.010097    0.501297    
#> 108: 0.939487    1.146831    3.547241    0.772181    0.028791    0.009593    0.496320    
#> 109: 0.916252    1.145645    3.547039    0.733572    0.027351    0.009113    0.495075    
#> 110: 0.911088    1.134252    3.545095    0.696893    0.026553    0.009786    0.501127    
#> 111: 0.926998    1.139240    3.551288    0.662049    0.025226    0.009437    0.496776    
#> 112: 0.913746    1.118668    3.543922    0.670024    0.025523    0.009100    0.519176    
#> 113: 0.897306    1.133625    3.544707    0.636523    0.024247    0.008645    0.516040    
#> 114: 0.909740    1.143542    3.547880    0.608153    0.024577    0.009159    0.517452    
#> 115: 0.957636    1.122367    3.545668    0.705838    0.023348    0.011072    0.514493    
#> 116: 0.941913    1.135933    3.542345    0.670546    0.025401    0.010568    0.497177    
#> 117: 0.884155    1.152528    3.542866    0.637019    0.024131    0.010040    0.503728    
#> 118: 0.908932    1.154761    3.541975    0.628603    0.027864    0.009538    0.506466    
#> 119: 0.916978    1.154562    3.542676    0.674335    0.026471    0.009061    0.514591    
#> 120: 0.866030    1.143713    3.541124    0.643078    0.025148    0.008854    0.504806    
#> 121: 0.891264    1.155419    3.533521    0.613284    0.026707    0.008411    0.504973    
#> 122: 0.877551    1.138043    3.535167    0.620490    0.026060    0.007990    0.508103    
#> 123: 0.864227    1.166412    3.530460    0.589465    0.025777    0.008503    0.510104    
#> 124: 0.919659    1.176402    3.530940    0.715729    0.024716    0.008476    0.508435    
#> 125: 0.922176    1.164441    3.526611    0.679943    0.031708    0.008817    0.505876    
#> 126: 0.891059    1.167737    3.520640    0.645946    0.032743    0.008770    0.514793    
#> 127: 0.912305    1.147456    3.526998    0.690215    0.031484    0.008332    0.517660    
#> 128: 0.912430    1.149361    3.534598    0.713039    0.031713    0.007915    0.511677    
#> 129: 0.925868    1.147999    3.544872    0.679304    0.030128    0.008160    0.521868    
#> 130: 0.928202    1.155641    3.537549    0.691521    0.028621    0.007843    0.514609    
#> 131: 0.906850    1.139474    3.533355    0.656945    0.030267    0.007451    0.511162    
#> 132: 0.901307    1.138791    3.528455    0.700922    0.032159    0.008694    0.505663    
#> 133: 0.857180    1.129944    3.533677    0.665876    0.030551    0.008611    0.514845    
#> 134: 0.903216    1.124341    3.542782    0.688280    0.029024    0.009614    0.508354    
#> 135: 0.931930    1.121893    3.548173    0.653866    0.027573    0.009133    0.502042    
#> 136: 0.930701    1.123078    3.547451    0.621173    0.026194    0.009268    0.498710    
#> 137: 0.968905    1.129880    3.560043    0.625751    0.029374    0.009931    0.520402    
#> 138: 0.942826    1.139121    3.558729    0.613789    0.029774    0.011016    0.514162    
#> 139: 0.905157    1.119416    3.560672    0.643903    0.028285    0.010465    0.514069    
#> 140: 0.913824    1.126648    3.554847    0.611708    0.026871    0.010367    0.511641    
#> 141: 0.884818    1.119995    3.559203    0.581122    0.025527    0.011307    0.517236    
#> 142: 0.934389    1.107907    3.562461    0.618894    0.024251    0.011799    0.525351    
#> 143: 0.923234    1.118863    3.552450    0.683168    0.023330    0.011209    0.519530    
#> 144: 0.978115    1.131104    3.560565    0.779866    0.022163    0.012084    0.508360    
#> 145: 0.950273    1.143266    3.541508    0.852589    0.021055    0.011480    0.513880    
#> 146: 0.930923    1.150747    3.551337    0.809959    0.020002    0.011191    0.504131    
#> 147: 0.882920    1.140813    3.535751    0.769462    0.021705    0.012785    0.499913    
#> 148: 0.890024    1.146460    3.541120    0.730988    0.026532    0.012145    0.498600    
#> 149: 0.902126    1.122696    3.544054    0.694439    0.025206    0.012024    0.505071    
#> 150: 0.916528    1.122099    3.536263    0.709680    0.027765    0.011423    0.506240    
#> 151: 0.867938    1.160511    3.530665    0.674196    0.026376    0.010851    0.506640    
#> 152: 0.832041    1.156854    3.531568    0.649433    0.016855    0.013030    0.524933    
#> 153: 0.865480    1.169799    3.534255    0.589602    0.019130    0.012763    0.524158    
#> 154: 0.894409    1.175006    3.540738    0.635357    0.020212    0.013104    0.522916    
#> 155: 0.856881    1.152085    3.532067    0.595002    0.021604    0.014491    0.505092    
#> 156: 0.891656    1.154373    3.536142    0.587645    0.020501    0.014979    0.514567    
#> 157: 0.875060    1.143111    3.540413    0.609772    0.019846    0.014387    0.511212    
#> 158: 0.917774    1.147445    3.540247    0.627914    0.018641    0.013637    0.513916    
#> 159: 0.870137    1.162208    3.549986    0.550906    0.021467    0.011807    0.502858    
#> 160: 0.924927    1.131867    3.560545    0.563612    0.020147    0.014650    0.509300    
#> 161: 0.937157    1.135822    3.556144    0.551117    0.018765    0.016457    0.523419    
#> 162: 0.908453    1.143098    3.548572    0.522758    0.019155    0.013151    0.526301    
#> 163: 0.914024    1.127623    3.546080    0.557741    0.022171    0.012769    0.513156    
#> 164: 0.912556    1.138295    3.545461    0.584644    0.020266    0.013109    0.511221    
#> 165: 0.942738    1.139940    3.550035    0.640550    0.026044    0.012912    0.516910    
#> 166: 0.960490    1.153156    3.550862    0.592012    0.022641    0.013926    0.529427    
#> 167: 0.972382    1.161296    3.558088    0.674461    0.024944    0.012870    0.515933    
#> 168: 0.973569    1.160310    3.557918    0.703502    0.025169    0.009754    0.503580    
#> 169: 0.932636    1.166806    3.542346    0.561617    0.023687    0.010939    0.514507    
#> 170: 0.935342    1.143784    3.551616    0.563051    0.029013    0.011415    0.508861    
#> 171: 0.868358    1.137273    3.544630    0.522159    0.022771    0.010008    0.515519    
#> 172: 0.885966    1.145655    3.547660    0.541962    0.029912    0.009282    0.501011    
#> 173: 0.927475    1.153886    3.531871    0.698086    0.025344    0.007993    0.499477    
#> 174: 0.936269    1.158976    3.543753    0.642875    0.028017    0.008204    0.507314    
#> 175: 0.926916    1.152836    3.537943    0.680760    0.020630    0.008263    0.513821    
#> 176: 0.935129    1.164554    3.544311    0.659252    0.028351    0.007294    0.510670    
#> 177: 0.908647    1.150882    3.549112    0.612226    0.023673    0.007361    0.514069    
#> 178: 0.908524    1.137168    3.544833    0.641434    0.029468    0.007666    0.508609    
#> 179: 0.939737    1.155975    3.540183    0.631866    0.034089    0.008735    0.510342    
#> 180: 0.902048    1.162832    3.539479    0.599737    0.029623    0.008155    0.508853    
#> 181: 0.894445    1.130951    3.532755    0.571387    0.022940    0.009024    0.511822    
#> 182: 0.850805    1.143170    3.527115    0.576689    0.022124    0.010073    0.526959    
#> 183: 0.854805    1.149926    3.527507    0.524028    0.026502    0.011395    0.511259    
#> 184: 0.865140    1.131096    3.537073    0.506829    0.033775    0.008975    0.511188    
#> 185: 0.849479    1.151280    3.527690    0.541829    0.030754    0.008169    0.512474    
#> 186: 0.872861    1.168392    3.531854    0.639877    0.037807    0.008162    0.506141    
#> 187: 0.893332    1.147942    3.541686    0.633453    0.034205    0.009511    0.511671    
#> 188: 0.868671    1.157290    3.533486    0.655364    0.033520    0.010002    0.520827    
#> 189: 0.907180    1.154973    3.541413    0.684639    0.030570    0.011954    0.498898    
#> 190: 0.893709    1.145055    3.540589    0.651837    0.026243    0.010615    0.496622    
#> 191: 0.865332    1.148092    3.534448    0.614569    0.022147    0.011573    0.491407    
#> 192: 0.876590    1.153069    3.530784    0.669670    0.021849    0.011050    0.492052    
#> 193: 0.916108    1.157825    3.540598    0.674916    0.023282    0.010805    0.503416    
#> 194: 0.886691    1.146427    3.532872    0.622049    0.024976    0.010629    0.497376    
#> 195: 0.856814    1.151510    3.524904    0.671644    0.021851    0.011063    0.497405    
#> 196: 0.886565    1.147820    3.534222    0.593530    0.019990    0.012716    0.515912    
#> 197: 0.857052    1.151662    3.545191    0.595734    0.021536    0.012552    0.525993    
#> 198: 0.928773    1.140507    3.535323    0.677793    0.017596    0.013742    0.512234    
#> 199: 0.902027    1.133856    3.540286    0.713970    0.015699    0.013941    0.505325    
#> 200: 0.872189    1.123660    3.538136    0.706452    0.014993    0.013876    0.510528    
#> 201: 0.887632    1.126411    3.541756    0.710887    0.016615    0.013756    0.507224    
#> 202: 0.897968    1.129375    3.542970    0.709736    0.016869    0.013583    0.507415    
#> 203: 0.912180    1.130591    3.542668    0.714933    0.017438    0.013114    0.506405    
#> 204: 0.926380    1.129276    3.544122    0.726926    0.019533    0.012303    0.506352    
#> 205: 0.929397    1.134168    3.544039    0.714787    0.020560    0.011752    0.507335    
#> 206: 0.924655    1.138672    3.545138    0.693193    0.020866    0.011824    0.508299    
#> 207: 0.924961    1.139002    3.544415    0.684690    0.020837    0.011607    0.508265    
#> 208: 0.925860    1.138961    3.544553    0.686462    0.020568    0.011482    0.507275    
#> 209: 0.927736    1.136989    3.544879    0.681007    0.020519    0.011441    0.508207    
#> 210: 0.927342    1.135577    3.545718    0.675715    0.020465    0.011458    0.507401    
#> 211: 0.926338    1.134754    3.546245    0.670707    0.020942    0.011378    0.505493    
#> 212: 0.926935    1.134530    3.546752    0.668284    0.021071    0.011380    0.505118    
#> 213: 0.928944    1.133411    3.547471    0.663743    0.021214    0.011472    0.505900    
#> 214: 0.929541    1.132520    3.547939    0.659090    0.021436    0.011473    0.506302    
#> 215: 0.930492    1.132253    3.548672    0.657461    0.021539    0.011517    0.505694    
#> 216: 0.935277    1.130702    3.549656    0.660234    0.021471    0.011729    0.505482    
#> 217: 0.936740    1.130721    3.550163    0.660583    0.021366    0.011874    0.505168    
#> 218: 0.935403    1.131527    3.549912    0.656613    0.021466    0.012006    0.504498    
#> 219: 0.932860    1.131742    3.549211    0.653265    0.021326    0.012102    0.504152    
#> 220: 0.929500    1.131808    3.548644    0.653451    0.021140    0.012180    0.504212    
#> 221: 0.927740    1.132319    3.547957    0.653269    0.021150    0.012156    0.504081    
#> 222: 0.925651    1.132497    3.547711    0.648872    0.021241    0.012205    0.503939    
#> 223: 0.925935    1.132989    3.547783    0.647815    0.021267    0.012165    0.504089    
#> 224: 0.927597    1.133418    3.548012    0.647411    0.021117    0.012043    0.503910    
#> 225: 0.928201    1.132855    3.548462    0.645967    0.021101    0.012026    0.503717    
#> 226: 0.927767    1.132153    3.548464    0.643680    0.021120    0.011966    0.503720    
#> 227: 0.927967    1.132116    3.548533    0.643224    0.021156    0.012005    0.503397    
#> 228: 0.927912    1.132348    3.548266    0.642967    0.021007    0.012007    0.503580    
#> 229: 0.926825    1.132121    3.548066    0.641500    0.020761    0.012002    0.503671    
#> 230: 0.927518    1.131732    3.548026    0.642518    0.020557    0.011977    0.503858    
#> 231: 0.927359    1.131893    3.548278    0.641050    0.020467    0.011978    0.503955    
#> 232: 0.927139    1.131930    3.548520    0.640020    0.020371    0.011931    0.503885    
#> 233: 0.926992    1.131560    3.548786    0.641519    0.020379    0.011912    0.503999    
#> 234: 0.927375    1.131338    3.548889    0.641227    0.020504    0.011899    0.504011    
#> 235: 0.928311    1.130722    3.548917    0.640907    0.020706    0.011897    0.503859    
#> 236: 0.928643    1.130901    3.548822    0.643227    0.020848    0.011870    0.503853    
#> 237: 0.929524    1.130764    3.548945    0.645229    0.020832    0.011823    0.503858    
#> 238: 0.930420    1.130765    3.549001    0.648874    0.020837    0.011791    0.504172    
#> 239: 0.929575    1.130969    3.548884    0.647582    0.020885    0.011773    0.504385    
#> 240: 0.929786    1.131025    3.549006    0.647717    0.020964    0.011719    0.504242    
#> 241: 0.930033    1.130788    3.549092    0.646559    0.021113    0.011689    0.504263    
#> 242: 0.929851    1.130343    3.549117    0.645854    0.021261    0.011644    0.504507    
#> 243: 0.929742    1.130461    3.549097    0.645852    0.021449    0.011625    0.504440    
#> 244: 0.929443    1.130223    3.549194    0.645844    0.021471    0.011632    0.504441    
#> 245: 0.929633    1.130020    3.549258    0.644651    0.021679    0.011620    0.504625    
#> 246: 0.929656    1.130088    3.549295    0.643843    0.021692    0.011586    0.504656    
#> 247: 0.928676    1.130029    3.549293    0.642129    0.021700    0.011570    0.504546    
#> 248: 0.928345    1.130418    3.549094    0.641347    0.021686    0.011506    0.504545    
#> 249: 0.927773    1.130558    3.549085    0.640774    0.021766    0.011448    0.504543    
#> 250: 0.928369    1.130724    3.549204    0.639675    0.021850    0.011416    0.504667    
#> 251: 0.928524    1.130795    3.549427    0.639277    0.021908    0.011401    0.504782    
#> 252: 0.929986    1.130754    3.549730    0.641232    0.022011    0.011380    0.505044    
#> 253: 0.930698    1.130724    3.549842    0.644782    0.022121    0.011333    0.505300    
#> 254: 0.930417    1.131091    3.549848    0.644166    0.022177    0.011290    0.505501    
#> 255: 0.930409    1.131292    3.549851    0.643764    0.022185    0.011287    0.505783    
#> 256: 0.930645    1.131219    3.549845    0.643924    0.022202    0.011217    0.506201    
#> 257: 0.930608    1.130787    3.549852    0.643313    0.022353    0.011172    0.506225    
#> 258: 0.930504    1.130314    3.549884    0.644148    0.022408    0.011191    0.506256    
#> 259: 0.930318    1.130230    3.549817    0.643780    0.022338    0.011219    0.506309    
#> 260: 0.930056    1.130493    3.549742    0.644056    0.022341    0.011223    0.506370    
#> 261: 0.930184    1.130511    3.549599    0.645427    0.022333    0.011219    0.506537    
#> 262: 0.929623    1.130774    3.549528    0.644828    0.022352    0.011214    0.506580    
#> 263: 0.929104    1.131016    3.549484    0.644457    0.022440    0.011206    0.506682    
#> 264: 0.928466    1.131100    3.549446    0.644525    0.022607    0.011193    0.506584    
#> 265: 0.928417    1.131279    3.549339    0.645135    0.022785    0.011188    0.506452    
#> 266: 0.928946    1.131604    3.549268    0.646030    0.022889    0.011171    0.506458    
#> 267: 0.929238    1.131702    3.549355    0.648286    0.022922    0.011180    0.506530    
#> 268: 0.928985    1.131703    3.549303    0.647640    0.022883    0.011184    0.506511    
#> 269: 0.929324    1.131426    3.549351    0.647976    0.022803    0.011213    0.506562    
#> 270: 0.929414    1.131386    3.549207    0.647270    0.022684    0.011259    0.506599    
#> 271: 0.929435    1.131348    3.549123    0.647129    0.022583    0.011297    0.506654    
#> 272: 0.929136    1.131283    3.549095    0.645010    0.022521    0.011333    0.506913    
#> 273: 0.928982    1.131374    3.549073    0.645498    0.022502    0.011354    0.507093    
#> 274: 0.928740    1.131455    3.549066    0.645886    0.022478    0.011385    0.506889    
#> 275: 0.928594    1.131595    3.548951    0.645512    0.022471    0.011364    0.506842    
#> 276: 0.928476    1.131738    3.549065    0.645627    0.022483    0.011372    0.506790    
#> 277: 0.928305    1.131939    3.549204    0.645713    0.022484    0.011403    0.506931    
#> 278: 0.928213    1.131962    3.549281    0.646076    0.022550    0.011408    0.506823    
#> 279: 0.928056    1.131844    3.549279    0.646016    0.022549    0.011397    0.506929    
#> 280: 0.927904    1.131955    3.549237    0.645588    0.022523    0.011386    0.506980    
#> 281: 0.927242    1.132122    3.549181    0.644582    0.022570    0.011359    0.506901    
#> 282: 0.926889    1.132372    3.549141    0.644305    0.022623    0.011354    0.506810    
#> 283: 0.926898    1.132573    3.549182    0.645007    0.022683    0.011353    0.506807    
#> 284: 0.926652    1.132520    3.549093    0.645243    0.022693    0.011350    0.506702    
#> 285: 0.926501    1.132626    3.548875    0.646069    0.022721    0.011346    0.506745    
#> 286: 0.925946    1.132905    3.548602    0.645597    0.022768    0.011352    0.506777    
#> 287: 0.925291    1.133217    3.548377    0.645511    0.022759    0.011377    0.506929    
#> 288: 0.924791    1.133383    3.548180    0.644739    0.022793    0.011389    0.506970    
#> 289: 0.924618    1.133552    3.548029    0.645300    0.022848    0.011387    0.506965    
#> 290: 0.924586    1.133623    3.547945    0.645932    0.022847    0.011394    0.507049    
#> 291: 0.924584    1.133761    3.547875    0.646157    0.022799    0.011387    0.506990    
#> 292: 0.924439    1.133834    3.547790    0.646028    0.022746    0.011377    0.506975    
#> 293: 0.924113    1.133941    3.547657    0.646015    0.022710    0.011378    0.506946    
#> 294: 0.923723    1.134070    3.547524    0.646339    0.022725    0.011366    0.506875    
#> 295: 0.923657    1.133965    3.547504    0.646355    0.022703    0.011367    0.506799    
#> 296: 0.923651    1.133907    3.547423    0.646674    0.022699    0.011378    0.506841    
#> 297: 0.923635    1.133944    3.547432    0.646865    0.022634    0.011385    0.507085    
#> 298: 0.923925    1.133856    3.547405    0.647615    0.022603    0.011378    0.506947    
#> 299: 0.923935    1.133864    3.547298    0.648180    0.022572    0.011378    0.506877    
#> 300: 0.924373    1.133774    3.547374    0.649557    0.022555    0.011397    0.506812    
#> 301: 0.924632    1.133729    3.547306    0.650552    0.022555    0.011402    0.506788    
#> 302: 0.924605    1.133773    3.547199    0.650493    0.022592    0.011413    0.506652    
#> 303: 0.924335    1.133938    3.547113    0.650183    0.022608    0.011411    0.506474    
#> 304: 0.924089    1.134225    3.546914    0.650070    0.022641    0.011394    0.506390    
#> 305: 0.923879    1.134254    3.546827    0.650075    0.022649    0.011384    0.506329    
#> 306: 0.923051    1.134273    3.546665    0.649245    0.022697    0.011357    0.506208    
#> 307: 0.922949    1.134261    3.546603    0.650088    0.022737    0.011341    0.506198    
#> 308: 0.922592    1.134240    3.546545    0.650428    0.022783    0.011324    0.506215    
#> 309: 0.922650    1.134124    3.546501    0.650990    0.022831    0.011295    0.506216    
#> 310: 0.923154    1.134114    3.546352    0.652844    0.022890    0.011277    0.506179    
#> 311: 0.923135    1.134152    3.546255    0.653960    0.022939    0.011265    0.506145    
#> 312: 0.922854    1.134077    3.546133    0.654238    0.022951    0.011241    0.506203    
#> 313: 0.922433    1.134004    3.545996    0.654497    0.022956    0.011232    0.506273    
#> 314: 0.921817    1.133948    3.545897    0.654517    0.022948    0.011211    0.506320    
#> 315: 0.921426    1.134045    3.545813    0.654468    0.022976    0.011198    0.506357    
#> 316: 0.920971    1.134158    3.545688    0.654040    0.022996    0.011192    0.506360    
#> 317: 0.921016    1.134230    3.545608    0.654456    0.023045    0.011174    0.506558    
#> 318: 0.921247    1.134292    3.545528    0.655044    0.023071    0.011170    0.506572    
#> 319: 0.921353    1.134304    3.545530    0.655785    0.023074    0.011162    0.506718    
#> 320: 0.920869    1.134353    3.545437    0.656044    0.023077    0.011137    0.506871    
#> 321: 0.920588    1.134489    3.545303    0.656057    0.023126    0.011102    0.507005    
#> 322: 0.920315    1.134539    3.545192    0.656082    0.023180    0.011102    0.506943    
#> 323: 0.919874    1.134568    3.545079    0.656309    0.023216    0.011115    0.506888    
#> 324: 0.919946    1.134719    3.545017    0.657178    0.023234    0.011121    0.506878    
#> 325: 0.919731    1.134867    3.544892    0.657249    0.023213    0.011115    0.506931    
#> 326: 0.919499    1.134792    3.544845    0.656853    0.023200    0.011122    0.506944    
#> 327: 0.919296    1.134777    3.544800    0.656602    0.023172    0.011109    0.506930    
#> 328: 0.919082    1.134910    3.544755    0.656498    0.023168    0.011098    0.506922    
#> 329: 0.918998    1.134993    3.544708    0.656848    0.023122    0.011087    0.506813    
#> 330: 0.918521    1.135098    3.544623    0.656137    0.023097    0.011071    0.506764    
#> 331: 0.918450    1.135370    3.544554    0.656032    0.023097    0.011060    0.506646    
#> 332: 0.918262    1.135412    3.544570    0.655925    0.023096    0.011097    0.506581    
#> 333: 0.918187    1.135473    3.544558    0.655677    0.023087    0.011120    0.506522    
#> 334: 0.918034    1.135505    3.544491    0.656048    0.023080    0.011124    0.506420    
#> 335: 0.918089    1.135603    3.544386    0.656627    0.023077    0.011122    0.506477    
#> 336: 0.917975    1.135820    3.544271    0.657183    0.023092    0.011120    0.506523    
#> 337: 0.917731    1.136069    3.544112    0.658067    0.023131    0.011121    0.506563    
#> 338: 0.917596    1.136150    3.544024    0.658376    0.023150    0.011120    0.506606    
#> 339: 0.917377    1.136229    3.543986    0.658417    0.023135    0.011121    0.506580    
#> 340: 0.917312    1.136186    3.543957    0.658746    0.023131    0.011118    0.506596    
#> 341: 0.917635    1.136131    3.543993    0.659491    0.023109    0.011112    0.506572    
#> 342: 0.917341    1.136138    3.544010    0.659440    0.023120    0.011110    0.506585    
#> 343: 0.917086    1.136235    3.544085    0.658576    0.023146    0.011111    0.506566    
#> 344: 0.917229    1.136235    3.544135    0.658035    0.023166    0.011107    0.506557    
#> 345: 0.917082    1.136176    3.544157    0.657726    0.023146    0.011098    0.506497    
#> 346: 0.916994    1.136225    3.544139    0.657593    0.023111    0.011099    0.506436    
#> 347: 0.917128    1.136104    3.544111    0.658152    0.023115    0.011093    0.506448    
#> 348: 0.916850    1.136124    3.544105    0.657182    0.023118    0.011074    0.506519    
#> 349: 0.916691    1.136104    3.544103    0.656685    0.023132    0.011067    0.506438    
#> 350: 0.916532    1.136164    3.544116    0.656704    0.023132    0.011063    0.506450    
#> 351: 0.916541    1.136326    3.544108    0.656725    0.023132    0.011057    0.506500    
#> 352: 0.916552    1.136422    3.544119    0.656655    0.023111    0.011067    0.506620    
#> 353: 0.916926    1.136516    3.544142    0.657367    0.023083    0.011074    0.506594    
#> 354: 0.917214    1.136588    3.544125    0.658184    0.023067    0.011071    0.506559    
#> 355: 0.917620    1.136529    3.544105    0.659270    0.023022    0.011073    0.506598    
#> 356: 0.917812    1.136480    3.544150    0.660046    0.022993    0.011081    0.506643    
#> 357: 0.918119    1.136505    3.544172    0.660169    0.022966    0.011100    0.506624    
#> 358: 0.918030    1.136545    3.544210    0.660124    0.022946    0.011099    0.506648    
#> 359: 0.917815    1.136614    3.544216    0.659714    0.022912    0.011107    0.506671    
#> 360: 0.917804    1.136622    3.544204    0.659722    0.022879    0.011102    0.506608    
#> 361: 0.917579    1.136711    3.544169    0.659723    0.022896    0.011092    0.506500    
#> 362: 0.917557    1.136773    3.544119    0.659677    0.022914    0.011076    0.506344    
#> 363: 0.917578    1.136752    3.544150    0.659900    0.022919    0.011065    0.506285    
#> 364: 0.917716    1.136807    3.544134    0.660064    0.022918    0.011063    0.506253    
#> 365: 0.917883    1.136856    3.544118    0.660906    0.022943    0.011057    0.506278    
#> 366: 0.917990    1.136907    3.544060    0.661637    0.022948    0.011052    0.506426    
#> 367: 0.918006    1.136958    3.544017    0.662222    0.022977    0.011042    0.506496    
#> 368: 0.918241    1.136970    3.543985    0.662780    0.023016    0.011029    0.506455    
#> 369: 0.918105    1.136970    3.543963    0.662684    0.023028    0.011035    0.506541    
#> 370: 0.917792    1.137034    3.543872    0.662523    0.023065    0.011031    0.506565    
#> 371: 0.917752    1.137052    3.543809    0.662521    0.023077    0.011026    0.506537    
#> 372: 0.917356    1.137108    3.543713    0.662803    0.023106    0.011015    0.506558    
#> 373: 0.917214    1.137257    3.543620    0.663874    0.023116    0.011011    0.506593    
#> 374: 0.917129    1.137314    3.543560    0.664214    0.023105    0.011011    0.506614    
#> 375: 0.916894    1.137395    3.543525    0.664398    0.023136    0.011012    0.506681    
#> 376: 0.916828    1.137431    3.543484    0.665242    0.023164    0.011024    0.506598    
#> 377: 0.916772    1.137520    3.543417    0.665829    0.023138    0.011024    0.506598    
#> 378: 0.916502    1.137561    3.543342    0.665533    0.023111    0.011037    0.506631    
#> 379: 0.916294    1.137457    3.543306    0.665839    0.023095    0.011051    0.506557    
#> 380: 0.916104    1.137435    3.543266    0.665615    0.023061    0.011057    0.506522    
#> 381: 0.915823    1.137419    3.543202    0.665577    0.023055    0.011067    0.506475    
#> 382: 0.915637    1.137410    3.543170    0.665766    0.023011    0.011084    0.506413    
#> 383: 0.915648    1.137439    3.543131    0.666216    0.022992    0.011092    0.506320    
#> 384: 0.915519    1.137409    3.543109    0.666598    0.022957    0.011100    0.506337    
#> 385: 0.915757    1.137412    3.543081    0.667595    0.022938    0.011113    0.506337    
#> 386: 0.916155    1.137492    3.543103    0.668670    0.022948    0.011126    0.506362    
#> 387: 0.916281    1.137595    3.543133    0.668631    0.022967    0.011137    0.506401    
#> 388: 0.916328    1.137649    3.543140    0.668736    0.022998    0.011150    0.506373    
#> 389: 0.916252    1.137556    3.543117    0.668963    0.023020    0.011149    0.506439    
#> 390: 0.916183    1.137579    3.543077    0.669153    0.023028    0.011145    0.506455    
#> 391: 0.916270    1.137560    3.543036    0.669270    0.023027    0.011151    0.506407    
#> 392: 0.916533    1.137542    3.543061    0.669923    0.023037    0.011160    0.506378    
#> 393: 0.916642    1.137490    3.543063    0.670639    0.023053    0.011169    0.506330    
#> 394: 0.916480    1.137468    3.543027    0.670506    0.023057    0.011168    0.506301    
#> 395: 0.916404    1.137520    3.543027    0.670307    0.023069    0.011165    0.506265    
#> 396: 0.916314    1.137544    3.543015    0.670295    0.023064    0.011168    0.506182    
#> 397: 0.916272    1.137481    3.543018    0.670351    0.023057    0.011167    0.506190    
#> 398: 0.916367    1.137358    3.543052    0.670401    0.023036    0.011174    0.506182    
#> 399: 0.916387    1.137334    3.543094    0.670102    0.023040    0.011175    0.506156    
#> 400: 0.916392    1.137343    3.543132    0.670032    0.023048    0.011169    0.506118    
#> 401: 0.916317    1.137346    3.543127    0.669683    0.023043    0.011155    0.506162    
#> 402: 0.916416    1.137361    3.543094    0.669542    0.023050    0.011148    0.506216    
#> 403: 0.916536    1.137357    3.543074    0.669386    0.023039    0.011144    0.506260    
#> 404: 0.916424    1.137338    3.543011    0.669200    0.023063    0.011124    0.506266    
#> 405: 0.916267    1.137323    3.542979    0.669569    0.023107    0.011107    0.506225    
#> 406: 0.916243    1.137320    3.542927    0.670238    0.023160    0.011094    0.506213    
#> 407: 0.916333    1.137396    3.542933    0.670410    0.023201    0.011082    0.506240    
#> 408: 0.916561    1.137482    3.542922    0.670460    0.023237    0.011072    0.506259    
#> 409: 0.916629    1.137484    3.542914    0.670282    0.023237    0.011061    0.506240    
#> 410: 0.916502    1.137504    3.542889    0.669821    0.023241    0.011060    0.506272    
#> 411: 0.916436    1.137408    3.542870    0.669357    0.023262    0.011058    0.506391    
#> 412: 0.916387    1.137396    3.542856    0.669051    0.023270    0.011052    0.506455    
#> 413: 0.916214    1.137437    3.542786    0.668779    0.023273    0.011049    0.506463    
#> 414: 0.915885    1.137533    3.542752    0.668466    0.023283    0.011042    0.506464    
#> 415: 0.915762    1.137560    3.542712    0.668426    0.023284    0.011034    0.506435    
#> 416: 0.915852    1.137622    3.542716    0.668895    0.023286    0.011032    0.506350    
#> 417: 0.915900    1.137617    3.542708    0.669495    0.023284    0.011027    0.506273    
#> 418: 0.915864    1.137638    3.542719    0.669606    0.023286    0.011022    0.506219    
#> 419: 0.915979    1.137709    3.542738    0.669734    0.023293    0.011027    0.506190    
#> 420: 0.916016    1.137745    3.542741    0.669787    0.023287    0.011036    0.506144    
#> 421: 0.915849    1.137716    3.542738    0.669577    0.023281    0.011040    0.506055    
#> 422: 0.915896    1.137665    3.542752    0.669644    0.023255    0.011048    0.506030    
#> 423: 0.916045    1.137600    3.542778    0.669562    0.023235    0.011053    0.506040    
#> 424: 0.915937    1.137533    3.542720    0.669395    0.023221    0.011047    0.506107    
#> 425: 0.916087    1.137580    3.542691    0.669913    0.023252    0.011048    0.506174    
#> 426: 0.916078    1.137629    3.542654    0.670306    0.023256    0.011042    0.506223    
#> 427: 0.915876    1.137644    3.542658    0.670114    0.023241    0.011037    0.506250    
#> 428: 0.915930    1.137650    3.542687    0.670327    0.023244    0.011029    0.506291    
#> 429: 0.916052    1.137681    3.542726    0.670395    0.023235    0.011024    0.506287    
#> 430: 0.916173    1.137677    3.542739    0.670611    0.023239    0.011013    0.506318    
#> 431: 0.916041    1.137675    3.542764    0.670374    0.023240    0.011002    0.506268    
#> 432: 0.915915    1.137734    3.542745    0.670264    0.023265    0.010988    0.506235    
#> 433: 0.915894    1.137827    3.542756    0.670392    0.023277    0.010977    0.506235    
#> 434: 0.915932    1.137798    3.542762    0.670385    0.023250    0.010976    0.506263    
#> 435: 0.916020    1.137800    3.542745    0.670631    0.023231    0.010983    0.506248    
#> 436: 0.915939    1.137807    3.542746    0.670670    0.023247    0.010989    0.506231    
#> 437: 0.916021    1.137812    3.542738    0.670812    0.023258    0.010989    0.506205    
#> 438: 0.915916    1.137864    3.542714    0.670538    0.023265    0.010992    0.506186    
#> 439: 0.915846    1.137934    3.542711    0.670207    0.023269    0.010992    0.506105    
#> 440: 0.915838    1.138000    3.542725    0.669940    0.023271    0.010992    0.506037    
#> 441: 0.915872    1.137979    3.542748    0.669740    0.023288    0.010995    0.505968    
#> 442: 0.915809    1.138029    3.542775    0.669508    0.023305    0.010987    0.505915    
#> 443: 0.915815    1.138086    3.542755    0.669037    0.023314    0.010988    0.505900    
#> 444: 0.915719    1.138135    3.542707    0.669217    0.023329    0.010996    0.505891    
#> 445: 0.915539    1.138141    3.542657    0.669480    0.023326    0.011008    0.506007    
#> 446: 0.915368    1.138200    3.542629    0.669046    0.023301    0.011009    0.506052    
#> 447: 0.915296    1.138268    3.542604    0.668758    0.023292    0.011014    0.506111    
#> 448: 0.915089    1.138290    3.542610    0.668178    0.023305    0.011011    0.506136    
#> 449: 0.915053    1.138361    3.542667    0.667832    0.023330    0.011010    0.506194    
#> 450: 0.915058    1.138366    3.542679    0.667481    0.023351    0.011013    0.506198    
#> 451: 0.915007    1.138429    3.542678    0.667396    0.023368    0.011016    0.506205    
#> 452: 0.914895    1.138481    3.542658    0.667092    0.023381    0.011011    0.506162    
#> 453: 0.914850    1.138526    3.542631    0.666785    0.023379    0.011007    0.506117    
#> 454: 0.914666    1.138616    3.542589    0.666554    0.023393    0.011002    0.506084    
#> 455: 0.914652    1.138633    3.542576    0.666611    0.023383    0.010996    0.506096    
#> 456: 0.914544    1.138607    3.542554    0.666579    0.023395    0.010995    0.506114    
#> 457: 0.914524    1.138609    3.542567    0.666575    0.023399    0.011000    0.506090    
#> 458: 0.914553    1.138669    3.542597    0.666448    0.023402    0.011004    0.506051    
#> 459: 0.914469    1.138759    3.542601    0.666421    0.023403    0.011006    0.506038    
#> 460: 0.914262    1.138802    3.542569    0.666083    0.023405    0.011012    0.506016    
#> 461: 0.914083    1.138843    3.542584    0.665987    0.023410    0.011019    0.506005    
#> 462: 0.913969    1.138879    3.542592    0.665717    0.023397    0.011027    0.505959    
#> 463: 0.914037    1.138910    3.542634    0.665481    0.023380    0.011036    0.505904    
#> 464: 0.914118    1.138953    3.542621    0.665344    0.023372    0.011037    0.505846    
#> 465: 0.914035    1.138880    3.542613    0.665066    0.023363    0.011039    0.505805    
#> 466: 0.913953    1.138882    3.542576    0.664995    0.023345    0.011043    0.505773    
#> 467: 0.913814    1.138897    3.542612    0.664677    0.023328    0.011047    0.505728    
#> 468: 0.913938    1.138893    3.542668    0.664690    0.023330    0.011054    0.505746    
#> 469: 0.914032    1.138986    3.542703    0.664707    0.023362    0.011050    0.505739    
#> 470: 0.914018    1.139047    3.542681    0.664793    0.023372    0.011056    0.505759    
#> 471: 0.913968    1.139077    3.542662    0.664833    0.023365    0.011055    0.505774    
#> 472: 0.913901    1.139109    3.542650    0.664804    0.023386    0.011047    0.505700    
#> 473: 0.913705    1.139158    3.542619    0.664543    0.023382    0.011042    0.505673    
#> 474: 0.913597    1.139188    3.542590    0.664388    0.023383    0.011041    0.505734    
#> 475: 0.913589    1.139269    3.542560    0.664407    0.023395    0.011039    0.505732    
#> 476: 0.913487    1.139287    3.542510    0.664678    0.023409    0.011038    0.505724    
#> 477: 0.913410    1.139321    3.542495    0.664613    0.023399    0.011042    0.505682    
#> 478: 0.913269    1.139311    3.542486    0.664378    0.023385    0.011044    0.505675    
#> 479: 0.913320    1.139325    3.542485    0.664526    0.023387    0.011042    0.505642    
#> 480: 0.913205    1.139328    3.542464    0.664600    0.023389    0.011042    0.505682    
#> 481: 0.913182    1.139343    3.542417    0.664977    0.023397    0.011041    0.505679    
#> 482: 0.913198    1.139354    3.542363    0.665488    0.023412    0.011044    0.505702    
#> 483: 0.913172    1.139375    3.542336    0.665712    0.023430    0.011043    0.505724    
#> 484: 0.913294    1.139304    3.542368    0.666145    0.023430    0.011050    0.505757    
#> 485: 0.913313    1.139309    3.542399    0.666175    0.023442    0.011055    0.505784    
#> 486: 0.913339    1.139318    3.542441    0.666088    0.023444    0.011054    0.505792    
#> 487: 0.913631    1.139312    3.542493    0.666413    0.023435    0.011057    0.505790    
#> 488: 0.913649    1.139294    3.542526    0.666262    0.023438    0.011057    0.505775    
#> 489: 0.913791    1.139254    3.542530    0.666330    0.023452    0.011060    0.505781    
#> 490: 0.913838    1.139268    3.542535    0.666288    0.023470    0.011055    0.505738    
#> 491: 0.913927    1.139317    3.542534    0.666423    0.023469    0.011057    0.505681    
#> 492: 0.914008    1.139362    3.542516    0.666692    0.023464    0.011050    0.505683    
#> 493: 0.914097    1.139363    3.542535    0.667083    0.023458    0.011043    0.505673    
#> 494: 0.914279    1.139369    3.542539    0.667112    0.023460    0.011047    0.505671    
#> 495: 0.914409    1.139385    3.542549    0.667212    0.023454    0.011040    0.505660    
#> 496: 0.914562    1.139444    3.542532    0.667074    0.023452    0.011042    0.505674    
#> 497: 0.914636    1.139470    3.542547    0.666991    0.023434    0.011043    0.505722    
#> 498: 0.914663    1.139468    3.542539    0.666927    0.023434    0.011043    0.505668    
#> 499: 0.914668    1.139526    3.542526    0.666949    0.023421    0.011048    0.505669    
#> 500: 0.914652    1.139547    3.542514    0.666864    0.023396    0.011054    0.505645    
#> [====|====|=>-------------------------------------]  25%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.346494    0.973646    3.446383    0.392401    0.078972    0.016800    1.678691    
#> 002: 0.543145    1.028533    3.487166    0.372781    0.075024    0.016235    1.139165    
#> 003: 0.717662    1.021149    3.504140    0.354142    0.071273    0.015424    0.857182    
#> 004: 0.708971    1.050257    3.512992    0.378445    0.067709    0.016060    0.777334    
#> 005: 0.773569    1.061994    3.505648    0.415341    0.064323    0.015257    0.758218    
#> 006: 0.737262    1.076866    3.501799    0.467371    0.061107    0.014494    0.734317    
#> 007: 0.676017    1.082579    3.491127    0.444002    0.058052    0.013769    0.735265    
#> 008: 0.660811    1.078970    3.481212    0.421802    0.055149    0.015326    0.757474    
#> 009: 0.643518    1.088090    3.470360    0.400712    0.052392    0.014560    0.746650    
#> 010: 0.669948    1.060195    3.467484    0.450212    0.052118    0.013832    0.757264    
#> 011: 0.737624    1.084746    3.484852    0.477148    0.051316    0.015382    0.750341    
#> 012: 0.719913    1.072035    3.482626    0.453291    0.048750    0.017301    0.741047    
#> 013: 0.662140    1.056868    3.475477    0.506523    0.046313    0.017197    0.766322    
#> 014: 0.709011    1.051595    3.488368    0.481197    0.056073    0.016338    0.763371    
#> 015: 0.740456    1.049488    3.509218    0.457137    0.053269    0.015521    0.753916    
#> 016: 0.732395    1.083720    3.511371    0.434280    0.050606    0.015598    0.775513    
#> 017: 0.754639    1.077729    3.513518    0.450048    0.048076    0.014818    0.743467    
#> 018: 0.739312    1.088399    3.499202    0.448855    0.045672    0.014077    0.734871    
#> 019: 0.709356    1.082524    3.496137    0.486813    0.044792    0.013373    0.720026    
#> 020: 0.735499    1.068251    3.504429    0.505079    0.043838    0.013963    0.742890    
#> 021: 0.800806    1.049582    3.518523    0.532429    0.041646    0.013265    0.732990    
#> 022: 0.783255    1.043543    3.528884    0.505808    0.039564    0.013719    0.733771    
#> 023: 0.848270    1.040729    3.541169    0.480517    0.037586    0.017274    0.758705    
#> 024: 0.839473    1.056436    3.532160    0.456492    0.035706    0.019455    0.753452    
#> 025: 0.801119    1.025068    3.524231    0.509619    0.034592    0.018483    0.763005    
#> 026: 0.740334    1.082141    3.497780    0.526130    0.033722    0.017559    0.729452    
#> 027: 0.729880    1.097018    3.487798    0.499824    0.036268    0.016681    0.755398    
#> 028: 0.760627    1.069494    3.505097    0.497981    0.034455    0.018439    0.717814    
#> 029: 0.768851    1.053532    3.496194    0.473082    0.037287    0.019427    0.732631    
#> 030: 0.737291    1.084602    3.498468    0.489013    0.037367    0.018455    0.733397    
#> 031: 0.734135    1.060959    3.510830    0.464562    0.036007    0.020593    0.733256    
#> 032: 0.714886    1.073730    3.504420    0.507833    0.036398    0.019563    0.733377    
#> 033: 0.741634    1.070180    3.495962    0.537754    0.034578    0.020631    0.736989    
#> 034: 0.749460    1.066769    3.493310    0.581222    0.032849    0.019600    0.740160    
#> 035: 0.710374    1.063287    3.485280    0.552161    0.031207    0.018620    0.715380    
#> 036: 0.724937    1.083586    3.485898    0.524553    0.032853    0.017689    0.736606    
#> 037: 0.709907    1.074408    3.488612    0.498325    0.031210    0.018408    0.745890    
#> 038: 0.755714    1.079626    3.484308    0.586170    0.029650    0.019341    0.732316    
#> 039: 0.743803    1.072294    3.501333    0.571408    0.031046    0.019456    0.724727    
#> 040: 0.720082    1.075963    3.509158    0.542837    0.032107    0.018484    0.734915    
#> 041: 0.717509    1.070031    3.495530    0.515695    0.034395    0.017559    0.728440    
#> 042: 0.775495    1.043343    3.502055    0.529516    0.032788    0.016681    0.723078    
#> 043: 0.710594    1.058672    3.489626    0.503041    0.036522    0.015847    0.730216    
#> 044: 0.697698    1.087787    3.490024    0.477889    0.034696    0.015623    0.729184    
#> 045: 0.690473    1.097942    3.482941    0.488954    0.032961    0.014841    0.721442    
#> 046: 0.702837    1.086443    3.493106    0.507149    0.031313    0.015051    0.730560    
#> 047: 0.703604    1.082741    3.484595    0.481792    0.032163    0.014298    0.720357    
#> 048: 0.718425    1.072205    3.493171    0.468226    0.049139    0.013584    0.725632    
#> 049: 0.701020    1.048026    3.491755    0.444815    0.046682    0.013447    0.746913    
#> 050: 0.766091    1.028290    3.498690    0.441841    0.051565    0.014214    0.748313    
#> 051: 0.760344    1.056870    3.496552    0.551618    0.048986    0.013504    0.750711    
#> 052: 0.766522    1.060567    3.504534    0.524037    0.051840    0.012828    0.744049    
#> 053: 0.815563    1.060676    3.505759    0.603943    0.056991    0.012187    0.733846    
#> 054: 0.702706    1.072790    3.498583    0.573746    0.062009    0.011578    0.737787    
#> 055: 0.710224    1.092934    3.487061    0.555571    0.058908    0.010999    0.726959    
#> 056: 0.727088    1.079359    3.481273    0.540829    0.055963    0.011908    0.742864    
#> 057: 0.707157    1.065288    3.486863    0.513788    0.053165    0.012685    0.753494    
#> 058: 0.701138    1.077197    3.481719    0.528211    0.050506    0.015056    0.731937    
#> 059: 0.727524    1.031380    3.490768    0.602632    0.047981    0.020666    0.751894    
#> 060: 0.760979    1.025768    3.508032    0.572501    0.048441    0.019633    0.752708    
#> 061: 0.790985    1.030348    3.518012    0.543876    0.046019    0.018651    0.775866    
#> 062: 0.757024    1.051579    3.518227    0.516682    0.044805    0.017719    0.773695    
#> 063: 0.759894    1.033324    3.519500    0.490848    0.042564    0.017952    0.784843    
#> 064: 0.769917    1.049893    3.524641    0.469243    0.042448    0.017054    0.752565    
#> 065: 0.790774    1.053340    3.507913    0.529495    0.042902    0.017303    0.732303    
#> 066: 0.770861    1.077290    3.512407    0.503020    0.040757    0.019515    0.740626    
#> 067: 0.727897    1.069473    3.498524    0.477869    0.038719    0.019338    0.725204    
#> 068: 0.688627    1.081246    3.505767    0.453976    0.043529    0.018371    0.746075    
#> 069: 0.700295    1.089474    3.490119    0.443854    0.048401    0.017453    0.726665    
#> 070: 0.707839    1.073567    3.497624    0.515019    0.046780    0.018750    0.722036    
#> 071: 0.702587    1.090059    3.485717    0.514149    0.052199    0.018900    0.726812    
#> 072: 0.700391    1.066561    3.490610    0.488441    0.049589    0.017955    0.727521    
#> 073: 0.707238    1.078422    3.500270    0.464019    0.047110    0.017058    0.733574    
#> 074: 0.720067    1.066964    3.496542    0.454505    0.044754    0.016205    0.736801    
#> 075: 0.726950    1.068934    3.511914    0.537314    0.042517    0.015394    0.736696    
#> 076: 0.717630    1.085604    3.493747    0.586088    0.040391    0.014625    0.739280    
#> 077: 0.771315    1.069630    3.492877    0.613302    0.038461    0.013894    0.723496    
#> 078: 0.764230    1.064611    3.500002    0.582636    0.038597    0.013199    0.728079    
#> 079: 0.770769    1.078795    3.501602    0.629211    0.047283    0.013274    0.734282    
#> 080: 0.773238    1.062758    3.502357    0.637903    0.046396    0.012949    0.747736    
#> 081: 0.778944    1.072553    3.498129    0.621935    0.044076    0.014524    0.738647    
#> 082: 0.811334    1.043011    3.507099    0.743533    0.041872    0.017195    0.733004    
#> 083: 0.738967    1.061661    3.503911    0.706357    0.039779    0.019455    0.749231    
#> 084: 0.767086    1.052695    3.501994    0.671039    0.037790    0.019664    0.730500    
#> 085: 0.756606    1.058806    3.509270    0.637487    0.044657    0.018681    0.723976    
#> 086: 0.771481    1.076500    3.502271    0.605613    0.042424    0.017747    0.744232    
#> 087: 0.744977    1.068006    3.495333    0.575332    0.044653    0.016860    0.723493    
#> 088: 0.663428    1.048674    3.493713    0.546565    0.042421    0.016017    0.730486    
#> 089: 0.685007    1.074816    3.482744    0.519237    0.045355    0.015216    0.742034    
#> 090: 0.664291    1.087545    3.473134    0.512974    0.043087    0.014455    0.753171    
#> 091: 0.670842    1.073049    3.469802    0.530163    0.040933    0.014716    0.726851    
#> 092: 0.661313    1.098593    3.462869    0.570408    0.038886    0.016110    0.732042    
#> 093: 0.621096    1.086468    3.462710    0.541887    0.036942    0.015361    0.738590    
#> 094: 0.664182    1.100273    3.460802    0.514793    0.035095    0.018018    0.740217    
#> 095: 0.685041    1.098071    3.489116    0.581896    0.037358    0.018848    0.728821    
#> 096: 0.658524    1.103764    3.474217    0.633853    0.038399    0.017906    0.723977    
#> 097: 0.704555    1.119680    3.473794    0.602161    0.041710    0.017011    0.732482    
#> 098: 0.720535    1.112771    3.481949    0.572053    0.042541    0.016797    0.728285    
#> 099: 0.649651    1.064150    3.472154    0.543450    0.040414    0.015957    0.748173    
#> 100: 0.680657    1.067093    3.485417    0.569560    0.038393    0.015159    0.727131    
#> 101: 0.689266    1.057743    3.474753    0.541082    0.036473    0.014401    0.735848    
#> 102: 0.643175    1.061880    3.459521    0.514028    0.034650    0.013681    0.726816    
#> 103: 0.708531    1.067258    3.482519    0.488326    0.032917    0.012997    0.718823    
#> 104: 0.692060    1.061399    3.499638    0.475734    0.031271    0.014036    0.733277    
#> 105: 0.693015    1.059529    3.485319    0.481395    0.030542    0.015349    0.730589    
#> 106: 0.734376    1.063363    3.488118    0.522630    0.030697    0.014582    0.737227    
#> 107: 0.748757    1.088821    3.495491    0.582110    0.031344    0.013853    0.738209    
#> 108: 0.681236    1.087311    3.479660    0.553005    0.039644    0.015564    0.740610    
#> 109: 0.694732    1.071550    3.495563    0.525354    0.037662    0.014786    0.744861    
#> 110: 0.700786    1.077069    3.483990    0.499087    0.035779    0.014046    0.734827    
#> 111: 0.708520    1.084238    3.491793    0.474132    0.039889    0.013344    0.721456    
#> 112: 0.688540    1.059908    3.483774    0.450426    0.042319    0.014381    0.726482    
#> 113: 0.710537    1.084448    3.498569    0.453495    0.040203    0.016910    0.720367    
#> 114: 0.717293    1.070094    3.498732    0.430820    0.042926    0.018820    0.714322    
#> 115: 0.722366    1.049317    3.500411    0.430996    0.040780    0.017879    0.739103    
#> 116: 0.720391    1.066150    3.490309    0.485061    0.038741    0.019614    0.746481    
#> 117: 0.671045    1.086593    3.489050    0.460808    0.036853    0.022627    0.739120    
#> 118: 0.722612    1.088750    3.492626    0.489235    0.041297    0.021496    0.732625    
#> 119: 0.764203    1.065865    3.495010    0.611766    0.041247    0.020421    0.735490    
#> 120: 0.705315    1.072387    3.484488    0.581178    0.040500    0.019400    0.731194    
#> 121: 0.728728    1.091326    3.491343    0.552119    0.043735    0.018430    0.731855    
#> 122: 0.731658    1.065561    3.501086    0.524513    0.041548    0.018933    0.728280    
#> 123: 0.750804    1.076564    3.496289    0.498287    0.039471    0.018043    0.728652    
#> 124: 0.729412    1.106290    3.486292    0.539202    0.044927    0.019417    0.729086    
#> 125: 0.753960    1.105810    3.485441    0.523067    0.042680    0.018446    0.737633    
#> 126: 0.704546    1.121204    3.474378    0.569978    0.049480    0.017524    0.734084    
#> 127: 0.677927    1.083506    3.475942    0.541479    0.047006    0.016647    0.721967    
#> 128: 0.686315    1.078700    3.495313    0.556693    0.044655    0.015815    0.731286    
#> 129: 0.765683    1.092703    3.505311    0.643496    0.060026    0.015024    0.734291    
#> 130: 0.797253    1.082965    3.504410    0.611321    0.058359    0.014273    0.734253    
#> 131: 0.753817    1.066190    3.497460    0.580755    0.058931    0.017125    0.721429    
#> 132: 0.738443    1.051637    3.495413    0.551717    0.055985    0.016269    0.731562    
#> 133: 0.740460    1.059204    3.505615    0.524131    0.053185    0.015780    0.729817    
#> 134: 0.772397    1.053367    3.516573    0.497925    0.050526    0.014991    0.739008    
#> 135: 0.762632    1.065826    3.514859    0.473029    0.048000    0.016639    0.739799    
#> 136: 0.763554    1.046243    3.505885    0.452011    0.045600    0.015807    0.741693    
#> 137: 0.810873    1.055507    3.501158    0.564977    0.043320    0.015016    0.769361    
#> 138: 0.730862    1.043220    3.498979    0.543003    0.041154    0.014266    0.748881    
#> 139: 0.700370    1.059523    3.507835    0.515853    0.043705    0.013552    0.749534    
#> 140: 0.771329    1.081346    3.509060    0.547652    0.041519    0.016415    0.752137    
#> 141: 0.761075    1.059895    3.516363    0.555092    0.040668    0.015594    0.750811    
#> 142: 0.760308    1.038284    3.519948    0.527338    0.050957    0.014881    0.759611    
#> 143: 0.713198    1.050242    3.515071    0.500971    0.048409    0.016722    0.754216    
#> 144: 0.734288    1.071768    3.511273    0.475922    0.045989    0.015886    0.729388    
#> 145: 0.682962    1.088292    3.486985    0.452126    0.043689    0.015092    0.735140    
#> 146: 0.717282    1.097418    3.493820    0.476923    0.041505    0.014578    0.716398    
#> 147: 0.666444    1.082451    3.487412    0.453077    0.039429    0.014926    0.723320    
#> 148: 0.693743    1.088638    3.505002    0.430423    0.037458    0.015331    0.723748    
#> 149: 0.679800    1.066580    3.502227    0.408902    0.035585    0.014840    0.753080    
#> 150: 0.742239    1.068719    3.496677    0.463822    0.034167    0.014098    0.753055    
#> 151: 0.656251    1.096249    3.486465    0.440631    0.045451    0.013394    0.726616    
#> 152: 0.659296    1.079593    3.491627    0.417567    0.044511    0.014428    0.735173    
#> 153: 0.704169    1.096140    3.503988    0.434447    0.049930    0.013463    0.732927    
#> 154: 0.735053    1.094958    3.508080    0.441711    0.037606    0.013835    0.760448    
#> 155: 0.728369    1.080606    3.490168    0.585265    0.040392    0.014640    0.722823    
#> 156: 0.730137    1.077281    3.506036    0.498441    0.038919    0.018920    0.746797    
#> 157: 0.681078    1.063624    3.513431    0.409158    0.037925    0.016568    0.734842    
#> 158: 0.756254    1.065024    3.494866    0.486292    0.041641    0.012235    0.737847    
#> 159: 0.718995    1.094839    3.501630    0.427282    0.038109    0.012512    0.749383    
#> 160: 0.773120    1.053446    3.512461    0.451385    0.039270    0.013907    0.755778    
#> 161: 0.765362    1.044064    3.516822    0.434836    0.044169    0.014110    0.749237    
#> 162: 0.737052    1.070640    3.495679    0.483555    0.034900    0.013500    0.758234    
#> 163: 0.712290    1.057606    3.493009    0.538091    0.036275    0.013368    0.743234    
#> 164: 0.716436    1.081550    3.491660    0.442503    0.036821    0.011128    0.738366    
#> 165: 0.719666    1.070866    3.489193    0.518223    0.043646    0.009198    0.739349    
#> 166: 0.739963    1.102885    3.488257    0.572171    0.050287    0.008939    0.730960    
#> 167: 0.724678    1.098718    3.500193    0.468067    0.039286    0.011405    0.723394    
#> 168: 0.740075    1.094227    3.497499    0.466188    0.036351    0.014453    0.706520    
#> 169: 0.725931    1.116327    3.485642    0.483558    0.035931    0.011350    0.721760    
#> 170: 0.714459    1.092068    3.484497    0.463873    0.032738    0.013226    0.733370    
#> 171: 0.713686    1.072718    3.490503    0.469224    0.036756    0.009819    0.729792    
#> 172: 0.701479    1.083453    3.492052    0.498109    0.034179    0.011024    0.721266    
#> 173: 0.669255    1.097506    3.476195    0.402083    0.033193    0.013038    0.728484    
#> 174: 0.679726    1.104855    3.479756    0.423143    0.030501    0.012663    0.725772    
#> 175: 0.669256    1.098742    3.472770    0.473990    0.037437    0.011336    0.754605    
#> 176: 0.670340    1.118814    3.488802    0.396783    0.040232    0.012724    0.727072    
#> 177: 0.696189    1.099107    3.489941    0.515619    0.038845    0.007500    0.742216    
#> 178: 0.697531    1.083147    3.481984    0.471216    0.038920    0.006249    0.743983    
#> 179: 0.742846    1.105163    3.490452    0.588929    0.051478    0.007583    0.738434    
#> 180: 0.673364    1.084708    3.489001    0.392158    0.058102    0.008078    0.734128    
#> 181: 0.686729    1.069041    3.478791    0.401455    0.054573    0.008854    0.730375    
#> 182: 0.676890    1.100166    3.460478    0.481642    0.053322    0.008976    0.739561    
#> 183: 0.640817    1.133501    3.454070    0.433856    0.038559    0.007564    0.741630    
#> 184: 0.644854    1.105865    3.460875    0.513069    0.042483    0.007633    0.737650    
#> 185: 0.651447    1.124990    3.462813    0.453066    0.053317    0.008360    0.746410    
#> 186: 0.625831    1.121082    3.467152    0.458955    0.051038    0.006901    0.749440    
#> 187: 0.647464    1.107950    3.483725    0.474689    0.046031    0.008491    0.735431    
#> 188: 0.643826    1.097465    3.474259    0.518921    0.039015    0.009803    0.732218    
#> 189: 0.701433    1.089075    3.480363    0.515095    0.036636    0.011368    0.725400    
#> 190: 0.731800    1.087686    3.493641    0.500622    0.033781    0.010996    0.723426    
#> 191: 0.715399    1.092130    3.486577    0.512908    0.037381    0.010794    0.719788    
#> 192: 0.680114    1.078196    3.488991    0.467826    0.037835    0.010595    0.739065    
#> 193: 0.720771    1.095188    3.481645    0.490000    0.044902    0.009291    0.730517    
#> 194: 0.692373    1.104382    3.477488    0.494053    0.049750    0.011329    0.745034    
#> 195: 0.667099    1.097480    3.477106    0.435070    0.042258    0.011639    0.746903    
#> 196: 0.657485    1.104741    3.486162    0.467333    0.046227    0.010432    0.747461    
#> 197: 0.632959    1.110925    3.500849    0.399301    0.046884    0.007799    0.764339    
#> 198: 0.755360    1.082878    3.487200    0.482879    0.047506    0.010060    0.738651    
#> 199: 0.710491    1.061677    3.498576    0.397239    0.045759    0.010494    0.730267    
#> 200: 0.659377    1.076931    3.491110    0.345252    0.039755    0.013289    0.745719    
#> 201: 0.649836    1.074202    3.487373    0.367456    0.039148    0.012717    0.740156    
#> 202: 0.655495    1.079432    3.484508    0.376237    0.039637    0.013104    0.742952    
#> 203: 0.677054    1.079766    3.483028    0.407886    0.042017    0.012922    0.739953    
#> 204: 0.682878    1.077476    3.484038    0.423198    0.044696    0.012587    0.742573    
#> 205: 0.685314    1.082347    3.486210    0.430659    0.047254    0.012397    0.746067    
#> 206: 0.694066    1.083051    3.489714    0.448206    0.046017    0.012971    0.750710    
#> 207: 0.690647    1.081855    3.489342    0.448441    0.044055    0.013328    0.750819    
#> 208: 0.685890    1.080722    3.489545    0.439088    0.042934    0.013632    0.749829    
#> 209: 0.689000    1.079297    3.489591    0.442233    0.041617    0.013787    0.748933    
#> 210: 0.690841    1.077670    3.489722    0.448111    0.041328    0.013726    0.746149    
#> 211: 0.695900    1.077794    3.490479    0.455377    0.040980    0.013627    0.743941    
#> 212: 0.696779    1.077255    3.490818    0.458147    0.040917    0.013482    0.743594    
#> 213: 0.695239    1.074941    3.490386    0.455525    0.041081    0.013793    0.744295    
#> 214: 0.696528    1.073245    3.490519    0.456714    0.041341    0.014009    0.744455    
#> 215: 0.696535    1.072287    3.491723    0.455015    0.041291    0.014375    0.743700    
#> 216: 0.699553    1.070826    3.492293    0.461054    0.040971    0.014567    0.744406    
#> 217: 0.700089    1.071370    3.492303    0.464274    0.040543    0.014622    0.743470    
#> 218: 0.700872    1.072116    3.491987    0.463048    0.040586    0.014444    0.743297    
#> 219: 0.700126    1.072665    3.491033    0.464707    0.040715    0.014316    0.743395    
#> 220: 0.698525    1.073728    3.490781    0.462258    0.041028    0.014284    0.743169    
#> 221: 0.696658    1.074107    3.490509    0.461136    0.041074    0.014244    0.742574    
#> 222: 0.698388    1.074057    3.490508    0.464083    0.040500    0.014417    0.742572    
#> 223: 0.700813    1.073497    3.490935    0.466985    0.040105    0.014649    0.742183    
#> 224: 0.702779    1.073346    3.491500    0.468487    0.039855    0.014677    0.742402    
#> 225: 0.702795    1.071988    3.492259    0.466908    0.040005    0.014605    0.742812    
#> 226: 0.702597    1.070797    3.491894    0.467663    0.039949    0.014484    0.743317    
#> 227: 0.701531    1.070168    3.492004    0.469912    0.040195    0.014322    0.743290    
#> 228: 0.700602    1.071185    3.491776    0.468657    0.040463    0.014247    0.743562    
#> 229: 0.700073    1.071449    3.491349    0.469325    0.040604    0.014153    0.743086    
#> 230: 0.699841    1.071636    3.490851    0.467044    0.040563    0.014060    0.743325    
#> 231: 0.698959    1.071799    3.491063    0.464997    0.040472    0.013998    0.743463    
#> 232: 0.697412    1.071271    3.491226    0.463422    0.040521    0.013955    0.743791    
#> 233: 0.696580    1.070360    3.491308    0.464508    0.040447    0.014014    0.743808    
#> 234: 0.696818    1.070562    3.491350    0.466899    0.040337    0.014085    0.743186    
#> 235: 0.696635    1.070120    3.491129    0.467878    0.040215    0.014044    0.742589    
#> 236: 0.696385    1.070239    3.491036    0.467965    0.040268    0.014013    0.742593    
#> 237: 0.698869    1.070324    3.491277    0.469734    0.040302    0.014036    0.743252    
#> 238: 0.700641    1.070857    3.491447    0.472656    0.040339    0.014073    0.743372    
#> 239: 0.701704    1.071345    3.491798    0.474188    0.040087    0.014121    0.743462    
#> 240: 0.702900    1.071529    3.492091    0.475685    0.040094    0.014146    0.742710    
#> 241: 0.704640    1.071311    3.492301    0.477604    0.039920    0.014225    0.742419    
#> 242: 0.704350    1.071090    3.492235    0.478029    0.039819    0.014273    0.743009    
#> 243: 0.703627    1.071937    3.492275    0.477653    0.039953    0.014310    0.743494    
#> 244: 0.703391    1.072020    3.492477    0.475476    0.039993    0.014391    0.743738    
#> 245: 0.704075    1.071645    3.492907    0.474408    0.039825    0.014420    0.744335    
#> 246: 0.704290    1.071175    3.493341    0.471786    0.039740    0.014500    0.744354    
#> 247: 0.705178    1.071062    3.493626    0.473262    0.039793    0.014558    0.744207    
#> 248: 0.705910    1.071350    3.493775    0.474024    0.039917    0.014558    0.744134    
#> 249: 0.706025    1.071267    3.493882    0.475420    0.039763    0.014609    0.744305    
#> 250: 0.706829    1.071661    3.494123    0.477251    0.039589    0.014638    0.744672    
#> 251: 0.707468    1.072260    3.494263    0.479268    0.039480    0.014651    0.745001    
#> 252: 0.708817    1.072314    3.494618    0.480898    0.039338    0.014686    0.744860    
#> 253: 0.708858    1.072330    3.494896    0.480995    0.039306    0.014701    0.744881    
#> 254: 0.709234    1.072779    3.495191    0.480061    0.039226    0.014736    0.744936    
#> 255: 0.709376    1.072711    3.495412    0.478294    0.039251    0.014726    0.745187    
#> 256: 0.710185    1.072496    3.495543    0.478397    0.039321    0.014772    0.745104    
#> 257: 0.710619    1.071987    3.495592    0.478759    0.039482    0.014774    0.744699    
#> 258: 0.710438    1.071069    3.495655    0.478210    0.039526    0.014695    0.744653    
#> 259: 0.710652    1.070489    3.495669    0.477979    0.039663    0.014677    0.744943    
#> 260: 0.710434    1.070539    3.495788    0.477308    0.039698    0.014660    0.744783    
#> 261: 0.710056    1.070775    3.495718    0.476004    0.039705    0.014619    0.744741    
#> 262: 0.709931    1.071281    3.495648    0.476134    0.039721    0.014612    0.744534    
#> 263: 0.709579    1.071608    3.495623    0.476613    0.039741    0.014579    0.744305    
#> 264: 0.709677    1.071480    3.495676    0.477835    0.039793    0.014581    0.744419    
#> 265: 0.709985    1.071242    3.495747    0.478314    0.039812    0.014596    0.744455    
#> 266: 0.710059    1.071172    3.495667    0.479664    0.039766    0.014614    0.744303    
#> 267: 0.710510    1.071366    3.495863    0.480746    0.039780    0.014622    0.744146    
#> 268: 0.710831    1.071359    3.495802    0.481371    0.039840    0.014635    0.743905    
#> 269: 0.710971    1.071346    3.495789    0.481969    0.039829    0.014637    0.743645    
#> 270: 0.711040    1.071448    3.495557    0.482641    0.039743    0.014660    0.743219    
#> 271: 0.710857    1.071801    3.495205    0.482584    0.039734    0.014651    0.743076    
#> 272: 0.711035    1.071876    3.495103    0.482418    0.039753    0.014631    0.743231    
#> 273: 0.710219    1.072199    3.495060    0.481538    0.039866    0.014626    0.743463    
#> 274: 0.710098    1.072250    3.495137    0.480261    0.039899    0.014592    0.743398    
#> 275: 0.710055    1.072176    3.495111    0.479767    0.039937    0.014552    0.743046    
#> 276: 0.710645    1.072068    3.495372    0.479442    0.039889    0.014546    0.743003    
#> 277: 0.710786    1.072151    3.495677    0.478848    0.039827    0.014556    0.742802    
#> 278: 0.711460    1.072088    3.495904    0.479090    0.039811    0.014573    0.742841    
#> 279: 0.711813    1.071825    3.496034    0.480241    0.039821    0.014585    0.743014    
#> 280: 0.712071    1.071984    3.496091    0.480477    0.039798    0.014584    0.742998    
#> 281: 0.711608    1.072054    3.496196    0.480861    0.039821    0.014543    0.743100    
#> 282: 0.711491    1.072243    3.496271    0.481713    0.039891    0.014539    0.743108    
#> 283: 0.711331    1.072295    3.496329    0.482749    0.039939    0.014560    0.743302    
#> 284: 0.711117    1.072122    3.496289    0.483643    0.039934    0.014564    0.743126    
#> 285: 0.711614    1.072121    3.496216    0.485001    0.039958    0.014619    0.743212    
#> 286: 0.711928    1.072281    3.495945    0.486140    0.039886    0.014666    0.743421    
#> 287: 0.712107    1.072516    3.495729    0.487195    0.039865    0.014706    0.743630    
#> 288: 0.712505    1.072428    3.495649    0.487875    0.039806    0.014725    0.743624    
#> 289: 0.712573    1.072385    3.495538    0.488443    0.039774    0.014738    0.743434    
#> 290: 0.712432    1.072585    3.495396    0.488415    0.039860    0.014731    0.743381    
#> 291: 0.712566    1.072967    3.495324    0.489083    0.039953    0.014704    0.743209    
#> 292: 0.712494    1.072977    3.495257    0.490049    0.039946    0.014708    0.743087    
#> 293: 0.712508    1.072964    3.495136    0.491039    0.039905    0.014714    0.742896    
#> 294: 0.712396    1.073406    3.495011    0.491667    0.039970    0.014687    0.742850    
#> 295: 0.712893    1.073385    3.494956    0.493113    0.040078    0.014633    0.742676    
#> 296: 0.712891    1.073307    3.494889    0.492570    0.040121    0.014587    0.742798    
#> 297: 0.713370    1.073395    3.494865    0.492962    0.040173    0.014547    0.742716    
#> 298: 0.713483    1.073303    3.494802    0.493441    0.040193    0.014511    0.742601    
#> 299: 0.713442    1.073461    3.494606    0.494097    0.040188    0.014498    0.742302    
#> 300: 0.713427    1.073519    3.494585    0.494164    0.040158    0.014512    0.742012    
#> 301: 0.713451    1.073453    3.494427    0.493392    0.040085    0.014488    0.742158    
#> 302: 0.713590    1.073398    3.494400    0.492888    0.040045    0.014479    0.742257    
#> 303: 0.713746    1.073420    3.494334    0.493189    0.039955    0.014473    0.742115    
#> 304: 0.714021    1.073549    3.494161    0.494591    0.039934    0.014453    0.742131    
#> 305: 0.713671    1.073293    3.494009    0.495062    0.039926    0.014431    0.742132    
#> 306: 0.713060    1.073134    3.493919    0.494366    0.039883    0.014412    0.742212    
#> 307: 0.712854    1.073015    3.493829    0.493468    0.039896    0.014428    0.742330    
#> 308: 0.712247    1.072834    3.493732    0.493073    0.039877    0.014425    0.742610    
#> 309: 0.712109    1.072811    3.493601    0.493337    0.039865    0.014413    0.742899    
#> 310: 0.712536    1.072886    3.493487    0.493921    0.039879    0.014372    0.743021    
#> 311: 0.712396    1.072918    3.493393    0.493859    0.039868    0.014355    0.742967    
#> 312: 0.712244    1.072899    3.493296    0.494163    0.039797    0.014338    0.742946    
#> 313: 0.712280    1.072614    3.493234    0.494031    0.039744    0.014342    0.743005    
#> 314: 0.712233    1.072470    3.493281    0.494042    0.039676    0.014364    0.742883    
#> 315: 0.712369    1.072413    3.493279    0.494131    0.039645    0.014356    0.742697    
#> 316: 0.712597    1.072454    3.493316    0.494378    0.039635    0.014330    0.742523    
#> 317: 0.712912    1.072445    3.493433    0.493949    0.039674    0.014324    0.742373    
#> 318: 0.713132    1.072479    3.493403    0.494293    0.039703    0.014321    0.742269    
#> 319: 0.713286    1.072428    3.493440    0.494399    0.039719    0.014330    0.742211    
#> 320: 0.712627    1.072585    3.493294    0.494320    0.039777    0.014302    0.742245    
#> 321: 0.712255    1.072816    3.493081    0.494804    0.039804    0.014265    0.742227    
#> 322: 0.711818    1.072934    3.492832    0.494625    0.039831    0.014239    0.742259    
#> 323: 0.711063    1.073071    3.492651    0.494556    0.039822    0.014206    0.742296    
#> 324: 0.711062    1.073099    3.492721    0.494429    0.039846    0.014217    0.742176    
#> 325: 0.710942    1.073231    3.492651    0.494120    0.039847    0.014216    0.742063    
#> 326: 0.710879    1.073067    3.492603    0.494542    0.039859    0.014205    0.741893    
#> 327: 0.710957    1.073126    3.492576    0.494881    0.039759    0.014219    0.741709    
#> 328: 0.710973    1.073255    3.492531    0.494702    0.039754    0.014194    0.741726    
#> 329: 0.711313    1.073377    3.492583    0.494290    0.039740    0.014167    0.741677    
#> 330: 0.711176    1.073574    3.492528    0.493797    0.039802    0.014160    0.741695    
#> 331: 0.711346    1.073903    3.492488    0.493704    0.039821    0.014140    0.741774    
#> 332: 0.711707    1.073807    3.492610    0.493783    0.039808    0.014126    0.741814    
#> 333: 0.711864    1.073771    3.492586    0.494330    0.039836    0.014106    0.741744    
#> 334: 0.711594    1.073813    3.492468    0.494398    0.039845    0.014115    0.741710    
#> 335: 0.711317    1.073856    3.492377    0.494858    0.039892    0.014122    0.741613    
#> 336: 0.710966    1.073909    3.492334    0.494822    0.039889    0.014141    0.741672    
#> 337: 0.710882    1.074254    3.492208    0.495284    0.039962    0.014147    0.741563    
#> 338: 0.710714    1.074470    3.492155    0.495850    0.039993    0.014131    0.741557    
#> 339: 0.710339    1.074662    3.492095    0.495603    0.040014    0.014120    0.741456    
#> 340: 0.710113    1.074615    3.492062    0.495746    0.040064    0.014097    0.741396    
#> 341: 0.710297    1.074535    3.492091    0.495528    0.040063    0.014072    0.741374    
#> 342: 0.709846    1.074475    3.491988    0.495276    0.040077    0.014047    0.741287    
#> 343: 0.709429    1.074530    3.492011    0.494420    0.040048    0.014038    0.741291    
#> 344: 0.709461    1.074524    3.492055    0.494309    0.040041    0.014041    0.741292    
#> 345: 0.709203    1.074520    3.492066    0.493672    0.040091    0.014035    0.741212    
#> 346: 0.708854    1.074655    3.491954    0.493241    0.040072    0.014022    0.741149    
#> 347: 0.708903    1.074641    3.491819    0.493902    0.040055    0.014026    0.741226    
#> 348: 0.708577    1.074909    3.491677    0.494084    0.040031    0.014005    0.741403    
#> 349: 0.708649    1.074834    3.491633    0.494923    0.040036    0.014015    0.741356    
#> 350: 0.708718    1.074788    3.491623    0.495401    0.040059    0.014013    0.741209    
#> 351: 0.708957    1.074926    3.491614    0.495851    0.040066    0.014010    0.741227    
#> 352: 0.708657    1.075114    3.491552    0.495999    0.040106    0.013989    0.741447    
#> 353: 0.708850    1.075310    3.491572    0.495826    0.040177    0.013949    0.741461    
#> 354: 0.709104    1.075375    3.491541    0.496005    0.040246    0.013911    0.741401    
#> 355: 0.709373    1.075308    3.491533    0.495607    0.040278    0.013882    0.741267    
#> 356: 0.709201    1.075344    3.491542    0.494890    0.040339    0.013859    0.741312    
#> 357: 0.709300    1.075387    3.491552    0.495077    0.040372    0.013844    0.741264    
#> 358: 0.709306    1.075551    3.491582    0.494729    0.040342    0.013835    0.741320    
#> 359: 0.709096    1.075794    3.491549    0.494423    0.040393    0.013831    0.741272    
#> 360: 0.708898    1.075889    3.491398    0.494139    0.040390    0.013820    0.741114    
#> 361: 0.708429    1.076041    3.491265    0.493391    0.040421    0.013818    0.740966    
#> 362: 0.708385    1.075982    3.491213    0.492822    0.040405    0.013819    0.740832    
#> 363: 0.708227    1.075959    3.491196    0.492367    0.040392    0.013828    0.740654    
#> 364: 0.708152    1.076031    3.491151    0.492128    0.040307    0.013853    0.740649    
#> 365: 0.708157    1.075971    3.491195    0.491683    0.040292    0.013885    0.740666    
#> 366: 0.708115    1.075945    3.491166    0.491435    0.040261    0.013909    0.740668    
#> 367: 0.708001    1.075962    3.491165    0.491254    0.040258    0.013900    0.740630    
#> 368: 0.707920    1.076163    3.491033    0.491741    0.040292    0.013881    0.740560    
#> 369: 0.707885    1.076230    3.490924    0.492378    0.040299    0.013858    0.740582    
#> 370: 0.707910    1.076335    3.490854    0.492758    0.040383    0.013839    0.740618    
#> 371: 0.707577    1.076374    3.490650    0.492996    0.040375    0.013819    0.740630    
#> 372: 0.707285    1.076561    3.490512    0.493359    0.040400    0.013798    0.740719    
#> 373: 0.706697    1.076752    3.490355    0.493159    0.040421    0.013784    0.740818    
#> 374: 0.706531    1.076756    3.490264    0.492949    0.040437    0.013766    0.740779    
#> 375: 0.706154    1.076831    3.490198    0.492299    0.040448    0.013747    0.740803    
#> 376: 0.705897    1.076934    3.490071    0.491672    0.040464    0.013740    0.740723    
#> 377: 0.705727    1.077131    3.489967    0.491254    0.040565    0.013718    0.740732    
#> 378: 0.705628    1.077222    3.489864    0.490851    0.040564    0.013700    0.740763    
#> 379: 0.705444    1.077164    3.489831    0.490575    0.040593    0.013670    0.740726    
#> 380: 0.705387    1.077210    3.489788    0.490483    0.040658    0.013662    0.740640    
#> 381: 0.705223    1.077331    3.489732    0.490431    0.040673    0.013640    0.740683    
#> 382: 0.705135    1.077359    3.489736    0.489887    0.040697    0.013628    0.740685    
#> 383: 0.705400    1.077351    3.489796    0.489856    0.040712    0.013621    0.740500    
#> 384: 0.705553    1.077368    3.489844    0.490005    0.040778    0.013602    0.740420    
#> 385: 0.705763    1.077427    3.489890    0.489989    0.040817    0.013581    0.740413    
#> 386: 0.706108    1.077493    3.490028    0.489859    0.040868    0.013562    0.740385    
#> 387: 0.706390    1.077473    3.490124    0.489861    0.040878    0.013557    0.740359    
#> 388: 0.706704    1.077415    3.490189    0.490343    0.040885    0.013547    0.740332    
#> 389: 0.706943    1.077222    3.490224    0.490450    0.040905    0.013537    0.740471    
#> 390: 0.706885    1.077154    3.490243    0.490482    0.040908    0.013537    0.740538    
#> 391: 0.706964    1.077097    3.490221    0.490523    0.040895    0.013553    0.740510    
#> 392: 0.707139    1.077140    3.490261    0.490722    0.040907    0.013558    0.740501    
#> 393: 0.707169    1.077166    3.490155    0.490791    0.040930    0.013543    0.740577    
#> 394: 0.707192    1.077220    3.490076    0.490766    0.040920    0.013541    0.740637    
#> 395: 0.707386    1.077343    3.490064    0.490740    0.040938    0.013536    0.740605    
#> 396: 0.707449    1.077399    3.490097    0.491141    0.040928    0.013539    0.740542    
#> 397: 0.707658    1.077490    3.490140    0.491762    0.040954    0.013546    0.740505    
#> 398: 0.707962    1.077590    3.490142    0.492354    0.040947    0.013536    0.740461    
#> 399: 0.708108    1.077695    3.490171    0.492664    0.040971    0.013538    0.740395    
#> 400: 0.708331    1.077774    3.490205    0.493184    0.040987    0.013532    0.740378    
#> 401: 0.708327    1.077829    3.490198    0.493106    0.040962    0.013538    0.740424    
#> 402: 0.708315    1.077783    3.490202    0.493016    0.040993    0.013542    0.740340    
#> 403: 0.708793    1.077682    3.490237    0.493620    0.041016    0.013556    0.740284    
#> 404: 0.708905    1.077686    3.490210    0.493521    0.041061    0.013565    0.740159    
#> 405: 0.708812    1.077658    3.490202    0.493573    0.041087    0.013566    0.740165    
#> 406: 0.708953    1.077722    3.490212    0.493443    0.041164    0.013572    0.740309    
#> 407: 0.708960    1.077846    3.490207    0.493598    0.041162    0.013562    0.740373    
#> 408: 0.709102    1.078016    3.490195    0.493259    0.041124    0.013557    0.740450    
#> 409: 0.709096    1.078065    3.490159    0.492701    0.041099    0.013537    0.740445    
#> 410: 0.709081    1.078140    3.490110    0.492556    0.041100    0.013521    0.740357    
#> 411: 0.709130    1.078073    3.490056    0.492833    0.041110    0.013505    0.740311    
#> 412: 0.709154    1.078091    3.490058    0.492335    0.041070    0.013512    0.740299    
#> 413: 0.709162    1.078086    3.490050    0.491809    0.041068    0.013503    0.740294    
#> 414: 0.709107    1.078080    3.490074    0.491902    0.041108    0.013492    0.740159    
#> 415: 0.709426    1.078065    3.490065    0.492065    0.041099    0.013487    0.740165    
#> 416: 0.709624    1.078079    3.490088    0.492063    0.041112    0.013493    0.740086    
#> 417: 0.709649    1.078033    3.490104    0.492024    0.041090    0.013503    0.740091    
#> 418: 0.709460    1.078073    3.490086    0.491781    0.041067    0.013516    0.740088    
#> 419: 0.709384    1.078177    3.490041    0.491822    0.041032    0.013525    0.740149    
#> 420: 0.709464    1.078155    3.490046    0.491875    0.040998    0.013530    0.740103    
#> 421: 0.709434    1.078139    3.490090    0.491440    0.041000    0.013532    0.740040    
#> 422: 0.709464    1.077999    3.490072    0.491605    0.041002    0.013548    0.739998    
#> 423: 0.709478    1.077996    3.490021    0.491350    0.041048    0.013551    0.739968    
#> 424: 0.709287    1.078012    3.489926    0.491330    0.041101    0.013552    0.739949    
#> 425: 0.709335    1.078070    3.489888    0.491571    0.041145    0.013555    0.740059    
#> 426: 0.709360    1.078106    3.489947    0.491711    0.041126    0.013556    0.740051    
#> 427: 0.709329    1.078085    3.489974    0.491608    0.041075    0.013565    0.740036    
#> 428: 0.709231    1.078109    3.490000    0.491136    0.041055    0.013550    0.740103    
#> 429: 0.709341    1.078055    3.490060    0.490872    0.041015    0.013545    0.740172    
#> 430: 0.709231    1.078015    3.490083    0.490519    0.041022    0.013543    0.740215    
#> 431: 0.709228    1.078037    3.490089    0.490166    0.041013    0.013541    0.740152    
#> 432: 0.709284    1.078071    3.490091    0.490394    0.041060    0.013538    0.740146    
#> 433: 0.709349    1.078129    3.490129    0.491115    0.041117    0.013551    0.740093    
#> 434: 0.709209    1.078029    3.490111    0.491463    0.041160    0.013552    0.740021    
#> 435: 0.709297    1.077999    3.490115    0.491657    0.041170    0.013567    0.739904    
#> 436: 0.709291    1.077894    3.490115    0.491520    0.041142    0.013577    0.739905    
#> 437: 0.709148    1.077876    3.490084    0.491286    0.041108    0.013588    0.739983    
#> 438: 0.709118    1.078055    3.490047    0.491373    0.041112    0.013580    0.739930    
#> 439: 0.709196    1.078167    3.490001    0.491691    0.041089    0.013579    0.739868    
#> 440: 0.709018    1.078226    3.489998    0.491376    0.041071    0.013560    0.739874    
#> 441: 0.709193    1.078197    3.490050    0.491208    0.041082    0.013568    0.739825    
#> 442: 0.709226    1.078291    3.490091    0.491221    0.041065    0.013568    0.739743    
#> 443: 0.709530    1.078363    3.490112    0.491551    0.041025    0.013586    0.739655    
#> 444: 0.709523    1.078471    3.490058    0.491508    0.040969    0.013598    0.739569    
#> 445: 0.709599    1.078541    3.490008    0.492116    0.040954    0.013606    0.739514    
#> 446: 0.709562    1.078599    3.489972    0.492779    0.040917    0.013614    0.739488    
#> 447: 0.709656    1.078678    3.489966    0.493031    0.040868    0.013625    0.739493    
#> 448: 0.709671    1.078737    3.489973    0.493069    0.040823    0.013638    0.739494    
#> 449: 0.709815    1.078799    3.490071    0.492778    0.040786    0.013644    0.739502    
#> 450: 0.709884    1.078783    3.490120    0.492714    0.040735    0.013636    0.739588    
#> 451: 0.710104    1.078775    3.490141    0.493402    0.040702    0.013632    0.739664    
#> 452: 0.710226    1.078766    3.490154    0.493674    0.040703    0.013631    0.739644    
#> 453: 0.710272    1.078743    3.490161    0.493533    0.040734    0.013616    0.739591    
#> 454: 0.710193    1.078855    3.490127    0.493472    0.040746    0.013599    0.739506    
#> 455: 0.710166    1.078870    3.490097    0.493233    0.040777    0.013579    0.739568    
#> 456: 0.710009    1.078870    3.490091    0.493214    0.040797    0.013570    0.739572    
#> 457: 0.710007    1.078816    3.490125    0.493370    0.040788    0.013558    0.739575    
#> 458: 0.710048    1.078883    3.490157    0.493623    0.040766    0.013552    0.739571    
#> 459: 0.709907    1.078996    3.490149    0.493682    0.040782    0.013543    0.739580    
#> 460: 0.709786    1.079099    3.490083    0.493588    0.040813    0.013537    0.739592    
#> 461: 0.709731    1.079246    3.490046    0.493751    0.040825    0.013527    0.739643    
#> 462: 0.709587    1.079439    3.490009    0.493926    0.040840    0.013520    0.739688    
#> 463: 0.709490    1.079513    3.490020    0.493594    0.040859    0.013511    0.739723    
#> 464: 0.709444    1.079621    3.489999    0.493355    0.040866    0.013513    0.739688    
#> 465: 0.709397    1.079551    3.489978    0.493090    0.040848    0.013522    0.739629    
#> 466: 0.709412    1.079548    3.489949    0.493053    0.040825    0.013520    0.739615    
#> 467: 0.709198    1.079551    3.489951    0.493015    0.040779    0.013525    0.739535    
#> 468: 0.709364    1.079528    3.490008    0.493241    0.040762    0.013528    0.739534    
#> 469: 0.709523    1.079591    3.490061    0.493443    0.040735    0.013549    0.739519    
#> 470: 0.709546    1.079593    3.490044    0.493846    0.040684    0.013564    0.739504    
#> 471: 0.709497    1.079597    3.490021    0.493907    0.040666    0.013570    0.739509    
#> 472: 0.709464    1.079589    3.490012    0.493867    0.040623    0.013574    0.739602    
#> 473: 0.709384    1.079617    3.489996    0.493511    0.040580    0.013580    0.739561    
#> 474: 0.709314    1.079609    3.489939    0.493461    0.040598    0.013579    0.739537    
#> 475: 0.709314    1.079659    3.489917    0.493187    0.040585    0.013584    0.739513    
#> 476: 0.709275    1.079616    3.489912    0.493391    0.040582    0.013585    0.739570    
#> 477: 0.709026    1.079603    3.489926    0.493035    0.040566    0.013592    0.739528    
#> 478: 0.708804    1.079620    3.489899    0.492818    0.040551    0.013593    0.739536    
#> 479: 0.708891    1.079666    3.489883    0.492646    0.040547    0.013591    0.739585    
#> 480: 0.708898    1.079689    3.489893    0.492452    0.040528    0.013593    0.739668    
#> 481: 0.708778    1.079663    3.489853    0.492068    0.040524    0.013588    0.739728    
#> 482: 0.708831    1.079646    3.489823    0.491978    0.040522    0.013592    0.739792    
#> 483: 0.708733    1.079700    3.489766    0.492300    0.040490    0.013590    0.739754    
#> 484: 0.708672    1.079683    3.489726    0.492625    0.040492    0.013581    0.739782    
#> 485: 0.708654    1.079740    3.489686    0.493109    0.040536    0.013571    0.739743    
#> 486: 0.708511    1.079803    3.489671    0.492929    0.040550    0.013559    0.739728    
#> 487: 0.708745    1.079833    3.489740    0.492714    0.040547    0.013556    0.739661    
#> 488: 0.708768    1.079769    3.489784    0.492590    0.040518    0.013564    0.739580    
#> 489: 0.708893    1.079727    3.489827    0.492351    0.040499    0.013569    0.739553    
#> 490: 0.709103    1.079733    3.489881    0.492480    0.040463    0.013577    0.739586    
#> 491: 0.709288    1.079734    3.489976    0.492496    0.040433    0.013595    0.739591    
#> 492: 0.709389    1.079782    3.490006    0.492537    0.040404    0.013613    0.739624    
#> 493: 0.709249    1.079804    3.490026    0.492487    0.040379    0.013639    0.739636    
#> 494: 0.709375    1.079897    3.490031    0.492360    0.040364    0.013658    0.739617    
#> 495: 0.709483    1.079992    3.490045    0.492775    0.040359    0.013673    0.739533    
#> 496: 0.709701    1.080000    3.490049    0.492928    0.040354    0.013683    0.739521    
#> 497: 0.709702    1.079962    3.490032    0.493236    0.040365    0.013690    0.739506    
#> 498: 0.709569    1.079974    3.489965    0.493220    0.040364    0.013694    0.739490    
#> 499: 0.709489    1.080018    3.489929    0.493395    0.040386    0.013698    0.739473    
#> 500: 0.709503    1.079970    3.489891    0.493501    0.040397    0.013688    0.739466    
#> [====|====|=>-------------------------------------]  25%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.282968    0.967584    3.447598    0.392401    0.078940    0.016800    1.735794    
#> 002: 0.146577    1.034059    3.481324    0.372781    0.074993    0.015960    1.061638    
#> 003: 0.198727    0.996534    3.511705    0.354142    0.073357    0.015162    0.869937    
#> 004: 0.169441    1.023829    3.511878    0.336435    0.077976    0.014404    0.771889    
#> 005: 0.233362    1.016925    3.508551    0.319613    0.092166    0.013684    0.701567    
#> 006: 0.194173    1.035009    3.485726    0.303633    0.108688    0.013000    0.653678    
#> 007: 0.129868    1.054532    3.470416    0.288451    0.121413    0.013941    0.641198    
#> 008: 0.121952    1.026718    3.469655    0.274029    0.115343    0.018353    0.647772    
#> 009: 0.100961    1.016010    3.470102    0.260327    0.109575    0.019742    0.653353    
#> 010: 0.142443    1.024220    3.464956    0.247311    0.104097    0.018755    0.660624    
#> 011: 0.185532    1.043571    3.468077    0.234945    0.119489    0.017817    0.659553    
#> 012: 0.159234    1.038441    3.461236    0.223198    0.113515    0.016927    0.663108    
#> 013: 0.111926    1.049441    3.458518    0.212038    0.109393    0.016080    0.660765    
#> 014: 0.165432    1.055893    3.470587    0.201436    0.103923    0.015276    0.675331    
#> 015: 0.179068    1.063926    3.479818    0.191364    0.098727    0.018784    0.622664    
#> 016: 0.181449    1.048234    3.492259    0.181796    0.097195    0.020046    0.621303    
#> 017: 0.188637    1.055046    3.489991    0.172706    0.092335    0.020627    0.619433    
#> 018: 0.196306    1.038003    3.475752    0.175679    0.087718    0.021138    0.629530    
#> 019: 0.146717    1.041616    3.474663    0.169348    0.083332    0.021081    0.625892    
#> 020: 0.161033    1.046668    3.480993    0.178166    0.089012    0.021688    0.632660    
#> 021: 0.173099    1.029818    3.487320    0.169257    0.084562    0.023714    0.630907    
#> 022: 0.177407    1.039017    3.490548    0.167281    0.080334    0.022528    0.615018    
#> 023: 0.265441    1.023786    3.510750    0.167488    0.088656    0.023300    0.636379    
#> 024: 0.251849    1.030044    3.512884    0.159113    0.087647    0.022135    0.637640    
#> 025: 0.255064    1.006247    3.512135    0.151158    0.083264    0.023120    0.635696    
#> 026: 0.243030    1.021753    3.507557    0.143600    0.080869    0.021964    0.631489    
#> 027: 0.195022    1.042221    3.502995    0.136420    0.081383    0.020866    0.644513    
#> 028: 0.253941    1.029056    3.516951    0.129599    0.085442    0.019823    0.617880    
#> 029: 0.249356    1.032733    3.507836    0.135031    0.105617    0.020379    0.622700    
#> 030: 0.240047    1.039732    3.511659    0.128280    0.100336    0.022004    0.637594    
#> 031: 0.234397    0.996421    3.513048    0.127919    0.095319    0.021063    0.632442    
#> 032: 0.216042    0.997092    3.516809    0.126884    0.090553    0.022444    0.624549    
#> 033: 0.200922    1.034407    3.508692    0.133817    0.099265    0.021907    0.626237    
#> 034: 0.235790    1.042025    3.513534    0.127236    0.094302    0.020812    0.641630    
#> 035: 0.203177    1.019925    3.503387    0.130636    0.091345    0.019771    0.619482    
#> 036: 0.223364    1.037317    3.505751    0.124105    0.093254    0.020354    0.609634    
#> 037: 0.225452    1.035630    3.498401    0.125973    0.092855    0.019336    0.604820    
#> 038: 0.204760    1.040804    3.485802    0.119674    0.104633    0.018370    0.625452    
#> 039: 0.205515    1.051277    3.488435    0.158109    0.101684    0.017451    0.616454    
#> 040: 0.188374    1.041889    3.496362    0.150203    0.096600    0.016579    0.645668    
#> 041: 0.168287    1.048570    3.486987    0.142693    0.091770    0.017252    0.637566    
#> 042: 0.201881    1.041188    3.477901    0.135558    0.087181    0.017793    0.633370    
#> 043: 0.180643    1.053209    3.477400    0.128780    0.085334    0.017901    0.641541    
#> 044: 0.170670    1.063398    3.471323    0.123786    0.094766    0.017006    0.613764    
#> 045: 0.168681    1.085986    3.466788    0.130445    0.090028    0.016156    0.623756    
#> 046: 0.203905    1.057190    3.470405    0.136638    0.096021    0.018449    0.616203    
#> 047: 0.186533    1.051443    3.480409    0.141113    0.099271    0.020232    0.601288    
#> 048: 0.191912    1.045969    3.482103    0.153598    0.106947    0.019221    0.615367    
#> 049: 0.191347    1.062681    3.488255    0.145918    0.113698    0.018260    0.630842    
#> 050: 0.242177    1.005265    3.504932    0.138623    0.108013    0.017347    0.636894    
#> 051: 0.229218    1.019386    3.504918    0.149041    0.102613    0.019091    0.638430    
#> 052: 0.195704    1.017931    3.509146    0.141589    0.097482    0.019712    0.618604    
#> 053: 0.194070    1.025884    3.486503    0.182947    0.107040    0.018727    0.626917    
#> 054: 0.156928    1.029992    3.488461    0.173799    0.110514    0.020719    0.630549    
#> 055: 0.189608    1.019374    3.493760    0.165109    0.110444    0.020845    0.624106    
#> 056: 0.225042    1.004607    3.509631    0.156854    0.104921    0.023179    0.627862    
#> 057: 0.221484    1.016387    3.515044    0.159174    0.100503    0.022020    0.631002    
#> 058: 0.201798    1.024225    3.497680    0.151215    0.096769    0.020919    0.620166    
#> 059: 0.224680    1.009764    3.507573    0.143654    0.098020    0.020062    0.647542    
#> 060: 0.181726    0.993097    3.497836    0.136472    0.098836    0.019059    0.657032    
#> 061: 0.200365    1.027303    3.505198    0.140712    0.093895    0.018106    0.662793    
#> 062: 0.214472    1.033481    3.514738    0.165442    0.116528    0.017201    0.659800    
#> 063: 0.265542    1.001185    3.511632    0.157169    0.110701    0.016341    0.657608    
#> 064: 0.232473    1.026240    3.511126    0.149311    0.107158    0.015524    0.650355    
#> 065: 0.226434    1.016149    3.498994    0.141845    0.111215    0.014748    0.636135    
#> 066: 0.231609    1.045570    3.502037    0.141646    0.114366    0.014010    0.634563    
#> 067: 0.207348    1.023343    3.495428    0.151946    0.108647    0.015178    0.632209    
#> 068: 0.170316    1.013599    3.497550    0.146261    0.109785    0.014419    0.653642    
#> 069: 0.159775    1.042403    3.480725    0.138948    0.117676    0.013698    0.656291    
#> 070: 0.189410    1.036928    3.495292    0.132000    0.125712    0.014649    0.632173    
#> 071: 0.189621    1.022116    3.482236    0.125400    0.119426    0.013917    0.640374    
#> 072: 0.151036    1.006447    3.484556    0.135325    0.113455    0.013221    0.641755    
#> 073: 0.157020    1.034922    3.485160    0.134248    0.129048    0.012560    0.639604    
#> 074: 0.164886    1.015742    3.480114    0.144268    0.124467    0.011932    0.645554    
#> 075: 0.178892    1.003550    3.505223    0.137054    0.130519    0.011397    0.655889    
#> 076: 0.227998    1.018916    3.515766    0.130201    0.123993    0.010827    0.630388    
#> 077: 0.265940    1.011965    3.513686    0.136157    0.117793    0.015184    0.621120    
#> 078: 0.234911    1.018434    3.506097    0.135677    0.111903    0.014424    0.617299    
#> 079: 0.217804    1.022452    3.506781    0.128893    0.131578    0.013703    0.639600    
#> 080: 0.230999    1.016995    3.512272    0.125228    0.124999    0.014858    0.646246    
#> 081: 0.213178    1.033199    3.500227    0.119623    0.118750    0.017468    0.632106    
#> 082: 0.185079    1.012746    3.505805    0.113642    0.117529    0.016595    0.647047    
#> 083: 0.198388    1.036377    3.503439    0.130302    0.111652    0.015765    0.636208    
#> 084: 0.220130    1.034675    3.493113    0.125910    0.107798    0.019347    0.628186    
#> 085: 0.205265    1.041068    3.501897    0.135330    0.102408    0.019158    0.642861    
#> 086: 0.197716    1.029431    3.496627    0.135209    0.101092    0.018460    0.626778    
#> 087: 0.173084    1.031433    3.488122    0.137596    0.096037    0.021355    0.642038    
#> 088: 0.122635    1.048198    3.485710    0.144540    0.092732    0.020846    0.669054    
#> 089: 0.158702    1.065062    3.469593    0.147715    0.099773    0.021976    0.637683    
#> 090: 0.131752    1.054528    3.463322    0.152669    0.099434    0.020921    0.621334    
#> 091: 0.128798    1.040343    3.454407    0.145035    0.094462    0.020072    0.634739    
#> 092: 0.140773    1.051254    3.466200    0.137783    0.089739    0.019068    0.652028    
#> 093: 0.130852    1.040662    3.474382    0.130894    0.085252    0.021950    0.648036    
#> 094: 0.145272    1.064008    3.480142    0.124350    0.103700    0.020853    0.638392    
#> 095: 0.150490    1.033802    3.481402    0.118132    0.098515    0.019810    0.629817    
#> 096: 0.139385    1.036398    3.475468    0.124848    0.096297    0.018820    0.622462    
#> 097: 0.181106    1.056715    3.474928    0.122618    0.098037    0.017879    0.623985    
#> 098: 0.166116    1.080436    3.482855    0.130004    0.099730    0.017064    0.630506    
#> 099: 0.163088    1.039858    3.477167    0.123504    0.103095    0.016210    0.624058    
#> 100: 0.152298    1.035276    3.472537    0.120089    0.108981    0.015400    0.622814    
#> 101: 0.141098    1.038809    3.465508    0.122392    0.105623    0.014630    0.618606    
#> 102: 0.132179    1.061695    3.450385    0.116272    0.109053    0.013898    0.634924    
#> 103: 0.146614    1.054848    3.466138    0.122810    0.104613    0.013203    0.626448    
#> 104: 0.133224    1.047462    3.474500    0.132933    0.101300    0.012543    0.643407    
#> 105: 0.181301    1.017422    3.478939    0.175143    0.114392    0.012545    0.630019    
#> 106: 0.195824    1.036986    3.481844    0.166385    0.108672    0.014348    0.636980    
#> 107: 0.228037    1.040692    3.489824    0.158066    0.105786    0.015170    0.643215    
#> 108: 0.206608    1.015699    3.490036    0.150163    0.101523    0.016283    0.647075    
#> 109: 0.208318    1.033050    3.509537    0.142655    0.096447    0.017727    0.640709    
#> 110: 0.198123    1.010279    3.486119    0.135522    0.091625    0.018827    0.633673    
#> 111: 0.203040    1.025127    3.495668    0.128746    0.087043    0.018928    0.617740    
#> 112: 0.190555    1.003504    3.480667    0.122309    0.087415    0.018436    0.631007    
#> 113: 0.197294    1.047019    3.488052    0.128308    0.089105    0.018796    0.616628    
#> 114: 0.209764    1.034855    3.491972    0.125345    0.087516    0.017857    0.630698    
#> 115: 0.204086    1.018787    3.486916    0.122651    0.083786    0.021614    0.665014    
#> 116: 0.175887    1.050146    3.481725    0.116518    0.092855    0.027369    0.662549    
#> 117: 0.175507    1.067455    3.489412    0.112041    0.088212    0.026001    0.642848    
#> 118: 0.192964    1.035436    3.485737    0.141385    0.083802    0.024701    0.628963    
#> 119: 0.196511    1.032382    3.484956    0.135321    0.088313    0.023466    0.613290    
#> 120: 0.213128    1.016953    3.500748    0.128555    0.083897    0.022293    0.609644    
#> 121: 0.205632    1.062513    3.496924    0.122127    0.088751    0.021316    0.632101    
#> 122: 0.208891    1.028055    3.495107    0.150721    0.085603    0.020250    0.625310    
#> 123: 0.200718    1.038116    3.481291    0.153242    0.091741    0.019238    0.630234    
#> 124: 0.218609    1.045657    3.487058    0.145580    0.087154    0.018276    0.626599    
#> 125: 0.231805    1.052412    3.496132    0.144350    0.105838    0.017940    0.624982    
#> 126: 0.192837    1.045144    3.487616    0.153424    0.100546    0.021994    0.638332    
#> 127: 0.183757    1.026155    3.482536    0.158895    0.095519    0.020894    0.640812    
#> 128: 0.142577    1.052682    3.484544    0.150951    0.100963    0.019849    0.642258    
#> 129: 0.187692    1.050285    3.490144    0.143403    0.095914    0.018857    0.648966    
#> 130: 0.210860    1.054219    3.492505    0.136233    0.092491    0.017914    0.628275    
#> 131: 0.149459    1.051112    3.469147    0.129421    0.108929    0.017018    0.628403    
#> 132: 0.125074    1.062020    3.454621    0.122950    0.103483    0.016167    0.625830    
#> 133: 0.127999    1.064215    3.456454    0.138562    0.098308    0.015637    0.622862    
#> 134: 0.142381    1.040395    3.474982    0.131634    0.093393    0.017603    0.620102    
#> 135: 0.144253    1.049120    3.477646    0.126911    0.096535    0.016723    0.623213    
#> 136: 0.173458    1.034348    3.490434    0.120565    0.092223    0.017501    0.619873    
#> 137: 0.225629    1.023787    3.509260    0.114537    0.087612    0.022118    0.620697    
#> 138: 0.208198    1.033551    3.497678    0.108810    0.093900    0.021164    0.619987    
#> 139: 0.218647    1.035124    3.504554    0.103370    0.111692    0.021211    0.631618    
#> 140: 0.228568    1.035479    3.511012    0.110484    0.106107    0.020151    0.636943    
#> 141: 0.226601    1.016397    3.513161    0.104960    0.100802    0.019143    0.633031    
#> 142: 0.241411    1.016883    3.522872    0.099712    0.095762    0.019529    0.649385    
#> 143: 0.203746    1.049296    3.509209    0.097569    0.093522    0.021329    0.641060    
#> 144: 0.258717    1.019625    3.529740    0.128740    0.088846    0.020263    0.633082    
#> 145: 0.190179    1.029861    3.496564    0.133509    0.089902    0.019250    0.631914    
#> 146: 0.204106    1.043138    3.508309    0.126834    0.085407    0.018287    0.644111    
#> 147: 0.173922    1.021606    3.493547    0.153822    0.083672    0.022872    0.608519    
#> 148: 0.201666    1.040952    3.503849    0.157459    0.092009    0.021729    0.627528    
#> 149: 0.202050    1.013108    3.494416    0.160484    0.087409    0.020982    0.631540    
#> 150: 0.213977    1.017407    3.486500    0.152459    0.092097    0.021222    0.634389    
#> 151: 0.173560    1.052653    3.486679    0.144836    0.100650    0.020161    0.624586    
#> 152: 0.174929    1.035899    3.488868    0.133032    0.094900    0.013656    0.625884    
#> 153: 0.183939    1.034156    3.500238    0.126461    0.105701    0.014443    0.639612    
#> 154: 0.216178    1.035364    3.502240    0.119689    0.111564    0.014954    0.640803    
#> 155: 0.195267    1.006550    3.496678    0.094373    0.124165    0.012673    0.662718    
#> 156: 0.192843    1.022433    3.496908    0.105772    0.115009    0.014593    0.649606    
#> 157: 0.222103    1.025796    3.507251    0.107358    0.120624    0.012855    0.643346    
#> 158: 0.225590    1.030694    3.497536    0.097837    0.120337    0.013373    0.632149    
#> 159: 0.169101    1.048839    3.489569    0.103383    0.123134    0.014840    0.624017    
#> 160: 0.210095    0.999635    3.508710    0.103487    0.114175    0.016694    0.628685    
#> 161: 0.177598    1.033903    3.494949    0.093295    0.098558    0.016135    0.647185    
#> 162: 0.202031    1.044612    3.483658    0.086416    0.103780    0.016202    0.648734    
#> 163: 0.233279    1.025525    3.499428    0.095625    0.098166    0.017139    0.646862    
#> 164: 0.227146    1.009013    3.499068    0.113576    0.091227    0.014778    0.621547    
#> 165: 0.239855    1.014689    3.508783    0.106435    0.087276    0.022759    0.609632    
#> 166: 0.263942    1.042435    3.514234    0.128718    0.090155    0.021155    0.632449    
#> 167: 0.256202    1.046317    3.517105    0.134919    0.094834    0.018234    0.631829    
#> 168: 0.220454    1.060405    3.499014    0.115662    0.104490    0.012390    0.654919    
#> 169: 0.178550    1.072559    3.487614    0.099102    0.103211    0.013987    0.647727    
#> 170: 0.184895    1.049632    3.485404    0.102889    0.114530    0.012930    0.632885    
#> 171: 0.152214    1.033267    3.484486    0.098993    0.103172    0.012951    0.618412    
#> 172: 0.152158    1.073143    3.476295    0.120395    0.115319    0.012773    0.631768    
#> 173: 0.175380    1.059369    3.468445    0.133570    0.106044    0.012566    0.618724    
#> 174: 0.180881    1.037613    3.484233    0.131829    0.096558    0.015219    0.618012    
#> 175: 0.187498    1.040419    3.473062    0.131774    0.102781    0.015658    0.627047    
#> 176: 0.191388    1.059688    3.489829    0.124370    0.112449    0.015142    0.616658    
#> 177: 0.195000    1.043121    3.498279    0.102707    0.098019    0.012749    0.630372    
#> 178: 0.222221    1.042824    3.496438    0.115041    0.107768    0.014534    0.620042    
#> 179: 0.227771    1.063819    3.493125    0.101623    0.114731    0.017794    0.615644    
#> 180: 0.168352    1.049173    3.492368    0.136672    0.098622    0.016665    0.621615    
#> 181: 0.175204    1.055107    3.473443    0.130597    0.091734    0.018821    0.619849    
#> 182: 0.150155    1.066001    3.463822    0.118013    0.096005    0.023178    0.644712    
#> 183: 0.132179    1.077227    3.463273    0.119509    0.105894    0.023470    0.636601    
#> 184: 0.149747    1.032470    3.476339    0.109822    0.094628    0.019050    0.629359    
#> 185: 0.122110    1.067505    3.465257    0.120532    0.100049    0.018577    0.627585    
#> 186: 0.138933    1.070642    3.463731    0.112584    0.113810    0.022206    0.642660    
#> 187: 0.152407    1.041469    3.481450    0.109243    0.087043    0.019837    0.647983    
#> 188: 0.124592    1.061542    3.474737    0.106251    0.106521    0.016565    0.658472    
#> 189: 0.166473    1.071348    3.482834    0.105191    0.099703    0.015794    0.647408    
#> 190: 0.171663    1.042954    3.491193    0.110494    0.088210    0.019265    0.626678    
#> 191: 0.179353    1.058480    3.484554    0.137098    0.089973    0.019333    0.627902    
#> 192: 0.196179    1.058662    3.490929    0.138913    0.068055    0.021583    0.620379    
#> 193: 0.229081    1.047462    3.503162    0.123245    0.070061    0.021286    0.621502    
#> 194: 0.187951    1.042305    3.491632    0.128095    0.083681    0.018265    0.637840    
#> 195: 0.160257    1.036375    3.482792    0.113772    0.088780    0.020450    0.633349    
#> 196: 0.162495    1.038867    3.493488    0.115041    0.086384    0.023425    0.645983    
#> 197: 0.150093    1.045708    3.509071    0.136826    0.092985    0.026052    0.654848    
#> 198: 0.202601    1.018243    3.505090    0.134184    0.088351    0.024442    0.616836    
#> 199: 0.168250    1.030185    3.493207    0.138399    0.097420    0.021918    0.628016    
#> 200: 0.182316    1.044900    3.486779    0.132262    0.112850    0.020564    0.624626    
#> 201: 0.168450    1.048792    3.488655    0.124836    0.105833    0.019971    0.621124    
#> 202: 0.166666    1.050178    3.487654    0.127982    0.104546    0.019369    0.623366    
#> 203: 0.174002    1.048015    3.486163    0.128516    0.101637    0.019115    0.623946    
#> 204: 0.179844    1.041531    3.487933    0.128724    0.102010    0.019654    0.626930    
#> 205: 0.180279    1.045347    3.488006    0.127405    0.101478    0.020025    0.630130    
#> 206: 0.182129    1.047645    3.489032    0.127669    0.101745    0.019816    0.632937    
#> 207: 0.181131    1.043604    3.489181    0.124836    0.100549    0.019984    0.634352    
#> 208: 0.182859    1.043172    3.490175    0.124508    0.099378    0.019934    0.635116    
#> 209: 0.185345    1.040450    3.490746    0.122476    0.099779    0.019730    0.634640    
#> 210: 0.187620    1.037257    3.491740    0.120251    0.099764    0.019415    0.634190    
#> 211: 0.188397    1.037821    3.492412    0.118132    0.099911    0.019227    0.633029    
#> 212: 0.188315    1.037068    3.492835    0.118539    0.099322    0.018999    0.632209    
#> 213: 0.191298    1.035662    3.493752    0.119832    0.098600    0.018895    0.632187    
#> 214: 0.194706    1.034653    3.494623    0.118928    0.098169    0.018797    0.632234    
#> 215: 0.196083    1.034958    3.495489    0.118008    0.098075    0.018695    0.632641    
#> 216: 0.196440    1.033955    3.495760    0.118280    0.098615    0.018637    0.632822    
#> 217: 0.195376    1.034732    3.495486    0.118024    0.098737    0.018396    0.632576    
#> 218: 0.194718    1.036033    3.494865    0.118513    0.099173    0.018165    0.631864    
#> 219: 0.194128    1.035698    3.493928    0.117777    0.099127    0.017798    0.632370    
#> 220: 0.192527    1.036339    3.493691    0.117278    0.099106    0.017725    0.632432    
#> 221: 0.192151    1.036111    3.493470    0.117657    0.098752    0.017734    0.632041    
#> 222: 0.192498    1.036211    3.492974    0.117567    0.099085    0.017715    0.632507    
#> 223: 0.192712    1.036785    3.492427    0.116587    0.099356    0.017609    0.633418    
#> 224: 0.193818    1.036911    3.492685    0.116246    0.099674    0.017472    0.633406    
#> 225: 0.194100    1.035050    3.493098    0.116039    0.099358    0.017416    0.633904    
#> 226: 0.195096    1.033100    3.493239    0.115947    0.099851    0.017279    0.634708    
#> 227: 0.194276    1.032984    3.493499    0.115802    0.100023    0.017317    0.635442    
#> 228: 0.193494    1.033810    3.492875    0.116350    0.100094    0.017306    0.635709    
#> 229: 0.193503    1.033562    3.492841    0.116397    0.099843    0.017261    0.635047    
#> 230: 0.194228    1.032990    3.492949    0.116734    0.099227    0.017215    0.634433    
#> 231: 0.194701    1.032552    3.493476    0.116757    0.099034    0.017329    0.633763    
#> 232: 0.195127    1.032164    3.493892    0.116917    0.098440    0.017313    0.633002    
#> 233: 0.195206    1.031058    3.494269    0.116796    0.098133    0.017311    0.632641    
#> 234: 0.195357    1.031072    3.494360    0.116283    0.098068    0.017307    0.632343    
#> 235: 0.194822    1.031093    3.493931    0.115459    0.097611    0.017302    0.632311    
#> 236: 0.194276    1.031793    3.493500    0.115045    0.097667    0.017282    0.632806    
#> 237: 0.193982    1.032206    3.493032    0.114938    0.097799    0.017269    0.633570    
#> 238: 0.193991    1.032759    3.492704    0.114344    0.098245    0.017216    0.634671    
#> 239: 0.193448    1.033319    3.492275    0.113844    0.098630    0.017184    0.634984    
#> 240: 0.193467    1.033613    3.492061    0.113789    0.098631    0.017163    0.634747    
#> 241: 0.193500    1.032988    3.492260    0.113497    0.098877    0.017169    0.634809    
#> 242: 0.193548    1.032791    3.492238    0.112955    0.098805    0.017234    0.634996    
#> 243: 0.192796    1.033166    3.492085    0.112899    0.098694    0.017266    0.635101    
#> 244: 0.192432    1.033442    3.492049    0.113493    0.098581    0.017342    0.634963    
#> 245: 0.192987    1.032900    3.492311    0.113810    0.098744    0.017310    0.635185    
#> 246: 0.192785    1.032613    3.492252    0.113745    0.098601    0.017292    0.635320    
#> 247: 0.192591    1.032393    3.492435    0.113916    0.098766    0.017309    0.635251    
#> 248: 0.192595    1.032168    3.492387    0.114411    0.098586    0.017237    0.635083    
#> 249: 0.192628    1.032112    3.492644    0.114428    0.098582    0.017235    0.635009    
#> 250: 0.193091    1.032143    3.492992    0.114886    0.098738    0.017308    0.634803    
#> 251: 0.193836    1.032424    3.493334    0.115470    0.098725    0.017307    0.634570    
#> 252: 0.195158    1.032780    3.493804    0.115432    0.098846    0.017267    0.634510    
#> 253: 0.196204    1.032729    3.494262    0.115446    0.099010    0.017297    0.634272    
#> 254: 0.196878    1.032251    3.494673    0.115426    0.099150    0.017241    0.634044    
#> 255: 0.197380    1.032086    3.494939    0.115784    0.099160    0.017207    0.633985    
#> 256: 0.197821    1.031891    3.495012    0.115699    0.099086    0.017154    0.633938    
#> 257: 0.197927    1.031750    3.494879    0.115629    0.099319    0.017066    0.634048    
#> 258: 0.197539    1.031631    3.494769    0.115725    0.099229    0.017009    0.633962    
#> 259: 0.197289    1.031381    3.494758    0.116203    0.099192    0.016989    0.633773    
#> 260: 0.197020    1.031603    3.494742    0.116486    0.099086    0.016995    0.633436    
#> 261: 0.196943    1.031523    3.494719    0.117376    0.098990    0.016993    0.633180    
#> 262: 0.197386    1.031599    3.494814    0.117893    0.098871    0.017019    0.633161    
#> 263: 0.197117    1.031776    3.494846    0.118064    0.098845    0.017021    0.633257    
#> 264: 0.197173    1.031568    3.494999    0.117960    0.098791    0.017010    0.633585    
#> 265: 0.197476    1.031499    3.495069    0.118002    0.098863    0.017006    0.633558    
#> 266: 0.197628    1.031430    3.495155    0.117935    0.098888    0.016952    0.633503    
#> 267: 0.198025    1.031392    3.495437    0.117530    0.098946    0.016937    0.633590    
#> 268: 0.198427    1.031141    3.495460    0.117584    0.098974    0.016921    0.633643    
#> 269: 0.198970    1.031332    3.495455    0.117620    0.099236    0.016900    0.633904    
#> 270: 0.199274    1.031623    3.495391    0.117438    0.099270    0.016897    0.633952    
#> 271: 0.199188    1.031464    3.495310    0.117408    0.099145    0.016902    0.634225    
#> 272: 0.199152    1.031299    3.495325    0.117620    0.098946    0.016928    0.634539    
#> 273: 0.199138    1.031451    3.495414    0.117897    0.098902    0.016934    0.634581    
#> 274: 0.198859    1.031558    3.495376    0.117915    0.098822    0.016905    0.634708    
#> 275: 0.198754    1.031614    3.495270    0.117834    0.098849    0.016892    0.634762    
#> 276: 0.198405    1.032126    3.495109    0.117891    0.098860    0.016877    0.634646    
#> 277: 0.197995    1.032406    3.495121    0.117788    0.098780    0.016886    0.634498    
#> 278: 0.197884    1.032488    3.495143    0.117441    0.098899    0.016890    0.634815    
#> 279: 0.197941    1.032272    3.495257    0.117369    0.099052    0.016907    0.635095    
#> 280: 0.198216    1.032401    3.495316    0.117403    0.099101    0.016891    0.635164    
#> 281: 0.198207    1.032532    3.495447    0.117352    0.099009    0.016882    0.635066    
#> 282: 0.198177    1.032536    3.495529    0.117178    0.099014    0.016896    0.635119    
#> 283: 0.198377    1.032605    3.495722    0.116975    0.098868    0.016934    0.635280    
#> 284: 0.198616    1.032286    3.495829    0.116448    0.098585    0.016971    0.635206    
#> 285: 0.199007    1.032420    3.495709    0.115949    0.098400    0.016988    0.635327    
#> 286: 0.199254    1.032518    3.495750    0.115660    0.098472    0.017018    0.635116    
#> 287: 0.199316    1.032411    3.495771    0.115455    0.098368    0.017047    0.635316    
#> 288: 0.199722    1.032194    3.495942    0.115190    0.098539    0.017067    0.635374    
#> 289: 0.200183    1.031980    3.496117    0.115164    0.098581    0.017103    0.635361    
#> 290: 0.200306    1.031820    3.496171    0.115061    0.098555    0.017119    0.635337    
#> 291: 0.200570    1.031833    3.496381    0.114965    0.098360    0.017161    0.635346    
#> 292: 0.200719    1.031894    3.496455    0.115154    0.098199    0.017178    0.635167    
#> 293: 0.200516    1.031900    3.496438    0.115306    0.098121    0.017163    0.635064    
#> 294: 0.200156    1.032058    3.496262    0.115411    0.098139    0.017131    0.634864    
#> 295: 0.200125    1.031903    3.496217    0.115493    0.097964    0.017174    0.634753    
#> 296: 0.200005    1.032077    3.496139    0.115615    0.097724    0.017235    0.634681    
#> 297: 0.200184    1.032351    3.495994    0.115668    0.097535    0.017284    0.634608    
#> 298: 0.200288    1.032458    3.495920    0.115751    0.097419    0.017313    0.634537    
#> 299: 0.200439    1.032557    3.495919    0.115716    0.097304    0.017353    0.634377    
#> 300: 0.200748    1.032296    3.496062    0.115756    0.097287    0.017350    0.634346    
#> 301: 0.200886    1.032201    3.496028    0.115982    0.097270    0.017348    0.634376    
#> 302: 0.200765    1.032227    3.495937    0.116114    0.097405    0.017329    0.634271    
#> 303: 0.200548    1.032406    3.495841    0.116072    0.097393    0.017347    0.634167    
#> 304: 0.200681    1.032638    3.495766    0.116210    0.097362    0.017385    0.634070    
#> 305: 0.200463    1.032661    3.495637    0.116169    0.097399    0.017412    0.634072    
#> 306: 0.200109    1.032932    3.495388    0.116157    0.097355    0.017415    0.634086    
#> 307: 0.199837    1.033113    3.495213    0.116170    0.097375    0.017389    0.634284    
#> 308: 0.199678    1.033108    3.495171    0.116174    0.097511    0.017362    0.634459    
#> 309: 0.199906    1.032687    3.495176    0.116063    0.097463    0.017324    0.634707    
#> 310: 0.200125    1.032573    3.495185    0.115978    0.097648    0.017281    0.634796    
#> 311: 0.200149    1.032453    3.495239    0.115978    0.097669    0.017271    0.634785    
#> 312: 0.200167    1.032260    3.495205    0.116011    0.097689    0.017243    0.634635    
#> 313: 0.200121    1.032079    3.495121    0.116033    0.097893    0.017207    0.634754    
#> 314: 0.199790    1.032268    3.494942    0.115999    0.098045    0.017148    0.634666    
#> 315: 0.199327    1.032551    3.494792    0.116056    0.098201    0.017111    0.634688    
#> 316: 0.199445    1.032818    3.494722    0.115779    0.098228    0.017098    0.634803    
#> 317: 0.199513    1.033016    3.494734    0.115853    0.098256    0.017094    0.634774    
#> 318: 0.199726    1.033020    3.494777    0.115791    0.098083    0.017116    0.634796    
#> 319: 0.199871    1.032921    3.494855    0.115723    0.098045    0.017107    0.634846    
#> 320: 0.199660    1.032748    3.494832    0.115729    0.098009    0.017096    0.634993    
#> 321: 0.199559    1.032830    3.494777    0.115571    0.098201    0.017087    0.635161    
#> 322: 0.199304    1.032932    3.494608    0.115542    0.098392    0.017043    0.635304    
#> 323: 0.198890    1.032821    3.494557    0.115597    0.098433    0.017027    0.635368    
#> 324: 0.198898    1.032786    3.494539    0.115774    0.098590    0.016974    0.635384    
#> 325: 0.198893    1.032822    3.494570    0.115826    0.098692    0.016940    0.635385    
#> 326: 0.198743    1.032634    3.494518    0.115800    0.098855    0.016887    0.635519    
#> 327: 0.198709    1.032648    3.494568    0.115653    0.098993    0.016839    0.635675    
#> 328: 0.198907    1.032740    3.494556    0.115633    0.098929    0.016795    0.635615    
#> 329: 0.199040    1.032879    3.494540    0.115640    0.099001    0.016776    0.635597    
#> 330: 0.199079    1.033012    3.494573    0.115649    0.099011    0.016786    0.635604    
#> 331: 0.199125    1.033118    3.494612    0.115742    0.099009    0.016798    0.635467    
#> 332: 0.199190    1.033056    3.494767    0.115718    0.098855    0.016831    0.635406    
#> 333: 0.199213    1.033029    3.494769    0.115584    0.098786    0.016851    0.635311    
#> 334: 0.199374    1.032794    3.494765    0.115577    0.098656    0.016846    0.635338    
#> 335: 0.199433    1.032671    3.494769    0.115418    0.098536    0.016850    0.635357    
#> 336: 0.199090    1.032906    3.494724    0.115313    0.098543    0.016853    0.635462    
#> 337: 0.198988    1.033230    3.494590    0.115002    0.098544    0.016836    0.635584    
#> 338: 0.199106    1.033323    3.494676    0.115000    0.098449    0.016815    0.635678    
#> 339: 0.199053    1.033490    3.494665    0.115064    0.098390    0.016811    0.635722    
#> 340: 0.198883    1.033397    3.494647    0.115029    0.098313    0.016816    0.635721    
#> 341: 0.198893    1.033387    3.494629    0.115049    0.098385    0.016795    0.635764    
#> 342: 0.198724    1.033283    3.494558    0.114973    0.098446    0.016799    0.635685    
#> 343: 0.198383    1.033356    3.494500    0.115139    0.098486    0.016797    0.635509    
#> 344: 0.198429    1.033348    3.494522    0.115247    0.098425    0.016817    0.635445    
#> 345: 0.198294    1.033402    3.494500    0.115332    0.098366    0.016830    0.635373    
#> 346: 0.198115    1.033423    3.494342    0.115506    0.098283    0.016812    0.635256    
#> 347: 0.197944    1.033480    3.494221    0.115496    0.098314    0.016814    0.635289    
#> 348: 0.197832    1.033553    3.494138    0.115565    0.098414    0.016810    0.635224    
#> 349: 0.197839    1.033441    3.494114    0.115480    0.098499    0.016795    0.635265    
#> 350: 0.197507    1.033495    3.494004    0.115396    0.098564    0.016764    0.635276    
#> 351: 0.197417    1.033732    3.493881    0.115294    0.098572    0.016736    0.635366    
#> 352: 0.197211    1.034003    3.493810    0.115202    0.098705    0.016715    0.635498    
#> 353: 0.197211    1.034208    3.493690    0.115260    0.098701    0.016699    0.635470    
#> 354: 0.197344    1.034227    3.493576    0.115308    0.098702    0.016692    0.635532    
#> 355: 0.197750    1.034178    3.493620    0.115352    0.098613    0.016696    0.635464    
#> 356: 0.197905    1.034105    3.493689    0.115376    0.098600    0.016713    0.635427    
#> 357: 0.198238    1.034224    3.493758    0.115439    0.098515    0.016734    0.635325    
#> 358: 0.198277    1.034359    3.493779    0.115392    0.098442    0.016758    0.635235    
#> 359: 0.198258    1.034428    3.493757    0.115275    0.098397    0.016762    0.635222    
#> 360: 0.197895    1.034581    3.493557    0.115209    0.098431    0.016735    0.635085    
#> 361: 0.197675    1.034661    3.493428    0.115159    0.098496    0.016704    0.635024    
#> 362: 0.197497    1.034748    3.493271    0.115186    0.098570    0.016685    0.634972    
#> 363: 0.197356    1.034748    3.493234    0.115232    0.098569    0.016675    0.634948    
#> 364: 0.197196    1.034885    3.493141    0.115305    0.098706    0.016653    0.635045    
#> 365: 0.197024    1.034809    3.493116    0.115310    0.098826    0.016652    0.635074    
#> 366: 0.196940    1.034727    3.493063    0.115472    0.098885    0.016651    0.635081    
#> 367: 0.196877    1.034789    3.493050    0.115484    0.099004    0.016622    0.635133    
#> 368: 0.197018    1.034899    3.493030    0.115397    0.098996    0.016617    0.635044    
#> 369: 0.196836    1.034887    3.493014    0.115329    0.099009    0.016607    0.635044    
#> 370: 0.196567    1.034975    3.492980    0.115281    0.099035    0.016592    0.634967    
#> 371: 0.196334    1.035045    3.492849    0.115155    0.099011    0.016567    0.635082    
#> 372: 0.196199    1.035193    3.492800    0.115008    0.099052    0.016555    0.635091    
#> 373: 0.195996    1.035406    3.492734    0.115043    0.099089    0.016536    0.635048    
#> 374: 0.195904    1.035445    3.492602    0.115071    0.099062    0.016533    0.635013    
#> 375: 0.195868    1.035399    3.492618    0.115249    0.099184    0.016527    0.634966    
#> 376: 0.195836    1.035485    3.492638    0.115268    0.099195    0.016528    0.634820    
#> 377: 0.195788    1.035568    3.492615    0.115325    0.099104    0.016534    0.634783    
#> 378: 0.195607    1.035657    3.492545    0.115312    0.099061    0.016544    0.634729    
#> 379: 0.195414    1.035583    3.492537    0.115377    0.099068    0.016526    0.634707    
#> 380: 0.195163    1.035561    3.492513    0.115352    0.099026    0.016493    0.634720    
#> 381: 0.195039    1.035551    3.492458    0.115455    0.098989    0.016473    0.634722    
#> 382: 0.194933    1.035629    3.492403    0.115517    0.098993    0.016479    0.634641    
#> 383: 0.195059    1.035659    3.492413    0.115549    0.098928    0.016484    0.634603    
#> 384: 0.195025    1.035545    3.492432    0.115695    0.098882    0.016481    0.634585    
#> 385: 0.194954    1.035624    3.492392    0.115729    0.098914    0.016484    0.634568    
#> 386: 0.195010    1.035774    3.492419    0.115779    0.098924    0.016501    0.634570    
#> 387: 0.195081    1.035869    3.492450    0.115782    0.098941    0.016506    0.634563    
#> 388: 0.195035    1.035897    3.492481    0.115660    0.098905    0.016538    0.634587    
#> 389: 0.195013    1.035905    3.492447    0.115610    0.098804    0.016555    0.634653    
#> 390: 0.195066    1.035912    3.492479    0.115617    0.098639    0.016584    0.634612    
#> 391: 0.195080    1.035883    3.492507    0.115603    0.098532    0.016598    0.634503    
#> 392: 0.195135    1.035962    3.492554    0.115620    0.098431    0.016610    0.634491    
#> 393: 0.195085    1.035966    3.492505    0.115658    0.098319    0.016611    0.634430    
#> 394: 0.195004    1.036062    3.492446    0.115697    0.098295    0.016612    0.634366    
#> 395: 0.195048    1.036221    3.492419    0.115760    0.098269    0.016628    0.634316    
#> 396: 0.195203    1.036284    3.492439    0.115748    0.098207    0.016651    0.634242    
#> 397: 0.195260    1.036228    3.492512    0.115714    0.098154    0.016677    0.634217    
#> 398: 0.195400    1.036190    3.492572    0.115666    0.098068    0.016717    0.634217    
#> 399: 0.195518    1.036141    3.492638    0.115581    0.098023    0.016744    0.634164    
#> 400: 0.195588    1.036076    3.492726    0.115499    0.097958    0.016756    0.634148    
#> 401: 0.195625    1.036064    3.492748    0.115451    0.097930    0.016774    0.634150    
#> 402: 0.195645    1.036132    3.492756    0.115446    0.097876    0.016793    0.634072    
#> 403: 0.195701    1.036138    3.492739    0.115509    0.097895    0.016818    0.633953    
#> 404: 0.195716    1.036174    3.492636    0.115628    0.097894    0.016826    0.633875    
#> 405: 0.195727    1.036277    3.492623    0.115685    0.097880    0.016812    0.633840    
#> 406: 0.195670    1.036300    3.492611    0.115710    0.097876    0.016809    0.633902    
#> 407: 0.195661    1.036293    3.492649    0.115702    0.097913    0.016800    0.633907    
#> 408: 0.195655    1.036300    3.492658    0.115653    0.097917    0.016795    0.633901    
#> 409: 0.195562    1.036208    3.492612    0.115693    0.098002    0.016782    0.633866    
#> 410: 0.195538    1.036257    3.492535    0.115775    0.098079    0.016761    0.633862    
#> 411: 0.195597    1.036148    3.492473    0.115891    0.098079    0.016748    0.633908    
#> 412: 0.195619    1.036221    3.492419    0.115993    0.098035    0.016748    0.633916    
#> 413: 0.195630    1.036254    3.492355    0.116147    0.098017    0.016743    0.633961    
#> 414: 0.195484    1.036295    3.492346    0.116208    0.097975    0.016734    0.633966    
#> 415: 0.195534    1.036235    3.492328    0.116218    0.097926    0.016738    0.633969    
#> 416: 0.195660    1.036285    3.492351    0.116191    0.098004    0.016748    0.633916    
#> 417: 0.195625    1.036279    3.492352    0.116150    0.097999    0.016749    0.633855    
#> 418: 0.195472    1.036318    3.492390    0.116113    0.098032    0.016741    0.633841    
#> 419: 0.195527    1.036423    3.492424    0.116157    0.098101    0.016756    0.633845    
#> 420: 0.195613    1.036421    3.492429    0.116217    0.098100    0.016739    0.633822    
#> 421: 0.195622    1.036456    3.492467    0.116203    0.098165    0.016724    0.633848    
#> 422: 0.195814    1.036368    3.492515    0.116139    0.098237    0.016720    0.633910    
#> 423: 0.195910    1.036309    3.492533    0.116144    0.098218    0.016716    0.633972    
#> 424: 0.195761    1.036325    3.492473    0.116109    0.098273    0.016702    0.633933    
#> 425: 0.195864    1.036337    3.492444    0.116026    0.098279    0.016694    0.633972    
#> 426: 0.195857    1.036420    3.492404    0.115977    0.098322    0.016683    0.633948    
#> 427: 0.195680    1.036411    3.492398    0.115950    0.098288    0.016667    0.633923    
#> 428: 0.195457    1.036510    3.492374    0.116012    0.098283    0.016655    0.633945    
#> 429: 0.195396    1.036569    3.492366    0.116009    0.098314    0.016643    0.633951    
#> 430: 0.195316    1.036556    3.492325    0.116056    0.098325    0.016636    0.633912    
#> 431: 0.195176    1.036499    3.492265    0.116148    0.098329    0.016629    0.633826    
#> 432: 0.195010    1.036479    3.492274    0.116152    0.098317    0.016640    0.633734    
#> 433: 0.195028    1.036468    3.492346    0.116123    0.098267    0.016654    0.633730    
#> 434: 0.194986    1.036361    3.492408    0.115990    0.098172    0.016683    0.633755    
#> 435: 0.195125    1.036286    3.492452    0.115936    0.098102    0.016692    0.633776    
#> 436: 0.195142    1.036103    3.492545    0.115873    0.098069    0.016725    0.633802    
#> 437: 0.195121    1.035988    3.492559    0.115802    0.098041    0.016735    0.633824    
#> 438: 0.195198    1.035977    3.492601    0.115736    0.098023    0.016736    0.633837    
#> 439: 0.195216    1.036051    3.492643    0.115632    0.098018    0.016733    0.633851    
#> 440: 0.195126    1.036113    3.492691    0.115528    0.097994    0.016741    0.633869    
#> 441: 0.195108    1.036105    3.492725    0.115549    0.098006    0.016732    0.633863    
#> 442: 0.195113    1.036213    3.492756    0.115620    0.098070    0.016720    0.633746    
#> 443: 0.195098    1.036254    3.492750    0.115577    0.098043    0.016718    0.633721    
#> 444: 0.195075    1.036274    3.492696    0.115555    0.098081    0.016714    0.633745    
#> 445: 0.194942    1.036371    3.492641    0.115473    0.098139    0.016711    0.633806    
#> 446: 0.194790    1.036456    3.492556    0.115398    0.098208    0.016699    0.633768    
#> 447: 0.194870    1.036457    3.492555    0.115463    0.098177    0.016706    0.633723    
#> 448: 0.194953    1.036471    3.492590    0.115483    0.098190    0.016713    0.633676    
#> 449: 0.195066    1.036453    3.492709    0.115539    0.098239    0.016729    0.633663    
#> 450: 0.195009    1.036365    3.492735    0.115546    0.098185    0.016757    0.633595    
#> 451: 0.195062    1.036313    3.492739    0.115500    0.098190    0.016774    0.633631    
#> 452: 0.194933    1.036344    3.492721    0.115497    0.098184    0.016781    0.633609    
#> 453: 0.194873    1.036284    3.492690    0.115574    0.098120    0.016780    0.633551    
#> 454: 0.194760    1.036453    3.492638    0.115651    0.098185    0.016778    0.633514    
#> 455: 0.194700    1.036503    3.492602    0.115698    0.098096    0.016771    0.633523    
#> 456: 0.194599    1.036447    3.492578    0.115746    0.098019    0.016790    0.633467    
#> 457: 0.194536    1.036511    3.492593    0.115823    0.097933    0.016815    0.633402    
#> 458: 0.194474    1.036669    3.492573    0.115858    0.097881    0.016826    0.633319    
#> 459: 0.194343    1.036810    3.492523    0.115866    0.097767    0.016842    0.633286    
#> 460: 0.194208    1.036917    3.492437    0.115915    0.097693    0.016848    0.633248    
#> 461: 0.194052    1.036948    3.492407    0.115928    0.097659    0.016861    0.633232    
#> 462: 0.194042    1.037050    3.492417    0.115888    0.097619    0.016893    0.633191    
#> 463: 0.194194    1.037111    3.492462    0.115883    0.097587    0.016902    0.633178    
#> 464: 0.194262    1.037153    3.492484    0.115865    0.097598    0.016892    0.633097    
#> 465: 0.194358    1.037021    3.492534    0.115881    0.097567    0.016890    0.633060    
#> 466: 0.194463    1.037053    3.492532    0.115844    0.097562    0.016879    0.633015    
#> 467: 0.194406    1.036993    3.492531    0.115829    0.097583    0.016866    0.632958    
#> 468: 0.194559    1.037039    3.492574    0.115818    0.097564    0.016865    0.633010    
#> 469: 0.194588    1.037125    3.492608    0.115830    0.097594    0.016866    0.633022    
#> 470: 0.194547    1.037186    3.492582    0.115776    0.097631    0.016870    0.632976    
#> 471: 0.194414    1.037115    3.492555    0.115742    0.097583    0.016871    0.632961    
#> 472: 0.194382    1.037129    3.492562    0.115776    0.097643    0.016875    0.632928    
#> 473: 0.194301    1.037105    3.492561    0.115766    0.097662    0.016880    0.632902    
#> 474: 0.194351    1.037058    3.492568    0.115730    0.097712    0.016874    0.632945    
#> 475: 0.194389    1.037105    3.492578    0.115718    0.097728    0.016872    0.632946    
#> 476: 0.194479    1.037120    3.492588    0.115677    0.097789    0.016864    0.632998    
#> 477: 0.194517    1.037118    3.492624    0.115613    0.097845    0.016848    0.632977    
#> 478: 0.194507    1.037135    3.492634    0.115640    0.097853    0.016836    0.632993    
#> 479: 0.194621    1.037118    3.492684    0.115641    0.097854    0.016835    0.633012    
#> 480: 0.194766    1.037095    3.492719    0.115620    0.097868    0.016838    0.633101    
#> 481: 0.194739    1.037143    3.492722    0.115637    0.097895    0.016834    0.633096    
#> 482: 0.194786    1.037196    3.492701    0.115720    0.097922    0.016846    0.633048    
#> 483: 0.194762    1.037135    3.492709    0.115716    0.097917    0.016840    0.633042    
#> 484: 0.194758    1.037016    3.492730    0.115685    0.097884    0.016835    0.633089    
#> 485: 0.194829    1.037005    3.492739    0.115708    0.097871    0.016832    0.633063    
#> 486: 0.194900    1.036996    3.492752    0.115681    0.097833    0.016833    0.633002    
#> 487: 0.194963    1.036981    3.492794    0.115678    0.097821    0.016846    0.632981    
#> 488: 0.194916    1.036886    3.492804    0.115691    0.097814    0.016846    0.632951    
#> 489: 0.194914    1.036863    3.492771    0.115686    0.097867    0.016840    0.632970    
#> 490: 0.194992    1.036874    3.492802    0.115679    0.097910    0.016842    0.632963    
#> 491: 0.195038    1.036928    3.492842    0.115710    0.097969    0.016853    0.632917    
#> 492: 0.195043    1.036929    3.492835    0.115731    0.098010    0.016853    0.632947    
#> 493: 0.195040    1.036961    3.492818    0.115837    0.098021    0.016857    0.632948    
#> 494: 0.195026    1.037056    3.492764    0.115915    0.098017    0.016853    0.632897    
#> 495: 0.195090    1.037074    3.492741    0.115924    0.098004    0.016850    0.632867    
#> 496: 0.195118    1.037110    3.492703    0.115871    0.098025    0.016847    0.632898    
#> 497: 0.195004    1.037092    3.492681    0.115736    0.098057    0.016848    0.632904    
#> 498: 0.194968    1.037106    3.492616    0.115690    0.098109    0.016832    0.632848    
#> 499: 0.194894    1.037141    3.492549    0.115663    0.098126    0.016819    0.632822    
#> 500: 0.194855    1.037150    3.492521    0.115662    0.098133    0.016817    0.632784    
#> [====|====|=>-------------------------------------]  25%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.287940    0.992578    3.441305    0.392401    0.083389    0.016800    1.848002    
#> 002: 0.244705    1.023433    3.480910    0.372781    0.079219    0.017045    1.128066    
#> 003: 0.374338    0.999369    3.514607    0.354142    0.075258    0.016193    0.918613    
#> 004: 0.378141    1.040804    3.528265    0.336435    0.075637    0.015383    0.802871    
#> 005: 0.407722    1.068296    3.525572    0.319613    0.071855    0.014614    0.745317    
#> 006: 0.368125    1.080039    3.503487    0.386136    0.068263    0.013883    0.692154    
#> 007: 0.328531    1.099214    3.495924    0.369524    0.064849    0.013189    0.714198    
#> 008: 0.312661    1.098590    3.488458    0.392845    0.076201    0.012627    0.708956    
#> 009: 0.265736    1.090876    3.477131    0.401736    0.072391    0.012589    0.702473    
#> 010: 0.279692    1.080449    3.479868    0.391132    0.068771    0.011960    0.720107    
#> 011: 0.364755    1.100307    3.487202    0.439040    0.069968    0.011842    0.702796    
#> 012: 0.334820    1.101306    3.479115    0.417088    0.066469    0.011346    0.701835    
#> 013: 0.276635    1.113859    3.474158    0.404181    0.063146    0.011574    0.711803    
#> 014: 0.312613    1.099737    3.490274    0.383972    0.070772    0.012419    0.729238    
#> 015: 0.329320    1.100203    3.500244    0.442879    0.067234    0.013849    0.702401    
#> 016: 0.328107    1.112621    3.509824    0.420735    0.063872    0.016021    0.704339    
#> 017: 0.342018    1.108695    3.510513    0.452724    0.060679    0.015395    0.690916    
#> 018: 0.349411    1.115168    3.491328    0.525477    0.057645    0.014766    0.695316    
#> 019: 0.305735    1.100233    3.488634    0.499203    0.056346    0.015650    0.692362    
#> 020: 0.323381    1.102889    3.493199    0.474243    0.070779    0.014867    0.689054    
#> 021: 0.328902    1.086674    3.497641    0.454020    0.067240    0.014124    0.694459    
#> 022: 0.320892    1.097566    3.499421    0.431319    0.063878    0.015041    0.670761    
#> 023: 0.401228    1.103037    3.520213    0.449628    0.060684    0.018227    0.696300    
#> 024: 0.367452    1.115289    3.509425    0.427147    0.057650    0.019312    0.695049    
#> 025: 0.334657    1.100813    3.494292    0.405789    0.054767    0.018346    0.690945    
#> 026: 0.312366    1.138952    3.479954    0.385500    0.052029    0.018179    0.700320    
#> 027: 0.268460    1.143676    3.477070    0.388067    0.049428    0.019655    0.717971    
#> 028: 0.326133    1.131656    3.493721    0.406167    0.051809    0.020892    0.706348    
#> 029: 0.372534    1.138495    3.495302    0.439781    0.063152    0.020258    0.713427    
#> 030: 0.334279    1.132641    3.492034    0.417792    0.059995    0.020116    0.699911    
#> 031: 0.314287    1.106939    3.504728    0.425983    0.056995    0.019110    0.718592    
#> 032: 0.346533    1.107312    3.507230    0.443743    0.056031    0.020705    0.706327    
#> 033: 0.328546    1.112215    3.497431    0.421556    0.055226    0.020123    0.702427    
#> 034: 0.368622    1.113587    3.495945    0.400478    0.058179    0.019117    0.709554    
#> 035: 0.339115    1.089942    3.490815    0.380454    0.062390    0.018161    0.687461    
#> 036: 0.356213    1.098652    3.500088    0.449491    0.059271    0.017308    0.684694    
#> 037: 0.332021    1.125397    3.495783    0.452662    0.059173    0.016443    0.706329    
#> 038: 0.346646    1.110271    3.494426    0.430029    0.056214    0.015621    0.706960    
#> 039: 0.350750    1.124496    3.512096    0.408528    0.067772    0.014840    0.676135    
#> 040: 0.334196    1.132828    3.506496    0.388101    0.064383    0.014635    0.704099    
#> 041: 0.355401    1.098454    3.510845    0.369146    0.061164    0.016101    0.693253    
#> 042: 0.419230    1.087944    3.516538    0.350688    0.058106    0.015296    0.698416    
#> 043: 0.356994    1.101248    3.503999    0.380842    0.055200    0.016599    0.690734    
#> 044: 0.346491    1.115139    3.496921    0.411025    0.058292    0.015769    0.700331    
#> 045: 0.342259    1.135748    3.484784    0.401579    0.057142    0.014981    0.693544    
#> 046: 0.335081    1.117376    3.486848    0.399598    0.057351    0.014232    0.686519    
#> 047: 0.326931    1.126886    3.478406    0.480538    0.054484    0.014904    0.682162    
#> 048: 0.341848    1.112590    3.482654    0.493581    0.051760    0.014159    0.698115    
#> 049: 0.361428    1.095864    3.492130    0.504532    0.062001    0.013451    0.690239    
#> 050: 0.349140    1.089280    3.497657    0.479305    0.065105    0.012779    0.705018    
#> 051: 0.335127    1.095628    3.492423    0.455340    0.061850    0.013035    0.726837    
#> 052: 0.363421    1.121642    3.504228    0.432573    0.058757    0.013909    0.706891    
#> 053: 0.373714    1.096501    3.505816    0.410944    0.055819    0.013213    0.703235    
#> 054: 0.343711    1.100572    3.505775    0.412349    0.053028    0.016972    0.715728    
#> 055: 0.339762    1.104273    3.506473    0.444427    0.056960    0.016124    0.711245    
#> 056: 0.338570    1.081925    3.507796    0.422206    0.055781    0.015495    0.717315    
#> 057: 0.337974    1.099414    3.506485    0.401096    0.067333    0.015075    0.710540    
#> 058: 0.315207    1.088520    3.491274    0.381041    0.065157    0.014321    0.718636    
#> 059: 0.302781    1.069541    3.498924    0.361989    0.061899    0.016142    0.717626    
#> 060: 0.336849    1.071272    3.512044    0.343889    0.058804    0.020265    0.736862    
#> 061: 0.385109    1.081539    3.517018    0.385031    0.055864    0.022295    0.753341    
#> 062: 0.351758    1.111226    3.514010    0.365779    0.053071    0.022284    0.728364    
#> 063: 0.402695    1.078822    3.516996    0.361002    0.052186    0.021170    0.714308    
#> 064: 0.368119    1.091425    3.528165    0.362110    0.049576    0.020112    0.724047    
#> 065: 0.390284    1.078411    3.505081    0.420716    0.047098    0.022848    0.717382    
#> 066: 0.362912    1.083346    3.506419    0.413215    0.046593    0.021705    0.683023    
#> 067: 0.365827    1.094354    3.515646    0.392554    0.047522    0.022488    0.688343    
#> 068: 0.341305    1.096093    3.513342    0.372926    0.047628    0.021364    0.692169    
#> 069: 0.323192    1.118714    3.493596    0.390439    0.053642    0.020296    0.682496    
#> 070: 0.337827    1.102380    3.500437    0.400055    0.050960    0.019908    0.692454    
#> 071: 0.284434    1.146301    3.460595    0.381143    0.048412    0.020229    0.720791    
#> 072: 0.270081    1.106703    3.481430    0.372769    0.045991    0.020539    0.706835    
#> 073: 0.310970    1.118890    3.491361    0.418334    0.054273    0.019512    0.705748    
#> 074: 0.350647    1.107127    3.492058    0.414156    0.059284    0.019056    0.708531    
#> 075: 0.349269    1.107142    3.515258    0.432096    0.056320    0.018103    0.705572    
#> 076: 0.389814    1.121915    3.520365    0.522000    0.053504    0.020368    0.702044    
#> 077: 0.399492    1.093359    3.511743    0.534631    0.057597    0.019350    0.689077    
#> 078: 0.365246    1.092260    3.510455    0.507899    0.054717    0.018382    0.697714    
#> 079: 0.376198    1.108447    3.513510    0.482505    0.051982    0.017722    0.709043    
#> 080: 0.360344    1.101979    3.497880    0.458379    0.049383    0.021560    0.710055    
#> 081: 0.345242    1.099752    3.504224    0.435460    0.046913    0.020482    0.700877    
#> 082: 0.310546    1.092944    3.497525    0.436855    0.048985    0.019458    0.715062    
#> 083: 0.339910    1.120419    3.492637    0.555740    0.055347    0.018485    0.716311    
#> 084: 0.370360    1.116427    3.500174    0.527953    0.065119    0.017561    0.697118    
#> 085: 0.362833    1.125099    3.503334    0.501555    0.061863    0.016683    0.707254    
#> 086: 0.394089    1.109607    3.503530    0.476477    0.058770    0.017224    0.706601    
#> 087: 0.352478    1.113222    3.482427    0.455152    0.055832    0.016363    0.685868    
#> 088: 0.282553    1.114684    3.485383    0.432395    0.053040    0.015553    0.702846    
#> 089: 0.332544    1.124104    3.485033    0.410775    0.062193    0.017084    0.694135    
#> 090: 0.323729    1.129246    3.487201    0.391332    0.069429    0.016230    0.695956    
#> 091: 0.282806    1.093985    3.480933    0.409754    0.065958    0.015418    0.702660    
#> 092: 0.270627    1.125153    3.461908    0.389266    0.062660    0.016376    0.714362    
#> 093: 0.226937    1.116893    3.460774    0.418177    0.063121    0.015823    0.707417    
#> 094: 0.284563    1.141764    3.463752    0.516337    0.060405    0.015154    0.702318    
#> 095: 0.270824    1.162755    3.482985    0.490520    0.057385    0.018846    0.688597    
#> 096: 0.261090    1.134819    3.477576    0.465994    0.054516    0.017903    0.698682    
#> 097: 0.362729    1.136841    3.486559    0.497279    0.053065    0.017008    0.677992    
#> 098: 0.346404    1.140244    3.496961    0.497892    0.050412    0.016158    0.694930    
#> 099: 0.336754    1.117133    3.492496    0.472997    0.048643    0.015350    0.711578    
#> 100: 0.330659    1.115435    3.495522    0.449347    0.053183    0.014582    0.713070    
#> 101: 0.287002    1.092172    3.472124    0.426880    0.054565    0.013853    0.712464    
#> 102: 0.307068    1.088719    3.475037    0.420685    0.055230    0.013161    0.694183    
#> 103: 0.351319    1.114211    3.479538    0.475642    0.058543    0.012503    0.683329    
#> 104: 0.334771    1.119540    3.492941    0.553179    0.060986    0.011877    0.706048    
#> 105: 0.319882    1.114113    3.482475    0.525520    0.057937    0.011284    0.709601    
#> 106: 0.335256    1.116506    3.488177    0.499244    0.061321    0.011930    0.688445    
#> 107: 0.349441    1.106161    3.491177    0.474282    0.063693    0.013046    0.699485    
#> 108: 0.355012    1.102895    3.486878    0.450568    0.060508    0.014553    0.718565    
#> 109: 0.344785    1.117072    3.501827    0.428040    0.070287    0.016746    0.704599    
#> 110: 0.335929    1.107942    3.480679    0.459376    0.066773    0.015908    0.691525    
#> 111: 0.345870    1.124706    3.496098    0.436407    0.063434    0.015113    0.681838    
#> 112: 0.356226    1.081568    3.498097    0.414586    0.060262    0.015894    0.692129    
#> 113: 0.380590    1.113201    3.501628    0.488344    0.057249    0.017202    0.674763    
#> 114: 0.383994    1.105315    3.505103    0.493278    0.057104    0.017155    0.691603    
#> 115: 0.373532    1.084724    3.498842    0.492033    0.058557    0.019029    0.700097    
#> 116: 0.356863    1.134106    3.486685    0.498864    0.055629    0.019247    0.698631    
#> 117: 0.325666    1.127322    3.490943    0.474347    0.052985    0.018284    0.693340    
#> 118: 0.356190    1.112466    3.490283    0.450632    0.064980    0.017370    0.682118    
#> 119: 0.330453    1.123627    3.488502    0.428101    0.066561    0.016502    0.701143    
#> 120: 0.345428    1.109427    3.496140    0.478195    0.063233    0.015735    0.694273    
#> 121: 0.361460    1.131516    3.493272    0.459352    0.060071    0.017279    0.700147    
#> 122: 0.369493    1.106917    3.503219    0.436384    0.057068    0.016415    0.716780    
#> 123: 0.368188    1.134296    3.481035    0.494721    0.054214    0.015594    0.705820    
#> 124: 0.318262    1.160831    3.473673    0.469985    0.051504    0.014815    0.718219    
#> 125: 0.344859    1.146430    3.476327    0.446486    0.048929    0.014074    0.696380    
#> 126: 0.314558    1.147048    3.477698    0.424161    0.058386    0.013370    0.696287    
#> 127: 0.337603    1.133808    3.473329    0.402953    0.055466    0.012702    0.718135    
#> 128: 0.329820    1.122375    3.487246    0.384115    0.063942    0.012067    0.704957    
#> 129: 0.381106    1.124560    3.499949    0.376673    0.060745    0.011873    0.712887    
#> 130: 0.382707    1.129911    3.497177    0.357916    0.060463    0.013278    0.703551    
#> 131: 0.352511    1.104124    3.493262    0.340020    0.064096    0.013670    0.708117    
#> 132: 0.292181    1.130573    3.474424    0.351161    0.070527    0.015200    0.716135    
#> 133: 0.293565    1.115827    3.479810    0.363030    0.067001    0.015814    0.718466    
#> 134: 0.332296    1.098923    3.500017    0.388595    0.063651    0.023120    0.701159    
#> 135: 0.340934    1.082839    3.513807    0.369165    0.060468    0.021964    0.723144    
#> 136: 0.351082    1.065811    3.515328    0.352698    0.057445    0.022193    0.718565    
#> 137: 0.391397    1.070026    3.518995    0.408125    0.057880    0.023951    0.708400    
#> 138: 0.339697    1.099851    3.502461    0.387718    0.063979    0.022754    0.692806    
#> 139: 0.339822    1.109751    3.508183    0.442093    0.060780    0.021616    0.685019    
#> 140: 0.373071    1.110300    3.504554    0.422315    0.059332    0.020535    0.706150    
#> 141: 0.349829    1.090930    3.506770    0.472442    0.056366    0.019843    0.699178    
#> 142: 0.399307    1.066214    3.530400    0.448820    0.053547    0.018851    0.718485    
#> 143: 0.325854    1.086378    3.517045    0.426379    0.067396    0.019381    0.702686    
#> 144: 0.387979    1.094603    3.520032    0.496459    0.064026    0.018412    0.679274    
#> 145: 0.365578    1.103517    3.502329    0.471636    0.060825    0.017491    0.684280    
#> 146: 0.301304    1.125936    3.501546    0.448054    0.057784    0.018451    0.706715    
#> 147: 0.307204    1.110696    3.484668    0.425651    0.054895    0.019370    0.678541    
#> 148: 0.311119    1.120851    3.496255    0.439130    0.058171    0.020718    0.688197    
#> 149: 0.338039    1.104891    3.497486    0.500914    0.055262    0.020066    0.704610    
#> 150: 0.392191    1.091091    3.494544    0.498932    0.052499    0.022064    0.708090    
#> 151: 0.312262    1.124355    3.480676    0.512026    0.049874    0.020961    0.686465    
#> 152: 0.293271    1.126742    3.485492    0.415763    0.043642    0.020607    0.707399    
#> 153: 0.322648    1.125057    3.497664    0.405769    0.048402    0.023060    0.706844    
#> 154: 0.382805    1.109674    3.503341    0.399399    0.043167    0.024565    0.711540    
#> 155: 0.360156    1.085429    3.499159    0.376356    0.045386    0.019206    0.701232    
#> 156: 0.344294    1.100402    3.506878    0.374467    0.042890    0.017550    0.699771    
#> 157: 0.357428    1.113915    3.503120    0.442006    0.048029    0.020277    0.703927    
#> 158: 0.344089    1.136938    3.488241    0.431996    0.045855    0.018496    0.708091    
#> 159: 0.324935    1.134147    3.499153    0.398954    0.050915    0.020260    0.704427    
#> 160: 0.346572    1.095582    3.501275    0.422327    0.046404    0.019709    0.706945    
#> 161: 0.344828    1.100411    3.506825    0.400154    0.039786    0.023496    0.709498    
#> 162: 0.345101    1.124204    3.506468    0.422358    0.046340    0.020604    0.691587    
#> 163: 0.388514    1.111904    3.497344    0.432583    0.045314    0.023243    0.685898    
#> 164: 0.365906    1.106522    3.493153    0.439141    0.038831    0.020087    0.705471    
#> 165: 0.385590    1.093723    3.511719    0.424039    0.046418    0.023018    0.702403    
#> 166: 0.364286    1.124480    3.508689    0.405675    0.048410    0.021447    0.685483    
#> 167: 0.388999    1.128681    3.517311    0.369889    0.054531    0.019154    0.712417    
#> 168: 0.380437    1.141132    3.503350    0.405186    0.052826    0.019443    0.709264    
#> 169: 0.330484    1.162821    3.482037    0.366638    0.047506    0.026846    0.728347    
#> 170: 0.328749    1.122865    3.482035    0.337898    0.044923    0.022947    0.712188    
#> 171: 0.283739    1.131126    3.476543    0.371900    0.051696    0.019130    0.697426    
#> 172: 0.276516    1.127649    3.482346    0.410053    0.052862    0.019163    0.693084    
#> 173: 0.276905    1.159768    3.457117    0.404096    0.047284    0.016591    0.693485    
#> 174: 0.315150    1.131719    3.472208    0.443122    0.053997    0.013973    0.683523    
#> 175: 0.320762    1.129950    3.474127    0.399743    0.053791    0.015252    0.711780    
#> 176: 0.310535    1.142175    3.485371    0.430898    0.054868    0.015012    0.682584    
#> 177: 0.366664    1.118090    3.510635    0.422959    0.048253    0.014932    0.703985    
#> 178: 0.371449    1.099100    3.506094    0.417557    0.054264    0.018180    0.692113    
#> 179: 0.371604    1.103094    3.505681    0.360429    0.047392    0.018364    0.694391    
#> 180: 0.331820    1.107510    3.504625    0.348952    0.053852    0.017415    0.689318    
#> 181: 0.310038    1.095805    3.489185    0.330219    0.051034    0.014587    0.708442    
#> 182: 0.269452    1.114870    3.467708    0.364484    0.065369    0.011040    0.722760    
#> 183: 0.307725    1.139298    3.473525    0.399445    0.078513    0.011130    0.683939    
#> 184: 0.306376    1.119505    3.469792    0.477491    0.065170    0.013381    0.682819    
#> 185: 0.328066    1.130259    3.482756    0.461573    0.057268    0.013079    0.703049    
#> 186: 0.286972    1.149795    3.482608    0.451144    0.068479    0.011408    0.709390    
#> 187: 0.337623    1.113835    3.490389    0.467636    0.053282    0.012856    0.718031    
#> 188: 0.307648    1.133479    3.491177    0.472550    0.061999    0.014361    0.712253    
#> 189: 0.315601    1.147631    3.491883    0.439845    0.068329    0.012833    0.714676    
#> 190: 0.271860    1.144698    3.484511    0.428779    0.050332    0.011502    0.705568    
#> 191: 0.277305    1.141692    3.464301    0.533453    0.051793    0.010156    0.700463    
#> 192: 0.270268    1.156230    3.462863    0.532999    0.062532    0.012749    0.704492    
#> 193: 0.320935    1.147424    3.471677    0.508804    0.062910    0.013459    0.691496    
#> 194: 0.305470    1.135289    3.474652    0.425333    0.060113    0.012996    0.696386    
#> 195: 0.291492    1.137173    3.470249    0.448020    0.052883    0.013046    0.703779    
#> 196: 0.277438    1.138632    3.470932    0.481995    0.054176    0.012307    0.706292    
#> 197: 0.248030    1.146762    3.484536    0.435636    0.066089    0.015686    0.717674    
#> 198: 0.307464    1.129375    3.479529    0.409551    0.054788    0.017041    0.680080    
#> 199: 0.282240    1.104807    3.478640    0.387745    0.062827    0.015893    0.699427    
#> 200: 0.303806    1.110328    3.478878    0.429033    0.055689    0.014669    0.724378    
#> 201: 0.308705    1.113189    3.485317    0.413964    0.050993    0.015830    0.707964    
#> 202: 0.305580    1.114536    3.486865    0.416662    0.050995    0.015965    0.709419    
#> 203: 0.314968    1.114226    3.483837    0.429121    0.051166    0.016424    0.705965    
#> 204: 0.317662    1.109702    3.486126    0.421399    0.053932    0.016530    0.705104    
#> 205: 0.320506    1.112631    3.485998    0.431404    0.055106    0.016601    0.706544    
#> 206: 0.319815    1.117063    3.486442    0.439387    0.053795    0.016126    0.707574    
#> 207: 0.319664    1.118384    3.485418    0.436075    0.054128    0.016050    0.706779    
#> 208: 0.321551    1.121096    3.484692    0.438014    0.054053    0.016121    0.705999    
#> 209: 0.327995    1.118874    3.485133    0.441814    0.055041    0.016145    0.706991    
#> 210: 0.329077    1.117305    3.485987    0.438028    0.054923    0.016125    0.707114    
#> 211: 0.333093    1.117269    3.487979    0.433871    0.054627    0.015905    0.706924    
#> 212: 0.337639    1.115571    3.490084    0.432200    0.054664    0.016058    0.707017    
#> 213: 0.343436    1.111364    3.492103    0.432101    0.055631    0.016230    0.707082    
#> 214: 0.346900    1.109767    3.493412    0.432769    0.056060    0.016224    0.708318    
#> 215: 0.351514    1.109074    3.494769    0.433848    0.056589    0.016306    0.708153    
#> 216: 0.355920    1.108709    3.495022    0.435950    0.056610    0.016209    0.709490    
#> 217: 0.355257    1.109170    3.494900    0.435198    0.056907    0.016044    0.708268    
#> 218: 0.354637    1.110102    3.494808    0.433565    0.057038    0.016084    0.707667    
#> 219: 0.352470    1.110213    3.493377    0.436282    0.056461    0.016264    0.707315    
#> 220: 0.348829    1.110902    3.492669    0.435690    0.056086    0.016625    0.706910    
#> 221: 0.347083    1.110654    3.492755    0.435600    0.055588    0.016563    0.706089    
#> 222: 0.346108    1.110553    3.492194    0.436287    0.055190    0.016656    0.706077    
#> 223: 0.345903    1.110847    3.491943    0.436189    0.054927    0.016694    0.705667    
#> 224: 0.346792    1.111797    3.492049    0.435571    0.055042    0.016525    0.705222    
#> 225: 0.346943    1.110260    3.492736    0.436406    0.055103    0.016398    0.705387    
#> 226: 0.346925    1.108950    3.492828    0.434746    0.055342    0.016354    0.705400    
#> 227: 0.346828    1.108756    3.493575    0.434491    0.055601    0.016407    0.704846    
#> 228: 0.345021    1.109270    3.493017    0.432812    0.055485    0.016641    0.704754    
#> 229: 0.344595    1.109152    3.492874    0.433935    0.055125    0.016662    0.704371    
#> 230: 0.344337    1.108741    3.492766    0.434572    0.054888    0.016591    0.704488    
#> 231: 0.344797    1.109188    3.492830    0.436333    0.054944    0.016592    0.703979    
#> 232: 0.344099    1.109183    3.493025    0.436379    0.054768    0.016611    0.703613    
#> 233: 0.342950    1.108441    3.493310    0.436157    0.054762    0.016681    0.703650    
#> 234: 0.341923    1.108361    3.493312    0.435154    0.054871    0.016657    0.703933    
#> 235: 0.341669    1.107756    3.493259    0.435281    0.054915    0.016572    0.703590    
#> 236: 0.341638    1.107952    3.493332    0.436036    0.054895    0.016544    0.703070    
#> 237: 0.342295    1.107647    3.493385    0.438560    0.054791    0.016522    0.702834    
#> 238: 0.343240    1.107509    3.493650    0.439163    0.054660    0.016511    0.702886    
#> 239: 0.343058    1.107714    3.493593    0.439439    0.054521    0.016442    0.703174    
#> 240: 0.342891    1.107576    3.493773    0.438740    0.054505    0.016307    0.703029    
#> 241: 0.343478    1.106975    3.494208    0.438472    0.054647    0.016255    0.702804    
#> 242: 0.343219    1.106726    3.494209    0.438315    0.054848    0.016195    0.702819    
#> 243: 0.343212    1.106584    3.494328    0.436714    0.054938    0.016190    0.703048    
#> 244: 0.342453    1.106292    3.494265    0.437517    0.054731    0.016190    0.703010    
#> 245: 0.342342    1.105778    3.494330    0.437288    0.054772    0.016199    0.703300    
#> 246: 0.342296    1.105488    3.494351    0.436006    0.054787    0.016163    0.703954    
#> 247: 0.342118    1.105312    3.494540    0.435308    0.054655    0.016101    0.703786    
#> 248: 0.341678    1.105660    3.494557    0.434415    0.054531    0.016013    0.703716    
#> 249: 0.340761    1.106009    3.494416    0.432631    0.054841    0.015922    0.704553    
#> 250: 0.341205    1.106454    3.494305    0.432869    0.055004    0.015871    0.704780    
#> 251: 0.341790    1.107011    3.494354    0.433952    0.054927    0.015836    0.705072    
#> 252: 0.342454    1.107191    3.494850    0.434055    0.054904    0.015737    0.705505    
#> 253: 0.342799    1.107364    3.495027    0.433893    0.054944    0.015723    0.705856    
#> 254: 0.343178    1.107875    3.495281    0.432678    0.054945    0.015717    0.706076    
#> 255: 0.343430    1.108230    3.495536    0.432312    0.054872    0.015725    0.706085    
#> 256: 0.343768    1.107978    3.495679    0.432161    0.054728    0.015749    0.706370    
#> 257: 0.344165    1.107572    3.495816    0.431942    0.054687    0.015728    0.706546    
#> 258: 0.343618    1.107286    3.495875    0.431317    0.054618    0.015701    0.706530    
#> 259: 0.342907    1.107388    3.495767    0.431331    0.054639    0.015735    0.706528    
#> 260: 0.342707    1.107777    3.495634    0.431503    0.054598    0.015712    0.706248    
#> 261: 0.342813    1.107815    3.495704    0.432404    0.054597    0.015693    0.706251    
#> 262: 0.342882    1.108035    3.495768    0.432519    0.054677    0.015693    0.706247    
#> 263: 0.343070    1.108462    3.495769    0.432385    0.054745    0.015676    0.706347    
#> 264: 0.342648    1.108799    3.495632    0.431594    0.054933    0.015641    0.706445    
#> 265: 0.342322    1.109072    3.495553    0.431230    0.055223    0.015618    0.706309    
#> 266: 0.341941    1.109404    3.495441    0.430811    0.055487    0.015550    0.706217    
#> 267: 0.341564    1.109411    3.495371    0.430056    0.055685    0.015497    0.706601    
#> 268: 0.341215    1.109328    3.495146    0.428715    0.055760    0.015440    0.706726    
#> 269: 0.341597    1.109211    3.495019    0.429008    0.055749    0.015413    0.706856    
#> 270: 0.341654    1.109339    3.494806    0.430455    0.055711    0.015389    0.706867    
#> 271: 0.341957    1.109410    3.494655    0.430234    0.055633    0.015394    0.707188    
#> 272: 0.342122    1.109054    3.494612    0.431086    0.055657    0.015410    0.707594    
#> 273: 0.341936    1.109167    3.494578    0.431121    0.055740    0.015423    0.707820    
#> 274: 0.341528    1.109203    3.494627    0.431510    0.055659    0.015407    0.707694    
#> 275: 0.341730    1.109139    3.494543    0.432972    0.055603    0.015401    0.707511    
#> 276: 0.341580    1.109457    3.494676    0.434019    0.055472    0.015486    0.707329    
#> 277: 0.341417    1.110065    3.494679    0.434659    0.055455    0.015602    0.707146    
#> 278: 0.340753    1.110308    3.494623    0.433978    0.055542    0.015633    0.707153    
#> 279: 0.340131    1.110250    3.494510    0.432820    0.055557    0.015666    0.707160    
#> 280: 0.339783    1.110500    3.494511    0.432298    0.055563    0.015684    0.706969    
#> 281: 0.339154    1.110574    3.494415    0.432293    0.055618    0.015660    0.706992    
#> 282: 0.338483    1.110769    3.494230    0.433267    0.055676    0.015672    0.707002    
#> 283: 0.338863    1.110830    3.494314    0.433710    0.055680    0.015691    0.706997    
#> 284: 0.338854    1.110748    3.494319    0.433345    0.055544    0.015697    0.706817    
#> 285: 0.338938    1.110731    3.494190    0.434555    0.055551    0.015684    0.706565    
#> 286: 0.338862    1.110818    3.494013    0.435566    0.055542    0.015710    0.706463    
#> 287: 0.338576    1.111015    3.493934    0.436728    0.055516    0.015735    0.706386    
#> 288: 0.338483    1.110939    3.493809    0.437628    0.055512    0.015716    0.706504    
#> 289: 0.338749    1.110799    3.493823    0.438780    0.055494    0.015679    0.706554    
#> 290: 0.339089    1.110735    3.493823    0.440351    0.055501    0.015667    0.706642    
#> 291: 0.339280    1.110856    3.493907    0.440582    0.055432    0.015643    0.706660    
#> 292: 0.339256    1.110956    3.493951    0.440005    0.055349    0.015621    0.706784    
#> 293: 0.338681    1.111022    3.493855    0.439229    0.055302    0.015613    0.706881    
#> 294: 0.338327    1.111205    3.493699    0.438657    0.055392    0.015611    0.706990    
#> 295: 0.337849    1.110979    3.493564    0.438368    0.055299    0.015632    0.706959    
#> 296: 0.337849    1.110862    3.493625    0.438186    0.055194    0.015682    0.707021    
#> 297: 0.337950    1.110927    3.493624    0.438245    0.055135    0.015739    0.706917    
#> 298: 0.338002    1.110899    3.493661    0.437541    0.055068    0.015788    0.707021    
#> 299: 0.338333    1.110980    3.493791    0.437355    0.055040    0.015804    0.706794    
#> 300: 0.338713    1.110698    3.493995    0.437248    0.055074    0.015772    0.706771    
#> 301: 0.339062    1.110364    3.494131    0.436990    0.055101    0.015762    0.706932    
#> 302: 0.339270    1.110076    3.494241    0.436685    0.055102    0.015765    0.706840    
#> 303: 0.339142    1.110344    3.494210    0.436389    0.055169    0.015768    0.706590    
#> 304: 0.339117    1.110578    3.494070    0.435776    0.055210    0.015765    0.706507    
#> 305: 0.338740    1.110507    3.493933    0.435702    0.055252    0.015796    0.706339    
#> 306: 0.338222    1.110591    3.493686    0.435902    0.055223    0.015804    0.706266    
#> 307: 0.337949    1.110627    3.493454    0.436525    0.055293    0.015797    0.706290    
#> 308: 0.337656    1.110781    3.493228    0.437546    0.055427    0.015790    0.706318    
#> 309: 0.337700    1.110637    3.493121    0.438113    0.055466    0.015770    0.706577    
#> 310: 0.338026    1.110502    3.493042    0.438731    0.055424    0.015778    0.706595    
#> 311: 0.338366    1.110539    3.493039    0.438798    0.055380    0.015794    0.706545    
#> 312: 0.338080    1.110415    3.492915    0.439464    0.055368    0.015801    0.706588    
#> 313: 0.337564    1.110289    3.492798    0.440390    0.055425    0.015796    0.706758    
#> 314: 0.337050    1.110327    3.492739    0.439824    0.055414    0.015780    0.706799    
#> 315: 0.336817    1.110433    3.492702    0.438880    0.055443    0.015790    0.706782    
#> 316: 0.337068    1.110456    3.492698    0.438761    0.055334    0.015808    0.706867    
#> 317: 0.337096    1.110405    3.492881    0.438141    0.055322    0.015801    0.707151    
#> 318: 0.337423    1.110410    3.492921    0.437724    0.055223    0.015783    0.707195    
#> 319: 0.337382    1.110343    3.492998    0.437664    0.055153    0.015759    0.707187    
#> 320: 0.336908    1.110256    3.492910    0.438014    0.055052    0.015741    0.707297    
#> 321: 0.336774    1.110295    3.492795    0.438280    0.054998    0.015699    0.707408    
#> 322: 0.336463    1.110265    3.492639    0.438061    0.054900    0.015676    0.707529    
#> 323: 0.335878    1.110362    3.492522    0.437958    0.054834    0.015684    0.707594    
#> 324: 0.335719    1.110432    3.492594    0.437265    0.054865    0.015673    0.707543    
#> 325: 0.335910    1.110564    3.492594    0.436689    0.054824    0.015667    0.707506    
#> 326: 0.335763    1.110497    3.492538    0.436827    0.054859    0.015650    0.707414    
#> 327: 0.335885    1.110518    3.492595    0.436805    0.054845    0.015622    0.707370    
#> 328: 0.336112    1.110563    3.492694    0.436179    0.054829    0.015621    0.707242    
#> 329: 0.336476    1.110608    3.492831    0.435896    0.054772    0.015608    0.707099    
#> 330: 0.336463    1.110669    3.492920    0.435409    0.054763    0.015623    0.707063    
#> 331: 0.336701    1.110907    3.493000    0.435114    0.054746    0.015644    0.706958    
#> 332: 0.336823    1.110826    3.493127    0.434647    0.054675    0.015674    0.706946    
#> 333: 0.337042    1.110773    3.493136    0.434567    0.054715    0.015698    0.706876    
#> 334: 0.336999    1.110653    3.493076    0.434700    0.054732    0.015683    0.706711    
#> 335: 0.336845    1.110736    3.492982    0.434421    0.054780    0.015647    0.706757    
#> 336: 0.336519    1.111060    3.492936    0.434022    0.054933    0.015619    0.706963    
#> 337: 0.336432    1.111374    3.492766    0.433938    0.054941    0.015600    0.706962    
#> 338: 0.336380    1.111537    3.492746    0.433612    0.054891    0.015576    0.706949    
#> 339: 0.336045    1.111814    3.492636    0.433193    0.054905    0.015561    0.706865    
#> 340: 0.335935    1.111734    3.492615    0.432776    0.054814    0.015570    0.706780    
#> 341: 0.335934    1.111634    3.492639    0.432319    0.054741    0.015582    0.706756    
#> 342: 0.335693    1.111486    3.492551    0.432034    0.054705    0.015591    0.706687    
#> 343: 0.335625    1.111510    3.492549    0.432372    0.054674    0.015614    0.706591    
#> 344: 0.335701    1.111540    3.492593    0.432894    0.054636    0.015626    0.706626    
#> 345: 0.335863    1.111516    3.492599    0.433308    0.054555    0.015639    0.706616    
#> 346: 0.335984    1.111597    3.492530    0.433440    0.054515    0.015631    0.706525    
#> 347: 0.335839    1.111632    3.492360    0.433409    0.054509    0.015626    0.706665    
#> 348: 0.335556    1.111809    3.492268    0.433070    0.054509    0.015618    0.706738    
#> 349: 0.335354    1.111852    3.492175    0.433318    0.054544    0.015594    0.706696    
#> 350: 0.334927    1.111834    3.492093    0.432797    0.054552    0.015570    0.706776    
#> 351: 0.334544    1.112037    3.491998    0.432318    0.054578    0.015559    0.706783    
#> 352: 0.333973    1.112188    3.491943    0.431934    0.054565    0.015588    0.706858    
#> 353: 0.334001    1.112348    3.491964    0.431562    0.054572    0.015610    0.706893    
#> 354: 0.333968    1.112347    3.491914    0.431324    0.054585    0.015628    0.706852    
#> 355: 0.334314    1.112185    3.492041    0.431196    0.054507    0.015670    0.706788    
#> 356: 0.334409    1.112118    3.492114    0.431125    0.054441    0.015678    0.706841    
#> 357: 0.334723    1.112135    3.492196    0.430747    0.054377    0.015695    0.706835    
#> 358: 0.334788    1.112182    3.492291    0.430711    0.054351    0.015687    0.706747    
#> 359: 0.334770    1.112237    3.492322    0.430745    0.054331    0.015663    0.706662    
#> 360: 0.334645    1.112258    3.492269    0.430870    0.054292    0.015628    0.706543    
#> 361: 0.334648    1.112271    3.492245    0.430830    0.054347    0.015607    0.706405    
#> 362: 0.334835    1.112284    3.492194    0.430820    0.054370    0.015590    0.706224    
#> 363: 0.334914    1.112140    3.492228    0.430702    0.054319    0.015578    0.706169    
#> 364: 0.334925    1.112185    3.492236    0.430677    0.054302    0.015569    0.706244    
#> 365: 0.335120    1.112058    3.492297    0.430863    0.054319    0.015545    0.706359    
#> 366: 0.335318    1.112121    3.492240    0.431380    0.054282    0.015522    0.706530    
#> 367: 0.335427    1.112174    3.492235    0.431383    0.054270    0.015504    0.706476    
#> 368: 0.335566    1.112244    3.492229    0.431393    0.054284    0.015482    0.706376    
#> 369: 0.335485    1.112199    3.492209    0.431247    0.054254    0.015471    0.706401    
#> 370: 0.335312    1.112304    3.492176    0.430930    0.054314    0.015449    0.706393    
#> 371: 0.335146    1.112313    3.492047    0.431009    0.054314    0.015441    0.706367    
#> 372: 0.335057    1.112392    3.492000    0.431243    0.054318    0.015413    0.706454    
#> 373: 0.334711    1.112570    3.491890    0.431382    0.054362    0.015399    0.706526    
#> 374: 0.334792    1.112568    3.491839    0.432079    0.054407    0.015382    0.706425    
#> 375: 0.334796    1.112575    3.491806    0.432452    0.054489    0.015376    0.706383    
#> 376: 0.334700    1.112757    3.491751    0.432528    0.054564    0.015363    0.706305    
#> 377: 0.334476    1.112897    3.491698    0.432434    0.054511    0.015357    0.706264    
#> 378: 0.334221    1.112912    3.491628    0.432162    0.054413    0.015364    0.706322    
#> 379: 0.334162    1.112759    3.491693    0.431805    0.054359    0.015381    0.706248    
#> 380: 0.334092    1.112679    3.491691    0.431882    0.054278    0.015376    0.706256    
#> 381: 0.333947    1.112549    3.491719    0.431690    0.054208    0.015384    0.706183    
#> 382: 0.334024    1.112544    3.491756    0.431835    0.054106    0.015387    0.706090    
#> 383: 0.334090    1.112473    3.491790    0.431617    0.054050    0.015378    0.706117    
#> 384: 0.333907    1.112391    3.491771    0.431533    0.054056    0.015362    0.706158    
#> 385: 0.333875    1.112306    3.491807    0.431063    0.054059    0.015360    0.706129    
#> 386: 0.334159    1.112430    3.491872    0.430927    0.054099    0.015368    0.706084    
#> 387: 0.334457    1.112495    3.491992    0.431112    0.054082    0.015375    0.705955    
#> 388: 0.334681    1.112528    3.492038    0.431382    0.054093    0.015372    0.705873    
#> 389: 0.334800    1.112402    3.492032    0.431766    0.054132    0.015365    0.705862    
#> 390: 0.335034    1.112366    3.492017    0.432302    0.054119    0.015363    0.705801    
#> 391: 0.335327    1.112305    3.492028    0.432461    0.054122    0.015344    0.705752    
#> 392: 0.335502    1.112259    3.492128    0.432364    0.054156    0.015344    0.705812    
#> 393: 0.335551    1.112111    3.492181    0.432081    0.054144    0.015351    0.705842    
#> 394: 0.335807    1.112074    3.492204    0.432227    0.054161    0.015331    0.705850    
#> 395: 0.335985    1.112074    3.492220    0.432264    0.054199    0.015325    0.705824    
#> 396: 0.336203    1.112054    3.492258    0.432369    0.054180    0.015325    0.705793    
#> 397: 0.336383    1.112024    3.492332    0.432413    0.054191    0.015307    0.705786    
#> 398: 0.336638    1.111888    3.492426    0.432915    0.054165    0.015308    0.705745    
#> 399: 0.336921    1.111794    3.492581    0.433042    0.054130    0.015314    0.705706    
#> 400: 0.337075    1.111755    3.492668    0.432614    0.054116    0.015315    0.705708    
#> 401: 0.337212    1.111733    3.492707    0.432373    0.054143    0.015305    0.705753    
#> 402: 0.337331    1.111764    3.492750    0.432253    0.054137    0.015302    0.705799    
#> 403: 0.337388    1.111721    3.492765    0.432040    0.054115    0.015311    0.705853    
#> 404: 0.337258    1.111820    3.492647    0.431950    0.054108    0.015308    0.705941    
#> 405: 0.337125    1.111870    3.492619    0.432044    0.054157    0.015296    0.706005    
#> 406: 0.337171    1.111944    3.492662    0.431759    0.054233    0.015278    0.706119    
#> 407: 0.337190    1.111941    3.492747    0.431305    0.054252    0.015272    0.706149    
#> 408: 0.337443    1.111995    3.492780    0.431098    0.054277    0.015270    0.706132    
#> 409: 0.337325    1.112008    3.492774    0.430766    0.054237    0.015276    0.706048    
#> 410: 0.337166    1.112038    3.492699    0.430520    0.054173    0.015277    0.705993    
#> 411: 0.337111    1.111945    3.492628    0.430538    0.054113    0.015289    0.705992    
#> 412: 0.337021    1.111972    3.492612    0.430406    0.054041    0.015313    0.706012    
#> 413: 0.336948    1.111978    3.492583    0.430099    0.054012    0.015332    0.706023    
#> 414: 0.336655    1.112016    3.492546    0.430185    0.053955    0.015339    0.706026    
#> 415: 0.336722    1.112048    3.492541    0.429981    0.053921    0.015344    0.705950    
#> 416: 0.336880    1.112064    3.492587    0.429658    0.053897    0.015343    0.705943    
#> 417: 0.337031    1.111967    3.492623    0.429604    0.053853    0.015354    0.705914    
#> 418: 0.337054    1.111919    3.492669    0.429708    0.053808    0.015353    0.705873    
#> 419: 0.337099    1.112151    3.492676    0.429878    0.053817    0.015365    0.705806    
#> 420: 0.337096    1.112373    3.492636    0.429892    0.053778    0.015370    0.705743    
#> 421: 0.336959    1.112489    3.492615    0.430010    0.053725    0.015365    0.705689    
#> 422: 0.336949    1.112533    3.492570    0.429848    0.053717    0.015355    0.705627    
#> 423: 0.336988    1.112538    3.492560    0.429989    0.053701    0.015358    0.705697    
#> 424: 0.336783    1.112571    3.492438    0.430250    0.053685    0.015350    0.705691    
#> 425: 0.336714    1.112732    3.492371    0.430626    0.053749    0.015337    0.705703    
#> 426: 0.336810    1.112898    3.492375    0.430885    0.053820    0.015337    0.705646    
#> 427: 0.336717    1.112930    3.492359    0.431148    0.053830    0.015346    0.705540    
#> 428: 0.336602    1.112988    3.492337    0.431070    0.053772    0.015356    0.705587    
#> 429: 0.336571    1.112983    3.492349    0.431148    0.053711    0.015376    0.705536    
#> 430: 0.336560    1.112953    3.492365    0.431232    0.053684    0.015395    0.705491    
#> 431: 0.336680    1.112905    3.492403    0.431107    0.053683    0.015400    0.705435    
#> 432: 0.336788    1.112938    3.492432    0.431076    0.053682    0.015400    0.705370    
#> 433: 0.336919    1.113011    3.492445    0.431124    0.053732    0.015383    0.705432    
#> 434: 0.336780    1.112915    3.492445    0.430804    0.053720    0.015378    0.705504    
#> 435: 0.336842    1.112887    3.492428    0.430873    0.053711    0.015379    0.705434    
#> 436: 0.336809    1.112865    3.492414    0.430607    0.053749    0.015371    0.705423    
#> 437: 0.336808    1.112827    3.492384    0.430559    0.053765    0.015369    0.705422    
#> 438: 0.336835    1.112922    3.492381    0.430837    0.053750    0.015363    0.705370    
#> 439: 0.336764    1.113078    3.492374    0.430612    0.053784    0.015368    0.705311    
#> 440: 0.336602    1.113169    3.492350    0.430585    0.053784    0.015368    0.705269    
#> 441: 0.336520    1.113157    3.492373    0.430384    0.053813    0.015370    0.705250    
#> 442: 0.336361    1.113190    3.492369    0.430329    0.053801    0.015364    0.705223    
#> 443: 0.336289    1.113245    3.492342    0.430329    0.053800    0.015374    0.705135    
#> 444: 0.336247    1.113292    3.492281    0.430287    0.053808    0.015389    0.705096    
#> 445: 0.336014    1.113301    3.492202    0.430120    0.053803    0.015406    0.705175    
#> 446: 0.335873    1.113380    3.492137    0.430200    0.053795    0.015418    0.705163    
#> 447: 0.335774    1.113493    3.492101    0.430027    0.053809    0.015429    0.705173    
#> 448: 0.335585    1.113474    3.492089    0.430016    0.053817    0.015442    0.705151    
#> 449: 0.335548    1.113571    3.492130    0.429980    0.053862    0.015451    0.705132    
#> 450: 0.335612    1.113607    3.492147    0.429942    0.053900    0.015471    0.705165    
#> 451: 0.335567    1.113635    3.492141    0.429778    0.053915    0.015478    0.705168    
#> 452: 0.335455    1.113614    3.492128    0.429641    0.053903    0.015468    0.705188    
#> 453: 0.335470    1.113578    3.492147    0.429671    0.053862    0.015465    0.705234    
#> 454: 0.335429    1.113637    3.492182    0.429511    0.053906    0.015468    0.705203    
#> 455: 0.335357    1.113588    3.492188    0.429418    0.053895    0.015484    0.705222    
#> 456: 0.335294    1.113536    3.492153    0.429480    0.053918    0.015493    0.705244    
#> 457: 0.335200    1.113537    3.492153    0.429455    0.053902    0.015518    0.705172    
#> 458: 0.335163    1.113650    3.492155    0.429528    0.053900    0.015539    0.705135    
#> 459: 0.334862    1.113823    3.492098    0.429530    0.053899    0.015551    0.705151    
#> 460: 0.334610    1.113881    3.492021    0.429316    0.053910    0.015560    0.705118    
#> 461: 0.334441    1.113984    3.491970    0.429066    0.053914    0.015561    0.705124    
#> 462: 0.334374    1.114128    3.491959    0.429173    0.053921    0.015559    0.705046    
#> 463: 0.334446    1.114204    3.491984    0.429300    0.053927    0.015567    0.704967    
#> 464: 0.334556    1.114262    3.491984    0.429454    0.053937    0.015575    0.704868    
#> 465: 0.334647    1.114102    3.492022    0.429649    0.053953    0.015573    0.704806    
#> 466: 0.334746    1.114075    3.492047    0.429681    0.053957    0.015575    0.704848    
#> 467: 0.334734    1.113993    3.492058    0.429684    0.053926    0.015589    0.704751    
#> 468: 0.334881    1.113996    3.492118    0.429850    0.053924    0.015607    0.704760    
#> 469: 0.335041    1.113998    3.492219    0.429574    0.053987    0.015629    0.704803    
#> 470: 0.334945    1.114086    3.492201    0.429535    0.054035    0.015639    0.704819    
#> 471: 0.334743    1.114099    3.492147    0.429455    0.054020    0.015645    0.704800    
#> 472: 0.334668    1.114192    3.492079    0.429320    0.054062    0.015645    0.704796    
#> 473: 0.334573    1.114233    3.492028    0.429200    0.054060    0.015643    0.704752    
#> 474: 0.334503    1.114285    3.491953    0.429338    0.054098    0.015635    0.704817    
#> 475: 0.334509    1.114433    3.491890    0.429605    0.054147    0.015628    0.704803    
#> 476: 0.334507    1.114522    3.491826    0.429865    0.054212    0.015619    0.704821    
#> 477: 0.334384    1.114575    3.491786    0.429738    0.054265    0.015611    0.704777    
#> 478: 0.334262    1.114573    3.491783    0.429561    0.054313    0.015601    0.704805    
#> 479: 0.334339    1.114663    3.491806    0.429524    0.054379    0.015595    0.704795    
#> 480: 0.334393    1.114685    3.491796    0.429588    0.054434    0.015603    0.704845    
#> 481: 0.334333    1.114711    3.491762    0.429627    0.054462    0.015608    0.704825    
#> 482: 0.334384    1.114706    3.491748    0.429444    0.054468    0.015628    0.704860    
#> 483: 0.334334    1.114630    3.491743    0.429277    0.054458    0.015630    0.704940    
#> 484: 0.334320    1.114475    3.491784    0.429332    0.054402    0.015640    0.704996    
#> 485: 0.334272    1.114525    3.491771    0.429131    0.054395    0.015640    0.705083    
#> 486: 0.334187    1.114569    3.491774    0.428999    0.054420    0.015637    0.705100    
#> 487: 0.334247    1.114674    3.491846    0.428822    0.054429    0.015639    0.705111    
#> 488: 0.334216    1.114579    3.491914    0.428589    0.054446    0.015632    0.705122    
#> 489: 0.334295    1.114509    3.491940    0.428533    0.054441    0.015633    0.705191    
#> 490: 0.334427    1.114543    3.491968    0.428575    0.054479    0.015628    0.705149    
#> 491: 0.334441    1.114621    3.492003    0.428670    0.054502    0.015621    0.705149    
#> 492: 0.334373    1.114726    3.492006    0.428789    0.054524    0.015606    0.705270    
#> 493: 0.334335    1.114745    3.492006    0.428738    0.054587    0.015602    0.705296    
#> 494: 0.334365    1.114822    3.491970    0.428637    0.054620    0.015606    0.705248    
#> 495: 0.334435    1.114858    3.491970    0.428727    0.054597    0.015596    0.705198    
#> 496: 0.334516    1.114938    3.491920    0.428952    0.054625    0.015587    0.705150    
#> 497: 0.334401    1.114918    3.491883    0.429028    0.054602    0.015576    0.705178    
#> 498: 0.334355    1.114887    3.491854    0.429173    0.054643    0.015566    0.705123    
#> 499: 0.334329    1.114937    3.491815    0.428892    0.054633    0.015561    0.705130    
#> 500: 0.334308    1.114872    3.491809    0.428699    0.054624    0.015558    0.705157    
#> [====|====|=>-------------------------------------]  25%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.321167    0.965573    3.442328    0.392401    0.072521    0.016800    1.702904    
#> 002: 0.451719    1.016499    3.478587    0.372781    0.068895    0.015960    1.194804    
#> 003: 0.635178    0.974154    3.510822    0.354142    0.065450    0.015162    0.884536    
#> 004: 0.624620    1.012289    3.514987    0.336435    0.062178    0.014404    0.787556    
#> 005: 0.715152    1.014717    3.514129    0.399488    0.059069    0.013684    0.726083    
#> 006: 0.670099    1.025529    3.498523    0.379514    0.056116    0.013000    0.707160    
#> 007: 0.608071    1.040961    3.481891    0.431349    0.053310    0.012350    0.728253    
#> 008: 0.622380    1.025201    3.487459    0.447446    0.050644    0.011732    0.717541    
#> 009: 0.588177    1.011253    3.485390    0.425073    0.048112    0.011146    0.731451    
#> 010: 0.615018    1.009634    3.488638    0.403820    0.045706    0.010588    0.745275    
#> 011: 0.634858    1.020825    3.499043    0.392486    0.047132    0.010059    0.729850    
#> 012: 0.671714    1.025803    3.487361    0.486550    0.044775    0.009556    0.722347    
#> 013: 0.598394    1.034531    3.473373    0.559011    0.054635    0.009078    0.722644    
#> 014: 0.644399    1.051331    3.482830    0.531060    0.051903    0.008624    0.738305    
#> 015: 0.669606    1.035624    3.509448    0.504507    0.049796    0.009524    0.717922    
#> 016: 0.664929    1.046536    3.501233    0.479282    0.047306    0.010105    0.718136    
#> 017: 0.633048    1.050135    3.497755    0.488790    0.044941    0.009600    0.717788    
#> 018: 0.675626    1.056213    3.480055    0.464351    0.044310    0.009120    0.742299    
#> 019: 0.606312    1.049976    3.480706    0.441133    0.046122    0.008664    0.725571    
#> 020: 0.662981    1.042322    3.487624    0.434150    0.045633    0.008230    0.724082    
#> 021: 0.660946    1.035690    3.490517    0.412443    0.043352    0.007819    0.724629    
#> 022: 0.616195    1.033621    3.495123    0.391821    0.045069    0.007428    0.727162    
#> 023: 0.726067    1.036826    3.513366    0.372230    0.042816    0.007938    0.754502    
#> 024: 0.682103    1.054949    3.507725    0.361447    0.040675    0.007777    0.739894    
#> 025: 0.649545    1.031441    3.499477    0.383969    0.045555    0.007388    0.739404    
#> 026: 0.616977    1.059319    3.489247    0.424100    0.051192    0.007997    0.738453    
#> 027: 0.614430    1.081661    3.482104    0.443606    0.048632    0.007793    0.732488    
#> 028: 0.677215    1.056580    3.494846    0.425742    0.046201    0.007413    0.703937    
#> 029: 0.658197    1.037836    3.487627    0.404454    0.047560    0.007042    0.735780    
#> 030: 0.670464    1.052439    3.492396    0.440997    0.048113    0.006690    0.724322    
#> 031: 0.625122    1.044086    3.497622    0.418947    0.048753    0.008423    0.718508    
#> 032: 0.644136    1.035145    3.509624    0.398000    0.046315    0.008002    0.719893    
#> 033: 0.606034    1.036754    3.490894    0.393139    0.043999    0.007602    0.718834    
#> 034: 0.629345    1.041302    3.487209    0.397676    0.053378    0.007222    0.729343    
#> 035: 0.590908    1.041108    3.482391    0.377792    0.054323    0.006861    0.726985    
#> 036: 0.610767    1.049836    3.484315    0.358902    0.051607    0.006518    0.736225    
#> 037: 0.620746    1.054091    3.490947    0.391991    0.049027    0.006894    0.753258    
#> 038: 0.648034    1.038843    3.483748    0.406821    0.046576    0.006549    0.734018    
#> 039: 0.652142    1.048363    3.488393    0.386480    0.045211    0.007507    0.719263    
#> 040: 0.606345    1.058226    3.487304    0.383308    0.042951    0.007132    0.722925    
#> 041: 0.598272    1.047562    3.488881    0.388793    0.049425    0.006775    0.705997    
#> 042: 0.659577    1.026735    3.486325    0.444387    0.051092    0.006503    0.714608    
#> 043: 0.610768    1.048983    3.477497    0.422168    0.048537    0.007501    0.719102    
#> 044: 0.602329    1.059731    3.478325    0.401496    0.046110    0.008395    0.720300    
#> 045: 0.586231    1.079152    3.469314    0.399023    0.043805    0.007976    0.721024    
#> 046: 0.592867    1.069584    3.465272    0.399855    0.041615    0.007577    0.716426    
#> 047: 0.605345    1.054331    3.465509    0.391205    0.043644    0.007198    0.721250    
#> 048: 0.594724    1.047567    3.478088    0.371645    0.042163    0.006838    0.729355    
#> 049: 0.611457    1.034799    3.491227    0.353063    0.050687    0.006496    0.734463    
#> 050: 0.646605    1.032153    3.486031    0.356452    0.050068    0.006171    0.764912    
#> 051: 0.605110    1.046614    3.486452    0.349216    0.052257    0.006969    0.761470    
#> 052: 0.594651    1.045286    3.489561    0.331755    0.051382    0.007078    0.755936    
#> 053: 0.650088    1.027819    3.494420    0.407329    0.049362    0.007052    0.729124    
#> 054: 0.618809    1.033919    3.494115    0.386962    0.049815    0.009089    0.721926    
#> 055: 0.633010    1.037076    3.484856    0.420683    0.053837    0.008705    0.720655    
#> 056: 0.645424    1.025429    3.493670    0.444978    0.051145    0.009190    0.724203    
#> 057: 0.622260    1.041430    3.487594    0.422729    0.050716    0.009660    0.731054    
#> 058: 0.658674    1.044601    3.488476    0.424784    0.048180    0.009177    0.728542    
#> 059: 0.636747    1.017780    3.487467    0.458992    0.045771    0.008718    0.741505    
#> 060: 0.643785    1.006561    3.498277    0.464794    0.049008    0.008282    0.751504    
#> 061: 0.701537    1.014456    3.501636    0.441554    0.057158    0.008789    0.736716    
#> 062: 0.632931    1.037865    3.505010    0.419476    0.071145    0.008350    0.743954    
#> 063: 0.639393    1.021755    3.490362    0.446741    0.067588    0.008740    0.736704    
#> 064: 0.615540    1.055183    3.489196    0.424404    0.064208    0.008303    0.721796    
#> 065: 0.677794    1.046460    3.477352    0.427380    0.060998    0.007888    0.719542    
#> 066: 0.639277    1.065978    3.477217    0.406011    0.057948    0.008009    0.716160    
#> 067: 0.610028    1.055850    3.477304    0.385710    0.055051    0.009261    0.712218    
#> 068: 0.601397    1.043176    3.491281    0.366425    0.052298    0.010968    0.730708    
#> 069: 0.563446    1.060079    3.468133    0.348104    0.049683    0.010920    0.754558    
#> 070: 0.605991    1.052614    3.486823    0.368821    0.051267    0.010374    0.735216    
#> 071: 0.593951    1.087696    3.465622    0.411789    0.048704    0.009856    0.754590    
#> 072: 0.594393    1.061229    3.464607    0.466579    0.049278    0.009363    0.713626    
#> 073: 0.635300    1.059369    3.488839    0.466813    0.050864    0.008895    0.697643    
#> 074: 0.630579    1.040187    3.490113    0.443472    0.051092    0.008502    0.723028    
#> 075: 0.675964    1.031731    3.506903    0.489377    0.048538    0.008760    0.714447    
#> 076: 0.690467    1.047843    3.508050    0.496944    0.046111    0.008322    0.737419    
#> 077: 0.715200    1.027807    3.504215    0.512376    0.043805    0.008819    0.708538    
#> 078: 0.662588    1.027391    3.501293    0.486757    0.048770    0.009967    0.698039    
#> 079: 0.690177    1.054132    3.504080    0.534925    0.059001    0.009469    0.710470    
#> 080: 0.656521    1.027161    3.500851    0.508179    0.064949    0.010467    0.744122    
#> 081: 0.656115    1.042531    3.499130    0.482770    0.061702    0.011326    0.720162    
#> 082: 0.651297    1.035306    3.510175    0.458631    0.058617    0.010760    0.764515    
#> 083: 0.708564    1.044902    3.512959    0.525815    0.056090    0.010222    0.743515    
#> 084: 0.696106    1.050124    3.498681    0.499743    0.053286    0.009711    0.730709    
#> 085: 0.639539    1.039079    3.508416    0.474756    0.055030    0.009225    0.737089    
#> 086: 0.644299    1.032916    3.501648    0.451018    0.052279    0.008764    0.756813    
#> 087: 0.658591    1.034125    3.495451    0.428467    0.049665    0.008326    0.749608    
#> 088: 0.567294    1.046256    3.483095    0.407044    0.047698    0.007909    0.729202    
#> 089: 0.627106    1.063181    3.478943    0.386692    0.057433    0.007514    0.743650    
#> 090: 0.589614    1.053711    3.482048    0.367357    0.054562    0.007138    0.725955    
#> 091: 0.607097    1.034237    3.481619    0.371509    0.051834    0.007208    0.715375    
#> 092: 0.602097    1.055282    3.473671    0.402018    0.049587    0.009433    0.714553    
#> 093: 0.558379    1.038355    3.473277    0.381917    0.049787    0.009241    0.716022    
#> 094: 0.580450    1.059872    3.469098    0.362821    0.059297    0.010574    0.726423    
#> 095: 0.614028    1.065629    3.479501    0.401458    0.056332    0.010045    0.734976    
#> 096: 0.584223    1.066391    3.473642    0.428314    0.053515    0.009543    0.724954    
#> 097: 0.642349    1.078093    3.480008    0.485856    0.050840    0.009066    0.717504    
#> 098: 0.630886    1.092668    3.485121    0.461563    0.048298    0.008613    0.732986    
#> 099: 0.579099    1.046768    3.479216    0.438485    0.045883    0.008182    0.732343    
#> 100: 0.597190    1.049296    3.470424    0.426815    0.043589    0.007773    0.716700    
#> 101: 0.544643    1.043557    3.456568    0.408316    0.041409    0.007384    0.729074    
#> 102: 0.565428    1.042873    3.449642    0.445368    0.045563    0.007015    0.724654    
#> 103: 0.586153    1.061906    3.462578    0.423099    0.047047    0.006664    0.724555    
#> 104: 0.575355    1.047554    3.475280    0.401944    0.045441    0.006331    0.749275    
#> 105: 0.579274    1.057328    3.479878    0.381847    0.053582    0.006015    0.739244    
#> 106: 0.582836    1.060501    3.473580    0.448889    0.052862    0.005714    0.728020    
#> 107: 0.647475    1.065676    3.473541    0.442685    0.050219    0.006333    0.744759    
#> 108: 0.605500    1.054670    3.473484    0.420551    0.047708    0.006070    0.740310    
#> 109: 0.589440    1.052297    3.481093    0.403970    0.045322    0.006610    0.743266    
#> 110: 0.626079    1.049956    3.474690    0.402704    0.050732    0.006279    0.716221    
#> 111: 0.615750    1.060515    3.472721    0.424347    0.052093    0.005965    0.708881    
#> 112: 0.595715    1.045558    3.471762    0.403129    0.069040    0.005667    0.710367    
#> 113: 0.601136    1.061225    3.477133    0.382973    0.067561    0.005384    0.722826    
#> 114: 0.611207    1.064738    3.480770    0.363824    0.066256    0.005143    0.728319    
#> 115: 0.607757    1.031906    3.473135    0.376709    0.067875    0.004886    0.756929    
#> 116: 0.609208    1.047939    3.480832    0.371138    0.064482    0.004641    0.749588    
#> 117: 0.597775    1.080111    3.478887    0.403225    0.066077    0.004409    0.718135    
#> 118: 0.631529    1.063437    3.477617    0.383064    0.062773    0.004189    0.722102    
#> 119: 0.593321    1.048021    3.475818    0.394547    0.059634    0.004898    0.734843    
#> 120: 0.596835    1.052822    3.477860    0.477945    0.056653    0.004950    0.728764    
#> 121: 0.611474    1.080133    3.474205    0.454048    0.053820    0.005225    0.723513    
#> 122: 0.601651    1.042199    3.478252    0.431346    0.051129    0.005242    0.723673    
#> 123: 0.628853    1.067741    3.478090    0.409778    0.048573    0.005582    0.739552    
#> 124: 0.620035    1.094274    3.476300    0.412956    0.046144    0.006603    0.726999    
#> 125: 0.605107    1.095598    3.465802    0.433076    0.043837    0.006427    0.703073    
#> 126: 0.562040    1.096362    3.455980    0.411423    0.041645    0.006798    0.730773    
#> 127: 0.601813    1.071821    3.458585    0.408538    0.040967    0.006458    0.737379    
#> 128: 0.597510    1.061987    3.469403    0.416726    0.048192    0.006135    0.721884    
#> 129: 0.636147    1.059864    3.480840    0.419602    0.061468    0.005828    0.722333    
#> 130: 0.682438    1.073016    3.488436    0.413688    0.058395    0.005537    0.711730    
#> 131: 0.647186    1.045726    3.482289    0.426253    0.055475    0.006400    0.723454    
#> 132: 0.617369    1.064861    3.472987    0.435415    0.052701    0.006737    0.716547    
#> 133: 0.610493    1.060453    3.472451    0.413645    0.058122    0.006543    0.729482    
#> 134: 0.624992    1.047993    3.488020    0.422302    0.055216    0.006802    0.720369    
#> 135: 0.628998    1.034354    3.490775    0.402359    0.061236    0.006844    0.731207    
#> 136: 0.623850    1.029822    3.487976    0.430623    0.058174    0.006502    0.724237    
#> 137: 0.648046    1.021080    3.492208    0.484977    0.055266    0.006177    0.728659    
#> 138: 0.622489    1.057519    3.477606    0.490941    0.057901    0.005868    0.720696    
#> 139: 0.619653    1.058890    3.477168    0.542522    0.055095    0.005863    0.710996    
#> 140: 0.620566    1.075437    3.482335    0.515396    0.059302    0.005569    0.728957    
#> 141: 0.644046    1.045705    3.496372    0.489626    0.072365    0.005291    0.738824    
#> 142: 0.690786    1.027007    3.502078    0.501755    0.069418    0.005026    0.745390    
#> 143: 0.660591    1.074581    3.497005    0.503395    0.065947    0.005301    0.752712    
#> 144: 0.648258    1.060657    3.498162    0.478225    0.062650    0.005418    0.743790    
#> 145: 0.632846    1.048661    3.490435    0.454314    0.059525    0.006127    0.715191    
#> 146: 0.642982    1.050290    3.492308    0.510888    0.057497    0.006956    0.723317    
#> 147: 0.605072    1.024168    3.488395    0.485344    0.054623    0.007617    0.705715    
#> 148: 0.615169    1.048753    3.499717    0.461076    0.051891    0.008232    0.705159    
#> 149: 0.590412    1.043692    3.485364    0.438023    0.055506    0.008111    0.727742    
#> 150: 0.634610    1.046550    3.485196    0.416121    0.063964    0.007705    0.718752    
#> 151: 0.571264    1.057972    3.475249    0.395315    0.062316    0.007320    0.737086    
#> 152: 0.543960    1.065494    3.477535    0.317969    0.051958    0.006211    0.727367    
#> 153: 0.563292    1.073222    3.475314    0.333826    0.050697    0.005757    0.746942    
#> 154: 0.614285    1.074339    3.481052    0.314901    0.048405    0.005696    0.756237    
#> 155: 0.589397    1.049924    3.481704    0.349873    0.056380    0.004507    0.737450    
#> 156: 0.596964    1.066182    3.491549    0.337239    0.057239    0.005669    0.746536    
#> 157: 0.637064    1.050364    3.493489    0.322463    0.064494    0.006184    0.742066    
#> 158: 0.651976    1.070076    3.489177    0.367146    0.056786    0.006072    0.730259    
#> 159: 0.586185    1.084906    3.477218    0.380220    0.045894    0.005988    0.733868    
#> 160: 0.647453    1.048413    3.488855    0.368014    0.051314    0.006357    0.735360    
#> 161: 0.622498    1.044477    3.484789    0.372293    0.063496    0.006326    0.730073    
#> 162: 0.606675    1.058635    3.478703    0.325551    0.063235    0.006382    0.742873    
#> 163: 0.619635    1.036906    3.483119    0.325511    0.062121    0.005683    0.739293    
#> 164: 0.587932    1.049861    3.479275    0.334568    0.055680    0.005849    0.741877    
#> 165: 0.639329    1.031969    3.491536    0.377199    0.052641    0.005728    0.742933    
#> 166: 0.644142    1.071061    3.489517    0.381823    0.054353    0.004819    0.737476    
#> 167: 0.647974    1.065878    3.495692    0.390531    0.049501    0.004147    0.747714    
#> 168: 0.676661    1.067020    3.498318    0.456214    0.052890    0.004579    0.728513    
#> 169: 0.676234    1.078150    3.495037    0.490574    0.056880    0.005497    0.739996    
#> 170: 0.644807    1.037290    3.494335    0.402689    0.051459    0.004577    0.746624    
#> 171: 0.642313    1.047950    3.492189    0.416091    0.054094    0.004607    0.714278    
#> 172: 0.611900    1.065069    3.494350    0.396148    0.051308    0.004313    0.707438    
#> 173: 0.607959    1.057372    3.485668    0.359225    0.054663    0.005484    0.717488    
#> 174: 0.628989    1.039964    3.491399    0.380169    0.043092    0.005716    0.717767    
#> 175: 0.631693    1.042980    3.481519    0.428386    0.050954    0.005732    0.730366    
#> 176: 0.578919    1.069954    3.478843    0.350057    0.049460    0.005145    0.721501    
#> 177: 0.637065    1.067433    3.481729    0.391196    0.041744    0.005292    0.722125    
#> 178: 0.632608    1.050624    3.483324    0.419828    0.042563    0.005651    0.720132    
#> 179: 0.657067    1.063032    3.483439    0.471361    0.049515    0.005905    0.723185    
#> 180: 0.605007    1.048522    3.487637    0.380668    0.045666    0.007768    0.713576    
#> 181: 0.601352    1.041209    3.462696    0.381577    0.047629    0.007959    0.703111    
#> 182: 0.612366    1.067140    3.460327    0.407643    0.054387    0.006430    0.708939    
#> 183: 0.566609    1.081244    3.452474    0.347287    0.041745    0.006264    0.718007    
#> 184: 0.532111    1.071279    3.454771    0.368672    0.039434    0.005723    0.738247    
#> 185: 0.555185    1.092862    3.452042    0.414187    0.039181    0.007665    0.729353    
#> 186: 0.533922    1.096777    3.450023    0.431699    0.036740    0.007516    0.738460    
#> 187: 0.564062    1.059584    3.467300    0.402308    0.040015    0.006870    0.729256    
#> 188: 0.550953    1.072327    3.466971    0.434912    0.044410    0.008677    0.743201    
#> 189: 0.604769    1.087654    3.479559    0.370039    0.042334    0.009341    0.732990    
#> 190: 0.628410    1.067846    3.489042    0.474350    0.036872    0.008320    0.733526    
#> 191: 0.582282    1.078108    3.465789    0.361351    0.037973    0.006218    0.732656    
#> 192: 0.553022    1.067367    3.466825    0.350358    0.032076    0.005000    0.742682    
#> 193: 0.582057    1.086980    3.463912    0.331073    0.037808    0.003901    0.744817    
#> 194: 0.537551    1.068737    3.453765    0.343145    0.047468    0.004422    0.743569    
#> 195: 0.530678    1.076730    3.448228    0.343888    0.041063    0.004011    0.757161    
#> 196: 0.529321    1.084595    3.457457    0.311736    0.042959    0.004635    0.790669    
#> 197: 0.557125    1.088521    3.475647    0.317599    0.053634    0.004995    0.750475    
#> 198: 0.607614    1.062947    3.463431    0.303971    0.054010    0.005628    0.734875    
#> 199: 0.548172    1.059965    3.469441    0.328058    0.053417    0.005174    0.733027    
#> 200: 0.554061    1.063385    3.469718    0.369053    0.059375    0.004503    0.719057    
#> 201: 0.561195    1.059588    3.468890    0.388551    0.059760    0.005143    0.714734    
#> 202: 0.559396    1.059700    3.466981    0.397331    0.055810    0.005375    0.720497    
#> 203: 0.577499    1.060657    3.465449    0.429885    0.054978    0.005908    0.718953    
#> 204: 0.584145    1.058162    3.466820    0.430391    0.055472    0.005908    0.721402    
#> 205: 0.588503    1.063027    3.468813    0.426048    0.055391    0.005825    0.723969    
#> 206: 0.591459    1.066489    3.470856    0.423625    0.056083    0.005768    0.728561    
#> 207: 0.594697    1.065745    3.471498    0.427369    0.055147    0.005528    0.728691    
#> 208: 0.597018    1.065470    3.472032    0.423353    0.053842    0.005486    0.727617    
#> 209: 0.600487    1.062441    3.473251    0.418231    0.053425    0.005468    0.727140    
#> 210: 0.604474    1.058636    3.475120    0.419387    0.054111    0.005434    0.725720    
#> 211: 0.609178    1.058378    3.476436    0.419907    0.054535    0.005295    0.724045    
#> 212: 0.611680    1.057455    3.477327    0.418892    0.054075    0.005326    0.723055    
#> 213: 0.613328    1.055227    3.477883    0.411738    0.053190    0.005420    0.723956    
#> 214: 0.614949    1.054871    3.478188    0.405484    0.052782    0.005391    0.724583    
#> 215: 0.612458    1.056384    3.478448    0.403354    0.052880    0.005333    0.724057    
#> 216: 0.613584    1.056379    3.478538    0.403508    0.053164    0.005218    0.725604    
#> 217: 0.613499    1.057211    3.478398    0.404726    0.053112    0.005132    0.725476    
#> 218: 0.613264    1.058152    3.477647    0.403335    0.052778    0.005056    0.725800    
#> 219: 0.612459    1.057753    3.477018    0.406871    0.053134    0.004984    0.726437    
#> 220: 0.610418    1.058041    3.476721    0.407240    0.053343    0.004968    0.726533    
#> 221: 0.609682    1.058532    3.476320    0.406968    0.053163    0.004919    0.726852    
#> 222: 0.609298    1.058538    3.476437    0.403507    0.053773    0.004936    0.727273    
#> 223: 0.610055    1.058951    3.476682    0.402369    0.053851    0.004965    0.726401    
#> 224: 0.610858    1.059063    3.477148    0.399634    0.053896    0.004965    0.726572    
#> 225: 0.610926    1.057201    3.477590    0.400664    0.053830    0.004945    0.726669    
#> 226: 0.609878    1.055592    3.477431    0.400344    0.053707    0.004922    0.727435    
#> 227: 0.608792    1.055257    3.477302    0.399736    0.053743    0.004878    0.726740    
#> 228: 0.607871    1.056500    3.476893    0.400294    0.054300    0.004895    0.726987    
#> 229: 0.605898    1.056135    3.476569    0.398169    0.054235    0.004910    0.726618    
#> 230: 0.606016    1.056082    3.476417    0.398199    0.053949    0.004871    0.726967    
#> 231: 0.606347    1.056172    3.476506    0.399552    0.054006    0.004887    0.726755    
#> 232: 0.604976    1.056279    3.476574    0.398348    0.054093    0.004854    0.727288    
#> 233: 0.604435    1.055516    3.476749    0.396290    0.054203    0.004792    0.727714    
#> 234: 0.604326    1.055640    3.477028    0.396581    0.054566    0.004753    0.727837    
#> 235: 0.604343    1.055371    3.476998    0.396375    0.054543    0.004687    0.728012    
#> 236: 0.604209    1.055771    3.476761    0.395489    0.054634    0.004659    0.728215    
#> 237: 0.605115    1.055459    3.476767    0.396595    0.054974    0.004648    0.728647    
#> 238: 0.605972    1.055749    3.476745    0.398773    0.055371    0.004623    0.728347    
#> 239: 0.605857    1.056113    3.476621    0.400893    0.055439    0.004593    0.728571    
#> 240: 0.607506    1.056012    3.476718    0.404480    0.055657    0.004577    0.728332    
#> 241: 0.608733    1.055590    3.477011    0.405880    0.055470    0.004582    0.728530    
#> 242: 0.609732    1.054894    3.477109    0.406177    0.055462    0.004603    0.729479    
#> 243: 0.609507    1.055390    3.477152    0.406533    0.055355    0.004616    0.730260    
#> 244: 0.608534    1.055339    3.477223    0.404933    0.055150    0.004630    0.730766    
#> 245: 0.608305    1.055204    3.477201    0.405820    0.054914    0.004640    0.731400    
#> 246: 0.608648    1.054854    3.477391    0.404011    0.054553    0.004708    0.731453    
#> 247: 0.609646    1.054720    3.477780    0.402755    0.054267    0.004750    0.731574    
#> 248: 0.609571    1.054593    3.478001    0.401350    0.054099    0.004755    0.731548    
#> 249: 0.609301    1.054631    3.478177    0.401489    0.054015    0.004768    0.731441    
#> 250: 0.609953    1.055080    3.478367    0.401121    0.053930    0.004804    0.731459    
#> 251: 0.610606    1.055191    3.478676    0.399715    0.053847    0.004817    0.731387    
#> 252: 0.610763    1.055209    3.478929    0.399660    0.053816    0.004834    0.731094    
#> 253: 0.610511    1.055282    3.479047    0.398631    0.053891    0.004871    0.730996    
#> 254: 0.610463    1.055869    3.479200    0.397443    0.054140    0.004884    0.730908    
#> 255: 0.609787    1.056170    3.479391    0.395715    0.054315    0.004926    0.730821    
#> 256: 0.610038    1.056024    3.479399    0.394470    0.054365    0.004960    0.730613    
#> 257: 0.610132    1.055571    3.479426    0.394056    0.054287    0.004962    0.730693    
#> 258: 0.610025    1.055238    3.479422    0.393381    0.054085    0.004946    0.730782    
#> 259: 0.610214    1.055132    3.479424    0.393916    0.054059    0.004933    0.730696    
#> 260: 0.609698    1.055111    3.479449    0.394747    0.054099    0.004918    0.730866    
#> 261: 0.610109    1.055012    3.479623    0.394556    0.054199    0.004914    0.731021    
#> 262: 0.610842    1.055226    3.479752    0.396636    0.054557    0.004901    0.730901    
#> 263: 0.610710    1.055532    3.479749    0.397056    0.054645    0.004921    0.730555    
#> 264: 0.610097    1.055854    3.479596    0.397148    0.054572    0.004922    0.730759    
#> 265: 0.609692    1.056195    3.479572    0.396391    0.054680    0.004910    0.731075    
#> 266: 0.610000    1.056443    3.479597    0.396538    0.054712    0.004902    0.731274    
#> 267: 0.610239    1.056696    3.479667    0.396342    0.054915    0.004895    0.731350    
#> 268: 0.610275    1.056824    3.479578    0.395893    0.054913    0.004878    0.731220    
#> 269: 0.610749    1.056756    3.479542    0.395617    0.054847    0.004875    0.731111    
#> 270: 0.610492    1.056994    3.479326    0.395653    0.054906    0.004862    0.730818    
#> 271: 0.610346    1.057131    3.479196    0.395792    0.055033    0.004857    0.730911    
#> 272: 0.610229    1.057204    3.479003    0.396120    0.055200    0.004858    0.731334    
#> 273: 0.609609    1.057611    3.478838    0.396005    0.055280    0.004848    0.731716    
#> 274: 0.609149    1.057761    3.478840    0.395782    0.055309    0.004826    0.731761    
#> 275: 0.609246    1.057995    3.478614    0.395850    0.055378    0.004818    0.731744    
#> 276: 0.608589    1.058261    3.478646    0.395546    0.055564    0.004835    0.731800    
#> 277: 0.608451    1.058550    3.478785    0.395329    0.055689    0.004850    0.731744    
#> 278: 0.608598    1.058433    3.478955    0.395354    0.055853    0.004866    0.731697    
#> 279: 0.608740    1.058177    3.479063    0.395961    0.056160    0.004887    0.731819    
#> 280: 0.609212    1.057875    3.479336    0.396075    0.056092    0.004896    0.731688    
#> 281: 0.609536    1.057703    3.479466    0.395577    0.056113    0.004892    0.731754    
#> 282: 0.609712    1.057678    3.479598    0.394619    0.056378    0.004876    0.731955    
#> 283: 0.610009    1.057462    3.479748    0.394628    0.056378    0.004861    0.732620    
#> 284: 0.609799    1.057199    3.479809    0.394095    0.056241    0.004856    0.732550    
#> 285: 0.609541    1.057425    3.479638    0.394116    0.056211    0.004843    0.732642    
#> 286: 0.609556    1.057720    3.479444    0.393939    0.056208    0.004842    0.732695    
#> 287: 0.609012    1.057981    3.479228    0.394282    0.056227    0.004847    0.732823    
#> 288: 0.609255    1.058088    3.479107    0.394455    0.056210    0.004855    0.732882    
#> 289: 0.609856    1.057976    3.479092    0.395081    0.056078    0.004855    0.732929    
#> 290: 0.609963    1.057932    3.479150    0.395093    0.056026    0.004854    0.733087    
#> 291: 0.610061    1.058028    3.479258    0.394795    0.056021    0.004855    0.732956    
#> 292: 0.609662    1.058211    3.479161    0.394780    0.055970    0.004857    0.732807    
#> 293: 0.609492    1.058408    3.478955    0.394294    0.055977    0.004855    0.732776    
#> 294: 0.608839    1.058749    3.478807    0.393969    0.055961    0.004841    0.732844    
#> 295: 0.608511    1.058699    3.478664    0.394120    0.055997    0.004827    0.732691    
#> 296: 0.608289    1.058736    3.478601    0.394003    0.055900    0.004825    0.732628    
#> 297: 0.608101    1.059081    3.478407    0.394126    0.055807    0.004825    0.732698    
#> 298: 0.607554    1.059206    3.478278    0.394393    0.055665    0.004823    0.732581    
#> 299: 0.607377    1.059543    3.478123    0.394889    0.055602    0.004817    0.732407    
#> 300: 0.607370    1.059569    3.478073    0.394553    0.055565    0.004811    0.732240    
#> 301: 0.607494    1.059378    3.478011    0.394231    0.055643    0.004809    0.732235    
#> 302: 0.607128    1.059471    3.477935    0.394085    0.055664    0.004807    0.732361    
#> 303: 0.606499    1.059602    3.477867    0.393801    0.055693    0.004824    0.732250    
#> 304: 0.606291    1.059770    3.477799    0.393078    0.055688    0.004851    0.732127    
#> 305: 0.606010    1.059419    3.477800    0.392860    0.055739    0.004876    0.732050    
#> 306: 0.605590    1.059349    3.477665    0.392179    0.055670    0.004897    0.732081    
#> 307: 0.605470    1.059240    3.477594    0.392478    0.055735    0.004903    0.732075    
#> 308: 0.604583    1.059245    3.477436    0.392169    0.055869    0.004906    0.732294    
#> 309: 0.604436    1.059258    3.477317    0.391968    0.055922    0.004921    0.732510    
#> 310: 0.604551    1.059329    3.477202    0.392005    0.055979    0.004924    0.732668    
#> 311: 0.604523    1.059400    3.477109    0.391661    0.055976    0.004936    0.732519    
#> 312: 0.603894    1.059543    3.477010    0.391082    0.055968    0.004947    0.732655    
#> 313: 0.603329    1.059592    3.476929    0.390335    0.055900    0.004952    0.732781    
#> 314: 0.602861    1.059600    3.476830    0.390376    0.055861    0.004957    0.732875    
#> 315: 0.602467    1.059705    3.476778    0.389988    0.055820    0.004961    0.732898    
#> 316: 0.602321    1.059771    3.476707    0.389703    0.055767    0.004960    0.733008    
#> 317: 0.602083    1.059776    3.476613    0.389515    0.055711    0.004956    0.733165    
#> 318: 0.602211    1.059830    3.476570    0.389454    0.055569    0.004956    0.733195    
#> 319: 0.602658    1.059826    3.476624    0.389741    0.055527    0.004956    0.733277    
#> 320: 0.602400    1.059773    3.476555    0.389533    0.055495    0.004955    0.733376    
#> 321: 0.602068    1.059710    3.476432    0.389906    0.055438    0.004958    0.733361    
#> 322: 0.601702    1.059834    3.476256    0.389217    0.055410    0.004958    0.733386    
#> 323: 0.601089    1.060058    3.476114    0.388862    0.055353    0.004952    0.733325    
#> 324: 0.600790    1.060229    3.476048    0.388405    0.055420    0.004939    0.733186    
#> 325: 0.600645    1.060429    3.475945    0.388141    0.055466    0.004934    0.733080    
#> 326: 0.600727    1.060287    3.475921    0.388477    0.055538    0.004935    0.732999    
#> 327: 0.600947    1.060285    3.475936    0.388522    0.055649    0.004942    0.732875    
#> 328: 0.601316    1.060386    3.475924    0.388238    0.055652    0.004939    0.732872    
#> 329: 0.601482    1.060589    3.475870    0.388028    0.055673    0.004933    0.732902    
#> 330: 0.601401    1.060688    3.475856    0.387707    0.055645    0.004937    0.732928    
#> 331: 0.601442    1.061000    3.475803    0.387672    0.055698    0.004938    0.732753    
#> 332: 0.601453    1.060995    3.475776    0.387587    0.055642    0.004936    0.732770    
#> 333: 0.601374    1.061026    3.475672    0.388048    0.055618    0.004933    0.732736    
#> 334: 0.601072    1.060844    3.475597    0.387957    0.055596    0.004933    0.732687    
#> 335: 0.600605    1.060855    3.475464    0.388020    0.055563    0.004933    0.732658    
#> 336: 0.600222    1.060980    3.475348    0.387898    0.055577    0.004938    0.732728    
#> 337: 0.600286    1.061251    3.475259    0.388358    0.055562    0.004936    0.732768    
#> 338: 0.600422    1.061462    3.475267    0.388655    0.055560    0.004933    0.732828    
#> 339: 0.600129    1.061695    3.475225    0.388919    0.055544    0.004923    0.732780    
#> 340: 0.600000    1.061709    3.475259    0.388885    0.055603    0.004916    0.732754    
#> 341: 0.600266    1.061676    3.475304    0.388831    0.055641    0.004913    0.732719    
#> 342: 0.599969    1.061602    3.475264    0.389062    0.055665    0.004912    0.732636    
#> 343: 0.599717    1.061723    3.475280    0.388776    0.055725    0.004907    0.732632    
#> 344: 0.599666    1.061798    3.475297    0.388386    0.055754    0.004907    0.732714    
#> 345: 0.599669    1.061721    3.475330    0.388034    0.055754    0.004904    0.732750    
#> 346: 0.599761    1.061699    3.475287    0.387786    0.055712    0.004894    0.732774    
#> 347: 0.599813    1.061529    3.475197    0.388067    0.055694    0.004901    0.732913    
#> 348: 0.599541    1.061585    3.475142    0.388226    0.055767    0.004914    0.732892    
#> 349: 0.599264    1.061459    3.475087    0.388864    0.055784    0.004917    0.732880    
#> 350: 0.599160    1.061322    3.475069    0.388859    0.055763    0.004919    0.732923    
#> 351: 0.599180    1.061472    3.475027    0.389347    0.055697    0.004923    0.732986    
#> 352: 0.598898    1.061606    3.475000    0.389462    0.055721    0.004931    0.733002    
#> 353: 0.599028    1.061696    3.474961    0.388990    0.055681    0.004927    0.732961    
#> 354: 0.598960    1.061695    3.474865    0.389289    0.055574    0.004919    0.732879    
#> 355: 0.599237    1.061573    3.474821    0.389171    0.055566    0.004913    0.733015    
#> 356: 0.599310    1.061405    3.474856    0.389011    0.055590    0.004911    0.733045    
#> 357: 0.599564    1.061440    3.474875    0.388928    0.055560    0.004909    0.733009    
#> 358: 0.599597    1.061332    3.474938    0.388628    0.055520    0.004911    0.732912    
#> 359: 0.599541    1.061451    3.474936    0.388325    0.055549    0.004909    0.732897    
#> 360: 0.599406    1.061564    3.474854    0.387837    0.055531    0.004902    0.732856    
#> 361: 0.599140    1.061657    3.474771    0.387513    0.055527    0.004900    0.732800    
#> 362: 0.599254    1.061682    3.474749    0.387579    0.055471    0.004902    0.732640    
#> 363: 0.599495    1.061691    3.474776    0.387502    0.055436    0.004902    0.732528    
#> 364: 0.599775    1.061655    3.474839    0.387268    0.055410    0.004906    0.732403    
#> 365: 0.599875    1.061586    3.474905    0.386911    0.055301    0.004911    0.732445    
#> 366: 0.600137    1.061566    3.474900    0.386716    0.055253    0.004917    0.732406    
#> 367: 0.600323    1.061612    3.474930    0.386744    0.055316    0.004921    0.732255    
#> 368: 0.600436    1.061660    3.474872    0.387140    0.055294    0.004928    0.732170    
#> 369: 0.600206    1.061630    3.474835    0.387278    0.055306    0.004919    0.732276    
#> 370: 0.600280    1.061683    3.474833    0.387282    0.055280    0.004923    0.732316    
#> 371: 0.600082    1.061847    3.474747    0.387317    0.055296    0.004917    0.732310    
#> 372: 0.599905    1.061980    3.474687    0.387350    0.055337    0.004912    0.732371    
#> 373: 0.599538    1.062096    3.474590    0.387302    0.055329    0.004907    0.732495    
#> 374: 0.599545    1.062121    3.474520    0.387178    0.055338    0.004907    0.732417    
#> 375: 0.599468    1.062189    3.474471    0.386969    0.055379    0.004916    0.732375    
#> 376: 0.599424    1.062166    3.474442    0.386890    0.055425    0.004918    0.732368    
#> 377: 0.599336    1.062183    3.474409    0.386964    0.055467    0.004922    0.732424    
#> 378: 0.599104    1.062163    3.474324    0.386884    0.055423    0.004927    0.732498    
#> 379: 0.598872    1.062071    3.474264    0.386887    0.055426    0.004917    0.732454    
#> 380: 0.598643    1.062074    3.474243    0.386807    0.055402    0.004915    0.732414    
#> 381: 0.598459    1.062022    3.474215    0.387060    0.055372    0.004911    0.732385    
#> 382: 0.598499    1.061967    3.474194    0.387194    0.055323    0.004904    0.732254    
#> 383: 0.598616    1.061955    3.474140    0.387319    0.055250    0.004898    0.732270    
#> 384: 0.598618    1.061908    3.474113    0.387151    0.055212    0.004895    0.732178    
#> 385: 0.598638    1.061950    3.474101    0.386778    0.055161    0.004897    0.732188    
#> 386: 0.598876    1.062075    3.474172    0.386650    0.055061    0.004900    0.732173    
#> 387: 0.599098    1.062156    3.474257    0.386636    0.055012    0.004914    0.732169    
#> 388: 0.599273    1.062160    3.474277    0.386838    0.054976    0.004922    0.732123    
#> 389: 0.599183    1.061986    3.474238    0.386952    0.054965    0.004926    0.732150    
#> 390: 0.599122    1.061914    3.474195    0.387078    0.054934    0.004937    0.732119    
#> 391: 0.599248    1.061826    3.474191    0.386975    0.054916    0.004940    0.732083    
#> 392: 0.599331    1.061768    3.474234    0.386913    0.054901    0.004939    0.732104    
#> 393: 0.599196    1.061703    3.474242    0.386657    0.054886    0.004938    0.732141    
#> 394: 0.599244    1.061692    3.474221    0.386462    0.054847    0.004936    0.732142    
#> 395: 0.599456    1.061829    3.474218    0.386201    0.054823    0.004939    0.732226    
#> 396: 0.599477    1.061898    3.474220    0.386130    0.054793    0.004942    0.732292    
#> 397: 0.599582    1.061904    3.474219    0.386890    0.054745    0.004957    0.732338    
#> 398: 0.599647    1.061957    3.474238    0.387687    0.054720    0.004971    0.732300    
#> 399: 0.599772    1.061903    3.474313    0.388190    0.054680    0.004981    0.732258    
#> 400: 0.599995    1.061882    3.474318    0.388577    0.054668    0.004994    0.732224    
#> 401: 0.599973    1.061882    3.474326    0.388523    0.054659    0.005011    0.732213    
#> 402: 0.600043    1.061891    3.474328    0.388431    0.054577    0.005024    0.732193    
#> 403: 0.600245    1.061963    3.474314    0.388390    0.054564    0.005031    0.732150    
#> 404: 0.600190    1.062078    3.474249    0.388402    0.054510    0.005028    0.732155    
#> 405: 0.600103    1.062123    3.474256    0.388366    0.054530    0.005027    0.732182    
#> 406: 0.600228    1.062168    3.474265    0.388853    0.054583    0.005029    0.732145    
#> 407: 0.600316    1.062162    3.474292    0.388995    0.054638    0.005026    0.732116    
#> 408: 0.600555    1.062247    3.474300    0.389193    0.054683    0.005026    0.732137    
#> 409: 0.600560    1.062186    3.474257    0.389322    0.054722    0.005028    0.732012    
#> 410: 0.600454    1.062284    3.474162    0.389493    0.054768    0.005031    0.732015    
#> 411: 0.600563    1.062122    3.474161    0.389631    0.054830    0.005034    0.732133    
#> 412: 0.600655    1.062116    3.474155    0.389777    0.054846    0.005033    0.732174    
#> 413: 0.600656    1.062137    3.474124    0.389820    0.054890    0.005036    0.732066    
#> 414: 0.600364    1.062071    3.474090    0.389508    0.054911    0.005038    0.731984    
#> 415: 0.600494    1.062031    3.474106    0.389497    0.054854    0.005042    0.731950    
#> 416: 0.600583    1.062130    3.474122    0.389555    0.054835    0.005048    0.731841    
#> 417: 0.600403    1.062182    3.474101    0.389378    0.054845    0.005052    0.731766    
#> 418: 0.600281    1.062222    3.474087    0.389366    0.054894    0.005050    0.731751    
#> 419: 0.600131    1.062279    3.474026    0.389410    0.054952    0.005046    0.731723    
#> 420: 0.600263    1.062257    3.474015    0.389638    0.055010    0.005039    0.731695    
#> 421: 0.600031    1.062251    3.474009    0.389735    0.055030    0.005030    0.731716    
#> 422: 0.600203    1.062107    3.474011    0.389758    0.055033    0.005023    0.731730    
#> 423: 0.600352    1.062050    3.474021    0.389553    0.055078    0.005018    0.731747    
#> 424: 0.600237    1.062020    3.473998    0.389331    0.055082    0.005011    0.731758    
#> 425: 0.600241    1.062095    3.473986    0.389512    0.055033    0.005007    0.731730    
#> 426: 0.600203    1.062192    3.473986    0.389587    0.055009    0.005004    0.731697    
#> 427: 0.600046    1.062173    3.473979    0.389448    0.054978    0.005010    0.731637    
#> 428: 0.599909    1.062250    3.473960    0.389339    0.054949    0.005020    0.731619    
#> 429: 0.599952    1.062268    3.474001    0.389492    0.054926    0.005025    0.731540    
#> 430: 0.599897    1.062328    3.474013    0.389487    0.054912    0.005024    0.731451    
#> 431: 0.599739    1.062350    3.473973    0.389467    0.054903    0.005021    0.731387    
#> 432: 0.599783    1.062480    3.473948    0.389608    0.054895    0.005024    0.731361    
#> 433: 0.599966    1.062667    3.473962    0.389946    0.054906    0.005031    0.731379    
#> 434: 0.600067    1.062593    3.474004    0.390225    0.054920    0.005039    0.731387    
#> 435: 0.600264    1.062577    3.474023    0.390345    0.054881    0.005049    0.731401    
#> 436: 0.600373    1.062549    3.474029    0.390115    0.054848    0.005049    0.731390    
#> 437: 0.600517    1.062533    3.474038    0.389978    0.054813    0.005052    0.731410    
#> 438: 0.600533    1.062669    3.474088    0.390241    0.054798    0.005047    0.731417    
#> 439: 0.600756    1.062745    3.474097    0.390241    0.054841    0.005039    0.731354    
#> 440: 0.600891    1.062707    3.474144    0.390320    0.054830    0.005034    0.731297    
#> 441: 0.600879    1.062652    3.474181    0.390179    0.054860    0.005035    0.731187    
#> 442: 0.600727    1.062728    3.474207    0.389906    0.054923    0.005031    0.731144    
#> 443: 0.600798    1.062872    3.474251    0.389732    0.054965    0.005026    0.731125    
#> 444: 0.600730    1.062925    3.474232    0.389671    0.054982    0.005021    0.731112    
#> 445: 0.600638    1.063008    3.474184    0.389629    0.054973    0.005022    0.731133    
#> 446: 0.600509    1.063092    3.474141    0.389484    0.054952    0.005022    0.731153    
#> 447: 0.600656    1.063143    3.474138    0.389372    0.054954    0.005024    0.731141    
#> 448: 0.600654    1.063112    3.474177    0.389489    0.054971    0.005031    0.731062    
#> 449: 0.600840    1.063092    3.474277    0.389547    0.054959    0.005039    0.731043    
#> 450: 0.600897    1.063099    3.474295    0.389746    0.054903    0.005041    0.731025    
#> 451: 0.601039    1.063119    3.474283    0.389846    0.054856    0.005040    0.730974    
#> 452: 0.600995    1.063118    3.474275    0.389841    0.054799    0.005036    0.730886    
#> 453: 0.600896    1.063094    3.474276    0.389625    0.054766    0.005034    0.730902    
#> 454: 0.600829    1.063133    3.474272    0.389542    0.054758    0.005036    0.730856    
#> 455: 0.600872    1.063119    3.474241    0.389675    0.054739    0.005039    0.730823    
#> 456: 0.600775    1.063076    3.474223    0.389544    0.054719    0.005044    0.730935    
#> 457: 0.600739    1.063061    3.474234    0.389508    0.054689    0.005046    0.730909    
#> 458: 0.600706    1.063107    3.474249    0.389584    0.054653    0.005048    0.730912    
#> 459: 0.600665    1.063131    3.474264    0.389750    0.054624    0.005047    0.730976    
#> 460: 0.600596    1.063095    3.474260    0.389840    0.054600    0.005050    0.731011    
#> 461: 0.600560    1.063109    3.474238    0.390195    0.054567    0.005057    0.730976    
#> 462: 0.600452    1.063168    3.474245    0.390502    0.054562    0.005060    0.730941    
#> 463: 0.600420    1.063203    3.474228    0.390679    0.054543    0.005060    0.730895    
#> 464: 0.600466    1.063169    3.474227    0.390683    0.054539    0.005059    0.730845    
#> 465: 0.600594    1.063073    3.474244    0.390639    0.054529    0.005055    0.730814    
#> 466: 0.600654    1.063065    3.474242    0.390733    0.054517    0.005054    0.730793    
#> 467: 0.600677    1.063022    3.474265    0.390544    0.054475    0.005061    0.730774    
#> 468: 0.600889    1.062981    3.474312    0.390689    0.054468    0.005066    0.730799    
#> 469: 0.600883    1.062994    3.474343    0.390776    0.054440    0.005071    0.730821    
#> 470: 0.600775    1.062952    3.474368    0.390998    0.054426    0.005075    0.730783    
#> 471: 0.600729    1.062885    3.474392    0.391239    0.054399    0.005076    0.730780    
#> 472: 0.600666    1.062846    3.474428    0.391014    0.054367    0.005080    0.730850    
#> 473: 0.600589    1.062804    3.474452    0.390770    0.054335    0.005079    0.730846    
#> 474: 0.600750    1.062639    3.474502    0.390888    0.054390    0.005078    0.730884    
#> 475: 0.600899    1.062660    3.474536    0.390728    0.054364    0.005076    0.730913    
#> 476: 0.600919    1.062668    3.474513    0.390714    0.054376    0.005072    0.731031    
#> 477: 0.600784    1.062656    3.474524    0.390516    0.054391    0.005072    0.731015    
#> 478: 0.600688    1.062591    3.474547    0.390519    0.054332    0.005074    0.731011    
#> 479: 0.600694    1.062665    3.474564    0.390611    0.054332    0.005072    0.731011    
#> 480: 0.600733    1.062686    3.474592    0.390672    0.054341    0.005074    0.731024    
#> 481: 0.600702    1.062640    3.474616    0.390580    0.054326    0.005079    0.731044    
#> 482: 0.600835    1.062659    3.474609    0.390527    0.054299    0.005086    0.731116    
#> 483: 0.600727    1.062599    3.474605    0.390402    0.054288    0.005086    0.731152    
#> 484: 0.600634    1.062490    3.474616    0.390365    0.054276    0.005082    0.731143    
#> 485: 0.600711    1.062444    3.474638    0.390475    0.054295    0.005082    0.731137    
#> 486: 0.600766    1.062465    3.474629    0.390448    0.054259    0.005079    0.731139    
#> 487: 0.600962    1.062502    3.474672    0.390336    0.054253    0.005083    0.731125    
#> 488: 0.601071    1.062425    3.474723    0.390247    0.054234    0.005084    0.731181    
#> 489: 0.601252    1.062285    3.474770    0.390110    0.054219    0.005090    0.731205    
#> 490: 0.601463    1.062260    3.474853    0.390158    0.054206    0.005095    0.731218    
#> 491: 0.601593    1.062263    3.474919    0.390144    0.054230    0.005092    0.731213    
#> 492: 0.601639    1.062294    3.474952    0.390084    0.054211    0.005091    0.731278    
#> 493: 0.601526    1.062322    3.474971    0.390041    0.054186    0.005093    0.731314    
#> 494: 0.601651    1.062378    3.474978    0.390102    0.054167    0.005095    0.731200    
#> 495: 0.601819    1.062396    3.475027    0.390211    0.054134    0.005098    0.731178    
#> 496: 0.602064    1.062470    3.475044    0.390653    0.054103    0.005101    0.731163    
#> 497: 0.601918    1.062472    3.475065    0.390747    0.054068    0.005103    0.731141    
#> 498: 0.601812    1.062430    3.475053    0.390851    0.054065    0.005103    0.731126    
#> 499: 0.601768    1.062477    3.475038    0.391013    0.054105    0.005104    0.731095    
#> 500: 0.601715    1.062438    3.475024    0.390975    0.054080    0.005107    0.731122    
#> [====|====|=>-------------------------------------]  25%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.301619    0.962496    3.434253    0.392401    0.075013    0.016800    1.661222    
#> 002: 0.363117    0.930429    3.463680    0.372781    0.071262    0.015960    1.131201    
#> 003: 0.473335    0.893653    3.447195    0.354142    0.067699    0.015162    0.906400    
#> 004: 0.447472    0.909165    3.445264    0.336435    0.064314    0.014404    0.755168    
#> 005: 0.479463    0.932230    3.427041    0.347589    0.061098    0.013684    0.713096    
#> 006: 0.396791    0.962652    3.408809    0.393909    0.058044    0.015102    0.668861    
#> 007: 0.347021    0.983452    3.393267    0.374213    0.055141    0.015670    0.687756    
#> 008: 0.340405    0.984538    3.379812    0.437924    0.052384    0.019762    0.682362    
#> 009: 0.284259    0.980179    3.364139    0.451933    0.049765    0.020555    0.664420    
#> 010: 0.313268    0.978342    3.379015    0.429336    0.047277    0.020345    0.684445    
#> 011: 0.380938    0.985122    3.378412    0.533628    0.044913    0.022292    0.686855    
#> 012: 0.394369    0.969345    3.377675    0.594358    0.042667    0.023869    0.665251    
#> 013: 0.331786    0.979776    3.368723    0.564640    0.040534    0.023982    0.674082    
#> 014: 0.378567    0.976754    3.377787    0.536408    0.038507    0.024211    0.685882    
#> 015: 0.408243    0.977144    3.404524    0.509588    0.036582    0.027559    0.659797    
#> 016: 0.397742    0.973075    3.405572    0.515822    0.035210    0.027022    0.670890    
#> 017: 0.388969    0.978621    3.398772    0.490031    0.036608    0.026446    0.656953    
#> 018: 0.393168    0.964936    3.384996    0.581619    0.034777    0.025124    0.664406    
#> 019: 0.393329    0.977150    3.391596    0.552538    0.035882    0.023868    0.655504    
#> 020: 0.431794    0.962168    3.403436    0.524911    0.043851    0.025265    0.660375    
#> 021: 0.442033    0.957319    3.409140    0.541174    0.041658    0.027368    0.656732    
#> 022: 0.395945    0.973039    3.402433    0.514115    0.039575    0.026000    0.665615    
#> 023: 0.408602    0.985839    3.393852    0.516493    0.037596    0.024700    0.676625    
#> 024: 0.376377    0.999462    3.390918    0.490668    0.035717    0.023465    0.658096    
#> 025: 0.374593    0.990764    3.376626    0.488704    0.033931    0.028729    0.655806    
#> 026: 0.367233    0.998943    3.369956    0.533252    0.032234    0.027292    0.659918    
#> 027: 0.336030    1.005109    3.364660    0.506589    0.030623    0.025928    0.670235    
#> 028: 0.392106    0.990667    3.386002    0.511146    0.029091    0.027240    0.664517    
#> 029: 0.400878    0.986978    3.385874    0.485588    0.027637    0.025878    0.684252    
#> 030: 0.348881    1.001879    3.378537    0.461309    0.028765    0.026895    0.680954    
#> 031: 0.324997    0.984536    3.385886    0.438244    0.027433    0.025550    0.677454    
#> 032: 0.363384    1.004889    3.386885    0.416331    0.030998    0.024438    0.676981    
#> 033: 0.351336    1.003938    3.375722    0.474780    0.031661    0.023216    0.665226    
#> 034: 0.377361    1.004064    3.386146    0.451041    0.042669    0.024912    0.649185    
#> 035: 0.360854    0.987063    3.382369    0.428489    0.040536    0.023667    0.636355    
#> 036: 0.357649    0.992331    3.378978    0.465424    0.038509    0.022483    0.653990    
#> 037: 0.359198    1.008352    3.373868    0.502633    0.048070    0.021359    0.662498    
#> 038: 0.349871    1.016220    3.364848    0.477501    0.045679    0.020291    0.661809    
#> 039: 0.345615    0.993251    3.374568    0.490593    0.043395    0.023362    0.649215    
#> 040: 0.355495    1.002551    3.384468    0.466063    0.041226    0.026241    0.674448    
#> 041: 0.339636    0.998520    3.368494    0.442760    0.039164    0.024928    0.663246    
#> 042: 0.382434    0.991684    3.378822    0.454022    0.037206    0.023682    0.655189    
#> 043: 0.349301    0.992975    3.364623    0.431321    0.037365    0.022498    0.683561    
#> 044: 0.349589    1.000457    3.372987    0.423097    0.035497    0.021373    0.690197    
#> 045: 0.347731    1.015135    3.368383    0.412638    0.046141    0.020304    0.671363    
#> 046: 0.340603    1.000564    3.374550    0.449428    0.049019    0.019289    0.676875    
#> 047: 0.344162    1.003187    3.362401    0.560130    0.046568    0.018325    0.668673    
#> 048: 0.333320    1.011306    3.363074    0.566622    0.045068    0.017408    0.655998    
#> 049: 0.363009    1.003649    3.370212    0.694788    0.048525    0.017364    0.670474    
#> 050: 0.359686    0.985646    3.379171    0.660048    0.046098    0.019909    0.665576    
#> 051: 0.356018    0.992882    3.363509    0.627046    0.046521    0.020304    0.662789    
#> 052: 0.363646    0.994901    3.381249    0.595694    0.044195    0.019288    0.660663    
#> 053: 0.381507    0.982885    3.381508    0.565909    0.041985    0.021560    0.644411    
#> 054: 0.379762    0.985645    3.395352    0.541092    0.039886    0.020482    0.663309    
#> 055: 0.408530    0.988106    3.395638    0.520630    0.037892    0.019488    0.670626    
#> 056: 0.417783    0.979654    3.396067    0.494598    0.035997    0.020770    0.685975    
#> 057: 0.397665    0.975403    3.400306    0.469868    0.034197    0.021339    0.696721    
#> 058: 0.362748    0.985496    3.383686    0.500445    0.032487    0.021329    0.689539    
#> 059: 0.318314    0.957939    3.382458    0.475423    0.030863    0.020771    0.699334    
#> 060: 0.384162    0.958608    3.397335    0.452899    0.039656    0.020906    0.682924    
#> 061: 0.422225    0.962814    3.407010    0.430254    0.039273    0.024100    0.692071    
#> 062: 0.403534    0.967131    3.402286    0.431592    0.046803    0.022895    0.705958    
#> 063: 0.430420    0.958707    3.401856    0.428449    0.044463    0.021750    0.715273    
#> 064: 0.403217    0.977249    3.407540    0.471575    0.042239    0.020663    0.687817    
#> 065: 0.405507    0.979073    3.380047    0.457433    0.040127    0.022441    0.679821    
#> 066: 0.388095    0.983107    3.380679    0.487646    0.038121    0.021319    0.674196    
#> 067: 0.369972    0.978904    3.385256    0.463264    0.036215    0.026698    0.673481    
#> 068: 0.353607    0.987955    3.387654    0.452014    0.034404    0.025363    0.681797    
#> 069: 0.362333    0.991543    3.381491    0.429413    0.038049    0.024095    0.682992    
#> 070: 0.369757    0.988999    3.400628    0.413723    0.045263    0.022890    0.676790    
#> 071: 0.368723    0.998776    3.388452    0.393036    0.044474    0.021746    0.673729    
#> 072: 0.359402    0.967051    3.389140    0.373385    0.042251    0.020658    0.684699    
#> 073: 0.390257    0.958622    3.414893    0.359268    0.046805    0.019626    0.668182    
#> 074: 0.390929    0.963017    3.404067    0.435661    0.045401    0.018644    0.665839    
#> 075: 0.441780    0.955613    3.420333    0.489196    0.044756    0.017712    0.675958    
#> 076: 0.473437    0.947744    3.430103    0.595219    0.042519    0.017325    0.680716    
#> 077: 0.463971    0.945983    3.410778    0.565458    0.046795    0.020815    0.689359    
#> 078: 0.415965    0.961408    3.404957    0.537185    0.044455    0.021274    0.675014    
#> 079: 0.410387    0.970730    3.397159    0.510326    0.047533    0.020210    0.679835    
#> 080: 0.401876    0.959872    3.398230    0.484809    0.048605    0.019200    0.677663    
#> 081: 0.373900    0.978828    3.393001    0.460569    0.046174    0.018240    0.669212    
#> 082: 0.375803    0.967691    3.397646    0.437541    0.043866    0.019738    0.674785    
#> 083: 0.405410    0.984198    3.404474    0.536391    0.041672    0.020477    0.673072    
#> 084: 0.400827    0.987085    3.386930    0.509572    0.039589    0.020003    0.678490    
#> 085: 0.417343    0.998594    3.391588    0.544468    0.041299    0.022090    0.675180    
#> 086: 0.371535    0.986337    3.386561    0.565340    0.039234    0.023199    0.663588    
#> 087: 0.389348    0.985481    3.384877    0.580111    0.037272    0.024884    0.649799    
#> 088: 0.341970    0.979799    3.384924    0.551106    0.037678    0.026248    0.669777    
#> 089: 0.361779    0.986143    3.376639    0.523550    0.040230    0.028506    0.650399    
#> 090: 0.326919    0.990983    3.366605    0.497373    0.038219    0.030058    0.657956    
#> 091: 0.328379    0.986390    3.353082    0.486410    0.036308    0.029565    0.660684    
#> 092: 0.358428    0.991621    3.360253    0.563730    0.034492    0.029801    0.652331    
#> 093: 0.319503    0.992376    3.363353    0.535544    0.033668    0.028311    0.670797    
#> 094: 0.344286    0.995755    3.375069    0.513539    0.031985    0.028822    0.659314    
#> 095: 0.333461    0.997621    3.381940    0.487862    0.030386    0.031606    0.677000    
#> 096: 0.311576    1.016239    3.361297    0.463469    0.036624    0.030025    0.663391    
#> 097: 0.327827    1.022281    3.351528    0.523552    0.034793    0.029932    0.654456    
#> 098: 0.381849    1.007661    3.381699    0.497374    0.033053    0.029548    0.675966    
#> 099: 0.357571    0.984643    3.366495    0.472505    0.033976    0.028071    0.664536    
#> 100: 0.348938    0.989451    3.381516    0.511064    0.032804    0.029306    0.659029    
#> 101: 0.338628    0.980570    3.361863    0.485510    0.031164    0.029654    0.655700    
#> 102: 0.333477    0.978347    3.356436    0.476278    0.033118    0.028172    0.657741    
#> 103: 0.378703    0.988111    3.364386    0.522125    0.034107    0.026763    0.658710    
#> 104: 0.366007    0.997489    3.384212    0.518148    0.035347    0.025425    0.670834    
#> 105: 0.405960    0.988527    3.389545    0.500045    0.037961    0.024538    0.667572    
#> 106: 0.397619    0.987125    3.394465    0.475043    0.036063    0.023521    0.657360    
#> 107: 0.395347    0.991220    3.387825    0.520387    0.034260    0.022734    0.658912    
#> 108: 0.343170    0.986411    3.388907    0.507665    0.032547    0.023666    0.683171    
#> 109: 0.346294    0.985489    3.387162    0.482281    0.030919    0.025428    0.680184    
#> 110: 0.372244    0.994246    3.362736    0.507841    0.032460    0.024157    0.664205    
#> 111: 0.373991    1.009422    3.367798    0.498586    0.030837    0.022949    0.653821    
#> 112: 0.323239    0.991830    3.352782    0.526011    0.031773    0.021801    0.658638    
#> 113: 0.323069    1.015668    3.364407    0.536116    0.030184    0.025335    0.650883    
#> 114: 0.355132    1.011728    3.362830    0.509311    0.030623    0.024069    0.668321    
#> 115: 0.396300    0.975379    3.364363    0.595880    0.032104    0.022865    0.675323    
#> 116: 0.427441    0.987341    3.369558    0.578988    0.038243    0.025196    0.671361    
#> 117: 0.415073    1.000147    3.388975    0.698844    0.036331    0.025453    0.674176    
#> 118: 0.410458    0.993548    3.386897    0.663902    0.034576    0.024181    0.671650    
#> 119: 0.382824    1.007638    3.388500    0.630707    0.032847    0.022972    0.696429    
#> 120: 0.326172    1.003738    3.373142    0.599172    0.039404    0.023943    0.678646    
#> 121: 0.343838    1.017616    3.359175    0.569213    0.037434    0.022746    0.680670    
#> 122: 0.350381    1.005235    3.372197    0.540752    0.038834    0.021608    0.686312    
#> 123: 0.380869    1.027935    3.366582    0.513715    0.044214    0.020528    0.664661    
#> 124: 0.350463    1.028094    3.370955    0.488029    0.042280    0.020547    0.667009    
#> 125: 0.385976    1.000615    3.374861    0.504410    0.040289    0.023381    0.677920    
#> 126: 0.364000    1.009043    3.363300    0.520912    0.038274    0.022212    0.682688    
#> 127: 0.378058    0.995284    3.367356    0.568528    0.036361    0.027902    0.684939    
#> 128: 0.369858    0.991173    3.376265    0.582708    0.037028    0.026507    0.684538    
#> 129: 0.411151    1.005142    3.382647    0.591185    0.040515    0.026118    0.671108    
#> 130: 0.446150    1.006494    3.385685    0.607249    0.045202    0.024812    0.665871    
#> 131: 0.352383    0.993450    3.375367    0.576887    0.043569    0.023572    0.674256    
#> 132: 0.327810    1.006013    3.366742    0.548042    0.047825    0.022393    0.667097    
#> 133: 0.342844    0.975108    3.369903    0.520640    0.049824    0.021273    0.670846    
#> 134: 0.335809    0.972921    3.372341    0.511325    0.047333    0.024582    0.674368    
#> 135: 0.343287    0.988099    3.373013    0.485759    0.047128    0.023352    0.668081    
#> 136: 0.350153    0.989102    3.374828    0.461471    0.047494    0.022185    0.661171    
#> 137: 0.379327    0.975292    3.388028    0.467615    0.046102    0.024762    0.660160    
#> 138: 0.332364    0.983166    3.373786    0.444234    0.052094    0.023524    0.668078    
#> 139: 0.337948    1.005874    3.386432    0.509029    0.049917    0.022348    0.677464    
#> 140: 0.363113    1.018812    3.391172    0.483578    0.047421    0.023142    0.669333    
#> 141: 0.365242    0.984214    3.397535    0.460806    0.045050    0.025411    0.650120    
#> 142: 0.411081    0.956309    3.411808    0.482655    0.042798    0.024140    0.661298    
#> 143: 0.382707    0.971033    3.400959    0.458523    0.040658    0.026196    0.678761    
#> 144: 0.448633    0.978865    3.415399    0.551359    0.038625    0.027335    0.637710    
#> 145: 0.380940    0.979288    3.385337    0.580839    0.036694    0.025968    0.655026    
#> 146: 0.381661    0.988550    3.382040    0.551797    0.036154    0.024670    0.655820    
#> 147: 0.351980    0.990954    3.366842    0.524207    0.040053    0.023842    0.656799    
#> 148: 0.360499    0.982936    3.385293    0.497997    0.041903    0.022650    0.662611    
#> 149: 0.356830    0.986779    3.374265    0.504406    0.043749    0.021517    0.657559    
#> 150: 0.382431    0.989911    3.369690    0.533685    0.041561    0.020442    0.675413    
#> 151: 0.357811    1.004997    3.377417    0.540468    0.045554    0.021573    0.650203    
#> 152: 0.346010    0.996244    3.383108    0.491101    0.033320    0.024817    0.656206    
#> 153: 0.353970    1.007441    3.383116    0.471510    0.041494    0.018726    0.674174    
#> 154: 0.380948    1.025894    3.378292    0.534478    0.046439    0.016472    0.678777    
#> 155: 0.374443    1.001073    3.373376    0.494974    0.050073    0.013783    0.662357    
#> 156: 0.367734    0.988951    3.378157    0.521978    0.056841    0.017984    0.660490    
#> 157: 0.369469    0.987002    3.382012    0.491790    0.066943    0.017027    0.666402    
#> 158: 0.391736    1.003888    3.376555    0.494595    0.052164    0.016846    0.649313    
#> 159: 0.354888    0.997120    3.388761    0.440053    0.040636    0.020318    0.658484    
#> 160: 0.380477    0.970892    3.389944    0.455924    0.043441    0.019667    0.676751    
#> 161: 0.363079    0.984635    3.384074    0.425386    0.046827    0.019246    0.684883    
#> 162: 0.349120    1.003748    3.376612    0.456053    0.056036    0.018107    0.689312    
#> 163: 0.385871    0.989113    3.373655    0.457196    0.060026    0.020225    0.664663    
#> 164: 0.383583    1.008075    3.380703    0.496614    0.049272    0.023181    0.653475    
#> 165: 0.403889    0.992749    3.378202    0.591777    0.045500    0.024316    0.664521    
#> 166: 0.421396    1.008625    3.387796    0.543149    0.056254    0.025124    0.665343    
#> 167: 0.396967    1.012561    3.388630    0.459146    0.046175    0.019772    0.685833    
#> 168: 0.361112    1.011363    3.382937    0.455469    0.046566    0.016801    0.672900    
#> 169: 0.351137    1.028439    3.369474    0.391578    0.042323    0.017450    0.686283    
#> 170: 0.351213    0.994278    3.382049    0.390989    0.039173    0.018505    0.681585    
#> 171: 0.346227    0.993553    3.381189    0.405533    0.032584    0.015458    0.665089    
#> 172: 0.353812    1.009383    3.374000    0.448029    0.039418    0.014525    0.662441    
#> 173: 0.349527    1.004437    3.371012    0.457699    0.035679    0.020015    0.667960    
#> 174: 0.350036    1.006705    3.376122    0.537316    0.035850    0.014480    0.666425    
#> 175: 0.335015    0.993578    3.369977    0.469820    0.033767    0.016960    0.678664    
#> 176: 0.337346    1.009391    3.368806    0.495964    0.031901    0.017091    0.679490    
#> 177: 0.361290    1.010896    3.374350    0.526124    0.035263    0.019128    0.687439    
#> 178: 0.388364    0.982492    3.379565    0.566875    0.036131    0.020246    0.680783    
#> 179: 0.395771    0.997461    3.371224    0.489774    0.036730    0.020658    0.686845    
#> 180: 0.349224    0.998213    3.375982    0.446364    0.039327    0.019282    0.676034    
#> 181: 0.329760    1.000780    3.359799    0.433975    0.035935    0.019707    0.668686    
#> 182: 0.330957    0.993209    3.357693    0.490658    0.029677    0.024566    0.668881    
#> 183: 0.365810    1.009048    3.367236    0.541546    0.028047    0.023270    0.666542    
#> 184: 0.374654    0.984154    3.379800    0.526602    0.025883    0.023853    0.649393    
#> 185: 0.349728    0.991047    3.372387    0.480214    0.024306    0.023805    0.667914    
#> 186: 0.349413    1.001510    3.380703    0.474041    0.023246    0.023660    0.710126    
#> 187: 0.353670    0.974789    3.391097    0.429730    0.029614    0.024060    0.713786    
#> 188: 0.333508    0.983308    3.388835    0.388419    0.035721    0.026773    0.705326    
#> 189: 0.337085    0.993932    3.386583    0.410982    0.032643    0.025784    0.705439    
#> 190: 0.354942    0.981943    3.389871    0.421604    0.027120    0.025155    0.695417    
#> 191: 0.368032    1.000633    3.384363    0.433792    0.030933    0.024571    0.686340    
#> 192: 0.345131    1.000800    3.388356    0.393997    0.033706    0.024988    0.689746    
#> 193: 0.392721    1.010771    3.388384    0.420416    0.030367    0.025315    0.699226    
#> 194: 0.368878    0.995280    3.380877    0.392020    0.043395    0.022266    0.706718    
#> 195: 0.317168    1.004118    3.365984    0.422637    0.051415    0.016903    0.692260    
#> 196: 0.348923    0.997985    3.375966    0.480176    0.049411    0.020077    0.691476    
#> 197: 0.325507    1.006283    3.389928    0.472164    0.048565    0.021196    0.689580    
#> 198: 0.363598    0.987807    3.379632    0.495479    0.037312    0.023758    0.666355    
#> 199: 0.327965    0.986397    3.378767    0.471891    0.038764    0.024053    0.666962    
#> 200: 0.356050    0.986391    3.381307    0.530477    0.031745    0.024631    0.677915    
#> 201: 0.341673    0.983882    3.383395    0.490061    0.033002    0.027125    0.679029    
#> 202: 0.336562    0.985237    3.379711    0.471620    0.034461    0.025795    0.676447    
#> 203: 0.352889    0.985918    3.378883    0.480700    0.035493    0.024984    0.675835    
#> 204: 0.350589    0.981749    3.380119    0.491846    0.037558    0.025044    0.675868    
#> 205: 0.360819    0.987970    3.379700    0.521784    0.038520    0.024921    0.677001    
#> 206: 0.362390    0.993883    3.379245    0.528195    0.038709    0.024499    0.677172    
#> 207: 0.357096    0.996422    3.376240    0.530650    0.037908    0.023926    0.678106    
#> 208: 0.355133    0.996319    3.375599    0.530355    0.037081    0.023203    0.678722    
#> 209: 0.356762    0.994844    3.375353    0.525118    0.037465    0.022751    0.678028    
#> 210: 0.358573    0.993968    3.375722    0.526933    0.037606    0.022163    0.675789    
#> 211: 0.362424    0.993718    3.376406    0.523278    0.037474    0.022079    0.674175    
#> 212: 0.364461    0.993568    3.376641    0.522114    0.038240    0.022073    0.672792    
#> 213: 0.366293    0.992023    3.376936    0.519979    0.038745    0.022141    0.671991    
#> 214: 0.366564    0.991481    3.377692    0.516196    0.039241    0.021927    0.671612    
#> 215: 0.366974    0.991230    3.378461    0.512645    0.039833    0.021608    0.671528    
#> 216: 0.370637    0.989694    3.379564    0.509312    0.040970    0.021338    0.672417    
#> 217: 0.370805    0.990102    3.380053    0.507134    0.042367    0.021047    0.671921    
#> 218: 0.371583    0.990606    3.380072    0.502569    0.042793    0.020886    0.671455    
#> 219: 0.369962    0.990246    3.379819    0.499853    0.042991    0.020855    0.671432    
#> 220: 0.369758    0.989855    3.380237    0.499279    0.043011    0.020841    0.670937    
#> 221: 0.369692    0.989685    3.380136    0.497775    0.043277    0.020855    0.670881    
#> 222: 0.370306    0.989595    3.380326    0.494828    0.043311    0.021073    0.670859    
#> 223: 0.369985    0.990600    3.379868    0.496156    0.043378    0.021201    0.669875    
#> 224: 0.371346    0.991707    3.379864    0.494763    0.043126    0.021243    0.669312    
#> 225: 0.370024    0.991406    3.379769    0.493495    0.042961    0.021243    0.668658    
#> 226: 0.368966    0.990450    3.379382    0.493315    0.042868    0.021299    0.668562    
#> 227: 0.368729    0.990694    3.379130    0.492308    0.042658    0.021371    0.668326    
#> 228: 0.367184    0.991638    3.378601    0.490434    0.042509    0.021359    0.668478    
#> 229: 0.366071    0.991724    3.378273    0.489844    0.042214    0.021297    0.668147    
#> 230: 0.365741    0.991982    3.378028    0.491888    0.042180    0.021295    0.667483    
#> 231: 0.365590    0.992496    3.378059    0.495563    0.042150    0.021352    0.667095    
#> 232: 0.364526    0.992626    3.378178    0.493639    0.041907    0.021298    0.667104    
#> 233: 0.364486    0.992291    3.378127    0.494837    0.041647    0.021311    0.666941    
#> 234: 0.364147    0.992672    3.377982    0.495462    0.041375    0.021338    0.666772    
#> 235: 0.363384    0.992637    3.377867    0.495359    0.041230    0.021326    0.666825    
#> 236: 0.362965    0.993460    3.377278    0.495820    0.041139    0.021250    0.666920    
#> 237: 0.363907    0.993322    3.377147    0.498282    0.040947    0.021238    0.667638    
#> 238: 0.363896    0.993250    3.377383    0.497822    0.040815    0.021261    0.668352    
#> 239: 0.363420    0.993716    3.377118    0.496951    0.040764    0.021296    0.668971    
#> 240: 0.362969    0.993875    3.376844    0.496483    0.040648    0.021340    0.669061    
#> 241: 0.362903    0.993459    3.376854    0.496445    0.040397    0.021269    0.669585    
#> 242: 0.362562    0.992993    3.376667    0.495961    0.040420    0.021207    0.669887    
#> 243: 0.362642    0.993097    3.376911    0.494312    0.040299    0.021226    0.670707    
#> 244: 0.362672    0.993017    3.377094    0.492464    0.040243    0.021274    0.670709    
#> 245: 0.362986    0.992561    3.377459    0.491593    0.040251    0.021205    0.670956    
#> 246: 0.363359    0.992216    3.377789    0.490831    0.040338    0.021191    0.671081    
#> 247: 0.363240    0.991873    3.377923    0.491058    0.040295    0.021153    0.670897    
#> 248: 0.362623    0.991975    3.377810    0.489614    0.040178    0.021145    0.670847    
#> 249: 0.362416    0.991938    3.377933    0.488749    0.040204    0.021158    0.670571    
#> 250: 0.363161    0.992033    3.378276    0.489456    0.040290    0.021211    0.670528    
#> 251: 0.364288    0.992210    3.378662    0.490272    0.040250    0.021242    0.670425    
#> 252: 0.365269    0.992067    3.379005    0.491903    0.040375    0.021247    0.670532    
#> 253: 0.365306    0.992091    3.379041    0.492822    0.040465    0.021262    0.670543    
#> 254: 0.365467    0.992177    3.379068    0.492134    0.040466    0.021315    0.670257    
#> 255: 0.365442    0.992487    3.379136    0.492174    0.040240    0.021385    0.670033    
#> 256: 0.366273    0.992556    3.379129    0.492847    0.040084    0.021415    0.669879    
#> 257: 0.366246    0.992251    3.379217    0.492057    0.039997    0.021408    0.669478    
#> 258: 0.366091    0.991948    3.379275    0.491814    0.039955    0.021434    0.669262    
#> 259: 0.366422    0.991718    3.379442    0.491716    0.039846    0.021443    0.669124    
#> 260: 0.366798    0.991801    3.379538    0.491824    0.039802    0.021352    0.668945    
#> 261: 0.367150    0.991519    3.379813    0.490842    0.039674    0.021283    0.668905    
#> 262: 0.367000    0.991384    3.379917    0.490296    0.039691    0.021193    0.668946    
#> 263: 0.366086    0.991548    3.379912    0.488789    0.039663    0.021173    0.669359    
#> 264: 0.365193    0.991996    3.379599    0.488098    0.039862    0.021186    0.669505    
#> 265: 0.365158    0.992372    3.379645    0.487612    0.039923    0.021191    0.669762    
#> 266: 0.365517    0.992542    3.379789    0.487665    0.040024    0.021165    0.669821    
#> 267: 0.365469    0.992536    3.380026    0.487076    0.040036    0.021129    0.669845    
#> 268: 0.365428    0.992426    3.380113    0.486246    0.040014    0.021084    0.669590    
#> 269: 0.366428    0.992362    3.380300    0.487026    0.039884    0.021106    0.669342    
#> 270: 0.367206    0.992484    3.380410    0.488163    0.039812    0.021140    0.669118    
#> 271: 0.367342    0.992490    3.380371    0.489674    0.039794    0.021185    0.669032    
#> 272: 0.367496    0.992368    3.380294    0.490970    0.039885    0.021218    0.669313    
#> 273: 0.367317    0.992596    3.380196    0.491647    0.039897    0.021213    0.669352    
#> 274: 0.366730    0.992771    3.380103    0.493125    0.039942    0.021142    0.669324    
#> 275: 0.366620    0.993003    3.379822    0.495448    0.039903    0.021162    0.669250    
#> 276: 0.366296    0.993324    3.379722    0.497195    0.039893    0.021178    0.669139    
#> 277: 0.366314    0.993717    3.379781    0.498132    0.039853    0.021190    0.669079    
#> 278: 0.365777    0.993881    3.379704    0.497703    0.039889    0.021167    0.669014    
#> 279: 0.365109    0.993912    3.379675    0.496044    0.040009    0.021124    0.668985    
#> 280: 0.364987    0.994134    3.379683    0.494543    0.040045    0.021079    0.668891    
#> 281: 0.364789    0.994284    3.379717    0.493859    0.040140    0.021047    0.668745    
#> 282: 0.365029    0.994407    3.379806    0.494096    0.040195    0.021051    0.668719    
#> 283: 0.365409    0.994449    3.379968    0.494701    0.040318    0.021127    0.668793    
#> 284: 0.365664    0.994215    3.380061    0.495343    0.040335    0.021150    0.668595    
#> 285: 0.365944    0.994133    3.379987    0.497046    0.040359    0.021190    0.668448    
#> 286: 0.366130    0.994242    3.379818    0.498341    0.040440    0.021194    0.668594    
#> 287: 0.366251    0.994534    3.379652    0.499493    0.040522    0.021192    0.668565    
#> 288: 0.366386    0.994634    3.379556    0.500277    0.040492    0.021198    0.668524    
#> 289: 0.366980    0.994629    3.379597    0.501731    0.040510    0.021223    0.668495    
#> 290: 0.367109    0.994659    3.379633    0.501884    0.040452    0.021227    0.668676    
#> 291: 0.367328    0.994756    3.379787    0.501116    0.040409    0.021226    0.668522    
#> 292: 0.367438    0.994766    3.379890    0.500251    0.040475    0.021204    0.668448    
#> 293: 0.367648    0.994629    3.379995    0.500326    0.040415    0.021189    0.668337    
#> 294: 0.367800    0.994675    3.380005    0.500104    0.040403    0.021207    0.668224    
#> 295: 0.367866    0.994413    3.380017    0.499810    0.040365    0.021203    0.668143    
#> 296: 0.368033    0.994326    3.380043    0.499111    0.040437    0.021200    0.668169    
#> 297: 0.367979    0.994324    3.380012    0.498816    0.040472    0.021185    0.668153    
#> 298: 0.368063    0.994238    3.380053    0.498829    0.040505    0.021215    0.668023    
#> 299: 0.368310    0.994316    3.380063    0.499085    0.040460    0.021248    0.667933    
#> 300: 0.368536    0.994241    3.380199    0.498600    0.040487    0.021236    0.667783    
#> 301: 0.368763    0.994011    3.380255    0.497482    0.040476    0.021217    0.667786    
#> 302: 0.368706    0.993832    3.380224    0.496848    0.040406    0.021213    0.667660    
#> 303: 0.368617    0.993840    3.380132    0.496818    0.040371    0.021218    0.667439    
#> 304: 0.368473    0.994040    3.379990    0.496808    0.040400    0.021233    0.667354    
#> 305: 0.368247    0.993838    3.379916    0.496881    0.040415    0.021250    0.667166    
#> 306: 0.368022    0.993705    3.379843    0.496938    0.040414    0.021335    0.667008    
#> 307: 0.367629    0.993625    3.379754    0.497272    0.040353    0.021380    0.667098    
#> 308: 0.367429    0.993640    3.379602    0.497429    0.040344    0.021365    0.667193    
#> 309: 0.367370    0.993619    3.379573    0.497538    0.040382    0.021330    0.667419    
#> 310: 0.367548    0.993534    3.379507    0.498096    0.040344    0.021327    0.667563    
#> 311: 0.367458    0.993408    3.379470    0.499157    0.040271    0.021360    0.667531    
#> 312: 0.367423    0.993237    3.379387    0.499390    0.040285    0.021369    0.667527    
#> 313: 0.367231    0.993034    3.379287    0.499838    0.040363    0.021336    0.667686    
#> 314: 0.366732    0.993008    3.379195    0.499102    0.040358    0.021305    0.667689    
#> 315: 0.366212    0.993058    3.379082    0.498032    0.040381    0.021273    0.667626    
#> 316: 0.365963    0.993081    3.378984    0.497811    0.040440    0.021236    0.667585    
#> 317: 0.365867    0.993123    3.379011    0.497150    0.040500    0.021246    0.667658    
#> 318: 0.365718    0.993196    3.378940    0.496731    0.040573    0.021268    0.667547    
#> 319: 0.365628    0.993154    3.378949    0.496706    0.040594    0.021301    0.667609    
#> 320: 0.365316    0.993112    3.378844    0.497265    0.040620    0.021319    0.667606    
#> 321: 0.365125    0.993225    3.378691    0.497754    0.040720    0.021317    0.667672    
#> 322: 0.364846    0.993237    3.378534    0.497420    0.040743    0.021315    0.667826    
#> 323: 0.364428    0.993272    3.378437    0.497117    0.040760    0.021271    0.667829    
#> 324: 0.364118    0.993473    3.378371    0.496269    0.040806    0.021228    0.667820    
#> 325: 0.363862    0.993786    3.378310    0.495592    0.040876    0.021220    0.667753    
#> 326: 0.363849    0.993735    3.378233    0.495546    0.040893    0.021225    0.667725    
#> 327: 0.363978    0.993715    3.378209    0.495674    0.040969    0.021258    0.667643    
#> 328: 0.363958    0.993887    3.378080    0.495864    0.041002    0.021296    0.667583    
#> 329: 0.363999    0.993995    3.377988    0.495926    0.040921    0.021332    0.667402    
#> 330: 0.363789    0.994059    3.377936    0.496010    0.040863    0.021348    0.667319    
#> 331: 0.363915    0.994246    3.377965    0.496035    0.040822    0.021378    0.667285    
#> 332: 0.363941    0.994089    3.378008    0.496079    0.040703    0.021411    0.667355    
#> 333: 0.364271    0.993899    3.378033    0.496321    0.040610    0.021489    0.667334    
#> 334: 0.364315    0.993731    3.377996    0.496146    0.040593    0.021550    0.667207    
#> 335: 0.364304    0.993689    3.377965    0.495788    0.040524    0.021579    0.667231    
#> 336: 0.363921    0.993693    3.377927    0.495300    0.040389    0.021596    0.667350    
#> 337: 0.363580    0.993782    3.377859    0.494818    0.040289    0.021612    0.667397    
#> 338: 0.363372    0.993839    3.377894    0.494126    0.040243    0.021606    0.667394    
#> 339: 0.363121    0.993910    3.377818    0.493740    0.040271    0.021607    0.667269    
#> 340: 0.362923    0.993855    3.377737    0.493745    0.040316    0.021637    0.667231    
#> 341: 0.362723    0.993844    3.377698    0.493518    0.040326    0.021664    0.667154    
#> 342: 0.362399    0.993782    3.377606    0.493492    0.040369    0.021661    0.667054    
#> 343: 0.362300    0.993824    3.377599    0.493463    0.040406    0.021679    0.667029    
#> 344: 0.362395    0.993802    3.377572    0.494165    0.040407    0.021672    0.667000    
#> 345: 0.362571    0.993726    3.377606    0.494492    0.040419    0.021683    0.666964    
#> 346: 0.362645    0.993757    3.377526    0.494539    0.040406    0.021683    0.666902    
#> 347: 0.362437    0.993575    3.377457    0.493832    0.040409    0.021692    0.666937    
#> 348: 0.362214    0.993535    3.377461    0.493671    0.040503    0.021720    0.666964    
#> 349: 0.361999    0.993428    3.377435    0.493618    0.040578    0.021736    0.666882    
#> 350: 0.361794    0.993237    3.377402    0.493132    0.040604    0.021750    0.666835    
#> 351: 0.361669    0.993335    3.377392    0.492827    0.040632    0.021742    0.666887    
#> 352: 0.361673    0.993448    3.377363    0.492357    0.040630    0.021744    0.667044    
#> 353: 0.361777    0.993621    3.377382    0.492078    0.040618    0.021757    0.667054    
#> 354: 0.361789    0.993651    3.377362    0.492057    0.040615    0.021760    0.666951    
#> 355: 0.362031    0.993615    3.377343    0.491769    0.040619    0.021730    0.666945    
#> 356: 0.361943    0.993592    3.377330    0.491544    0.040619    0.021708    0.667068    
#> 357: 0.361834    0.993611    3.377293    0.491273    0.040599    0.021695    0.666945    
#> 358: 0.361859    0.993625    3.377338    0.491043    0.040616    0.021690    0.666791    
#> 359: 0.361991    0.993639    3.377341    0.491215    0.040614    0.021686    0.666699    
#> 360: 0.362020    0.993647    3.377260    0.491438    0.040605    0.021668    0.666597    
#> 361: 0.362214    0.993670    3.377223    0.491749    0.040595    0.021676    0.666465    
#> 362: 0.362403    0.993628    3.377244    0.492089    0.040582    0.021651    0.666354    
#> 363: 0.362672    0.993582    3.377311    0.492601    0.040577    0.021672    0.666196    
#> 364: 0.363004    0.993579    3.377331    0.492715    0.040593    0.021667    0.666161    
#> 365: 0.363249    0.993421    3.377426    0.492934    0.040558    0.021665    0.666343    
#> 366: 0.363627    0.993293    3.377453    0.493437    0.040527    0.021665    0.666408    
#> 367: 0.363568    0.993271    3.377408    0.493127    0.040477    0.021631    0.666436    
#> 368: 0.363587    0.993339    3.377356    0.492944    0.040474    0.021633    0.666380    
#> 369: 0.363427    0.993423    3.377341    0.492860    0.040486    0.021631    0.666389    
#> 370: 0.363256    0.993555    3.377302    0.492230    0.040523    0.021594    0.666442    
#> 371: 0.363046    0.993680    3.377225    0.492176    0.040528    0.021586    0.666436    
#> 372: 0.362675    0.993761    3.377143    0.492114    0.040528    0.021568    0.666431    
#> 373: 0.362268    0.993938    3.377025    0.492191    0.040493    0.021565    0.666488    
#> 374: 0.362082    0.993999    3.376927    0.492274    0.040432    0.021567    0.666504    
#> 375: 0.361888    0.994097    3.376863    0.492223    0.040452    0.021566    0.666490    
#> 376: 0.361689    0.994187    3.376791    0.492313    0.040426    0.021580    0.666435    
#> 377: 0.361591    0.994251    3.376731    0.492344    0.040383    0.021590    0.666371    
#> 378: 0.361531    0.994222    3.376711    0.492508    0.040376    0.021585    0.666401    
#> 379: 0.361296    0.994031    3.376735    0.492596    0.040367    0.021597    0.666390    
#> 380: 0.361237    0.993955    3.376760    0.492424    0.040353    0.021594    0.666369    
#> 381: 0.361024    0.993877    3.376753    0.492709    0.040348    0.021580    0.666440    
#> 382: 0.360959    0.993858    3.376776    0.492616    0.040333    0.021569    0.666402    
#> 383: 0.360847    0.993921    3.376753    0.492706    0.040333    0.021583    0.666281    
#> 384: 0.360603    0.994017    3.376690    0.492541    0.040304    0.021587    0.666299    
#> 385: 0.360479    0.994046    3.376603    0.492430    0.040275    0.021625    0.666288    
#> 386: 0.360412    0.994160    3.376545    0.492700    0.040280    0.021646    0.666294    
#> 387: 0.360638    0.994242    3.376604    0.493299    0.040287    0.021659    0.666283    
#> 388: 0.360831    0.994391    3.376646    0.494069    0.040303    0.021687    0.666217    
#> 389: 0.360845    0.994303    3.376632    0.494554    0.040352    0.021710    0.666241    
#> 390: 0.360714    0.994392    3.376507    0.495061    0.040421    0.021706    0.666237    
#> 391: 0.360606    0.994445    3.376425    0.495059    0.040471    0.021690    0.666216    
#> 392: 0.360319    0.994522    3.376378    0.494572    0.040500    0.021678    0.666229    
#> 393: 0.360110    0.994521    3.376259    0.494451    0.040531    0.021658    0.666264    
#> 394: 0.360050    0.994556    3.376202    0.494507    0.040513    0.021672    0.666236    
#> 395: 0.359981    0.994630    3.376166    0.494626    0.040497    0.021692    0.666163    
#> 396: 0.360032    0.994692    3.376136    0.494848    0.040505    0.021706    0.666103    
#> 397: 0.360093    0.994675    3.376138    0.495142    0.040529    0.021696    0.666065    
#> 398: 0.360256    0.994662    3.376142    0.495721    0.040542    0.021694    0.666062    
#> 399: 0.360167    0.994684    3.376158    0.496037    0.040542    0.021694    0.665960    
#> 400: 0.360156    0.994789    3.376149    0.495984    0.040530    0.021700    0.665930    
#> 401: 0.360169    0.994906    3.376086    0.495935    0.040561    0.021711    0.665969    
#> 402: 0.360192    0.995008    3.376013    0.495905    0.040564    0.021730    0.665969    
#> 403: 0.360062    0.995102    3.375948    0.495879    0.040519    0.021736    0.665974    
#> 404: 0.359893    0.995231    3.375851    0.495771    0.040472    0.021726    0.666001    
#> 405: 0.359808    0.995338    3.375828    0.495843    0.040410    0.021731    0.665997    
#> 406: 0.359687    0.995393    3.375795    0.495543    0.040394    0.021729    0.666019    
#> 407: 0.359703    0.995377    3.375841    0.495313    0.040389    0.021723    0.665985    
#> 408: 0.359918    0.995393    3.375872    0.495223    0.040384    0.021727    0.665986    
#> 409: 0.360013    0.995366    3.375909    0.495074    0.040385    0.021718    0.665988    
#> 410: 0.360100    0.995345    3.375916    0.494576    0.040413    0.021683    0.666097    
#> 411: 0.360227    0.995179    3.375981    0.494475    0.040444    0.021686    0.666122    
#> 412: 0.360154    0.995163    3.375983    0.494426    0.040441    0.021689    0.666162    
#> 413: 0.360069    0.995224    3.375942    0.494546    0.040420    0.021692    0.666151    
#> 414: 0.359834    0.995297    3.375924    0.494264    0.040389    0.021707    0.666220    
#> 415: 0.359766    0.995375    3.375873    0.494034    0.040357    0.021708    0.666234    
#> 416: 0.359645    0.995445    3.375866    0.493861    0.040327    0.021718    0.666266    
#> 417: 0.359686    0.995457    3.375834    0.494130    0.040278    0.021709    0.666276    
#> 418: 0.359797    0.995414    3.375873    0.494310    0.040246    0.021699    0.666211    
#> 419: 0.359844    0.995467    3.375879    0.494262    0.040261    0.021699    0.666154    
#> 420: 0.359976    0.995471    3.375921    0.494208    0.040253    0.021696    0.666121    
#> 421: 0.359877    0.995495    3.375959    0.494389    0.040227    0.021700    0.666071    
#> 422: 0.360034    0.995417    3.376024    0.494553    0.040221    0.021727    0.665979    
#> 423: 0.360242    0.995296    3.376078    0.494484    0.040243    0.021727    0.666027    
#> 424: 0.360269    0.995236    3.376037    0.494515    0.040244    0.021732    0.666079    
#> 425: 0.360206    0.995292    3.375990    0.494357    0.040248    0.021751    0.666139    
#> 426: 0.360152    0.995364    3.375964    0.494261    0.040224    0.021737    0.666100    
#> 427: 0.359977    0.995402    3.375963    0.494043    0.040215    0.021727    0.666124    
#> 428: 0.359855    0.995470    3.375958    0.493616    0.040181    0.021718    0.666155    
#> 429: 0.359751    0.995536    3.375970    0.493308    0.040170    0.021719    0.666147    
#> 430: 0.359704    0.995559    3.375963    0.493275    0.040123    0.021729    0.666187    
#> 431: 0.359698    0.995559    3.375921    0.493225    0.040107    0.021707    0.666158    
#> 432: 0.359567    0.995640    3.375919    0.493165    0.040121    0.021702    0.666124    
#> 433: 0.359552    0.995788    3.375910    0.493066    0.040135    0.021689    0.666172    
#> 434: 0.359474    0.995725    3.375920    0.492981    0.040147    0.021672    0.666151    
#> 435: 0.359421    0.995711    3.375906    0.492810    0.040174    0.021654    0.666108    
#> 436: 0.359418    0.995653    3.375976    0.492595    0.040200    0.021648    0.666090    
#> 437: 0.359468    0.995620    3.375954    0.492555    0.040222    0.021660    0.666072    
#> 438: 0.359512    0.995768    3.375954    0.492651    0.040203    0.021653    0.666071    
#> 439: 0.359453    0.995912    3.375924    0.492313    0.040198    0.021646    0.666127    
#> 440: 0.359372    0.996036    3.375929    0.492080    0.040195    0.021636    0.666187    
#> 441: 0.359333    0.996041    3.375900    0.491755    0.040165    0.021632    0.666215    
#> 442: 0.359193    0.996129    3.375896    0.491552    0.040184    0.021628    0.666200    
#> 443: 0.359100    0.996232    3.375824    0.491496    0.040161    0.021634    0.666160    
#> 444: 0.358915    0.996339    3.375747    0.491377    0.040123    0.021646    0.666078    
#> 445: 0.358642    0.996434    3.375647    0.491167    0.040064    0.021670    0.666073    
#> 446: 0.358434    0.996534    3.375544    0.491380    0.040028    0.021680    0.666040    
#> 447: 0.358196    0.996621    3.375434    0.491270    0.040019    0.021676    0.666072    
#> 448: 0.357999    0.996684    3.375418    0.491361    0.040028    0.021678    0.666028    
#> 449: 0.358002    0.996740    3.375492    0.491257    0.040049    0.021664    0.666076    
#> 450: 0.358029    0.996736    3.375538    0.490912    0.040060    0.021664    0.666112    
#> 451: 0.358096    0.996777    3.375561    0.490864    0.040072    0.021657    0.666191    
#> 452: 0.358171    0.996761    3.375580    0.490812    0.040069    0.021642    0.666151    
#> 453: 0.358197    0.996787    3.375588    0.490920    0.040058    0.021633    0.666089    
#> 454: 0.358213    0.996867    3.375588    0.490918    0.040026    0.021641    0.666027    
#> 455: 0.358349    0.996858    3.375632    0.491073    0.040004    0.021644    0.665959    
#> 456: 0.358427    0.996832    3.375650    0.491325    0.039995    0.021646    0.665960    
#> 457: 0.358371    0.996801    3.375654    0.491257    0.039989    0.021662    0.665886    
#> 458: 0.358430    0.996914    3.375662    0.491293    0.039996    0.021659    0.665848    
#> 459: 0.358071    0.997019    3.375616    0.491041    0.040004    0.021639    0.665904    
#> 460: 0.357763    0.997073    3.375530    0.490853    0.040064    0.021619    0.665889    
#> 461: 0.357545    0.997122    3.375509    0.490697    0.040077    0.021612    0.665861    
#> 462: 0.357425    0.997191    3.375527    0.490725    0.040082    0.021618    0.665831    
#> 463: 0.357435    0.997232    3.375539    0.490854    0.040096    0.021620    0.665788    
#> 464: 0.357454    0.997194    3.375541    0.490866    0.040075    0.021624    0.665764    
#> 465: 0.357564    0.997049    3.375581    0.490863    0.040059    0.021629    0.665728    
#> 466: 0.357638    0.997018    3.375588    0.490856    0.040036    0.021629    0.665737    
#> 467: 0.357546    0.997003    3.375586    0.490948    0.039988    0.021631    0.665700    
#> 468: 0.357746    0.996983    3.375611    0.491307    0.039949    0.021618    0.665770    
#> 469: 0.357699    0.997014    3.375622    0.491159    0.039940    0.021609    0.665872    
#> 470: 0.357661    0.997026    3.375609    0.491246    0.039933    0.021616    0.665922    
#> 471: 0.357635    0.997042    3.375576    0.491335    0.039902    0.021621    0.665985    
#> 472: 0.357522    0.997061    3.375536    0.491446    0.039909    0.021612    0.665982    
#> 473: 0.357354    0.997084    3.375488    0.491371    0.039917    0.021599    0.665930    
#> 474: 0.357189    0.997094    3.375429    0.491390    0.039927    0.021588    0.665957    
#> 475: 0.357243    0.997109    3.375428    0.491339    0.039942    0.021586    0.665919    
#> 476: 0.357249    0.997094    3.375438    0.491493    0.039948    0.021581    0.665942    
#> 477: 0.357228    0.997122    3.375469    0.491212    0.039957    0.021578    0.665924    
#> 478: 0.357159    0.997079    3.375485    0.490846    0.039952    0.021576    0.665943    
#> 479: 0.357233    0.997050    3.375499    0.490633    0.039952    0.021568    0.665959    
#> 480: 0.357273    0.997070    3.375486    0.490458    0.040012    0.021557    0.665974    
#> 481: 0.357258    0.997085    3.375484    0.490279    0.040022    0.021533    0.665975    
#> 482: 0.357235    0.997105    3.375454    0.490016    0.040044    0.021505    0.665999    
#> 483: 0.357134    0.997096    3.375465    0.489859    0.040076    0.021478    0.666008    
#> 484: 0.357125    0.997004    3.375488    0.489677    0.040083    0.021462    0.666021    
#> 485: 0.357122    0.996988    3.375517    0.489501    0.040095    0.021460    0.666026    
#> 486: 0.357178    0.997017    3.375592    0.489523    0.040111    0.021449    0.666036    
#> 487: 0.357278    0.997034    3.375668    0.489289    0.040127    0.021438    0.666068    
#> 488: 0.357240    0.996961    3.375709    0.489185    0.040120    0.021428    0.666055    
#> 489: 0.357375    0.996848    3.375745    0.489156    0.040125    0.021430    0.666104    
#> 490: 0.357561    0.996787    3.375782    0.489295    0.040132    0.021424    0.666091    
#> 491: 0.357738    0.996780    3.375854    0.489314    0.040093    0.021441    0.666058    
#> 492: 0.357854    0.996815    3.375892    0.489130    0.040112    0.021443    0.666095    
#> 493: 0.357834    0.996824    3.375908    0.488963    0.040119    0.021452    0.666033    
#> 494: 0.357970    0.996825    3.375914    0.488913    0.040140    0.021447    0.665949    
#> 495: 0.358116    0.996851    3.375918    0.488837    0.040131    0.021437    0.665895    
#> 496: 0.358288    0.996894    3.375935    0.488883    0.040125    0.021425    0.665875    
#> 497: 0.358193    0.996886    3.375937    0.488557    0.040104    0.021426    0.665943    
#> 498: 0.358301    0.996891    3.375957    0.488622    0.040105    0.021432    0.665885    
#> 499: 0.358281    0.996922    3.375923    0.488542    0.040118    0.021438    0.665858    
#> 500: 0.358227    0.996941    3.375899    0.488489    0.040129    0.021438    0.665827    
#> [====|====|=>-------------------------------------]  25%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.294497    0.962941    3.441558    0.392401    0.074566    0.016800    1.849466    
#> 002: 0.358245    1.042516    3.439993    0.372781    0.070838    0.015960    1.218211    
#> 003: 0.569918    1.001323    3.468951    0.453064    0.067296    0.015162    0.846196    
#> 004: 0.544769    0.990764    3.457846    0.570438    0.063931    0.014404    0.753565    
#> 005: 0.606312    0.993520    3.449519    0.729771    0.060735    0.013684    0.708648    
#> 006: 0.578225    0.987742    3.430862    0.795260    0.057698    0.013000    0.685389    
#> 007: 0.586521    1.003470    3.422951    0.849829    0.055523    0.012350    0.679317    
#> 008: 0.565373    0.993988    3.429589    0.807337    0.057941    0.011732    0.674547    
#> 009: 0.531962    0.973279    3.419052    0.766971    0.055044    0.011146    0.678314    
#> 010: 0.579168    0.976614    3.423232    0.829609    0.052292    0.010588    0.683335    
#> 011: 0.640124    0.985871    3.437597    0.897146    0.049677    0.010059    0.684869    
#> 012: 0.588271    0.970258    3.428464    0.895814    0.047194    0.009556    0.690127    
#> 013: 0.568640    0.972814    3.436084    0.851024    0.044834    0.009083    0.684559    
#> 014: 0.628243    0.968000    3.442506    1.000886    0.042592    0.008629    0.689794    
#> 015: 0.636589    0.970108    3.453034    0.950841    0.044258    0.008197    0.678996    
#> 016: 0.641040    0.972877    3.458755    0.927188    0.052593    0.007788    0.662879    
#> 017: 0.623424    0.966656    3.459275    0.880829    0.052221    0.007398    0.670167    
#> 018: 0.663030    0.958109    3.451527    0.868134    0.049610    0.007028    0.679501    
#> 019: 0.641183    0.969997    3.447872    0.900203    0.048021    0.006677    0.660705    
#> 020: 0.667988    0.947890    3.457391    0.855193    0.045620    0.008161    0.670688    
#> 021: 0.649696    0.936044    3.468104    0.872594    0.043495    0.007753    0.663265    
#> 022: 0.723034    0.932943    3.475957    0.917390    0.041321    0.007365    0.665618    
#> 023: 0.773074    0.951466    3.483053    0.965434    0.053559    0.007859    0.671191    
#> 024: 0.745061    0.974513    3.475592    0.976025    0.052741    0.007517    0.647101    
#> 025: 0.737516    0.962614    3.461822    1.042672    0.050104    0.007141    0.657612    
#> 026: 0.691869    0.983972    3.444098    1.122624    0.047776    0.006784    0.668805    
#> 027: 0.673403    0.985375    3.450645    1.066493    0.054707    0.006706    0.657213    
#> 028: 0.745325    0.981958    3.450850    1.228049    0.059654    0.006370    0.643201    
#> 029: 0.735153    0.976072    3.445415    1.288099    0.056671    0.007172    0.647872    
#> 030: 0.764239    0.983978    3.440926    1.464767    0.053837    0.007178    0.667211    
#> 031: 0.709577    0.966351    3.447316    1.391529    0.051146    0.006819    0.669451    
#> 032: 0.629945    0.972716    3.449144    1.321952    0.052425    0.006478    0.667885    
#> 033: 0.684670    0.975850    3.449274    1.255855    0.054807    0.006154    0.650602    
#> 034: 0.729399    0.996870    3.448633    1.193062    0.052067    0.006912    0.666460    
#> 035: 0.659503    0.970231    3.444576    1.133409    0.049464    0.006567    0.647126    
#> 036: 0.656599    0.964703    3.452518    1.076738    0.049014    0.006238    0.668664    
#> 037: 0.688725    0.983181    3.455355    1.022902    0.048592    0.005926    0.687544    
#> 038: 0.652324    0.980321    3.447828    1.006698    0.053973    0.006142    0.674484    
#> 039: 0.717981    0.994190    3.450161    1.312298    0.053351    0.006058    0.669440    
#> 040: 0.691825    0.981202    3.444009    1.246683    0.050683    0.005755    0.674862    
#> 041: 0.664814    0.959494    3.450113    1.184349    0.056448    0.005468    0.667815    
#> 042: 0.729303    0.977385    3.443318    1.185668    0.053626    0.006136    0.664454    
#> 043: 0.636585    0.995987    3.437668    1.126385    0.050944    0.006145    0.667562    
#> 044: 0.671312    1.002593    3.439487    1.070065    0.048397    0.005838    0.664490    
#> 045: 0.669857    1.003312    3.439422    1.017515    0.051281    0.005546    0.654656    
#> 046: 0.734159    0.969687    3.444325    1.161564    0.048717    0.005269    0.652419    
#> 047: 0.721047    0.990960    3.433491    1.110378    0.052223    0.005351    0.662018    
#> 048: 0.630007    0.986001    3.427579    1.054860    0.052709    0.005272    0.660725    
#> 049: 0.580809    0.966980    3.434708    1.002117    0.050073    0.005046    0.655930    
#> 050: 0.679926    0.947980    3.434217    1.127346    0.055735    0.005430    0.675633    
#> 051: 0.632198    0.960467    3.437053    1.070979    0.058485    0.005975    0.684692    
#> 052: 0.649973    0.981601    3.445713    1.017430    0.060080    0.006114    0.670055    
#> 053: 0.676173    0.967220    3.449089    0.966558    0.057437    0.005808    0.666436    
#> 054: 0.633740    0.967863    3.453462    0.918230    0.055851    0.005518    0.674242    
#> 055: 0.604777    0.977282    3.437205    0.872319    0.061508    0.005242    0.676888    
#> 056: 0.648940    0.986314    3.441359    0.902950    0.062493    0.004980    0.695853    
#> 057: 0.637186    0.982035    3.448879    0.872491    0.064985    0.004731    0.671806    
#> 058: 0.616815    0.989864    3.442416    0.828866    0.061735    0.004858    0.673749    
#> 059: 0.655324    0.947225    3.448969    0.951857    0.058649    0.007130    0.681571    
#> 060: 0.692850    0.937818    3.458060    0.931776    0.061091    0.007776    0.680172    
#> 061: 0.743973    0.968619    3.461974    1.167998    0.058656    0.007387    0.681157    
#> 062: 0.730657    0.986589    3.459778    1.109598    0.071974    0.008270    0.686760    
#> 063: 0.760193    0.958508    3.457979    1.252441    0.068375    0.008781    0.701211    
#> 064: 0.680368    0.948470    3.460031    1.189819    0.064956    0.008616    0.688031    
#> 065: 0.704357    0.967008    3.438922    1.130328    0.063771    0.008185    0.670216    
#> 066: 0.704872    0.980362    3.443075    1.169121    0.060582    0.009384    0.655803    
#> 067: 0.654042    0.982325    3.441860    1.110665    0.057553    0.010036    0.654392    
#> 068: 0.613009    0.979907    3.448971    1.055131    0.056049    0.009534    0.667518    
#> 069: 0.623725    0.974933    3.432542    1.002375    0.053247    0.009057    0.663387    
#> 070: 0.637730    0.977071    3.446852    0.996620    0.063983    0.008605    0.674403    
#> 071: 0.633238    1.018179    3.425590    0.977089    0.070168    0.008174    0.671246    
#> 072: 0.581369    0.992262    3.425314    0.995513    0.066659    0.007766    0.687218    
#> 073: 0.637167    0.976615    3.449511    0.985601    0.063326    0.007377    0.674026    
#> 074: 0.606509    0.963633    3.440709    0.936321    0.060160    0.007008    0.664822    
#> 075: 0.666731    0.957922    3.453738    1.045620    0.057152    0.006658    0.698244    
#> 076: 0.653241    0.978427    3.453723    0.993339    0.054294    0.006325    0.681061    
#> 077: 0.679979    0.954229    3.445716    0.943672    0.051580    0.007780    0.692923    
#> 078: 0.688921    0.966508    3.457147    0.917550    0.049001    0.008873    0.663640    
#> 079: 0.754147    0.988823    3.459834    1.113731    0.050076    0.008430    0.684090    
#> 080: 0.744992    0.978506    3.458827    1.189715    0.050959    0.008008    0.701769    
#> 081: 0.609985    0.985444    3.444684    1.130229    0.048411    0.007608    0.683434    
#> 082: 0.633194    0.977645    3.461424    1.073718    0.051046    0.007227    0.686320    
#> 083: 0.668514    0.968528    3.469722    1.020032    0.048493    0.006866    0.676561    
#> 084: 0.683696    0.968754    3.462607    0.969030    0.058549    0.006604    0.698153    
#> 085: 0.684674    0.962307    3.467068    0.992271    0.070756    0.006593    0.666921    
#> 086: 0.727819    0.963062    3.462996    1.045844    0.067218    0.006263    0.654431    
#> 087: 0.637221    0.957240    3.455296    0.993552    0.063857    0.005950    0.659761    
#> 088: 0.678527    0.942537    3.462564    0.973695    0.060664    0.005653    0.666688    
#> 089: 0.706195    0.972632    3.446426    1.070107    0.057631    0.005370    0.660740    
#> 090: 0.674432    0.971528    3.447164    1.016602    0.065918    0.005101    0.663138    
#> 091: 0.688427    0.954562    3.437149    1.054988    0.062622    0.004846    0.668902    
#> 092: 0.662952    0.980927    3.442850    1.002239    0.059491    0.005636    0.665174    
#> 093: 0.602824    0.974714    3.440374    0.952127    0.056517    0.006090    0.685632    
#> 094: 0.572119    0.989662    3.432670    0.904520    0.053691    0.007014    0.676158    
#> 095: 0.626360    0.990283    3.433192    0.977627    0.051006    0.007921    0.655964    
#> 096: 0.665562    0.990361    3.437920    1.067702    0.053225    0.007867    0.656158    
#> 097: 0.627411    1.016232    3.428629    1.029691    0.050564    0.007473    0.663271    
#> 098: 0.682301    1.012677    3.442335    1.039796    0.052079    0.007591    0.665200    
#> 099: 0.637331    0.971576    3.437398    0.987806    0.057716    0.007212    0.680399    
#> 100: 0.643387    0.981263    3.435135    1.029695    0.064508    0.006851    0.658227    
#> 101: 0.572426    0.981617    3.417735    0.978210    0.061283    0.006509    0.656067    
#> 102: 0.591195    0.977821    3.416046    0.929300    0.058219    0.006183    0.657261    
#> 103: 0.579683    0.997676    3.421472    0.882835    0.055308    0.005874    0.657507    
#> 104: 0.645198    0.983517    3.432410    1.071305    0.052543    0.005580    0.662365    
#> 105: 0.691097    0.980844    3.434025    1.162559    0.055636    0.005301    0.674057    
#> 106: 0.735567    0.976459    3.429378    1.467507    0.055113    0.005319    0.672400    
#> 107: 0.801162    0.992514    3.436813    1.622350    0.056879    0.005895    0.671389    
#> 108: 0.676552    0.994787    3.433859    1.541232    0.062875    0.005628    0.672465    
#> 109: 0.685651    0.988009    3.437559    1.464171    0.062915    0.005346    0.673015    
#> 110: 0.665142    0.992443    3.425730    1.390962    0.059769    0.005079    0.675220    
#> 111: 0.654185    0.994280    3.432662    1.321414    0.056781    0.004825    0.677915    
#> 112: 0.595245    0.968745    3.435470    1.255343    0.062305    0.004584    0.680701    
#> 113: 0.556626    0.999909    3.433997    1.192576    0.059189    0.004550    0.660975    
#> 114: 0.633081    0.981362    3.434054    1.132947    0.056230    0.004323    0.654654    
#> 115: 0.716146    0.958097    3.439021    1.213800    0.053418    0.004107    0.674043    
#> 116: 0.766742    0.992174    3.440214    1.516792    0.053527    0.004069    0.654801    
#> 117: 0.719734    1.000452    3.441467    1.440952    0.050850    0.004698    0.653303    
#> 118: 0.713623    0.988256    3.439795    1.368904    0.048308    0.004463    0.663828    
#> 119: 0.647544    0.998117    3.435746    1.300459    0.051991    0.004240    0.677782    
#> 120: 0.652904    0.981650    3.435744    1.235436    0.049724    0.004028    0.677529    
#> 121: 0.658003    1.014001    3.431145    1.173664    0.057073    0.003879    0.670091    
#> 122: 0.617048    0.990690    3.437569    1.114981    0.057685    0.005099    0.673299    
#> 123: 0.671489    1.008635    3.435634    1.059232    0.060817    0.005457    0.675466    
#> 124: 0.629252    1.016046    3.431840    1.007124    0.057776    0.005443    0.656463    
#> 125: 0.716170    1.010592    3.432650    1.210520    0.054887    0.007290    0.644962    
#> 126: 0.664341    1.017044    3.424860    1.149994    0.052143    0.007166    0.656897    
#> 127: 0.744875    1.003260    3.424013    1.274726    0.049536    0.006808    0.663035    
#> 128: 0.725205    0.992941    3.443900    1.378966    0.052788    0.006468    0.658188    
#> 129: 0.680345    0.993351    3.449801    1.310018    0.050149    0.007114    0.663894    
#> 130: 0.733511    1.003794    3.447640    1.244517    0.049446    0.006759    0.657109    
#> 131: 0.635541    0.973092    3.443009    1.182291    0.052914    0.006421    0.645199    
#> 132: 0.620587    0.977967    3.437333    1.123177    0.050268    0.006100    0.659600    
#> 133: 0.639131    0.966815    3.438005    1.067018    0.047755    0.006028    0.653251    
#> 134: 0.613207    0.952567    3.442457    1.013667    0.045367    0.007306    0.663252    
#> 135: 0.616989    0.969385    3.447409    0.962984    0.048243    0.007097    0.672602    
#> 136: 0.613628    0.960377    3.449647    0.914834    0.057068    0.006742    0.673667    
#> 137: 0.660082    0.960854    3.453497    0.904802    0.054215    0.008416    0.673659    
#> 138: 0.646554    0.980034    3.454237    0.859562    0.051504    0.008022    0.674718    
#> 139: 0.663896    0.985473    3.463436    0.883192    0.055085    0.007621    0.678598    
#> 140: 0.678577    0.970113    3.463191    0.939687    0.064185    0.007240    0.679212    
#> 141: 0.670519    0.960167    3.465555    0.943574    0.061696    0.006878    0.667561    
#> 142: 0.705722    0.926795    3.474466    0.984972    0.058611    0.006534    0.667516    
#> 143: 0.634638    0.951537    3.473819    0.972564    0.055680    0.008988    0.678347    
#> 144: 0.726935    0.970585    3.466960    1.083280    0.052896    0.008539    0.657320    
#> 145: 0.713830    0.991238    3.454283    1.070427    0.050252    0.008112    0.670240    
#> 146: 0.700289    0.978659    3.461767    1.016906    0.047739    0.008766    0.659424    
#> 147: 0.639673    0.966402    3.452173    0.966060    0.045352    0.008984    0.640302    
#> 148: 0.612714    0.969908    3.458308    0.917757    0.046457    0.008535    0.653079    
#> 149: 0.649444    0.961724    3.462221    0.871870    0.044134    0.008108    0.657454    
#> 150: 0.687732    0.957321    3.451075    0.905375    0.041927    0.007702    0.674005    
#> 151: 0.641928    0.981637    3.443687    0.935879    0.039831    0.007317    0.674848    
#> 152: 0.575143    0.979083    3.438720    0.841494    0.038025    0.007214    0.681221    
#> 153: 0.584471    1.019936    3.437111    0.793733    0.058452    0.006049    0.663777    
#> 154: 0.597931    1.016434    3.437741    0.821412    0.055983    0.005428    0.668718    
#> 155: 0.599736    0.986891    3.435554    0.781200    0.053379    0.004499    0.671604    
#> 156: 0.631199    1.002287    3.446103    0.784846    0.052926    0.004848    0.668255    
#> 157: 0.624396    0.995944    3.455987    0.821902    0.063266    0.005606    0.670661    
#> 158: 0.642212    0.992973    3.450779    0.785097    0.051212    0.005922    0.671212    
#> 159: 0.621556    0.989094    3.457892    0.858704    0.053029    0.006007    0.663028    
#> 160: 0.662897    0.967705    3.461347    0.942091    0.048771    0.006333    0.649642    
#> 161: 0.663486    0.969817    3.453691    0.920463    0.046148    0.006058    0.659804    
#> 162: 0.641765    0.989011    3.439205    0.834255    0.045904    0.005825    0.683801    
#> 163: 0.602914    0.978784    3.431820    0.831147    0.048844    0.006318    0.673132    
#> 164: 0.658831    0.980746    3.445054    0.981628    0.047346    0.006254    0.667886    
#> 165: 0.705876    0.957914    3.452773    1.104786    0.047784    0.007876    0.673648    
#> 166: 0.790366    0.995909    3.450931    1.268379    0.053774    0.006800    0.668416    
#> 167: 0.774638    0.999413    3.458139    1.363262    0.050887    0.006401    0.668141    
#> 168: 0.739509    0.986519    3.459842    1.112455    0.063257    0.005909    0.665481    
#> 169: 0.715698    0.999889    3.457250    1.162339    0.063662    0.005005    0.678528    
#> 170: 0.694189    0.964331    3.459455    0.935667    0.062874    0.005199    0.675034    
#> 171: 0.650347    0.964087    3.459987    0.827713    0.052594    0.004174    0.668834    
#> 172: 0.586656    0.972301    3.456647    0.760326    0.048177    0.003975    0.660385    
#> 173: 0.626284    0.995421    3.448907    0.819185    0.060166    0.003844    0.656997    
#> 174: 0.680530    0.966432    3.451711    0.856717    0.055828    0.003961    0.652055    
#> 175: 0.626775    0.983526    3.443194    0.859443    0.065729    0.003049    0.670223    
#> 176: 0.691125    0.985332    3.445923    1.024433    0.050726    0.002581    0.668516    
#> 177: 0.663650    0.968822    3.450754    0.953606    0.046856    0.002001    0.683830    
#> 178: 0.682349    0.966496    3.451926    0.860543    0.050537    0.002534    0.673527    
#> 179: 0.717435    0.990733    3.450400    1.129759    0.052386    0.003143    0.677551    
#> 180: 0.648762    0.996440    3.443971    0.994431    0.045534    0.003518    0.669843    
#> 181: 0.652115    0.978144    3.431052    1.007353    0.052286    0.003828    0.661343    
#> 182: 0.558961    0.986863    3.424384    0.772144    0.058200    0.004450    0.666021    
#> 183: 0.602813    0.989359    3.424230    0.834759    0.054488    0.004552    0.662832    
#> 184: 0.577253    0.979218    3.429010    0.736990    0.050181    0.003841    0.677865    
#> 185: 0.574860    0.999535    3.423137    0.824508    0.051623    0.003202    0.684730    
#> 186: 0.587062    1.018888    3.422987    0.859181    0.057001    0.002831    0.690349    
#> 187: 0.598485    1.017425    3.428657    0.830623    0.050366    0.002082    0.674633    
#> 188: 0.583756    1.014052    3.421042    0.811424    0.054862    0.001699    0.686162    
#> 189: 0.611670    1.016906    3.423580    0.880046    0.057773    0.001353    0.699744    
#> 190: 0.635043    1.004720    3.425547    0.962536    0.046329    0.001202    0.678955    
#> 191: 0.641857    1.003180    3.423344    1.003409    0.048698    0.001352    0.680597    
#> 192: 0.595333    0.985625    3.426499    0.860238    0.056041    0.001719    0.676203    
#> 193: 0.643393    1.012622    3.427530    0.965371    0.055462    0.001560    0.661844    
#> 194: 0.591532    0.990482    3.426790    0.850712    0.077247    0.001887    0.664516    
#> 195: 0.603969    0.992070    3.421223    0.933459    0.085059    0.001992    0.673620    
#> 196: 0.609138    0.990518    3.425440    0.840519    0.079054    0.001595    0.683040    
#> 197: 0.597374    0.991244    3.430454    0.867472    0.074558    0.001507    0.694524    
#> 198: 0.643604    0.984430    3.425977    1.054234    0.061448    0.001796    0.674352    
#> 199: 0.627219    0.988146    3.428819    1.096566    0.057447    0.001859    0.672709    
#> 200: 0.602877    1.004740    3.433297    0.959319    0.066796    0.001554    0.686432    
#> 201: 0.589663    0.996155    3.435110    0.834618    0.061352    0.001780    0.678452    
#> 202: 0.612752    0.996926    3.435206    0.894223    0.065384    0.001830    0.678179    
#> 203: 0.623260    0.996032    3.434890    0.915566    0.065342    0.001898    0.675696    
#> 204: 0.637096    0.991409    3.435991    0.940879    0.065287    0.002005    0.676341    
#> 205: 0.642171    0.994828    3.436503    0.937827    0.064939    0.002079    0.677723    
#> 206: 0.640376    0.996835    3.437816    0.913674    0.064979    0.002064    0.679066    
#> 207: 0.640689    0.996785    3.437808    0.936113    0.064781    0.002062    0.679655    
#> 208: 0.642894    0.994374    3.438413    0.934608    0.064410    0.002099    0.677853    
#> 209: 0.643466    0.990273    3.439209    0.929968    0.064813    0.002122    0.677822    
#> 210: 0.643264    0.987475    3.439893    0.924853    0.064128    0.002158    0.678418    
#> 211: 0.643435    0.987413    3.440979    0.913881    0.063624    0.002167    0.678615    
#> 212: 0.647845    0.985728    3.441588    0.921655    0.063309    0.002213    0.677535    
#> 213: 0.652579    0.983020    3.441894    0.935444    0.063608    0.002286    0.676866    
#> 214: 0.652632    0.981510    3.442007    0.933175    0.062933    0.002365    0.675495    
#> 215: 0.653631    0.980748    3.442123    0.935640    0.062111    0.002434    0.674470    
#> 216: 0.653340    0.979513    3.442558    0.924909    0.062362    0.002484    0.674509    
#> 217: 0.655230    0.979429    3.442756    0.925710    0.062476    0.002563    0.673340    
#> 218: 0.654116    0.980060    3.442542    0.922535    0.062783    0.002643    0.672546    
#> 219: 0.652625    0.980453    3.441591    0.921474    0.062740    0.002704    0.673169    
#> 220: 0.648075    0.980677    3.441218    0.918120    0.062668    0.002747    0.673089    
#> 221: 0.645429    0.980655    3.440588    0.917146    0.062576    0.002767    0.672482    
#> 222: 0.642757    0.980758    3.440087    0.916011    0.062637    0.002803    0.672247    
#> 223: 0.642539    0.980170    3.439700    0.921117    0.062425    0.002828    0.671658    
#> 224: 0.642525    0.980710    3.439628    0.920932    0.062359    0.002833    0.671438    
#> 225: 0.640237    0.979527    3.439449    0.920495    0.062348    0.002826    0.671420    
#> 226: 0.639169    0.978502    3.439157    0.920613    0.062486    0.002802    0.672106    
#> 227: 0.638996    0.978647    3.439166    0.920114    0.062765    0.002796    0.672530    
#> 228: 0.637760    0.979524    3.438715    0.920153    0.062808    0.002797    0.672280    
#> 229: 0.636570    0.979551    3.438450    0.915578    0.062663    0.002780    0.672280    
#> 230: 0.636528    0.979501    3.438373    0.915592    0.062567    0.002754    0.672250    
#> 231: 0.636556    0.979754    3.438454    0.918463    0.062625    0.002742    0.672087    
#> 232: 0.635378    0.979352    3.438778    0.919041    0.062759    0.002735    0.671941    
#> 233: 0.635054    0.978652    3.438955    0.920537    0.062996    0.002741    0.671578    
#> 234: 0.636546    0.978456    3.439210    0.925661    0.063116    0.002745    0.671431    
#> 235: 0.636678    0.978017    3.439208    0.924380    0.062944    0.002737    0.671545    
#> 236: 0.637410    0.978224    3.439118    0.925383    0.062870    0.002725    0.671361    
#> 237: 0.638769    0.978032    3.439065    0.928759    0.062794    0.002717    0.671386    
#> 238: 0.638116    0.978203    3.439077    0.925299    0.062925    0.002727    0.671734    
#> 239: 0.636712    0.978807    3.439075    0.919859    0.062899    0.002723    0.671850    
#> 240: 0.636890    0.978797    3.439117    0.918169    0.062686    0.002715    0.672073    
#> 241: 0.637580    0.978174    3.439273    0.917334    0.062482    0.002716    0.672608    
#> 242: 0.637943    0.977805    3.439349    0.915169    0.062474    0.002727    0.673263    
#> 243: 0.636941    0.977899    3.439307    0.912680    0.062498    0.002749    0.673827    
#> 244: 0.636250    0.977518    3.439341    0.908599    0.062386    0.002780    0.674229    
#> 245: 0.637515    0.977291    3.439463    0.908102    0.062125    0.002781    0.674722    
#> 246: 0.638148    0.977479    3.439602    0.908502    0.062082    0.002783    0.674571    
#> 247: 0.637949    0.977086    3.439834    0.907359    0.062052    0.002794    0.674496    
#> 248: 0.637541    0.977315    3.439921    0.907067    0.061935    0.002790    0.674337    
#> 249: 0.637337    0.977752    3.439938    0.908788    0.061899    0.002782    0.674496    
#> 250: 0.638058    0.978274    3.439940    0.908153    0.062014    0.002792    0.674541    
#> 251: 0.638821    0.979197    3.440081    0.908752    0.062084    0.002789    0.674811    
#> 252: 0.638862    0.979575    3.440264    0.908145    0.062219    0.002770    0.675028    
#> 253: 0.639268    0.980007    3.440418    0.909546    0.062198    0.002777    0.674931    
#> 254: 0.639581    0.980152    3.440558    0.909399    0.062060    0.002782    0.674867    
#> 255: 0.639558    0.980216    3.440749    0.909726    0.061744    0.002800    0.674745    
#> 256: 0.640469    0.979878    3.440753    0.911919    0.061636    0.002807    0.674871    
#> 257: 0.640210    0.979490    3.440752    0.911340    0.061792    0.002817    0.674994    
#> 258: 0.640092    0.979320    3.440744    0.911743    0.061838    0.002805    0.674929    
#> 259: 0.640582    0.979368    3.440728    0.913702    0.061670    0.002806    0.674746    
#> 260: 0.640253    0.979529    3.440800    0.912523    0.061605    0.002810    0.674606    
#> 261: 0.640110    0.979392    3.440917    0.911864    0.061736    0.002818    0.674594    
#> 262: 0.640428    0.979531    3.441036    0.913705    0.061733    0.002835    0.674643    
#> 263: 0.640320    0.979744    3.441009    0.915065    0.061657    0.002856    0.674672    
#> 264: 0.639791    0.979920    3.440960    0.914481    0.061509    0.002868    0.674911    
#> 265: 0.639573    0.980203    3.440944    0.917009    0.061421    0.002873    0.675279    
#> 266: 0.640053    0.980665    3.440970    0.918122    0.061405    0.002871    0.675482    
#> 267: 0.640563    0.980635    3.441056    0.919898    0.061360    0.002881    0.675806    
#> 268: 0.641401    0.980550    3.441093    0.920788    0.061412    0.002876    0.675524    
#> 269: 0.641966    0.980535    3.441108    0.920512    0.061564    0.002872    0.675397    
#> 270: 0.642535    0.980603    3.440967    0.921191    0.061558    0.002873    0.675373    
#> 271: 0.642761    0.980464    3.440892    0.923066    0.061491    0.002869    0.675407    
#> 272: 0.642599    0.980252    3.440784    0.922254    0.061319    0.002873    0.675738    
#> 273: 0.642363    0.980564    3.440768    0.922232    0.061252    0.002883    0.675870    
#> 274: 0.642299    0.980859    3.440808    0.922150    0.061073    0.002888    0.675688    
#> 275: 0.642424    0.981047    3.440800    0.921998    0.060893    0.002907    0.675593    
#> 276: 0.642292    0.981223    3.440886    0.923206    0.060737    0.002940    0.675797    
#> 277: 0.641997    0.981673    3.440996    0.923659    0.060694    0.002965    0.675980    
#> 278: 0.641468    0.981598    3.441034    0.923335    0.060597    0.002989    0.675911    
#> 279: 0.640948    0.981272    3.441063    0.922839    0.060540    0.003011    0.675712    
#> 280: 0.641011    0.981210    3.441137    0.923518    0.060352    0.003032    0.675688    
#> 281: 0.640805    0.981163    3.441253    0.922129    0.060161    0.003044    0.675609    
#> 282: 0.641030    0.981045    3.441372    0.921502    0.059956    0.003068    0.675701    
#> 283: 0.641890    0.980909    3.441545    0.923153    0.059847    0.003089    0.675829    
#> 284: 0.641989    0.980541    3.441657    0.922640    0.059783    0.003102    0.675566    
#> 285: 0.641547    0.980555    3.441580    0.920715    0.059632    0.003116    0.675701    
#> 286: 0.641542    0.980715    3.441517    0.920125    0.059498    0.003129    0.675695    
#> 287: 0.641148    0.980880    3.441388    0.919975    0.059412    0.003138    0.675640    
#> 288: 0.641503    0.981067    3.441279    0.920067    0.059389    0.003138    0.675703    
#> 289: 0.642065    0.981097    3.441224    0.920671    0.059341    0.003140    0.675545    
#> 290: 0.642293    0.981200    3.441154    0.921038    0.059225    0.003143    0.675704    
#> 291: 0.642061    0.981355    3.441130    0.920876    0.059179    0.003150    0.675551    
#> 292: 0.641927    0.981340    3.441079    0.920939    0.059154    0.003151    0.675319    
#> 293: 0.641947    0.981348    3.440983    0.921076    0.059077    0.003167    0.675107    
#> 294: 0.641762    0.981362    3.440879    0.920727    0.059043    0.003184    0.675012    
#> 295: 0.641306    0.981014    3.440832    0.919241    0.059050    0.003188    0.674967    
#> 296: 0.640904    0.980832    3.440881    0.918183    0.058975    0.003202    0.675001    
#> 297: 0.640753    0.980954    3.440832    0.917770    0.058944    0.003211    0.675103    
#> 298: 0.640659    0.980872    3.440814    0.917762    0.058889    0.003216    0.674960    
#> 299: 0.640483    0.980989    3.440789    0.917793    0.058818    0.003215    0.674727    
#> 300: 0.640753    0.980884    3.440809    0.920007    0.058889    0.003210    0.674601    
#> 301: 0.640692    0.980780    3.440741    0.920691    0.058925    0.003209    0.674742    
#> 302: 0.640410    0.980952    3.440685    0.919454    0.058873    0.003209    0.674839    
#> 303: 0.639942    0.981336    3.440649    0.918628    0.058756    0.003214    0.674816    
#> 304: 0.639888    0.981737    3.440569    0.919686    0.058621    0.003224    0.674801    
#> 305: 0.640066    0.981638    3.440584    0.921013    0.058504    0.003231    0.674710    
#> 306: 0.639956    0.981588    3.440533    0.922364    0.058408    0.003230    0.674709    
#> 307: 0.640289    0.981593    3.440432    0.924247    0.058362    0.003220    0.674840    
#> 308: 0.639842    0.981590    3.440334    0.924027    0.058455    0.003215    0.675027    
#> 309: 0.639786    0.981380    3.440264    0.924379    0.058504    0.003212    0.675219    
#> 310: 0.639736    0.981353    3.440149    0.923315    0.058631    0.003207    0.675242    
#> 311: 0.639673    0.981293    3.440116    0.923757    0.058640    0.003206    0.675295    
#> 312: 0.639098    0.981206    3.440061    0.923065    0.058670    0.003201    0.675450    
#> 313: 0.638436    0.981097    3.439999    0.922066    0.058836    0.003197    0.675465    
#> 314: 0.637726    0.981098    3.439913    0.921973    0.058959    0.003189    0.675383    
#> 315: 0.637146    0.981233    3.439842    0.921377    0.059005    0.003189    0.675318    
#> 316: 0.637226    0.981263    3.439763    0.921993    0.058963    0.003184    0.675291    
#> 317: 0.637431    0.981196    3.439737    0.922084    0.058915    0.003172    0.675328    
#> 318: 0.637336    0.981375    3.439628    0.921937    0.058828    0.003169    0.675338    
#> 319: 0.637251    0.981468    3.439581    0.922322    0.058783    0.003169    0.675268    
#> 320: 0.636667    0.981547    3.439454    0.921950    0.058828    0.003173    0.675340    
#> 321: 0.636254    0.981725    3.439291    0.922131    0.058886    0.003167    0.675412    
#> 322: 0.635995    0.981840    3.439126    0.921469    0.058897    0.003159    0.675618    
#> 323: 0.635478    0.981940    3.439020    0.920339    0.058927    0.003155    0.675638    
#> 324: 0.635772    0.982119    3.439030    0.920993    0.058973    0.003152    0.675617    
#> 325: 0.635895    0.982217    3.438998    0.920688    0.058955    0.003154    0.675515    
#> 326: 0.636005    0.982044    3.438972    0.921307    0.059023    0.003158    0.675587    
#> 327: 0.635779    0.981968    3.439022    0.920291    0.059052    0.003158    0.675644    
#> 328: 0.635758    0.981934    3.439069    0.919691    0.058972    0.003157    0.675621    
#> 329: 0.635827    0.982077    3.439105    0.919576    0.058970    0.003163    0.675571    
#> 330: 0.635294    0.982197    3.439120    0.919018    0.058980    0.003164    0.675478    
#> 331: 0.635313    0.982400    3.439111    0.918888    0.058934    0.003167    0.675383    
#> 332: 0.635172    0.982401    3.439077    0.918655    0.058853    0.003173    0.675306    
#> 333: 0.634948    0.982449    3.439001    0.918093    0.058799    0.003178    0.675220    
#> 334: 0.634830    0.982491    3.438906    0.918604    0.058766    0.003179    0.675157    
#> 335: 0.634378    0.982703    3.438768    0.918130    0.058829    0.003182    0.675109    
#> 336: 0.634043    0.982957    3.438643    0.918008    0.058858    0.003184    0.675182    
#> 337: 0.633407    0.983205    3.438513    0.917492    0.058776    0.003183    0.675288    
#> 338: 0.633041    0.983320    3.438445    0.917637    0.058742    0.003178    0.675361    
#> 339: 0.632705    0.983455    3.438401    0.917448    0.058712    0.003177    0.675337    
#> 340: 0.632672    0.983328    3.438416    0.917750    0.058697    0.003177    0.675336    
#> 341: 0.632928    0.983365    3.438413    0.918803    0.058691    0.003182    0.675228    
#> 342: 0.632469    0.983303    3.438360    0.918195    0.058660    0.003186    0.675114    
#> 343: 0.632004    0.983410    3.438338    0.917063    0.058650    0.003184    0.675058    
#> 344: 0.631915    0.983482    3.438341    0.916879    0.058609    0.003183    0.674977    
#> 345: 0.632062    0.983420    3.438382    0.917189    0.058627    0.003186    0.674868    
#> 346: 0.632108    0.983436    3.438361    0.917155    0.058564    0.003186    0.674763    
#> 347: 0.632159    0.983375    3.438275    0.917384    0.058510    0.003192    0.674698    
#> 348: 0.631944    0.983487    3.438210    0.917342    0.058463    0.003201    0.674574    
#> 349: 0.632007    0.983440    3.438162    0.917589    0.058398    0.003208    0.674458    
#> 350: 0.631821    0.983378    3.438100    0.917671    0.058306    0.003217    0.674340    
#> 351: 0.631851    0.983491    3.438026    0.917754    0.058266    0.003220    0.674302    
#> 352: 0.631428    0.983682    3.437994    0.916807    0.058221    0.003227    0.674326    
#> 353: 0.631506    0.983901    3.437937    0.916821    0.058160    0.003236    0.674390    
#> 354: 0.631484    0.984006    3.437815    0.917197    0.058072    0.003246    0.674446    
#> 355: 0.631831    0.983927    3.437775    0.918366    0.057991    0.003254    0.674513    
#> 356: 0.632076    0.983889    3.437786    0.918669    0.058019    0.003258    0.674549    
#> 357: 0.632262    0.983957    3.437783    0.918591    0.057982    0.003267    0.674482    
#> 358: 0.632100    0.983988    3.437831    0.917805    0.058023    0.003268    0.674399    
#> 359: 0.632050    0.984028    3.437864    0.917727    0.058096    0.003274    0.674323    
#> 360: 0.631983    0.983990    3.437830    0.917079    0.058129    0.003279    0.674237    
#> 361: 0.631877    0.983947    3.437782    0.916650    0.058134    0.003281    0.674230    
#> 362: 0.631596    0.983904    3.437745    0.916070    0.058120    0.003281    0.674102    
#> 363: 0.631639    0.983766    3.437751    0.916010    0.058068    0.003289    0.674076    
#> 364: 0.631607    0.983683    3.437748    0.915489    0.058079    0.003292    0.674027    
#> 365: 0.631700    0.983572    3.437768    0.915309    0.058054    0.003290    0.674054    
#> 366: 0.632020    0.983520    3.437792    0.914959    0.058012    0.003288    0.674134    
#> 367: 0.631956    0.983494    3.437838    0.914259    0.057953    0.003283    0.674155    
#> 368: 0.632170    0.983537    3.437809    0.914642    0.057919    0.003285    0.674131    
#> 369: 0.632123    0.983497    3.437801    0.914738    0.057951    0.003281    0.674124    
#> 370: 0.632106    0.983504    3.437768    0.915584    0.058012    0.003275    0.674109    
#> 371: 0.631952    0.983492    3.437690    0.916227    0.058004    0.003272    0.674111    
#> 372: 0.631697    0.983489    3.437635    0.916420    0.057985    0.003271    0.674143    
#> 373: 0.631498    0.983614    3.437557    0.917250    0.058026    0.003266    0.674128    
#> 374: 0.631533    0.983635    3.437449    0.917329    0.058000    0.003268    0.674104    
#> 375: 0.631128    0.983632    3.437389    0.917425    0.058017    0.003266    0.674159    
#> 376: 0.630728    0.983698    3.437325    0.917412    0.057932    0.003268    0.674144    
#> 377: 0.630698    0.983732    3.437302    0.918114    0.057896    0.003272    0.674077    
#> 378: 0.630561    0.983692    3.437269    0.917979    0.057914    0.003280    0.674028    
#> 379: 0.630377    0.983537    3.437299    0.918492    0.057927    0.003286    0.673972    
#> 380: 0.630118    0.983437    3.437296    0.917788    0.057942    0.003291    0.673962    
#> 381: 0.629854    0.983401    3.437304    0.917407    0.057914    0.003289    0.673993    
#> 382: 0.629978    0.983413    3.437271    0.918205    0.057898    0.003289    0.673917    
#> 383: 0.630187    0.983494    3.437274    0.918613    0.057940    0.003287    0.673898    
#> 384: 0.630039    0.983454    3.437304    0.918516    0.057901    0.003287    0.673895    
#> 385: 0.630005    0.983488    3.437329    0.918326    0.057881    0.003288    0.673846    
#> 386: 0.630377    0.983634    3.437358    0.919118    0.057847    0.003285    0.673839    
#> 387: 0.630614    0.983760    3.437381    0.919779    0.057824    0.003289    0.673775    
#> 388: 0.630514    0.983753    3.437379    0.919689    0.057784    0.003286    0.673795    
#> 389: 0.630326    0.983637    3.437342    0.919563    0.057773    0.003289    0.673818    
#> 390: 0.630384    0.983650    3.437327    0.920161    0.057726    0.003290    0.673729    
#> 391: 0.630320    0.983675    3.437297    0.919595    0.057665    0.003290    0.673607    
#> 392: 0.630300    0.983743    3.437340    0.919682    0.057634    0.003291    0.673528    
#> 393: 0.630305    0.983680    3.437355    0.919943    0.057563    0.003295    0.673434    
#> 394: 0.630248    0.983719    3.437336    0.919834    0.057559    0.003298    0.673364    
#> 395: 0.630252    0.983834    3.437327    0.920144    0.057503    0.003301    0.673307    
#> 396: 0.630331    0.983836    3.437323    0.920993    0.057448    0.003304    0.673270    
#> 397: 0.630285    0.983804    3.437342    0.920879    0.057444    0.003302    0.673205    
#> 398: 0.630288    0.983818    3.437348    0.920381    0.057374    0.003308    0.673222    
#> 399: 0.630474    0.983821    3.437416    0.920677    0.057377    0.003314    0.673199    
#> 400: 0.630693    0.983887    3.437498    0.920708    0.057365    0.003315    0.673135    
#> 401: 0.630783    0.983871    3.437563    0.920787    0.057396    0.003319    0.673122    
#> 402: 0.630875    0.983869    3.437608    0.920597    0.057384    0.003325    0.673034    
#> 403: 0.631006    0.983894    3.437641    0.920342    0.057358    0.003330    0.672936    
#> 404: 0.631116    0.983884    3.437634    0.919906    0.057320    0.003332    0.672907    
#> 405: 0.630732    0.983933    3.437644    0.918948    0.057250    0.003335    0.672936    
#> 406: 0.630770    0.983974    3.437655    0.918618    0.057184    0.003338    0.672982    
#> 407: 0.630885    0.984043    3.437703    0.918263    0.057096    0.003342    0.673074    
#> 408: 0.631099    0.984161    3.437729    0.918203    0.057003    0.003341    0.673118    
#> 409: 0.631162    0.984137    3.437717    0.918300    0.056957    0.003340    0.673078    
#> 410: 0.630999    0.984144    3.437675    0.917889    0.056965    0.003338    0.673117    
#> 411: 0.631029    0.983960    3.437658    0.917815    0.056985    0.003339    0.673127    
#> 412: 0.631055    0.984031    3.437643    0.917674    0.056992    0.003340    0.673127    
#> 413: 0.631003    0.984022    3.437605    0.917091    0.056999    0.003340    0.673152    
#> 414: 0.630808    0.984084    3.437595    0.917170    0.056940    0.003338    0.673230    
#> 415: 0.630813    0.984083    3.437585    0.916649    0.056906    0.003339    0.673256    
#> 416: 0.631121    0.984188    3.437588    0.917620    0.056881    0.003339    0.673195    
#> 417: 0.631374    0.984245    3.437571    0.918175    0.056856    0.003335    0.673167    
#> 418: 0.631401    0.984341    3.437572    0.918118    0.056887    0.003332    0.673077    
#> 419: 0.631508    0.984421    3.437567    0.918361    0.056895    0.003332    0.673000    
#> 420: 0.631568    0.984433    3.437563    0.918570    0.056892    0.003331    0.672970    
#> 421: 0.631431    0.984490    3.437558    0.918264    0.056892    0.003328    0.672923    
#> 422: 0.631524    0.984418    3.437561    0.918145    0.056877    0.003325    0.672913    
#> 423: 0.631515    0.984352    3.437567    0.917926    0.056892    0.003323    0.672918    
#> 424: 0.631411    0.984353    3.437556    0.917658    0.056903    0.003322    0.672855    
#> 425: 0.631458    0.984401    3.437546    0.918040    0.056924    0.003323    0.672831    
#> 426: 0.631364    0.984484    3.437546    0.917856    0.056927    0.003319    0.672847    
#> 427: 0.631069    0.984440    3.437526    0.917275    0.056869    0.003317    0.672887    
#> 428: 0.630806    0.984445    3.437520    0.916456    0.056773    0.003316    0.672920    
#> 429: 0.630715    0.984408    3.437555    0.916129    0.056765    0.003319    0.672879    
#> 430: 0.630664    0.984410    3.437567    0.916567    0.056726    0.003322    0.672886    
#> 431: 0.630669    0.984432    3.437528    0.917097    0.056647    0.003323    0.672858    
#> 432: 0.630705    0.984546    3.437523    0.917532    0.056604    0.003327    0.672841    
#> 433: 0.630616    0.984643    3.437523    0.917639    0.056588    0.003326    0.672828    
#> 434: 0.630400    0.984559    3.437515    0.917525    0.056562    0.003326    0.672838    
#> 435: 0.630335    0.984550    3.437487    0.917452    0.056573    0.003327    0.672777    
#> 436: 0.630284    0.984516    3.437463    0.917271    0.056587    0.003329    0.672724    
#> 437: 0.630175    0.984534    3.437420    0.916794    0.056556    0.003333    0.672685    
#> 438: 0.629924    0.984658    3.437387    0.916513    0.056530    0.003331    0.672617    
#> 439: 0.629943    0.984792    3.437347    0.916298    0.056511    0.003328    0.672547    
#> 440: 0.629868    0.984894    3.437330    0.916264    0.056561    0.003323    0.672462    
#> 441: 0.629833    0.984890    3.437325    0.915816    0.056608    0.003322    0.672390    
#> 442: 0.629703    0.984958    3.437337    0.915735    0.056579    0.003320    0.672329    
#> 443: 0.629806    0.985036    3.437354    0.915592    0.056547    0.003321    0.672286    
#> 444: 0.629848    0.985051    3.437348    0.915750    0.056557    0.003323    0.672295    
#> 445: 0.629703    0.985088    3.437316    0.915745    0.056567    0.003323    0.672325    
#> 446: 0.629696    0.985166    3.437278    0.915672    0.056578    0.003322    0.672293    
#> 447: 0.629663    0.985190    3.437274    0.915684    0.056571    0.003326    0.672241    
#> 448: 0.629521    0.985224    3.437271    0.915325    0.056533    0.003334    0.672179    
#> 449: 0.629432    0.985300    3.437324    0.915119    0.056514    0.003338    0.672133    
#> 450: 0.629522    0.985300    3.437357    0.915165    0.056523    0.003340    0.672077    
#> 451: 0.629735    0.985383    3.437365    0.915667    0.056567    0.003338    0.672069    
#> 452: 0.629793    0.985408    3.437337    0.915916    0.056578    0.003338    0.672076    
#> 453: 0.629852    0.985387    3.437326    0.916379    0.056613    0.003335    0.672009    
#> 454: 0.629800    0.985487    3.437308    0.916611    0.056636    0.003332    0.671952    
#> 455: 0.629850    0.985530    3.437279    0.916943    0.056651    0.003332    0.671908    
#> 456: 0.629783    0.985452    3.437268    0.916814    0.056643    0.003332    0.671999    
#> 457: 0.629666    0.985416    3.437263    0.916457    0.056640    0.003332    0.671950    
#> 458: 0.629600    0.985485    3.437303    0.916261    0.056674    0.003334    0.671933    
#> 459: 0.629518    0.985582    3.437346    0.916322    0.056707    0.003337    0.671927    
#> 460: 0.629374    0.985580    3.437367    0.916072    0.056712    0.003340    0.671936    
#> 461: 0.629271    0.985623    3.437400    0.915835    0.056740    0.003339    0.671893    
#> 462: 0.629155    0.985601    3.437427    0.915380    0.056736    0.003341    0.671831    
#> 463: 0.629006    0.985674    3.437408    0.914929    0.056727    0.003345    0.671759    
#> 464: 0.629036    0.985676    3.437403    0.914531    0.056709    0.003347    0.671659    
#> 465: 0.628912    0.985543    3.437381    0.914102    0.056680    0.003343    0.671678    
#> 466: 0.628930    0.985568    3.437349    0.914627    0.056659    0.003339    0.671647    
#> 467: 0.628958    0.985547    3.437326    0.915019    0.056656    0.003337    0.671543    
#> 468: 0.629165    0.985537    3.437319    0.915811    0.056626    0.003338    0.671527    
#> 469: 0.629225    0.985603    3.437328    0.915874    0.056589    0.003341    0.671595    
#> 470: 0.629072    0.985605    3.437321    0.915857    0.056560    0.003347    0.671613    
#> 471: 0.628838    0.985553    3.437306    0.915526    0.056531    0.003349    0.671628    
#> 472: 0.628818    0.985523    3.437315    0.916000    0.056555    0.003349    0.671594    
#> 473: 0.628704    0.985481    3.437292    0.915715    0.056563    0.003351    0.671568    
#> 474: 0.628557    0.985464    3.437286    0.915215    0.056575    0.003353    0.671617    
#> 475: 0.628472    0.985546    3.437273    0.914827    0.056595    0.003358    0.671610    
#> 476: 0.628437    0.985558    3.437256    0.914920    0.056624    0.003358    0.671597    
#> 477: 0.628420    0.985631    3.437260    0.915164    0.056631    0.003356    0.671553    
#> 478: 0.628254    0.985633    3.437279    0.914954    0.056606    0.003353    0.671605    
#> 479: 0.628238    0.985719    3.437287    0.914566    0.056620    0.003351    0.671652    
#> 480: 0.628298    0.985720    3.437328    0.914419    0.056625    0.003355    0.671655    
#> 481: 0.628361    0.985709    3.437347    0.914138    0.056636    0.003356    0.671601    
#> 482: 0.628552    0.985720    3.437366    0.914578    0.056608    0.003356    0.671604    
#> 483: 0.628649    0.985667    3.437389    0.914736    0.056639    0.003357    0.671613    
#> 484: 0.628662    0.985570    3.437419    0.914639    0.056632    0.003356    0.671629    
#> 485: 0.628693    0.985588    3.437449    0.914737    0.056615    0.003357    0.671698    
#> 486: 0.628792    0.985678    3.437445    0.914750    0.056634    0.003357    0.671765    
#> 487: 0.628894    0.985722    3.437455    0.914789    0.056640    0.003358    0.671771    
#> 488: 0.628878    0.985723    3.437456    0.914992    0.056630    0.003359    0.671801    
#> 489: 0.628924    0.985686    3.437458    0.915056    0.056625    0.003358    0.671795    
#> 490: 0.629057    0.985653    3.437484    0.915497    0.056651    0.003360    0.671785    
#> 491: 0.629443    0.985694    3.437514    0.916746    0.056677    0.003361    0.671791    
#> 492: 0.629565    0.985711    3.437508    0.917476    0.056718    0.003362    0.671875    
#> 493: 0.629751    0.985691    3.437511    0.918799    0.056759    0.003365    0.671882    
#> 494: 0.629987    0.985771    3.437506    0.919273    0.056819    0.003366    0.671860    
#> 495: 0.629969    0.985849    3.437507    0.919134    0.056850    0.003366    0.671818    
#> 496: 0.630112    0.985925    3.437486    0.919202    0.056852    0.003362    0.671841    
#> 497: 0.630096    0.985888    3.437498    0.919412    0.056874    0.003362    0.671874    
#> 498: 0.630038    0.985812    3.437489    0.919513    0.056881    0.003361    0.671863    
#> 499: 0.630001    0.985852    3.437443    0.919237    0.056879    0.003361    0.671848    
#> 500: 0.630100    0.985816    3.437424    0.919708    0.056897    0.003361    0.671840    
#> [====|====|=>-------------------------------------]  25%; 0:00:00 [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  [====|====|====|====|====|====|====|====|====|====] 100%; 0:00:00                                                                                  001: 0.285286    0.958822    3.448187    0.392401    0.079823    0.016800    1.579830    
#> 002: 0.314541    1.018406    3.454234    0.372781    0.075832    0.015960    1.048378    
#> 003: 0.360922    0.949250    3.456281    0.354142    0.072040    0.015162    0.834413    
#> 004: 0.379576    0.967649    3.466152    0.336435    0.068438    0.014404    0.758404    
#> 005: 0.411254    0.968364    3.456427    0.319613    0.065016    0.013684    0.714361    
#> 006: 0.357439    0.949718    3.448962    0.303633    0.061765    0.013000    0.660239    
#> 007: 0.329344    0.964889    3.440710    0.288451    0.058677    0.012350    0.655284    
#> 008: 0.324648    0.962089    3.438522    0.274029    0.055743    0.011732    0.672014    
#> 009: 0.298653    0.945928    3.427151    0.260327    0.052956    0.011146    0.669243    
#> 010: 0.303085    0.947256    3.431662    0.247311    0.050308    0.010588    0.673707    
#> 011: 0.355374    0.955066    3.439327    0.234945    0.057974    0.010097    0.675922    
#> 012: 0.364447    0.944742    3.437559    0.238174    0.055075    0.009841    0.670358    
#> 013: 0.333277    0.953641    3.435880    0.233836    0.052321    0.012460    0.657491    
#> 014: 0.363331    0.948767    3.450253    0.222144    0.049705    0.011837    0.678630    
#> 015: 0.401453    0.943149    3.467793    0.212452    0.051208    0.011245    0.659694    
#> 016: 0.342244    0.978319    3.452849    0.233100    0.056591    0.010683    0.666973    
#> 017: 0.345071    0.975418    3.447540    0.239759    0.055307    0.010428    0.657113    
#> 018: 0.305624    0.981658    3.423014    0.233637    0.066586    0.009907    0.653431    
#> 019: 0.281998    0.981117    3.421535    0.221955    0.063257    0.009411    0.652291    
#> 020: 0.323173    0.983032    3.432925    0.210857    0.060094    0.008941    0.665983    
#> 021: 0.343206    0.975519    3.443890    0.200314    0.057090    0.008494    0.666572    
#> 022: 0.328486    0.961821    3.447414    0.190299    0.054235    0.008069    0.649028    
#> 023: 0.398332    0.967438    3.460249    0.214046    0.051523    0.007665    0.686863    
#> 024: 0.411789    0.979924    3.456326    0.205897    0.048947    0.007282    0.673354    
#> 025: 0.367750    0.955652    3.441447    0.216609    0.046500    0.006918    0.663789    
#> 026: 0.345438    0.981670    3.434676    0.209002    0.047194    0.008322    0.653103    
#> 027: 0.345691    0.996023    3.440348    0.212516    0.050055    0.009889    0.667013    
#> 028: 0.356831    0.977622    3.445329    0.201891    0.047967    0.009395    0.653981    
#> 029: 0.367888    0.981847    3.441254    0.223032    0.058114    0.008925    0.668211    
#> 030: 0.378364    0.985507    3.437938    0.211881    0.055208    0.008513    0.679769    
#> 031: 0.350292    0.977676    3.448643    0.201287    0.052448    0.008087    0.668790    
#> 032: 0.323551    0.964177    3.444199    0.191222    0.049825    0.007683    0.655222    
#> 033: 0.319380    0.978261    3.440119    0.181661    0.050360    0.008135    0.650137    
#> 034: 0.336769    0.974733    3.436550    0.191719    0.049676    0.007729    0.657331    
#> 035: 0.312413    0.968477    3.436125    0.182133    0.047192    0.007342    0.671483    
#> 036: 0.328013    0.982790    3.430112    0.188274    0.058233    0.006975    0.659031    
#> 037: 0.314394    0.985162    3.430603    0.184391    0.064414    0.006626    0.658024    
#> 038: 0.346317    0.981743    3.429570    0.195395    0.063713    0.006511    0.660313    
#> 039: 0.340289    0.976867    3.438156    0.185625    0.060528    0.007224    0.666100    
#> 040: 0.301212    0.992163    3.438721    0.176344    0.057501    0.007305    0.685301    
#> 041: 0.310139    0.962309    3.439528    0.167527    0.054626    0.008772    0.667601    
#> 042: 0.357033    0.964776    3.436090    0.196142    0.051895    0.009679    0.648005    
#> 043: 0.304626    0.977594    3.427297    0.186335    0.049300    0.009195    0.656351    
#> 044: 0.343638    0.985223    3.430699    0.203845    0.048762    0.008735    0.654764    
#> 045: 0.324060    0.998717    3.428603    0.235347    0.046324    0.008935    0.641869    
#> 046: 0.309090    0.982713    3.430093    0.223579    0.052204    0.008489    0.650106    
#> 047: 0.333147    0.974123    3.431926    0.217982    0.049594    0.009827    0.651924    
#> 048: 0.357378    0.967758    3.439355    0.212297    0.047114    0.010532    0.671751    
#> 049: 0.316681    0.959756    3.442131    0.201683    0.048099    0.011306    0.673881    
#> 050: 0.383585    0.935968    3.450437    0.239137    0.045694    0.011129    0.658693    
#> 051: 0.359769    0.955710    3.448367    0.228974    0.048542    0.012309    0.659567    
#> 052: 0.345236    0.972450    3.450622    0.217525    0.048404    0.011693    0.641647    
#> 053: 0.344469    0.962717    3.440618    0.206649    0.049128    0.011109    0.651792    
#> 054: 0.332018    0.968256    3.442356    0.204772    0.055440    0.011002    0.671784    
#> 055: 0.350707    0.953138    3.445667    0.209653    0.052668    0.010452    0.682814    
#> 056: 0.368382    0.956860    3.445992    0.214862    0.057063    0.011123    0.681096    
#> 057: 0.343901    0.937169    3.455310    0.215180    0.054210    0.011590    0.696279    
#> 058: 0.365929    0.955488    3.446450    0.212333    0.051500    0.012332    0.671439    
#> 059: 0.374324    0.925655    3.461729    0.201717    0.053733    0.012604    0.674695    
#> 060: 0.373621    0.923196    3.463745    0.191649    0.058418    0.013786    0.673104    
#> 061: 0.362652    0.940935    3.451589    0.182066    0.055583    0.013097    0.674246    
#> 062: 0.359967    0.954377    3.462288    0.173081    0.077476    0.012442    0.677983    
#> 063: 0.391470    0.925109    3.458751    0.194313    0.073603    0.011820    0.678617    
#> 064: 0.379572    0.944880    3.458263    0.188369    0.069922    0.011229    0.677422    
#> 065: 0.383634    0.952905    3.447861    0.178950    0.066426    0.010667    0.677914    
#> 066: 0.345443    1.004914    3.446459    0.184395    0.063105    0.010134    0.657727    
#> 067: 0.330559    0.967588    3.432008    0.175175    0.059950    0.009627    0.678684    
#> 068: 0.303566    0.965443    3.440096    0.177316    0.057630    0.011295    0.691715    
#> 069: 0.286619    0.976423    3.429486    0.193155    0.054749    0.010730    0.692813    
#> 070: 0.334265    0.965410    3.450122    0.183497    0.055437    0.010194    0.697300    
#> 071: 0.351229    0.980422    3.439096    0.174322    0.059695    0.009684    0.715563    
#> 072: 0.302816    0.964072    3.428870    0.165606    0.056710    0.009200    0.694902    
#> 073: 0.369874    0.983347    3.443838    0.238632    0.060531    0.008770    0.665182    
#> 074: 0.342461    0.952210    3.436419    0.226700    0.058141    0.008331    0.661157    
#> 075: 0.338602    0.961679    3.455133    0.215365    0.055234    0.007915    0.655085    
#> 076: 0.361499    0.979735    3.458569    0.204597    0.055326    0.007592    0.650783    
#> 077: 0.374383    0.952086    3.450054    0.194367    0.052559    0.007213    0.662339    
#> 078: 0.369667    0.962264    3.451980    0.209333    0.049931    0.009107    0.635517    
#> 079: 0.374027    0.983493    3.450253    0.251044    0.057214    0.010063    0.648070    
#> 080: 0.364980    0.958655    3.446706    0.249427    0.062017    0.010567    0.653440    
#> 081: 0.366900    0.957306    3.445388    0.236956    0.058916    0.011214    0.644084    
#> 082: 0.328234    0.959484    3.448704    0.225108    0.059036    0.011335    0.666546    
#> 083: 0.338055    0.991089    3.440101    0.250031    0.056085    0.013367    0.661126    
#> 084: 0.343646    1.001426    3.427004    0.237529    0.053280    0.012699    0.665147    
#> 085: 0.308329    0.999429    3.435875    0.225653    0.052378    0.012064    0.664756    
#> 086: 0.314174    0.979978    3.433034    0.214370    0.054556    0.011931    0.656300    
#> 087: 0.298677    0.984808    3.425755    0.217034    0.054699    0.011335    0.652416    
#> 088: 0.252223    0.972971    3.425075    0.239463    0.051964    0.010768    0.671302    
#> 089: 0.303216    0.992282    3.425427    0.227490    0.060088    0.010230    0.669329    
#> 090: 0.284942    0.985328    3.421687    0.216116    0.065930    0.009718    0.675004    
#> 091: 0.319034    0.955999    3.422514    0.210455    0.067425    0.009232    0.667320    
#> 092: 0.314901    0.982410    3.424426    0.199932    0.064054    0.008771    0.678461    
#> 093: 0.303526    0.975624    3.427700    0.189936    0.060851    0.008332    0.666088    
#> 094: 0.312641    0.981189    3.426295    0.180439    0.057808    0.007916    0.672319    
#> 095: 0.328571    0.989262    3.438093    0.178883    0.060216    0.007520    0.665554    
#> 096: 0.279548    0.985573    3.435101    0.192471    0.057206    0.007741    0.647223    
#> 097: 0.364014    0.983440    3.445221    0.204483    0.054346    0.007487    0.657610    
#> 098: 0.342539    0.990798    3.447565    0.200941    0.051628    0.007806    0.681639    
#> 099: 0.327248    0.968627    3.431823    0.232198    0.060189    0.007662    0.651078    
#> 100: 0.307272    0.982329    3.434910    0.220588    0.057180    0.007900    0.644733    
#> 101: 0.283761    0.967340    3.418219    0.209559    0.056511    0.008139    0.651633    
#> 102: 0.277209    0.982350    3.412262    0.199081    0.057158    0.007732    0.653142    
#> 103: 0.308544    0.994976    3.415956    0.196270    0.054300    0.008415    0.663231    
#> 104: 0.301672    0.989874    3.426692    0.237086    0.061008    0.008964    0.668346    
#> 105: 0.309085    0.966679    3.423958    0.225232    0.059330    0.008516    0.664714    
#> 106: 0.337973    0.978436    3.428248    0.213970    0.056363    0.009159    0.657349    
#> 107: 0.342348    0.975248    3.433451    0.230206    0.053545    0.010518    0.649242    
#> 108: 0.361884    0.973991    3.440930    0.218695    0.054306    0.009992    0.674681    
#> 109: 0.338359    0.956992    3.450558    0.207760    0.051590    0.009492    0.690620    
#> 110: 0.340100    0.953996    3.427477    0.202674    0.049011    0.009018    0.663486    
#> 111: 0.332327    0.961839    3.430315    0.199992    0.046560    0.008567    0.647076    
#> 112: 0.306756    0.958248    3.423460    0.215576    0.046706    0.009237    0.663820    
#> 113: 0.316785    0.992692    3.431050    0.204797    0.051336    0.008776    0.648412    
#> 114: 0.318267    0.995384    3.435250    0.252330    0.048769    0.008356    0.657778    
#> 115: 0.318205    0.969550    3.428932    0.239714    0.046331    0.009813    0.662007    
#> 116: 0.308898    0.979808    3.428323    0.227728    0.050125    0.011144    0.663671    
#> 117: 0.296911    0.990110    3.428858    0.216342    0.048104    0.010587    0.659651    
#> 118: 0.360279    0.975290    3.444398    0.234203    0.045698    0.010057    0.656434    
#> 119: 0.369319    0.962507    3.454866    0.222493    0.046666    0.010301    0.654861    
#> 120: 0.334236    0.938160    3.452996    0.221275    0.046946    0.012319    0.652257    
#> 121: 0.346615    0.975337    3.443128    0.210211    0.054535    0.011703    0.667151    
#> 122: 0.345674    0.954073    3.452495    0.199701    0.067004    0.011118    0.666729    
#> 123: 0.364562    0.961143    3.445430    0.203209    0.063654    0.010562    0.667345    
#> 124: 0.358914    0.998090    3.447715    0.205230    0.060471    0.010591    0.646299    
#> 125: 0.368106    0.987237    3.440678    0.231000    0.057448    0.010061    0.649792    
#> 126: 0.344670    0.999566    3.436085    0.219450    0.054575    0.012698    0.662973    
#> 127: 0.354688    0.985102    3.436238    0.218223    0.051846    0.012063    0.663782    
#> 128: 0.324842    0.959254    3.451111    0.207312    0.057970    0.011460    0.647452    
#> 129: 0.360184    0.967663    3.451992    0.196946    0.058850    0.010887    0.663634    
#> 130: 0.389915    0.973915    3.447926    0.187099    0.059762    0.010343    0.650634    
#> 131: 0.373860    0.966519    3.444743    0.177744    0.064433    0.009826    0.648435    
#> 132: 0.319473    0.969094    3.436875    0.168857    0.061211    0.009334    0.680314    
#> 133: 0.338961    0.954264    3.445425    0.160414    0.066178    0.008922    0.690521    
#> 134: 0.362588    0.950160    3.457576    0.157826    0.076770    0.008476    0.670890    
#> 135: 0.337572    0.944556    3.452073    0.162324    0.082750    0.008136    0.674315    
#> 136: 0.350217    0.933879    3.452201    0.154207    0.078613    0.007729    0.676424    
#> 137: 0.370808    0.948376    3.458562    0.146497    0.074682    0.007343    0.686074    
#> 138: 0.351096    0.967528    3.451189    0.143949    0.070948    0.009295    0.667538    
#> 139: 0.362838    0.963738    3.467309    0.148954    0.067401    0.009925    0.679270    
#> 140: 0.366489    0.959749    3.467508    0.159847    0.066714    0.009429    0.678048    
#> 141: 0.349737    0.932719    3.457612    0.180979    0.063379    0.008958    0.652959    
#> 142: 0.383530    0.927245    3.468749    0.171930    0.061660    0.008510    0.665751    
#> 143: 0.369200    0.960982    3.458689    0.176354    0.058577    0.009621    0.673824    
#> 144: 0.393781    0.944918    3.475294    0.205976    0.055648    0.009140    0.656887    
#> 145: 0.365189    0.957381    3.454092    0.195677    0.052866    0.008683    0.651808    
#> 146: 0.358208    0.968168    3.454499    0.203886    0.050223    0.008249    0.657573    
#> 147: 0.322817    0.951109    3.438065    0.193692    0.047712    0.007836    0.651782    
#> 148: 0.322870    0.970722    3.446144    0.184127    0.050271    0.007444    0.655681    
#> 149: 0.319450    0.971610    3.433152    0.193094    0.058472    0.007072    0.684250    
#> 150: 0.327912    0.970845    3.426710    0.183439    0.064529    0.006719    0.688841    
#> 151: 0.289679    0.996872    3.430280    0.211224    0.065010    0.006679    0.674512    
#> 152: 0.303218    0.973756    3.437265    0.179357    0.053587    0.006225    0.682909    
#> 153: 0.334154    0.977894    3.443625    0.173033    0.051720    0.008349    0.682573    
#> 154: 0.355968    0.979191    3.443185    0.147451    0.047778    0.007955    0.695934    
#> 155: 0.339171    0.969743    3.445245    0.129976    0.063485    0.008320    0.669621    
#> 156: 0.353457    0.964968    3.456821    0.131096    0.067455    0.007580    0.695804    
#> 157: 0.377078    0.951564    3.459624    0.174161    0.072200    0.007372    0.694056    
#> 158: 0.386603    0.977963    3.457908    0.160328    0.066962    0.009168    0.693747    
#> 159: 0.323827    0.979815    3.450470    0.169427    0.069448    0.007631    0.672805    
#> 160: 0.363620    0.934224    3.456628    0.226953    0.066113    0.008048    0.663020    
#> 161: 0.360148    0.957585    3.448808    0.192474    0.057596    0.008583    0.672437    
#> 162: 0.328202    0.980148    3.434492    0.161001    0.063362    0.007404    0.683798    
#> 163: 0.334282    0.964875    3.432700    0.156840    0.078047    0.006771    0.660639    
#> 164: 0.327598    0.974348    3.439211    0.172003    0.067170    0.008829    0.670652    
#> 165: 0.357742    0.951905    3.451973    0.165285    0.058451    0.012242    0.657399    
#> 166: 0.367333    0.970684    3.447176    0.173315    0.054691    0.011672    0.658778    
#> 167: 0.359250    0.986879    3.449546    0.184210    0.050356    0.011286    0.660484    
#> 168: 0.358918    0.982374    3.440112    0.202045    0.055824    0.009484    0.657586    
#> 169: 0.347191    0.999378    3.440709    0.163757    0.054001    0.009571    0.677853    
#> 170: 0.331833    0.975863    3.442601    0.142066    0.058335    0.006966    0.657609    
#> 171: 0.325313    0.956278    3.448056    0.155173    0.044962    0.008497    0.642815    
#> 172: 0.316867    0.979252    3.438103    0.161096    0.049985    0.008389    0.652886    
#> 173: 0.284374    1.011799    3.416392    0.205888    0.061378    0.010174    0.655283    
#> 174: 0.287930    0.980859    3.427295    0.212103    0.044714    0.012473    0.643080    
#> 175: 0.288616    0.996033    3.414329    0.191500    0.049465    0.013604    0.661192    
#> 176: 0.297285    1.001028    3.431382    0.195737    0.041393    0.011359    0.660755    
#> 177: 0.315773    1.000389    3.428872    0.213615    0.039638    0.011901    0.654153    
#> 178: 0.323641    0.977459    3.437279    0.172959    0.037672    0.010487    0.653661    
#> 179: 0.377724    0.990422    3.444520    0.159820    0.046195    0.010945    0.670332    
#> 180: 0.355175    0.988022    3.447340    0.158111    0.042300    0.008793    0.673151    
#> 181: 0.330020    0.954766    3.429811    0.173114    0.044108    0.009473    0.660356    
#> 182: 0.285079    0.972770    3.422369    0.169816    0.038155    0.010981    0.665319    
#> 183: 0.305837    0.983983    3.427196    0.170922    0.038843    0.011641    0.646261    
#> 184: 0.314816    0.975443    3.436739    0.168537    0.045249    0.009542    0.685596    
#> 185: 0.313164    0.995771    3.427491    0.171861    0.048066    0.007942    0.664025    
#> 186: 0.320713    1.002326    3.435874    0.180456    0.047587    0.010527    0.653248    
#> 187: 0.328782    0.986257    3.437474    0.197198    0.054891    0.010639    0.653054    
#> 188: 0.289966    0.989655    3.432594    0.164863    0.057796    0.008382    0.674456    
#> 189: 0.322810    1.006741    3.435011    0.170425    0.049937    0.008568    0.675471    
#> 190: 0.297363    0.993921    3.430503    0.147908    0.049884    0.009147    0.667511    
#> 191: 0.301714    0.994560    3.425178    0.188338    0.052985    0.007400    0.664471    
#> 192: 0.279312    1.001805    3.420352    0.212703    0.043594    0.008916    0.660538    
#> 193: 0.322611    1.003714    3.426680    0.230244    0.044916    0.007517    0.645044    
#> 194: 0.302727    0.974428    3.428589    0.199603    0.055814    0.009176    0.667265    
#> 195: 0.316113    0.988470    3.429067    0.224079    0.051996    0.011667    0.645292    
#> 196: 0.328719    0.985754    3.432146    0.237953    0.044034    0.012022    0.647990    
#> 197: 0.301029    0.987275    3.443378    0.212726    0.048842    0.011573    0.668459    
#> 198: 0.328956    0.983282    3.435681    0.209633    0.043288    0.011125    0.658245    
#> 199: 0.330014    0.966894    3.440130    0.206497    0.049352    0.009285    0.641266    
#> 200: 0.336358    0.963175    3.436529    0.245096    0.053494    0.012018    0.653898    
#> 201: 0.346556    0.966351    3.442605    0.236815    0.051649    0.012859    0.653228    
#> 202: 0.346251    0.969536    3.441742    0.237389    0.050155    0.012608    0.650754    
#> 203: 0.354262    0.969118    3.440898    0.235380    0.047285    0.012451    0.652661    
#> 204: 0.351623    0.965647    3.443505    0.230024    0.045479    0.012072    0.657832    
#> 205: 0.349485    0.969050    3.443664    0.228482    0.043750    0.011758    0.662050    
#> 206: 0.347711    0.971112    3.443571    0.227175    0.044833    0.011485    0.662035    
#> 207: 0.345231    0.972881    3.442217    0.225175    0.045804    0.011276    0.660242    
#> 208: 0.341347    0.974077    3.441541    0.220630    0.046501    0.011029    0.658324    
#> 209: 0.343142    0.971582    3.441557    0.219244    0.047095    0.010980    0.657757    
#> 210: 0.343942    0.969990    3.441419    0.217634    0.047556    0.010901    0.657479    
#> 211: 0.348011    0.971078    3.442519    0.217644    0.047781    0.010888    0.658018    
#> 212: 0.350689    0.970202    3.442793    0.214808    0.047837    0.010705    0.659117    
#> 213: 0.351698    0.968720    3.443210    0.210840    0.048103    0.010500    0.660394    
#> 214: 0.352149    0.968310    3.443367    0.208443    0.047897    0.010361    0.661243    
#> 215: 0.351624    0.968484    3.443253    0.206227    0.047764    0.010207    0.662347    
#> 216: 0.352064    0.968141    3.443150    0.203860    0.048253    0.010245    0.663820    
#> 217: 0.351408    0.968778    3.442853    0.202245    0.048734    0.010148    0.663460    
#> 218: 0.350488    0.969212    3.442421    0.200122    0.049002    0.010081    0.663326    
#> 219: 0.348887    0.968705    3.441733    0.198468    0.048871    0.010019    0.663451    
#> 220: 0.347672    0.968887    3.441406    0.197737    0.048791    0.009973    0.663632    
#> 221: 0.346253    0.969455    3.441071    0.196561    0.048956    0.009908    0.663070    
#> 222: 0.345316    0.969435    3.440733    0.195344    0.049380    0.009904    0.663259    
#> 223: 0.345545    0.970322    3.440226    0.195099    0.049467    0.009928    0.663348    
#> 224: 0.345257    0.970972    3.440117    0.195473    0.049314    0.009994    0.663270    
#> 225: 0.344874    0.969926    3.440226    0.195322    0.049154    0.010068    0.663170    
#> 226: 0.344913    0.969230    3.440141    0.194768    0.049209    0.010019    0.663296    
#> 227: 0.344629    0.969304    3.440338    0.196489    0.049198    0.010015    0.663025    
#> 228: 0.343844    0.970187    3.439677    0.197388    0.049353    0.010044    0.662805    
#> 229: 0.342077    0.970403    3.439016    0.197420    0.049445    0.010086    0.661862    
#> 230: 0.341617    0.970748    3.438523    0.197378    0.049496    0.010113    0.661927    
#> 231: 0.340275    0.971716    3.438060    0.197213    0.049791    0.010132    0.661697    
#> 232: 0.338412    0.971561    3.438146    0.197544    0.049780    0.010151    0.661724    
#> 233: 0.337319    0.971316    3.438072    0.198858    0.049692    0.010158    0.661664    
#> 234: 0.336353    0.971673    3.438011    0.199292    0.049661    0.010245    0.661553    
#> 235: 0.336611    0.971314    3.437928    0.200155    0.049327    0.010343    0.661470    
#> 236: 0.336828    0.971361    3.437827    0.200562    0.049190    0.010393    0.661799    
#> 237: 0.337999    0.970794    3.438033    0.199928    0.049020    0.010421    0.662225    
#> 238: 0.338318    0.970581    3.438234    0.199312    0.049191    0.010392    0.662921    
#> 239: 0.338341    0.970928    3.438268    0.198840    0.049545    0.010414    0.662664    
#> 240: 0.338573    0.970416    3.438754    0.198347    0.049613    0.010426    0.662613    
#> 241: 0.339559    0.969395    3.439164    0.199144    0.049623    0.010403    0.662680    
#> 242: 0.339918    0.969053    3.439110    0.199703    0.049512    0.010398    0.663202    
#> 243: 0.339606    0.969489    3.439022    0.200026    0.049582    0.010349    0.663546    
#> 244: 0.338939    0.969418    3.439083    0.200718    0.049370    0.010305    0.664082    
#> 245: 0.339159    0.969329    3.439299    0.200506    0.049478    0.010292    0.664431    
#> 246: 0.339201    0.969088    3.439396    0.200056    0.049424    0.010284    0.664310    
#> 247: 0.339352    0.968703    3.439605    0.199504    0.049616    0.010241    0.664075    
#> 248: 0.339123    0.968961    3.439693    0.199017    0.049757    0.010190    0.663699    
#> 249: 0.338688    0.968800    3.439816    0.198348    0.049822    0.010155    0.663926    
#> 250: 0.339692    0.969062    3.440048    0.198787    0.049891    0.010148    0.663886    
#> 251: 0.339808    0.969305    3.440286    0.198471    0.050041    0.010113    0.663909    
#> 252: 0.340219    0.969122    3.440503    0.197935    0.050244    0.010092    0.663747    
#> 253: 0.340160    0.968824    3.440584    0.197718    0.050378    0.010087    0.663829    
#> 254: 0.340138    0.968710    3.440860    0.197958    0.050573    0.010078    0.663794    
#> 255: 0.340403    0.968848    3.441011    0.198757    0.050596    0.010051    0.664004    
#> 256: 0.341064    0.968669    3.441067    0.198708    0.050472    0.010035    0.664177    
#> 257: 0.340857    0.968505    3.441019    0.198251    0.050530    0.010038    0.664228    
#> 258: 0.340577    0.968331    3.441032    0.198658    0.050559    0.010016    0.664080    
#> 259: 0.340109    0.968202    3.440927    0.198954    0.050601    0.010039    0.663928    
#> 260: 0.339814    0.968508    3.440907    0.198922    0.050620    0.010067    0.663563    
#> 261: 0.339751    0.968661    3.440890    0.199304    0.050608    0.010064    0.663471    
#> 262: 0.340169    0.969041    3.440851    0.199733    0.050463    0.010033    0.663424    
#> 263: 0.339923    0.969484    3.440847    0.199727    0.050341    0.010005    0.663554    
#> 264: 0.339859    0.969585    3.440823    0.199718    0.050215    0.009993    0.663710    
#> 265: 0.340316    0.969494    3.440927    0.200064    0.049960    0.010018    0.663789    
#> 266: 0.341146    0.969434    3.441118    0.200500    0.049866    0.010031    0.663547    
#> 267: 0.341537    0.969164    3.441353    0.200976    0.049806    0.010033    0.663551    
#> 268: 0.341361    0.968788    3.441320    0.200782    0.049851    0.010010    0.663713    
#> 269: 0.342144    0.968460    3.441393    0.200825    0.049934    0.010011    0.663828    
#> 270: 0.342519    0.968743    3.441393    0.200662    0.049910    0.009996    0.663487    
#> 271: 0.342457    0.968772    3.441380    0.200672    0.049875    0.009994    0.663556    
#> 272: 0.342296    0.968737    3.441344    0.200134    0.049825    0.010001    0.663764    
#> 273: 0.342127    0.969038    3.441327    0.200205    0.049709    0.010015    0.663677    
#> 274: 0.341559    0.969333    3.441232    0.199936    0.049668    0.010033    0.663687    
#> 275: 0.341060    0.969478    3.440993    0.199685    0.049546    0.010045    0.663654    
#> 276: 0.340858    0.969767    3.441056    0.200037    0.049558    0.010040    0.663675    
#> 277: 0.341182    0.970027    3.441241    0.200345    0.049593    0.010057    0.663561    
#> 278: 0.341407    0.969960    3.441396    0.200346    0.049653    0.010060    0.663544    
#> 279: 0.341293    0.969775    3.441459    0.199701    0.049800    0.010046    0.663759    
#> 280: 0.341320    0.969842    3.441507    0.199562    0.049860    0.010025    0.663774    
#> 281: 0.341163    0.969950    3.441583    0.199572    0.049916    0.010017    0.663980    
#> 282: 0.341429    0.970082    3.441739    0.199825    0.050021    0.009998    0.664085    
#> 283: 0.341679    0.969956    3.441902    0.199645    0.050186    0.009987    0.664473    
#> 284: 0.341776    0.969807    3.441978    0.199414    0.050228    0.009993    0.664587    
#> 285: 0.341643    0.969926    3.441852    0.199184    0.050331    0.010001    0.664701    
#> 286: 0.341591    0.970051    3.441791    0.199117    0.050530    0.009999    0.664600    
#> 287: 0.341290    0.970201    3.441664    0.199414    0.050523    0.009975    0.664691    
#> 288: 0.341420    0.970113    3.441671    0.199552    0.050551    0.009991    0.664705    
#> 289: 0.341727    0.969889    3.441743    0.199702    0.050593    0.010000    0.664435    
#> 290: 0.341917    0.969781    3.441836    0.200038    0.050569    0.010014    0.664556    
#> 291: 0.342162    0.969917    3.441937    0.199923    0.050538    0.010041    0.664287    
#> 292: 0.342202    0.969941    3.441951    0.199818    0.050553    0.010043    0.664179    
#> 293: 0.342112    0.969873    3.441915    0.199672    0.050473    0.010040    0.664059    
#> 294: 0.342028    0.969896    3.441898    0.199820    0.050496    0.010046    0.664074    
#> 295: 0.341878    0.969625    3.441910    0.199539    0.050464    0.010046    0.663985    
#> 296: 0.341878    0.969523    3.441889    0.199316    0.050423    0.010059    0.664061    
#> 297: 0.341820    0.969773    3.441859    0.199070    0.050425    0.010077    0.664075    
#> 298: 0.341807    0.969925    3.441854    0.199143    0.050376    0.010085    0.664061    
#> 299: 0.341912    0.970111    3.441798    0.199177    0.050301    0.010082    0.663946    
#> 300: 0.342185    0.970098    3.441831    0.199491    0.050358    0.010090    0.663685    
#> 301: 0.342325    0.970125    3.441772    0.199928    0.050315    0.010089    0.663517    
#> 302: 0.342558    0.970284    3.441746    0.200299    0.050289    0.010094    0.663297    
#> 303: 0.342529    0.970334    3.441753    0.200271    0.050297    0.010103    0.663091    
#> 304: 0.342333    0.970573    3.441616    0.200218    0.050293    0.010103    0.662968    
#> 305: 0.342234    0.970387    3.441591    0.200265    0.050327    0.010092    0.662787    
#> 306: 0.342003    0.970162    3.441598    0.200199    0.050307    0.010081    0.662807    
#> 307: 0.342143    0.970008    3.441575    0.200433    0.050333    0.010076    0.662785    
#> 308: 0.341963    0.969895    3.441500    0.200409    0.050372    0.010070    0.662909    
#> 309: 0.341650    0.969725    3.441381    0.200470    0.050401    0.010082    0.662904    
#> 310: 0.341666    0.969700    3.441252    0.200473    0.050425    0.010102    0.662980    
#> 311: 0.341259    0.969813    3.441101    0.200643    0.050427    0.010098    0.662987    
#> 312: 0.340834    0.969987    3.440834    0.200805    0.050390    0.010109    0.662933    
#> 313: 0.340449    0.969883    3.440667    0.201044    0.050357    0.010117    0.663142    
#> 314: 0.340151    0.969954    3.440557    0.201421    0.050350    0.010113    0.663189    
#> 315: 0.339583    0.970051    3.440423    0.201431    0.050363    0.010101    0.663238    
#> 316: 0.339475    0.970152    3.440380    0.201383    0.050439    0.010088    0.663210    
#> 317: 0.339589    0.970099    3.440442    0.201259    0.050461    0.010067    0.663442    
#> 318: 0.339734    0.970175    3.440386    0.201269    0.050335    0.010069    0.663492    
#> 319: 0.339599    0.970239    3.440345    0.201384    0.050274    0.010086    0.663522    
#> 320: 0.339336    0.970253    3.440253    0.201677    0.050300    0.010101    0.663452    
#> 321: 0.339066    0.970316    3.440112    0.201891    0.050378    0.010117    0.663438    
#> 322: 0.338829    0.970406    3.439961    0.201933    0.050467    0.010126    0.663314    
#> 323: 0.338378    0.970537    3.439913    0.201730    0.050502    0.010132    0.663287    
#> 324: 0.338234    0.970765    3.439863    0.201422    0.050504    0.010122    0.663232    
#> 325: 0.338108    0.970956    3.439730    0.201234    0.050505    0.010110    0.663295    
#> 326: 0.337931    0.970906    3.439644    0.201009    0.050516    0.010100    0.663348    
#> 327: 0.337911    0.970822    3.439640    0.200824    0.050579    0.010099    0.663378    
#> 328: 0.337891    0.970891    3.439605    0.200445    0.050568    0.010094    0.663485    
#> 329: 0.337884    0.971016    3.439615    0.200156    0.050609    0.010078    0.663515    
#> 330: 0.337790    0.971144    3.439613    0.199890    0.050592    0.010071    0.663636    
#> 331: 0.337764    0.971260    3.439589    0.199841    0.050592    0.010062    0.663591    
#> 332: 0.337953    0.971293    3.439675    0.200064    0.050556    0.010064    0.663450    
#> 333: 0.337894    0.971335    3.439628    0.199937    0.050503    0.010071    0.663310    
#> 334: 0.337759    0.971199    3.439588    0.199781    0.050439    0.010068    0.663285    
#> 335: 0.337628    0.971186    3.439530    0.199814    0.050345    0.010069    0.663246    
#> 336: 0.337443    0.971212    3.439439    0.200055    0.050249    0.010093    0.663268    
#> 337: 0.337388    0.971288    3.439389    0.200169    0.050179    0.010114    0.663370    
#> 338: 0.337369    0.971336    3.439379    0.200235    0.050174    0.010143    0.663435    
#> 339: 0.337273    0.971395    3.439366    0.200137    0.050215    0.010152    0.663378    
#> 340: 0.337209    0.971328    3.439365    0.199985    0.050332    0.010152    0.663399    
#> 341: 0.337270    0.971246    3.439379    0.200009    0.050462    0.010147    0.663357    
#> 342: 0.337064    0.971011    3.439367    0.199918    0.050503    0.010150    0.663335    
#> 343: 0.336896    0.971018    3.439349    0.199942    0.050551    0.010135    0.663286    
#> 344: 0.336978    0.970942    3.439366    0.200055    0.050561    0.010129    0.663305    
#> 345: 0.337087    0.970861    3.439436    0.200116    0.050616    0.010122    0.663279    
#> 346: 0.336894    0.970934    3.439366    0.200038    0.050690    0.010099    0.663245    
#> 347: 0.337038    0.970867    3.439303    0.199971    0.050762    0.010095    0.663347    
#> 348: 0.336780    0.970961    3.439243    0.199776    0.050882    0.010100    0.663398    
#> 349: 0.336619    0.970862    3.439244    0.199870    0.050894    0.010124    0.663471    
#> 350: 0.336599    0.970709    3.439231    0.199895    0.050951    0.010131    0.663467    
#> 351: 0.336706    0.970823    3.439196    0.199930    0.051010    0.010134    0.663391    
#> 352: 0.336520    0.970981    3.439178    0.199731    0.051075    0.010132    0.663476    
#> 353: 0.336427    0.971131    3.439131    0.199525    0.051055    0.010137    0.663450    
#> 354: 0.336380    0.971157    3.439033    0.199389    0.051004    0.010140    0.663493    
#> 355: 0.336616    0.971102    3.439003    0.199376    0.050974    0.010126    0.663488    
#> 356: 0.336649    0.971093    3.439023    0.199544    0.050997    0.010110    0.663546    
#> 357: 0.336807    0.971224    3.439035    0.199613    0.050994    0.010109    0.663405    
#> 358: 0.336870    0.971168    3.439089    0.199620    0.051027    0.010102    0.663300    
#> 359: 0.336855    0.971242    3.439062    0.199593    0.051041    0.010083    0.663248    
#> 360: 0.336861    0.971379    3.438958    0.199795    0.051095    0.010072    0.663103    
#> 361: 0.336652    0.971438    3.438905    0.199760    0.051097    0.010055    0.662983    
#> 362: 0.336641    0.971517    3.438837    0.199620    0.051119    0.010041    0.662878    
#> 363: 0.336574    0.971460    3.438805    0.199466    0.051086    0.010030    0.662859    
#> 364: 0.336618    0.971592    3.438757    0.199431    0.051141    0.010018    0.662793    
#> 365: 0.336597    0.971515    3.438790    0.199533    0.051147    0.010024    0.662866    
#> 366: 0.336646    0.971534    3.438774    0.199390    0.051153    0.010030    0.662965    
#> 367: 0.336677    0.971495    3.438765    0.199317    0.051095    0.010033    0.662976    
#> 368: 0.336860    0.971553    3.438753    0.199144    0.051072    0.010052    0.662887    
#> 369: 0.336855    0.971571    3.438749    0.199061    0.051129    0.010053    0.662896    
#> 370: 0.336858    0.971714    3.438733    0.198979    0.051186    0.010074    0.662853    
#> 371: 0.336784    0.971631    3.438682    0.198792    0.051187    0.010068    0.662904    
#> 372: 0.336816    0.971720    3.438678    0.198911    0.051183    0.010083    0.662841    
#> 373: 0.336458    0.971936    3.438545    0.198981    0.051156    0.010094    0.662771    
#> 374: 0.336231    0.971941    3.438445    0.199137    0.051096    0.010113    0.662619    
#> 375: 0.335944    0.972019    3.438344    0.199157    0.051099    0.010128    0.662607    
#> 376: 0.335782    0.972127    3.438248    0.199234    0.051058    0.010151    0.662496    
#> 377: 0.335556    0.972172    3.438193    0.199459    0.051044    0.010165    0.662386    
#> 378: 0.335376    0.972152    3.438130    0.199673    0.051044    0.010178    0.662366    
#> 379: 0.335170    0.972112    3.438056    0.199788    0.051060    0.010191    0.662380    
#> 380: 0.334940    0.972067    3.437966    0.200049    0.051108    0.010198    0.662325    
#> 381: 0.334749    0.972035    3.437946    0.200055    0.051096    0.010196    0.662254    
#> 382: 0.334783    0.972052    3.437973    0.200165    0.051100    0.010197    0.662131    
#> 383: 0.334853    0.972061    3.437988    0.200073    0.051102    0.010188    0.662047    
#> 384: 0.334941    0.971971    3.438072    0.199984    0.051106    0.010183    0.662071    
#> 385: 0.335069    0.971914    3.438161    0.199798    0.051139    0.010183    0.662070    
#> 386: 0.335199    0.971943    3.438271    0.199685    0.051157    0.010183    0.662050    
#> 387: 0.335513    0.971958    3.438374    0.199560    0.051191    0.010183    0.662083    
#> 388: 0.335751    0.971943    3.438474    0.199579    0.051156    0.010189    0.662031    
#> 389: 0.335773    0.971707    3.438498    0.199538    0.051167    0.010186    0.662072    
#> 390: 0.335634    0.971660    3.438509    0.199435    0.051154    0.010183    0.662101    
#> 391: 0.335637    0.971546    3.438516    0.199221    0.051105    0.010179    0.662085    
#> 392: 0.335646    0.971520    3.438559    0.199324    0.051062    0.010176    0.662100    
#> 393: 0.335416    0.971389    3.438482    0.199306    0.051020    0.010176    0.662075    
#> 394: 0.335328    0.971408    3.438419    0.199500    0.051032    0.010179    0.662091    
#> 395: 0.335420    0.971440    3.438382    0.199580    0.051068    0.010186    0.662121    
#> 396: 0.335388    0.971484    3.438368    0.199537    0.051097    0.010179    0.662185    
#> 397: 0.335397    0.971479    3.438371    0.199595    0.051118    0.010184    0.662224    
#> 398: 0.335373    0.971478    3.438343    0.199544    0.051110    0.010186    0.662261    
#> 399: 0.335503    0.971512    3.438398    0.199656    0.051170    0.010186    0.662144    
#> 400: 0.335566    0.971631    3.438421    0.199623    0.051225    0.010180    0.662108    
#> 401: 0.335559    0.971732    3.438391    0.199719    0.051316    0.010181    0.662166    
#> 402: 0.335524    0.971835    3.438326    0.199790    0.051364    0.010180    0.662199    
#> 403: 0.335631    0.971852    3.438311    0.199756    0.051385    0.010182    0.662191    
#> 404: 0.335595    0.971879    3.438237    0.199744    0.051376    0.010180    0.662156    
#> 405: 0.335477    0.971941    3.438187    0.199837    0.051343    0.010176    0.662184    
#> 406: 0.335482    0.971989    3.438150    0.200045    0.051333    0.010176    0.662181    
#> 407: 0.335427    0.972057    3.438152    0.199941    0.051375    0.010166    0.662134    
#> 408: 0.335503    0.972096    3.438164    0.199993    0.051362    0.010156    0.662140    
#> 409: 0.335530    0.972069    3.438127    0.199945    0.051331    0.010142    0.662150    
#> 410: 0.335588    0.972115    3.438065    0.199955    0.051368    0.010131    0.662188    
#> 411: 0.335658    0.972040    3.438035    0.200133    0.051337    0.010129    0.662210    
#> 412: 0.335639    0.972131    3.437954    0.200112    0.051337    0.010118    0.662175    
#> 413: 0.335622    0.972146    3.437911    0.200047    0.051328    0.010117    0.662237    
#> 414: 0.335479    0.972174    3.437918    0.200093    0.051303    0.010117    0.662226    
#> 415: 0.335521    0.972207    3.437900    0.200031    0.051275    0.010119    0.662219    
#> 416: 0.335590    0.972281    3.437919    0.200014    0.051278    0.010123    0.662253    
#> 417: 0.335581    0.972251    3.437892    0.200188    0.051295    0.010122    0.662210    
#> 418: 0.335387    0.972278    3.437881    0.200377    0.051285    0.010127    0.662134    
#> 419: 0.335425    0.972341    3.437895    0.200574    0.051334    0.010133    0.662108    
#> 420: 0.335424    0.972374    3.437881    0.200593    0.051391    0.010126    0.662064    
#> 421: 0.335368    0.972399    3.437863    0.200665    0.051422    0.010121    0.662016    
#> 422: 0.335362    0.972370    3.437819    0.200664    0.051427    0.010113    0.661991    
#> 423: 0.335408    0.972317    3.437803    0.200690    0.051486    0.010111    0.661993    
#> 424: 0.335330    0.972335    3.437741    0.200683    0.051538    0.010111    0.662020    
#> 425: 0.335272    0.972407    3.437741    0.200781    0.051496    0.010106    0.662111    
#> 426: 0.335245    0.972447    3.437721    0.200726    0.051478    0.010099    0.662091    
#> 427: 0.335117    0.972379    3.437709    0.200699    0.051446    0.010084    0.662110    
#> 428: 0.335041    0.972373    3.437702    0.200616    0.051484    0.010071    0.662131    
#> 429: 0.335016    0.972374    3.437713    0.200487    0.051553    0.010059    0.662152    
#> 430: 0.335000    0.972469    3.437711    0.200362    0.051573    0.010046    0.662263    
#> 431: 0.334940    0.972389    3.437681    0.200236    0.051611    0.010036    0.662234    
#> 432: 0.334775    0.972426    3.437661    0.200020    0.051643    0.010027    0.662179    
#> 433: 0.334783    0.972524    3.437719    0.199885    0.051610    0.010034    0.662220    
#> 434: 0.334752    0.972415    3.437736    0.199773    0.051593    0.010030    0.662233    
#> 435: 0.334881    0.972333    3.437771    0.199711    0.051582    0.010029    0.662224    
#> 436: 0.335031    0.972267    3.437811    0.199621    0.051618    0.010019    0.662278    
#> 437: 0.335045    0.972261    3.437832    0.199530    0.051663    0.010009    0.662303    
#> 438: 0.335067    0.972392    3.437867    0.199410    0.051660    0.010004    0.662327    
#> 439: 0.335070    0.972499    3.437859    0.199345    0.051656    0.010004    0.662275    
#> 440: 0.335033    0.972555    3.437859    0.199417    0.051674    0.009998    0.662244    
#> 441: 0.335069    0.972577    3.437857    0.199496    0.051673    0.009995    0.662165    
#> 442: 0.334919    0.972628    3.437841    0.199464    0.051695    0.009985    0.662126    
#> 443: 0.334847    0.972719    3.437780    0.199371    0.051661    0.009978    0.662131    
#> 444: 0.334753    0.972783    3.437728    0.199399    0.051650    0.009974    0.662113    
#> 445: 0.334644    0.972816    3.437660    0.199348    0.051644    0.009970    0.662137    
#> 446: 0.334444    0.972857    3.437587    0.199344    0.051647    0.009963    0.662134    
#> 447: 0.334544    0.972856    3.437616    0.199392    0.051606    0.009962    0.662068    
#> 448: 0.334581    0.972855    3.437644    0.199516    0.051557    0.009969    0.661991    
#> 449: 0.334623    0.972915    3.437706    0.199532    0.051513    0.009977    0.661961    
#> 450: 0.334662    0.972908    3.437728    0.199541    0.051504    0.009986    0.661951    
#> 451: 0.334791    0.972944    3.437729    0.199453    0.051507    0.009980    0.661970    
#> 452: 0.334782    0.972981    3.437708    0.199583    0.051498    0.009975    0.661918    
#> 453: 0.334759    0.973035    3.437696    0.199660    0.051480    0.009972    0.661823    
#> 454: 0.334747    0.973103    3.437704    0.199574    0.051449    0.009966    0.661771    
#> 455: 0.334793    0.973065    3.437718    0.199574    0.051437    0.009962    0.661716    
#> 456: 0.334770    0.973048    3.437704    0.199651    0.051448    0.009950    0.661724    
#> 457: 0.334748    0.973018    3.437754    0.199575    0.051494    0.009942    0.661645    
#> 458: 0.334740    0.973066    3.437790    0.199475    0.051506    0.009934    0.661638    
#> 459: 0.334689    0.973144    3.437800    0.199497    0.051495    0.009925    0.661636    
#> 460: 0.334745    0.973181    3.437805    0.199432    0.051496    0.009921    0.661611    
#> 461: 0.334720    0.973222    3.437809    0.199206    0.051502    0.009921    0.661645    
#> 462: 0.334612    0.973263    3.437818    0.199101    0.051516    0.009925    0.661662    
#> 463: 0.334673    0.973349    3.437811    0.199014    0.051505    0.009927    0.661675    
#> 464: 0.334635    0.973408    3.437779    0.199012    0.051504    0.009927    0.661661    
#> 465: 0.334615    0.973286    3.437782    0.199016    0.051499    0.009918    0.661676    
#> 466: 0.334639    0.973268    3.437779    0.198996    0.051510    0.009908    0.661666    
#> 467: 0.334620    0.973218    3.437802    0.199026    0.051546    0.009894    0.661633    
#> 468: 0.334719    0.973223    3.437842    0.199073    0.051550    0.009889    0.661694    
#> 469: 0.334822    0.973294    3.437868    0.199074    0.051530    0.009880    0.661731    
#> 470: 0.334781    0.973327    3.437828    0.199119    0.051528    0.009878    0.661704    
#> 471: 0.334702    0.973291    3.437813    0.199204    0.051498    0.009879    0.661646    
#> 472: 0.334674    0.973321    3.437806    0.199179    0.051510    0.009873    0.661633    
#> 473: 0.334580    0.973317    3.437804    0.199147    0.051507    0.009866    0.661563    
#> 474: 0.334492    0.973287    3.437809    0.199059    0.051495    0.009870    0.661546    
#> 475: 0.334488    0.973370    3.437818    0.199063    0.051460    0.009871    0.661536    
#> 476: 0.334472    0.973386    3.437781    0.198987    0.051439    0.009875    0.661576    
#> 477: 0.334435    0.973460    3.437793    0.198889    0.051410    0.009878    0.661495    
#> 478: 0.334327    0.973523    3.437784    0.198829    0.051385    0.009888    0.661500    
#> 479: 0.334324    0.973565    3.437784    0.198870    0.051336    0.009894    0.661557    
#> 480: 0.334362    0.973570    3.437778    0.198867    0.051328    0.009895    0.661651    
#> 481: 0.334376    0.973582    3.437755    0.198961    0.051359    0.009900    0.661692    
#> 482: 0.334366    0.973625    3.437740    0.198965    0.051371    0.009892    0.661671    
#> 483: 0.334314    0.973622    3.437728    0.198898    0.051407    0.009887    0.661630    
#> 484: 0.334381    0.973495    3.437747    0.198868    0.051414    0.009885    0.661633    
#> 485: 0.334392    0.973480    3.437768    0.198757    0.051404    0.009879    0.661676    
#> 486: 0.334515    0.973482    3.437805    0.198690    0.051394    0.009874    0.661686    
#> 487: 0.334724    0.973487    3.437855    0.198675    0.051375    0.009873    0.661761    
#> 488: 0.334795    0.973462    3.437898    0.198710    0.051383    0.009863    0.661798    
#> 489: 0.334848    0.973417    3.437892    0.198664    0.051409    0.009852    0.661855    
#> 490: 0.334891    0.973494    3.437905    0.198562    0.051417    0.009837    0.661854    
#> 491: 0.334865    0.973569    3.437928    0.198514    0.051400    0.009833    0.661812    
#> 492: 0.334868    0.973599    3.437938    0.198623    0.051384    0.009836    0.661801    
#> 493: 0.334814    0.973625    3.437937    0.198573    0.051366    0.009834    0.661862    
#> 494: 0.334998    0.973625    3.437962    0.198581    0.051377    0.009828    0.661866    
#> 495: 0.335095 