% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/topo.R
\name{download.topo}
\alias{download.topo}
\title{Download and Cache a topo File}
\usage{
download.topo(west, east, south, north, resolution, destdir, destfile,
  format, server, debug = getOption("oceDebug"))
}
\arguments{
\item{west, east}{Longitudes of the western and eastern sides of the box.}

\item{south, north}{Latitudes of the southern and northern sides of the box.}

\item{resolution}{Optional grid spacing, in minutes. If not supplied,
a default value of 4 (corresponding to 7.4km, or 4 nautical
miles) is used. Note that (as of August 2016) the original data are on
a 1-minute grid, which limits the possibilities for \code{resolution}.}

\item{destdir}{Optional string indicating the directory in which to store
downloaded files. If not supplied, \code{"."} is used, i.e. the data file
is stored in the present working directory.}

\item{destfile}{Optional string indicating the name of the file.
If not supplied, the file name is constructed from the other
parameters of the function call, so subsequent calls with the same
parameters will yield the same result, thus providing the key
to the caching scheme.}

\item{format}{Optional string indicating the type of file to download. If
not supplied, this defaults to \code{"gmt"}. See \dQuote{Details}.}

\item{server}{Optional string indicating the server from which to get the data.
If not supplied, the default
\samp{"https://gis.ngdc.noaa.gov/cgi-bin/public/wcs/etopo1.xyz"}
will be used.}

\item{debug}{an integer specifying whether debugging information is
to be printed during the processing. This is a general parameter that
is used by many \code{oce} functions. Generally, setting \code{debug=0}
turns off the printing, while higher values suggest that more information
be printed. If one function calls another, it usually reduces the value of
\code{debug} first, so that a user can often obtain deeper debugging
by specifying higher \code{debug} values.}
}
\value{
String indicating the full pathname to the downloaded file.
}
\description{
Data are downloaded (from \samp{https://maps.ngdc.noaa.gov/viewers/wcs-client/}, by
default) and a string containing the full path to the downloaded file is returned.
Typically, this return value is used with \code{\link{read.topo}} to read the
data. Subsequent calls to \code{download.topo}
with identical parameters will simply return the name of the cached file,
assuming the user has not deleted it in the meantime.
For convenience, if \code{destfile} is not
given, then \code{download.topo} will construct a filename from the other arguments,
rounding longitude and latitude limits to 0.01 degrees.
}
\details{
The data are downloaded with \code{\link[utils]{download.file}}, using a URL
devised from reverse engineering web-based queries constructed by
the default \code{server} used here. Note that the data source is "etopo1",
which is a 1 arc-second file [1,2].

Three values are permitted for \code{format},
each named after the
targets of menu items on the
NOAA website (as of August 2016): (1) \code{"aaigrid"} (for
the menu item "ArcGIS ASCII Grid"), which
yields a text file, (2) \code{"netcdf"} (the default,
for the menu item named
"NetCDF"), which yields a NetCDF file
and (3) \code{"gmt"} (for the menu item named
"GMT NetCDF"), which yields a NetCDF file in
another format. All of these file formats are
recognized by \code{\link{read.topo}}.
(The NOAA server has more options, and if
\code{\link{read.topo}} is extended to handle them, they will
also be added here.)
}
\section{Webserver history}{

All versions of \code{download.topo} to date have used a NOAA server as
the data source, but the URL has not been static. A list of the
servers that have been used is provided below,
in hopes that it can help users to make guesses
for \code{server}, should \code{download.topo} fail because of
a fail to download the data. Another
hint is to look at the source code for
\code{\link[marmap]{getNOAA.bathy}} in the \CRANpkg{marmap} package,
which is also forced to track the moving target that is NOAA.

\itemize{
\item August 2016.
\samp{http://maps.ngdc.noaa.gov/mapviewer-support/wcs-proxy/wcs.groovy}

\item December 2016.
\samp{http://mapserver.ngdc.noaa.gov/cgi-bin/public/wcs/etopo1.xyz}

\item June-September 2017.
\samp{https://gis.ngdc.noaa.gov/cgi-bin/public/wcs/etopo1.xyz}
}
}

\examples{
\dontrun{
library(oce)
topoFile <- download.topo(west=-64, east=-60, south=43, north=46,
                          resolution=1, destdir="~/data/topo")
topo <- read.topo(topoFile)
imagep(topo, zlim=c(-400, 400), drawTriangles=TRUE)
data(coastlineWorldFine, package="ocedata")
lines(coastlineWorldFine[["longitude"]], coastlineWorldFine[["latitude"]])
}

}
\references{
1. \samp{https://www.ngdc.noaa.gov/mgg/global/global.html}

2. Amante, C. and B.W. Eakins, 2009. ETOPO1 1 Arc-Minute Global Relief
Model: Procedures, Data Sources and Analysis. NOAA Technical Memorandum
NESDIS NGDC-24. National Geophysical Data Center, NOAA. doi:10.7289/V5C8276M
[access date: Aug 30, 2017].
}
\seealso{
The work is done with \code{\link[utils]{download.file}}.

Other functions that download files: \code{\link{download.amsr}},
  \code{\link{download.coastline}},
  \code{\link{download.met}}

Other things related to topo data: \code{\link{[[,topo-method}},
  \code{\link{[[<-,topo-method}}, \code{\link{as.topo}},
  \code{\link{plot,topo-method}}, \code{\link{read.topo}},
  \code{\link{subset,topo-method}},
  \code{\link{summary,topo-method}},
  \code{\link{topo-class}}, \code{\link{topoInterpolate}},
  \code{\link{topoWorld}}
}
\author{
Dan Kelley
}
\concept{functions that download files}
\concept{things related to topo data}
