% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/map.R
\name{mapPlot}
\alias{mapPlot}
\title{Draw a Map}
\usage{
mapPlot(
  longitude,
  latitude,
  longitudelim,
  latitudelim,
  grid = TRUE,
  geographical = 0,
  bg,
  fill,
  border = NULL,
  col = NULL,
  clip = TRUE,
  type = "polygon",
  axes = TRUE,
  axisStyle = 1,
  cex,
  cex.axis = 1,
  mgp = c(0, 0.5, 0),
  drawBox = TRUE,
  showHemi = TRUE,
  polarCircle = 0,
  lonlabels = TRUE,
  latlabels = TRUE,
  projection = "+proj=moll",
  tissot = FALSE,
  trim = TRUE,
  debug = getOption("oceDebug"),
  ...
)
}
\arguments{
\item{longitude}{either a numeric vector of longitudes of points to be plotted, or
something (an \code{oce} object, a list, or a data frame) from which both
longitude and latitude may be inferred (in which case the \code{latitude}
argument is ignored).  If \code{longitude} is missing, both it and
\code{latitude} are taken from the built-in \link{coastlineWorld} dataset.}

\item{latitude}{numeric vector of latitudes of points to be plotted (ignored
if the first argument contains both latitude and longitude).}

\item{longitudelim, latitudelim}{optional numeric vectors of length two, indicating the
limits of the plot. A warning is issued if these are not specified together.
See \dQuote{Examples} for a polar-region example, noting that
the whole-globe span of \code{longitudelim} is used to centre the plot
at the north pole.}

\item{grid}{either a number (or pair of numbers) indicating the spacing of
longitude and latitude lines, in degrees, or a logical value (or pair of
values) indicating whether to draw an auto-scaled grid, or whether to skip
the grid drawing.  In the case of numerical values, \code{NA} can be used to
turn off the grid in longitude or latitude.  Grids are set up based on
examination of the scale used in middle 10 percent of the plot area, and for
most projections this works quite well.  If not, one may set
\code{grid=FALSE} and add a grid later with \code{\link[=mapGrid]{mapGrid()}}.}

\item{geographical}{flag indicating the style of axes.  With
\code{geographical=0}, the axes are conventional, with decimal degrees as
the unit, and negative signs indicating the southern and western
hemispheres.  With \code{geographical=1}, the signs are dropped, with axis
values being in decreasing order within the southern and western
hemispheres.  With \code{geographical=2}, the signs are dropped and the axes
are labelled with degrees, minutes and seconds, as appropriate, and
hemispheres are indicated with letters. With \code{geographical=3}, things
are the same as for \code{geographical=2}, but the hemisphere indication
is omitted. Finally, with \code{geographical=4}, unsigned numbers are used,
followed by letters \code{N} in the northern hemisphere, \code{S} in the southern,
\code{E} in the eastern, and \code{W} in the western.}

\item{bg}{color of the background (ignored).}

\item{fill}{is a deprecated argument; see \link{oce-deprecated}.}

\item{border}{color of coastlines and international borders (ignored unless
\code{type="polygon"}.}

\item{col}{either the color for filling polygons (if \code{type="polygon"})
or the color of the points and line segments (if \code{type="p"},
\code{type="l"}, or \code{type="o"}). If \code{col=NULL} then a default
will be set: no coastline filling for the \code{type="polygon"} case,
or black coastlines, for \code{type="p"}, \code{type="l"}, or
\code{type="o"}.}

\item{clip}{logical value indicating whether to trim any coastline elements that lie wholly
outside the plot region. This can prevent e.g. a problem of filling the whole plot area of
an Arctic stereopolar view, because the projected trace for Antarctica lies outside all
other regions so the whole of the world ends up being "land".  Setting \code{clip=FALSE}
disables this action, which may be of benefit in rare instances in the line connecting
two points on a coastline may cross the plot domain, even if those points are outside
that domain.}

\item{type}{indication of type; may be \code{"polygon"}, for a filled polygon,
\code{"p"} for points, \code{"l"} for line segments, or \code{"o"} for points
overlain with line segments.}

\item{axes}{a logical value indicating whether to draw longitude and latitude
values in the lower and left margin, respectively.  This may not work well
for some projections or scales.  See also \code{axisStyle}, \code{lonlabels}
and \code{latlabels}, which offer more granular control of labelling.}

\item{axisStyle}{an integer specifying the style of labels for the numbers
on axes.  The choices are:
1 for signed numbers without additional labels;
2 (the default) for unsigned numbers followed by letters indicating the hemisphere;
3 for signed numbers followed by a degree sign;
4 for unsigned numbers followed by a degree sign; and
5 for signed numbers followed by a degree sign and letters indicating the hemisphere.}

\item{cex}{character expansion factor for plot symbols,
used if \code{type='p'} or any other value that yields symbols.}

\item{cex.axis}{axis-label expansion factor (see \code{\link[=par]{par()}}).}

\item{mgp}{three-element numerical vector describing axis-label
placement, passed to \code{\link[=mapAxis]{mapAxis()}}.}

\item{drawBox}{logical value indicating whether to draw a box around the plot.
This is helpful for many projections at sub-global scale.}

\item{showHemi}{logical value indicating whether to show the hemisphere in
axis tick labels.}

\item{polarCircle}{a number indicating the number of degrees of latitude
extending from the poles, within which zones are not drawn.}

\item{lonlabels}{An optional logical value or numeric vector that controls
the labelling along the horizontal axis. There are four possibilities:
(1) If \code{lonlabels} is \code{TRUE} (the default), then reasonable values are inferred
and axes are drawn with ticks and labels alongside those ticks;
(2) if \code{lonlabels} is \code{FALSE}, then ticks are drawn, but no labels;
(3) if \code{lonlabels} is \code{NULL}, then no axis ticks or labels are drawn; and
(4) if \code{lonlabels} is  a vector of finite numerical values, then tick marks
are placed  at those longitudes, and labels are put alongside them.
Note that R tries to avoid overwriting labels on axes, so the instructions
in case 4 might not be obeyed exactly.
See also \code{latlabels}, and note that setting \code{axes=FALSE}
ensures that no longitude or latitude axes will be drawn regardless
of the values of \code{lonlabels} and \code{latlabels}.}

\item{latlabels}{As \code{lonlabels}, but for latitude, on the left
plot axis.}

\item{projection}{either character value indicating the map projection, or
the output from \code{\link[sf:st_crs]{sf::st_crs()}}. In the first case, see a table
in \dQuote{Details} for the projections that are available.
In the second case, note that \code{\link[=mapPlot]{mapPlot()}} reports an error if
a similar function from the old \code{sp} package is used.}

\item{tissot}{logical value indicating whether to use \code{\link[=mapTissot]{mapTissot()}}
to plot Tissot indicatrices, i.e. ellipses at grid intersection points, which
indicate map distortion.}

\item{trim}{logical value indicating whether to trim islands or lakes
containing only points that are off-scale of the current plot box.  This
solves the problem of Antarctica overfilling the entire domain, for an
Arctic-centred stereographic projection.  It is not a perfect solution,
though, because the line segment joining two off-scale points might
intersect the plotting box.}

\item{debug}{a flag that turns on debugging.  Set to 1 to get a moderate
amount of debugging information, or to 2 to get more.}

\item{...}{optional arguments passed to some plotting functions.  This can
be useful in many ways, e.g.  Example 5 shows how to use \code{xlim} etc to
reproduce a scale exactly between two plots.}
}
\description{
Plot coordinates as a map, using one of the subset of projections
provided by the \CRANpkg{sf} package.  The projection information specified
with the \code{mapPlot()} call is stored in a global variable that can be retrieved
by related functions, making it easy to add points, lines, text, images
or contours to an existing map.
The \dQuote{Details} section, below, provides a list of
available projections. The "Using map projections" vignette
offers examples of several map plots, in addition to the single
example provided in the \dQuote{Examples} section, below.
}
\details{
The calculations for map projections are done with
the \CRANpkg{sf} package.  Importantly, though, not all
the \CRANpkg{sf} projections are
available in \code{oce}, for reasons relating to limitations
of \CRANpkg{sf}, for example relating to inverse-projection
calculations. The \code{oce} choices are tabulated
below, e.g. \code{projection="+proj=aea"} selects the Albers equal area
projection.  (See also the warning, below, about a problem
with \CRANpkg{sf} version 0.9-8.)

Further details of the vast array of map projections provided by PROJ
are given in reference 4.  This system has been in rapid development
since about 2018, and reference 5 provides a helpful overview of the changes
and the reasons why they were necessary.  Practical examples of map projections in
\CRANpkg{oce} are provided in reference 6, along with some notes on problems.
A fascinating treatment of the history of map projections is provided
in reference 7.  To get an idea of how projections are being created
nowadays, see reference 8, about the \code{eqearth} projection that was added
to \CRANpkg{oce} in August 2020.
}
\section{Available Projections}{


The following table lists projections available in \CRANpkg{oce}, and was generated
by reformatting a subset of the output of the unix
command \code{proj -lP}. Most of the arguments have default values, and many
projections also have optional arguments.  Although e.g. \code{proj -l=aea}
provides a little more information about particular projections,
users ought to consult reference 4 for fuller details and illustrations.\tabular{lll}{
   \strong{Projection} \tab \strong{Code} \tab \strong{Arguments} \cr
   Albers equal area \tab \code{aea} \tab \code{lat_1}, \code{lat_2} \cr
   Azimuthal equidistant \tab \code{aeqd} \tab \code{lat_0}, \code{guam} \cr
   Aitoff \tab \code{aitoff} \tab - \cr
   Mod. stererographics of Alaska \tab \code{alsk} \tab - \cr
   Bipolar conic of western hemisphere \tab \code{bipc} \tab - \cr
   Bonne Werner \tab \code{bonne} \tab \code{lat_1} \cr
   Cassini \tab \code{cass} \tab - \cr
   Central cylindrical \tab \code{cc} \tab - \cr
   Equal area cylindrical \tab \code{cea} \tab \code{lat_ts} \cr
   Collignon \tab \code{collg} \tab - \cr
   Craster parabolic Putnins P4 \tab \code{crast} \tab - \cr
   Eckert I \tab \code{eck1} \tab - \cr
   Eckert II \tab \code{eck2} \tab - \cr
   Eckert III \tab \code{eck3} \tab - \cr
   Eckert IV \tab \code{eck4} \tab - \cr
   Eckert V \tab \code{eck5} \tab - \cr
   Eckert VI \tab \code{eck6} \tab - \cr
   Equidistant cylindrical plate (Caree) \tab \code{eqc} \tab \code{lat_ts}, \code{lat_0} \cr
   Equidistant conic \tab \code{eqdc} \tab \code{lat_1}, \code{lat_2} \cr
   Equal earth \tab \code{eqearth} \tab - \cr
   Euler \tab \code{euler} \tab \code{lat_1}, \code{lat_2} \cr
   Extended transverse Mercator \tab \code{etmerc} \tab - \cr
   Fahey \tab \code{fahey} \tab - \cr
   Foucault \tab \code{fouc} \tab - \cr
   Foucault sinusoidal \tab \code{fouc_s} \tab - \cr
   Gall stereographic \tab \code{gall} \tab - \cr
   Geostationary satellite view \tab \code{geos} \tab \code{h} \cr
   General sinusoidal series \tab \code{gn_sinu} \tab \code{m}, \code{n} \cr
   Gnomonic \tab \code{gnom} \tab - \cr
   Goode homolosine \tab \code{goode} \tab - \cr
   Hatano asymmetrical equal area \tab \code{hatano} \tab - \cr
   Interrupted Goode homolosine \tab \code{igh} \tab - \cr
   Kavraisky V \tab \code{kav5} \tab - \cr
   Kavraisky VII \tab \code{kav7} \tab - \cr
   Lambert azimuthal equal area \tab \code{laea} \tab - \cr
   Longitude and latitude \tab \code{longlat} \tab - \cr
   Longitude and latitude \tab \code{latlong} \tab - \cr
   Lambert conformal conic \tab \code{lcc} \tab \code{lat_1}, \code{lat_2} or \code{lat_0}, \code{k_0} \cr
   Lambert equal area conic \tab \code{leac} \tab \code{lat_1}, \code{south} \cr
   Loximuthal \tab \code{loxim} \tab - \cr
   Space oblique for Landsat \tab \code{lsat} \tab \code{lsat}, \code{path} \cr
   McBryde-Thomas flat-polar sine, no. 1 \tab \code{mbt_s} \tab - \cr
   McBryde-Thomas flat-polar sine, no. 2 \tab \code{mbt_fps} \tab - \cr
   McBryde-Thomas flat-polar parabolic \tab \code{mbtfpp} \tab - \cr
   McBryde-Thomas flat-polar quartic \tab \code{mbtfpq} \tab - \cr
   McBryde-Thomas flat-polar sinusoidal \tab \code{mbtfps} \tab - \cr
   Mercator \tab \code{merc} \tab \code{lat_ts} \cr
   Miller oblated stereographic \tab \code{mil_os} \tab - \cr
   Miller cylindrical \tab \code{mill} \tab - \cr
   Mollweide \tab \code{moll} \tab - \cr
   Murdoch I \tab \code{murd1} \tab \code{lat_1}, \code{lat_2} \cr
   Murdoch II \tab \code{murd2} \tab \code{lat_1}, \code{lat_2} \cr
   murdoch III \tab \code{murd3} \tab \code{lat_1}, \code{lat_2} \cr
   Natural earth \tab \code{natearth} \tab - \cr
   Nell \tab \code{nell} \tab - \cr
   Nell-Hammer \tab \code{nell_h} \tab - \cr
   Near-sided perspective \tab \code{nsper} \tab \code{h} \cr
   New Zealand map grid \tab \code{nzmg} \tab - \cr
   General oblique transformation \tab \code{ob_tran} \tab \code{o_proj}, \code{o_lat_p}, \code{o_lon_p}, \cr
    \tab  \tab \code{o_alpha}, \code{o_lon_c}, \code{o_lat_c}, \cr
    \tab  \tab \code{o_lon_1}, \code{o_lat_1}, \cr
    \tab  \tab \code{o_lon_2}, \code{o_lat_2} \cr
   Oblique cylindrical equal area \tab \code{ocea} \tab \code{lat_1}, \code{lat_2}, \code{lon_1}, \code{lon_2} \cr
   Oblated equal area \tab \code{oea} \tab \code{n}, \code{m}, \code{theta} \cr
   Oblique Mercator \tab \code{omerc} \tab \code{alpha}, \code{gamma}, \code{no_off}, \cr
    \tab  \tab \code{lonc}, \code{lon_1}, \code{lat_1}, \cr
    \tab  \tab \code{lon_2}, \code{lat_2} \cr
   Orthographic \tab \code{ortho} \tab - \cr
   Polyconic American \tab \code{poly} \tab - \cr
   Putnins P1 \tab \code{putp1} \tab - \cr
   Putnins P2 \tab \code{putp2} \tab - \cr
   Putnins P3 \tab \code{putp3} \tab - \cr
   Putnins P3' \tab \code{putp3p} \tab - \cr
   Putnins P4' \tab \code{putp4p} \tab - \cr
   Putnins P5 \tab \code{putp5} \tab - \cr
   Putnins P5' \tab \code{putp5p} \tab - \cr
   Putnins P6 \tab \code{putp6} \tab - \cr
   Putnins P6' \tab \code{putp6p} \tab - \cr
   Quartic authalic \tab \code{qua_aut} \tab - \cr
   Quadrilateralized spherical cube \tab \code{qsc} \tab - \cr
   Robinson \tab \code{robin} \tab - \cr
   Roussilhe stereographic \tab \code{rouss} \tab - \cr
   Sinusoidal aka Sanson-Flamsteed \tab \code{sinu} \tab - \cr
   Swiss. oblique Mercator \tab \code{somerc} \tab - \cr
   Stereographic \tab \code{stere} \tab \code{lat_ts} \cr
   Oblique stereographic alternative \tab \code{sterea} \tab - \cr
   Transverse cylindrical equal area \tab \code{tcea} \tab - \cr
   Tissot \tab \code{tissot} \tab \code{lat_1}, \code{lat_2} \cr
   Transverse Mercator \tab \code{tmerc} \tab \code{approx} \cr
   Two point equidistant \tab \code{tpeqd} \tab \code{lat_1}, \code{lon_1}, \code{lat_2}, \code{lon_2} \cr
   Tilted perspective \tab \code{tpers} \tab \code{tilt}, \code{azi}, \code{h} \cr
   Universal polar stereographic \tab \code{ups} \tab \code{south} \cr
   Urmaev flat-polar sinusoidal \tab \code{urmfps} \tab \code{n} \cr
   Universal transverse Mercator \tab \code{utm} \tab \code{zone}, \code{south}, \code{approx} \cr
   van der Grinten I \tab \code{vandg} \tab - \cr
   Vitkovsky I \tab \code{vitk1} \tab \code{lat_1}, \code{lat_2} \cr
   Wagner I Kavraisky VI \tab \code{wag1} \tab - \cr
   Wagner II \tab \code{wag2} \tab - \cr
   Wagner III \tab \code{wag3} \tab \code{lat_ts} \cr
   Wagner IV \tab \code{wag4} \tab - \cr
   Wagner V \tab \code{wag5} \tab - \cr
   Wagner VI \tab \code{wag6} \tab - \cr
   Werenskiold I \tab \code{weren} \tab - \cr
   Winkel I \tab \code{wink1} \tab \code{lat_ts} \cr
   Winkel Tripel \tab \code{wintri} \tab \code{lat_ts} \cr
}
}

\section{Choosing a projection}{

The best choice of projection depends on the application.
Users may find \code{projection="+proj=moll"} useful for world-wide
plots, \code{ortho} for hemispheres viewed from the equator, \code{stere}
for polar views, \code{lcc} for wide meridional ranges in mid latitudes,
\code{merc} in limited-area cases where angle preservation is
important, or either \code{aea} or \code{eqearth} (on local and global
scales, respectively) where area preservation is important.
The choice becomes more important, the larger the size of the region
represented.  When it comes to publication, it can be sensible to use the
same projection as used in previous reports.
}

\section{Problems}{

Map projection is a complicated matter that is addressed here
in a limited and pragmatic way.  For example, \code{mapPlot} tries to draw
axes along a box containing the map, instead of trying to find spots along
the ``edge'' of the map at which to put longitude and latitude labels.
This design choice greatly simplifies the coding effort, freeing up time to
work on issues regarded as more pressing.  Chief among those issues are (a)
the occurrence of horizontal lines in maps that have prime meridians
(b) inaccurate filling of land regions that (again) occur with shifted
meridians and (c) inaccurate filling of Antarctica in some projections.
Generally, issues are tackled first for commonly used projections, such as
those used in the examples.
}

\section{Historical Notes}{

\itemize{
\item 2020-12-24: complete switch from \code{rgdal} to \CRANpkg{sf},
removing the testing scheme created on 2020-08-03.
\item 2020-08-03: added support for the \code{eqearth} projection (like \code{robin} but
an equal-area method).
\item 2020-08-03: dropped support for the \code{healpix}, \code{pconic}
and \code{rhealpix} projections, which caused errors with the
\CRANpkg{sf} package.  (This is not a practical loss, since these
interrupted projections were handled badly by \code{mapPlot()}
in any case.)
\item 2020-08-03: switch from \code{rgdal} to \CRANpkg{sf} for
calculations related to map projection, owing to some
changes in the former package that broke \CRANpkg{oce}
code.  (To catch problems, \CRANpkg{oce} was set up to use
both packages temporarily, issuing warnings if the results differed
by more than 1 metre in easting or northing values.)
\item 2017-11-19: \code{imw_p} removed, because it has problems doing
inverse calculations.
This is a also problem in the standalone PROJ.4 application version
4.9.3, downloaded and built on OSX.
See \verb{https://github.com/dankelley/oce/issues/1319} for details.
\item 2017-11-17: \code{lsat} removed, because it does not work in
\code{rgdal} or in the latest standalone PROJ.4 application.
This is a also problem in the standalone PROJ.4 application version
4.9.3, downloaded and built on OSX.
See \verb{https://github.com/dankelley/oce/issues/1337} for details.
\item 2017-09-30: \code{lcca} removed, because its inverse was
wildly inaccurate in a Pacific Antarctic-Alaska application
(see \verb{https://github.com/dankelley/oce/issues/1303}).
}
}

\examples{
# NOTE: the map-projection vignette has many more examples.
library(oce)
data(coastlineWorld)
# Demonstrate a high-latitude view using a built-in "CRS" value that is used
# by the National Snow and Ice Data Center (NSIDC) for representing
# the northern-hemisphere ice zone.  The view is meant to mimic the figure
# at the top of the document entitled "A Guide to NSIDC's Polar Stereographic
# Projection" at https://nsidc.org/data/user-resources/help-center, with the
# box indicating the region of the NSIDC grid.
library(oce)
data(coastlineWorld)
projection <- sf::st_crs("EPSG:3413")
cat(projection$proj4string, "\n") # see the projection details
par(mar=c(2, 2, 1, 1))            # tighten margins
mapPlot(coastlineWorld, projection=projection,
   col=gray(0.9), geographical=4,
   longitudelim=c(-180, 180), latitudelim=c(10, 90))
# Coordinates of box from Table 6 of the NSIDC document
box <- cbind(-360+c(168.35,102.34,350.3,279.26,168.35),
    c(30.98, 31.37, 34.35, 33.92, 30.98))
mapLines(box[,1], box[,2], lwd=2)


}
\references{
\enumerate{
\item Snyder, John P., 1987.  Map Projections: A Working Manual.  USGS
Professional Paper: 1395
\verb{https://pubs.er.usgs.gov/publication/pp1395}
\item Natural Resources Canada
\verb{https://www.nrcan.gc.ca/earth-sciences/geography/topographic-information/maps/9805}
\item "List of Map Projections." In Wikipedia, January 26, 2021.
\verb{https://en.wikipedia.org/w/index.php?title=List_of_map_projections}.
\item PROJ contributors (2020).
"PROJ Coordinate Transformation Software Library."
Open Source Geospatial Foundation, n.d.
\verb{https://proj.org}.
\item Bivand, Roger (2020) Why have CRS, projections and transformations changed?
\item A gallery of map plots is provided at
\verb{https://dankelley.github.io/r/2020/08/02/oce-proj.html}
\item Snyder, John Parr.
Flattening the Earth: Two Thousand Years of Map Projections.
Chicago, IL: University of Chicago Press, 1993.
\verb{https://press.uchicago.edu/ucp/books/book/chicago/F/bo3632853.html}
\item Šavrič, Bojan, Tom Patterson, and Bernhard Jenny.
"The Equal Earth Map Projection."
International Journal of Geographical Information Science 33, no. 3 (March 4, 2019): 454-65.
\doi{10.1080/13658816.2018.1504949}
}
}
\seealso{
Points may be added to a map with \code{\link[=mapPoints]{mapPoints()}}, lines with
\code{\link[=mapLines]{mapLines()}}, text with \code{\link[=mapText]{mapText()}}, polygons with
\code{\link[=mapPolygon]{mapPolygon()}}, images with \code{\link[=mapImage]{mapImage()}}, and scale bars
with \code{\link[=mapScalebar]{mapScalebar()}}.  Points on a map may be determined with mouse
clicks using \code{\link[=mapLocator]{mapLocator()}}.  Great circle paths can be calculated
with \code{\link[=geodGc]{geodGc()}}.  See reference 8 for a demonstration of the available map
projections (with graphs).

Other functions related to maps: 
\code{\link{formatPosition}()},
\code{\link{lonlat2map}()},
\code{\link{lonlat2utm}()},
\code{\link{map2lonlat}()},
\code{\link{mapArrows}()},
\code{\link{mapAxis}()},
\code{\link{mapContour}()},
\code{\link{mapCoordinateSystem}()},
\code{\link{mapDirectionField}()},
\code{\link{mapGrid}()},
\code{\link{mapImage}()},
\code{\link{mapLines}()},
\code{\link{mapLocator}()},
\code{\link{mapLongitudeLatitudeXY}()},
\code{\link{mapPoints}()},
\code{\link{mapPolygon}()},
\code{\link{mapScalebar}()},
\code{\link{mapText}()},
\code{\link{mapTissot}()},
\code{\link{oceCRS}()},
\code{\link{shiftLongitude}()},
\code{\link{usrLonLat}()},
\code{\link{utm2lonlat}()}
}
\author{
Dan Kelley and Clark Richards
}
\concept{functions related to maps}
