% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/look_up.R
\name{look_up}
\alias{look_up}
\title{Look up geological intervals and assign geological stages}
\usage{
look_up(
  occdf,
  early_interval = "early_interval",
  late_interval = "late_interval",
  int_key = FALSE,
  assign_with_GTS = "GTS2020",
  return_unassigned = FALSE
)
}
\arguments{
\item{occdf}{\code{dataframe}. A dataframe of fossil occurrences or other
geological data, with columns of class \code{character} specifying the
earliest and the latest possible interval associated with each occurrence.}

\item{early_interval}{\code{character}. Name of the column in \code{occdf} that
contains the earliest interval from which the occurrences are from. Defaults
to "early_interval".}

\item{late_interval}{\code{character}. Name of the column in \code{occdf} that
contains the latest interval from which the occurrences are from. Defaults
to "late_interval".}

\item{int_key}{\code{dataframe}. A dataframe linking interval names to
international geological stage names from the ICS, or other, user-defined
intervals.
This dataframe should contain the following named columns containing
\code{character} values: \cr
\itemize{
\item \code{interval_name} contains the names to be matched from \code{occdf} \cr
\item \code{early_stage} contains the names of the earliest stages
corresponding to the intervals \cr
\item \code{late_stage} contains the latest stage corresponding to the
intervals \cr
}
Optionally, named \code{numeric} columns provide maximum and minimum ages
for the intervals: \cr
\itemize{
\item \code{max_ma}
\item \code{min_ma}
}
If set to \code{FALSE} (default), stages and numerical ages can be assigned
based on one of the GTS tables (see below).}

\item{assign_with_GTS}{\code{character} or \code{FALSE}. Allows intervals to
be searched in the \code{GTS2020} (default) or the \code{GTS2012} table. Set to
\code{FALSE} to disable.}

\item{return_unassigned}{\code{logical}. Return interval names which could
not be assigned, instead of the dataframe with assignments.
Defaults to \code{FALSE}.}
}
\value{
A \code{dataframe} of the original input \code{data} with the following
appended columns is returned: \code{early_stage} and \code{late_stage}, corresponding
to the earliest and latest international geological stage which
could be assigned to the occurrences based on the given interval names.
\code{interval_max_ma} and \code{interval_min_ma} return maximum and minimum interval
ages if provided in the interval key, or if they can be fetched from GTS2012
or GTS2020. A column \code{interval_mid_ma} is appended to provide the midpoint
ages of the intervals.
}
\description{
A function that uses interval names to assign either
\href{https://stratigraphy.org/ICSchart/ChronostratChart2022-02.pdf}{international geological stages}
and numeric ages from the International Commission on Stratigraphy (ICS), or
user-defined intervals, to fossil occurrences.
}
\details{
If \code{int_key} is set to \code{FALSE} (default), this function can be used to
assign numerical ages solely based on stages from a GTS table, and to assign
stages based on GTS interval names.

Instead of  geological stages, the user can supply any names in the
\code{early_stage} and \code{late_stage} column of \code{int_key}.
\code{assign_with_GTS} should then be set to \code{FALSE}.

An exemplary \code{int_key} has been included within this package
(\code{\link{interval_key}}). This key works well for assigning
geological stages to many of the intervals from the
\href{https://paleobiodb.org}{Paleobiology Database}
and the \href{https://www.paleo-reefs.pal.uni-erlangen.de/}{PaleoReefs Database}.
\code{palaeoverse} cannot guarantee that all of
the stage assignments with the exemplary key are accurate.
The table corresponding to this key can be loaded with
\code{palaeoverse::interval_key}.
}
\section{Developer(s)}{

Kilian Eichenseer & William Gearty
}

\section{Reviewer(s)}{

Lewis A. Jones & Christopher D. Dean
}

\examples{
## Just use GTS2020 (default):
# create exemplary dataframe
taxdf <- data.frame(name = c("A", "B", "C"),
early_interval = c("Maastrichtian", "Campanian", "Sinemurian"),
late_interval = c("Maastrichtian", "Campanian", "Bartonian"))
# assign stages and numerical ages
taxdf <- look_up(taxdf)

## Use exemplary int_key
# Get internal reef data
occdf <- reefs
 # assign stages and numerical ages
occdf <- look_up(occdf,
                early_interval = "interval",
                late_interval = "interval",
                int_key = interval_key)

## Use exemplary int_key and return unassigned
# Get internal tetrapod data
occdf <- tetrapods
# assign stages and numerical ages
occdf <- look_up(occdf, int_key = palaeoverse::interval_key)
# return unassigned intervals
unassigned <- look_up(occdf, int_key = palaeoverse::interval_key,
                      return_unassigned = TRUE)

## Use own key and GTS2012:
# create example data
occdf <- data.frame(
  stage = c("any Permian", "first Permian stage",
            "any Permian", "Roadian"))
# create example key
interval_key <- data.frame(
  interval_name = c("any Permian", "first Permian stage"),
  early_stage = c("Asselian", "Asselian"),
  late_stage = c("Changhsingian", "Asselian"))
# assign stages and numerical ages:
occdf <- look_up(occdf,
                 early_interval = "stage", late_interval = "stage",
                 int_key = interval_key, assign_with_GTS = "GTS2012")

}
