% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simFossilRecordMethods.R
\name{simFossilRecordMethods}
\alias{simFossilRecordMethods}
\alias{timeSliceFossilRecord}
\alias{fossilRecord2fossilTaxa}
\alias{fossilRecord2fossilRanges}
\alias{fossilRecord2fossilTaxa}
\alias{fossilTaxa2fossilRecord}
\alias{fossilRecord2fossilRanges}
\title{Methods for Editing or Converting Output from simFossilRecord}
\usage{
timeSliceFossilRecord(fossilRecord, sliceTime, shiftRoot4TimeSlice = FALSE,
  modern.samp.prob = 1, tolerance = 10^-4)

fossilRecord2fossilTaxa(fossilRecord)

fossilTaxa2fossilRecord(fossilTaxa)

fossilRecord2fossilRanges(fossilRecord, merge.cryptic = TRUE,
  ranges.only = TRUE)
}
\arguments{
\item{fossilRecord}{A list object output by \code{simFossilRecord}, often composed
of multiple elements, each of which is data for 'one taxon', with the first
element being a distinctive six-element vector composed of numbers, corresponding
to the six fields in tables output by the deprecated function \code{simFossilTaxa}.}

\item{sliceTime}{The date to slice the \code{simFossilRecord} output at, given
in time-units before the modern, on the same scale as the input \code{fossilRecord}.}

\item{shiftRoot4TimeSlice}{Should the dating of events be shifted, so that the
date given for \code{sliceTime} is now 0, or should the dates not be shifted,
so that they remain on the same scale as the input? This argument accepts a
logical TRUE or FALSE, but also accepts the string \code{"withExtantOnly"},
which will only 'shift' the time-scale if living taxa are present, as
determined by having ranges that overlap within \code{tolerance} of \code{sliceTime}.}

\item{modern.samp.prob}{The probability that a taxon is sampled at the modern time
(or, for \code{timeSliceFossilRecord}, the time at which the simulation data is
slice). Must be a number between 0 and 1. If 1, all taxa that survive to the modern
day (to the \code{sliceTime}) are sampled, if 0, none are.}

\item{tolerance}{A small number which sets a range around the \code{sliceTime} within
which taxa will be considered extant for the purposes of output.}

\item{fossilTaxa}{A \code{fossilTaxa} object, composed of a table
containing information on the true first and last appearance times of taxa,
as well as their ancestor-descendant relationships.}

\item{merge.cryptic}{If \code{TRUE}, sampling events for cryptic taxon-units (i.e.
those in the same cryptic complex) will be merged into sampling events for a single
taxon-unit (with the name of the first taxon in that cryptic complex).}

\item{ranges.only}{If \code{TRUE} (the default), \code{fossilRecord2fossilRanges}
will return the dates of the first and last sampled occurrences of each taxon-unit
(i.e. the stratigraphic range of each taxon). If \code{FALSE}, instead a list will be
output, with each element being a vector of dates for all sampling events of each taxon-unit.}
}
\value{
Depends on the function and the arguments given. See Details.
}
\description{
These are a set of functions available for manipulating, translating
and editing the objects of class \code{fossilRecordSimulation} output
from function \code{simFossilRecord}.
}
\details{
These functions exist to manipulate \code{fossilRecordSimulation} objects
output from \code{simFossilRecord}, particularly so that they can be interfaced
with functions in library \code{paleotree} in the same way that output from the
deprecated 'legacy' simulation function \code{simFossilTaxa} was used.

\code{timeSliceFossilRecord} takes a given \code{fossilRecordSimulation} object
and 'slices' the data to remove any events that occur after the given
\code{sliceTime} and make it so any taxa still alive as of \code{sliceTime}
are now listed as extant.

\code{fossilRecord2fossilTaxa} converts a \code{fossilRecordSimulation} object
to the flat table format of taxon data as was originally output by deprecated function 
\code{simFossilTaxa}, and can be taken as input by a number of \code{paleotree} functions such as
\code{sampleRanges},\code{taxa2phylo} and \code{taxa2cladogram}. 

\code{fossilTaxa2fossilRecord} does the reverse, converting a \code{simFossilTaxa}
table into a \code{fossilRecordSimulation} list object,
but returns a \code{fossilRecordSimulation} object that considers each species as un-sampled (as sampling
information is not contained within a \code{simFossilTaxa} table).

\code{fossilRecord2fossilRanges} converts a \code{fossilRecordSimulation} object
to the flat table format of observed taxon ranges, as is typically output by processing
\code{simFossilRecord} simulation output with \code{paleotree} function
\code{sampleRanges}.
}
\examples{

set.seed(44)
record <- simFossilRecord(p = 0.1, q = 0.1, r = 0.1, nruns = 1,
	nTotalTaxa = c(20,30) ,nExtant = 0, plot = TRUE)

# time-slicing

# let's try slicing this record at 940 time-units
slicedRecord <- timeSliceFossilRecord(fossilRecord = record, sliceTime = 940)
# and let's plot it
divCurveFossilRecordSim(slicedRecord)

# now with shiftRoot4TimeSlice = TRUE to shift the root age
slicedRecord <- timeSliceFossilRecord(fossilRecord = record, sliceTime = 940,
	shiftRoot4TimeSlice = TRUE)
# and let's plot it
divCurveFossilRecordSim(slicedRecord)

# plot look a little different due to how axis limits are treated...
# notice that in both, 'modern' (extant) taxa are sampled with probability = 1
	#let's try it again, make that probability = 0

# now with shiftRoot4TimeSlice = TRUE
slicedRecord <- timeSliceFossilRecord(fossilRecord = record, sliceTime = 940,
	shiftRoot4TimeSlice = TRUE, modern.samp.prob = 0)
# and let's plot it
divCurveFossilRecordSim(slicedRecord)

############################

# converting to taxa objects and observed ranges

# convert to taxa data
taxa <- fossilRecord2fossilTaxa(record)
# convert to ranges
ranges <- fossilRecord2fossilRanges(record)

# plot diversity curves with multiDiv
multiDiv(list(taxa,ranges),plotMultCurves = TRUE)
# should look a lot like what we got earlier

# get the cladogram we'd obtain for these taxa with taxa2cladogram
cladogram <- taxa2cladogram(taxa,plot = TRUE)

# now get the time-scaled phylogenies with taxa2phylo

# first, with tips extending to the true times of extinction
treeExt <- taxa2phylo(taxa,plot = TRUE)

# now, with tips extending to the first appearance dates (FADs) of taxa
	# get the FADs from the ranges
FADs <- ranges[,1]
treeFAD <- taxa2phylo(taxa,FADs,plot = TRUE)

}
\seealso{
\code{\link{simFossilRecord}}
}
\author{
David W. Bapst
}
