% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modifyTerminalBranches.R
\name{modifyTerminalBranches}
\alias{modifyTerminalBranches}
\alias{dropZLB}
\alias{dropExtinct}
\alias{dropExtant}
\alias{addTermBranchLength}
\alias{dropPaleoTip}
\alias{bindPaleoTip}
\title{Modify, Drop or Bind Terminal Branches of Various Types (Mainly for Paleontological Phylogenies)}
\usage{
dropZLB(tree)

dropExtinct(tree, tol = 0.01, ignore.root.time = FALSE)

dropExtant(tree, tol = 0.01)

addTermBranchLength(tree, addtime = 0.001)

dropPaleoTip(tree, ...)

bindPaleoTip(
  tree,
  tipLabel,
  nodeAttach = NULL,
  tipAge = NULL,
  edgeLength = NULL,
  positionBelow = 0,
  noNegativeEdgeLength = TRUE
)
}
\arguments{
\item{tree}{A phylogeny, as an object of class \code{phylo}.
\code{dropPaleoTip} requires this
input object to also have a \code{tree$root.time} element. If not provided for
\code{bindPaleoTip}, then the \code{$root.time} will be presumed to be such that the
furthest tip from the root is at \code{time = 0}.}

\item{tol}{Tolerance for determining modern age; used for distinguishing
extinct from extant taxa. Tips which end within \code{tol} of the furthest
distance from the root will be treated as 'extant' taxa for the purpose of
keeping or dropping.}

\item{ignore.root.time}{Ignore \code{tree$root.time} in calculating which tips are
extinct? \code{tree$root.time} will still be adjusted,
if the operation alters the \code{tree$root.time}.}

\item{addtime}{Extra amount of time to add to all terminal branch lengths.}

\item{...}{additional arguments passed to \code{dropPaleoTip} are passed to \code{\link{drop.tip}}.}

\item{tipLabel}{A character string of \code{length = 1} containing the name of the new tip
to be added to \code{tree}.}

\item{nodeAttach}{Node or tip ID number (as given in \code{tree$edge}) at which to attach the new tip. 
See documentation of \code{bind.tip} for more details.}

\item{tipAge}{The age of the tip taxon added to the tree, in time before present (i.e. where
present is 0), given in the same units as the edges of the tree are already scaled. Cannot be
given if \code{edgeLength} is given.}

\item{edgeLength}{The new \code{edge.length} of the terminal branch this tip is connected to.
Cannot be given if \code{tipAge} is given.}

\item{positionBelow}{The distance along the edge below the node to be attached to 
(given in \code{nodeAttach} to add the new tip. Cannot be negative or greater than the length of the
edge below \code{nodeAttach}.}

\item{noNegativeEdgeLength}{Return an error if a negative terminal edge length is calculated
for the new tip.}
}
\value{
Gives back a modified phylogeny as a \code{phylo} object, generally with a
modified \code{$root.time} element.
}
\description{
These functions modify terminal branches or drop certain terminal branches
based on various criteria.
\code{dropZLB} drops tip-taxa that are attached to the tree via 
zero-length terminal branches ("ZLBs"). 
This is sometimes useful for phylogenies of fossil taxa, as
various time-scaling methods often produce these 'ZLBs', taxa whose early
appearance causes them to be functionally interpreted as ancestors in some
time-scaling methods. Removing 'ZLBs' is advised for analyses of
diversification/diversity, as these will appear as simultaneous
speciation/extinction events. Note this function only drops tips attached to
a terminal zero-length branch; if you want to collapse internal zero-length
branches, see the ape function \code{\link{di2multi}}.
}
\details{
\code{dropExtinct} drops all terminal branches which end before the modern (i.e.
extinct taxa). \code{DropExtant} drops all terminal branches which end at the
modern (i.e. extant/still-living taxa). In both cases, the modern is defined
based on \code{tree$root.time} if available, or the modern is inferred to be the
point in time when the tip furthest from the root (the latest tip)
terminates.

If the input tree has a \code{$root.time} element,
as expected for most phylogeny containing fossil taxa
objects handled by this library, that \code{$root.time} is adjusted if the relative
time of the root divergence changes when terminal branches are dropped.
This is typically performed via the function \code{\link{fixRootTime}}.
Adjusted \code{$root.time} elements are only given if
the input tree has a \code{$root.time} element.

\code{addTermBranchLength} adds an amount equal to the argument \code{addtime} to the
terminal branch lengths of the tree. If there is a \code{$root.time} element, this
is increased by an amount equal to \code{addtime}. A negative amount can be input
to reduce the length of terminal branches. However, if negative branch
lengths are produced, the function fails and a warning is produced.
The function \code{addTermBranchLength} does \emph{not} call \code{fixRootTime},
so the root.time elements in the result tree may
be nonsensical, particularly if negative amounts are input.

\code{dropPaleoTip} is a wrapper for \code{ape}'s \code{\link{drop.tip}} which also modifies the
\code{$root.time} element if necessary, using \code{fixRootTime}. Similarly,
\code{bindPaleoTip} is a wrapper for phytool's \code{bind.tip} which allows tip age
as input and modifies the \code{$root.time} element if necessary (i.e. if a tip
is added to edge leading up to the root).

Note that for \code{bindPaleoTip}, tips added below the root are subtracted from
any existing \code{$root.edge} element,
as per behavior of \code{link{bind.tip}} and \code{\link{bind.tree}}.
However, \code{bindPaleoTip} will append a \code{$root.edge} of
the appropriate value (i.e., root edge length)
if one does not exist (or is not long enough) to avoid an error. After
binding is finished, any \code{$root.edge} equal to 0 is removed before the
resulting tree is output.
}
\examples{

set.seed(444)
# Simulate some fossil ranges with simFossilRecord
record <- simFossilRecord(
    p = 0.1, q = 0.1, 
    nruns = 1, 
    nTotalTaxa = c(30,40), 
    nExtant = 0
    )
taxa <- fossilRecord2fossilTaxa(record)
# simulate a fossil record 
    # with imperfect sampling with sampleRanges
rangesCont <- sampleRanges(taxa,r = 0.5)
# Now let's make a tree using taxa2phylo
tree <- taxa2phylo(taxa,obs_time = rangesCont[,2])
# compare the two trees
layout(1:2)
plot(ladderize(tree))
plot(ladderize(dropZLB(tree)))

# reset
layout(1)


# example using dropExtinct and dropExtant
set.seed(444)
record <- simFossilRecord(
    p = 0.1, q = 0.1, 
    nruns = 1, 
    nTotalTaxa = c(30,40), 
    nExtant = c(10,20)
    )
taxa <- fossilRecord2fossilTaxa(record)
tree <- taxa2phylo(taxa)
phyloDiv(tree)
tree1 <- dropExtinct(tree)
phyloDiv(tree1)
tree2 <- dropExtant(tree)
phyloDiv(tree2)


# example using addTermBranchLength
set.seed(444)
treeA <- rtree(10)
treeB <- addTermBranchLength(treeA,1)
compareTermBranches(treeA,treeB)

#########################
# test dropPaleoTip
	# (and fixRootTime by extension...)

# simple example
tree <- read.tree(text = "(A:3,(B:2,(C:5,D:3):2):3);")
tree$root.time <- 10
plot(tree, no.margin = FALSE)
axisPhylo()

# now a series of tests, dropping various tips
(test <- dropPaleoTip(tree,"A")$root.time) #  = 7
(test[2] <- dropPaleoTip(tree,"B")$root.time) #  = 10
(test[3] <- dropPaleoTip(tree,"C")$root.time) #  = 10
(test[4] <- dropPaleoTip(tree,"D")$root.time) #  = 10
(test[5] <- dropPaleoTip(tree,c("A","B"))$root.time) #  = 5
(test[6] <- dropPaleoTip(tree,c("B","C"))$root.time) #  = 10
(test[7] <- dropPaleoTip(tree,c("A","C"))$root.time) #  = 7
(test[8] <- dropPaleoTip(tree,c("A","D"))$root.time) #  = 7

# is it all good? if not, fail so paleotree fails...
if(!identical(test,c(7,10,10,10,5,10,7,7))){
     stop("fixRootTime fails!")
     }


##############
# testing bindPaleoTip

# simple example 
tree <- read.tree(text = "(A:3,(B:2,(C:5,D:3):2):3);")
tree$root.time <- 20
plot(tree, no.margin = FALSE)
axisPhylo()

\dontrun{

require(phytools)

# bindPaleoTip effectively wraps bind.tip from phytools
# using a conversion like below

tipAge <- 5
node <- 6

# the new tree length (tip to root depth) should be:
# new length = the root time - tipAge - nodeheight(tree,node)

newLength <- tree$root.time-tipAge-nodeheight(tree,node)
tree1 <- bind.tip(tree,
    "tip.label",
    where = node,\
    edge.length = newLength)

layout(1:2)
plot(tree)
axisPhylo()
plot(tree1)
axisPhylo()

# reset
layout(1)

}

# now with bindPaleoTip

tree1 <- bindPaleoTip(tree,"new",nodeAttach = 6,tipAge = 5)

layout(1:2)
plot(tree)
axisPhylo()
plot(tree1)
axisPhylo()

# reset
layout(1)

#then the tip age of "new" should 5
test <- dateNodes(tree1)[which(tree1$tip.label == "new")] == 5
if(!test){
    stop("bindPaleoTip fails!")
    }

# with positionBelow

tree1 <- bindPaleoTip(
    tree,
    "new",
    nodeAttach = 6,
    tipAge = 5,
    positionBelow = 1
    )

layout(1:2)
plot(tree)
axisPhylo()
plot(tree1)
axisPhylo()

# reset
layout(1)

# at the root

tree1 <- bindPaleoTip(
    tree,
    "new", 
    nodeAttach = 5,
    tipAge = 5)

layout(1:2)
plot(tree)
axisPhylo()
plot(tree1)
axisPhylo()

# reset
layout(1)

#then the tip age of "new" should 5
test <- dateNodes(tree1)[which(tree1$tip.label == "new")] == 5
if(!test){
     stop("bindPaleoTip fails!")
     }

# at the root with positionBelow

tree1 <- bindPaleoTip(tree,"new",nodeAttach = 5,tipAge = 5,
	positionBelow = 3)

layout(1:2)
plot(tree)
axisPhylo()
plot(tree1)
axisPhylo()

# reset
layout(1)

#then the tip age of "new" should 5
test <- dateNodes(tree1)[which(tree1$tip.label == "new")] == 5
#and the root age should be 23
test1 <- tree1$root.time == 23
if(!test | !test1){
     stop("bindPaleoTip fails!")
     }

}
\seealso{
\code{\link{compareTermBranches}}, \code{\link{phyloDiv}}, 
\code{\link{drop.tip}}, \code{\link{bind.tip}}
}
\author{
David W. Bapst. The functions \code{dropTipPaleo} and \code{bindTipPaleo} are modified imports of
\code{\link{drop.tip}} and \code{\link{bind.tip}} from packages \code{ape} and \code{phytools}.
}
