#' plot_phenology plots the phenology.
#' @title Plot the phenology from a result.
#' @author Marc Girondot
#' @return Return A map generated with map_phenology
#' @param result A result file generated by fit_phenology
#' @param data A dataset generated by add_format
#' @param pdf TRUE or FALSE, indicates if a pdf file is generated.
#' @param parametersfixed Set of fixed parameters
#' @param parameters Set of parameters to be changed
#' @param series Number of series to be analyzed or 'all'
#' @param ... Parameters used for graphics
#' @param help If TRUE, an help is displayed
#' @description The function "plot_phenology" plots the phenology graph from a result.
#' @examples
#' library(phenology)
#' # Read a file with data
#' # Gratiot<-read.delim("http://max2.ese.u-psud.fr/epc/conservation/BI/Complete.txt", , header=FALSE)
#' data(Gratiot)
#' # Generate a formatted list nammed data_Gratiot 
#' data_Gratiot<-add_format(origin=NULL, add=Gratiot, name="Complete", reference=as.Date("2001-01-01"), format="%d/%m/%Y")
#' # Generate initial points for the optimisation
#' parg<-par_init(data_Gratiot, parametersfixed=NULL)
#' # Run the optimisation
#' # result_Gratiot<-fit_phenology(data=data_Gratiot, parametersfit=parg, parametersfixed=NULL, trace=1)
#' data(result_Gratiot)
#' # Plot the phenology and get some stats
#' plot_phenology(result=result_Gratiot, pdf=FALSE)
#' @export



plot_phenology <-
function(result=NULL, data=NULL, parameters=NULL, parametersfixed=NA, series="all", pdf=FALSE, help=FALSE, ...) {

if(help || (is.null(data) && is.null(result))) {
	cat("The syntaxe of this function is :\n")
	cat("plot_phenology(result) or \n")
	cat("plot_phenology(result=res, series=x, pdf=TRUE, ...)\n")
	cat("with res being the output of fit_phenology.\n")
	cat("series is used to indicate the series to plot.\n")
	cat("If only the data must be printed, use :\n")
	cat("plot_phenology(data=dataset, series=x, pdf=TRUE, ...)\n")
	cat("x can be a vector with all the series to be printed, for example\n")
	cat("plot_phenology(result=res, series=c(1,3,4), pdf=TRUE, ...)\n")
	cat("x can be 'all' to print all the series at one time. In this\n")
	cat("case pdf should be TRUE if graphs are necessary.\n")
	cat("Optional parameters ... are:\n")
	cat("xlab: Label for X axis [default Days]\n")
	cat("ylab: Label for Y axis [default Number]\n")
	cat("ylim: maximum value for Y axis [default estimated from data]\n")
	cat("cex: size of the points [default 0.5]\n")
	cat("pch: symbol of the points [default 16]\n")
	
} else {

pnp<-c(...)
pnp<-ifelse(is.null(pnp), NA, pnp)

if (is.null(data))  {data<-result$data}

if (series=="all") {series<-c(1:length(data))}

for(kseries in 1:length(series)) {

reference<-data[[series[kseries]]]$Date[1]-(data[[series[kseries]]]$ordinal[1]+1)

if (!is.null(result)) {
	# si il y en a pas estimable, je les mets à 0
	res_se<-result$se

	## je stocke les données ajustées dans parres
	parres<-result$par
	parametersfixed<-result$parametersfixed
	
} else {
	# je n ai pas de result, je lis parameters
	parres<-parameters
	res_se<-0
}


res_se[is.na(res_se)]<-0


# length(result$par) donne le nombre de paramètres pour lesquels on a result
# length(parametersfixed) donne le nombre de paramètres fixés
# je passe les parametersfixed en revue pour savoir si le sd# est fourni pour ce paramètre

if (!all(is.na(parametersfixed))) {

for(i in 1:length(parametersfixed)) {
	nm<-names(parametersfixed[i])
# est-ce que j ai un sd pour ce paramètre et le paramètre
# si oui je stocke les deux
# sinon, je mets le sd à 0

	if (substr(nm,1,3)!="sd#")
	{
		parres<-c(parres, parametersfixed[i])
		if (is.na(parametersfixed[paste("sd#", nm, sep="")])) {
# je n'ai pas de sd pour ce paramètre
			res_se<-c(res_se, 0)
		} else {
# j'ai un sd pour ce paramètre		
			res_se<-c(res_se, parametersfixed[paste("sd#", nm, sep="")])
		}
	}
}

}

## crée un tableau dans lequel on met ce qu'on va utiliser pour le graph
val=matrix(c(0:364, rep(NA, 365), rep(NA, 365), rep(NA, 365), rep(NA, 365), rep(NA, 365), rep(NA, 365)), ncol=7)

## On le nomme, c'est plus simple
colnames(val)=c("days", "Obs", "Theor", "Theor-2SE", "Theor+2SE", "Theor-2SD", "Theor+2SD")

if (is.null(data)) {nmser=""} else {nmser<-names(data[series[kseries]])}

if (!is.null(parres)) {

## crée un tableau avec des réplicats des modèles
par2=matrix(rep(NA, length(parres)*1000), ncol=length(parres))

## On nomme les colonnes comme les noms des paramètres
colnames(par2)=names(parres)


# on générère 1000 jeux de paramètres à partir des paramètres obtenus en vraisemblance
# ici il faut que je prenne aussi ceux transmis en parmetersfixed
for(i in 1:length(parres)) par2[,i]=rnorm(1000, mean=as.numeric(parres[i]), sd=res_se[i])

## je génère une matrice de 1000 saisons de pontes
ponte2=matrix(rep(NA, 365*1000), ncol=365)

cat("\n")

pb<-txtProgressBar(min=1, max=1000, style=3)



## je remplis les 1000 saisons de ponte
for(j in 1:1000) {

setTxtProgressBar(pb, j)

# j'ai tous les paramètres dans xpar
# maintenant tous les paramètre fixés appraissent dans resfit
	xparec<-.format_par(par2[j,], nmser)
	
#	for(i in 1:365) {
#	print(i)
#		ponte2[j,i]=.daily_count(i, xparec, print=FALSE)
#	}
	
	ponte2[j,1:365]=.daily_count(1:365, xparec, print=FALSE)
	
	
# je viens de générer les pontes du jour j
}

cat("\n")
cat(nmser, "\n", sep="")
cat(rep("-", nchar(nmser)), "\n", sep="")

## je calcule les écart-types des nb de pontes générées chaque jour
## sd2=sd(ponte2)
sd2<-apply(ponte2, 2, sd)

mnponte<-mean(apply(ponte2, 1, sum))
sdponte<-sd(apply(ponte2, 1, sum))
cat("Estimation without the observed data\n")
if (sdponte!=0) {
cat("Total number of counts: ", format(mnponte, digits=floor(log10(mnponte)+4)), " ; SD ", format(sdponte, digits=floor(log10(sdponte)+4)), "\n", sep="")
} else {
cat("Total number of counts: ", format(mnponte, digits=floor(log10(mnponte)+4)), " ; SD 0.000\n", sep="")
}

cat("Estimation taking into account the observed data\n")

# dans ponte2[nbsimul 1:1000, jour 1:365] j'ai la donnée théorique
for(i in 1:dim(data[[series[kseries]]])[1]) {
		if (!is.na(data[[series[kseries]]]$ordinal2[i])) {
			for(j in (1+data[[series[kseries]]]$ordinal[i]):data[[series[kseries]]]$ordinal2[i]) {
# plus la peine, vérifié maintenant à l'entrée des données: 4/2/2012
#				if ((j<1)||(j>365)) {
#					print(paste("Error in the date ",data[[series[kseries]]]$date[i], " or ",
# data[[series[kseries]]]$date2[i], "; please check", sep=""))
#				} else {
					ponte2[1:1000, j]<-0
#				}
			}
		}
}

for(i in 1:dim(data[[series[kseries]]])[1]) {
	ponte2[1:1000, data[[series[kseries]]]$ordinal[i]]<-data[[series[kseries]]]$nombre[i]
}


mnponte<-mean(apply(ponte2, 1, sum))
sdponte<-sd(apply(ponte2, 1, sum))
if (sdponte!=0) {
cat("Total number of counts: ", format(mnponte, digits=floor(log10(mnponte)+4)), " ; SD ", format(sdponte, digits=floor(log10(sdponte)+4)), "\n", sep="")
} else {
cat("Total number of counts: ", format(mnponte, digits=floor(log10(mnponte)+4)), " ; SD 0.000\n", sep="")
}



## je remplis le tableau val avec les nb théoriques

xparec<-.format_par(parres, nmser)

val[1:365, "Theor"]=.daily_count(1:365, xparec, print=FALSE)

## je remplis le tableau val avec les nb théoriques +/- 2 SD
for(i in 1:365) {val[i, "Theor-2SE"]=max(0, val[i, "Theor"]-2*sd2[i])}
val[1:365, "Theor+2SE"]=val[1:365, "Theor"]+2*sd2[1:365]


## je calcule la distribution théorique des points minimaux
for(i in 1:365) {val[i, "Theor-2SD"]=max(c(subset(0:trunc(3*val[i, "Theor"]), pnbinom(0:trunc(3*val[i, "Theor"]), size=abs(as.numeric(xparec["Theta"])), mu=val[i, "Theor"])<=0.05), 0))}

## je calcule la distribution théoriques des points maximaux
for(i in 1:365) {val[i, "Theor+2SD"]=min(c(subset(0:trunc(3*val[i, "Theor"]), pnbinom(0:trunc(3*val[i, "Theor"]), size=abs(as.numeric(xparec["Theta"])), mu=val[i, "Theor"])>=0.95), trunc(3*val[i, "Theor"])+1))}


}

vmaxx<-c(reference, reference+364)

if (is.na(pnp['ylim'])) {

if ((!is.null(data)) && (!is.null(parres))) {
	vmaxy<-c(0, max(val[, "Theor+2SD"], data[[series[kseries]]]$nombre[(is.na(data[[series[kseries]]]$ordinal2)) & (!is.na(data[[series[kseries]]]$nombre))]))
} else {
	if (!is.null(data)) {
		vmaxy<-c(0, max(data[[series[kseries]]]$nombre[(is.na(data[[series[kseries]]]$ordinal2)) & (!is.na(data[[series[kseries]]]$nombre))]))
	} else {
		vmaxy<-c(0, max(val[, "Theor+2SD"]))
	}
}
} else {

	vmaxy<-ifelse(length(pnp['ylim'])==1, c(0, pnp['ylim']), pnp['ylim'])
}

x<-seq(from=reference, to=reference+364, by="1 day")

xlab<-ifelse(is.na(pnp['xlab']), "Days", pnp['xlab'])
ylab<-ifelse(is.na(pnp['ylab']), "Number", pnp['ylab'])
pch<-ifelse(is.na(pnp['pch']), 16, pnp['pch'])
cex<-ifelse(is.na(pnp['cex']), 0.5, pnp['cex'])


if (pdf) {
pdf(paste(names(data[series[kseries]]),".pdf", sep=""))


## je fais les graphiques
## Pour les dates seules
par(new=FALSE);



plot(x, rep(-1, 365) , type="l", xlim=vmaxx, ylim=vmaxy, bty="n", xlab=xlab, ylab=ylab)

par(new=TRUE);


if (!is.null(data)) {
plot(data[[series[kseries]]]$Date[is.na(data[[series[kseries]]]$Date2)], data[[series[kseries]]]$nombre[is.na(data[[series[kseries]]]$Date2)] , 
	type="p", xlim=vmaxx, ylim=vmaxy, xlab="", ylab="", axes=FALSE, bty="n", cex=cex, col="black", pch=pch)


## Pour les dates avec incertitudes
par(new=TRUE);
for(i in 1:dim(data[[series[kseries]]])[1]) {
	if (!is.na(data[[series[kseries]]]$ordinal2[i])) {
		x0<-data[[series[kseries]]]$Date[i]
		x1<-data[[series[kseries]]]$Date2[i]
		lgt01<-as.numeric(data[[series[kseries]]]$Date2[i]-data[[series[kseries]]]$Date[i]+1)
		y0<-data[[series[kseries]]]$nombre[i]/lgt01
		y1<-y0
		segments(x0, y0, x1=x1, y1=y1, col="green", lwd=2)
	}
}



par(new=TRUE);
}

if (!is.null(parres)) {

plot((reference+val[, "days"]),val[, "Theor"] , type="l", xlim=vmaxx, ylim=vmaxy,  xlab="", ylab="", axes = FALSE, bty="n");
par(new=TRUE);
plot((reference+val[, "days"]),val[, "Theor-2SE"] , type="l", xlim=vmaxx, ylim=vmaxy, xlab="", ylab="", axes = FALSE, lty=2, bty="n");
par(new=TRUE);
plot((reference+val[, "days"]),val[, "Theor+2SE"],  type="l", xlim=vmaxx, ylim=vmaxy, xlab="", ylab="", axes = FALSE, lty=2, bty="n");
par(new=TRUE);
plot((reference+val[, "days"]),val[, "Theor-2SD"],  type="l", xlim=vmaxx, ylim=vmaxy, xlab="", ylab="", axes = FALSE, lty=2, bty="n", col="red");
par(new=TRUE);
plot((reference+val[, "days"]), val[, "Theor+2SD"], type="l", xlim=vmaxx, ylim=vmaxy, xlab="", ylab="", axes = FALSE, lty=2, bty="n", col="red");
}

mtext(names(data[series[kseries]]), side=3)

dev.off()

}

## je fais les graphiques
## Pour les dates seules
par(new=FALSE);

plot(x, rep(-1, 365) , type="l", xlim=vmaxx, ylim=vmaxy, bty="n", xlab=xlab, ylab=ylab)

par(new=TRUE);


if (!is.null(data)) {
plot(data[[series[kseries]]]$Date[is.na(data[[series[kseries]]]$Date2)], data[[series[kseries]]]$nombre[is.na(data[[series[kseries]]]$Date2)] , 
	type="p", xlim=vmaxx, ylim=vmaxy, xlab="", ylab="", axes=FALSE, bty="n", cex=cex, col="black", pch=pch)


## Pour les dates avec incertitudes
par(new=TRUE);
for(i in 1:dim(data[[series[kseries]]])[1]) {
	if (!is.na(data[[series[kseries]]]$ordinal2[i])) {
		x0<-data[[series[kseries]]]$Date[i]
		x1<-data[[series[kseries]]]$Date2[i]
		lgt01<-as.numeric(data[[series[kseries]]]$Date2[i]-data[[series[kseries]]]$Date[i]+1)
		y0<-data[[series[kseries]]]$nombre[i]/lgt01
		y1<-y0
		segments(x0, y0, x1=x1, y1=y1, col="green", lwd=2)
	}
}



par(new=TRUE);
}

if (!is.null(parres)) {

plot((reference+val[, "days"]),val[, "Theor"] , type="l", xlim=vmaxx, ylim=vmaxy,  xlab="", ylab="", axes = FALSE, bty="n");
par(new=TRUE);
plot((reference+val[, "days"]),val[, "Theor-2SE"] , type="l", xlim=vmaxx, ylim=vmaxy, xlab="", ylab="", axes = FALSE, lty=2, bty="n");
par(new=TRUE);
plot((reference+val[, "days"]),val[, "Theor+2SE"],  type="l", xlim=vmaxx, ylim=vmaxy, xlab="", ylab="", axes = FALSE, lty=2, bty="n");
par(new=TRUE);
plot((reference+val[, "days"]),val[, "Theor-2SD"],  type="l", xlim=vmaxx, ylim=vmaxy, xlab="", ylab="", axes = FALSE, lty=2, bty="n", col="red");
par(new=TRUE);
plot((reference+val[, "days"]), val[, "Theor+2SD"], type="l", xlim=vmaxx, ylim=vmaxy, xlab="", ylab="", axes = FALSE, lty=2, bty="n", col="red");
}
mtext(names(data[series[kseries]]), side=3)


}

}

	growlnotify('Plot is done!')


}
