% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/test_pathway.R
\name{test_pathway}
\alias{test_pathway}
\title{Pathway differential analysis based on longitudinal metabolomics data}
\usage{
test_pathway(
  quantif,
  design,
  pathways = "auto",
  fixed,
  random,
  npc = 1L,
  model = c("lmer", "blmer"),
  organism = NULL,
  min_size = 2
)
}
\arguments{
\item{quantif}{data.frame or matrix of the metabolite quantification with 
samples in rows (sample identifiers must be row names) and metabolites in 
columns (metabolite codes must be column names)}

\item{design}{data.frame or matrix with samples in rows (sample identifiers
must be row names) and the different effects to be included in the model
in columns. Column names must be provided and are used as arguments for
\code{fixed} and \code{random}}

\item{pathways}{data.frame or matrix with metabolites in rows (all 
metabolites in columns of \code{quantif} must have a row in this input) and the
following information in columns: \itemize{
 \item \code{metabolite_code} metabolite code
 \item \code{metabolite_name} metabolite name
 \item \code{pathway_code} pathway code (identifier)
 \item \code{pathway_name} name of the pathway
}}

\item{fixed}{character vector of fixed effects to be included in the model.
They must correspond to column names of \code{design}}

\item{random}{character vector of random effects to be included in the model.
They must correspond to column names of \code{design}}

\item{npc}{number of PCs for the PCA step}

\item{model}{a character string indicating if the model has to be fitted
using \link[lme4]{lmer} or \link[blme]{blmer}. Default to \code{"lmer"}}

\item{organism}{organism code in KEGG database. Required if
\code{pathways = "auto"} and ignored otherwise}

\item{min_size}{minimal number of metabolites in a pathway. Required if
\code{pathways = "auto"} and ignored otherwise. Default to \code{2}}
}
\value{
an object of class \code{pathwayRes} which is a list of pathway 
results. Each element of the list contain the following entries:
\item{pathway_name}{a character corresponding to the pathway name}
\item{pathway_code}{a character corresponding to the pathway code}
\item{metabolites}{a data.frame with the names and codes of the quantified
metabolites in the pathway}
\item{PCA}{the result of the pathway PCA (a \code{PCA} object as obtained 
from \link[FactoMineR]{PCA})}
\item{model}{the output of the mixed model fit}
\item{test_pathway}{a data.frame with the p-values for each tested fixed
effect}
}
\description{
Perform a differential analysis at pathway level based on 
metabolite quantifications and information on pathway metabolite composition.
The method relies on a PCA step.
}
\details{
If \code{pathways = "auto"}, information on pathways in which metabolites are
present is automatically obtained by the function \link{pathway_search} using
\code{KEGGREST} that queries KEGG database. Results are specific to a given
organism (passed in \code{organism}). Pathways containing less than
\code{min_size} metabolites are filtered out.
}
\examples{
data("MTBLS422")
quantif <- from_ASICS_to_PHOENICS(quantif)
out_test <- test_pathway(quantif, design, pathways, 
                         fixed = c("Age", "Treatment"), random = "Mouse", 
                         npc = 2, model = "blmer")
out_test

if (requireNamespace("KEGGREST", quietly = TRUE)) { 
\donttest{
  out_test2 <- test_pathway(quantif, design, pathways = "auto", 
                            fixed = c("Age", "Treatment"), random = "Mouse", 
                            npc = 2, model = "blmer", organism = "mmu")
  out_test2
 }
}

}
\seealso{
\link{pathway_search} for the automatic annotation of metabolites in KEGG 
pathways.
}
\author{
Camille Guilmineau <camille.guilmineau@inrae.fr>\cr
Remi Servien <remi.servien@inrae.fr>\cr
Nathalie Vialaneix <nathalie.vialaneix@inrae.fr>
}
