% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/conductance.R
\encoding{UTF-8}
\name{CO2_conductance}
\alias{CO2_conductance}
\alias{.get_gtc}
\alias{.get_guc}
\alias{.get_gbc}
\alias{.get_gmc}
\alias{.get_gsc}
\title{Conductance to CO2 (umol / (m^2 s Pa))}
\usage{
.get_gtc(pars, unitless)

.get_guc(pars, surface, unitless)

.get_gbc(pars, surface, unitless)

.get_gmc(pars, surface, unitless)

.get_gsc(pars, surface, unitless)
}
\arguments{
\item{pars}{Concatenated parameters (\code{leaf_par}, \code{enviro_par}, and \code{constants})}

\item{unitless}{Logical. Should \code{units} be set? The function is faster when FALSE, but input must be in correct units or else results will be incorrect without any warning.}

\item{surface}{Leaf surface (lower or upper)}
}
\description{
Conductance to CO2 (umol / (m^2 s Pa))

- g_tc: total conductance to CO2

- g_uc: cuticular conductance to CO2

- g_bc: boundary layer conductance to CO2

- g_mc: mesophyll conductance to CO2

- g_sc: stomatal conductance to CO2
}
\details{
Total conductance to CO2 is the sum of parallel conductances on the lower (\eqn{g_\mathrm{c,lower}}{gc_lower}) and upper (\eqn{g_\mathrm{c,upper}}{gc_upper}) leaf portions:

 \deqn{g_\mathrm{c,total} = g_\mathrm{c,lower} + g_\mathrm{c,upper}}{gc_total = gc_lower + gc_upper}
 
Each partial conductance consists of two parallel conductances, the cuticular conductance (\eqn{g_\mathrm{u,c}}{g_uc}) and the in-series conductances through mesophyll (\eqn{g_\mathrm{m,c}}{g_mc}), stomata (\eqn{g_\mathrm{s,c}}{g_sc}), and boundary layer (\eqn{g_\mathrm{b,c}}{g_bc}). To simplify the formula, I use substitute resistance where \eqn{r_x = 1 / g_x}. For surface \eqn{i}:

\deqn{g_{\mathrm{c},i} = g_{\mathrm{u},i} + (1 / (r_{\mathrm{m},i} + r_{\mathrm{s},i} + r_{\mathrm{b},i}))}{g_ci = g_ui + (1 / (r_mi + r_si + r_bi))}

The cuticular, stomatal, and mesophyll conductances can be the same or different for upper and lower. The partitioning factors (\eqn{k_x}) divide the conductance between surfaces while keeping the total conductance constant:

\deqn{g_{x,\mathrm{lower}} = g_x (1 / (1 + k_x))}{gx_lower = g_x (1 / (1 + k_x))}
\deqn{g_{x,\mathrm{upper}} = g_x (k_x / (1 + k_x))}{gx_upper = g_x (k_x / (1 + k_x))}
\deqn{g_x = g_{x,\mathrm{lower}} + g_{x,\mathrm{upper}}}{g_x = gx_lower + gx_upper}

How the partitioning factors work: \cr
\tabular{ll}{
\eqn{k_x} \tab description \cr
0 \tab all conductance on \bold{lower} surface/portion \cr
0.5 \tab 2/3 conductance on \bold{lower} surface \cr
1 \tab conductance evenly divided between surfaces/portions \cr
2 \tab 2/3 conductance on \bold{upper} surface \cr
Inf \tab all conductance on \bold{upper} surface/portion
}

The boundary layer conductances for each are calculated on the basis of mass and heat transfer (see \code{\link{.get_gbc}}).

\tabular{lllll}{
\emph{Symbol} \tab \emph{R} \tab \emph{Description} \tab \emph{Units} \tab \emph{Default}\cr
\eqn{g_\mathrm{mc}}{g_mc} \tab \code{g_mc} \tab mesophyll conductance to CO2 (T_leaf) \tab \eqn{\mu}mol CO2 / (m\eqn{^2} s Pa) \tab \link[=bake]{calculated} \cr
\eqn{g_\mathrm{sc}}{g_sc} \tab \code{g_sc} \tab stomatal conductance to CO2 \tab \eqn{\mu}mol CO2 / (m\eqn{^2} s Pa) \tab 4 \cr
\eqn{g_\mathrm{uc}}{g_uc} \tab \code{g_uc} \tab cuticular conductance to CO2 \tab \eqn{\mu}mol CO2 / (m\eqn{^2} s Pa) \tab 0.1 \cr
\eqn{k_\mathrm{mc}}{k_mc} \tab \code{k_mc} \tab partition of \eqn{g_\mathrm{mc}}{g_mc} to lower mesophyll \tab none \tab 1 \cr
\eqn{k_\mathrm{sc}}{k_sc} \tab \code{k_sc} \tab partition of \eqn{g_\mathrm{sc}}{g_sc} to lower surface \tab none \tab 1 \cr
\eqn{k_\mathrm{uc}}{k_uc} \tab \code{k_uc} \tab partition of \eqn{g_\mathrm{uc}}{g_uc} to lower surface \tab none \tab 1 \cr
}
}
\examples{
# Total conductance to CO2

library(magrittr)
library(photosynthesis)

## Hypostomatous leaf; default parameters
bake_par <- make_bakepar()
constants <- make_constants(use_tealeaves = FALSE)
enviro_par <- make_enviropar(use_tealeaves = FALSE)
leaf_par <- make_leafpar(replace = list(k_sc = set_units(0)),
                         use_tealeaves = FALSE)
leaf_par \%<>\% bake(bake_par, constants)
enviro_par$T_air <- leaf_par$T_leaf
pars <- c(leaf_par, enviro_par, constants)

gbw_upper <- photosynthesis:::.get_gbc(pars, "upper", FALSE)

# Lower surface ----
gbc_lower <- photosynthesis:::.get_gbc(pars, "lower", FALSE)
gmc_lower <- photosynthesis:::.get_gmc(pars, "lower", FALSE)
gsc_lower <- photosynthesis:::.get_gsc(pars, "lower", FALSE)
guc_lower <- photosynthesis:::.get_guc(pars, "lower", FALSE)

rc_lower <- 1 / gmc_lower + 1 / gsc_lower + 1 / gbc_lower
gc_lower <- 1 / rc_lower
gc_lower \%<>\% magrittr::add(guc_lower)

# Upper surface ----
gbc_upper <- photosynthesis:::.get_gbc(pars, "upper", FALSE)
gmc_upper <- photosynthesis:::.get_gmc(pars, "upper", FALSE)
gsc_upper <- photosynthesis:::.get_gsc(pars, "upper", FALSE)
guc_upper <- photosynthesis:::.get_guc(pars, "upper", FALSE)

rc_upper <- 1 / gmc_upper + 1 / gsc_upper + 1 / gbc_upper
gc_upper <- 1 / rc_upper
gc_upper \%<>\% magrittr::add(guc_upper)

## Lower and upper surface are in parallel
g_tc <- gc_lower + gc_upper

g_tc

}
