% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tbl_store.R
\name{tbl_source}
\alias{tbl_source}
\title{Obtain a table-prep formula from a table store}
\usage{
tbl_source(tbl, store = NULL)
}
\arguments{
\item{tbl}{The table name associated with a table-prep formula. This is part
of the table \code{store}. This table could be identified by its name (e.g.,
\code{tbl = "large_table"}) or by supplying a reference using a subset (with
\code{$}) of the \code{tbl_store} object (e.g., \code{tbl = store$large_table}). If using
the latter method then nothing needs to be supplied to \code{store}.}

\item{store}{Either a table store object created by the \code{\link[=tbl_store]{tbl_store()}}
function or a path to a table store YAML file created by \code{\link[=yaml_write]{yaml_write()}}.}
}
\value{
A table-prep formula.
}
\description{
The \code{tbl_source()} function provides a convenient means to access a
table-prep formula from either a \code{tbl_store} object or a table store YAML
file (which can be created with the \code{\link[=yaml_write]{yaml_write()}} function). A call to
\code{tbl_source()} is most useful as an input to the \code{tbl} argument of
\code{\link[=create_agent]{create_agent()}}, \code{\link[=create_informant]{create_informant()}}, or \code{\link[=set_tbl]{set_tbl()}}.

Should you need to obtain the table itself (that is generated via the
table-prep formula), then the \code{\link[=tbl_get]{tbl_get()}} function should be used for that.
}
\section{Function ID}{

1-9
}

\examples{
if (interactive()) {

# Let's create a `tbl_store` object by
# giving two table-prep formulas to
# `tbl_store()`
tbls <- 
  tbl_store(
    small_table_duck ~ db_tbl(
      table = small_table,
      dbname = ":memory:",
      dbtype = "duckdb"
    ),
    sml_table ~ pointblank::small_table
  )

# We can pass a table-prep formula
# to `create_agent()` and interrogate
# the table shortly thereafter
agent <- 
  create_agent(
    tbl = ~ tbl_source("sml_table", tbls),
    label = "An example that uses a table store.",
    actions = action_levels(warn_at = 0.10)
  ) \%>\% 
  col_exists(vars(date, date_time)) \%>\%
  interrogate()

# Both the `tbl_store` object and the
# `agent` can be transformed to YAML with
# the `yaml_write()` function

# This writes the `tbl_store.yml` file
# by default (but a different name
# could be used)
yaml_write(tbls)

# Let's modify the agent's target
# to point to the table labeled as
# `"sml_table"` in the YAML
# representation of the `tbl_store`
agent <-
  agent \%>\% 
  set_tbl(
    ~ tbl_source(
        tbl = "sml_table",
        store = "tbl_store.yml"
      )
  )

# Then we can write agent to a YAML
# file (writes to `agent-sml_table.yml`
# by default)
yaml_write(agent)

# Now that both are in this on-disk format
# an interrogation can be done by accessing
# the agent YAML
agent <-
  yaml_agent_interrogate(
    filename = "agent-sml_table.yml"
  )

}

}
\seealso{
Other Planning and Prep: 
\code{\link{action_levels}()},
\code{\link{create_agent}()},
\code{\link{create_informant}()},
\code{\link{db_tbl}()},
\code{\link{draft_validation}()},
\code{\link{file_tbl}()},
\code{\link{scan_data}()},
\code{\link{tbl_get}()},
\code{\link{tbl_store}()},
\code{\link{validate_rmd}()}
}
\concept{Planning and Prep}
