% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/policy_tree.R
\name{predict.policy_tree}
\alias{predict.policy_tree}
\title{Predict method for policy_tree}
\usage{
\method{predict}{policy_tree}(object, newdata, type = c("action.id", "node.id"), ...)
}
\arguments{
\item{object}{policy_tree object}

\item{newdata}{Points at which predictions should be made. Note that this matrix should have the
same number of columns as the training matrix, and that the columns must appear in the same order.}

\item{type}{The type of prediction required, "action.id" is the action id and
"node.id" is the integer id of the leaf node the sample falls into. Default is "action.id".}

\item{...}{Additional arguments (currently ignored).}
}
\value{
A vector of predictions. For type = "action.id" each element is an integer from 1 to d where d is
the number of columns in the reward matrix. For type = "node.id" each element is an integer corresponding
to the node the sample falls into (level-ordered).
}
\description{
Predict values based on fitted policy_tree object.
}
\examples{
\donttest{
# Construct doubly robust scores using a causal forest.
n <- 10000
p <- 10
# Discretizing continuous covariates decreases runtime for policy learning.
X <- round(matrix(rnorm(n * p), n, p), 2)
colnames(X) <- make.names(1:p)
W <- rbinom(n, 1, 1 / (1 + exp(X[, 3])))
tau <- 1 / (1 + exp((X[, 1] + X[, 2]) / 2)) - 0.5
Y <- X[, 3] + W * tau + rnorm(n)
c.forest <- grf::causal_forest(X, Y, W)

# Retrieve doubly robust scores.
dr.scores <- double_robust_scores(c.forest)

# Learn a depth-2 tree on a training set.
train <- sample(1:n, n / 2)
tree <- policy_tree(X[train, ], dr.scores[train, ], 2)
tree

# Evaluate the tree on a test set.
test <- -train

# One way to assess the policy is to see whether the leaf node (group) the test set samples
# are predicted to belong to have mean outcomes in accordance with the prescribed policy.

# Get the leaf node assigned to each test sample.
node.id <- predict(tree, X[test, ], type = "node.id")

# Doubly robust estimates of E[Y(control)] and E[Y(treated)] by leaf node.
values <- aggregate(dr.scores[test, ], by = list(leaf.node = node.id),
                    FUN = function(dr) c(mean = mean(dr), se = sd(dr) / sqrt(length(dr))))
print(values, digits = 1)

# Take cost of treatment into account by, for example, offsetting the objective
# with an estimate of the average treatment effect.
ate <- grf::average_treatment_effect(c.forest)
cost.offset <- ate[["estimate"]]
dr.scores[, "treated"] <- dr.scores[, "treated"] - cost.offset
tree.cost <- policy_tree(X, dr.scores, 2)

# Predict treatment assignment for each sample.
predicted <- predict(tree, X)

# If there are too many covariates to make tree search computationally feasible, then one
# approach is to consider for example only the top features according to GRF's variable importance.
var.imp <- grf::variable_importance(c.forest)
top.5 <- order(var.imp, decreasing = TRUE)[1:5]
tree.top5 <- policy_tree(X[, top.5], dr.scores, 2, split.step = 50)
}
}
