\name{ratiocalc}
\alias{ratiocalc}

\title{Calculation of ratios/propagated errors/confidence intervals/permutation p-values from qPCR runs with/without reference data}

\description{For multiple qPCR data from type 'pcrbatch', this function calculates ratios between two samples, 
 using normalization against a reference gene, if supplied. The input can be single qPCR data or (more likely) data containing replicates. 
Errors and confidence intervals for the obtained ratios can be calculated by Monte-Carlo simulation, a permutation (bootstrap) approach
  similar to the popular REST software and by (first-order) error propagation. 
Statistical significance for the ratios is calculated by a permutation approach of reallocated vs. non-reallocated data. See 'Details'. 
}

\usage{
ratiocalc(data, group = NULL, which.eff = c("sig", "sli", "exp"), 
          type.eff = c("individual", "mean.single", "median.single", 
                       "mean.pair", "median.pair"),
          which.cp = c("cpD2", "cpD1", "cpE", "cpR", "cpT", "Cy0"),
          perm = c("cp", "eff", "both", NULL), 
          pval = c("up", "down"), ...)
}

\arguments{
  \item{data}{multiple qPCR data generated by \code{\link{pcrbatch}}.}
  \item{group}{a character vector defining the replicates (if any) as well as target and reference data. See 'Details'}.
  \item{which.eff}{efficiency calculated by which method. Defaults to sigmoidal fit. See output of \code{\link{pcrbatch}}. Alternatively, a fixed numeric value between 1 and 2 that is used for all runs.}
  \item{type.eff}{type of efficiency to be supplied for the error analysis. See 'Details'.}
  \item{which.cp}{type of crossing point to be used for the analysis. See output of \code{\link{efficiency}}.}
  \item{perm}{the variables subjected to the permutation approach (reallocation). See 'Details'.}
  \item{pval}{the direction of gene-regulation to be confirmed by permutation p-value. See 'Details'.}
  \item{...}{other parameters to be passed to \code{\link{propagate}}.}
}

\details{
The replicates of the 'pcrbatch' data columns are to be defined as a character vector with the following abbreviations:\cr

"gs":   gene-of-interest in target sample\cr
"gc":   gene-of-interest in control sample\cr
"rs":   reference gene in target sample\cr
"rc":   reference gene in control sample\cr

There is no distinction between the different runs of the same sample, so that three different runs of a gene of interest in
 a target sample are defined as c("gs", "gs", "gs"). The error analysis calculates statistics from ALL replicates, so that
 a further sub-categorization of runs seems superfluous.

Examples:\cr
No replicates: NULL.\cr
2 runs with 2 replicates each, no reference: c("gs", "gs", "gs", "gs", "gc", "gc", "gc", "gc").\cr
1 run with two replicates each and reference data:  c("gs", "gs", "gc", "gc", "rs", "rs", "rc", "rc").\cr

\code{type.eff} defines the pre-processing of the efficiencies before being transferred to \code{\link{propagate}}.
The qPCR community sometimes uses single efficiencies, or averaged over replicates etc., so that different settings
 were implemented. In detail, these are the following:\cr

"individual":     The individual efficiencies from each run are used.\cr
"mean.single":    Efficiencies are averaged over all replicates.\cr
"median.single":  Same as above but median instead of mean.\cr 
"mean.pair":      Efficiencies are averaged from all replicates of target sample and control.\cr
"median.pair":    Same as above but median instead of mean.\cr

The ratios are calculated according to the following formulas:\cr
Without reference PCR: \deqn{\frac{E.gc^{cp.gc}}{E.gs^{cp.gs}}}
With reference PCR: \deqn{\frac{E.gc^{cp.gc}}{E.gs^{cp.gs}}\cdot\frac{E.rs^{cp.rs}}{E.rc^{cp.rc}}}


The permutation approach reallocates either crossing points (\code{"cp"}), efficiencies (\code{"E"}) or \code{"both"} between
 target sample and control groups. Ratios are calculated for each reallocation and compared to ratios obtained if samples were permutated but not
  reallocated. A p-value is calculated from all scenarios in which the reallocation gave a similar/higher ratio than the original data if
 \code{pval} = "up" or similar/lower if \code{pval} = "down". The resulting p-value is thus a measure for the frequency of difference in means of
 the original data and randomly perturbed data.

Confidence values are returned for all three methods (Monte Carlo, permutation, error propagation) as follows:

Monte-Carlo:  From the evaluations of the Monte-Carlo simulated data.\cr
Permutation:  From the evaluations of the within-group permutated data.\cr
Propagation:  From the propagated error, assuming normality.
}

\value{
A list with the following components:\cr
The complete output from \code{\link{propagate}}, attached with the data that was transferred to \code{\link{propagate}} for
 the error analysis as item \code{data}. 
}

\author{
Andrej-Nikolai Spiess
}

\note{
All error types are calculated by \code{\link{propagate}} and often seem quite high. This largely depends on the error of the exponent (i.e. threshold cycles)
 of the exponential function. The error usually decreases when setting \code{use.cov = TRUE} in the \code{...} part of the function. It can be debated anyhow,
 if the variables 'efficiency' and 'threshold cycles' have a covariance structure. As the efficiency is deduced at the second derivative maximum of the sigmoidal
 curve, variance in the second should show an effect on the first, such that the use of a var-covar matrix might be feasible. It is also commonly encountered
 that the propagated error is much higher when using reference data, as the number of partial derivative functions increases.
}

\references{
Livak KJ \emph{et al.} (2001) Analysis of relative gene expression data using real-time quantitative PCR and the 2(-Delta Delta C(T)) method. \emph{Methods}, \bold{25}: 402-428.\cr
Tichopad A \emph{et al.} (2003) Standardized determination of real-time PCR efficiency from a single reaction set-up. \emph{Nucleic Acids Res}, \bold{31}: e122.\cr
Liu W & Saint DA (2002) Validation of a quantitative method for real time PCR kinetics. \emph{Biochem Biophys Res Commun}, \bold{294}: 347-53.\cr
Pfaffl M \emph{et al.} (2002) Relative expression software tool (REST) for group-wise comparison and statistical analysis of relative expression results in real-time PCR. \emph{Nucl Acids Res}, \bold{30}: e36. 
}


\examples{
## only target sample and control,
## no reference, 4 replicates each
## individual efficiencies for error
## calculation
DAT <- pcrbatch(reps, 2:9, l4)
GROUP <- c("gs", "gs", "gs", "gs", "gc", "gc", "gc", "gc")
res <- ratiocalc(DAT, GROUP, which.eff = "sli", type.eff = "individual",
          which.cp = "cpD2", perm = "cp", pval = "up")

## Typical for using individual efficiencies,
## this inflates the error. 95\% confidence intervals 
## include 1 (no differential regulation) and errors are
## also extremely high (over 100\%).
res$conf.Sim
res$conf.Perm
res$error.Prop/res$eval.Prop

## Gets better using averaged efficiencies 
## over all replicates
res2 <- ratiocalc(DAT, GROUP, which.eff = "sli", type.eff = "mean.single",
          which.cp = "cpD2", perm = "cp", pval = "up")
res2$conf.Sim
res2$conf.Perm
res2$error.Prop/res$eval.Prop

## p-value indicates significant upregulation
## in comparison to randomly reallocated 
## threshold cycles (similar to REST software)
res2$pval.Perm

## using reference data.
## toy example is same data as above
## but replicated as reference such
## that the ratio should be 1.
\dontrun{
DAT2 <- pcrbatch(reps, c(2:9, 2:9), l4)
GROUP2 <- c("gs", "gs", "gs", "gs", 
            "gc", "gc", "gc", "gc",
            "rs", "rs", "rs", "rs",
            "rc", "rc", "rc", "rc")
res2 <- ratiocalc(DAT2, GROUP2, which.eff = "sli", type.eff = "mean.single",
          which.cp = "cpD2", perm = "cp", pval = "up")
res2$conf.Sim
res2$conf.Perm
res2$error.Prop/res$eval.Prop
res2$pval.Perm
}

## same as above, but reference data
## is mirrored such that the ratio 
## is squared.
\dontrun{
DAT3 <- pcrbatch(reps, c(2:9, 9:2), l4)
GROUP3 <- c("gs", "gs", "gs", "gs", 
            "gc", "gc", "gc", "gc",
            "rs", "rs", "rs", "rs",
            "rc", "rc", "rc", "rc")
res3 <- ratiocalc(DAT3, GROUP3, which.eff = "sli", type.eff = "mean.single",
          which.cp = "cpD2", perm = "cp", pval = "up")
res3$conf.Sim
res3$conf.Perm
res3$error.Prop/res$eval.Prop
res3$pval.Perm 
}

## compare 'propagate' to REST software
## using the data from the REST 2008
## manual (http://rest.gene-quantification.info/),
## have to create dataframe with values as we do
## not use 'pcrbatch', but external cp's & eff's!
## ties define random reallocation of crossing points
## between control and sample.
## See help for 'propagate'.
\dontrun{
EXPR <- expression((2.01^(cp.gc - cp.gs)/1.97^(cp.rc - cp.rs)))
cp.rc <- c(26.74, 26.85, 26.83, 26.68, 27.39, 27.03, 26.78, 27.32, NA, NA)
cp.rs <- c(26.77, 26.47, 27.03, 26.92, 26.97, 26.97, 26.07, 26.3, 26.14, 26.81)
cp.gc <- c(27.57, 27.61, 27.82, 27.12, 27.76, 27.74, 26.91, 27.49, NA, NA)
cp.gs <- c(24.54, 24.95, 24.57, 24.63, 24.66, 24.89, 24.71, 24.9, 24.26, 24.44)
DAT <- cbind(cp.rc, cp.rs, cp.gc, cp.gs)
res <- propagate(EXPR, DAT, do.sim = TRUE, do.perm = TRUE, ties = c(1, 1, 2, 2))
res$conf.Sim
res$conf.Perm
res$eval.Prop
res$error.Prop
}

## Does error propagation in qPCR quantitation make sense?
## In ratio calculations based on (E1^cp1)/(E2^cp2),
## only 2\% error in each of the variables result in
## over 50\% propagated error!
\dontrun{
x <- NULL
y <- NULL
for (i in seq(0, 0.1, by = 0.01)) {
      E1 <- c(1.7, 1.7 * i)
      cp1 <- c(15, 15 * i)
      E2 <- c(1.7, 1.7 * i)
      cp2 <- c(18, 18 * i)
      DF <- cbind(E1, cp1, E2, cp2)
      res <- propagate(expression((E1^cp1)/(E2^cp2)), DF, type = "stat", plot = FALSE)
      x <- c(x, i * 100)
      y <- c(y, (res$error.Prop/res$eval.Prop) * 100)
}
plot(x, y, xlim = c(0, 10), lwd = 2, xlab = "c.v. [\%]", ylab = "c.v. (prop) [\%]")
}
}



\keyword{nonlinear}

