% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/factional.operators.R
\name{matern.operators}
\alias{matern.operators}
\title{Rational approximations of stationary Gaussian Matern random fields}
\usage{
matern.operators(kappa, sigma, nu, G, C, d = NULL, m = 1)
}
\arguments{
\item{kappa}{Range parameter of the covariance function.}

\item{sigma}{Standard deviation of the covariance function.}

\item{nu}{Shape parameter of the covariance function.}

\item{G}{The stiffness matrix of a finite element discretization of the domain of interest.}

\item{C}{The mass matrix of a finite element discretization of the domain of interest.}

\item{d}{The dimension of the domain.}

\item{m}{The order of the rational approximation, which needs to be a positive integer.
The default value is 1.}
}
\value{
\code{matern.operators} returns an object of class "rSPDEobj". This object contains the
quantities listed in the output of \code{\link{fractional.operators}} as well as the 
parameters of the covariance functoin.
}
\description{
\code{matern.operators} is used for computing a rational SPDE approximation of a stationary Gaussian random
fields on \eqn{R^d} with a Matern covariance function
\deqn{C(h) = \frac{\sigma^2}{2^(\nu-1)\Gamma(\nu)}(\kappa h)^\nu K_\nu(\kappa h)}{C(h) =
(\sigma^2/(2^(\nu-1)\Gamma(\nu))(\kappa h)^\nu K_\nu(\kappa h)}
}
\details{
The approximation is based on a rational approximation of the fractional operator
\eqn{(\kappa^2 -\Delta)^\beta}, where \eqn{\beta = (\nu + d/2)/2}.
This results in an approximate model of the form \deqn{P_l u(s) = P_r W,}
where \eqn{P_j = p_j(L)} are non-fractional operators defined in terms of polynomials \eqn{p_j} for
\eqn{j=l,r}. The order of \eqn{p_r} is given by \code{m} and the order of \eqn{p_l} is \eqn{m + m_\beta}
where \eqn{m_\beta} is the integer part of \eqn{\beta} if \eqn{\beta>1} and
\eqn{m_\beta = 1} otherwise.

The discrete approximation can be written as \eqn{u = P_r x} where \eqn{x \sim N(0,Q^{-1})}{x ~ N(0,Q^{-1})}
and \eqn{Q = P_l^T C^{-1} P_l}. Note that the matrices \eqn{P_r} and \eqn{Q} may be be ill-conditioned for \eqn{m>1}.
In this case, the metehods in \code{\link{operator.operations}} should be used for operations
involving the matrices, since these methods are more numerically stable.
}
\examples{
#Compute rational approximation of a Gaussian process with a 
#Matern covariance function on R
kappa <- 10
sigma <- 1
nu <- 0.8

#create mass and stiffness matrices for a FEM discretization
x <- seq(from = 0, to = 1, length.out = 101)
fem <- rSPDE.fem1d(x)

#compute rational approximation of covariance function at 0.5
op <- matern.operators(kappa = kappa, sigma = sigma, nu = nu,
                       G = fem$G, C = fem$C, d = 1)
                       
v = t(rSPDE.A1d(x,0.5))
c.approx = Sigma.mult(op,v)

#plot the result and compare with the true Matern covariance
plot(x, matern.covariance(abs(x - 0.5), kappa, nu, sigma), type = "l", ylab = "C(h)",
     xlab="h", main = "Matern covariance and rational approximation")
lines(x,c.approx,col=2)
}
\seealso{
\code{\link{fractional.operators}}, \code{\link{spde.matern.operators}}
}
\author{
David Bolin \email{davidbolin@gmail.com}
}
