\name{optPenalty.aLOOCV}
\alias{optPenalty.aLOOCV}
\title{
Select optimal penalty parameter by approximate leave-one-out cross-validation
}
\description{
Function that selects the optimal penalty parameter for the \code{\link{ridgeS}} call by usage of approximate 
leave-one-out cross-validation. Its output includes (a.o.) the precision matrix under the optimal value of the
penalty parameter.
}
\usage{
optPenalty.aLOOCV(Y, lambdaMin, lambdaMax, step, type = "Alt", target = 
diag(1/diag(covML(Y))), output = "light", graph = TRUE, verbose = TRUE)
}
\arguments{
  \item{Y}{
Data \code{matrix}. Variables assumed to be represented by columns.
}
  \item{lambdaMin}{
A \code{numeric} giving the minimum value for the penalty parameter.
}
  \item{lambdaMax}{
A \code{numeric} giving the maximum value for the penalty parameter.
}
  \item{step}{
An \code{integer} determining the number of steps in moving through the grid [\code{lambdaMin}, \code{lambdaMax}].
}
  \item{type}{
A \code{character} indicating the type of ridge estimator to be used. Must be one of: "Alt", "ArchI", "ArchII".
}
  \item{target}{
A target \code{matrix} (in precision terms) for Type I ridge estimators.
}
  \item{output}{
A \code{character} indicating if the output is either heavy or light. Must be one of: "all", "light".
}
  \item{graph}{
A \code{logical} indicating if the grid search for the optimal penalty parameter should be visualized.
}
  \item{verbose}{
A \code{logical} indicating if intermediate output should be printed on screen.
}
}
\details{
The function calculates an approximate leave-one-out cross-validated (aLOOCV) negative log-likelihood score (using a regularized ridge estimator for the precision matrix) for each 
value of the penalty parameter contained in the search grid. The utilized aLOOCV score was proposed by Lian (2011) and Vujacic et al. (2014).
The aLOOCV negative log-likelihood score is computationally more efficient than its non-approximate counterpart (see
\code{\link{optPenaltyCV}}). For details on the aLOOCV negative log-likelihood score see Lian (2011) and Vujacic et al (2014).

The value of the penalty parameter that achieves the lowest aLOOCV negative log-likelihood score is deemed optimal. The penalty parameter must be positive such that \code{lambdaMin} must 
be a positive scalar. The maximum allowable value of \code{lambdaMax} depends on the type of ridge estimator employed. For details on the type of 
ridge estimator one may use (one of: "Alt", "ArchI", "ArchII") see \code{\link{ridgeS}}. The ouput consists of an object of class list 
(see below). When \code{output = "light"} (default) only the \code{optLambda} and \code{optPrec} elements of the list are given.
}
\value{
An object of class list:
\item{optLambda}{A \code{numeric} giving the optimal value of the penalty parameter.}
\item{optPrec}{A \code{matrix} representing the precision matrix of the chosen type (see \code{\link{ridgeS}}) under the optimal value of the penalty parameter.}
\item{lambdas}{A \code{numeric} vector representing all values of the penalty parameter for which approximate cross-validation was performed; Only given when \code{output = "all"}.}
\item{aLOOCVs}{A \code{numeric} vector representing the approximate cross-validated negative log-likelihoods for each value of the penalty parameter given in \code{lambdas}; Only given when \code{output = "all"}.}
}
\references{
Lian, H. (2011). Shrinkage tuning parameter selection in precision matrices estimation. Journal of Statistical
Planning and Inference, 141: 2839-2848.

van Wieringen, W.N. and Peeters, C.F.W. (2014). Ridge Estimation of Inverse Covariance Matrices from High-Dimensional Data. arXiv:1403.0904 [stat.ME].

Vujacic, I., Abbruzzo, A., and Wit, E.C. (2014). A computationally fast alternative to cross-validation in 
penalized Gaussian graphical models. arXiv: 1309.6216v2 [stat.ME].
}
\author{
Carel F.W. Peeters <cf.peeters@vumc.nl>, Wessel N. van Wieringen
}
\seealso{
\code{\link{ridgeS}}, \code{\link{optPenaltyCV}}
}
\examples{
## Obtain some (high-dimensional) data
p = 25
n = 10
set.seed(333)
X = matrix(rnorm(n*p), nrow = n, ncol = p)
colnames(X)[1:25] = letters[1:25]

## Obtain regularized precision under optimal penalty
OPT  <- optPenalty.aLOOCV(X, lambdaMin = .001, lambdaMax = 25, step = 100); OPT	
OPT$optLambda	# Optimal penalty
OPT$optPrec	# Regularized precision under optimal penalty
}