% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bison_solr.r
\name{bison_solr}
\alias{bison_solr}
\title{Search for and collect occurrence data from the USGS Bison API using
their solr endpoint.}
\usage{
bison_solr(decimalLatitude = NULL, decimalLongitude = NULL,
  year = NULL, providerID = NULL, resourceID = NULL,
  pointPath = NULL, basisOfRecord = NULL, eventDate = NULL,
  computedCountyFips = NULL, computedStateFips = NULL,
  scientificName = NULL, hierarchy_homonym_string = NULL,
  TSNs = NULL, recordedBy = NULL, occurrenceID = NULL,
  catalogNumber = NULL, ITIScommonName = NULL, kingdom = NULL,
  collectorNumber = NULL, provider = NULL,
  ownerInstitutionCollectionCode = NULL, providedScientificName = NULL,
  ITISscientificName = NULL, providedCommonName = NULL,
  ITIStsn = NULL, centroid = NULL, higherGeographyID = NULL,
  providedCounty = NULL, calculatedCounty = NULL,
  stateProvince = NULL, calculatedState = NULL, countryCode = NULL,
  callopts = list(), verbose = TRUE, ...)
}
\arguments{
\item{decimalLatitude}{Geographic coordinate that specifies the north south
position of a location on the Earth surface.}

\item{decimalLongitude}{Geographic coordinate that specifies the
east-west position of a location on the Earth surface.}

\item{year}{The year the collection was taken.}

\item{providerID}{(character) Unique identifier assigned by GBIF.}

\item{resourceID}{(character) A unique identifier that is a concatentation
of the provider identifier and the resource id seperated by a comma.}

\item{pointPath}{A dynamic field that contains the location in longitude and
latitude followed by the basis of record and an optional Geo (Spatial)
precision. Geo (Spatial) precision is an added descriptor when the record
is a county centroid.}

\item{basisOfRecord}{One of these enumerated values: Observation, Germplasm,
Fossil, Specimen, Literature, Unknown, or Living.}

\item{eventDate}{The date when the occurrence was recorded. Dates need to
be of the form YYYY-MM-DD}

\item{computedCountyFips}{County FIPS code conforming to standard FIPS 6-4
but with leading zeros removed. See \link{fips} dataset for codes}

\item{computedStateFips}{The normalized state fips code. See \link{fips} dataset
for codes}

\item{scientificName}{The species scientific name that is searchable in
a case insensitive way.}

\item{hierarchy_homonym_string}{hierarachy of the accepted or valid species
name starting at kingdom. If the name is a taxonomic homonym more than
one string is provided seperated by ';'.}

\item{TSNs}{Accepted or valid name is provided. If the name is a taxonmic
homonym more than one TSN is provided.}

\item{recordedBy}{Individual responsible for the scientific record.}

\item{occurrenceID}{Non-persistent unique identifier.}

\item{catalogNumber}{Unique key for every record (occurrence/row) within a
dataset that is not manipulated nor changed (nor generated, if not provided)
during the data ingest.}

\item{ITIScommonName}{Common name(s) from ITIS, e.g. "Canada goose"}

\item{kingdom}{Kingdom name, from GBIF raw occurrence or BISON provider.}

\item{collectorNumber}{An identifier given to the occurrence at the time it
was recorded, such as a specimen collector's number. / e.g., "SJM030022".}

\item{provider}{Non-persistent unique identifier.}

\item{ownerInstitutionCollectionCode}{Name for the dataset, format =
OwnerInstitution-Collection. / e.g., "USGS NAWQA BioData - Fish
Occurrence Records"}

\item{providedScientificName}{Full scientific name as provided in the dataset,
with authorship and date information if known.}

\item{ITISscientificName}{Scientific name from join on ITIS table, calculated
e.g, "Bison bison"}

\item{providedCommonName}{A list (concatenated and separated) of the available
vernacular species names. / e.g., "common shrew, Masked Shrew"}

\item{ITIStsn}{Phase II: ITIS TSN corresponding to
clean_provided_scientific_name. May be invalid,unaccepted. Calculated.
e.g., "3250", "05713"}

\item{centroid}{Text string indicating that provided lat/lon point represents
a polygon centroid. Text provides description of the centroid.}

\item{higherGeographyID}{5-digit numeric text string geographic code for the
state-county combination provided by data provider. / e.g,. "13029"}

\item{providedCounty}{Full county, parish, or organized borough name, as
provided in the dataset. If provided, Verbatum State is required. Is not
changed during data ingest. / e.g., "Fairfax"}

\item{calculatedCounty}{Full county, parish, or organized borough name of
the occurrence calculated. / e.g., "Fairfax"}

\item{stateProvince}{Full name of state or territory of the occurrence, as
provided in the dataset.}

\item{calculatedState}{U.S. State or territory name calculated.
e.g., "Puerto Rico"}

\item{countryCode}{The geographic location of the specific occurrence,
expressed through a constrained vocabulary of countries using 2-letter
ISO country code.}

\item{callopts}{Further args passed on to \code{\link[crul:HttpClient]{crul::HttpClient()}} for HTTP
debugging/inspecting. In \code{bison}, \code{bison_providers}, and
\code{bison_stats}, \code{...} is used instead of
callopts, but \code{...} is used here to pass additional Solr params.}

\item{verbose}{Print message with url (\code{TRUE}, default).}

\item{...}{Additional SOLR query arguments. See details.}
}
\value{
An object of class bison_solr - which is a list with slots for
number of records found (num_found), records, highlight, or facets.
}
\description{
This fxn is somewhat similar to \code{\link[=bison]{bison()}}, but interacts with the SOLR
interface \url{https://bison.usgs.gov/#solr} instead of the
OpenSearch interface \url{https://bison.usgs.gov/#opensearch}, which \code{\link[=bison]{bison()}}
uses.
}
\details{
Named parameters in this function are combined with \code{AND} and passed on
to \code{q} SOLR parameter.  Of course parameters can be more flexibly
combined - let us know if you want that flexibility and we can
think about that.
}
\section{SOLR search parameters passed on via \code{...}}{

\itemize{
\item fl: Fields to return in the query
\item rows: Number of records to return
\item sort: Field to sort by, see examples
\item facet: Facet or not, logical
\item facet.field: Fields to facet by
}

You can also use highlighting in solr search, but I'm not sure I see a
use case for it with BISON data, though you can do it with this function.

For a tutorial see here
\url{http://lucene.apache.org/solr/3_6_2/doc-files/tutorial.html}
}

\section{Range searches}{

If you pass a vector of length 2 to a parameter we construct a range
query for you. For example, \code{c(4, 5)} turns into \code{[4 TO 5]}. The \code{[]}
syntax means the search is inclusive, meaning 4 to 5, including 4 and 5.
Let us know if you think you need more flexible searching. That is,
doing exclusive \code{\{\}} or mixed searches (\code{\{]} or \code{[\}}). Range
searches can only be done with variables that are numeric/integer
or dates or strings that can be coerced to dates. Dates need to
be of the form YYYY-MM-DD
}

\examples{
\dontrun{
x=bison_solr(scientificName='Ursus americanus')

bison_solr(scientificName='Ursus americanus', computedStateFips='02',
 fl="scientificName")

x <- bison_solr(scientificName='Ursus americanus', computedStateFips='02', rows=50)
x$points$computedStateFips
head(x$points)

bison_solr(ITISscientificName='Ursus americanus', rows=50)

bison_solr(providerID = 220)

# combining parameters
x <- bison_solr(eventDate = c('2008-01-01', '2010-12-31'), 
  ITISscientificName="Helianthus annuus", rows = 100)
head(x$points)
sort(x$points$eventDate)

# range queries
## range search with providerID
bison_solr(providerID = c(220, 221))
## date range search
x <- bison_solr(eventDate = c('2010-08-08', '2010-08-21'))
sort(x$points$eventDate)
## TSN range search
x <- bison_solr(TSNs = c(174773, 174775), rows = 100)
sort(x$points$TSN)
## can't do range searches with character strings (that are not dates)
# bison_solr(kingdom = c("Animalia", "Plantae"))

# more examples
bison_solr(TSNs = 174773)
bison_solr(occurrenceID = 576630651)
bison_solr(catalogNumber = 'OBS101299944')
bison_solr(ITIScommonName = "Canada goose")
bison_solr(kingdom = "Animalia")
bison_solr(kingdom = "Plantae")

# Mapping
out <- bison_solr(scientificName='Ursus americanus', rows=200)
bisonmap(out)

out <- bison_solr(scientificName='Helianthus annuus', rows=800)
bisonmap(out)

# Using additional solr fields
## Faceting
bison_solr(scientificName='Helianthus annuus', rows=0, facet='true',
 facet.field='computedStateFips')

## Highlighting
bison_solr(scientificName='Helianthus annuus', rows=10, hl='true',
 hl.fl='scientificName')

## Use of hierarchy_homonym_string
bison_solr(hierarchy_homonym_string = '-202423-914154-914156-158852-')
## -- This is a bit unwieldy, but you can find this string in the output 
## of a call, like this
x <- bison_solr(scientificName='Ursus americanus', rows=1)
string <- x$points$hierarchy_homonym_string
bison_solr(hierarchy_homonym_string = string)

# The pointPath parameter
bison_solr(pointPath = '/-110.0,45.0/specimen')

# Curl options
bison_solr(scientificName='Ursus americanus', callopts=list(verbose = TRUE))
}
}
\references{
\url{https://bison.usgs.gov/#solr}
}
\seealso{
\code{\link[=bison_tax]{bison_tax()}}, \code{\link[=bison]{bison()}}

The USGS BISON Solr installation version as of 2014-10-14 was 4.4.
}
