# This is the server logic for a Shiny web application using state.x77 data
#
# https://CRAN.R-project.org/package=recmap

# source: vignette of https://CRAN.R-project.org/package=colorspace 
pal <- function(col, border = "light gray", ...) {
      n <- length(col)
      plot(0, 0, type="n", xlim = c(0, 1), ylim = c(0, 1),
            axes = FALSE, xlab = "", ylab = "", ...)
    rect(0:(n-1)/n, 0, 1:n/n, 1, col = col, border = border) 
      }

shinyServer(function(input, output, session) {
  
  output$plot_hoverinfo <- renderText({
    # cat("Hover (throttled):\n")
    res <- cartogram()$cartogram
    x <- input$plot_hover$x
    y <- input$plot_hover$y
    query <- ((res$x - res$dx) < x) & (x < (res$x + res$dx)) & ((res$y - res$dy) < y) & (y < (res$y + res$dy))
    
    
    # init
    rv <- "no object identified."
    
    if (sum(query)==1){
      rv <- paste(res$name[query])
    }
    
    paste("state name:", rv)
  })
  
  
  # some taken from the vignette of https://CRAN.R-project.org/package=colorspace 
  colormap <- reactive({
    list(colorspace_heat_hcl = heat_hcl(12, c = c(80, 30), l = c(30, 90), power = c(1/5, 2)),
         colorspace_rev_heat_hcl = rev(heat_hcl(12, h = c(0, -100), c = c(40, 80), l = c(75, 40),  power = 1)),
         colorspace_sequential_hcl_0 = sequential_hcl(12, c = 0, power = 2.2),
         colorspace_sequential_hcl_1 = sequential_hcl(12, power = 2.2),
         colorspace_diverge_hcl = diverge_hcl(7),
         colorspace_rainbow_dynamic4 = rainbow_hcl(4, start = 30, end = 300),
         colorspace_rainbow_dynamic10 = rainbow_hcl(10, start = 30, end = 300),
         colorspace_rainbow_warm10 = rainbow_hcl(10, start = 90, end = -30),
         colorspace_terrain_hcl = terrain_hcl(12, c = c(65, 0), l = c(45, 90), power = c(1/2, 1.5)),
         heat.colors = heat.colors(12),
         DanKeim = rev(c('#C6CF32', '#88E53B', '#50E258', '#29C67D', '#19999E', '#2064AF', '#3835AB', '#561493', '#6E086D', '#790D43', '#741F1E', '#5F3307')
    ),
      DanKeim_HSV = rev(c('#BECC3D', '#70C337', '#31B93C', '#2BB077', '#269FA7', '#21589E', '#221C94', '#56188B', '#82147F', '#791043', '#6F0E0D', '#66380A')))
         })

  output$colormap <- renderUI({
    selectInput('colormapname', 'colormap name', names(colormap()))
  })
  
  output$colormapPlot <- renderPlot({
    par(mar=c(0,0,0,0));
    pal(unlist(colormap()[input$colormapname]))
    })

  
  cartogram <- reactive({
    
    usa <- data.frame(x = state.center$x, 
                      y = state.center$y, 
                      # make the rectangles overlapping by correcting lines of longitude distance
                      dx = sqrt(state.area) / 2 / (0.8 * 60 * cos(state.center$y*pi/180)), 
                      dy = sqrt(state.area) / 2 / (0.8 * 60), 
                      z = sqrt(state.area),
                      name = state.name)
    
    usa$z <- state.x77[, input$area]
    
    M <- usa[!usa$name %in% c("Hawaii", "Alaska"), ]
    
    recmapFitness <- function(idxOrder, Map, ...){
      Cartogram <- recmap(Map[idxOrder, ])
      #1 / sum(Cartogram$z / (sqrt(sum(Cartogram$z^2))) * Cartogram$relpos.error)
      1 / sum(Cartogram$relpos.error)
    }
    
    set.seed(1)
    
    recmapGA <- ga(type = "permutation", 
                   min = 1, max = nrow(M),
                   fitness = recmapFitness, 
                   Map = M,
                   popSize = input$GApopulation * nrow(M), 
                   maxiter = input$GAmaxiter,
                   pmutation = input$GApmutation,
                   parallel=TRUE,
                   run = input$GArun)
    
    C <- recmap(M[recmapGA@solution[1, ], ])
 
    list(cartogram=C, solution=recmapGA@solution[1, ])
  })
  
  output$cartogramPlot <- renderPlot({
    res <- cartogram()
    
    cm <- unlist(colormap()[input$colormapname])
    
    S <- state.x77[which(rownames(state.x77) %in% res$cartogram$name), input$color]
    S <- S[res$solution]
    S <- round((length(cm) - 1) * (S - min(S)) / (max(S)  - min(S))) + 1
    plot.recmap(res$cartogram, col=cm[S], col.text = 'black')
    legend("topleft", c(paste("Area ~", input$area), paste("Color ~", input$color)), box.col = 'white')
  })
  
  
  output$pdfx77 = downloadHandler(
    filename = "recmap_x77.pdf",
    content = function(file) {
      pdf(file, 16,12)
      
      res <- cartogram()
      
      cm <- unlist(colormap()[input$colormapname])
      
      S <- state.x77[which(rownames(state.x77) %in% res$cartogram$name), input$color]
      S <- S[res$solution]
      S <- round((length(cm) - 1) * (S - min(S)) / (max(S)  - min(S))) + 1
      plot.recmap(res$cartogram, col=cm[S], col.text = 'black',
                  sub=paste('U.S. rectangular cartogram generated by using https://CRAN.R-project.org/package=recmap version',
                            packageVersion('recmap')))
      
      legend("topleft", c(paste("Area ~", input$area), paste("Color ~", input$color)), box.col = 'white')
                  
      dev.off()
    })
  

})
