#' @name redcapProjectInfo
#' @aliases redcapProjectInfo.redcapApiConnection
#' @aliases redcapProjectInfo.redcapDbConnection
#' @export redcapProjectInfo
#' @export redcapProjectInfo.redcapApiConnection
#' @export redcapProjectInfo.redcapDbConnection
#' 
#' @title Export a Project's Supplemental Information
#' @description The meta data, users, arms, events, mappings, and REDCap 
#'   version can be used to do some validation during data exports and imports.  
#'   In order to reduce calls to the API, these tables can be stored in an 
#'   object and referenced as needed.
#'   
#' @param rcon A REDCap connection object as generated by \code{redcapConnection}
#' @param date Logical. If \code{TRUE}, user expiration dates are converted to 
#'   \code{POSIXct} objects.
#' @param label Logical.  If \code{TRUE}, the user form permissions are 
#'   converted to labelled factors.
#' @param meta_data Logical.  Indicates if the meta data (data dictionary) 
#'   should be exported.
#' @param users Logical. Indicates if the users table should be exported.
#' @param instruments Logical. Indicates if the instruments table should be exported.
#' @param events Logical. Indicates if the event names should be exported.
#' @param arms Logical. Indicates if the arms table should be exported.
#' @param mappings Logical. Indicates if the form-event mappings should 
#'   be exported.
#' @param version Indicates if the REDCap version number should be exported.  
#'   Only applicable in REDCap 6.0.0 and higher.
#' @param v.number Character string.  Can be used to manually set the REDCap 
#'   version number for users running a version earlier than 6.0.  
#'   This is used to validate calls that may only be available in certain versions.
#' @param ... Arguments to be passed to other methods
#' 
#' @details The project information is stored in the option 
#'   \code{redcap_project_info}.  If the project is not longitudinal, the 
#'   events, arms, and event-form mappings elements will be assigned character 
#'   vectors instead of data frames.
#'   
#' @author Benjamin Nutter
#' 
#' @examples
#' \dontrun{
#' > #*** Note: I cannot provide working examples without
#' > #*** compromising security.  Instead, I will try to 
#' > #*** offer up sample code with the matching results
#' > 
#' > 
#' > #*** Create the connection object
#' > rcon <- redcapConnection(url=[YOUR_REDCAP_URL], token=[API_TOKEN])
#' > 
#' > redcapProjectInfo(rcon)
#' }
#' 

redcapProjectInfo <- function(rcon, date=TRUE, label=TRUE, 
                              meta_data=TRUE, users=TRUE, instruments=TRUE,
                              events=TRUE, arms=TRUE, mappings=TRUE,
                              version=TRUE, ...) UseMethod("redcapProjectInfo")

#' @rdname redcapProjectInfo

redcapProjectInfo.redcapDbConnection <- function(rcon, date=TRUE, label=TRUE, 
                              meta_data=TRUE, users=TRUE, instruments=TRUE,
                              events=TRUE, arms=TRUE, mappings=TRUE,
                              version=TRUE, ...){
  message("Please accept my apologies.  The redcapProjectInfo method for redcapDbConnection objects\n",
          "has not yet been written.  Please consider using the API.")
}

#' @rdname redcapProjectInfo

redcapProjectInfo.redcapApiConnection <- function(rcon, date=TRUE, label=TRUE, 
                              meta_data=TRUE, users=TRUE, instruments=TRUE,
                              events=TRUE, arms=TRUE, mappings=TRUE,
                              version=TRUE, ...,
                              v.number = ""){
  
  version <- if (version) tryCatch(exportVersion(rcon),
                                   error = function(cond){
                                     warning(paste("Error in exportVersion: ", cond))
                                     return(cond)
                                   }) else v.number
  meta_data <- if (meta_data) tryCatch(exportMetaData(rcon),
                                   error = function(cond){
                                     warning(paste("Error in exportMetaData: ", cond))
                                     return(cond)
                                   }) else NULL
  users <- if (users) tryCatch(exportUsers(rcon, date, label),
                                   error = function(cond){
                                     warning(paste("Error in exportUsers: ", cond))
                                     return(cond)
                                   }) else NULL
  instruments <- if (instruments) tryCatch(exportInstruments(rcon),
                                   error = function(cond){
                                     warning(paste("Error in exportInstruments: ", cond))
                                     return(cond)
                                   }) else NULL
  events <- if (events) tryCatch(exportEvents(rcon),
                                   error = function(cond){
                                     warning(paste("Error in exportEvents: ", cond))
                                     return(cond)
                                   }) else NULL
  arms <- if (arms) tryCatch(exportArms(rcon),
                                   error = function(cond){
                                     warning(paste("Error in exportArms: ", cond))
                                     return(cond)
                                   }) else NULL
  mappings <- if (mappings) tryCatch(exportMappings(rcon),
                                   error = function(cond){
                                     warning(paste("Error in exportMappings: ", cond))
                                     return(cond)
                                   }) else NULL
  
  
  proj <- list(meta_data = meta_data,
               users=users,
               instruments=instruments,
               events=events,
               arms=arms,
               mappings=mappings,
               version=version)
  class(proj) <- c("redcapProject", "list")
  return(proj)
}
