% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/redcap_to_db.R
\name{redcap_to_db}
\alias{redcap_to_db}
\title{Transfer 'REDCap' Data to a Database}
\usage{
redcap_to_db(
  conn,
  redcap_uri,
  token,
  data_table_name = "data",
  log_table_name = "log",
  raw_or_label = "raw",
  raw_or_label_headers = "raw",
  export_checkbox_label = FALSE,
  export_survey_fields = FALSE,
  export_data_access_groups = FALSE,
  blank_for_gray_form_status = FALSE,
  filter_logic = "",
  datetime_range_begin = as.POSIXct(NA),
  datetime_range_end = as.POSIXct(NA),
  fields = NULL,
  forms = NULL,
  events = NULL,
  record_id_name = "record_id",
  chunk_size = 1000,
  chunk_delay = 0.5,
  max_retries = 10,
  verbose = TRUE,
  beep = TRUE,
  ...
)
}
\arguments{
\item{conn}{A DBI connection object to a database.}

\item{redcap_uri}{Character string specifying the URI (uniform resource identifier)
of the REDCap server's API.}

\item{token}{Character string containing the REDCap API token specific to your project.
This token is used for authentication and must have export permissions.}

\item{data_table_name}{Character string specifying the name of the table to create or append
data to. Default is "data". Can include schema name (e.g. "schema.table").}

\item{log_table_name}{Character string specifying the name of the table to store
transfer logs. Default is "log". Can include schema name (e.g. "schema.log").
Set to NULL to disable logging.}

\item{raw_or_label}{A string (either "raw" or "label") that specifies
whether to export the raw coded values or the labels for the options of
multiple choice fields. Default is "raw".}

\item{raw_or_label_headers}{A string (either "raw" or "label") that
specifies for the CSV headers whether to export the variable/field names
(raw) or the field labels (label). Default is "raw".}

\item{export_checkbox_label}{Logical that specifies the format of checkbox field values
specifically when exporting the data as labels. If \code{raw_or_label} is
"label" and \code{export_checkbox_label} is TRUE, the values will be the text
displayed to the users. Otherwise, the values will be 0/1. Default is FALSE.}

\item{export_survey_fields}{Logical that specifies whether to export the
survey identifier field (e.g., 'redcap_survey_identifier') or survey
timestamp fields. Default is FALSE.}

\item{export_data_access_groups}{Logical that specifies whether or not to export
the \code{redcap_data_access_group} field when data access groups are utilized
in the project. Default is FALSE.}

\item{blank_for_gray_form_status}{Logical that specifies whether
or not to export blank values for instrument complete status fields that have
a gray status icon. Default is FALSE.}

\item{filter_logic}{String of logic text (e.g., \verb{[gender] = 'male'}) for
filtering the data to be returned, where the API will only return records
where the logic evaluates as TRUE. Default is an empty string.}

\item{datetime_range_begin}{To return only records that have been created or
modified \emph{after} a given datetime, provide a POSIXct value.
Default is NA (no begin time).}

\item{datetime_range_end}{To return only records that have been created or
modified \emph{before} a given datetime, provide a POSIXct value.
Default is NA (no end time).}

\item{fields}{Character vector specifying which fields to export.
Default is NULL (all fields).}

\item{forms}{Character vector specifying which forms to export.
Default is NULL (all forms).}

\item{events}{Character vector specifying which events to export.
Default is NULL (all events).}

\item{record_id_name}{Character string specifying the field name that contains record
identifiers used for chunking requests. Default is "record_id".}

\item{chunk_size}{Integer specifying the number of record IDs to process per chunk.
Default is 1000. Consider decreasing this for projects with many fields.}

\item{chunk_delay}{Numeric value specifying the delay in seconds between chunked
requests. Default is 0.5 seconds. Adjust to respect REDCap server limits.}

\item{max_retries}{Integer specifying the maximum number of retry attempts for failed
API connection or HTTP 504 error. Default is 10.}

\item{verbose}{Logical indicating whether to show progress and completion messages.
Default is TRUE.}

\item{beep}{Logical indicating whether to play sound notifications when the process
completes or encounters errors. Default is TRUE.}

\item{...}{Additional arguments passed to the REDCap API call.}
}
\value{
Returns a list with the following components:
\itemize{
\item \code{success}: Logical if the transfer was completed with no failed processing
\item \code{error_chunks}: Vector of chunk numbers that failed processing
\item \code{time_s}: Numeric value for total seconds to transfer and optimize data
}
}
\description{
Transfer REDCap data to a database in chunks to minimize memory usage.
}
\details{
This function transfers data from REDCap to any database in chunks, which helps manage memory
usage when dealing with large projects. It creates two tables in the database:
\itemize{
\item \code{data_table_name}: Contains all transferred REDCap records
\item \code{log_table_name}: Contains timestamped logs of the transfer process
}

The function automatically detects existing databases and handles them in three ways:
\itemize{
\item If no table exists, starts a new transfer process
\item If a table exists but is incomplete, resumes from the last processed record ID
\item If a table exists and is complete, returns success without reprocessing
}

The function fetches record IDs, then processes records in chunks.
If any error occurs during processing, the function will continue with remaining chunks
but mark the transfer as incomplete.

Data is first set to \strong{VARCHAR/TEXT} type for consistent handling across chunks.
For DuckDB, data types are automatically optimized after all data is inserted:
\itemize{
\item \strong{INTEGER}: Columns with only whole numbers
\item \strong{DOUBLE}: Columns with decimal numbers
\item \strong{DATE}: Columns with valid dates
\item \strong{TIMESTAMP}: Columns with valid timestamps
\item \strong{VARCHAR/TEXT}: All other columns remain as strings
}
}
\examples{
\dontrun{
# install.packages("pak")
# pak::pak(c("duckdb", "keyring", "redquack"))

library(redquack)

duckdb <- DBI::dbConnect(duckdb::duckdb(), "redcap.duckdb")

result <- redcap_to_db(
  redcap_uri = "https://redcap.example.org/api/",
  token = keyring::key_get("redcap_token"),
  conn = duckdb,
)

data <- DBI::dbGetQuery(duckdb, "SELECT * FROM data LIMIT 1000")
log <- DBI::dbGetQuery(duckdb, "SELECT * FROM log")

DBI::dbDisconnect(duckdb)
}
}
\seealso{
\code{\link[DBI]{dbConnect}} for database connection details
}
