% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nsga2.R
\name{nsga2}
\alias{nsga2}
\title{Non-Dominated Sorting in Genetic Algorithms II}
\usage{
nsga2(
  type = c("binary", "real-valued", "permutation"),
  fitness,
  ...,
  lower,
  upper,
  nBits,
  population = nsgaControl(type)$population,
  selection = nsgaControl(type)$selection,
  crossover = nsgaControl(type)$crossover,
  mutation = nsgaControl(type)$mutation,
  popSize = 50,
  nObj = ncol(fitness(matrix(10000, ncol = 100, nrow = 100))),
  pcrossover = 0.8,
  pmutation = 0.1,
  maxiter = 100,
  run = maxiter,
  maxFitness = Inf,
  names = NULL,
  suggestions = NULL,
  monitor = if (interactive()) nsgaMonitor else FALSE,
  summary = FALSE,
  seed = NULL
)
}
\arguments{
\item{type}{the type of genetic algorithm to be run depending on the nature
of decision variables. Possible values are:
\describe{
\item{'binary'}{for binary representations of decision variables.}
\item{'real-valued'}{for optimization problems where the decision
variables are floating-point representations of real numbers.}
\item{'permutation'}{for problems that involves reordering of a list of
objects.}
}}

\item{fitness}{the fitness function, any allowable R function which takes as
input an individual string representing a potential solution, and returns a
numerical value describing its 'fitness'.}

\item{...}{additional arguments to be passed to the fitness function. This
allows to write fitness functions that keep some variables fixed during the
search}

\item{lower}{a vector of length equal to the decision variables providing the
lower bounds of the search space in case of real-valued or permutation
encoded optimizations. Formerly this argument was named min; its usage is
allowed but deprecated.}

\item{upper}{a vector of length equal to the decision variables providing the
upper bounds of the search space in case of real-valued or permutation
encoded optimizations. Formerly this argument was named max; its usage is
allowed but deprecated.}

\item{nBits}{a value specifying the number of bits to be used in binary
encoded optimizations}

\item{population}{an R function for randomly generating an initial population.
See \code{\link[=nsga_Population]{nsga_Population()}} for available functions.}

\item{selection}{an R function performing selection, i.e. a function which
generates a new population of individuals from the current population
probabilistically according to individual fitness. See \code{\link[=nsga_Selection]{nsga_Selection()}}
for available functions.}

\item{crossover}{an R function performing crossover, i.e. a function which
forms offsprings by combining part of the genetic information from their
parents. See \code{\link[=nsga_Crossover]{nsga_Crossover()}} for available functions.}

\item{mutation}{an R function performing mutation, i.e. a function which
randomly alters the values of some genes in a parent chromosome.
See \code{\link[=nsga_Mutation]{nsga_Mutation()}} for available functions.}

\item{popSize}{the population size.}

\item{nObj}{number of objective in the fitness function.}

\item{pcrossover}{the probability of crossover between pairs of chromosomes.
Typically this is a large value and by default is set to 0.8.}

\item{pmutation}{the probability of mutation in a parent chromosome. Usually
mutation occurs with a small probability, and by default is set to 0.1.}

\item{maxiter}{the maximum number of iterations to run before the NSGA search
is halted.}

\item{run}{the number of consecutive generations without any improvement in
the best fitness value before the NSGA is stopped}

\item{maxFitness}{the upper bound on the fitness function after that the NSGA
search is interrupted.}

\item{names}{a vector of character strings providing the names of decision
variables.}

\item{suggestions}{a matrix of solutions strings to be included in the
initial population. If provided the number of columns must match the number
of decision variables.}

\item{monitor}{a logical or an R function which takes as input the current
state of the nsga-class object and show the evolution of the search. By
default, for interactive sessions the function nsgaMonitor prints the average
and best fitness values at each iteration. If set to plot these information
are plotted on a graphical device. Other functions can be written by the user
and supplied as argument. In non interactive sessions, by default
monitor = FALSE so any output is suppressed.}

\item{summary}{If there will be a summary generation after generation.}

\item{seed}{an integer value containing the random number generator state.
This argument can be used to replicate the results of a NSGA search. Note
that if parallel computing is required, the doRNG package must be installed.}
}
\value{
Returns an object of class nsga2-class. See \linkS4class{nsga2} for a
description of available slots information.
}
\description{
Minimization of a fitness function using non-dominated sorting genetic
algorithms - II (NSGA-IIs). Multiobjective evolutionary algorithms
}
\details{
The Non-dominated genetic algorithms II is a meta-heuristic proposed by
K. Deb, A. Pratap, S. Agarwal and T. Meyarivan in 2002. The purpose of the
algorithms is to find an efficient way to optimize multi-objectives functions
(two or more).
}
\examples{
#Example
#Two Objectives - Real Valued
zdt1 <- function (x) {
 if (is.null(dim(x))) {
   x <- matrix(x, nrow = 1)
 }
 n <- ncol(x)
 g <- 1 + rowSums(x[, 2:n, drop = FALSE]) * 9/(n - 1)
 return(cbind(x[, 1], g * (1 - sqrt(x[, 1]/g))))
}

#Not run:
\dontrun{
result <- nsga2(type = "real-valued",
                fitness = zdt1,
                lower = c(0,0),
                upper = c(1,1),
                popSize = 100,
                monitor = FALSE,
                maxiter = 500)
}

#Example 2
#Three Objectives - Real Valued
dtlz1 <- function (x, nobj = 3){
    if (is.null(dim(x))) {
        x <- matrix(x, 1)
    }
    n <- ncol(x)
    y <- matrix(x[, 1:(nobj - 1)], nrow(x))
    z <- matrix(x[, nobj:n], nrow(x))
    g <- 100 * (n - nobj + 1 + rowSums((z - 0.5)^2 - cos(20 * pi * (z - 0.5))))
    tmp <- t(apply(y, 1, cumprod))
    tmp <- cbind(t(apply(tmp, 1, rev)), 1)
    tmp2 <- cbind(1, t(apply(1 - y, 1, rev)))
    f <- tmp * tmp2 * 0.5 * (1 + g)
    return(f)
}

#Not run:
\dontrun{
result <- nsga2(type = "real-valued",
                fitness = dtlz1,
                lower = c(0,0,0), upper = c(1,1,1),
                popSize = 92,
                monitor = FALSE,
                maxiter = 500)
}

}
\references{
K. Deb, A. Pratap, S. Agarwal and T. Meyarivan, 'A fast and
elitist multiobjective genetic algorithm: NSGA-II,' in IEEE Transactions on
Evolutionary Computation, vol. 6, no. 2, pp. 182-197, April 2002,
doi: 10.1109/4235.996017.

Scrucca, L. (2017) On some extensions to 'GA' package: hybrid optimisation,
parallelisation and islands evolution. The R Journal, 9/1, 187-206.
doi: 10.32614/RJ-2017-008
}
\seealso{
\code{\link[=nsga]{nsga()}}, \code{\link[=nsga3]{nsga3()}}
}
\author{
Francisco Benitez
\email{benitezfj94@gmail.com}
}
