## ----setup, include = FALSE----------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

required <- c("viridis")
if (!all(sapply(required, requireNamespace, quietly = TRUE))) {
  knitr::opts_chunk$set(eval = FALSE)
}

## ---- message = FALSE----------------------------------------------------
# First step is to load the libraries. Not all of these libraries are stricly
# needed; some are used for convenience and visualization for this tutorial.
library("samc")
library("raster")
library("viridis")

## ---- fig.show='hold'----------------------------------------------------
# "Load" the data. In this case we are using data built into the package.
# In practice, users will likely load raster data using the raster() function
# from the raster package.
res_data <- samc::ex_res_data
abs_data <- samc::ex_abs_data
occ_data <- samc::ex_occ_data


# Plot the data and make sure it looks good. The built-in data is in matrices, 
# so we use the raster() function to help with the plotting. Note that when
# matrices are used by the package, it sets the extents based on the number of
# rows/cols. We do the same thing here when converting to a raster, otherwise
# the default extents will be (0,1) for both x and y, which is not only
# uninformative, but can result in "stretching" when visualizing datasets
# based non-square matrices.
plot(raster(res_data, xmn = 1, xmx = ncol(res_data), ymn = 1, ymx = nrow(res_data)),
     main = "Example Resistance Data", xlab = "x", ylab = "y", col = viridis(256))

plot(raster(abs_data, xmn = 1, xmx = ncol(abs_data), ymn = 1, ymx = nrow(abs_data)),
     main = "Example Absorption Data", xlab = "x", ylab = "y", col = viridis(256))

plot(raster(occ_data, xmn = 1, xmx = ncol(occ_data), ymn = 1, ymx = nrow(occ_data)),
     main = "Example Occupancy Data", xlab = "x", ylab = "y", col = viridis(256))

## ------------------------------------------------------------------------
# Create a `samc-class` object using the resistance and absorption data. We use the
# recipricol of the arithmetic mean for calculating the transition matrix. Note,
# the input data here are matrices, not RasterLayers. If using RasterLayers, the
# `latlon` parameter must be set.
samc_obj <- samc(res_data, abs_data, tr_fun = function(x) 1/mean(x))


# Print out the samc object and make sure everything is filled out. Try to
# double check some of the values, such as the nrows/ncols of the landscape
# data. The dimensions of the matrix (slot p) should be the number of non-NA
# cells in your data +1. In this case, our data has 2624 non-NA cells, so the
# matrix should be 2625 x 2625
str(samc_obj)

## ------------------------------------------------------------------------
# Convert the occupancy data to probability of occurrence
occ_prob_data <- occ_data / sum(occ_data, na.rm = TRUE)


# Calculate short- and long-term mortality metrics and long-term dispersal
short_mort <- mortality(samc_obj, occ_prob_data, time = 4800)
long_mort <- mortality(samc_obj, occ_prob_data)
long_disp <- dispersal(samc_obj, occ_prob_data)

## ---- fig.show='hold'----------------------------------------------------
# Create rasters using the vector result data for plotting.
short_mort_map <- map(samc_obj, short_mort)
long_mort_map <- map(samc_obj, long_mort)
long_disp_map <- map(samc_obj, long_disp)


# Plot the mortality and dispersal results
plot(short_mort_map, main = "Short-term Mortality", xlab = "x", ylab = "y", col = viridis(256))
plot(long_mort_map, main = "Long-term Mortality", xlab = "x", ylab = "y", col = viridis(256))
plot(long_disp_map, main = "Long-term Dispersal", xlab = "x", ylab = "y", col = viridis(256))

