#' seminr estimate_pls() function
#'
#' The \code{seminr} package provides a natural syntax for researchers to describe PLS
#' structural equation models. \code{seminr} is compatible with semPLS, simplePLS and matrixPLS,
#' meaning that once a model is specified, it can be used across models for comparison.
#'
#' @param data A \code{dataframe} containing the indicator measurement data.
#'
#' @param measurement_model A source-to-target matrix representing the outer/measurement model,
#'   generated by \code{constructs}.
#'
#' @param interactions An object of type \code{interactions} as generated by the \code{interactions}
#'   method. Default setting is \code{NULL} and can be excluded for models with no interactions.
#'
#' @param structural_model A source-to-target matrix representing the inner/structural model,
#'   generated by \code{relationships}.
#'
#' @param inner_weights A parameter declaring which inner weighting scheme should be used
#'   path_weighting is default, alternately path_factorial can be used.
#'
#' @usage
#' estimate_pls(data, measurement_model, interactions=NULL, structural_model,
#'              inner_weights = path_weighting)
#'
#' @seealso \code{\link{relationships}} \code{\link{constructs}} \code{\link{paths}} \code{\link{interactions}}
#'          \code{\link{bootstrap_model}}
#'
#' @examples
#' mobi <- mobi
#'
#' #seminr syntax for creating measurement model
#' mobi_mm <- constructs(
#'              reflective("Image",        multi_items("IMAG", 1:5)),
#'              reflective("Expectation",  multi_items("CUEX", 1:3)),
#'              reflective("Quality",      multi_items("PERQ", 1:7)),
#'              reflective("Value",        multi_items("PERV", 1:2)),
#'              reflective("Satisfaction", multi_items("CUSA", 1:3)),
#'              reflective("Complaints",   single_item("CUSCO")),
#'              reflective("Loyalty",      multi_items("CUSL", 1:3))
#'            )
#' #seminr syntax for creating structural model
#' mobi_sm <- relationships(
#'   paths(from = "Image",        to = c("Expectation", "Satisfaction", "Loyalty")),
#'   paths(from = "Expectation",  to = c("Quality", "Value", "Satisfaction")),
#'   paths(from = "Quality",      to = c("Value", "Satisfaction")),
#'   paths(from = "Value",        to = c("Satisfaction")),
#'   paths(from = "Satisfaction", to = c("Complaints", "Loyalty")),
#'   paths(from = "Complaints",   to = "Loyalty")
#' )
#'
#' mobi_pls <- estimate_pls(data = mobi,
#'                          measurement_model = mobi_mm,
#'                          structural_model = mobi_sm)
#'
#' summary(mobi_pls)
#' plot_scores(mobi_pls)
#' @export
estimate_pls <- function(data, measurement_model, interactions=NULL, structural_model, inner_weights = path_weighting) {
  cat("Generating the seminr model\n")
  warnings(measurement_model, data, structural_model)
  data <- stats::na.omit(data)
  rawdata <- data
  if(!is.null(interactions)) {
    # update data with new interaction items
    intxns_list <- interactions(data, measurement_model)
    get_data <- function(intxn) { intxn$data }
    interaction_data <- do.call("cbind", lapply(intxns_list, get_data))

    # Append data with interaction data
    data <- cbind(data, interaction_data)

    # update measurement model with interaction factors
    measure_interaction <- function(intxn) {
      composite(intxn$name, names(intxn$data),weights = mode_A)
    }
    intxns_mm <- constructs(do.call("c", lapply(intxns_list, measure_interaction)))
    measurement_model <- rbind(measurement_model, intxns_mm)
  }

  # warning if the model is incorrectly specified
  warning_struc_meas_model_complete(structural_model,measurement_model,data)

  # Make a named list of construct measurement_mode functions for simplePLS
  measurement_mode_scheme <- sapply(unique(c(structural_model[,1],structural_model[,2])), get_measure_mode, measurement_model, USE.NAMES = TRUE)

  # Run the model in simplePLS
  seminr_model = seminr::simplePLS(obsData = data, smMatrix = structural_model, mmMatrix = measurement_model, inner_weights = inner_weights, measurement_mode_scheme = measurement_mode_scheme)
  seminr_model$data <- data
  seminr_model$mobi_xm <- interactions
  seminr_model$rawdata <- rawdata

  # Correct for Bias in Reflective models using PLS Consistent
  seminr_model <- model_consistent(seminr_model)

  class(seminr_model) <- "seminr_model"
  return(seminr_model)
}

