\name{monitorTrial}
\alias{monitorTrial}
\title{
Group Sequential Monitoring of Simulated Efficacy Trials for the Event of Potential-Harm, Non-Efficacy, and High-Efficacy
}
\description{
\code{monitorTrial} applies a group sequential monitoring procedure to data-sets generated by \code{simTrial}, which may result in modification or termination of each simulated trial.
}
\usage{
monitorTrial(dataFile, stage1, stage2, harmMonitorRange, 
             alphaPerTest, minCnt, minPct, week1, minCnt2, 
             week2, nonEffInterval, nullVE, altVE, highVE,
             alpha, estimand = c("combined", "cox", "cuminc"), 
             VEcutoffWeek, saveDir = NULL)
}
\arguments{
  \item{dataFile}{if \code{saveDir = NULL}, a list returned by \code{simTrial}; otherwise a name (character string) of an \code{.RData} file created by \code{simTrial}}
  \item{stage1}{the final week of stage 1 in a two-stage trial}
  \item{stage2}{the final week of stage 2 in a two-stage trial, i.e., the maximum follow-up time}
  \item{harmMonitorRange}{a 2-component numeric vector specifying the range for pooled numbers of infections (pooled over the placebo and vaccine arm accruing infections the fastest) for which potential-harm stopping boundaries will be computed}
  \item{alphaPerTest}{a per-test nominal/unadjusted alpha level for potential-harm monitoring}
  \item{minCnt}{a minumum number of infections (pooled over the placebo and vaccine arm accruing infections the fastest) required for the initiation of non-efficacy monitoring [criterion 1]}
  \item{minPct}{a minimum proportion of infections after \code{week1} (pooled over the placebo and vaccine arm accruing infections the fastest) required for the initiation of non-efficacy monitoring [criterion 2]}
  \item{week1}{a time point (in weeks) used, together with \code{minPct}, for defining criterion 2}
  \item{minCnt2}{a minumum number of infections after \code{week2} (pooled over the placebo and vaccine arm accruing infections the fastest) required for the initiation of non-efficacy monitoring [criterion 3]}
  \item{week2}{a time point (in weeks) used, together with \code{minCnt2}, for defining criterion 3}
  \item{nonEffInterval}{a number of infections between two adjacent non-efficacy interim analyses}  
  \item{nullVE}{specifies criterion 1 for declaring non-efficacy: the 95\% confidence interval(s) for the VE estimand(s) do(es) not lie above \code{nullVE} (typically set equal to 0)}
  \item{altVE}{specifies criterion 2 for declaring non-efficacy: the 95\% confidence interval(s) for the VE estimand(s) lie(s) below \code{altVE} (typically a number in the 0--1 range)}
  \item{highVE}{specifies a criterion for declaring high-efficacy: the 95\% confidence interval(s) for the VE estimand(s) lie(s) above \code{highVE} (typically a number in the 0--1 range)}
  \item{alpha}{the nominal significance level for the one-sided test of the null hypothesis that VE(0--\code{stage1}) \eqn{\le} 0\%}
  \item{estimand}{a character string specifying the choice of VE estimands/test statistics used in non-efficacy and high-efficacy monitoring. Three options are implemented: (1) the `pure' Cox approach (\code{"cox"}), where VE is defined as 1-hazard ratio (treatment/control) and estimated by the maximum partial likelihood estimator in the Cox model, with the log-rank test used to evaluate efficacy, (2) the `pure' cumulative incidence approach (\code{"cuminc"}), where VE is defined as 1-cumulative incidence ratio (treatment/control) and estimated by the transformation of the Nelson-Aalen estimator for the cumulative hazard function, with the cumulative hazard-based Wald test used for evaluating efficacy, and (3) the combined approach (\code{"combined"}), where both above VE estimands are used for non-efficacy and high-efficacy testing, and the cumulative hazard-based Wald test is used for evaluating efficacy. Only the first three characters are necessary.}
  \item{VEcutoffWeek}{a time-origin (in weeks) from which the per-protocol VE is assessed (typically taken as the date of the last immunization or the date of the visit following the last immunization)}
  \item{saveDir}{a character string specifying a path for \code{dataFile}. If supplied, the output is also saved as an \code{.RData} file in this directory; otherwise the output is returned as a list.}  
}
\details{
All time variables use week as the unit of time. Month is defined as 52/12 weeks.

Potential-harm monitoring starts at the \code{harmMonitorRange[1]}-th infection pooled over the placebo group and the vaccine regimen that accrues infections the fastest. The potential-harm analyses continue at each additional infection until the first interim analysis for non-efficacy. The monitoring is implemented with exact one-sided binomial tests of H0: \eqn{p \le p0} versus H1: \eqn{p > p0}, where \eqn{p} is the probability that an infected participant was assigned to the vaccine group, and \eqn{p0} is a fixed constant that represents the null hypothesis that an infection is equally likely to be assigned vaccine or placebo. Each test is performed at the same prespecified nominal/unadjusted alpha-level (\code{alphaPerTest}), chosen based on simulations such that, for each vaccine regimen, the overall type I error rate by the \code{harmMonitorRange[2]}-th arm-pooled infection (i.e., the probability that the potential-harm boundary is reached when the vaccine is actually safe, \eqn{p = p0}) equals 0.05.

Non-efficacy is defined as evidence that it is highly unlikely that the vaccine has a beneficial effect on acquisition of VE(0--\code{stage1}) of \code{altVE} x 100\% or more. The non-efficacy analyses for each vaccine regimen will start at the first infection at or after the \code{minCnt}-th (pooled over the vaccine arm and placebo) when at least \code{minPct} x 100\% of the accumulated infections are diagnosed after \code{week1} and at least \code{minCnt2} infections are diagnosed after \code{week2}. Stopping for non-efficacy will lead to a reported 95\% CI for VE(0--\code{stage1}) lying below \code{altVE} x 100\% and covering \code{nullVE} x 100\%, where \code{estimand} determines the choice of the VE(0--\code{stage1}) estimator. This approach is similar to the inefficacy monitoring approach of Freidlin B, Korn EL, Gray R. (2010) A general inefficacy interim monitoring rule for randomized trials. Clinical Trials, 7:197-208. For \code{estimand = "combined"}, stopping for non-efficacy will lead to the reported 95\% CIs for both VE parameters lying below \code{altVE} x 100\% and covering \code{nullVE} x 100\%.


High-efficacy monitoring allows early detection of a highly protective vaccine if there is evidence that VE(0--\code{stage2}) \eqn{>} \code{highVE} x 100\%, based on two planned interim analyses, the first at the time of the fifth planned non-efficacy analysis, and the second at the expected mid-point between the number of infections at the first interim analysis and the number of infections observed at the end of \code{stage2}. Whereas the monitoring for potential-harm and non-efficacy restricts to \code{stage1} infections, the monitoring for high-efficacy counts all infections during \code{stage1} or \code{stage2}, given that early-stopping for high-efficacy would only be warranted under evidence for durability of the efficacy.

The following principles and rules are applied in the monitoring procedure:
\itemize{
\item Exclude all follow-up data from the analysis post-unblinding (and include all data pre-unblinding).
\item The monitoring is based on modified ITT analysis, i.e., all subjects documented to be free of the study endpoint at baseline are included and analyzed according to the treatment assigned by randomization, ignoring how many vaccinations they received (only pre-unblinding follow-up included).
\item If a vaccine hits the harm boundary, immediately discontinue vaccinations and accrual into this vaccine arm, and unblind this vaccine arm (continue post-unblinded follow-up until the end of Stage 1 for this vaccine arm).  
\item If a vaccine hits the non-efficacy boundary, immediately discontinue vaccinations and accrual into this vaccine arm, keep blinded and continue follow-up until the end of Stage 1 for this vaccine arm. 
\item If and when the last vaccine arm hits the non-efficacy (or harm) boundary, discontinue vaccinations and accrual into this vaccine arm, and unblind (the trial is over, completed in Stage 1).
\item Stage 1 for the whole trial is over on the earliest date of the two events: (1) all vaccine arms have hit the harm or non-efficacy boundary; and (2) the last enrolled subject in the trial reaches the final \code{stage1} visit.
\item Continue blinded follow-up until the end of Stage 2 for each vaccine arm that reaches the end of \code{stage1} with a positive efficacy or high-efficacy result.
\item If at least one vaccine arm reaches the end of \code{stage1} with a positive efficacy or high-efficacy result, continue blinded follow-up in the placebo arm until the end of Stage 2.
\item Stage 2 for the whole trial is over on the earliest date of the two events: (1) all subjects in the placebo arm and each vaccine arm that registered efficacy or high-efficacy in \code{stage1} have failed or been censored; and (2) all subjects in the placebo arm and each vaccine arm that registered efficacy or high-efficacy in \code{stage1} have completed the final \code{stage2} visit.
}
The above rules have the following implications:
\itemize{
\item If a vaccine hits the non-efficacy boundary but Stage 1 for the whole trial is not over, then one includes in the analysis all follow-up through the final \code{stage1} visit for that vaccine regimen, including all individuals accrued up through the date of hitting the non-efficacy boundary (which will be the total number accrued to this vaccine arm).
\item If a vaccine hits the harm boundary, all follow-up information through the date of hitting the harm boundary is included for this vaccine; no follow-up data are included after this date.
\item If and when the last vaccine arm hits the non-efficacy (or harm) boundary, all follow-up information through the date of hitting the non-efficacy (or harm) boundary is included for this vaccine; no follow-up data are included after this date.
}
}
\value{
If \code{saveDir} is specified, the output list (named \code{out}) is saved as an \code{.RData} file in \code{saveDir} (the path to \code{saveDir} is printed); otherwise it is returned. The output object is a list of length equal to the number of simulated trials, each of which is a list of length equal to the number of treatment arms, each of which is a list with (at least) the following components:
\item{boundHit}{a character string stating the monitoring outcome in this treatment arm, i.e., one of \code{"Harm"}, \code{"NonEffInterim"}, \code{"NonEffFinal"}, \code{"Eff"}, or \code{"HighEff"}. The first four outcomes can occur in Stage 1, whereas the last outcome can combine data over Stage 1 and Stage 2.}
\item{stopTime}{the time of hitting a stopping boundary since the first subject enrolled in the trial}
\item{stopInfectCnt}{the pooled number of infections at \code{stopTime}}
\item{summObj}{a \code{data.frame} containing summary information from each non-/high-efficacy interim analysis}
\item{finalHRci}{the final 95\% CI for the hazard ratio, available if \code{estimand!="cuminc"} and there is at least 1 infection in each arm}
\item{firstNonEffCnt}{the number of infections that triggered non-efficacy monitoring (if available)}
\item{totInfecCnt}{the total number of \code{stage1} (\code{stage2} if \code{boundHit = "HighEff"}) infections}
\item{totInfecSplit}{a table with the numbers of \code{stage1} (\code{stage2} if \code{boundHit = "HighEff"}) infections in the treatment and control arm}
\item{lastExitTime}{the time between the first subject's enrollment and the last subject's exiting from the trial}
}
\seealso{
\code{\link{simTrial}}, \code{\link{censTrial}}, and \code{\link{rankTrial}}
}
\examples{
simData <- simTrial(N=c(1000, rep(700, 2)), aveVE=seq(0, 0.4, by=0.2), 
                    VEmodel="half", vePeriods=c(1, 27, 79), enrollPeriod=78, 
                    enrollPartial=13, enrollPartialRelRate=0.5, dropoutRate=0.05, 
                    infecRate=0.04, fuTime=156, 
                    visitSchedule=c(0, (13/3)*(1:4), seq(13*6/3, 156, by=13*2/3)),
                    missVaccProb=c(0,0.05,0.1,0.15), VEcutoffWeek=26, nTrials=5, 
                    stage1=78, randomSeed=300)
                    
monitorData <- monitorTrial(dataFile=simData, stage1=78, stage2=156, 
                            harmMonitorRange=c(10,100), alphaPerTest=0.0106, 
                            minCnt=50, minPct=0.33, week1=26, minCnt2=2, week2=52, 
                            nonEffInterval=20, nullVE=0, altVE=0.4, highVE=0.6, 
                            alpha=0.025, estimand="combined", VEcutoffWeek=26)

### alternatively, to save the .RData output file (no '<-' needed):
###
### simTrial(N=c(1400, rep(1000, 2)), aveVE=seq(0, 0.4, by=0.2), VEmodel="half", 
###          vePeriods=c(1, 27, 79), enrollPeriod=78, enrollPartial=13, 
###          enrollPartialRelRate=0.5, dropoutRate=0.05, infecRate=0.04, fuTime=156, 
###          visitSchedule=c(0, (13/3)*(1:4), seq(13*6/3, 156, by=13*2/3)), 
###          missVaccProb=c(0,0.05,0.1,0.15), VEcutoffWeek=26, nTrials=30, 
###          stage1=78, saveDir="./", randomSeed=300)
###
### monitorTrial(dataFile=
###          "simTrial_nPlac=1400_nVacc=1000_1000_aveVE=0.2_0.4_infRate=0.04.RData", 
###          stage1=78, stage2=156, harmMonitorRange=c(10,100), alphaPerTest=0.0106, 
###          minCnt=50, minPct=0.33, week1=26, minCnt2=2, week2=52, nonEffInterval=20, 
###          nullVE=0, altVE=0.4, highVE=0.6, alpha=0.025, estimand="combined", 
###          VEcutoffWeek=26, saveDir="./")
}
