#' Show Group Enrichment Result
#'
#' See [group_enrichment] for examples.
#'
#' @inheritParams ggplot2::facet_wrap
#' @param df_enrich result `data.frame` from [group_enrichment].
#' @param return_list if `TRUE`, return a list of `ggplot` object so user
#' can combine multiple plots by other R packages like `patchwork`.
#' @param add_text_annotation if `TRUE`, add text annotation in box.
#' When show p value with filled color, the text indicates relative change;
#' when show relative change with filled color, the text indicates p value.
#' @param fill_by_p_value if `TRUE`, show log10 based p values with filled color.
#' The +/- of p values indicates change direction.
#' @param fill_scale a `Scale` object generated by `ggplot2` package to
#' set color for continuous values.
#' @param ... other parameters passing to [ggplot2::facet_wrap], only used
#' when `return_list` is `FALSE`.
#'
#' @return a (list of ) `ggplot` object.
#' @export
show_group_enrichment <- function(df_enrich,
                                  return_list = FALSE,
                                  scales = "free",
                                  add_text_annotation = TRUE,
                                  fill_by_p_value = TRUE,
                                  fill_scale = scale_fill_gradient2(
                                    low = "blue", mid = "white", high = "red",
                                    midpoint = ifelse(fill_by_p_value, 0, 1)),
                                  ...) {

  if (fill_by_p_value) {
    df_enrich$p_value_up <- abs(log10(df_enrich$p_value))
    df_enrich$p_value_up <- data.table::fifelse(
      df_enrich$measure_observed >= 1,
      df_enrich$p_value_up,
      -df_enrich$p_value_up
    )
  }

  if (return_list) {
    df_enrich %>%
      dplyr::group_nest(.data$grp_var) %>%
      dplyr::mutate(
        gg = purrr::map(.data$data,
                        plot_enrichment_simple,
                        x = "enrich_var", y = "grp1",
                        fill_scale = fill_scale,
                        fill_by_p_value = fill_by_p_value,
                        add_text_annotation = add_text_annotation)
      ) -> xx
    p <- xx$gg
    names(p) <- xx$grp_var

  } else {
    p <- plot_enrichment_simple(df_enrich, x = "enrich_var", y = "grp1",
                           fill_scale = fill_scale,
                           fill_by_p_value = fill_by_p_value,
                           add_text_annotation = add_text_annotation) +
      facet_wrap(~grp_var, scales = scales, ...)
  }

  return(p)
}

plot_enrichment_simple <- function(data, x, y, fill_scale,
                                   fill_by_p_value = TRUE,
                                   add_text_annotation = TRUE) {
  if (fill_by_p_value) {
    data$measure_observed <- round(data$measure_observed, 2)
  } else {
    data$p_value <- round(data$p_value, 3)
  }

  p <- ggplot(data,
              aes_string(
                x = x,
                y = y)) +
    geom_tile(mapping = aes_string(fill = if (fill_by_p_value) "p_value_up" else "measure_observed")) +
    fill_scale +
    labs(
      x = "Variable",
      y = "Subgroup",
      fill = if (fill_by_p_value) "log10\n(p-value)" else "Relative\nchange") +
    scale_x_discrete(expand = expansion(mult = c(0, 0))) +
    scale_y_discrete(expand = expansion(mult = c(0, 0)))

  if (add_text_annotation) {
    p <- p +
      geom_text(mapping = aes_string(
        label = if (fill_by_p_value) "measure_observed" else "p_value"
      ))
  }

  p
}
