% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spLMexact.R
\name{spLMexact}
\alias{spLMexact}
\title{Univariate Bayesian spatial linear model}
\usage{
spLMexact(
  formula,
  data = parent.frame(),
  coords,
  cor.fn,
  priors,
  spParams,
  noise_sp_ratio,
  n.samples,
  loopd = FALSE,
  loopd.method = "exact",
  verbose = TRUE,
  ...
)
}
\arguments{
\item{formula}{a symbolic description of the regression model to be fit.
See example below.}

\item{data}{an optional data frame containing the variables in the model.
If not found in \code{data}, the variables are taken from
\code{environment(formula)}, typically the environment from which
\code{spLMexact} is called.}

\item{coords}{an \eqn{n \times 2}{n x 2} matrix of the observation
coordinates in \eqn{\mathbb{R}^2} (e.g., easting and northing).}

\item{cor.fn}{a quoted keyword that specifies the correlation function used
to model the spatial dependence structure among the observations. Supported
covariance model key words are: \code{'exponential'} and \code{'matern'}.
See below for details.}

\item{priors}{a list with each tag corresponding to a parameter name and
containing prior details.}

\item{spParams}{fixed value of spatial process parameters.}

\item{noise_sp_ratio}{noise-to-spatial variance ratio.}

\item{n.samples}{number of posterior samples to be generated.}

\item{loopd}{logical. If \code{loopd=TRUE}, returns leave-one-out predictive
densities, using method as given by \code{loopd.method}. Default is
\code{FALSE}.}

\item{loopd.method}{character. Ignored if \code{loopd=FALSE}. If \code{loopd=TRUE},
valid inputs are \code{'exact'} and \code{'PSIS'}. The option \code{'exact'} corresponds to
exact leave-one-out predictive densities which requires computation almost
equivalent to fitting the model \eqn{n} times. The option \code{'PSIS'} is
faster and finds approximate leave-one-out predictive densities using
Pareto-smoothed importance sampling (Gelman \emph{et al.} 2024).}

\item{verbose}{logical. If \code{verbose = TRUE}, prints model description.}

\item{...}{currently no additional argument.}
}
\value{
An object of class \code{spLMexact}, which is a list with the
following tags -
\describe{
\item{samples}{a list of length 3, containing posterior samples of fixed
effects (\code{beta}), variance parameter (\code{sigmaSq}), spatial effects
(\code{z}).}
\item{loopd}{If \code{loopd=TRUE}, contains leave-one-out predictive
densities.}
\item{model.params}{Values of the fixed parameters that includes
\code{phi} (spatial decay), \code{nu} (spatial smoothness) and
\code{noise_sp_ratio} (noise-to-spatial variance ratio).}
}
The return object might include additional data used for subsequent
prediction and/or model fit evaluation.
}
\description{
Fits a Bayesian spatial linear model with spatial process
parameters and the noise-to-spatial variance ratio fixed to a value supplied
by the user. The output contains posterior samples of the fixed effects,
variance parameter, spatial random effects and, if required, leave-one-out
predictive densities.
}
\details{
Suppose \eqn{\chi = (s_1, \ldots, s_n)} denotes the \eqn{n}
spatial locations the response \eqn{y} is observed. With this function, we
fit a conjugate Bayesian hierarchical spatial model
\deqn{
\begin{aligned}
y \mid z, \beta, \sigma^2 &\sim N(X\beta + z, \delta^2 \sigma^2 I_n), \quad
z \mid \sigma^2 \sim N(0, \sigma^2 R(\chi; \phi, \nu)), \\
\beta \mid \sigma^2 &\sim N(\mu_\beta, \sigma^2 V_\beta), \quad
\sigma^2 \sim \mathrm{IG}(a_\sigma, b_\sigma)
\end{aligned}
}
where we fix the spatial process parameters \eqn{\phi} and \eqn{\nu}, the
noise-to-spatial variance ratio \eqn{\delta^2} and the hyperparameters
\eqn{\mu_\beta}, \eqn{V_\beta}, \eqn{a_\sigma} and \eqn{b_\sigma}. We utilize
a composition sampling strategy to sample the model parameters from their
joint posterior distribution which can be written as
\deqn{
p(\sigma^2, \beta, z \mid y) = p(\sigma^2 \mid y) \times
p(\beta \mid \sigma^2, y) \times p(z \mid \beta, \sigma^2, y).
}
We proceed by first sampling \eqn{\sigma^2} from its marginal posterior,
then given the samples of \eqn{\sigma^2}, we sample \eqn{\beta} and
subsequently, we sample \eqn{z} conditioned on the posterior samples of
\eqn{\beta} and \eqn{\sigma^2} (Banerjee 2020).
}
\examples{
# load data
data(simGaussian)
dat <- simGaussian[1:100, ]

# setup prior list
muBeta <- c(0, 0)
VBeta <- cbind(c(1.0, 0.0), c(0.0, 1.0))
sigmaSqIGa <- 2
sigmaSqIGb <- 0.1
prior_list <- list(beta.norm = list(muBeta, VBeta),
                   sigma.sq.ig = c(sigmaSqIGa, sigmaSqIGb))

# supply fixed values of model parameters
phi0 <- 3
nu0 <- 0.75
noise.sp.ratio <- 0.8

mod1 <- spLMexact(y ~ x1, data = dat,
                  coords = as.matrix(dat[, c("s1", "s2")]),
                  cor.fn = "matern",
                  priors = prior_list,
                  spParams = list(phi = phi0, nu = nu0),
                  noise_sp_ratio = noise.sp.ratio,
                  n.samples = 100,
                  loopd = TRUE, loopd.method = "exact")

beta.post <- mod1$samples$beta
z.post.median <- apply(mod1$samples$z, 1, median)
dat$z.post.median <- z.post.median
plot1 <- surfaceplot(dat, coords_name = c("s1", "s2"),
                     var_name = "z_true")
plot2 <- surfaceplot(dat, coords_name = c("s1", "s2"),
                     var_name = "z.post.median")
plot1
plot2
}
\references{
Banerjee S (2020). "Modeling massive spatial datasets using a
conjugate Bayesian linear modeling framework." \emph{Spatial Statistics}, \strong{37},
100417. ISSN 2211-6753. \doi{10.1016/j.spasta.2020.100417}.

Vehtari A, Simpson D, Gelman A, Yao Y, Gabry J (2024). "Pareto
Smoothed Importance Sampling." \emph{Journal of Machine Learning Research},
\strong{25}(72), 1-58. URL \url{https://jmlr.org/papers/v25/19-556.html}.
}
\seealso{
\code{\link[=spLMstack]{spLMstack()}}
}
\author{
Soumyakanti Pan \href{mailto:span18@ucla.edu}{span18@ucla.edu},\cr
Sudipto Banerjee \href{mailto:sudipto@ucla.edu}{sudipto@ucla.edu}
}
