% Generated by roxygen2 (4.0.2): do not edit by hand
\name{Mestimator}
\alias{Mestimator}
\title{Function \code{Mestimator}}
\usage{
Mestimator(x, locations, pairIndices, k, model, Tol = 1e-05,
  startingValue = NULL, Omega = diag(nrow(pairIndices)), iterate = TRUE,
  covMat = TRUE)
}
\arguments{
\item{x}{An \eqn{n} x \eqn{d} data matrix.}

\item{locations}{A \eqn{d} x 2 matrix containing the Cartesian coordinates of \eqn{d} points in the plane.}

\item{pairIndices}{A \eqn{q} x 2 matrix containing the indices of \eqn{q} pairs of points from the matrix \code{locations}.}

\item{k}{The threshold parameter in the definition of the empirical stable tail dependence function.}

\item{model}{Choose between "smith" and "BR".}

\item{Tol}{The tolerance parameter in the numerical integration procedure; defaults to 1e-05.}

\item{startingValue}{Initial value of the parameters in the minimization routine. Defaults to diag(2) for the Smith model and (1, 1.5, 0.75, 0.75) for the BR process.}

\item{Omega}{A \eqn{q} x \eqn{q} matrix specifying the metric with which the distance between the parametric and nonparametric estimates will be computed. The default is the identity matrix, i.e., the Euclidean metric.}

\item{iterate}{A Boolean variable. If \code{TRUE} (the default), then the estimator is calculated twice, first with \code{Omega} specified by the user, and then a second time with the optimal \code{Omega} calculated at the initial estimate.}

\item{covMat}{A Boolean variable. If \code{TRUE} (the default), the covariance matrix is calculated.}
}
\value{
A list with the following components:
\tabular{ll}{
\code{theta} \tab The estimator with estimated optimal weight matrix. \cr
\code{theta_pilot} \tab The estimator without the optimal weight matrix.\cr
\code{covMatrix} \tab The estimated covariance matrix for the estimator. \cr
\code{Omega} \tab The weight matrix with which the estimator was calculated.
}
}
\description{
Function to compute the pairwise M-estimator for the parameters of the Smith model or the Brown-Resnick process.
}
\details{
For a detailed description of the estimation procedure, see Einmahl et al. (2014).
Some tips for using this function:
\itemize{
\item{\code{n} versus \code{d}: }{ if the number of locations \eqn{d} is small (\eqn{d < 8} say), a sufficiently
large sample size (eg \eqn{n > 2000}) is needed to obtain a satisfying result, especially for the Brown-Resnick process.
However, if \eqn{d} is large, a sample size of \eqn{n = 500} should suffice.}
\item{\code{pairIndices}: }{ if the number of pairs \eqn{q} is large, \code{Mestimator} will be rather slow. This is due
to the calculation of \code{Omega} and \code{covMat}. Setting \code{iterate = FALSE} and
\code{covMat = FALSE} will make this procedure fast even for several hundreds of pairs of locations.}
\item{\code{Tol}: }{ the tolerance parameter is used when calculating the three- and four-dimensional integrals
in the asymptotic covariance matrix (see Appendix B in Einmahl et al. (2014)). A tolerance of 1e-04 often suffices, although
the default tolerance is a safer choice.}
\item{\code{StartingValue}: }{ for the Smith model, the estimator usually doesn't depend on the starting value
at all. For the Brown-Resnick process, it is advised to try a couple of starting values if \eqn{d}
is very small, preferably a starting value with \eqn{c < 1} and one with \eqn{c > 1}.}
\item{\code{iterate}: }{ if \code{iterate = TRUE}, the matrix \code{Omega} is calculated. This weight matrix tends to have a larger
effect when \eqn{d} is large and/or when the Smith model is used.}
\item{\code{covMat}: }{ if the resulting covariance matrix is incorrect (eg negative diagonal values), then \code{Tol} is set too high.
For the Smith model, the order of the parameters is \eqn{(\sigma_{11},\sigma_{22},\sigma_{12})}. }
}
}
\examples{
## define the locations of 4 stations
(locations <- rbind(c(1,1),c(2,1),c(1,2),c(2,2)))
## select the pairs of locations; here, we select all locations
(pairIndices <- selectPairIndices(locations, maxDistance = 2))

## We use the rmaxstab function from the package SpatialExtremes to
## simulate from the Smith and the Brown-Resnick process.

## The Smith model
set.seed(2)
x<-rmaxstab(n = 5000, coord = locations,cov.mod="gauss",cov11=1,cov22=2,cov12=0.5)
## calculate the pairwise M-estimator. This may take up to one minute or longer.
## Mestimator(x, locations, pairIndices, 100, model="smith",Tol = 5e-04)

## The Brown-Resnick process
set.seed(2)
x <- rmaxstab(n = 5000, coord = locations, cov.mod = "brown", range = 3, smooth = 1)
## We can only simulate isotropic processes with rmaxstab, so we multiply the coordinates
## of the locations with V^(-1) (beta,c). Here we choose beta = 0.25 and c = 1.5
(Vmat<-matrix(c(cos(0.25),1.5*sin(0.25),-sin(0.25),1.5*cos(0.25)),nrow=2))
(locationsAniso <- locations \%*\% t(solve(Vmat)))
## calculate the pairwise M-estimator. This may take up to one minute or longer.
## Mestimator(x, locationsAniso, pairIndices, 300, model="BR",Tol = 5e-04)
}
\references{
Einmahl, J.H.J., Kiriliouk, A., Krajina, A. and Segers, J. (2014), "An M-estimator of spatial tail dependence". See \url{http://arxiv.org/abs/1403.1975}.
}
\seealso{
\code{\link{selectPairIndices}}, \code{\link{pairCoordinates}}
}

