cont.analysis <- function(sites=NULL, subpop=NULL, design=NULL, data.cont=NULL,
   sigma=NULL, var.sigma=NULL, N.cluster=NULL, popsize=NULL, stage1size=NULL,
   popcorrect=FALSE, sizeweight=FALSE, unitsize=NULL, vartype="Local", conf=95,
   pctval=c(5,10,25,50,75,90,95), psurvey.obj=NULL) {

################################################################################
# Function: cont.analysis
# Programmers: Tony Olsen
#              Tom Kincaid
# Date: July 18, 2002
# Last Revised: September 5, 2006
# Description:
#   This function organizes input and output for analysis of continuous data
#   generated by a probability survey.  Input can be either an object of class
#   psurvey.analysis (see the documentation for function psurvey.analysis)
#   or through use of the other arguments to this function.
#   Input:
#      sites = a data frame consisting of two variables: the first variable is 
#         site IDs, and the second variable is a logical vector indicating which
#         sites to use in the analysis.  If psurvey.obj is not provided, then
#         this argument is required.  The default is NULL.
#      subpop = a data frame describing sets of populations and subpopulations 
#         for which estimates will be calculated.  The first variable is site  
#         IDs.  Each subsequent variable identifies a Type of population, where
#         the variable name is used to identify Type.  A Type variable
#         identifies each site with one of the subpopulations of that Type.  If
#         psurvey.obj is not provided, then this argument is required.  The
#         default is NULL.
#      design = a data frame consisting of design variables.  If psurvey.obj is
#        not provided, then this argument is required.  The default is NULL.
#        Variables should be named as follows:
#            siteID = site IDs
#            wgt = final adjusted weights, which are either the weights for a
#               single-stage sample or the stage two weights for a two-stage
#               sample
#            xcoord = x-coordinates for location, which are either the x-
#               coordinates for a single-stage sample or the stage two x-
#               coordinates for a two-stage sample
#            ycoord = y-coordinates for location, which are either the y-
#               coordinates for a single-stage sample or the stage two y-
#               coordinates for a two-stage sample
#            stratum = the stratum codes
#            cluster = the stage one sampling unit (primary sampling unit or  
#               cluster) codes
#            wgt1 = final adjusted stage one weights
#            xcoord1 = the stage one x-coordinates for location
#            ycoord1 = the stage one y-coordinates for location
#            support = support values - the value one (1) for a site from a 
#               finite resource or the measure of the sampling unit associated 
#               with a site from an extensive resource, which is required for 
#               calculation of finite and continuous population correction 
#               factors.
#            swgt = size-weights, which is the stage two size-weight for a two-
#               stage sample.
#            swgt1 = stage one size-weights.
#      data.cont = a data frame of continuous response variables.  The first 
#         variable is site IDs.  Subsequent variables are response variables.
#         Missing data (NA) is allowed.  The default is NULL.
#      sigma = measurement error variance.  This variable must be a vector
#         containing a value for each response variable and must have the names
#         attribute set to identify the response variable names.  Missing data
#         (NA) is allowed.  The default is NULL.
#      var.sigma = variance of the measurement error variance.  This variable
#         must be a vector containing a value for each response variable and
#         must have the names attribute set to identify the response variable
#         names.  Missing data (NA) is allowed.  The default is NULL.
#      N.cluster = the number of stage one sampling units in the resource, which 
#         is required for calculation of finite and continuous population 
#         correction factors for a two-stage sample.  For a stratified sample 
#         this variable must be a vector containing a value for each stratum and
#         must have the names attribute set to identify the stratum codes.  The
#         default is NULL.
#      popsize = the known size of the resource - the total number of sampling 
#         units of a finite resource or the measure of an extensive resource,
#         which is used to adjust estimators for the known size of a resource.
#         This argument also is required for calculation of finite and
#         continuous population correction factors for a single-stage sample.   
#         The argument must be in the form of a list containing an element for   
#         each population Type in the subpop data frame, where NULL is a valid   
#         choice for a population Type.  The list must be named using the column  
#         names for the population Types in subpop. If a population Type doesn't  
#         contain subpopulations, then each element of the list is either a  
#         single value for an unstratified sample or a vector containing a value  
#         for each stratum for a stratified sample, where elements of the vector
#         are named using the stratum codes.  If a population Type contains 
#         subpopulations, then each element of the list is a list containing an 
#         element for each subpopulation, where the list is named using the 
#         subpopulation names.  The element for each subpopulation will be 
#         either a single value for an unstratified sample or a named vector of 
#         values for a stratified sample.  The default is NULL.
#         Example popsize for a stratified sample:
#            popsize = list("Pop 1"=c("Stratum 1"=750, "Stratum 2"=500,
#               "Stratum 3"=250), "Pop2"=list("SubPop 1"=c("Stratum 1"=350,
#               "Stratum 2"=250, "Stratum 3"=150), "SubPop 2"=c("Stratum 1"=250,
#               "Stratum 2"=150, "Stratum 3"=100), "SubPop 3"=c("Stratum 1"=150,
#               "Stratum 2"=150, "Stratum 3"=75)), "Pop 3"=NULL)
#         Example popsize for an unstratified sample:
#            popsize = list("Pop 1"=1500, "Pop2"=list("SubPop 1"=750,
#               "SubPop 2"=500, "SubPop 3"=375), "Pop 3"=NULL)
#      stage1size = the known size of the stage one sampling units of a two-
#         stage sample, which is required for calculation of finite and  
#         continuous population correction factors for a two-stage sample and 
#         must have the names attribute set to identify the stage one sampling 
#         unit codes.  For a stratified sample, the names attribute must be set
#         to identify both stratum codes and stage one sampling unit codes using
#         a convention where the two codes are separated by the & symbol, e.g.,
#         "Stratum 1&Cluster 1".  The default is NULL.
#      popcorrect = a logical value that indicates whether finite or continuous 
#         population correction factors should be employed during variance 
#         estimation, where TRUE = use the correction factors and FALSE = do not 
#         use the correction factors.  The default is FALSE.
#      sizeweight = a logical value that indicates whether size-weights should 
#         be used in the analysis, where TRUE = use the size-weights and FALSE = 
#         do not use the size-weights.  The default is FALSE.
#      unitsize = the known sum of the size-weights of the resource.  The 
#         argument must be in the form of a list containing an element for each  
#         population Type in the subpop data frame, where NULL is a valid choice  
#         for a population Type.  The list must be named using the column  
#         names for population Types in subpop.  If a population Type doesn't  
#         contain subpopulations, then each element of the list is either a  
#         single value for an unstratified sample or a vector containing a value  
#         for each stratum for a stratified sample, where elements of the vector
#         are named using the stratum codes.  If a population Type contains 
#         subpopulations, then each element of the list is a list containing an 
#         element for each subpopulation, where the list is named using the 
#         subpopulation names.  The element for each subpopulation will be 
#         either a single value for an unstratified sample or a named vector of 
#         values for a stratified sample.  The default is NULL.
#      vartype = the choice of variance estimator, where "Local" = local mean
#         estimator and "SRS" = SRS estimator.  The default is "Local".
#      conf = the confidence level.  The default is 95%.
#      pctval = the set of values at which percentiles are estimated.  The 
#         default set is: {5, 10, 25, 50, 75, 90, 95}.
#      psurvey.obj = A list of class psurvey.analysis that was produced by the
#         function psurvey.analysis.  Depending on input to that function,
#         some elements of the list may be NULL.  The default is NULL.
#   Output:
#      A list containing either two or four data frames of population
#      estimates for all combinations of population Types, subpopulations within
#      Types, and response variables.  The data frames containing deconvoluted
#      CDF estimates and deconvoluted percentile estimates are only included in
#      the output list when an input value for measurement error variance is
#      provided to the function.  CDF estimates are calculated for both
#      proportion and size of the population.  Standard error estimates and
#      confidence interval estimates also are calculated.  
#      The four data frames are:
#         CDF - a data frame containing the CDF estimates
#         Pct - a data frame containing the percentile estimates plus population
#            mean, standard deviation, and variance estimates
#         CDF.D - a data frame containing the deconvoluted CDF estimates
#         Pct.D - a data frame containing the deconvoluted percentile estimates
#      If an input value for measurement error variance is not provided to the
#      function, then CDF.D and Pct.D are assigned the value NULL.
#   Other Functions Required:
#      dframe.check - check site IDs, the sites data frame, the subpop data
#         frame, the data.cat data frame, type.cat, and the data.cont data frame
#         to assure valid contents and, as necessary, create the sites data
#         frame, the subpop data frame, and type.cat
#      vecprint - takes an input vector and outputs a character string with
#         line breaks inserted
#      uniqueID - creates unique site IDs by appending a unique number to
#         each occurrence of a site ID
#      input.check - check input values for errors, consistency, and
#         compatibility with psurvey.analysis analytical functions
#      cdf.est - estimate the cumulative distribution function (CDF) for the 
#         proportion (expressed as percent) and the total of a response variable
#      cdf.decon - estimate the deconvoluted CDF for the proportion and the
#         total of a response variable
#      total.est - estimate the population total, mean, variance, and
#         standard deviation of a response variable
#   Examples:
#      Continuous variable example
#      mysiteID <- paste("Site", 1:100, sep="")
#      mysites <- data.frame(siteID=mysiteID, Active=rep(TRUE, 100))
#      mysubpop <- data.frame(siteID=mysiteID, All.Sites=rep("All Sites",100),
#         Resource.Class=rep(c("Good","Poor"), c(55,45)))
#      mydesign <- data.frame(siteID=mysiteID, wgt=runif(100, 10, 100),
#         xcoord=runif(100), ycoord=runif(100))
#      mypopsize <- list(All.Sites=c(Stratum1=3500, Stratum2=2000),
#         Resource.Class=list(Good=c(Stratum1=2500, Stratum2=1500),
#         Poor=c(Stratum1=1000, Stratum2=500)))
#      ContVar <- rnorm(100, 10, 1)
#      mydata.cont <- data.frame(siteID=mysiteID, ContVar=ContVar)
#      cont.analysis(sites=mysites, subpop=mysubpop, design=mydesign,
#         data.cont=mydata.cont, popsize=mypopsize)
#
#      Include deconvolution estimates
#      mydata.cont <- data.frame(siteID=mysiteID, ContVar=ContVar,
#         ContVar.1=ContVar + rnorm(100, 0, sqrt(0.25)),
#         ContVar.2=ContVar + rnorm(100, 0, sqrt(0.50)))
#      mysigma <- c(ContVar=NA, ContVar.1=0.25, ContVar.2=0.50)
#      cont.analysis(sites=mysites, subpop=mysubpop[,1:2], design=mydesign,
#         data.cont=mydata.cont, sigma=mysigma, popsize=mypopsize[1])
################################################################################

# Create a data frame for warning messages

   warn.ind <- FALSE
   warn.df <- NULL
   fname <- "cont.analysis"

# Begin the section when an object of class "psurvey.analysis" was input to the 
# function

   if(inherits(psurvey.obj, "psurvey.analysis")) {

# Assign variables from the input list

      sites <- psurvey.obj$sites
      subpop <- psurvey.obj$subpop
      design <- psurvey.obj$design
      data.cont <- psurvey.obj$data.cont
      sigma <- psurvey.obj$sigma
      var.sigma <- psurvey.obj$var.sigma
      N.cluster <- psurvey.obj$N.cluster
      popsize <- psurvey.obj$popsize
      stage1size <- psurvey.obj$stage1size
      unitsize <- psurvey.obj$unitsize
      stratum.ind <- psurvey.obj$stratum.ind
      cluster.ind <- psurvey.obj$cluster.ind
      pcfactor.ind <- psurvey.obj$pcfactor.ind
      swgt.ind <- psurvey.obj$swgt.ind
      vartype <- psurvey.obj$vartype
      conf <- psurvey.obj$conf
      pctval <- psurvey.obj$pctval

# Assign some required values from the subpop data frame

      ntypes <- dim(subpop)[2]
      typenames <- names(subpop)

# Begin the section when an object of class "psurvey.analysis" was not input to
# the function

   } else {

# Check that the required data frames have been provided

      if(is.null(sites))
         stop("\nThe sites data frame must be provided.")
      if(!is.data.frame(sites))
         stop("\nThe sites argument must be a data frame.")
      if(is.null(subpop))
         stop("\nThe subpop data frame must be provided.")
      if(!is.data.frame(subpop))
         stop("\nThe subpop argument must be a data frame.")
      if(is.null(design))
         stop("\nThe design data frame must be provided.")
      if(!is.data.frame(design))
         stop("\nThe design argument must be a data frame.")
      if(is.null(data.cont))
         stop("\nThe data.cont data frame must be provided.")
      if(!is.data.frame(data.cont))
         stop("\nThe data.cont argument must be a data frame.")

# Check the design data frame for required names

      design.names <- names(design)
      temp <- match(design.names, c("siteID", "wgt", "xcoord", "ycoord",
         "stratum", "cluster", "wgt1", "xcoord1", "ycoord1", "support", "swgt",
         "swgt1"), nomatch=0)
      if(any(temp == 0)) {
         temp.str <- vecprint(design.names[temp == 0])
         stop(paste("\nThe following names used in the design data frame do not match the required names:\n", temp.str))
      }

# Check the sites data frame, the design data frame, the subpop data frame, and
# the data.cont data frame to assure valid contents

      temp <- dframe.check(sites, design, subpop, NULL, data.cont, design.names)
      design <- temp$design
      subpop <- temp$subpop
      data.cont <- temp$data.cont

# Assign variables from the design data frame

      siteID <- design$siteID
      wgt <- design$wgt
      xcoord <- design$xcoord
      ycoord <- design$ycoord
      stratum <- design$stratum
      cluster <- design$cluster
      wgt1 <- design$wgt1
      xcoord1 <- design$xcoord1
      ycoord1 <- design$ycoord1
      support <- design$support
      swgt <- design$swgt
      swgt1 <- design$swgt1

# Check site IDs for repeat values and, as necessary, create unique site IDs and
# output a warning message

      temp <- sapply(split(siteID, siteID), length)
      if(any(temp > 1)) {
         warn.ind <- TRUE
         temp.str <- vecprint(names(temp)[temp > 1])
         warn <- paste("The following site ID values occur more than once among the values that were \ninput to the function:\n", temp.str, sep="")
         act <- "Unique site ID values were created.\n"
         warn.df <- rbind(warn.df, data.frame(func=I(fname),
            subpoptype=NA, subpop=NA, indicator=NA, stratum=NA, warning=I(warn),
            action=I(act)))
         siteID <- uniqueID(siteID)
         subpop[,1] <- siteID
         data.cont[,1] <- siteID
      }

# Assign some required values from the subpop data frame

      ntypes <- dim(subpop)[2]
      typenames <- names(subpop)

# Determine whether the sample is stratified

      stratum.ind <- length(unique(stratum)) > 1

# If the sample is stratified, convert stratum to a factor, determine stratum 
# levels, and calculate number of strata

      if(stratum.ind) {
         stratum <- factor(stratum)
         stratum.levels <- levels(stratum)
         nstrata <- length(stratum.levels)
      } else {
         stratum.levels <- NULL
         nstrata <- NULL
      }

# Determine whether the sample has two stages

      cluster.ind <- length(unique(cluster)) > 1

# If the sample has two stages, convert cluster to a factor, determine cluster 
# levels, and calculate number of clusters

      if(cluster.ind) {
         if(stratum.ind) {
            cluster.in <- cluster
            cluster <- tapply(cluster, stratum, factor)
            cluster.levels <- sapply(cluster, levels, simplify=FALSE)
            ncluster <- sapply(cluster.levels, length)
         } else {
            cluster <- factor(cluster)
            cluster.levels <- levels(cluster)
            ncluster <- length(cluster.levels)
         }
      }

# Ensure that popsize and unitsize are lists

      if(!is.null(popsize) && !is.list(popsize))
         stop("\nThe popsize argument must be a list")
      if(!is.null(unitsize) && !is.list(unitsize))
         stop("\nThe unitsize argument must be a list")

# Determine whether the population correction factor is to be used

      if(popcorrect && is.null(support))
         stop("\nThe logical value that indicates whether finite or continuous population \ncorrection factors should be employed during variance estimation was set to \nTRUE, but support values were not provided in the design data frame.")
      pcfactor.ind <- popcorrect

# Determine whether the sample uses size-weights

      if(sizeweight) {
         if(is.null(swgt))
            stop("\nThe logical value that indicates whether size-weights should be employed in the analysis was set to \nTRUE, but size-weights were not provided in the design data frame.")
         if(cluster.ind && is.null(swgt1))
            stop("\nThe sample has two stages and the logical value that indicates whether size- \nweights should be employed in the analysis was set to TRUE, but stage one \nsize-weights were not provided in the design data frame.")
      }
      swgt.ind <- sizeweight

# Check the vector of measurement error variance values for correct names

      if(!is.null(sigma)) {
         temp.names <- names(data.cont)[-1]
         if(length(sigma) != length(temp.names))
            stop("\nThe vector of measurement error variance values is not the correct length.")
         if(is.null(names(sigma)))
            stop("\nThe vector of measurement error variance values must be named.")
         temp <- match(temp.names, names(sigma), nomatch=0)
         if(any(temp == 0)) {
            temp.str <- vecprint(temp.names[temp == 0])
            stop(paste("\nThe following names for the response variables do not occur among the names for \nthe vector of measurement error variance values:\n", temp.str, sep=""))
         }
         temp <- match(names(sigma), temp.names, nomatch=0)
         if(any(temp == 0)) {
            temp.str <- vecprint(names(sigma)[temp == 0])
            stop(paste("\nThe following names for the vector of measurement error variance values do not \noccur among the names for the response variables:\n", temp.str, sep=""))
         }
         sigma <- sigma[temp]
      }

# Check the vector of values for variance of the measurement error variance for
# correct names

      if(!is.null(var.sigma)) {
         if(length(var.sigma) != length(temp.names))
            stop("\nThe vector of values for variance of the measurement error variance is not the \ncorrect length.")
         if(is.null(names(var.sigma)))
            stop("\nThe vector of values for variance of the measurement error variance must be \nnamed.")
         temp <- match(temp.names, names(var.sigma), nomatch=0)
         if(any(temp == 0)) {
            temp.str <- vecprint(temp.names[temp == 0])
            stop(paste("\nThe following names for the response variables do not occur among the names for \nthe vector of values for variance of the measurement error variance:\n", temp.str, sep=""))
         }
         temp <- match(names(var.sigma), temp.names, nomatch=0)
         if(any(temp == 0)) {
            temp.str <- vecprint(names(var.sigma)[temp == 0])
            stop(paste("\nThe following names for the vector of values for variance of the measurement \nerror variance do not occur among the names for the response variables:\n", temp.str, sep=""))
         }
         var.sigma <- var.sigma[temp]
      }

# Determine the number of response values

      nresp <- dim(design)[1]

# Check for compatibility of input values

      temp <- input.check(nresp, wgt, sigma, var.sigma, xcoord, ycoord,
         stratum.ind, stratum, stratum.levels, nstrata, cluster.ind, cluster,
         cluster.levels, ncluster, N.cluster, wgt1, xcoord1, ycoord1, popsize,
         stage1size, pcfactor.ind, support, swgt.ind, swgt, swgt1, unitsize,
         vartype, conf, pctval=pctval, subpop=subpop)

      N.cluster <- temp$N.cluster
      popsize <- temp$popsize
      stage1size <- temp$stage1size
      unitsize <- temp$unitsize

# If the sample has two stages, determine whether there are a sufficient number
# of sites in each stage one sampling unit to allow variance calculation

      if(cluster.ind) {
         temp.ind <- FALSE
         for(itype in 2:ntypes) {
            temp <- apply(table(cluster, subpop[,itype]) == 1, 2, sum)
            if(any(temp > 0)) {
               temp.ind <- TRUE
               temp.str <- vecprint(names(temp)[temp > 0])
               warn <- paste("A variance estimate cannot be calculated since the following subpopulation(s) of \npopulation ", typenames[itype], " include one or more stage one sampling units \nwith a single site:\n", temp.str, sep="")
               act <- "Program execution was terminated.\n"
               warn.df <- rbind(warn.df, data.frame(func=I(fname),
                  subpoptype=NA, subpop=NA, indicator=NA, stratum=NA,
                  warning=I(warn), action=I(act)))
            }
         }
         if(temp.ind) {
            stop("\nThe program was terminated due to the presence of one or more stage one sampling \nunits with a single site.  Enter the following command to view the warning \nmessages that were generated: warnprnt() \n")
         }
      }

# If the sample was stratified and had two stages, then reset cluster to its 
# input value

      if(stratum.ind && cluster.ind)
         cluster <- cluster.in

# As necessary, assign missing values to the design variables

      if(is.null(xcoord))
         xcoord <- rep(NA, nresp)
      if(is.null(ycoord))
         ycoord <- rep(NA, nresp)
      if(is.null(stratum))
         stratum <- rep(NA, nresp)
      if(is.null(cluster))
         cluster <- rep(NA, nresp)
      if(is.null(wgt1))
         wgt1 <- rep(NA, nresp)
      if(is.null(xcoord1))
         xcoord1 <- rep(NA, nresp)
      if(is.null(ycoord1))
         ycoord1 <- rep(NA, nresp)
      if(is.null(support))
         support <- rep(NA, nresp)
      if(is.null(swgt))
         swgt <- rep(NA, nresp)
      if(is.null(swgt1))
         swgt1 <- rep(NA, nresp)

# Recreate the design data frame

      design <- data.frame(siteID=siteID, wgt=wgt, xcoord=xcoord, ycoord=ycoord,
         stratum=stratum, cluster=cluster, wgt1=wgt1, xcoord1=xcoord1,
         ycoord1=ycoord1, support=support, swgt=swgt, swgt1=swgt1)

# End the section when an object of class "psurvey.analysis" was not input to
# the function

   }

# Loop through all response variables

   nvar <- dim(data.cont)[2]
   varnames <- names(data.cont)
   nrow1a <- 0 
   nrow2a <- 0 
   for(ivar in 2:nvar) {

# Find unique data values across all subpopulations of this type of population

      cdfval <- sort(unique(data.cont[!is.na(data.cont[,ivar]),ivar]))

# Loop through all types of populations

      for(itype in 2:ntypes) {

# Find unique subpopulations of this type of population

         subpopnames <- levels(factor(subpop[,itype]))	

# Loop through all subpopulations of this type

         for(isubpop in 1:length(subpopnames)) {

# Select sites in a subpopulation

            subpop.ind <- subpop[,itype] == subpopnames[isubpop]

# Determine whether the subpopulation is empty

            if(all(is.na(data.cont[subpop.ind,ivar]))) {
               warn.ind <- TRUE
               warn <- paste("Subpopulation", subpopnames[isubpop], "of population type", typenames[itype], "for indicator", varnames[ivar], "\ncontains no data.\n")
               act <- "None.\n"
               warn.df <- rbind(warn.df, data.frame(func=I(fname),
                  subpoptype=I(typenames[itype]),
                  subpop=I(subpopnames[isubpop]), indicator=I(varnames[ivar]),
                  stratum=NA,  warning=I(warn), action=I(act)))
               next
            }

# Determine whether the subpopulation contains a single value

            if(sum(!is.na(data.cont[subpop.ind,ivar])) == 1) {
               warn.ind <- TRUE
               warn <- paste("Subpopulation", subpopnames[isubpop], "of population type", typenames[itype], "for indicator", varnames[ivar], "\ncontains a single value.  No analysis was performed.\n")
               act <- "None.\n"
               warn.df <- rbind(warn.df, data.frame(func=I(fname),
                  subpoptype=I(typenames[itype]),
                  subpop=I(subpopnames[isubpop]), indicator=I(varnames[ivar]),
                  stratum=NA,  warning=I(warn), action=I(act)))
               next
            }

# For a stratified sample, remove values from N.cluster and stage1size for
# strata that do not occur in the subpopulation

            if(stratum.ind) {
               temp.N.cluster <- N.cluster[!is.na(match(names(N.cluster),
                  unique(design[subpop.ind, 5])))]
               temp.stage1size <- stage1size[!is.na(match(names(stage1size),
                  unique(design[subpop.ind, 5])))]
            } else {
               temp.N.cluster <- N.cluster
               temp.stage1size <- stage1size
           }

# Select values from popsize and unitsize

            if(is.list(popsize[[itype-1]]))
               temp.popsize <- popsize[[itype-1]][[isubpop]]
            else
               temp.popsize <- popsize[[itype-1]]

            if(is.list(unitsize[[itype-1]]))
               temp.unitsize <- unitsize[[itype-1]][[isubpop]]
            else
               temp.unitsize <- unitsize[[itype-1]]

# Calculate estimates for the response variable

            warn.vec <- c(typenames[itype], subpopnames[isubpop],
               varnames[ivar])

            temp <- cdf.est(z=data.cont[subpop.ind,ivar],
               wgt=design[subpop.ind,2], x=design[subpop.ind,3],
               y=design[subpop.ind,4], stratum=design[subpop.ind,5],
               cluster=design[subpop.ind,6], N.cluster=temp.N.cluster,
               wgt1=design[subpop.ind,7], x1=design[subpop.ind,8],
               y1=design[subpop.ind,9], popsize=temp.popsize,
               stage1size=temp.stage1size, support=design[subpop.ind,10],
               swgt=design[subpop.ind,11], swgt1=design[subpop.ind,12],
               unitsize=temp.unitsize, vartype=vartype, conf=conf,
               cdfval=cdfval, pctval=pctval, check.ind=FALSE, warn.ind=warn.ind,
               warn.df=warn.df, warn.vec=warn.vec)
            temp1.cont <- temp$Results
            warn.ind <- temp$warn.ind
            warn.df <- temp$warn.df

            if(!is.null(sigma) && !is.na(sigma[ivar-1]))
               temp <- cdf.decon(z=data.cont[subpop.ind,ivar],
                  wgt=design[subpop.ind,2], sigma=sigma[ivar-1],
                  var.sigma=var.sigma[ivar-1], x=design[subpop.ind,3],
                  y=design[subpop.ind,4], stratum=design[subpop.ind,5],
                  cluster=design[subpop.ind,6], N.cluster=temp.N.cluster,
                  wgt1=design[subpop.ind,7], x1=design[subpop.ind,8],
                  y1=design[subpop.ind,9], popsize=temp.popsize,
                  stage1size=temp.stage1size, support=design[subpop.ind,10],
                  swgt=design[subpop.ind,11], swgt1=design[subpop.ind,12],
                  unitsize=temp.unitsize, vartype=vartype, conf=conf,
                  cdfval=cdfval, pctval=pctval, check.ind=FALSE,
                  warn.ind=warn.ind, warn.df=warn.df, warn.vec=warn.vec)
               temp2.cont <- temp$Results
               warn.ind <- temp$warn.ind
               warn.df <- temp$warn.df

            temp <- total.est(z=data.cont[subpop.ind,ivar],
               wgt=design[subpop.ind,2], x=design[subpop.ind,3],
               y=design[subpop.ind,4], stratum=design[subpop.ind,5],
               cluster=design[subpop.ind,6], N.cluster=temp.N.cluster,
               wgt1=design[subpop.ind,7], x1=design[subpop.ind,8],
               y1=design[subpop.ind,9], popsize=temp.popsize,
               stage1size=temp.stage1size, support=design[subpop.ind,10],
               swgt=design[subpop.ind,11], swgt1=design[subpop.ind,12],
               unitsize=temp.unitsize, vartype=vartype, conf=conf,
               check.ind=FALSE, warn.ind=warn.ind, warn.df=warn.df,
               warn.vec=warn.vec)
            temp3.cont <- temp$Results[-1,]
            warn.ind <- temp$warn.ind
            warn.df <- temp$warn.df

# Assign estimates for the response variable to data frames

            if(nrow1a == 0) {
               nn <- dim(temp1.cont$CDF)[1]
               cdfsum <- data.frame(Type=rep(typenames[itype],nn), 
                  Subpopulation=rep(subpopnames[isubpop],nn), 
                  Indicator=rep(varnames[ivar],nn), temp1.cont$CDF)
               nrow1a <- nn

               nn <- dim(temp1.cont$Pct)[1] + dim(temp3.cont)[1]
               pctsum <- data.frame(Type=rep(typenames[itype],nn), 
                  Subpopulation=rep(subpopnames[isubpop],nn), 
                  Indicator=rep(varnames[ivar],nn), 
                  rbind(temp1.cont$Pct[,1:6], temp3.cont))
               nrow1b <- nn
            } else {
               nn <- dim(temp1.cont$CDF)[1]
               cdfsum <- rbind(cdfsum, data.frame(Type=rep(typenames[itype],nn), 
                  Subpopulation=rep(subpopnames[isubpop],nn), 
                  Indicator=rep(varnames[ivar],nn), temp1.cont$CDF, 
                  row.names=(nrow1a+1):(nrow1a+nn)))
               nrow1a <- nrow1a + nn

               nn <- dim(temp1.cont$Pct)[1] + dim(temp3.cont)[1]
               pctsum <- rbind(pctsum, data.frame(Type=rep(typenames[itype],
                  nn), Subpopulation=rep(subpopnames[isubpop],nn),
                  Indicator=rep(varnames[ivar],nn), 
                  rbind(temp1.cont$Pct[,1:6], temp3.cont), 
                  row.names=(nrow1b+1):(nrow1b+nn)))
               nrow1b <- nrow1b + nn
            }

            if(!is.null(sigma) && !is.na(sigma[ivar-1])) {
               if(nrow2a == 0) {
                  nn <- dim(temp2.cont$CDF)[1]
                  cdfsum.D <- data.frame(Type=rep(typenames[itype],nn), 
                     Subpopulation=rep(subpopnames[isubpop],nn), 
                     Indicator=rep(varnames[ivar],nn), temp2.cont$CDF)
                  nrow2a <- nn

                  nn <- dim(temp2.cont$Pct)[1]
                  pctsum.D <- data.frame(Type=rep(typenames[itype],nn), 
                     Subpopulation=rep(subpopnames[isubpop],nn), 
                     Indicator=rep(varnames[ivar],nn), temp2.cont$Pct[,1:6])
                  nrow2b <- nn
               } else {
                  nn <- dim(temp2.cont$CDF)[1]
                  cdfsum.D <- rbind(cdfsum.D, data.frame(Type=rep(typenames
                     [itype], nn), Subpopulation=rep(subpopnames[isubpop],nn), 
                     Indicator=rep(varnames[ivar],nn), temp2.cont$CDF, 
                     row.names=(nrow2a+1):(nrow2a+nn)))
                  nrow2a <- nrow2a + nn

                  nn <- dim(temp2.cont$Pct)[1]
                  pctsum.D <- rbind(pctsum.D, data.frame(Type=rep(typenames
                     [itype],nn), Subpopulation=rep(subpopnames[isubpop],nn), 
                     Indicator=rep(varnames[ivar],nn), temp2.cont$Pct[,1:6], 
                     row.names=(nrow2b+1):(nrow2b+nn)))
                  nrow2b <- nrow2b + nn
               }
            }

# End of the loop for subpopulations

         }

# End of the loop for type of population

      }

# End of the loop for response variables

   }

# As necessary, output a message indicating that warning messages were generated
# during execution of the program

   if(warn.ind) {
      warn.df <<- warn.df
      if(nrow(warn.df) == 1)
         cat("During execution of the program, a warning message was generated.  The warning \nmessage is stored in a data frame named 'warn.df'.  Enter the following command \nto view the warning message: warnprnt()\n")
      else
         cat(paste("During execution of the program,", nrow(warn.df), "warning messages were generated.  The warning \nmessages are stored in a data frame named 'warn.df'.  Enter the following \ncommand to view the warning messages: warnprnt() \nTo view a subset of the warning messages (say, messages number 1, 3, and 5), \nenter the following command: warnprnt(m=c(1,3,5))\n"))
   }

# Return the list

   if(!is.null(sigma))
      list(CDF=cdfsum, Pct=pctsum, CDF.D=cdfsum.D, Pct.D=pctsum.D)
   else
      list(CDF=cdfsum, Pct=pctsum, CDF.D=NULL, Pct.D=NULL)
}
