% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SQUANT.R
\name{eval_squant}
\alias{eval_squant}
\title{SQUANT performance evaluation}
\usage{
eval_squant(
  yvar,
  censorvar,
  trtvar,
  trtcd = 1,
  dir,
  type,
  data,
  squant.out,
  brief = FALSE
)
}
\arguments{
\item{yvar}{A character. The response variable name in the \code{data}.
The corresponding column in the \code{data} should be numeric.}

\item{censorvar}{A character or NULL. The event indicator variable name in
the \code{data}. The corresponding column in the \code{data} should
be 0(censor) or 1(event). Use NULL when it is not a time-to-event case.}

\item{trtvar}{A character or NULL. The trt variable name in the \code{data}
for the predictive case. The corresponding column in the \code{data}
should contain the treatment assignments, and can be either numeric
or character. Use NULL for the prognostics case.}

\item{trtcd}{The code for the treatment arm for the predictive case,
e.g. trtcd="treatment" or trtcd=1, etc.}

\item{dir}{A character, "larger" or "smaller".
When dir == "larger", larger response is preferred for the target subgroup.
In the predictive case, it means the derived signature from \code{squant}
selects patients satisfying
               E(Y|X,TRT)-E(Y|X,CTRL)>=quant.
In the prognostic case, it means the derived signature from \code{squant}
selects patients satisfying
               E(Y|X)>=quant.
When dir == "smaller", smaller response is preferred for the target subgroup.
In the predictive case, it means the derived signature from \code{squant}
selects patients satisfying
               E(Y|X,CTRL)-E(Y|X,TRT)>=quant.
In the prognostic case, it means the derived signature from \code{squant}
selects patients satisfying
               E(Y|X)<=quant.}

\item{type}{The response type. Use "s" for survival, "b" for binary, and
"c" for continuous.}

\item{data}{The data frame for performance evaluation of the derived signature.}

\item{squant.out}{The squant object, the signature of which will be applied
to the specified data. The output of \code{squant} function.}

\item{brief}{A logical value, TRUE or FALSE. When TRUE, only the most important p value
will be reported.}
}
\value{
An object of "eval_squant". A list containing the following elements.
  \item{inter.pval}{Treatment*subgroup Interaction p value (predictive case only).}
  \item{pos.group.pval}{The p value of the trt difference in the selected positive
    group (predictive case only).}
  \item{neg.group.pval}{The p value of the trt difference in the negative group
    (predictive case only).}
  \item{pval}{The p value of group comparison (prognostic case only).}
  \item{group.stats}{The performance of each arm by group (predictive case) or
    the performance of each group (prognostic case).}
  \item{data.pred}{The data with the predicted subgroup in the last column.}
}
\description{
\code{eval_squant} evaluates the subgroup identification performance.
}
\details{
This function evaluates the subgroup identification performance through
applying the derived signature (the squant object) to a specified dataset.
Note that when the specified dataset is the same as the training set,
the performance is usually over-optimistic and is subject to over-fitting.
Ideally, use an independent testing set to have an honest evaluation of
the performance.
}
\examples{
#toy example#
set.seed(888)
x=as.data.frame(matrix(rnorm(200),100,2))
names(x) = c("x1", "x2")
trt = sample(0:1, size=100, replace=TRUE)
y= 2*x[,2]*trt+rnorm(100)
data = cbind(y=y, trt=trt, x)
res = squant(yvar="y", censorvar=NULL, xvars=c("x1", "x2"),
             trtvar="trt", trtcd=1, data=data, type="c", weight=NULL,
             dir="larger", quant=NULL, xvars.keep=NULL, alpha=1,
             fold=2, n.cv = 10, FDR = 0.1, progress=FALSE)


\donttest{
#predictive case with continuous response#
set.seed(888)
x=as.data.frame(matrix(rnorm(20000),200,100))
names(x) = paste("x", 1:100,sep="")
trt = sample(0:1, size=200, replace=TRUE)
y=x[,1]+x[,2]*trt+rnorm(200)
data = cbind(y=y, trt=trt, x)
res = squant(yvar="y", censorvar=NULL, xvars=paste("x", 1:100,sep=""),
             trtvar="trt", trtcd=1, data=data, type="c", weight=NULL,
             dir="larger", quant=NULL, xvars.keep=NULL, alpha=1,
             fold=5, n.cv = 50, FDR = 0.1)
res
#fitted signature#
res$squant.fit
#performance of the identified subgroup#
#including:
#  interaction p value,
#  p valve of trt difference in positive group,
#  p value of trt difference in negative group,
#  and stats for each arm in each group.
res$performance
#interpretation#
res$interpretation1
res$interpretation2

#evaluation of prediction performance#
eval.res = eval_squant(yvar="y", censorvar=NULL, trtvar="trt", trtcd=1, dir="larger",
                       type="c", data=data, squant.out=res, brief=FALSE)
#plot the subgroups#
plot(res, trt.name="Trt", ctrl.name="Ctrl")
plot(eval.res, trt.name="Trt", ctrl.name="Ctrl")



#prognostic case with survival response#
set.seed(888)
x=as.data.frame(matrix(rnorm(20000),200,100))
names(x) = paste("x", 1:100,sep="")
y=10*(10+x[,1]+rnorm(200))
data = cbind(y=y, x)
data$event = sample(c(rep(1,150),rep(0,50)))
res = squant(yvar="y", censorvar="event", xvars=paste("x", 1:100,sep=""),
             trtvar=NULL, trtcd=NULL, data=data, type="s", weight=NULL,
             dir="larger", quant=NULL, xvars.keep=NULL, alpha=1,
             fold=5, n.cv = 50, FDR = 0.1)
res

#fitted signature#
res$squant.fit
#performance of the identified subgroup#
res$performance
#evaluation of prediction performance#
eval.res = eval_squant(yvar="y", censorvar="event", trtvar=NULL, trtcd=NULL, dir="larger",
                       type="s", data=data, squant.out=res, brief=FALSE)

#plot the subgroups#
plot(res, trt.name=NULL, ctrl.name=NULL)
plot(eval.res, trt.name=NULL, ctrl.name=NULL)

}
}
