#' Write a deployable Plumber file for a vetiver model
#'
#' Use `vetiver_write_plumber()` to create a `plumber.R` file for a
#' [vetiver_model()] that has been versioned and stored via
#' [vetiver_pin_write()].
#'
#' @inheritParams pins::pin_read
#' @param ... Other arguments passed to [vetiver_pr_predict()] such as
#' the endpoint `path` or prediction `type`.
#' @param file A path to write the Plumber file. Defaults to `plumber.R` in the
#' working directory. See [plumber::plumb()] for naming precedence rules.
#'
#' @details
#' By default, this function will find and use the latest version of your
#' vetiver model; the model API (when deployed) will be linked to that specific
#' version. You can override this default behavior by choosing a specific
#' `version`.
#'
#' @return
#' The content of the `plumber.R` file, invisibly.
#'
#' @export
#'
#' @examples
#' library(pins)
#' tmp <- tempfile()
#' b <- board_temp(versioned = TRUE)
#' cars_lm <- lm(mpg ~ ., data = mtcars)
#' v <- vetiver_model(cars_lm, "cars_linear")
#' vetiver_pin_write(b, v)
#'
#' vetiver_write_plumber(b, "cars_linear", file = tmp)
#'
vetiver_write_plumber <- function(board, name, version = NULL,
                                  ...,
                                  file = "plumber.R") {

    plumber_dots <- rlang::list2(...)

    if (board$versioned) {
        if (is_null(version)) {
            version <- pins::pin_versions(board, name)
            version <- choose_version(version)
        }
        pin_read <- glue('v <- vetiver_pin_read(b, "{name}", version = "{version}")')
        v <- vetiver_pin_read(board, name, version = version)
    } else {
        pin_read <- glue('v <- vetiver_pin_read(b, "{name}")')
        v <- vetiver_pin_read(board, name)
    }

    load_infra_pkgs <- glue_collapse(glue("library({infra_pkgs})"), sep = "\n")
    load_required_pkgs <- glue_required_pkgs(v$metadata$required_pkgs)

    board <- rlang::expr_deparse(pins::board_deparse(board))
    board <- glue('b <- {board}')

    if (rlang::is_empty(plumber_dots)) {
        pr_predict <- "pr %>% vetiver_pr_predict(v)"
    } else {
        plumber_args <- glue('{names(plumber_dots)} = "{plumber_dots}"')
        plumber_args <- glue_collapse(plumber_args, sep = ", ")
        pr_predict <- glue("pr %>% vetiver_pr_predict(v, {plumber_args})")
    }

    plumber <- glue("\n
         #* @plumber
         function(pr) {{
             {pr_predict}
         }}
         ")

    ret <- compact(list(
        "# Generated by the vetiver package; edit with care\n",
        load_infra_pkgs,
        load_required_pkgs,
        board,
        pin_read,
        plumber
    ))
    readr::write_lines(ret, file = file)
}

infra_pkgs <- c("pins", "plumber", "rapidoc", "vetiver")


glue_required_pkgs <- function(required_pkgs) {
    if (!is_null(required_pkgs)) {
        required_pkgs <- sort(required_pkgs)
        required_pkgs <- glue_collapse(glue("    library({required_pkgs})"),
                                       sep = "\n")
        load_required_pkgs <- glue("\n
            # Packages needed to generate model predictions
            if (FALSE) {{
            {required_pkgs}
            }}
            ")
        return(load_required_pkgs)
    }
    NULL
}

choose_version <- function(df) {
    if (has_name(df, "active")) {
        version <- vec_slice(df, df$active)
    } else if (has_name(df, "created")) {
        idx <- head(order(df$created, decreasing = TRUE), 1)
        version <- vec_slice(df, idx)
    } else {
        version <- vec_slice(df, 1)
        warn(
            c("Pinned vetiver model has no active version and no datetime on versions",
              "Do you need to check your pinned model?",
              glue('Using version {version[["version"]]}'))
        )
    }
    version[["version"]]
}

