#' Evaluate metrics at multiple scales of aggregation
#'
#' @section Raster inputs:
#'
#' If `data` is `NULL`, then `truth` and `estimate` should both be `SpatRaster`
#' objects, as created via [terra::rast()]. These rasters will then be
#' aggregated to each grid using [exactextractr::exact_extract()].
#'
#' Grids are calculated using the bounding box of `truth`, under the assumption
#' that you may have extrapolated into regions which do not have matching "true"
#' values. This function does not check that `truth` and `estimate` overlap at
#' all, or that they are at all contained within the grid.
#'
#' @section Creating grid blocks:
#'
#' The grid blocks can be controlled by passing arguments to
#' [sf::st_make_grid()] via `...`. Some particularly useful arguments include:
#'
#' * `cellsize`: Target cellsize, expressed as the "diameter" (shortest
#' straight-line distance between opposing sides; two times the apothem)
#' of each block, in map units.
#' * `n`: The number of grid blocks in the x and y direction (columns, rows).
#' * `square`: A logical value indicating whether to create square (`TRUE`) or
#' hexagonal (`FALSE`) cells.
#'
#' If both `cellsize` and `n` are provided, then the number of blocks requested
#' by `n` of sizes specified by `cellsize` will be returned, likely not
#' lining up with the bounding box of `data`. If only `cellsize`
#' is provided, this function will return as many blocks of size
#' `cellsize` as fit inside the bounding box of `data`. If only `n` is provided,
#' then `cellsize` will be automatically adjusted to create the requested
#' number of cells.
#'
#' This function can be used for geographic or projected coordinate reference
#' systems and expects 2D data.
#'
#' @param data Either a point geometry `sf` object containing the columns
#' specified by the `truth` and `estimate` arguments, or `NULL` if `truth` and
#' `estimate` are `SpatRaster` objects.
#' @param truth,estimate If `data` is an `sf` object, the names (optionally
#' unquoted) for the columns in `data` containing the true and predicted values,
#' respectively. If `data` is `NULL`, `SpatRaster` objects with a single layer
#' containing the true and predicted values, respectively.
#' @param metrics Either a [yardstick::metric_set()] object, or a list of
#' functions which will be used to construct a [yardstick::metric_set()] object
#' specifying the performance metrics to evaluate at each scale.
#' @param grids Optionally, a list of pre-computed `sf` or `sfc` objects
#' specifying polygon boundaries to use for assessments.
#' @param ... Arguments passed to [sf::st_make_grid()].
#' **You almost certainly should provide these arguments as lists.**
#' For instance, passing `n = list(c(1, 2))` will create a single 1x2 grid;
#' passing `n = c(1, 2)` will create a 1x1 grid _and_ a 2x2 grid.
#' @param aggregation_function The function to use to aggregate predictions and
#' true values at various scales, by default [mean()]. For the `sf` method,
#' you can pass any function which takes a single vector and returns a scalar.
#' For raster methods, any function accepted by
#' [exactextractr::exact_extract()] (note that built-in function names must be
#' quoted). Note that this function does _not_ pay attention to the value of
#' `na_rm`; any NA handling you want to do during aggregation should be handled
#' by this function directly.
#' @param na_rm Boolean: Should polygons with NA values be removed before
#' calculating metrics? Note that this does _not_ impact how values are
#' aggregated to polygons: if you want to remove NA values before aggregating,
#' provide a function to `aggregation_function` which will remove NA values.
#' @param autoexpand_grid Boolean: if `data` is in geographic coordinates and
#' `grids` aren't provided, the grids generated by [sf::st_make_grid()] may not
#' contain all observations. If `TRUE`, this function will automatically expand
#' generated grids by a tiny factor to attempt to capture all observations.
#' @param progress Boolean: if `data` is `NULL`, should aggregation via
#' [exactextractr::exact_extract()] show a progress bar? Separate progress bars
#' will be shown for each time `truth` and `estimate` are aggregated.
#'
#' @return A tibble with six columns: `.metric`, with the name
#' of the metric that the row describes; `.estimator`, with the name of the
#' estimator used, `.estimate`, with the output of the metric function;
#' `.grid_args`, with the arguments passed to [sf::st_make_grid()] via `...`
#' (if any), `.grid`, containing the grids used to aggregate predictions,
#' as well as the aggregated values of `truth` and `estimate` as well as the
#' count of non-NA values for each, and `.notes`, which (if `data` is an `sf`
#' object) will indicate any observations which were not used in a given
#' assessment.
#'
#' @examplesIf rlang::is_installed("modeldata")
#' data(ames, package = "modeldata")
#' ames_sf <- sf::st_as_sf(ames, coords = c("Longitude", "Latitude"), crs = 4326)
#' ames_model <- lm(Sale_Price ~ Lot_Area, data = ames_sf)
#' ames_sf$predictions <- predict(ames_model, ames_sf)
#'
#' ww_multi_scale(
#'   ames_sf,
#'   Sale_Price,
#'   predictions,
#'   n = list(
#'     c(10, 10),
#'     c(1, 1)
#'   ),
#'   square = FALSE
#' )
#'
#' # or, mostly equivalently
#' # (there will be a slight difference due to `autoexpand_grid = TRUE`)
#' grids <- list(
#'   sf::st_make_grid(ames_sf, n = c(10, 10), square = FALSE),
#'   sf::st_make_grid(ames_sf, n = c(1, 1), square = FALSE)
#' )
#' ww_multi_scale(ames_sf, Sale_Price, predictions, grids = grids)
#'
#' @references
#' Riemann, R., Wilson, B. T., Lister, A., and Parks, S. (2010). "An effective
#' assessment protocol for continuous geospatial datasets of forest
#' characteristics using USFS Forest Inventory and Analysis (FIA) data."
#' Remote Sensing of Environment 114(10), pp 2337-2352,
#' doi: 10.1016/j.rse.2010.05.010 .
#'
#' @export
ww_multi_scale <- function(
    data = NULL,
    truth,
    estimate,
    metrics = list(yardstick::rmse, yardstick::mae),
    grids = NULL,
    ...,
    na_rm = TRUE,
    aggregation_function = "mean",
    autoexpand_grid = TRUE,
    progress = TRUE) {
  if (length(na_rm) != 1 || !is.logical(na_rm)) {
    rlang::abort("Only one logical value can be passed to `na_rm`.")
  }

  if (is.null(data)) {
    UseMethod("ww_multi_scale", truth)
  } else {
    UseMethod("ww_multi_scale", data)
  }
}

#' @exportS3Method
ww_multi_scale.SpatRaster <- function(
    data = NULL,
    truth,
    estimate,
    metrics = list(yardstick::rmse, yardstick::mae),
    grids = NULL,
    ...,
    na_rm = TRUE,
    aggregation_function = "mean",
    autoexpand_grid = TRUE,
    progress = TRUE) {
  rlang::check_installed("terra")
  rlang::check_installed("exactextractr")

  if (!inherits(truth, "SpatRaster") || terra::nlyr(truth) != 1) {
    rlang::abort("`truth` must be a SpatRaster with only one layer.")
  }

  if (!inherits(estimate, "SpatRaster") || terra::nlyr(estimate) != 1) {
    rlang::abort("`estimate` must be a SpatRaster with only one layer.")
  }

  if (sf::st_crs(truth) != sf::st_crs(estimate)) {
    rlang::abort("`truth` and `estimate` must share a CRS.")
  }

  metrics <- handle_metrics(metrics)
  grid_list <- handle_grids(truth, grids, autoexpand_grid, ...)

  grid_list$grids <- purrr::map(
    grid_list$grids,
    function(grid) {
      grid <- sf::st_as_sf(grid)
      sf::st_geometry(grid) <- "geometry"
      if (rlang::is_function(aggregation_function) || aggregation_function == "count") {
        grid$.truth <- exactextractr::exact_extract(
          truth,
          grid,
          fun = aggregation_function,
          progress = progress
        )
        grid$.estimate <- exactextractr::exact_extract(
          estimate,
          grid,
          fun = aggregation_function,
          progress = progress
        )
        grid[c(".truth", ".estimate", "geometry")]
      } else {
        truth_df <- exactextractr::exact_extract(
          truth,
          grid,
          fun = c(aggregation_function, "count"),
          progress = progress
        )
        names(truth_df) <- c(".truth", ".truth_count")

        estimate_df <- exactextractr::exact_extract(
          estimate,
          grid,
          fun = c(aggregation_function, "count"),
          progress = progress
        )
        names(estimate_df) <- c(".estimate", ".estimate_count")

        cbind(grid, truth_df, estimate_df)[c(
          ".truth",
          ".truth_count",
          ".estimate",
          ".estimate_count",
          "geometry"
        )]
      }
    }
  )

  .notes <- purrr::map(
    seq_along(grid_list$grids),
    function(idx) {
      tibble::tibble(
        note = character(0),
        missing_indices = list()
      )
    }
  )

  purrr::pmap_dfr(
    list(
      grid = grid_list$grids,
      grid_arg = grid_list$grid_arg_idx,
      .notes = .notes
    ),
    function(grid, grid_arg, .notes) {
      out <- metrics(grid, .truth, .estimate, na_rm = na_rm)
      out[attr(out, "sf_column")] <- NULL
      out$.grid_args <- list(grid_list$grid_args[grid_arg, ])
      out$.grid <- list(grid)
      out$.notes <- list(.notes)
      out
    }
  )
}

#' @exportS3Method
ww_multi_scale.sf <- function(
    data,
    truth,
    estimate,
    metrics = list(yardstick::rmse, yardstick::mae),
    grids = NULL,
    ...,
    na_rm = TRUE,
    aggregation_function = "mean",
    autoexpand_grid = TRUE,
    progress = TRUE) {
  if (nrow(data) == 0) {
    rlang::abort(
      "0 rows were passed to `data`."
    )
  }

  geom_type <- unique(sf::st_geometry_type(data))
  if (!(length(geom_type) == 1 && geom_type == "POINT")) {
    rlang::abort(
      c(
        "ww_multi_scale is currently only implemented for point geometries.",
        i = "Consider casting your data to points."
      )
    )
  }

  metrics <- handle_metrics(metrics)

  truth_var <- tidyselect::eval_select(rlang::expr({{ truth }}), data)
  estimate_var <- tidyselect::eval_select(rlang::expr({{ estimate }}), data)

  if (!is.numeric(data[[truth_var]])) {
    rlang::abort("`truth` must be numeric.")
  }

  if (!is.numeric(data[[estimate_var]])) {
    rlang::abort("`estimate` must be numeric.")
  }

  grid_list <- handle_grids(data, grids, autoexpand_grid, ...)

  grid_list$grid_intersections <- purrr::map(
    grid_list$grids,
    function(grid) {
      out <- sf::st_intersects(grid, data)
      out[purrr::map_lgl(out, function(x) !identical(x, integer(0)))]
    }
  )

  .notes <- purrr::map(
    grid_list$grid_intersections,
    function(idx) {
      missing <- setdiff(seq_len(nrow(data)), unlist(idx))

      note <- character(0)

      if (length(missing) > 0) {
        note <- "Some observations were not within any grid cell, and as such were not used in any assessments. Their row numbers are in the `missing_indices` column."
        missing <- list(missing)
      } else {
        missing <- list()
      }

      tibble::tibble(
        note = note,
        missing_indices = missing
      )
    }
  )

  if (any(purrr::map_lgl(.notes, function(x) nrow(x) > 0))) {
    rlang::warn(
      c(
        "Some observations were not within any grid cell, and as such were not used in any assessments.",
        i = "See the `.notes` column for details."
      )
    )
  }

  grid_list$grid_intersections <- purrr::map(
    grid_list$grid_intersections,
    function(idx_list) {
      out <- purrr::map_dfr(
        idx_list,
        function(idx) {
          dplyr::summarise(
            data[idx, , drop = FALSE],
            .truth = rlang::exec(.env[["aggregation_function"]], {{ truth }}),
            .truth_count = sum(!is.na({{ truth }})),
            .estimate = rlang::exec(.env[["aggregation_function"]], {{ estimate }}),
            .estimate_count = sum(!is.na({{ estimate }})),
            .groups = "keep"
          )
        }
      )

      if (dplyr::is_grouped_df(data)) {
        dplyr::group_by(out, !!!dplyr::groups(data))
      } else {
        out
      }
    }
  )

  purrr::pmap_dfr(
    list(
      dat = grid_list$grid_intersections,
      grid = grid_list$grids,
      grid_arg = grid_list$grid_arg_idx,
      .notes = .notes
    ),
    function(dat, grid, grid_arg, .notes) {
      out <- metrics(dat, .truth, .estimate, na_rm = na_rm)
      out[attr(out, "sf_column")] <- NULL
      out$.grid_args <- list(grid_list$grid_args[grid_arg, ])
      out$.grid <- list(
        suppressMessages( # We want to ignore a "names repair" message here
          sf::st_join(
            sf::st_as_sf(grid),
            dat,
            sf::st_contains
          )
        )
      )
      out$.notes <- list(.notes)
      out
    }
  )
}

handle_metrics <- function(metrics) {
  if (inherits(metrics, "metric")) metrics <- list(metrics)
  if (!inherits(metrics, "metric_set")) {
    metrics <- do.call(yardstick::metric_set, metrics)
  }
  metrics
}

handle_grids <- function(data, grids, autoexpand_grid, ...) {
  if (is.null(grids)) {
    grid_args <- rlang::list2(...)
    grid_args <- tibble::as_tibble(do.call(cbind, grid_args))
    grid_arg_idx <- seq_len(nrow(grid_args))

    grid_box <- sf::st_bbox(data)
    if (is_longlat(data) && autoexpand_grid) {
      # cf https://github.com/ropensci/stplanr/pull/467
      # basically: spherical geometry means sometimes the straight line of the
      # grid will exclude points within the bounding box
      #
      # so here we'll expand our boundary by a small bit in order to always contain our
      # points within the grid
      grid_box <- expand_grid(grid_box)
    }

    grids <- apply(
      grid_args,
      1,
      function(g_args) {
        do.call(
          sf::st_make_grid,
          c(g_args, x = list(grid_box))
        )
      }
    )
  } else {
    grid_args <- tibble::tibble()
    grid_arg_idx <- 0
  }

  if (!is.na(sf::st_crs(data))) {
    grids <- purrr::map(grids, sf::st_transform, sf::st_crs(data))
  }

  list(
    grids = grids,
    grid_args = grid_args,
    grid_arg_idx = grid_arg_idx
  )
}

#' Expand geographic bounding boxes slightly
#'
#' Because we're drawing straight lines on spheres when working with geographic
#' coordinates, it's entirely possible to have points within a bounding box but
#' outside of the straight lines between the corners. As this is almost never
#' expected, this function adds a tiny fudge factor to bounding boxes in order
#' to "catch" points.
#'
#' @param grid_box The output from [sf::st_bbox()]
#' @param expansion The expansion factor: what fraction should each coordinate
#' be adjusted by?
#'
#' @return A very slightly buffered bounding box
#'
#' @references
#' https://github.com/ropensci/stplanr/pull/467
#'
#' @noRd
expand_grid <- function(grid_box, expansion = 0.00001) {
  grid_box[1] <- grid_box[1] - abs(grid_box[1] * expansion)
  grid_box[2] <- grid_box[2] - abs(grid_box[2] * expansion)
  grid_box[3] <- grid_box[3] + abs(grid_box[3] * expansion)
  grid_box[4] <- grid_box[4] + abs(grid_box[4] * expansion)
  grid_box
}

#' Check if an sf object is in geographic coordinates
#'
#' This function adjusts [sf::st_is_longlat()] so that data without a CRS,
#' such as simulated data on arbitrary grids, is treated as non-geographic.
#'
#' @inheritParams sf::st_is_longlat
#'
#' @noRd
is_longlat <- function(x) {
  !(sf::st_crs(x) == sf::NA_crs_) && sf::st_is_longlat(x)
}
