% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mixture_identification.R
\encoding{UTF-8}
\name{mixmod_regression}
\alias{mixmod_regression}
\alias{mixmod_regression.wt_cdf_estimation}
\title{Mixture Model Identification using Segmented Regression}
\usage{
mixmod_regression(x, ...)

\method{mixmod_regression}{wt_cdf_estimation}(
  x,
  distribution = c("weibull", "lognormal", "loglogistic"),
  conf_level = 0.95,
  k = 2,
  control = segmented::seg.control(),
  ...
)
}
\arguments{
\item{x}{Object of class \code{wt_cdf_estimation} returned from
\code{\link{estimate_cdf}}.}

\item{...}{Further arguments passed to or from other methods. Currently not used.}

\item{distribution}{Supposed distribution of the random variable.}

\item{conf_level}{Confidence level of the interval. If \code{distribution} is
\code{"weibull"} this must be one of \code{0.9}, \code{0.95} or \code{0.99}.}

\item{k}{Number of mixture components. If the data should be split in an automated
fashion, \code{k} must be set to \code{NULL}. The argument \code{fix.psi}
of \code{control} is then set to \code{FALSE}.}

\item{control}{Output of the call to \code{\link[segmented]{seg.control}}, which
is passed to \code{\link[segmented:segmented]{segmented.lm}}.
See 'Examples' for usage.}
}
\value{
Returns a list with classes \code{wt_model} and
\code{wt_rank_regression} if no breakpoint was detected. See
\code{\link{rank_regression}}.

Returns a list with classes \code{wt_model} and \code{wt_mixmod_regression}
if at least one breakpoint was determined. The length of the list depends on
the number of identified subgroups. Each list element contains the
information provided by \code{\link{rank_regression}}. In addition, the
returned tibble \code{data} of each list element only retains information on
the failed units and has two more columns:
\itemize{
  \item \code{q} : Quantiles of the standard distribution calculated from
    column \code{prob}.
  \item \code{group} : Membership to the respective segment.
}

If more than one method was specified in \code{\link{estimate_cdf}}, the
resulting output is a list with classes \code{wt_model} and
\code{wt_mixmod_regression_list} where each list element has class
\code{wt_model} and \code{wt_mixmod_regression}.
}
\description{
This function uses piecewise linear regression to divide the data into
subgroups. See 'Details'.
}
\details{
The segmentation process is based on the lifetime realizations of failed
units and their corresponding estimated failure probabilities for which intact
items are taken into account. It is performed with the support of
\code{\link[segmented:segmented]{segmented.lm}}.

Segmentation can be done with a specified number of subgroups or in an automated
fashion (see argument \code{k}).
The algorithm tends to overestimate the number of breakpoints when the separation
is done automatically (see 'Warning' in \code{\link[segmented:segmented]{segmented.lm}}).

In the context of reliability analysis it is important that the main types of
failures can be identified and analyzed separately. These are
\itemize{
  \item early failures,
  \item random failures and
  \item wear-out failures.
}
In order to reduce the risk of overestimation as well as being able to consider
the main types of failures, a maximum of three subgroups (\code{k = 3}) is recommended.
}
\examples{
# Reliability data preparation:
## Data for mixture model:
data_mix <- reliability_data(
  voltage,
  x = hours,
  status = status
)

## Data for simple unimodal distribution:
data <- reliability_data(
  shock,
  x = distance,
  status = status
)

# Probability estimation with one method:
prob_mix <- estimate_cdf(
  data_mix,
  methods = "johnson"
)

prob <- estimate_cdf(
  data,
  methods = "johnson"
)

# Probability estimation for multiple methods:
prob_mix_mult <- estimate_cdf(
  data_mix,
  methods = c("johnson", "kaplan", "nelson")
)

# Example 1 - Mixture identification using k = 2 two-parametric Weibull models:
mix_mod_weibull <- mixmod_regression(
  x = prob_mix,
  distribution = "weibull",
  conf_level = 0.99,
  k = 2
)

# Example 2 - Mixture identification using k = 3 two-parametric lognormal models:
mix_mod_lognorm <- mixmod_regression(
  x = prob_mix,
  distribution = "lognormal",
  k = 3
)

# Example 3 - Mixture identification for multiple methods specified in estimate_cdf:
mix_mod_mult <- mixmod_regression(
  x = prob_mix_mult,
  distribution = "loglogistic"
)

# Example 4 - Mixture identification using control argument:
mix_mod_control <- mixmod_regression(
  x = prob_mix,
  distribution = "weibull",
  control = segmented::seg.control(display = TRUE)
)

# Example 5 - Mixture identification performs rank_regression for k = 1:
mod <- mixmod_regression(
  x = prob,
  distribution = "weibull",
  k = 1
)

}
\references{
Doganaksoy, N.; Hahn, G.; Meeker, W. Q., Reliability Analysis by
  Failure Mode, Quality Progress, 35(6), 47-52, 2002
}
