% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ranktree.R
\name{ranktree}
\alias{ranktree}
\title{Recursive partitioning method for the prediction of preference rankings based upon Kemeny distances}
\usage{
ranktree(Y, X, prunplot = FALSE, control = ranktreecontrol(...), ...)
}
\arguments{
\item{Y}{A n by m data matrix, in which there are n judges and m objects to be judged. Each row is a ranking of the objects which are represented by the columns.}

\item{X}{A dataframe containing the predictor, that must have n rows.}

\item{prunplot}{prunplot=TRUE returns the plot of the pruning sequence. Default value: FALSE}

\item{control}{a list of options that control details of the \code{ranktree} algorithm governed by the function
\code{ranktreecontrol}. The options govern the minimum size within node to split (the default value is 0.1*n, where n is the total sample size), the bound on the decrease in impurity,
(default, 0.01), the algorithm chosen to compute the median ranking (default, "quick"), and other options related
to the consrank algorithm, which is called by \code{ranktree}}

\item{\dots}{arguments passed bypassing \code{ranktreecontrol}}
}
\value{
An object of the class ranktree. See details for detailed information.
}
\description{
Recursive partitioning method for the prediction of preference rankings based upon Kemeny distances.
}
\details{
The user can use any algorithm implemented in the \code{consrank} function from the \pkg{ConsRank} package. All algorithms allow the user to set the option 'full=TRUE' 
if the median ranking(s) must be searched in the restricted space of permutations instead of in the unconstrained universe of rankings of n items including all possible ties. 
The output consists in a object of the class "ranktree". It contains:
\tabular{llll}{
X\tab \tab \tab the predictors: it must be a dataframe\cr
Y\tab \tab  \tab the response variable: the matrix of the rankings\cr
node\tab  \tab \tab a list containing teh tree-based structure:\cr
\tab number \tab \tab node number \cr
\tab terminal \tab \tab logical: TRUE is terminal node\cr
\tab father \tab \tab father node number of the current node\cr
\tab idfather \tab \tab id of the father node of the current node\cr
\tab size \tab \tab sample size within node\cr
\tab impur \tab \tab impurity at node\cr
\tab wimpur \tab \tab weighted impurity at node\cr
\tab idatnode \tab \tab id of the observations within node\cr
\tab class \tab \tab median ranking within node in terms of orderings\cr
\tab nclass \tab \tab median ranking within node in terms of rankings\cr
\tab mclass \tab \tab eventual multiple median rankings\cr
\tab tau \tab \tab Tau_x rank correlation coefficient at node\cr
\tab wtau \tab \tab weighted Tau_x rank correlation coefficient at node\cr
\tab error \tab \tab error at node\cr
\tab werror \tab \tab weighted error at node\cr
\tab varsplit \tab \tab variables generating split\cr
\tab varsplitid \tab \tab id of variables generating split\cr
\tab cutspli \tab \tab splitting point\cr
\tab children \tab \tab children nodes generated by current node\cr
\tab idchildren \tab \tab id of children nodes generated by current node\cr
\tab \dots \tab \tab other info about node\cr
control \tab \tab \tab parameters used to build the tree\cr
numnodes \tab \tab \tab number of nodes of the tree\cr
tsynt \tab \tab \tab list containing the synthesis of the tree:\cr
\tab children \tab \tab list containing all information about leaves\cr
\tab parents \tab \tab list containing all information about parent nodes\cr
geneaoly \tab  \tab \tab data frame containing information about all nodes\cr
idgenealogy \tab \tab \tab data frame containing information about all nodes in terms of nodes id\cr
idparents \tab \tab \tab id of the parents of all the nodes\cr
goodness \tab \tab \tab goodness -and badness- of fit measures of the tree: Tau_X, error, impurity\cr
nomin \tab \tab \tab information about nature of the predictors\cr 
alpha \tab \tab \tab alpha parameter for pruning sequence\cr
pruneinfo \tab \tab \tab list containing information about the pruning sequence:\cr
\tab prunelist \tab \tab information about the pruning\cr
\tab tau \tab \tab tau_X rank correlation coefficient of each subtree\cr
\tab error \tab \tab error of each subtree\cr
\tab termnodes \tab \tab number of terminal nodes of each subtree\cr
subtrees \tab \tab \tab list of each subtree created with the cost-complexity pruning procedure
}
}
\examples{
data("Univranks")
tree <- ranktree(Univranks$rankings,Univranks$predictors,num=50)

\donttest{
data(Irish)
#build the tree with default options
tree <- ranktree(Irish$rankings,Irish$predictors)

#plot the tree
plot(tree,dispclass=TRUE)

#visualize information
summary(tree)

#get information about the paths leading to terminal nodes (all the paths)
infopaths <- treepaths(tree)

#the terminal nodes
infopaths$leaves

#sample size within each terminal node
infopaths$size

#visualize the path of the second leave (terminal node number 8)
infopaths$paths[[2]]

#alternatively
nodepath(termnode=8,tree)


set.seed(132) #for reproducibility
#validation of the tree via v-fold cross-validation (default value of V=5)
vtree <- validatetree(tree,method="cv")

#extract the "best" tree
dtree <- getsubtree(tree,vtree$best_tau)

summary(dtree)

#plot the validated tree
plot(dtree,dispclass=TRUE)

#predicted rankings
rankfit <- predict(dtree,newx=Irish$predictors)

#fit of rankings
rankfit$rankings

#fit in terms of orderings
rankfit$orderings

#all info about the fit (id og the leaf, predictor values, and fit)
rankfit$orderings
}


}
\references{
D'Ambrosio, A., and Heiser W.J. (2016). A recursive partitioning method for the prediction of preference rankings based upon Kemeny distances. Psychometrika, vol. 81 (3), pp.774-94.
}
\seealso{
\code{ranktreecontrol}, \code{plot.ranktree}, \code{summary.ranktree}, \code{getsubtree}, \code{validatetree}, \code{treepaths}, \code{nodepath}
}
\author{
Antonio D'Ambrosio \email{antdambr@unina.it}
}
\keyword{Preference}
\keyword{Recursive}
\keyword{Tree-based}
\keyword{method}
\keyword{partitioning}
\keyword{rankings}
