% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PA.R
\name{PA}
\alias{PA}
\title{Parallel Analysis}
\usage{
PA(
  response,
  fa = "pc",
  n.iter = 100,
  type = "quant",
  nfact = 1,
  quant = 0.95,
  cor.type = "pearson",
  use = "pairwise.complete.obs",
  vis = TRUE,
  plot = TRUE
)
}
\arguments{
\item{response}{A required \code{N} × \code{I} matrix or data.frame consisting of the responses
of \code{N} individuals to \code{I} items.}

\item{fa}{A string that determines the method used to obtain eigenvalues in PA. If 'pc', it represents
Principal Component Analysis (PCA); if 'fa', it represents Principal Axis Factoring (a widely
used Factor Analysis method; @seealso \code{\link[EFAfactors]{factor.analysis}};
Auerswald & Moshagen, 2019). (Default = 'pc')}

\item{n.iter}{A numeric value that determines the number of simulations for the random data. (Default = 100)}

\item{type}{A string that determines the method used to calculate the reference eigenvalues from the simulated data.
If 'mean', the reference eigenvalue (\code{eigen.ref}) is the mean of the simulated eigenvalues (\code{eigen.sim});
if 'quant', the reference eigenvalue is the \code{quant} percentile of \code{eigen.sim}. (Default = 'quant')}

\item{nfact}{A numeric value that specifies the number of factors to extract, only effective when \code{fa = 'fa'}. (Default = 1)}

\item{quant}{A numeric value between 0 and 1, representing the quantile to be used for the reference
eigenvalues calculation when \code{type = 'quant'}. (Default = 0.95)}

\item{cor.type}{A character string indicating the correlation coefficient (or covariance) to
be computed. One of "pearson" (default), "kendall", or "spearman". @seealso \link[stats]{cor}.}

\item{use}{An optional character string specifying the method for computing covariances when
there are missing values. This must be one of "everything", "all.obs", "complete.obs",
"na.or.complete", or "pairwise.complete.obs" (default). @seealso \link[stats]{cor}.}

\item{vis}{A Boolean that determines whether to print the factor retention results. Set to \code{TRUE}
to print, or \code{FALSE} to suppress output. (Default = TRUE)}

\item{plot}{A Boolean that determines whether to display the PA plot. Set to \code{TRUE} to show the plot,
or \code{FALSE} to suppress it. @seealso \link[EFAfactors]{plot.PA}. (Default = TRUE)}
}
\value{
An object of class \code{PA}, which is a \code{list} containing the following components:
\item{nfact}{The number of factors to retain.}
\item{fa}{Indicates the method used to obtain eigenvalues in PA. 'pc' represents Principal Component Analysis, and 'fa' represents Principal Axis Factoring.}
\item{type}{Indicates the method used to calculate \code{eigen.ref}. If 'mean', \code{eigen.ref} is the mean of \code{eigen.sim}; if 'quant', \code{eigen.ref} is the \code{quant} percentile of \code{eigen.sim}.}
\item{eigen.value}{A vector containing the empirical eigenvalues.}
\item{eigen.ref}{A vector containing the reference eigenvalues, which depend on \code{type}.}
\item{eigen.sim}{A matrix containing the simulated eigenvalues for all iterations.}
}
\description{
This function performs Parallel Analysis (PA), which is a method used to determine the number of
factors to retain in exploratory factor analysis. It compares the empirical eigenvalues with those obtained
from simulated random data to identify the point where the observed eigenvalues are larger than those expected by chance.
The number of empirical eigenvalues that are greater than the corresponding reference eigenvalues is the number
of factors recommended to be retained by the PA method.
}
\details{
This function performs Parallel Analysis (PA; Horn, 1965; Auerswald & Moshagen, 2019) to determine the number of factors to retain.
PA is a widely used method and is considered the "gold standard" for factor retention due to its high accuracy and stability,
although it may underperform compared to methods like CD or EKC under certain conditions.
The core idea of PA is to simulate random data multiple times, for example, 100 times, and compute the eigenvalues from each simulation.
These simulated eigenvalues are then processed using either the mean or a quantile method to obtain the reference eigenvalues,
such as the i-th reference eigenvalue \eqn{\lambda_{i,ref}}.
The relationship between the i-th empirical eigenvalue \eqn{\lambda_{i}} and \eqn{\lambda_{i,ref}} indicates whether the i-th factor should be retained.
If \eqn{\lambda_{i} > \lambda_{i,ref}}, it suggests that the explanatory power of the i-th factor from the original data is stronger than that of the i-th factor from the random data,
and therefore the factor should be retained. Conversely, if \eqn{\lambda_{i} <= \lambda_{i,ref}},
it indicates that the explanatory power of the i-th factor from the original data is weaker or equal to that of the random data,
making it indistinguishable from noise, and thus the factor should not be retained. So,

\deqn{F = \sum_{i=1}^{I} I(\lambda_i > \lambda_{i,ref})}

Here, \( F \) represents the number of factors determined by the EKC, and \eqn{I(\cdot)} is the
indicator function, which equals 1 when the condition is true, and 0 otherwise.

Auerswald & Moshagen (2019) found that the most accurate results for PA were obtained when
using PCA to extract eigenvalues and using the 95th percentile of the simulated
eigenvalues to calculate the reference eigenvalues. Therefore,
the recommended settings for this function are \code{fa = 'pc'}, \code{type = 'quant'}, and \code{quant = 0.95}.
}
\examples{
library(EFAfactors)
set.seed(123)

##Take the data.bfi dataset as an example.
data(data.bfi)

response <- as.matrix(data.bfi[, 1:25]) ## loading data
response <- na.omit(response) ## Remove samples with NA/missing values

## Transform the scores of reverse-scored items to normal scoring
response[, c(1, 9, 10, 11, 12, 22, 25)] <- 6 - response[, c(1, 9, 10, 11, 12, 22, 25)] + 1


## Run PA function with default parameters.
\donttest{
PA.obj <- PA(response)

print(PA.obj)

plot(PA.obj)

## Get the eigen.value, eigen.ref and  nfact results.
eigen.value <- PA.obj$eigen.value
eigen.ref <- PA.obj$eigen.ref
nfact <- PA.obj$nfact

print(eigen.value)
print(eigen.ref)
print(nfact)

}

}
\references{
Auerswald, M., & Moshagen, M. (2019). How to determine the number of factors to retain in exploratory factor analysis: A comparison of extraction methods under realistic conditions. Psychological methods, 24(4), 468-491. https://doi.org/10.1037/met0000200.

Horn, J. L. (1965). A rationale and test for the number of factors in factor analysis. Psychometrika, 30, 179–185. http://dx.doi.org/10.1007/BF02289447.
}
\author{
Haijiang Qin <Haijiang133@outlook.com>
}
