% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/LGICARL_CUSUM_Down.R
\name{ARL_Clminus}
\alias{ARL_Clminus}
\title{ARL Estimation in CUSUM Control Charts with Gamma Distribution and Cautious Learning
for downward detection}
\usage{
ARL_Clminus(
  alpha,
  beta,
  alpha0_est,
  beta0_est,
  known_alpha,
  beta_ratio,
  H_delta,
  H_minus,
  n_I,
  replicates,
  K_l,
  delay,
  tau
)
}
\arguments{
\item{alpha}{Shape parameter of the Gamma distribution.}

\item{beta}{Scale parameter of the Gamma distribution.}

\item{alpha0_est}{Initial estimate of the shape parameter \code{alpha}. If \code{known_alpha} is \code{TRUE}, this value will be equal to \code{alpha}.}

\item{beta0_est}{Initial estimate of the scale parameter \code{beta}. This value is updated dynamically during the simulation.}

\item{known_alpha}{\code{TRUE} if \code{alpha0_est} is fixed, \code{FALSE} if it must be estimated.}

\item{beta_ratio}{Ratio between \code{beta} and its posterior estimate.}

\item{H_delta}{Increment of the lower control limit in the CUSUM chart.}

\item{H_minus}{Initial control limit of the CUSUM chart for downward detection.}

\item{n_I}{Sample size in Phase I.}

\item{replicates}{Number of Monte Carlo simulations.}

\item{K_l}{Secondary control threshold for parameter updating.}

\item{delay}{Number of observations before updating \code{beta0_est}.}

\item{tau}{Time point at which \code{beta} changes.}
}
\value{
A numeric value corresponding to the \strong{ARL} estimate for the downward CUSUM control chart with cautious learning.
}
\description{
This function calculates the \strong{Average Run Length (ARL)} of a CUSUM control chart based on the Gamma distribution,
incorporating a cautious learning scheme for the dynamic update of parameters.

The function allows the evaluation of the CUSUM chart’s performance under different parameterization scenarios,
ensuring efficient detection of process changes.

Based on the methodology presented in the work of Madrid-Alvarez, García-Díaz, and Tercero-Gómez (2024),
this implementation uses Monte Carlo simulations optimized in C++ for efficient execution
and progressive adjustment of the control chart parameters.

The values for \code{H_minus}, \code{H_delta}, \code{K_l}, \code{delay}, and \code{tau} can be referenced in the tables from the article:

\strong{Madrid-Alvarez, H. M., García-Díaz, J. C., & Tercero-Gómez, V. G. (2024).}
\emph{A CUSUM control chart for the Gamma distribution with cautious parameter learning.}
Quality Engineering, 1-23.
\subsection{Usage Scenarios:}{

\strong{Scenario 1: Known \code{alpha} and estimated \code{beta}}
\itemize{
\item The \code{alpha} parameter is assumed to be fixed and known in advance.
\item \code{beta} is estimated from a dataset or provided by the user.
\item The user must specify \code{alpha} and an initial estimate of \code{beta} (\code{beta0_est}).
}

\strong{Scenario 2: Both \code{alpha} and \code{beta} are estimated}
\itemize{
\item Both \code{alpha} and \code{beta} are estimated from an external dataset.
\item The user must calculate \code{alpha0_est} and \code{beta0_est} before calling the function.
\item \code{beta0_est} is dynamically updated during the simulation when a predefined condition is met.
}
}

\subsection{Features:}{
\itemize{
\item Implements Monte Carlo simulations for ARL estimation.
\item Allows dynamic updating of \code{beta0_est} to improve model adaptation.
\item Uses C++ optimization for efficient and precise execution.
\item Compatible with scenarios where \code{alpha} is either known or estimated.
\item Recommended values for \code{H_minus}, \code{H_delta}, \code{K_l}, \code{delay}, and \code{tau} can be found in the reference article.
}

This function is ideal for quality control studies where reliable detection of process changes
modeled with Gamma distributions is required.
}
}
\examples{
# Option 1: Provide parameters directly
ARL_Clminus(
   alpha = 1,
   beta = 1,
   alpha0_est = 1.067,  # alpha = known_alpha
   beta0_est = 0.2760,   # Estimated Beta
   known_alpha = TRUE,
   beta_ratio = 1/2,
   H_delta = 0.6946,
   H_minus = -4.8272,
   n_I = 500,
   replicates = 1000,
   K_l = 0.5,
   delay = 25,
   tau = 1
)

# Option 2: Use generated data
set.seed(123)
datos_faseI <- rgamma(n = 500, shape = 1, scale = 1)
alpha0_est <- mean(datos_faseI)^2 / var(datos_faseI)  # Alpha estimation
beta0_est <- mean(datos_faseI) / alpha0_est  # Beta estimation

ARL_Clminus(
   alpha = 1,
   beta = 1,
   alpha0_est = 1.067,  # alpha = known_alpha
   beta0_est = 0.2760,   # Estimated Beta
   known_alpha = FALSE,
   beta_ratio = 1/2,
   H_delta = 0.6946,
   H_minus = -4.8272,
   n_I = 500,
   replicates = 1000,
   K_l = 0.5,
   delay = 25,
   tau = 1
)
}
