% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimation.R
\name{one_step}
\alias{one_step}
\title{One-step estimator for unlabeled data (multi‐dist)}
\usage{
one_step(
  Y,
  Xhat,
  homoskedastic = FALSE,
  distribution = c("normal", "t", "laplace", "gamma", "beta"),
  nu = 4,
  gshape = 2,
  gscale = 1,
  ba = 2,
  bb = 2,
  intercept = TRUE
)
}
\arguments{
\item{Y}{Numeric response vector.}

\item{Xhat}{Numeric matrix of regressors \emph{excluding} the intercept. The
\strong{first} column must be the ML-generated regressor to correct.}

\item{homoskedastic}{Logical; if \code{TRUE}, assume a single error variance.}

\item{distribution}{Character: one of \code{"normal"}, \code{"t"}, \code{"laplace"}, \code{"gamma"}, or \code{"beta"}. Specifies which conditional density to use for residuals in the likelihood estimation.}

\item{nu}{Numeric; degrees of freedom (only used if \code{distribution = "t"}).}

\item{gshape, gscale}{Numeric; shape & scale for Gamma (only if \code{distribution = "gamma"}).}

\item{ba, bb}{Numeric; alpha & beta for Beta (only if \code{distribution = "beta"}).}

\item{intercept}{Logical; if \code{TRUE}, an intercept column of 1’s is prepended.}
}
\value{
An object of class \code{mlbc_fit} and subclass \code{mlbc_onestep} with:
\itemize{
\item \code{coef}: Named numeric vector of estimated coefficients.
\item \code{cov} : Variance–covariance matrix.
}
}
\description{
Fits the one-step estimator by maximizing the unlabeled likelihood via TMB,
automatically differentiating the objective, gradient, and Hessian.
}
\examples{
set.seed(2025)

# 1) Simulate “unlabeled” data
n     <- 200
p     <- 0.3
Xtrue <- rbinom(n, 1, p)
# ML regressor with 10\% false positives
Xhat  <- ifelse(runif(n) < 0.10, 1 - Xtrue, Xtrue)
Y     <- 1 + 2 * Xtrue + rnorm(n)

# 2) Small validation set to estimate fpr
m        <- 50
Xval_t   <- rbinom(m, 1, p)
Xval_h   <- ifelse(runif(m) < 0.10, 1 - Xval_t, Xval_t)
fpr_hat  <- mean(Xval_h == 1 & Xval_t == 0)

# 3) One‐step TMB estimator (Normal), with intercept
fit <- one_step(
  Y            = Y,
  Xhat         = matrix(Xhat, ncol = 1, dimnames = list(NULL, "Xhat")),
  homoskedastic = FALSE,
  distribution  = "normal",
  intercept     = TRUE
)
print(fit)
}
\references{
Battaglia, Christensen, Hansen, and Sacher (2025). "Inference for Regression with Variables Generated by AI or Machine Learning".
}
