% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_OSLcurve.R
\name{fit_OSLcurve}
\alias{fit_OSLcurve}
\title{Multi-exponential CW-OSL curve fitting}
\usage{
fit_OSLcurve(
  curve,
  K.max = 5,
  F.threshold = 150,
  stimulation.intensity = 30,
  stimulation.wavelength = 470,
  verbose = TRUE,
  output.complex = FALSE,
  parallel.computing = FALSE
)
}
\arguments{
\item{curve}{\linkS4class{RLum.Data.Curve} or \link{data.frame} or \link{matrix} (\strong{required}):
CW-OSL record or average CW-OSL curve created by \link{sum_OSLcurves}. If no column \verb{$time} exists, the first column is defined
as measurement time (x-axis). Time intervals must be constant. If no column \verb{$signal} exists, the second column is defined
as signal values (y-axis). Further columns will be ignored}

\item{K.max}{\link{numeric} (\emph{with default}):
Maximum number of components \emph{K}. The computing time increases exponentially with the component number.
\emph{K} < 7 is recommended}

\item{F.threshold}{\link{numeric} (\emph{with default}):
Fitting stop criterion. If the F-value is lower than this threshold, the fitting procedure stops and the K - 1 fit is returned}

\item{stimulation.intensity}{\link{numeric} (\emph{with default}):
Intensity of optical stimulation in \emph{mW / cm²}. Used to calculate photoionisation cross sections.}

\item{stimulation.wavelength}{\link{numeric} (\emph{with default}):
Wavelength of optical stimulation in \emph{nm}. Used to calculate photoionisation cross sections.
If a wavelength between 465 and 480 nm is chosen, the cross sections are set into
relation with literature values to name the signal components automatically.}

\item{verbose}{\link{logical} (\emph{with default}):
Enables console text output.}

\item{output.complex}{\link{logical} (\emph{with default}):
If \code{TRUE}, the function returns a list of objects, see section \strong{Value} for further information.
If \code{FALSE}, the function returns a \link{data.frame} with the CW-OSL model parameters of the fitting chosen by the F-test.
Setting the parameter to \code{FALSE} is not recommended when fitting a global average curve created by \link{sum_OSLcurves} as over-fitting is likely in such cases.}

\item{parallel.computing}{\link{logical} (\emph{with default}):
Enables the use of multiple CPU cores. This increases the execution speed significantly
but may need administrator rights and/or a firewall exception.
See \link[DEoptim:DEoptim.control]{DEoptim::DEoptim.control} for further information.}
}
\value{
If \code{output.complex = FALSE}, a \link{data.frame} is returned. It contains the signal decay rates
and signal intensities of the best fit. The best fit was either chosen by the F-test or
the last successful fitting iteration.

If \code{output.complex = TRUE}, a \link{list} of objects is returned:

\tabular{lll}{
\strong{Element} \tab \strong{Type} \tab \strong{Description}\cr
\code{decay.rates} \tab \code{numeric} \tab \link{vector} of the best suiting decay rates \cr
\code{K.selected} \tab \code{numeric} \tab number of components of the best fit \cr
\code{F.test} \tab \code{data.frame} \tab table containing the F-test parameter and the decay rates of each fitting model \cr
\code{F.test.print} \tab \code{data.frame} \tab the same table as above, but formated for pretty console and report output \cr
\code{info.text} \tab \code{character} \tab collected messages from the algorithms \cr
\code{component.tables} \tab \code{list} \tab result \link{data.frame}s for all tested models \cr
\code{curve} \tab \code{data.frame} \tab fitted time-signal-curve \cr
\code{components} \tab \code{data.frame} \tab best fit; same object as \code{output.complex = FALSE} returns \cr
\code{fit.results} \tab \code{list} \tab \link{list} of \link{nls} objects for all tested models \cr
\code{plot.data} \tab \code{data.frame} \tab factorized results for overview plotting with \link{plot_PhotoCrosssections} \cr
\code{parameters} \tab \code{list} \tab function arguments and the needed computing time
}
}
\description{
Fitting function for multi-exponentially decaying CW-OSL measurements,
based on the algorithm described by Bluszcz & Adamiec (2006).
}
\details{
The function assumes multiple exponentially decaying signal components with first-order kinetics:

\deqn{I(t) = n_1 \lambda_1 exp(-\lambda_1 t) + n_2 \lambda_2 exp(-\lambda_2 t) + ... + n_K \lambda_K exp(-\lambda_K t)}

with \eqn{I(t)} the CW-OSL signal, \eqn{n} the signal component intensity,
\eqn{\lambda} the signal component decay constant and \eqn{K} the number of signal components.
For actual fitting, the integrated version of this formula is used, see Mittelstraß et al. (2021) for details.

The fitting algorithm is an implementation of the \emph{hybrid evolutionary-linear algorithm} (HELA)
by Bluszcz & Adamiec (2006). See there or Mittelstraß et al. (in preparation) for details.
The differential evolution part of HELA is performed by \link[DEoptim:DEoptim]{DEoptim::DEoptim}.
The linear regression part of HELA is performed by \link{decompose_OSLcurve}.
The parameter refinement by Levenberg-Marquardt fitting is performed by \link[minpack.lm:nlsLM]{minpack.lm::nlsLM}.

\strong{F-test}

Bluszcz & Adamiec (2006) suggest the use of an F-test to determine the correct number of signal components.
This function compares the residual square sum (\emph{RSS_K}) value of each
fitting with the \emph{RSS_{K-1}} value of the previous fitting and calculates
an \emph{Improvement-in-fitting-quality} criterion:

\deqn{F_K = {(RSS_{K-1} - RSS_K)/2} / {RSS_K(N - 2K)}}

Here, \emph{N} is the number data points (channels) of the measurement and \emph{K} is the number of OSL components
in the fitting model. If \emph{F_K} falls below the threshold value (\code{F.threshold}), the fitting model
with \emph{K} components is apparently not significantly better than the \emph{K} - 1 model of the previous fitting cycle.
Thus, the \emph{K} - 1 model will be recommended as fitting solution.

\strong{Photoionisation cross sections}

While the function is suited for the analysis of a wide variety of multi-exponential decay problems,
it is targeted to CW-OSL measurements of quartz under SAR protocol conditions (470 nm stimulation at 125 °C).
To compare the calculated OSL components with OSL components reported in published literature,
photoionisation cross sections are calculated using the \code{stimulation.wavelength} \eqn{\lambda_{stim}}  and
\code{stimulation.intensity} \eqn{\Phi_{stim}}:

\deqn{\sigma_k=\lambda_k {hc / \Phi_{stim}\lambda_{stim}}}

Here \eqn{\sigma_k} is the photoionisation cross section of component \emph{k} in cm^2,
\eqn{\lambda_k} the CW-OSL decay constant in s^-1, \emph{h} the Planck constant and \emph{c} the speed of light.

If a \code{stimulation.intensity} between 460 nm and 485 nm is defined,
the components are named automatically in accordance to the
cross-sections published by Durcan and Duller (2011), Jain et al. (2003) and Singarayer and Bailey (2003).
For the Ultrafast and the Slow4 component, no consistent literature values could be found, so their range
is tentatively assigned:

\tabular{lll}{
\strong{Component} \tab \strong{Lower limit (cm^2)} \tab \strong{Upper limit (cm^2)}\cr
Ultrafast \tab 1e-16 \tab 1e-15 \cr
Fast \tab 1.9e-17 \tab 3.1e-17 \cr
Medium \tab 3e-18 \tab 9e-18 \cr
Slow1 \tab 1e-18 \tab 1.85e-18 \cr
Slow2 \tab 1.1e-19 \tab 4e-19 \cr
Slow3 \tab 1e-20 \tab 4.67e-20 \cr
Slow4 \tab 1e-21 \tab 1e-20
}
}
\section{Last update}{


2022-07-27, DM: Moved residual sum of squares (RSS) calculation during DE-optimization cycle to decompose_OSLcurve() to improve computing time by factor 3 to 4
}

\examples{

# Create a simple curve with just one component
curve <- data.frame(
  X = c(1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12),
  Y = c(377, 244, 163, 93, 59, 28, 17, 13, 10, 8, 9, 5))
# Perform fitting
components <- fit_OSLcurve(curve, F.threshold = 3)

# Display results
plot_OSLcurve(curve, components)

}
\references{
Bluszcz, A., Adamiec, G., 2006. Application of differential evolution to fitting OSL
decay curves. Radiation Measurements 41, 886–891.

Durcan, J.A., Duller, G.A.T., 2011. The fast ratio: A rapid measure for testing the dominance of the fast component in the initial OSL signal from quartz. Radiation Measurements 46, 1065–1072.

Jain, M., Murray, A.S., Bøtter-Jensen, L., 2003. Characterisation of blue-light stimulated luminescence components in different quartz samples: implications for dose measurement. Radiation Measurements 37, 441–449.

Mittelstraß, D., 2019. Decomposition of weak optically stimulated luminescence signals and
its application in retrospective dosimetry at quartz (Master thesis). TU Dresden, Dresden.

Singarayer, J.S., Bailey, R.M., 2003. Further investigations of the quartz optically stimulated luminescence components using linear modulation.
Radiation Measurements, Proceedings of the 10th international Conference on Luminescence and Electron-Spin Resonance Dating (LED 2002) 37, 451–458.
}
\seealso{
\link{RLum.OSL_decomposition}, \link{sum_OSLcurves}, \link{decompose_OSLcurve}, \link{plot_OSLcurve},
\link{plot_PhotoCrosssections}, \link[minpack.lm:nlsLM]{minpack.lm::nlsLM}, \link[DEoptim:DEoptim]{DEoptim::DEoptim}
}
\author{
Dirk Mittelstraß, \email{dirk.mittelstrass@luminescence.de}

Please cite the package the following way:

Mittelstraß, D., Schmidt, C., Beyer, J., Heitmann, J. and Straessner, A.:
R package OSLdecomposition: Automated identification and separation of quartz CW-OSL signal components, \emph{in preparation}.
}
