% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tiger_roads_sf.R
\name{tiger_roads_sf}
\alias{tiger_roads_sf}
\title{tiger_roads_sf}
\usage{
tiger_roads_sf(
  state = NULL,
  county = NULL,
  output_dir = tempdir(),
  delete_files = TRUE,
  vintage = 2020,
  entity = "us_roads",
  set_crs = NULL,
  transform_crs = NULL,
  sf_info = FALSE,
  do_progress = FALSE,
  shapefile = NULL,
  datafile = NULL,
  datafile_key = NULL,
  sf_key = "GEOID",
  express = NULL,
  check_na = FALSE
)
}
\arguments{
\item{state}{A required 2-digit FIPS code for the state of interest.
See \href{https://cran.r-project.org/package=usmap}{usmap::fips function} for finding FIPS codes.}

\item{county}{The \emph{three-digit} FIPS code for the county of interest.}

\item{output_dir}{A full directory path where the shapefile and its associated files will be downloaded.
The default is the directory defined by the value returned by \code{tempdir()}.}

\item{delete_files}{A logical which if \code{TRUE} will delete the shapefile and associated files in 'output_dir'.
The default is TRUE.}

\item{vintage}{A numeric that sets the vintage of interest. The default is 2020.
The value should be greater than 2010.}

\item{entity}{A string that defines the category of road geometries to return. Acceptable value are
"us_roads" (state & county arguments not required), "state_roads" (state argument required),
"county_roads" (state & county arguments required).}

\item{set_crs}{A numeric or character string which if non-NULL calls \code{sf::st_crs()} to set the crs of the geometries and transforms them.}

\item{transform_crs}{A numeric or character string which if non-NULL calls \code{sf::st_transform()}
to perform a crs transform of the geometries. Note that the crs of the shapefile must not be \code{NA}.}

\item{sf_info}{A logical which if \code{TRUE} displays info on the resulting simple feature object.}

\item{do_progress}{A logical which if \code{TRUE} displays a progress bar during the download.}

\item{shapefile}{A full file path to a shapefile folder with its unzipped files to be processed instead of downloading.}

\item{datafile}{A dataframe containing data that should be joined with this function's resultant simple feature object.}

\item{datafile_key}{The column name from 'datafile' dataframe used to key with the 'sf_key' column of the resultant simple feature dataframe.}

\item{sf_key}{The column from the resultant dataframe used to key with the 'datafile' dataframe.}

\item{express}{A logical expression object used to filter the resultant simple feature dataframe.
For example, one of the columns of the resultant simple feature dataframe is "STATEFP".
If you wanted to return just the geometries for Florida (which has a fips code of "12"),
then you assign 'express' equal to: expression(STATEFP == "12"). The expression will be
evaluated and only the geometries for Florida will be returned.}

\item{check_na}{A logical which if \code{TRUE} will remove rows that have missing values for any of the columns.
The default is to not check the columns for \code{NA} values.

\strong{Note: vintage must be greater than 2010}}
}
\value{
A data frame object of class sf
}
\description{
This function performs three tasks:
\enumerate{
\item Download to an output directory a zip file from the TIGER/Line Shapefiles database.
\item Unzip the zip file and locate the shape file of interest.
\item Read and convert the shape file to a simple feature object.
}
}
\details{
Returns simple feature (sf) of US Census roads boundary related geometric polygons,
provided by the US Census Bureau's TIGER/Line Shapefiles database. See
\href{https://r-spatial.github.io/sf/articles/sf1.html}{Simple Features for R}
for more information on simple features. Along with the geometries, additional roads related
variables are provided.  See \href{https://www2.census.gov/geo/pdfs/maps-data/data/tiger/tgrshp2022/TGRSHP2022_TechDoc_F-R.pdf}{Appendix L-3. Record Layouts: Roads Shapefile)}
for a description of road related variables of the sf file. For further information on the Census Bureau's shape files see
\href{https://www2.census.gov/geo/pdfs/maps-data/data/tiger/tgrshp2021/TGRSHP2021_TechDoc_Ch3.pdf}{About the 2021 TIGER/Line Shapefiles}.
From \href{https://www2.census.gov/geo/pdfs/maps-data/data/tiger/tgrshp2022/TGRSHP2022_TechDoc_Ch4.pdf}{Chapter 4.12.3 Roads – Primary, Secondary and All Roads} --
"The primary and secondary roads shapefile contains all linear street features with MTFCCs of primary
roads (S1100) or secondary roads (S1200) in the MAF/TIGER System. Secondary roads are main
arteries, usually in the U.S. highway, state highway, or county highway system. These roads have one or
more lanes of traffic in each direction, may or may not be divided, and usually have at-grade intersections
with many other roads and driveways. These roads often have both a local name and a route number."

The function returns the simple feature object which can easily be mapped (see \href{https://github.com/deandevl/RplotterPkg}{RplotterPkg::create_sf_plot()}) or
joined with US Census Bureau demographic data via the GEOID value.

Some earlier vintages may have NA for the crs so you may need to specify the 'crs_transform' to 3426.  Also
you may be interested in using a state level crs. See \href{https://epsg.io/}{epsg.io} to search worldwide for crs.
}
\examples{
library(httr)
library(sf)
library(data.table)
library(withr)
library(usmap)
library(RcensusPkg)

# Get the sf geometries of roads in Ohio
# Get the fips for Ohio
oh_fips <- usmap::fips(state = "ohio")

# Define a temporary, self deleting output folder for the downloaded shapefiles
output_dir <- withr::local_tempdir()
if(!dir.exists(output_dir)){
  dir.create(output_dir)
}

# Get the sf object
ohio_roads_sf <- RcensusPkg::tiger_roads_sf(
  state = oh_fips,
  entity = "state_roads",
  output_dir = output_dir,
  delete_files = FALSE
)

}
