% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derive_var_joined_exist_flag.R
\name{derive_var_joined_exist_flag}
\alias{derive_var_joined_exist_flag}
\title{Derives a Flag Based on an Existing Flag}
\usage{
derive_var_joined_exist_flag(
  dataset,
  dataset_add,
  by_vars,
  order,
  new_var,
  tmp_obs_nr_var = NULL,
  join_vars,
  join_type,
  first_cond_lower = NULL,
  first_cond_upper = NULL,
  filter_add = NULL,
  filter_join,
  true_value = "Y",
  false_value = NA_character_,
  check_type = "warning"
)
}
\arguments{
\item{dataset}{Input dataset

The variables specified by the \code{by_vars} and \code{join_vars} arguments are expected to be in the dataset.}

\item{dataset_add}{Additional dataset

The variables specified for \code{by_vars}, \code{join_vars}, and \code{order} are
expected.}

\item{by_vars}{Grouping variables

The specified variables are used for joining the input
dataset (\code{dataset}) with the additional dataset (\code{dataset_add}).

\emph{Permitted Values}: list of variables created by \code{exprs()}
e.g. \code{exprs(USUBJID, VISIT)}}

\item{order}{Order

The observations are ordered by the specified order.

For handling of \code{NA}s in sorting variables see \href{../articles/generic.html#sort_order}{Sort Order}.}

\item{new_var}{New variable

The specified variable is added to the input dataset.}

\item{tmp_obs_nr_var}{Temporary observation number

The specified variable is added to the input dataset (\code{dataset}) and the
additional dataset (\code{dataset_add}). It is set to the observation number
with respect to \code{order}. For each by group (\code{by_vars}) the observation
number starts with \code{1}. The variable can be used in the conditions
(\code{filter_join}, \code{first_cond_upper}, \code{first_cond_lower}). It is not included
in the output dataset. It can also be used to flag consecutive observations
or the last observation (see last example below).}

\item{join_vars}{Variables to keep from joined dataset

The variables needed from the other observations should be specified
for this parameter. The specified variables are added to the joined dataset
with suffix ".join". For example to flag all observations with \code{AVALC == "Y"} and \code{AVALC == "Y"} for at least one subsequent visit \code{join_vars = exprs(AVALC, AVISITN)} and \code{filter_join = AVALC == "Y" & AVALC.join == "Y" & AVISITN < AVISITN.join} could be specified.

The \verb{*.join} variables are not included in the output dataset.}

\item{join_type}{Observations to keep after joining

The argument determines which of the joined observations are kept with
respect to the original observation. For example, if \code{join_type = "after"}
is specified all observations after the original observations are kept.

For example for confirmed response or BOR in the oncology setting or
confirmed deterioration in questionnaires the confirmatory assessment must
be after the assessment. Thus \code{join_type = "after"} could be used.

Whereas, sometimes you might allow for confirmatory observations to occur
prior to the observation. For example, to identify AEs occurring on or
after seven days before a COVID AE. Thus \code{join_type = "all"} could be used.

\emph{Permitted Values:} \code{"before"}, \code{"after"}, \code{"all"}}

\item{first_cond_lower}{Condition for selecting range of data (before)

If this argument is specified, the other observations are restricted from
the first observation before the current observation where the specified
condition is fulfilled up to the current observation. If the condition is
not fulfilled for any of the other observations, no observations are
considered, i.e., the observation is not flagged.

This parameter should be specified if \code{filter_join} contains summary
functions which should not apply to all observations but only from a
certain observation before the current observation up to the current
observation. For an example see the last example below.}

\item{first_cond_upper}{Condition for selecting range of data (after)

If this argument is specified, the other observations are restricted up to
the first observation where the specified condition is fulfilled. If the
condition is not fulfilled for any of the other observations, no
observations are considered, i.e., the observation is not flagged.

This parameter should be specified if \code{filter_join} contains summary
functions which should not apply to all observations but only up to the
confirmation assessment. For an example see the third example below.}

\item{filter_add}{Filter for additional dataset (\code{dataset_add})

Only observations from \code{dataset_add} fulfilling the specified condition are
joined to the input dataset. If the argument is not specified, all
observations are joined.

Variables created by \code{order} or \code{new_vars} arguments can be used in the
condition.

The condition can include summary functions like \code{all()} or \code{any()}. The
additional dataset is grouped by the by variables (\code{by_vars}).

\emph{Permitted Values}: a condition}

\item{filter_join}{Condition for selecting observations

The filter is applied to the joined dataset for flagging the confirmed
observations. The condition can include summary functions like \code{all()} or
\code{any()}. The joined dataset is grouped by the original observations. I.e.,
the summary function are applied to all observations up to the confirmation
observation. For example, \code{filter_join = AVALC == "CR" & all(AVALC.join \%in\% c("CR", "NE")) & count_vals(var = AVALC.join, val = "NE") <= 1}
selects observations with response "CR" and for all observations up to the
confirmation observation the response is "CR" or "NE" and there is at most
one "NE".}

\item{true_value}{Value of \code{new_var} for flagged observations}

\item{false_value}{Value of \code{new_var} for observations not flagged}

\item{check_type}{Check uniqueness?

If \code{"warning"} or \code{"error"} is specified, the specified message is issued
if the observations of the input dataset are not unique with respect to the
by variables and the order.

\emph{Permitted Values:} \code{"none"}, \code{"warning"}, \code{"error"}}
}
\value{
The input dataset with the variable specified by \code{new_var} added.
}
\description{
Derive a flag which depends on other observations of the dataset. For
example, flagging events which need to be confirmed by a second event.
}
\details{
An example usage might be flagging if a patient received two required
medications within a certain timeframe of each other.

In the oncology setting, for example, the function could be used to flag if a
response value can be confirmed by an other assessment. This is commonly
used in endpoints such as best overall response.

The following steps are performed to produce the output dataset.
\subsection{Step 1}{
\itemize{
\item The variables specified by \code{order} are added to the additional dataset
(\code{dataset_add}).
\item The variables specified by \code{join_vars} are added to the additional dataset
(\code{dataset_add}).
\item The records from the additional dataset (\code{dataset_add}) are restricted to
those matching the \code{filter_add} condition.
}

The input dataset (\code{dataset}) is joined with the restricted additional
dataset by the variables specified for \code{by_vars}. From the additional
dataset only the variables specified for \code{join_vars} are kept. The suffix
".join" is added to those variables which also exist in the input dataset.

For example, for \code{by_vars = USUBJID}, \code{join_vars = exprs(AVISITN, AVALC)}
and input dataset and additional dataset

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# A tibble: 2 x 4
USUBJID AVISITN AVALC  AVAL
<chr>     <dbl> <chr> <dbl>
1             1 Y         1
1             2 N         0
}\if{html}{\out{</div>}}

the joined dataset is

\if{html}{\out{<div class="sourceCode r">}}\preformatted{A tibble: 4 x 6
USUBJID AVISITN AVALC  AVAL AVISITN.join AVALC.join
<chr>     <dbl> <chr> <dbl>        <dbl> <chr>
1             1 Y         1            1 Y
1             1 Y         1            2 N
1             2 N         0            1 Y
1             2 N         0            2 N
}\if{html}{\out{</div>}}
}

\subsection{Step 2}{

The joined dataset is restricted to observations with respect to
\code{join_type} and \code{order}.

The dataset from the example in the previous step with \code{join_type = "after"} and \code{order = exprs(AVISITN)} is restricted to

\if{html}{\out{<div class="sourceCode r">}}\preformatted{A tibble: 4 x 6
USUBJID AVISITN AVALC  AVAL AVISITN.join AVALC.join
<chr>     <dbl> <chr> <dbl>        <dbl> <chr>
1             1 Y         1            2 N
}\if{html}{\out{</div>}}
}

\subsection{Step 3}{

If \code{first_cond_lower} is specified, for each observation of the input
dataset the joined dataset is restricted to observations from the first
observation where \code{first_cond_lower} is fulfilled (the observation
fulfilling the condition is included) up to the observation of the input
dataset. If for an observation of the input dataset the condition is not
fulfilled, the observation is removed.

If \code{first_cond_upper} is specified, for each observation of the input
dataset the joined dataset is restricted to observations up to the first
observation where \code{first_cond_upper} is fulfilled (the observation
fulfilling the condition is included). If for an observation of the input
dataset the condition is not fulfilled, the observation is removed.

For an example see the last example in the "Examples" section.
}

\subsection{Step 4}{

The joined dataset is grouped by the observations from the input dataset
and restricted to the observations fulfilling the condition specified by
\code{filter_join}.
}

\subsection{Step 5}{

The first observation of each group is selected
}

\subsection{Step 6}{

The variable specified by \code{new_var} is added to the input dataset. It is
set to \code{true_value} for all observations which were selected in the
previous step. For the other observations it is set to \code{false_value}.

\strong{Note:} This function creates temporary datasets which may be much bigger
than the input datasets. If this causes memory issues, please try setting
the admiral option \code{save_memory} to \code{TRUE} (see \code{set_admiral_options()}).
This reduces the memory consumption but increases the run-time.
}
}
\examples{
library(tibble)

# flag observations with a duration longer than 30 and
# at, after, or up to 7 days before a COVID AE (ACOVFL == "Y")
adae <- tribble(
  ~USUBJID, ~ADY, ~ACOVFL, ~ADURN,
  "1",        10, "N",          1,
  "1",        21, "N",         50,
  "1",        23, "Y",         14,
  "1",        32, "N",         31,
  "1",        42, "N",         20,
  "2",        11, "Y",         13,
  "2",        23, "N",          2,
  "3",        13, "Y",         12,
  "4",        14, "N",         32,
  "4",        21, "N",         41
)

derive_var_joined_exist_flag(
  adae,
  dataset_add = adae,
  new_var = ALCOVFL,
  by_vars = exprs(USUBJID),
  join_vars = exprs(ACOVFL, ADY),
  join_type = "all",
  order = exprs(ADY),
  filter_join = ADURN > 30 & ACOVFL.join == "Y" & ADY >= ADY.join - 7
)

# flag observations with AVALC == "Y" and AVALC == "Y" at one subsequent visit
data <- tribble(
  ~USUBJID, ~AVISITN, ~AVALC,
  "1",      1,        "Y",
  "1",      2,        "N",
  "1",      3,        "Y",
  "1",      4,        "N",
  "2",      1,        "Y",
  "2",      2,        "N",
  "3",      1,        "Y",
  "4",      1,        "N",
  "4",      2,        "N",
)

derive_var_joined_exist_flag(
  data,
  dataset_add = data,
  by_vars = exprs(USUBJID),
  new_var = CONFFL,
  join_vars = exprs(AVALC, AVISITN),
  join_type = "after",
  order = exprs(AVISITN),
  filter_join = AVALC == "Y" & AVALC.join == "Y" & AVISITN < AVISITN.join
)

# select observations with AVALC == "CR", AVALC == "CR" at a subsequent visit,
# only "CR" or "NE" in between, and at most one "NE" in between
data <- tribble(
  ~USUBJID, ~AVISITN, ~AVALC,
  "1",      1,        "PR",
  "1",      2,        "CR",
  "1",      3,        "NE",
  "1",      4,        "CR",
  "1",      5,        "NE",
  "2",      1,        "CR",
  "2",      2,        "PR",
  "2",      3,        "CR",
  "3",      1,        "CR",
  "4",      1,        "CR",
  "4",      2,        "NE",
  "4",      3,        "NE",
  "4",      4,        "CR",
  "4",      5,        "PR"
)

derive_var_joined_exist_flag(
  data,
  dataset_add = data,
  by_vars = exprs(USUBJID),
  join_vars = exprs(AVALC),
  join_type = "after",
  order = exprs(AVISITN),
  new_var = CONFFL,
  first_cond_upper = AVALC.join == "CR",
  filter_join = AVALC == "CR" & all(AVALC.join \%in\% c("CR", "NE")) &
    count_vals(var = AVALC.join, val = "NE") <= 1
)

# flag observations with AVALC == "PR", AVALC == "CR" or AVALC == "PR"
# at a subsequent visit at least 20 days later, only "CR", "PR", or "NE"
# in between, at most one "NE" in between, and "CR" is not followed by "PR"
data <- tribble(
  ~USUBJID, ~ADY, ~AVALC,
  "1",         6, "PR",
  "1",        12, "CR",
  "1",        24, "NE",
  "1",        32, "CR",
  "1",        48, "PR",
  "2",         3, "PR",
  "2",        21, "CR",
  "2",        33, "PR",
  "3",        11, "PR",
  "4",         7, "PR",
  "4",        12, "NE",
  "4",        24, "NE",
  "4",        32, "PR",
  "4",        55, "PR"
)

derive_var_joined_exist_flag(
  data,
  dataset_add = data,
  by_vars = exprs(USUBJID),
  join_vars = exprs(AVALC, ADY),
  join_type = "after",
  order = exprs(ADY),
  new_var = CONFFL,
  first_cond_upper = AVALC.join \%in\% c("CR", "PR") & ADY.join - ADY >= 20,
  filter_join = AVALC == "PR" &
    all(AVALC.join \%in\% c("CR", "PR", "NE")) &
    count_vals(var = AVALC.join, val = "NE") <= 1 &
    (
      min_cond(var = ADY.join, cond = AVALC.join == "CR") >
        max_cond(var = ADY.join, cond = AVALC.join == "PR") |
        count_vals(var = AVALC.join, val = "CR") == 0
    )
)

# flag observations with CRIT1FL == "Y" at two consecutive visits or at the last visit
data <- tribble(
  ~USUBJID, ~AVISITN, ~CRIT1FL,
  "1",      1,        "Y",
  "1",      2,        "N",
  "1",      3,        "Y",
  "1",      5,        "N",
  "2",      1,        "Y",
  "2",      3,        "Y",
  "2",      5,        "N",
  "3",      1,        "Y",
  "4",      1,        "Y",
  "4",      2,        "N",
)

derive_var_joined_exist_flag(
  data,
  dataset_add = data,
  by_vars = exprs(USUBJID),
  new_var = CONFFL,
  tmp_obs_nr_var = tmp_obs_nr,
  join_vars = exprs(CRIT1FL),
  join_type = "all",
  order = exprs(AVISITN),
  filter_join = CRIT1FL == "Y" & CRIT1FL.join == "Y" &
    (tmp_obs_nr + 1 == tmp_obs_nr.join | tmp_obs_nr == max(tmp_obs_nr.join))
)

# first_cond_lower and first_cond_upper argument
myd <- tribble(
  ~subj, ~day, ~val,
  "1",      1, "++",
  "1",      2, "-",
  "1",      3, "0",
  "1",      4, "+",
  "1",      5, "++",
  "1",      6, "-",
  "2",      1, "-",
  "2",      2, "++",
  "2",      3, "+",
  "2",      4, "0",
  "2",      5, "-",
  "2",      6, "++"
)

# flag "0" where all results from the first "++" before the "0" up to the "0"
# (excluding the "0") are "+" or "++"
derive_var_joined_exist_flag(
  myd,
  dataset_add = myd,
  by_vars = exprs(subj),
  order = exprs(day),
  new_var = flag,
  join_vars = exprs(val),
  join_type = "before",
  first_cond_lower = val.join == "++",
  filter_join = val == "0" & all(val.join \%in\% c("+", "++"))
)

# flag "0" where all results from the "0" (excluding the "0") up to the first
# "++" after the "0" are "+" or "++"
derive_var_joined_exist_flag(
  myd,
  dataset_add = myd,
  by_vars = exprs(subj),
  order = exprs(day),
  new_var = flag,
  join_vars = exprs(val),
  join_type = "after",
  first_cond_upper = val.join == "++",
  filter_join = val == "0" & all(val.join \%in\% c("+", "++"))
)

# flag each dose which is lower than the previous dose per subject
ex <- tribble(
  ~USUBJID, ~EXSTDTM,          ~EXDOSE,
  "1",      "2024-01-01T08:00",      2,
  "1",      "2024-01-02T08:00",      4,
  "2",      "2024-01-01T08:30",      1,
  "2",      "2024-01-02T08:30",      4,
  "2",      "2024-01-03T08:30",      3,
  "2",      "2024-01-04T08:30",      2,
  "2",      "2024-01-05T08:30",      2
)

derive_var_joined_exist_flag(
  ex,
  dataset_add = ex,
  by_vars = exprs(USUBJID),
  order = exprs(EXSTDTM),
  new_var = DOSREDFL,
  tmp_obs_nr_var = tmp_dose_nr,
  join_vars = exprs(EXDOSE),
  join_type = "before",
  filter_join = (
    tmp_dose_nr == tmp_dose_nr.join + 1 # Look only at adjacent doses
    & EXDOSE > 0 & EXDOSE.join > 0 # Both doses are valid
    & EXDOSE < EXDOSE.join # Dose is lower than previous
  )
)

# derive definitive deterioration flag as any deterioration (CHGCAT1 = "Worsened")
# by parameter that is not followed by a non-deterioration
adqs <- tribble(
  ~USUBJID, ~PARAMCD, ~ADY, ~CHGCAT1,
  "1",      "QS1",      10, "Improved",
  "1",      "QS1",      21, "Improved",
  "1",      "QS1",      23, "Improved",
  "1",      "QS2",      32, "Worsened",
  "1",      "QS2",      42, "Improved",
  "2",      "QS1",      11, "Worsened",
  "2",      "QS1",      24, "Worsened"
)

derive_var_joined_exist_flag(
  adqs,
  dataset_add = adqs,
  new_var = DDETERFL,
  by_vars = exprs(USUBJID, PARAMCD),
  join_vars = exprs(CHGCAT1),
  join_type = "all",
  order = exprs(ADY),
  filter_join = all(CHGCAT1.join == "Worsened" | ADY > ADY.join)
)
}
\seealso{
\code{\link[=filter_joined]{filter_joined()}}, \code{\link[=derive_vars_joined]{derive_vars_joined()}}

General Derivation Functions for all ADaMs that returns variable appended to dataset:
\code{\link{derive_var_extreme_flag}()},
\code{\link{derive_var_merged_ef_msrc}()},
\code{\link{derive_var_merged_exist_flag}()},
\code{\link{derive_var_merged_summary}()},
\code{\link{derive_var_obs_number}()},
\code{\link{derive_var_relative_flag}()},
\code{\link{derive_vars_cat}()},
\code{\link{derive_vars_computed}()},
\code{\link{derive_vars_joined}()},
\code{\link{derive_vars_merged}()},
\code{\link{derive_vars_merged_lookup}()},
\code{\link{derive_vars_transposed}()}
}
\concept{der_gen}
\keyword{der_gen}
