% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ci.R
\name{ci}
\alias{ci}
\alias{ci.numeric}
\alias{ci.data.frame}
\alias{ci.brmsfit}
\title{Confidence/Credible/Compatibility Interval (CI)}
\usage{
ci(x, ...)

\method{ci}{numeric}(x, ci = 0.95, method = "ETI", verbose = TRUE, BF = 1, ...)

\method{ci}{data.frame}(x, ci = 0.95, method = "ETI", BF = 1, rvar_col = NULL, verbose = TRUE, ...)

\method{ci}{brmsfit}(
  x,
  ci = 0.95,
  method = "ETI",
  effects = "fixed",
  component = "conditional",
  parameters = NULL,
  verbose = TRUE,
  BF = 1,
  ...
)
}
\arguments{
\item{x}{A \code{stanreg} or \code{brmsfit} model, or a vector representing a posterior
distribution.}

\item{...}{Currently not used.}

\item{ci}{Value or vector of probability of the CI (between 0 and 1)
to be estimated. Default to \code{0.95} (\verb{95\%}).}

\item{method}{Can be \link[=eti]{"ETI"} (default), \link[=hdi]{"HDI"}, \link[=bci]{"BCI"},
\link[=spi]{"SPI"} or \link[=si]{"SI"}.}

\item{verbose}{Toggle off warnings.}

\item{BF}{The amount of support required to be included in the support interval.}

\item{rvar_col}{A single character - the name of an \code{rvar} column in the data
frame to be processed. See example in \code{\link[=p_direction]{p_direction()}}.}

\item{effects}{Should results for fixed effects (\code{"fixed"}, the default),
random effects (\code{"random"}) or both ("\verb{all"}) be returned? Only applies to
mixed models. May be abbreviated.}

\item{component}{Which type of parameters to return, such as parameters for
the conditional model, the zero-inflated part of the model, the dispersion
term, etc. See details in section \emph{Model Components}. May be abbreviated.
Note that the \emph{conditional} component also refers to the \emph{count} or \emph{mean}
component - names may differ, depending on the modeling package. There are
three convenient shortcuts (not applicable to \emph{all} model classes):
\itemize{
\item \code{component = "all"} returns all possible parameters.
\item If \code{component = "location"}, location parameters such as \code{conditional},
\code{zero_inflated}, \code{smooth_terms}, or \code{instruments} are returned (everything
that are fixed or random effects - depending on the \code{effects} argument -
but no auxiliary parameters).
\item For \code{component = "distributional"} (or \code{"auxiliary"}), components like
\code{sigma}, \code{dispersion}, \code{beta} or \code{precision} (and other auxiliary
parameters) are returned.
}}

\item{parameters}{Regular expression pattern that describes the parameters
that should be returned. Meta-parameters (like \code{lp__} or \code{prior_}) are
filtered by default, so only parameters that typically appear in the
\code{summary()} are returned. Use \code{parameters} to select specific parameters
for the output.}
}
\value{
A data frame with following columns:
\itemize{
\item \code{Parameter} The model parameter(s), if \code{x} is a model-object. If \code{x} is a
vector, this column is missing.
\item \code{CI} The probability of the credible interval.
\item \code{CI_low}, \code{CI_high} The lower and upper credible interval limits for the parameters.
}
}
\description{
Compute Confidence/Credible/Compatibility Intervals (CI) or Support Intervals
(SI) for Bayesian and frequentist models. The Documentation is accessible
for:
}
\details{
\itemize{
\item \href{https://easystats.github.io/bayestestR/articles/credible_interval.html}{Bayesian models}
\item \href{https://easystats.github.io/parameters/reference/ci.default.html}{Frequentist models}
}
}
\note{
When it comes to interpretation, we recommend thinking of the CI in terms of
an "uncertainty" or "compatibility" interval, the latter being defined as
"Given any value in the interval and the background assumptions,
the data should not seem very surprising" (\emph{Gelman & Greenland 2019}).

There is also a \href{https://easystats.github.io/see/articles/bayestestR.html}{\code{plot()}-method} implemented in the \href{https://easystats.github.io/see/}{\pkg{see}-package}.
}
\section{Model components}{


Possible values for the \code{component} argument depend on the model class.
Following are valid options:
\itemize{
\item \code{"all"}: returns all model components, applies to all models, but will only
have an effect for models with more than just the conditional model
component.
\item \code{"conditional"}: only returns the conditional component, i.e. "fixed
effects" terms from the model. Will only have an effect for models with
more than just the conditional model component.
\item \code{"smooth_terms"}: returns smooth terms, only applies to GAMs (or similar
models that may contain smooth terms).
\item \code{"zero_inflated"} (or \code{"zi"}): returns the zero-inflation component.
\item \code{"location"}: returns location parameters such as \code{conditional},
\code{zero_inflated}, or \code{smooth_terms} (everything that are fixed or random
effects - depending on the \code{effects} argument - but no auxiliary
parameters).
\item \code{"distributional"} (or \code{"auxiliary"}): components like \code{sigma},
\code{dispersion}, \code{beta} or \code{precision} (and other auxiliary parameters) are
returned.
}

For models of class \code{brmsfit} (package \strong{brms}), even more options are
possible for the \code{component} argument, which are not all documented in detail
here. See also \href{https://easystats.github.io/insight/reference/find_parameters.BGGM.html}{\code{?insight::find_parameters}}.
}

\examples{
\dontshow{if (require("rstanarm", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(bayestestR)

posterior <- rnorm(1000)
ci(posterior, method = "ETI")
ci(posterior, method = "HDI")

df <- data.frame(replicate(4, rnorm(100)))
ci(df, method = "ETI", ci = c(0.80, 0.89, 0.95))
ci(df, method = "HDI", ci = c(0.80, 0.89, 0.95))

model <- suppressWarnings(rstanarm::stan_glm(
  mpg ~ wt, data = mtcars, chains = 2, iter = 200, refresh = 0
))
ci(model, method = "ETI", ci = c(0.80, 0.89))
ci(model, method = "HDI", ci = c(0.80, 0.89))
\dontshow{\}) # examplesIf}
\dontshow{if (require("BayesFactor", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
bf <- BayesFactor::ttestBF(x = rnorm(100, 1, 1))
ci(bf, method = "ETI")
ci(bf, method = "HDI")
\dontshow{\}) # examplesIf}
\dontshow{if (require("emmeans", quietly = TRUE) && require("rstanarm", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
model <- emmeans::emtrends(model, ~1, "wt", data = mtcars)
ci(model, method = "ETI")
ci(model, method = "HDI")
\dontshow{\}) # examplesIf}
}
\references{
Gelman A, Greenland S. Are confidence intervals better termed "uncertainty
intervals"? BMJ 2019;l5381. 10.1136/bmj.l5381
}
\seealso{
Other ci: 
\code{\link{bci}()},
\code{\link{eti}()},
\code{\link{hdi}()},
\code{\link{si}()},
\code{\link{spi}()}
}
\concept{ci}
