#' Process: Recovering Fake Data
#'
#' @description
#' This function processes the synthetic datasets generated by
#'  `simulate_list`. For each of these simulated datasets, it then
#'  fits every model specified within the `fit_model` list. In essence,
#'  it iteratively calls the `optimize_para()` function for each
#'  generated object.
#'
#' The fitting procedure is analogous to that performed by `fit_p`,
#'  and it similarly leverages parallel computation across subjects
#'  to significantly accelerate the parameter estimation process.
#'
#' @param list [list] 
#' A list generated by function `simulate_list`
#' 
#' @param id [vector]
#' Specifies which subject's data to use. In parameter and model recovery
#'  analyses, the specific subject ID is often irrelevant. Although the
#'  experimental trial order might have some randomness for each subject,
#'  the sequence of reward feedback is typically pseudo-random.
#'
#' The default value for this argument is `NULL`. When `id` is `NULL`,
#'  the program automatically detects existing subject IDs within the
#'  dataset. It then randomly selects one subject as a sample, and the
#'  parameter and model recovery procedures are performed based on this
#'  selected subject's data.
#'  
#'  \code{default: id = NULL}
#' 
#' @param fit_model [function] 
#' fit model
#' 
#' @param funcs [character]
#' A character vector containing the names of all user-defined functions
#'  required for the computation. When parallel computation is enabled
#'  (i.e., `nc > 1`), user-defined models and their custom functions might
#'  not be automatically accessible within the parallel environment.
#'
#' Therefore, if you have created your own reinforcement learning model
#'  that modifies the package's default four default functions 
#'  (default functions: 
#'     \code{util_func = \link[binaryRL]{func_gamma}}, 
#'     \code{rate_func = \link[binaryRL]{func_eta}}, 
#'     \code{expl_func = \link[binaryRL]{func_epsilon}}
#'     \code{bias_func = \link[binaryRL]{func_pi}}
#'     \code{prob_func = \link[binaryRL]{func_tau}}
#'  ), 
#'  you must explicitly provide the names of your custom functions as a 
#'  vector here.
#' 
#' @param model_name [character] 
#' The name of your modal
#' 
#' @param n_params [integer] 
#' The number of free parameters in your model. 
#' 
#' @param n_trials [integer] 
#' The total number of trials in your experiment.
#'
#' @param initial_params [numeric]
#' Initial values for the free parameters that the optimization algorithm will
#'  search from. These are primarily relevant when using algorithms that require
#'  an explicit starting point, such as \code{L-BFGS-B}. If not specified,
#'  the function will automatically generate initial values close to zero.
#'  
#'  \code{default: initial_params = NA}.
#'
#' @param initial_size [integer]
#' This parameter corresponds to the \strong{population size} in genetic 
#'  algorithms (\code{GA}). It specifies the number of initial candidate
#'  solutions that the algorithm starts with for its evolutionary search.
#'  This parameter is only required for optimization algorithms that operate on
#'  a population, such as `GA` or `DEoptim`. 
#'  
#'  \code{Default: initial_size = 50}.
#' 
#' @param lower [vector] 
#' Lower bounds of free parameters
#' 
#' @param upper [vector] 
#' Upper bounds of free parameters
#' 
#' @param seed [integer] 
#' Random seed. This ensures that the results are 
#'  reproducible and remain the same each time the function is run. 
#'  
#'  \code{default: seed = 123}
#'  
#' @param iteration [integer] 
#' The number of iterations the optimization algorithm will perform
#'  when searching for the best-fitting parameters during the fitting
#'  phase. A higher number of iterations may increase the likelihood of 
#'  finding a global optimum but also increases computation time.
#'  
#' @param nc [integer]
#' Number of cores to use for parallel processing. Since fitting
#' optimal parameters for each subject is an independent task,
#' parallel computation can significantly speed up the fitting process:
#' \itemize{
#'   \item \strong{`nc = 1`}: The fitting proceeds sequentially.
#'   Parameters for one subject are fitted completely before moving
#'   to the next subject.
#'   \item \strong{`nc > 1`}: The fitting is performed in parallel
#'   across subjects. For example, if `nc = 4`, the algorithm will
#'   simultaneously fit data for four subjects. Once these are complete,
#'   it will proceed to fit the next batch of subjects (e.g., subjects
#'   5-8), and so on, until all subjects are processed.
#' }
#' 
#'  \code{default: nc = 1}
#' 
#' @param algorithm [character] 
#' Choose an algorithm package from
#'  `L-BFGS-B`, `GenSA`, `GA`, `DEoptim`, `PSO`, `Bayesian`, `CMA-ES`.
#'  
#' In addition, any algorithm from the `nloptr` package is also
#'  supported. If your chosen `nloptr` algorithm requires a local search,
#'  you need to input a character vector. The first element represents
#'  the algorithm used for global search, and the second element represents
#'  the algorithm used for local search.
#'  
#' @return a data frame for parameter recovery and model recovery
#' @examples
#' \dontrun{
#' binaryRL.res <- binaryRL::optimize_para(
#'   data = Mason_2024_Exp2,
#'   id = 1,
#'   n_params = 3,
#'   n_trials = 360,
#'   obj_func = binaryRL::RSTD,
#'   lower = c(0, 0, 0),
#'   upper = c(1, 1, 10),
#'   iteration = 100,
#'   algorithm = "L-BFGS-B"
#' )
#'
#' summary(binaryRL.res)
#' }
recovery_data <- function(
    list,
    id = 1,
    fit_model,
    funcs = NULL,
    model_name,
    n_params, 
    n_trials,
    lower,
    upper,
    initial_params = NA,
    initial_size = 50,
    iteration = 10,
    seed = 123,
    nc = 1,
    algorithm
){
  # 创建一个空数据集, 用于存放结果
  recovery <- data.frame(
    fit_model = rep(model_name, length(list)),
    ACC = NA,
    LL = NA,
    AIC = NA,
    BIC = NA
  )
  # 检测是都用同一个被试的题目, 还是每次都更换题目
  if (length(id) == 1) {
    id <- rep(id, length(list))
  }
  
  # 增加放置输入参数的列
  n_input_params <- length(list[[1]]$input)
  
  for (i in 1:n_input_params) {
    recovery[, i + 5] <- NA
    names(recovery)[i + 5] <- paste0("input_param_", i)
  }
  # 增加放置输出参数的列
  n_output_params <- length(lower)
  
  for (i in 1:n_output_params) {
    recovery[, i + 5 + n_input_params] <- NA
    names(recovery)[i + 5 + n_input_params] <- paste0("output_param_", i)
  }
  

  # Check for internally parallel algorithms
  if (nc == 1) {
    
    progressr::handlers(progressr::handler_txtprogressbar)
    
    progressr::with_progress({
      
      p <- progressr::progressor(steps = length(list))
      
      for (i in 1:length(list)) {
        
        p()
        
        data <- list[[i]][[1]]
        
        binaryRL.res <- binaryRL::optimize_para(
          data = data,
          id = id[i],
          obj_func = fit_model,
          n_params = n_params,
          n_trials = n_trials,
          lower = lower,
          upper = upper,
          initial_params = initial_params,
          initial_size = initial_size,
          iteration = iteration,
          seed = seed,
          algorithm = algorithm
        )
        
        recovery[i, 2] <- binaryRL.res$acc
        recovery[i, 3] <- binaryRL.res$ll
        recovery[i, 4] <- binaryRL.res$aic
        recovery[i, 5] <- binaryRL.res$bic
        
        for (j in 1:n_input_params) {
          recovery[i, j + 5] <- list[[i]]$input[j]
        }
        
        for (j in 1:n_output_params) {
          recovery[i, j + 5 + n_input_params] <- binaryRL.res$output[j]
        }
      }
    })
  }
  else {
    
    if (base::.Platform$OS.type == "windows") {
      future::plan(future::multisession, workers = nc)
    } else { # 包括 macOS, Linux, Unix
      future::plan(future::multicore, workers = nc)
    }
    
    doFuture::registerDoFuture()

    # 以迭代次数作为进度条
    n_iterations <- length(list) 
    
    progressr::handlers(progressr::handler_txtprogressbar)
    
    progressr::with_progress({
      
      p <- progressr::progressor(steps = n_iterations)
      
      doRNG::registerDoRNG(seed = seed)
      
      temp_recovery <- foreach::foreach(
        i = 1:n_iterations, .combine = rbind,
        .packages = "binaryRL", 
        .export = funcs
      ) %dorng% {
        
        data_i <- list[[i]][[1]]
        
        binaryRL.res <- binaryRL::optimize_para(
          data = data_i,
          id = id[i],
          obj_func = fit_model,
          n_params = n_params,
          n_trials = n_trials,
          lower = lower,
          upper = upper,
          initial_params = initial_params,
          initial_size = initial_size,
          iteration = iteration,
          seed = seed,
          algorithm = algorithm
        )
        
        row_i <- data.frame(matrix(NA, nrow = 1, ncol = 5 + n_input_params + n_output_params))
        row_i[1, 1] <- model_name
        row_i[1, 2] <- binaryRL.res$acc
        row_i[1, 3] <- binaryRL.res$ll
        row_i[1, 4] <- binaryRL.res$aic
        row_i[1, 5] <- binaryRL.res$bic
        
        for (j in 1:n_input_params) {
          row_i[1, 5 + j] <- list[[i]]$input[j]
        }
        for (j in 1:n_output_params) {
          row_i[1, 5 + n_input_params + j] <- binaryRL.res$output[j]
        }
        
        # 更新進度條
        p() 
        
        return(row_i)
      }
    })
    
    # 继承recovery的列名
    colnames(temp_recovery) <- colnames(recovery)
    recovery <- temp_recovery
  }
  
  return(recovery)
}
