\name{CE.ZINB}
\alias{CE.ZINB}
\title{
Multiple Break-point Detection via the CE Method with Zero-Inflated Negative Binomial Distribution}
\description{
Performs calculations to estimate both the number of break-points and their corresponding locations of discrete measurements with the CE method. Zero-inflated negative binomial distribution is used to model the excess zero observations and to model over-dispersesion in the oberved discrete (count) data. This function supports for the simulation of break-point locations in the CE algorithm based on the four parameter beta distribution and truncated normal distribution. The general BIC or AIC can be used to select the optimal number of break-points. 
}
\usage{
CE.ZINB(data, Nmax = 10, eps = 0.01, rho = 0.05, M = 200, h = 5, a = 0.8, b = 0.8,
distyp = 1, penalty = "BIC", parallel = FALSE)
}
\arguments{
  \item{data}{
data to be analysed. A single column array or a dataframe.
}
  \item{Nmax}{
maximum number of break-points. Default value is 10. 
}
  \item{eps}{
the cut-off value for the stopping criterion in the CE method. Default value is 0.01.
}
  \item{rho}{
the fraction which is used to obtain the best performing set of sample solutions (i.e., elite sample). Default value is 0.05.
}
  \item{M}{
sample size to be used in simulating the locations of break-points. Default value is 200.
}
  \item{h}{
minimum aberration width. Default is 5.
}
  \item{a}{
a smoothing parameter value. It is used in the four parameter beta distribution to smooth both shape parameters. When simulating from the truncated normal distribution, this value is used to smooth the estimates of the mean values. Default is 0.8.
}
  \item{b}{
a smoothing parameter value. It is used in the truncated normal distribution to smooth the estimates of the standard deviation. Default is 0.8.
}
  \item{distyp}{
distribution to simulate break-point locations. Options: 1 = four parameter beta distribution, 2 = truncated normal distribution. Default is 1. 
}
  \item{penalty}{
User can select either BIC or AIC to obtain the number of break-points. Options: "BIC", "AIC". Default is "BIC".
}
  \item{parallel}{
A logical argument specifying if parallel computation should be carried-out (TRUE) or not (FALSE). By default it is set as `FALSE'. In WINDOWS OS systems "snow" functionalities are used, whereas in Unix/Linux/MAC OSX "multicore" functionalities are used to carryout parallel computations with the maximum number of cores available.
}
}
\details{
Zero-inflated negative binomial (ZINB) distribution is used to model the discrete (count) data. ZINB model is preferred over the NB model when both excess zero values and over-dispersion observed in the count data. A performance function score (BIC) is calculated for each of the solutions generated by the statistical distribution (four parameter beta distribution or truncated normal distribution), which is used to simulate break-points from no break-point to the user provided maximum number of break-points. The solution that minimizes the BIC/AIC with respect to the number of break-points is reported as the optimal solution. Finally, a list containing a vector of break-point, BIC/AIC values and log-likelihood value is returned in the console. 
}
\value{
A list is returned with following items:
\item{No.BPs}{The number of break-points}
\item{BP.Loc}{A vector of break-point locations}
\item{BIC/AIC}{BIC/AIC value}
\item{ll}{Loglikelihood of the optimal solution}
}
\references{
  Priyadarshana, W. J. R. M. and Sofronov, G. (2012a) A Modified Cross- Entropy Method for Detecting Multiple Change-Points in DNA Count Data, In Proc. of the IEEE Conference on Evolutionary Computation (CEC), 1020-1027, DOI: 10.1109/CEC.2012.6256470.
  
  Priyadarshana, W. J. R. M. and Sofronov, G. (2012b) The Cross-Entropy Method and Multiple Change-Points Detection in Zero-Inflated DNA read count data, In: Y. T. Gu, S. C. Saha (Eds.) The 4th International Conference on Computational Methods (ICCM2012), 1-8, ISBN 978-1-921897-54-2.
  
  Rubinstein, R., and Kroese, D. (2004) The Cross-Entropy Method: A Unified Approach to Combinatorial Optimization, Monte-Carlo Simulation and Machine Learning. Springer-Verlag, New York.
  
  Schwarz, G. (1978) Estimating the dimension of a model, The Annals of Statistics, 6(2), 461-464.
}
\author{
Priyadarshana, W.J.R.M. <mjayawardana@swin.edu.au>
}

\seealso{
\code{\link{CE.NB}} for CE with negative binomial,

\code{\link{CE.NB.Init}} for CE with negative binomial with initial locations,

\code{\link{CE.ZINB.Init}} for CE with zero-inflated negative binomial with initial locations,

\code{\link{profilePlot}} to obtain mean profile plot.
}
\examples{
#### Simulated data example ###
# gamlss R package is used to simulate data from the ZINB.

\dontrun{
library(gamlss)
segs <- 6 # Number of segements
M <- c(1500, 2200, 800, 2500, 1000, 2000) # Segment width
#true.locations <- c(1501, 3701, 4501, 7001, 8001)  # True break-point locations 
seg <- NULL
p <- c(0.6, 0.1, 0.3, 0.05, 0.2, 0.4) # Specification of p's on each segment'
sigma.val <- c(1,2,3,4,5,6) # Specification of sigma vlaues 

for(j in 1:segs){
  seg <- c(seg, rZINBI(M[j], mu = 300, sigma = sigma.val[j], nu = p[j]))
}

simdata <- as.data.frame(seg)
rm(p, M, seg, segs, j, sigma.val)
#plot(data[, 1])

## CE with the four parameter beta distribution with BIC as the selection criterion  ##

obj1 <- CE.ZINB(simdata, distyp = 1, penalty = BIC, parallel = TRUE) # Parallel computation
obj1

profilePlot(obj1, simdata) # To obtain the mean profile plot

## CE with truncated normal distribution with BIC as the selection criterion  ##

obj2 <- CE.ZINB(simdata, distyp = 2, penalty = BIC, parallel = TRUE) # Parallel computation
obj2

profilePlot(obj2, simdata) # To obtain the mean profile plot
}
}
