% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/subsample_lat_band.R
\name{bandit}
\alias{bandit}
\title{Rarefy localities within latitudinal bands}
\usage{
bandit(
  dat,
  xy,
  iter,
  nSite,
  bin,
  centr = FALSE,
  absLat = FALSE,
  maxN = 90,
  maxS = -90,
  crs = "epsg:4326",
  output = "locs"
)
}
\arguments{
\item{dat}{A \code{data.frame} or \code{matrix} containing the coordinate
columns \code{xy} and any associated variables, e.g. taxon names.}

\item{xy}{A vector of two elements, specifying the name or numeric position
of columns in \code{dat} containing coordinates, e.g. longitude and latitude.
Coordinates for any shared sampling sites should be identical, and where sites
are raster cells, coordinates are usually expected to be cell centroids.}

\item{iter}{The number of times to subsample localities within \strong{each}
latitudinal band.}

\item{nSite}{The quota of unique locations to include in each subsample.}

\item{bin}{A positive numeric value for latitudinal band width, in degrees.}

\item{centr}{Logical: should a bin center on and cover the equator
(\code{TRUE}) or should the equator mark the boundary between the
lowest-latitude northern and southern bins (\code{FALSE}, default)?
Ignored if \code{absLat = TRUE}.}

\item{absLat}{Logical: should only the absolute values of latitude be
evaluated? If \code{absLat = TRUE}, \code{centr} argument is ignored.}

\item{maxN}{Optional argument to specify the northmost limit for subsampling,
if less than 90 degrees.}

\item{maxS}{Optional argument to specify the southmost limit for subsampling,
if not -90 degrees. Should be a negative value if in the southern hemisphere.}

\item{crs}{Coordinate reference system as a GDAL text string, EPSG code,
or object of class \code{crs}. Default is latitude-longitude (\code{EPSG:4326}).}

\item{output}{Whether the returned data should be two columns of
subsample site coordinates (\code{output = 'locs'}) or the subset of rows
from \code{dat} associated with those coordinates (\code{output = 'full'}).}
}
\value{
A list of subsamples, each a \code{data.frame} containing
coordinates of subsampled localities (if \code{output = 'locs'})
or the subset of occurrences from \code{dat} associated with those coordinates
(if \code{output = 'full'}). The latitudinal bounds of each subsample
are specified by its name in the list. If there are too few localities
in a given interval to draw a subsample, that interval is omitted from output.
}
\description{
\code{bandit} subsamples spatial point data to a specified number of sites
within bins of equal latitude
}
\details{
\code{bandit()} rarefies the number of spatial sites within latitudinal ranges
of specified bin width. (Compare with \code{\link[=cookies]{cookies()}} and \code{\link[=clustr]{clustr()}}, which spatially
subsample to a specified extent without regard to latitudinal position.)
Cases where it could be appropriate to control for latitudinal spread of localities
include characterisations of latitudinal diversity gradients (e.g. Marcot 2016)
or comparisons of ecosystem parameters that covary strongly with
latitude (e.g. diversity in reefal vs. non-reefal habitats). Note that
the total surface area of the Earth within equal-latitudinal increments
decreases from the equator towards the poles; \code{bandit()} standardises only
the amount of sites/area encompassed by each subsample, not the total area
that could have been available for species to inhabit.

As with all \code{divvy} subsampling functions, sites within a given
regional/latitudinal subsample are selected without replacement.

To calculate an integer number of degrees into which a given latitudinal
range divides evenly, the \code{palaeoverse} package (v 1.2.1) provides the
\code{\link[palaeoverse:lat_bins]{palaeoverse::lat_bins()}} function with argument \code{fit = TRUE}.
}
\examples{
# load bivalve occurrences to rasterise
library(terra)
data(bivalves)

# initialise Equal Earth projected coordinates
rWorld <- rast()
prj <- 'EPSG:8857'
rPrj <- project(rWorld, prj, res = 200000) # 200,000m is approximately 2 degrees

# coordinate column names for the current and target coordinate reference system
xyCartes <- c('paleolng','paleolat')
xyCell   <- c('centroidX','centroidY')

# project occurrences and retrieve cell centroids in new coordinate system
llOccs <- vect(bivalves, geom = xyCartes, crs = 'epsg:4326')
prjOccs <- project(llOccs, prj)
cellIds <- cells(rPrj, prjOccs)[,'cell']
bivalves[, xyCell] <- xyFromCell(rPrj, cellIds)

# subsample 20 equal-area sites within 10-degree bands of absolute latitude
n <- 20
reps <- 100
set.seed(11)
bandAbs <- bandit(dat = bivalves, xy = xyCell,
                  iter = reps, nSite = n, output = 'full',
                  bin = 10, absLat = TRUE,
                  crs = prj
)
head(bandAbs[[1]]) # inspect first subsample
names(bandAbs)[1] # degree interval (absolute value) of first subsample
#> [1] "[10,20)"
unique(names(bandAbs)) # all intervals containing sufficient data
#> [1] "[10,20)" "[20,30)" "[30,40)" "[40,50)"
# note insufficient coverage to subsample at equator or above 50 degrees

# subsample 20-degree bands, where central band spans the equator
# (-10 S to 10 N latitude), as in Allen et al. (2020)
# (An alternative, finer-grain way to divide 180 degrees evenly into an
# odd number of bands would be to set 'bin' = 4.)
bandCent <- bandit(dat = bivalves, xy = xyCell,
                   iter = reps, nSite = n, output = 'full',
                   bin = 20, centr = TRUE, absLat = FALSE,
                   crs = prj
)
unique(names(bandCent)) # all intervals containing sufficient data
#> [1] "[-50,-30)" "[10,30)" "[30,50)"

}
\references{
\insertRef{Allen2020}{divvy}

\insertRef{Marcot2016}{divvy}
}
\seealso{
\code{\link[=cookies]{cookies()}}

\code{\link[=clustr]{clustr()}}
}
