% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{dkw}
\alias{dkw}
\title{Density of the Kumaraswamy (Kw) Distribution}
\usage{
dkw(x, alpha, beta, log_prob = FALSE)
}
\arguments{
\item{x}{Vector of quantiles (values between 0 and 1).}

\item{alpha}{Shape parameter \code{alpha} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{beta}{Shape parameter \code{beta} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{log_prob}{Logical; if \code{TRUE}, the logarithm of the density is
returned (\eqn{\log(f(x))}). Default: \code{FALSE}.}
}
\value{
A vector of density values (\eqn{f(x)}) or log-density values
(\eqn{\log(f(x))}). The length of the result is determined by the recycling
rule applied to the arguments (\code{x}, \code{alpha}, \code{beta}).
Returns \code{0} (or \code{-Inf} if \code{log_prob = TRUE}) for \code{x}
outside the interval (0, 1), or \code{NaN} if parameters are invalid
(e.g., \code{alpha <= 0}, \code{beta <= 0}).
}
\description{
Computes the probability density function (PDF) for the two-parameter
Kumaraswamy (Kw) distribution with shape parameters \code{alpha} (\eqn{\alpha})
and \code{beta} (\eqn{\beta}). This distribution is defined on the interval (0, 1).
}
\details{
The probability density function (PDF) of the Kumaraswamy (Kw) distribution
is given by:
\deqn{
f(x; \alpha, \beta) = \alpha \beta x^{\alpha-1} (1 - x^\alpha)^{\beta-1}
}
for \eqn{0 < x < 1}, \eqn{\alpha > 0}, and \eqn{\beta > 0}.

The Kumaraswamy distribution is identical to the Generalized Kumaraswamy (GKw)
distribution (\code{\link{dgkw}}) with parameters \eqn{\gamma = 1},
\eqn{\delta = 0}, and \eqn{\lambda = 1}. It is also a special case of the
Exponentiated Kumaraswamy (\code{\link{dekw}}) with \eqn{\lambda = 1}, and
the Kumaraswamy-Kumaraswamy (\code{\link{dkkw}}) with \eqn{\delta = 0}
and \eqn{\lambda = 1}.
}
\examples{
\donttest{
# Example values
x_vals <- c(0.2, 0.5, 0.8)
alpha_par <- 2.0
beta_par <- 3.0

# Calculate density using dkw
densities <- dkw(x_vals, alpha_par, beta_par)
print(densities)

# Calculate log-density
log_densities <- dkw(x_vals, alpha_par, beta_par, log_prob = TRUE)
print(log_densities)
# Check: should match log(densities)
print(log(densities))

# Compare with dgkw setting gamma = 1, delta = 0, lambda = 1
densities_gkw <- dgkw(x_vals, alpha_par, beta_par, gamma = 1.0, delta = 0.0,
                      lambda = 1.0)
print(paste("Max difference:", max(abs(densities - densities_gkw)))) # Should be near zero

# Plot the density for different shape parameter combinations
curve_x <- seq(0.001, 0.999, length.out = 200)
plot(curve_x, dkw(curve_x, alpha = 2, beta = 3), type = "l",
     main = "Kumaraswamy Density Examples", xlab = "x", ylab = "f(x)",
     col = "blue", ylim = c(0, 4))
lines(curve_x, dkw(curve_x, alpha = 3, beta = 2), col = "red")
lines(curve_x, dkw(curve_x, alpha = 0.5, beta = 0.5), col = "green") # U-shaped
lines(curve_x, dkw(curve_x, alpha = 5, beta = 1), col = "purple") # J-shaped
lines(curve_x, dkw(curve_x, alpha = 1, beta = 3), col = "orange") # J-shaped (reversed)
legend("top", legend = c("a=2, b=3", "a=3, b=2", "a=0.5, b=0.5", "a=5, b=1", "a=1, b=3"),
       col = c("blue", "red", "green", "purple", "orange"), lty = 1, bty = "n", ncol = 2)
}

}
\references{
Kumaraswamy, P. (1980). A generalized probability density function for
double-bounded random processes. \emph{Journal of Hydrology}, \emph{46}(1-2), 79-88.

Jones, M. C. (2009). Kumaraswamy's distribution: A beta-type distribution
with some tractability advantages. \emph{Statistical Methodology}, \emph{6}(1), 70-81.
}
\seealso{
\code{\link{dgkw}} (parent distribution density),
\code{\link{dekw}}, \code{\link{dkkw}},
\code{\link{pkw}}, \code{\link{qkw}}, \code{\link{rkw}} (other Kw functions),
\code{\link[stats]{dbeta}}
}
\author{
Lopes, J. E.
}
\keyword{density}
\keyword{distribution}
\keyword{kumaraswamy}
