% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mediate_tsls.R
\name{mediate_tsls}
\alias{mediate_tsls}
\title{Two-stage Least Squares Estimation of the Average Causal Mediation Effects}
\usage{
mediate_tsls(model.m, model.y, treat = "treat.name", conf.level = 0.95,
  robustSE = FALSE, cluster = NULL, boot = FALSE, sims = 1000,
  est_se = TRUE, ...)
}
\arguments{
\item{model.m}{a fitted model object for mediator, of class \code{lm}.}

\item{model.y}{a fitted model object for outcome, of class \code{lm}.}

\item{treat}{a character string indicating the name of the treatment variable
used in the models.}

\item{conf.level}{level of the returned two-sided confidence intervals. 
Default is to return the 2.5 and 97.5 percentiles of the simulated 
quantities.}

\item{robustSE}{a logical value. If 'TRUE', heteroskedasticity-consistent 
standard errors will be used. Default is 'FALSE'.}

\item{cluster}{a variable indicating clusters for standard errors. Note that 
this should be a vector of cluster indicators itself, not a character 
string for the name of the variable.}

\item{boot}{a logical value. if \code{FALSE} analytic confidence intervals
based on Aroian (1947) will be used; if \code{TRUE} nonparametric
bootstrap will be used. Default is \code{FALSE}.}

\item{sims}{number of Monte Carlo draws for nonparametric bootstrap.}

\item{est_se}{estimate standard errors. Primarily for internal use. Default is \code{TRUE}.}

\item{...}{other arguments passed to vcovCL in the sandwich package:
typically the \code{type} argument for heteroskedasticity-consistent 
standard errors.}
}
\value{
\code{mediate} returns an object of class "\code{mediate}", 
  "\code{mediate.tsls}", a list that contains the components listed below.  
  
  The function \code{summary}  can be used to obtain a table of the results.  
  
  \item{d1}{point estimate for average causal mediation effects.}
  \item{d1.ci}{confidence intervals for average causal mediation 
  effect. The confidence level is set at the value specified in 
  'conf.level'.}
  \item{z0}{point estimates for average direct effect.}
  \item{z0.ci}{confidence intervals for average direct effect.}
  \item{z0.p}{two-sided p-values for average causal direct effect.}
  \item{n0}{the "proportions mediated", or the size of the average causal 
  mediation effect relative to the total effect.}
  \item{n0.ci}{confidence intervals for the proportion mediated.}
  \item{n0.p}{two-sided p-values for proportion mediated.}
  \item{tau.coef}{point estimate for total effect.}
  \item{tau.ci}{confidence interval for total effect.}
  \item{tau.p}{two-sided p-values for total effect.}
  \item{boot}{logical, the \code{boot} argument used.}
  \item{treat}{a character string indicating the name of the 'treat' variable 
  used.}
  \item{mediator}{a character string indicating the name of the 'mediator' 
  variable used.}
  \item{INT}{a logical value indicating whether the model specification 
  allows the effects to differ between the treatment and control conditions.}
  \item{conf.level}{the confidence level used. }
  \item{model.y}{the outcome model used.}
  \item{model.m}{the mediator model used.}
  \item{nobs}{number of observations in the model frame for 'model.m' and 
  'model.y'. May differ from the numbers in the original models input to 
  'mediate' if 'dropobs' was 'TRUE'.}
  \item{cluster}{the clusters used.}
}
\description{
Estimate quantities for causal mediation analysis using an instrumental 
variable estimator.
}
\examples{
# Generate data. We use TSLS to address unobserved confounding (n).
set.seed(123)
sims <- 1000
dat <- data.frame(z = sample(0:1, sims, replace = TRUE), 
                  t = sample(0:1, sims, replace = TRUE))
dat$n <- rnorm(sims, mean = 1)
dat$m <- rnorm(sims, mean = dat$z * 0.3 + dat$t * 0.2 + dat$n * 0.7, sd = 0.2)
dat$y <- rnorm(sims, mean = 5 + dat$t + dat$m * (-3) + dat$n, sd = 1)
model.m <- lm(m ~ t + z, data = dat)
model.y <- lm(y ~ t + m, data = dat)
cluster <- factor(sample(1:3, sims, replace = TRUE))
med <- mediate_tsls(model.m, model.y, cluster = cluster, treat = "t")
summary(med) 
}
\references{
Aroian, L. A. 1947. The probability function of the product of
  two normally distributed variables. *Annals of Mathematical Statistics,*
  18, 265-271.
}
