% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/S3_definitions.R
\name{summary.survtab}
\alias{summary.survtab}
\title{Summarize a survtab Object}
\usage{
\method{summary}{survtab}(object, t = NULL, subset = NULL, q = NULL, ...)
}
\arguments{
\item{object}{a \code{survtab} object}

\item{t}{a vector of times at which time points (actually intervals that
contain t) to print summary table of survival function estimates by strata;
values not existing in any interval cause rows containing only \code{NAs} to
be returned.}

\item{subset}{a logical condition to subset results table by
before printing; use this to limit to a certain stratum. E.g.
\code{subset = sex == "male"}}

\item{q}{a named \code{list} of quantiles to include in returned data set,
where names must match to estimates in \code{object};
returns intervals where the quantiles are reached first;
e.g. \code{list(surv.obs = 0.5)} finds the interval where \code{surv.obs}
is 0.45 and 0.55 at the beginning and end of the interval, respectively;
returns rows with \code{NA} values for quantiles not reached in estimates
(e.g. if \code{q = list(surv.obs = 0.5)} but lowest estimate is 0.6);
see Examples.}

\item{...}{unused; required for congruence with other \code{summary} methods}
}
\value{
A `data.table`: a slice from `object` based on `t`, `subset`, and `q`.
}
\description{
Summary method function for \code{survtab} objects; see
\code{\link{survtab_ag}}. Returns estimates at given time points
or all time points if \code{t} and \code{q} are both \code{NULL}.
}
\details{
Note that this function returns the intervals and NOT the time points
corresponding to quantiles / estimates corresponding to time points.
If you want precise estimates at time points that are not interval breaks,
add the time points as breaks and re-estimate the survival time function.
In interval-based estimation, the estimates denote e.g. probability of 
dying \emph{during} the interval, so time points within the intervals
are not usually considered at all. See e.g. Seppa, Dyba, and Hakulinen 
(2015).
}
\examples{

library(Epi)

## NOTE: recommended to use factor status variable
x <- Lexis(entry = list(FUT = 0, AGE = dg_age, CAL = get.yrs(dg_date)), 
           exit = list(CAL = get.yrs(ex_date)), 
           data = sire[sire$dg_date < sire$ex_date, ],
           exit.status = factor(status, levels = 0:2, 
           labels = c("alive", "canD", "othD")), 
           merge = TRUE)
## pretend some are male
set.seed(1L)
x$sex <- rbinom(nrow(x), 1, 0.5)
## observed survival
st <- survtab(Surv(time = FUT, event = lex.Xst) ~ sex, data = x, 
                  surv.type = "cif.obs",
                  breaks = list(FUT = seq(0, 5, 1/12)))

## estimates at full years of follow-up
summary(st, t = 1:5)

## interval estimate closest to 75th percentile, i.e.
## first interval where surv.obs < 0.75 at end
## (just switch 0.75 to 0.5 for median survival, etc.)
summary(st, q = list(surv.obs = 0.75))
## multiple quantiles
summary(st, q = list(surv.obs = c(0.75, 0.90), CIF_canD = 0.20))

## if you want all estimates in a new data.frame, you can also simply do

x <- as.data.frame(st)
}
\references{
Seppa K., Dyba T. and Hakulinen T.: Cancer Survival, 
Reference Module in Biomedical Sciences. Elsevier. 08-Jan-2015.
\doi{10.1016/B978-0-12-801238-3.02745-8}
}
\seealso{
Other survtab functions: 
\code{\link{Surv}()},
\code{\link{lines.survtab}()},
\code{\link{plot.survtab}()},
\code{\link{print.survtab}()},
\code{\link{survtab}()},
\code{\link{survtab_ag}()}
}
\author{
Joonas Miettinen
}
\concept{survtab functions}
