% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/redist_nuts.R
\name{redist_nuts}
\alias{redist_nuts}
\title{Data redistribution to NUTS subdivisions}
\usage{
redist_nuts(
  data,
  nuts_col,
  values_col,
  to_nuts = 2,
  redist_data = "population",
  redist_nuts_col = NULL,
  redist_values_col = NULL,
  population_year = 2023,
  nuts_year = 2016
)
}
\arguments{
\item{data}{A data frame containing the data at the country-level to
redistribute.}

\item{nuts_col}{A string specifying the column name in \code{data}
with the NUTS0 codes.}

\item{values_col}{A string or vector specifying the column name(s) in \code{data}
with the values to be redistributed.}

\item{to_nuts}{A numeric value (1, 2, or 3) specifying the NUTS level for redistribution.
Default 2, indicating redistribution to NUTS2.}

\item{redist_data}{A data frame containing values for each subdivision that will be
used as the basis for proportional redistribution. By default, this is set to
\code{"population"}, indicating redistribution based on human population data from
\href{https://ec.europa.eu/eurostat/databrowser/product/page/demo_r_pjangrp3}{Eurostat}.}

\item{redist_nuts_col}{A string specifying the column name in \code{redist_data}
that contains the destination NUTS codes. The NUTS level should correspond to the
value specified in  \code{to_nuts}. \code{NULL} (default) if a data frame is not
incorporated in \code{redist_data} (i.e., \code{redist_data = "population"}).}

\item{redist_values_col}{A string specifying the column name in \code{redist_data}
with the values for proportional redistribution. These values will serve as the
weights for the redistribution process. \code{NULL} (default) if a data frame
is not incorporated in \code{redist_data} (i.e., \code{redist_data = "population"}).}

\item{population_year}{A numeric value specifying the year(s) of the human population
data to be used for redistribution. Only necessary if \code{"population"} is specified in
\code{redist_data} (default is 2023). If multiple years are provided, the average
human population across those years will be used. Available years can be found at
\href{https://ec.europa.eu/eurostat/databrowser/product/page/demo_r_pjangrp3}{Eurostat}.}

\item{nuts_year}{Year of NUTS classification. Accepted values are '2003','2006','2010','2013',
'2016' (default),'2021', or '2024'. See
\href{https://ec.europa.eu/eurostat/web/nuts/history}{NUTS - History}.}
}
\value{
A data frame with the redistributed values across the specified NUTS
level. The data frame contains the columns \code{NUTSX} with the codes at the
selected NUTS level, \code{NUTS0} with the codes at country level, \code{proportion} with the
proportion according to which the values have been redistributed, and the columns
corresponding to the redistributed values with the same name specified in \code{values_col}.
}
\description{
Value redistribution from country-level (NUTS0) to smaller territories (i.e.,
NUTS1, NUTS2 or NUTS3). See
\href{https://ec.europa.eu/eurostat/web/nuts}{Nomenclature of territorial units for statistics}.
}
\details{
This function enables redistribution of values from national-level NUTS0
to smaller territorial units (i.e., NUTS1, NUTS2, or NUTS3), either proportionally
based on human population data from
\href{https://ec.europa.eu/eurostat/databrowser/product/page/demo_r_pjangrp3}{Eurostat}
or using user-supplied redistribution proportions.
Human population data for redistribution is automatically fetched for the specified time
period from the Eurostat database.

Note that more than one column of values provided in the data frame \code{data}
can be redistributed at the same time. The values in columns \code{values_col}
and \code{redist_values_col} must be numeric and positive.
\subsection{Common uses}{

In the context of quantitative pest risk assessment (qPRA) at the entry step,
this function can be applied to redistribute the quantity of potentially infested
commodities (\eqn{N_{trade}}, see \code{\link[=ntrade]{ntrade()}}) or the number of potential
founder populations (\eqn{NPFP}, see \code{\link[=pathway_model]{pathway_model()}}). For this purpose,
human population or consumption data from subdivisions are often used for redistribution.
}
}
\examples{
## Example of data redistribution to NUTS2 using human population data
data("datatrade_EU")
# extract NUTS0 codes (country level)
nuts0 <- unique(datatrade_EU$internal_production$reporter)
# simulate values for each country
nuts0_data <- data.frame(nuts0 = nuts0,
                         value = abs(rnorm(length(nuts0), 30000, 10000)))

\donttest{
# Redistribution
data_redist <- redist_nuts(data = nuts0_data,
                           nuts_col = "nuts0",
                           values_col = "value",
                           to_nuts = 2,
                           redist_data = "population",
                           population_year = c(2017, 2018, 2019))
head(data_redist)

# Plot
plot_nuts(data = data_redist,
          nuts_level = 2,
          nuts_col = "NUTS2", 
          values_col = "value")
}

## Example of data redistribution to NUTS1 using custom data
# consumption data at NUTS1 level
nuts1_data <- datatrade_EU$consumption_nuts1

# Redistribution
data_redist <- redist_nuts(data = nuts0_data,
                           nuts_col = "nuts0",
                           values_col = "value",
                           to_nuts = 1,
                           redist_data = nuts1_data,
                           redist_nuts_col = "NUTS_ID",
                           redist_values_col = "value")

head(data_redist)

# Plot
plot_nuts(data = data_redist,
          nuts_level = 1,
          nuts_col = "NUTS1", 
          values_col = "value")

}
\seealso{
\code{\link[=ntrade]{ntrade()}}, \code{\link[=pathway_model]{pathway_model()}}
}
