% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dtc_create_iso8601.R
\name{create_iso8601}
\alias{create_iso8601}
\title{Convert date or time collected values to ISO 8601}
\usage{
create_iso8601(
  ...,
  .format,
  .fmt_c = fmt_cmp(),
  .na = NULL,
  .cutoff_2000 = 68L,
  .check_format = FALSE,
  .warn = TRUE
)
}
\arguments{
\item{...}{Character vectors of dates, times or date-times' components.}

\item{.format}{Parsing format(s). Either a character vector or a list of
character vectors. If a character vector is passed then each element is
taken as parsing format for each vector passed in \code{...}. If a list is
provided, then each element must be a character vector of formats. The
first vector of formats is used for parsing the first vector passed in
\code{...}, and so on.}

\item{.fmt_c}{A list of regexps to use when parsing \code{.format}. Use \code{\link[=fmt_cmp]{fmt_cmp()}}
to create such an object to pass as argument to this parameter.}

\item{.na}{A character vector of string literals to be regarded as missing
values during parsing.}

\item{.cutoff_2000}{An integer value. Two-digit years smaller or equal to
\code{.cutoff_2000} are parsed as though starting with \code{20}, otherwise parsed as
though starting with \code{19}.}

\item{.check_format}{Whether to check the formats passed in \code{.format},
meaning to check against a selection of validated formats in
\link[=dtc_formats]{dtc_formats}; or to have a more permissible
interpretation of the formats.}

\item{.warn}{Whether to warn about parsing failures.}
}
\value{
A vector of dates, times or date-times in \href{https://en.wikipedia.org/wiki/ISO_8601}{ISO 8601} format
}
\description{
\code{\link[=create_iso8601]{create_iso8601()}} converts vectors of dates, times or date-times to \href{https://en.wikipedia.org/wiki/ISO_8601}{ISO 8601} format. Learn more in
\code{vignette("iso_8601")}.
}
\examples{
# Converting dates
create_iso8601(c("2020-01-01", "20200102"), .format = "y-m-d")
create_iso8601(c("2020-01-01", "20200102"), .format = "ymd")
create_iso8601(c("2020-01-01", "20200102"), .format = list(c("y-m-d", "ymd")))

# Two-digit years are supported
create_iso8601(c("20-01-01", "200101"), .format = list(c("y-m-d", "ymd")))

# `.cutoff_2000` sets the cutoff for two-digit to four-digit year conversion
# Default is at 68.
create_iso8601(c("67-01-01", "68-01-01", "69-01-01"), .format = "y-m-d")

# Change it to 80.
create_iso8601(c("79-01-01", "80-01-01", "81-01-01"), .format = "y-m-d", .cutoff_2000 = 80)

# Converting times
create_iso8601("15:10", .format = "HH:MM")
create_iso8601("2:10", .format = "HH:MM")
create_iso8601("2:1", .format = "HH:MM")
create_iso8601("02:01:56", .format = "HH:MM:SS")
create_iso8601("020156.5", .format = "HHMMSS")

# Converting date-times
create_iso8601("12 NOV 202015:15", .format = "dd mmm yyyyHH:MM")

# Indicate allowed missing values to make the parsing pass
create_iso8601("U DEC 201914:00", .format = "dd mmm yyyyHH:MM")
create_iso8601("U DEC 201914:00", .format = "dd mmm yyyyHH:MM", .na = "U")

create_iso8601("NOV 2020", .format = "m y")
create_iso8601(c("MAR 2019", "MaR 2020", "mar 2021"), .format = "m y")

create_iso8601("2019-04-041045-", .format = "yyyy-mm-ddHHMM-")

create_iso8601("20200507null", .format = "ymd(HH:MM:SS)")
create_iso8601("20200507null", .format = "ymd((HH:MM:SS)|null)")

# Fractional seconds
create_iso8601("2019-120602:20:13.1230001", .format = "y-mdH:M:S")

# Use different reserved characters in the format specification
# Here we change "H" to "x" and "M" to "w", for hour and minute, respectively.
create_iso8601("14H00M", .format = "HHMM")
create_iso8601("14H00M", .format = "xHwM", .fmt_c = fmt_cmp(hour = "x", min = "w"))

# Alternative formats with unknown values
datetimes <- c("UN UNK 201914:00", "UN JAN 2021")
format <- list(c("dd mmm yyyy", "dd mmm yyyyHH:MM"))
create_iso8601(datetimes, .format = format, .na = c("UN", "UNK"))

# Dates and times may come in many format variations
fmt <- "dd MMM yyyy HH nn ss"
fmt_cmp <- fmt_cmp(mon = "MMM", min = "nn", sec = "ss")
create_iso8601("05 feb 1985 12 55 02", .format = fmt, .fmt_c = fmt_cmp)

}
