% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/virtual-select.R
\name{updateVirtualSelect}
\alias{updateVirtualSelect}
\title{Update virtual select from server}
\usage{
updateVirtualSelect(
  inputId,
  label = NULL,
  choices = NULL,
  selected = NULL,
  disable = NULL,
  disabledChoices = NULL,
  open = NULL,
  session = shiny::getDefaultReactiveDomain()
)
}
\arguments{
\item{inputId}{The \code{input} slot that will be used to access the value.}

\item{label}{Display label for the control, or \code{NULL} for no label.}

\item{choices}{List of values to select from.
You can use:
\itemize{
\item \code{vector} use a simple vector for better performance.
\item \verb{named list} / \verb{named vector} in the same way as with \code{\link[shiny:selectInput]{shiny::selectInput()}}
\item custom formatted \code{list} allowing to use more options, must correspond to \href{https://sa-si-dev.github.io/virtual-select/#/properties}{virtual-select specifications}
\item output of \code{\link[=prepare_choices]{prepare_choices()}}
}}

\item{selected}{The initially selected value (or multiple values if \code{multiple = TRUE}). If not specified then defaults to the first value for
single-select lists and no values for multiple select lists.}

\item{disable}{Disable (\code{TRUE}) or enable (\code{FALSE}) the select menu.}

\item{disabledChoices}{List of disabled option's values.}

\item{open}{Open (\code{TRUE}) or close (\code{FALSE}) the dropdown.}

\item{session}{The \code{session} object passed to function given to
\code{shinyServer}. Default is \code{getDefaultReactiveDomain()}.}
}
\value{
No value.
}
\description{
Update a \code{\link[=virtualSelectInput]{virtualSelectInput()}} from the server.
}
\examples{
library(shiny)
library(shinyWidgets)

ui <- fluidPage(
  tags$h2("Virtual Select (update)"),

  virtualSelectInput(
    inputId = "sel1",
    label = "Update label:",
    choices = month.name
  ),
  verbatimTextOutput("res1"),
  textInput("label_text", label = "With text:"),
  textInput("label_html", label = "With HTML:"),

  virtualSelectInput(
    inputId = "sel2",
    label = "Update selected value:",
    choices = month.name
  ),
  verbatimTextOutput("res2"),
  radioButtons("selected", "Selected value:", month.name, inline = TRUE),

  virtualSelectInput(
    inputId = "sel3",
    label = "Update choices:",
    choices = tolower(month.name)
  ),
  verbatimTextOutput("res3"),
  radioButtons("choices", "Choices:", c("lowercase", "UPPERCASE"), inline = TRUE),

  virtualSelectInput(
    inputId = "sel4",
    label = "Update choices + selected:",
    choices = tolower(month.name)
  ),
  verbatimTextOutput("res4"),
  radioButtons("choices_select", "Choices:", c("lowercase", "UPPERCASE"), inline = TRUE),

  virtualSelectInput(
    inputId = "sel5",
    label = "Disable / enable:",
    choices = tolower(month.name)
  ),
  verbatimTextOutput("res5"),
  checkboxInput("disable", "Disable", value = FALSE),
  checkboxInput("disableChoices", "Disable march and june", value = FALSE),

  virtualSelectInput(
    inputId = "sel6",
    label = "Open / close:",
    choices = tolower(month.name)
  ),
  verbatimTextOutput("res6"),
  checkboxInput("open", "Open?", value = FALSE)

)

server <- function(input, output, session) {
  output$res1 <- renderPrint(input$sel1)
  observe({
    req(input$label_text)
    updateVirtualSelect(inputId = "sel1", label = input$label_text)
  })
  observe({
    req(input$label_html)
    updateVirtualSelect(
      inputId = "sel1",
      label = tags$span(input$label_html, style = "color: red;")
    )
  })

  output$res2 <- renderPrint(input$sel2)
  observe({
    updateVirtualSelect(inputId = "sel2", selected = input$selected)
  })

  output$res3 <- renderPrint(input$sel3)
  observe({
    if (identical(input$choices, "lowercase")) {
      updateVirtualSelect(inputId = "sel3", choices = tolower(month.name))
    } else {
      updateVirtualSelect(inputId = "sel3", choices = toupper(month.name))
    }
  })

  output$res4 <- renderPrint(input$sel4)
  observe({
    if (identical(input$choices_select, "lowercase")) {
      choices <- tolower(month.name)
    } else {
      choices <- toupper(month.name)
    }
    selected <- sample(choices, 1)
    updateVirtualSelect(inputId = "sel4", choices = choices, selected = selected)
  })

  output$res5 <- renderPrint(input$sel5)
  observe({
    if (isTRUE(input$disable)) {
      updateVirtualSelect(inputId = "sel5", disable = TRUE)
    } else {
      updateVirtualSelect(inputId = "sel5", disable = FALSE)
    }
  })
  observe({
    if (isTRUE(input$disableChoices)) {
      updateVirtualSelect(inputId = "sel5", disabledChoices = c("march", "june"))
    } else {
      updateVirtualSelect(inputId = "sel5", disabledChoices = character(0))
    }
  })

  observeEvent(input$open, {
    updateVirtualSelect(inputId = "sel6", open = input$open)
  }, ignoreInit = TRUE)

}

if (interactive())
  shinyApp(ui, server)
}
\seealso{
\code{\link[=virtualSelectInput]{virtualSelectInput()}} for creating a widget in the UI.
}
