% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/splinecopula.R
\name{spline.copula}
\alias{spline.copula}
\title{B-spline copula using the five M-spline basis functions}
\usage{
spline.copula(u, v, R = "PE1", mat = FALSE, density = FALSE)
}
\arguments{
\item{u}{A numeric vector of values in [0, 1] (uniform marginals for the first variable).}

\item{v}{A numeric vector of values in [0, 1] (uniform marginals for the second variable).}

\item{R}{A 5×5 non-negative coefficient matrix defining the copula structure. 
The matrix must satisfy the following conditions:
\itemize{
  \item All entries must be non-negative (\eqn{R_{kl} \ge 0}).
  \item The sum of all entries must be exactly 1.
  \item The row sums and column sums must equal:
        \code{c(1/8, 1/4, 1/4, 1/4, 1/8)} (in order).
}
These conditions ensure that the resulting function is a valid copula density.
You may also specify one of the built-in presets:
"PE1", "PE2",–"PE3", "PN1", "PN2", "PN3", "I", NE1", "NE2", "NE3", "NN1", "NN2", "NN3".
Default `"PE1"`.}

\item{mat}{Logical; if \code{TRUE}, returns the full matrix (outer product) of copula evaluations;
otherwise returns only the diagonal values, i.e., C(u_i, v_i) or c(u_i, v_i) for i = 1,...,n. Default is \code{FALSE}.}

\item{density}{Logical; if \code{TRUE}, evaluates the copula density; if \code{FALSE}, evaluates the copula function. Default is \code{FALSE}.}
}
\value{
A numeric vector or matrix. If \code{mat=FALSE}, returns a vector of length \code{length(u)};
if \code{mat=TRUE}, returns a matrix of size \code{length(u)} x \code{length(v)}.
}
\description{
\code{spline.copula} computes the B-spline copula (or its density) based on the five-parameter M-spline basis functions.
This copula is a specific instance of the B-spline copula family,
and can be implemented using matrix-based operations with \code{M.spline} and \code{I.spline} from the \pkg{joint.Cox} package.
}
\details{
If \code{density = TRUE}, the function computes the copula **density** \(c(u, v)\); otherwise, it returns the **copula distribution function** \(C(u, v)\).
If \code{density = FALSE}, it returns the copula function. The implementation uses five M-spline or I-spline
basis functions defined on [0,1]. The coefficient matrix is fixed internally but can be modified in advanced use.
}
\examples{
# Example data
library(joint.Cox)
library(ggplot2)

N = 50
u = v = seq(from = 0, to = 1, length.out = N)
U = rep(u, N)
V = rep(v, each = N)

c.data = data.frame(U = U, V = V, C = spline.copula(U, V, R = "PE1", density=TRUE))
ggplot(aes(x = U, y = V), data = c.data) +
  geom_contour(aes(x = U, y = V, z = C, colour = after_stat(level)),
  data = c.data, bins=25) + xlab("u") + ylab("v")

}
\seealso{
\code{\link[joint.Cox]{M.spline}}, \code{\link[joint.Cox]{I.spline}}
}
